#include <SYS/TYPES.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/VM.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/MPS.H>
#include <KERNEL/FEATURE.H>
#include <KERNEL/BOOTCFG.H>
#include <STRING.H>

#include <KERNEL/ACPI.H>

static char ACPI_FOUND = 0;
static __p_addr RSDT = 0;

#define n_rsdt_entries(rsdt)	((__32LE2CPU((rsdt)->table.length) - __offsetof(ACPI_RSDT, ptr)) / 4)

static int VM_INITIALIZED(void)
{
	__u32 cr0;
	__asm__ ("MOVL %%CR0, %0":"=r"(cr0)::"memory");
	if (cr0 & CR0_PG) return 1;
	else return 0;
}

static void *ACPI_MAP(__p_addr addr, unsigned length)
{
	void *r;
	if (!VM_INITIALIZED()) {
		if ((__p_addr)(unsigned long)(void *)(unsigned long)addr != addr)
			return __ERR_PTR(-EOVERFLOW);
		return (void *)(unsigned long)addr;
	}
	r = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, length, PAT_WB | PAT_RO);
	return r;
}

void ACPI_UNMAP(const void *ptr, unsigned length)
{
	if (!VM_INITIALIZED()) return;
	return KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(ptr, length);
}

static int ACPI_VERIFY_TABLE(const ACPI_TABLE *table, const char *signature, unsigned len)
{
	if (memcmp(table->signature, signature, 4))
		return -ENOENT;
	if (__32LE2CPU(table->length) < len)
		return -EINVAL;
	if (MPS_CHECKSUM(table, __32LE2CPU(table->length)))
		return -EIO;
	return 0;
}

static const void *ACPI_MAP_TABLE(__p_addr addr, const char *signature, unsigned len, unsigned *real_len)
{
	int r;
	const ACPI_TABLE *table = ACPI_MAP(addr, sizeof(ACPI_TABLE));
	if (__IS_ERR(table))
		return table;
	r = ACPI_VERIFY_TABLE(table, signature, len);
	*real_len = __32LE2CPU(table->length);
	ACPI_UNMAP(table, sizeof(ACPI_TABLE));
	if (r)
		return __ERR_PTR(r);
	return ACPI_MAP(addr, *real_len);
}

const void *ACPI_FIND_TABLE(const char *signature, unsigned len, unsigned *real_len)
{
	unsigned i;
	unsigned real_rsdt_len;
	const ACPI_RSDT *rsdt;
	if (!ACPI_FOUND)
		return __ERR_PTR(-ENXIO);
	rsdt = ACPI_MAP_TABLE(RSDT, ACPI_RSDT_SIGNATURE, ACPI_RSDT_LENGTH, &real_rsdt_len);
	if (__unlikely(__IS_ERR(rsdt)))
		return rsdt;
	for (i = 0; i < n_rsdt_entries(rsdt); i++) {
		const void *table;
		__p_addr ptr;
		ptr = rsdt->ptr[i];
		table = ACPI_MAP_TABLE(ptr, signature, len, real_len);
		if (!__IS_ERR(table)) {
			ACPI_UNMAP(rsdt, real_rsdt_len);
			return table;
		}
	}
	ACPI_UNMAP(rsdt, real_rsdt_len);
	return __ERR_PTR(-ENOENT);
}

static const __u8 *acpi_find_start;
static const __u8 *acpi_find_end;
static __u8 acpi_find_type;
static __u8 acpi_find_length;

void ACPI_FIND_RESET(const __u8 *start, const __u8 *end, __u8 type, __u8 length)
{
	acpi_find_start = start;
	acpi_find_end = end;
	acpi_find_type = type;
	acpi_find_length = length;
}

const void *ACPI_FIND_ENTRY(void)
{
	static union {
		__u8 data[256];
		__u64 align[256 / 8];
	} buffer;
	while (acpi_find_start <= acpi_find_end - sizeof(ACPI_ENTRY_HEADER)) {
		const ACPI_ENTRY_HEADER *entry = (const ACPI_ENTRY_HEADER *)acpi_find_start;
		if (acpi_find_end - acpi_find_start < entry->length) break;
		acpi_find_start += entry->length;
		if (entry->type == acpi_find_type) {
			if (entry->length < acpi_find_length) break;
			return memcpy(buffer.data, entry, entry->length);
		}
	}
	return NULL;
}

static int find_rsdp(unsigned long base, size_t len)
{
	const ACPI_RSDP *rsdp = (const ACPI_RSDP *)base;
	const ACPI_RSDT *rsdt;
	unsigned i;
	while (len >= ACPI_RSDP_LENGTH_1) {
		if (!memcmp(rsdp->signature, ACPI_RSDP_SIGNATURE, 8)) {
			if (MPS_CHECKSUM(rsdp, ACPI_RSDP_LENGTH_1)) goto bad_rsdp;
			rsdt = (ACPI_RSDT *)__32LE2CPU(rsdp->rsdt);
			if (!ACPI_VERIFY_TABLE(&rsdt->table, ACPI_RSDT_SIGNATURE, ACPI_RSDT_LENGTH)) {
				goto have_it;
			}
		}
		bad_rsdp:
		rsdp = (ACPI_RSDP *)((__u8 *)rsdp + 0x10);
		len -= 0x10;
	}
	return -1;

	have_it:
	ACPI_FOUND = 1;
	RSDT = (unsigned long)rsdt;
	VM_ADD_MEMORY_HOLE(RSDT, RSDT + __32LE2CPU(rsdp->length));
	for (i = 0; i < n_rsdt_entries(rsdt); i++) {
		const ACPI_TABLE *table = (const ACPI_TABLE *)rsdt->ptr[i];
		unsigned len = __32LE2CPU(table->length);
		VM_ADD_MEMORY_HOLE((unsigned long)table, (__u64)(unsigned long)table + len);
	}
	return 0;
}

static void FIND_ACPI(void)
{
	if (*(__u16 *)0x40e)
		if (!find_rsdp(*(__u16 *)0x40e << 4, 1024))
			goto got_rsdp;
	if (!find_rsdp(0xe0000, 0x20000))
		goto got_rsdp;
	return;

	got_rsdp:;
}

static __s8 PROXIMITY_DOMAIN_XLATE[MAX_NODES_BITS];

static void ACPI_PREINIT(void)
{
	const ACPI_MADT *madt;
	const ACPI_SRAT *srat;
	unsigned real_length;

	madt = ACPI_FIND_TABLE(ACPI_MADT_SIGNATURE, ACPI_MADT_LENGTH, &real_length);
	if (!__IS_ERR(madt)) {
		const ACPI_MADT_LOCAL_APIC *lapic;
		int procs;
		ACPI_FIND_RESET(madt->entries, (const __u8 *)madt + real_length, ACPI_MADT_TYPE_LOCAL_APIC, ACPI_MADT_LOCAL_APIC_LENGTH);
		procs = 0;
		while ((lapic = ACPI_FIND_ENTRY())) {
			if (__32LE2CPU(lapic->flags) & ACPI_MADT_LOCAL_APIC_FLAGS_ENABLED)
				procs++;
		}
#if __KERNEL_SUPPORT_SMP
		if (procs >= 2 && BOOTCFG->PARAM_SMP != PARAM_SMP_DISABLE) {
			FEATURE_ADD(FEATURE_SMP);
			if (procs >= CPU_ID_LIMIT) CPU_ID_LIMIT = procs;
		}
#endif
		ACPI_UNMAP(madt, real_length);
	}

	srat = ACPI_FIND_TABLE(ACPI_SRAT_SIGNATURE, ACPI_SRAT_LENGTH, &real_length);
	if (!__IS_ERR(srat)) {
		unsigned i, bits;

		__u32 proximity_domain_mask = 0;
		__u32 last_proximity_domain = 0;
		char valid_last_proximity_domain = 0;
		const ACPI_SRAT_MEMORY *mem;
		const ACPI_SRAT_PROCESSOR_LOCAL_APIC *lapic;
		const ACPI_SRAT_PROCESSOR_LOCAL_X2APIC *lx2apic;

		ACPI_FIND_RESET(srat->entries, (const __u8 *)srat + real_length, ACPI_SRAT_TYPE_MEMORY, ACPI_SRAT_MEMORY_LENGTH);
		while ((mem = ACPI_FIND_ENTRY())) {
			__u32 proximity_domain;
			__u64 end;
			if (!(__32LE2CPU(mem->flags) & ACPI_SRAT_MEMORY_FLAGS_ENABLED))
				continue;
			end = __64LE2CPU(mem->base_addr) + __64LE2CPU(mem->length);
			if (end > (__u64)1 << 32) {
				PREPARE_FOR_HIGHMEM = 1;
			}
			proximity_domain = acpi_srat_memory_proximity_domain(mem);
			if (valid_last_proximity_domain)
				proximity_domain_mask |= last_proximity_domain ^ proximity_domain;
			last_proximity_domain = proximity_domain;
			valid_last_proximity_domain = 1;
		}

		ACPI_FIND_RESET(srat->entries, (const __u8 *)srat + real_length, ACPI_SRAT_TYPE_PROCESSOR_LOCAL_APIC, ACPI_SRAT_PROCESSOR_LOCAL_APIC_LENGTH);
		while ((lapic = ACPI_FIND_ENTRY())) {
			__u32 proximity_domain;
			if (!(__32LE2CPU(lapic->flags) & ACPI_SRAT_PROCESSOR_LOCAL_APIC_FLAGS_ENABLED))
				continue;
			proximity_domain = acpi_srat_processor_local_apic_proximity_domain(lapic);
			if (valid_last_proximity_domain)
				proximity_domain_mask |= last_proximity_domain ^ proximity_domain;
			last_proximity_domain = proximity_domain;
			valid_last_proximity_domain = 1;
		}

		ACPI_FIND_RESET(srat->entries, (const __u8 *)srat + real_length, ACPI_SRAT_TYPE_PROCESSOR_LOCAL_X2APIC, ACPI_SRAT_PROCESSOR_LOCAL_X2APIC_LENGTH);
		while ((lx2apic = ACPI_FIND_ENTRY())) {
			__u32 proximity_domain;
			if (!(__32LE2CPU(lx2apic->flags) & ACPI_SRAT_PROCESSOR_LOCAL_X2APIC_FLAGS_ENABLED))
				continue;
			proximity_domain = acpi_srat_processor_local_x2apic_proximity_domain(lx2apic);
			if (valid_last_proximity_domain)
				proximity_domain_mask |= last_proximity_domain ^ proximity_domain;
			last_proximity_domain = proximity_domain;
			valid_last_proximity_domain = 1;
		}

		bits = 0;
		if (MAX_NODES_BITS > 0)
			for (i = 0; i < 32; i++)
				if (proximity_domain_mask & (1 << i)) {
					if (bits >= MAX_NODES_BITS) {
						memmove(PROXIMITY_DOMAIN_XLATE, PROXIMITY_DOMAIN_XLATE + 1, MAX_NODES_BITS - 1);
						bits = MAX_NODES_BITS - 1;
					}
					PROXIMITY_DOMAIN_XLATE[bits++] = i;
				}

		ACPI_UNMAP(srat, real_length);
	}
}

__node_id_t ACPI_PROXIMITY_DOMAIN_TO_NODE_ID(__u32 proximity_domain)
{
	unsigned i;
	__node_id_t node = 0;
	for (i = 0; i < MAX_NODES_BITS; i++) {
		__s8 s = PROXIMITY_DOMAIN_XLATE[i];
		if (s < 0) continue;
		node |= ((proximity_domain >> s) & 1) << i;
	}
	return node;
}

static void NODE_ID_INIT(void)
{
	NODE_ID_LIMIT = ACPI_PROXIMITY_DOMAIN_TO_NODE_ID(0xffffffff) + 1;
	if (NODE_ID_LIMIT <= 1) NODE_ID_LIMIT = 2;
	/* !!! FIXME: boot CPU doesn't have to be on node 0 */
}

void ACPI_DETECT(void)
{
	memset(PROXIMITY_DOMAIN_XLATE, -1, sizeof(PROXIMITY_DOMAIN_XLATE));
	FIND_ACPI();
	ACPI_PREINIT();
	NODE_ID_INIT();
}

