#ifndef __KERNEL_ACPITBL_H
#define __KERNEL_ACPITBL_H

#include <SYS/TYPES.H>

typedef struct {
	__u8 signature[8];
	__u8 checksum;
	__u8 oem_id[6];
	__u8 revision;
	__u32 rsdt;
	__u32 length;
	__u64 xsdt;
	__u8 extended_checksum;
	__u8 reserved[3];
} ACPI_RSDP;

#define ACPI_RSDP_SIGNATURE			"RSD PTR "

#define ACPI_RSDP_LENGTH_1			0x14


typedef struct {
	__u8 signature[4];
	__u32 length;
	__u8 revision;
	__u8 checksum;
	__u8 oem_id[6];
	__u8 oem_table_id[8];
	__u32 oem_revision;
	__u8 creator_id[4];
	__u32 creator_revision;
} ACPI_TABLE;

typedef struct {
	__u8 type;
	__u8 length;
} ACPI_ENTRY_HEADER;


/********
 * RSDT *
 ********/

typedef struct {
	ACPI_TABLE table;
	__u32 ptr[1];
} ACPI_RSDT;

#define ACPI_RSDT_SIGNATURE			"RSDT"
#define ACPI_RSDT_LENGTH			__offsetof(ACPI_RSDT, ptr)

/********
 * XSDT *
 ********/

typedef struct {
	ACPI_TABLE table;
	__u64 ptr[1];
} ACPI_XSDT;

#define ACPI_XSDT_SIGNATURE			"XSDT"
#define ACPI_XSDT_LENGTH			__offsetof(ACPI_XSDT, ptr)

/********
 * MADT *
 ********/

typedef struct {
	ACPI_TABLE table;
	__u32 local_apic_addr;
	__u32 flags;
	__u8 entries[1];
} ACPI_MADT;

#define ACPI_MADT_SIGNATURE			"APIC"
#define ACPI_MADT_LENGTH			__offsetof(ACPI_MADT, entries)

#define ACPI_MADT_FLAGS_PCAT_COMPAT		0x00000001

#define ACPI_MADT_TYPE_LOCAL_APIC		0x00
#define ACPI_MADT_TYPE_IO_APIC			0x01
#define ACPI_MADT_TYPE_INTERRUPT_SOURCE		0x02
#define ACPI_MADT_TYPE_NMI_SOURCE		0x03
#define ACPI_MADT_TYPE_LOCAL_APIC_NMI		0x04
#define ACPI_MADT_TYPE_LOCAL_APIC_ADDRESS	0x05
#define ACPI_MADT_TYPE_IO_SAPIC			0x06
#define ACPI_MADT_TYPE_LOCAL_SAPIC		0x07
#define ACPI_MADT_TYPE_PLATFORM_INTERRUPT_SOURCE 0x08
#define ACPI_MADT_TYPE_LOCAL_X2APIC		0x09
#define ACPI_MADT_TYPE_LOCAL_X2API_NMI		0x0a

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 acpi_processor_id;
	__u8 local_apic_id;
	__u32 flags;
} ACPI_MADT_LOCAL_APIC;

#define ACPI_MADT_LOCAL_APIC_LENGTH		sizeof(ACPI_MADT_LOCAL_APIC)

#define ACPI_MADT_LOCAL_APIC_FLAGS_ENABLED	0x00000001

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 io_apic_id;
	__u8 reserved;
	__u32 io_apic_addr;
	__u32 global_system_interrupt_base;
} ACPI_MADT_IO_APIC;

#define ACPI_MADT_IO_APIC_LENGTH		sizeof(ACPI_MADT_IO_APIC)

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 bus;
	__u8 source;
	__u32 global_system_interrupt;
	__u16 flags; /* MP_INTERRUPT_FLAGS_... */
} ACPI_INTERRUPT_SOURCE;

#define ACPI_INTERRUPT_SOURCE_LENGTH		0x0a

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 bus;
	__u8 source;
	__u16 flags; /* MP_INTERRUPT_FLAGS_... */
	__u32 global_system_interrupt;
} ACPI_NMI_SOURCE;

#define ACPI_NMI_SOURCE_LENGTH			sizeof(ACPI_NMI_SOURCE)

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 acpi_processor_id;
	__u8 flags0; /* MP_INTERRUPT_FLAGS_... */
	__u8 flags1;
	__u8 lint;
} ACPI_MADT_LOCAL_APIC_NMI;

#define ACPI_MADT_LOCAL_APIC_NMI_LENGTH		0x06

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u16 reserved;
	__u32 local_apic_addr_lo;
	__u32 local_apic_addr_hi;
} ACPI_MADT_LOCAL_APIC_ADDRESS;

#define ACPI_MADT_LOCAL_APIC_ADDRESS_LENGTH 	sizeof(ACPI_MADT_LOCAL_APIC_ADDRESS)

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 io_apic_id;
	__u8 reserved;
	__u32 global_system_interrupt_base;
	__u64 io_sapic_addr;
} ACPI_MADT_IO_SAPIC;

#define ACPI_MADT_IO_SAPIC_LENGTH		sizeof(ACPI_MADT_IO_SAPIC)

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 acpi_processor_id;
	__u8 local_sapic_id;
	__u8 local_sapic_eid;
	__u8 reserved[3];
	__u32 flags;
	__u32 acpi_processor_uid_value;
	__u8 acpi_processor_uid_string[1];
} ACPI_MADT_LOCAL_SAPIC;

#define ACPI_MADT_LOCAL_SAPIC_LENGTH		0x11

#define ACPI_MADT_LOCAL_SAPIC_FLAGS_ENABLED	0x00000001

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u16 flags; /* MP_INTERRUPT_FLAGS_... */
	__u8 type;
	__u8 processor_id;
	__u8 processor_eid;
	__u8 io_sapic_vector;
	__u32 global_system_interrupt;
	__u32 platform_flags;
} ACPI_MADT_PLATFORM_INTERRUPT_SOURCE;

#define ACPI_MADT_PLATFORM_INTERRUPT_SOURCE_LENGTH sizeof(ACPI_MADT_PLATFORM_INTERRUPT_SOURCE)

#define ACPI_MADT_PLATFORM_INTERRUPT_SOURCE_TYPE_PMI		0x01
#define ACPI_MADT_PLATFORM_INTERRUPT_SOURCE_TYPE_NMI		0x02
#define ACPI_MADT_PLATFORM_INTERRUPT_SOURCE_TYPE_CORRECTED	0x03

#define ACPI_MADT_PLATFORM_INTERRUPT_SOURCE_FLAGS_CPEI_OVERRIDE	0x00000001

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u16 reserved;
	__u32 loacl_x2apic_id;
	__u32 flags;
	__u32 acpi_processor_uid;
} ACPI_MADT_LOCAL_X2APIC;

#define ACPI_MADT_LOCAL_X2APIC_LENGTH		sizeof(ACPI_MADT_LOCAL_X2APIC)

#define ACPI_MADT_LOCAL_X2APIC_FLAGS_ENABLED	0x00000001

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u16 flags; /* MP_INTERRUPT_FLAGS_... */
	__u32 acpi_processor_uid;
	__u8 lint;
	__u8 reserved[3];
} ACPI_MADT_LOCAL_X2API_NMI;

#define ACPI_MADT_LOCAL_X2API_NMI_LENGTH	sizeof(ACPI_MADT_LOCAL_X2API_NMI)

/********
 * SRAT *
 ********/

typedef struct {
	ACPI_TABLE table;
	__u32 reserved[3];
	__u8 entries[1];
} ACPI_SRAT;

#define ACPI_SRAT_SIGNATURE			"SRAT"
#define ACPI_SRAT_LENGTH			__offsetof(ACPI_SRAT, entries)

#define ACPI_SRAT_TYPE_PROCESSOR_LOCAL_APIC	0x00
#define ACPI_SRAT_TYPE_MEMORY			0x01
#define ACPI_SRAT_TYPE_PROCESSOR_LOCAL_X2APIC	0x02

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 proximity_domain0;
	__u8 local_apic_id;
	__u32 flags;
	__u8 local_sapic_eid;
	__u8 proximity_domain1[3];
	__u32 clock_domain;
} ACPI_SRAT_PROCESSOR_LOCAL_APIC;

#define ACPI_SRAT_PROCESSOR_LOCAL_APIC_LENGTH	sizeof(ACPI_SRAT_PROCESSOR_LOCAL_APIC)

#define ACPI_SRAT_PROCESSOR_LOCAL_APIC_FLAGS_ENABLED 0x00000001

static __finline__ __u32 acpi_srat_processor_local_apic_proximity_domain(const ACPI_SRAT_PROCESSOR_LOCAL_APIC *lapic)
{
	return
		(lapic->proximity_domain0) |
		(lapic->proximity_domain1[0] << 8) |
		(lapic->proximity_domain1[1] << 16) |
		(lapic->proximity_domain1[2] << 24);
}

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 proximity_domain[4];
	__u16 reserved;
	__u64 base_addr;
	__u64 length;
	__u32 reserved1;
	__u32 flags;
	__u64 reserved2;
} ACPI_SRAT_MEMORY;

#define ACPI_SRAT_MEMORY_LENGTH			sizeof(ACPI_SRAT_MEMORY)

#define ACPI_SRAT_MEMORY_FLAGS_ENABLED		0x00000001
#define ACPI_SRAT_MEMORY_FLAGS_HOT_PLUGGABLE	0x00000002
#define ACPI_SRAT_MEMORY_FLAGS_NONVOLATILE	0x00000004

static __finline__ __u32 acpi_srat_memory_proximity_domain(const ACPI_SRAT_MEMORY *mem)
{
	return
		(mem->proximity_domain[0]) |
		(mem->proximity_domain[1] << 8) |
		(mem->proximity_domain[2] << 16) |
		(mem->proximity_domain[3] << 24);
}

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u16 reserved1;
	__u8 proximity_domain[4];
	__u32 local_x2apic_id;
	__u32 flags;
	__u32 clock_domain;
	__u32 reserved2;
} ACPI_SRAT_PROCESSOR_LOCAL_X2APIC;

#define ACPI_SRAT_PROCESSOR_LOCAL_X2APIC_LENGTH	sizeof(ACPI_SRAT_PROCESSOR_LOCAL_X2APIC)

#define ACPI_SRAT_PROCESSOR_LOCAL_X2APIC_FLAGS_ENABLED 0x00000001

static __finline__ __u32 acpi_srat_processor_local_x2apic_proximity_domain(const ACPI_SRAT_PROCESSOR_LOCAL_X2APIC *lx2apic)
{
	return
		(lx2apic->proximity_domain[0]) |
		(lx2apic->proximity_domain[1] << 8) |
		(lx2apic->proximity_domain[2] << 16) |
		(lx2apic->proximity_domain[3] << 24);
}

/********
 * SLIT *
 ********/

typedef struct {
	ACPI_TABLE table;
	__u32 num_localities;
	__u32 reserved;
	__u8 entries[1];
} ACPI_SLIT;

#define ACPI_SLIT_SIGNATURE			"SLIT"
#define ACPI_SLIT_LENGTH			__offsetof(ACPI_SLIT, entries)

/********
 * MSCT *
 ********/

typedef struct {
	ACPI_TABLE table;
	__u32 proximity_domain_offset;
	__u32 max_proximity_domains;
	__u32 max_clock_domains;
	__u64 max_physical_addr;
} ACPI_MSCT;

#define ACPI_MSCT_LENGTH			sizeof(ACPI_MSCT)

typedef struct {
	ACPI_ENTRY_HEADER header;
	__u8 proximity_domain_low[4];
	__u8 proximity_domain_high[4];
	__u8 max_processors[4];
	__u8 max_memory[8];
} ACPI_MSCT_PROXIMITY_DOMAIN;

#define ACPI_MSCT_TYPE_PROXIMITY_DOMAIN_LENGTH	1
#define ACPI_MSCT_PROXIMITY_DOMAIN_LENGTH	22

#endif
