#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/DL.H>
#include <KERNEL/LINK.H>
#include <SPAD/LINK.H>
#include <ARCH/CPU.H>
#include <ARCH/SETUP.H>
#include <ARCH/IO.H>
#include <SPAD/VM.H>
#include <KERNEL/FEATURE.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/ASM.H>
#include <KERNEL/TIME.H>
#include <KERNEL/IRQARCH.H>
#include <KERNEL/IRQ.H>
#include <KERNEL/SLAB.H>
#include <KERNEL/VM.H>
#include <KERNEL/SMP/SHARE.H>
#include <KERNEL/SMP/IPI.H>
#include <KERNEL/SMP/MATRIX.H>
#include <KERNEL/FSDLDR.H>
#include <KERNEL/BOOTCFG.H>
#include <KERNEL/MPS.H>
#include <KERNEL/ACPI.H>

/*
 * Kernel boot is tricky. Loader loads image to 0x100000 and jumps to
 * KERNEL$BOOT. KERNEL$BOOT is not relocated and so it can't touch global
 * variables, use switch statement etc. We relocate kernel for physical mapping,
 * then create paging structures, then relocate kernel again for virtual mapping
 * and finaly enable paging.
 */

static void RELOC_FAIL(void)
{
	/* Very cool - writing a message without touching any absolute addresses because the code is not relocated */
	unsigned short int *base = (void *)(*(char *)0x449 == 7 ? 0xb00a0 : 0xb80a0);
	base[0] = 'L' | 0x7000;
	base[1] = 'I' | 0x7000;
	base[2] = 'N' | 0x7000;
	base[3] = 'K' | 0x7000;
	base[4] = ' ' | 0x7000;
	base[5] = 'F' | 0x7000;
	base[6] = 'A' | 0x7000;
	base[7] = 'I' | 0x7000;
	base[8] = 'L' | 0x7000;
	base[9] = 'E' | 0x7000;
	base[10] = 'D' | 0x7000;
	while (1) __asm__ volatile ("HLT");
}

void BOOT_FAIL(const char *s)
{
	unsigned short int *base = (void *)(*(char *)0x449 == 7 ? 0xb00a0 : 0xb80a0);
	while (*s) *base++ = (unsigned char)*s++ | 0x7000;
	while (1) __asm__ volatile ("HLT");
}

void FEATURE_ADD_FAILURE(void)
{
	BOOT_FAIL("FEATURE BOX OVERFLOW");
}

char PREPARE_FOR_HIGHMEM;

struct cache_id {
	__u8 code;
	char type;
	unsigned size;
	unsigned assoc;
	unsigned line;
};

#define CID_END		0
#define CID_ITLB	1
#define CID_DTLB	2
#define CID_L1I		3
#define CID_L1D		4
#define CID_T		5
#define CID_L2		6
#define CID_L3		7
#define CID_PREFETCH	8

static const struct cache_id cid_types[] = {
	0x01,	CID_ITLB,	32,	  4,	4096,
	0x02,	CID_ITLB,	2,	  2,	4194304,
	0x03,	CID_DTLB,	64,	  4,	4096,
	0x04,	CID_DTLB,	8,	  4,	4194304,
	0x06,	CID_L1I,	8192,	  4,	32,
	0x08,	CID_L1I,	16384,	  4,	32,
	0x0A,	CID_L1D,	8192,	  2,	32,
	0x0C,	CID_L1D,	16384,	  4,	32,
	0x22,	CID_L3,		524288,	  4,	64,
	0x23,	CID_L3,		1048576,  8,	64,
	0x25,	CID_L3,		2097152,  8,	64,
	0x29,	CID_L3,		4194304,  8,	64,
	0x2C,	CID_L1D,	32768,	  8,	64,
	0x30,	CID_L1I,	32768,	  8,	64,
	0x39,	CID_L2,		131072,	  4,	64,
	0x3B,	CID_L2,		131072,	  2,	64,
	0x3C,	CID_L2,		262144,	  4,	64,
	0x41,	CID_L2,		131072,	  4,	32,
	0x42,	CID_L2,		262144,	  4,	32,
	0x43,	CID_L2,		524288,	  4,	32,
	0x44,	CID_L2,		1048576,  4,	32,
	0x45,	CID_L2,		2097152,  4,	32,
	0x50,	CID_ITLB,	64,	  64,	0,
	0x51,	CID_ITLB,	128,	  128,	0,
	0x52,	CID_ITLB,	256,	  256,	0,
	0x5B,	CID_DTLB,	64,	  64,	0,
	0x5C,	CID_DTLB,	128,	  128,	0,
	0x5D,	CID_DTLB,	256,	  256,	0,
	0x60,	CID_L1D,	16384,	  8,	64,
	0x66,	CID_L1D,	8192,	  4,	64,
	0x67,	CID_L1D,	16384,	  4,	64,
	0x68,	CID_L1D,	32768,	  4,	64,
	0x70,	CID_T,		12288,	  8,	0,
	0x71,	CID_T,		16384,	  8,	0,
	0x72,	CID_T,		32768,	  8,	0,
	0x78,	CID_L2,		1048576,  4,	64,
	0x79,	CID_L2,		131072,	  8,	64,
	0x7A,	CID_L2,		262144,	  8,	64,
	0x7B,	CID_L2,		524288,	  8,	64,
	0x7C,	CID_L2,		1048576,  8,	64,
	0x7D,	CID_L2,		2097152,  8,	64,
	0x7F,	CID_L2,		524288,	  2,	64,
	0x82,	CID_L2,		262144,	  8,	32,
	0x83,	CID_L2,		524288,	  8,	32,
	0x84,	CID_L2,		1048576,  8,	32,
	0x85,	CID_L2,		2097152,  8,	32,
	0x86,	CID_L2,		524288,	  4,	64,
	0x87,	CID_L2,		1048576,  8,	64,
	0xB0,	CID_ITLB,	128,	  4,	4096,
	0xB3,	CID_DTLB,	128,	  4,	4096,
	0xF0,	CID_PREFETCH,	0,	  0,	64,
	0xF1,	CID_PREFETCH,	0,	  0,	128,
	0,	CID_END,	0,	  0,	0
};

static void CHECK_CPU_FEATURES(void)
{
	__u32 f;
	__u32 cpuid = 0;
	__u32 cpuid2 = 0;
	__u32 amd_cpuid = 0;
	__u32 amd_cpuid2 = 0;
	__u32 maxc = 0, maxe = 0;
	unsigned family = 4;
	unsigned model = 0;
	unsigned stepping = 0;
	char vendor[13];
	__asm__ volatile ("					;\
		PUSHFL						;\
		POPL	%%EAX					;\
		XORL	%1, %%EAX				;\
		PUSHL	%%EAX					;\
		POPFL						;\
		PUSHFL						;\
		POPL	%%ECX					;\
		XORL	%%ECX, %%EAX				;\
		ANDL	%1, %%EAX				;\
	":"=a"(f):"i"(EFLAGS_ID):"cx","cc");
	if (!f) {
		DO_CPUID0(&maxc, vendor);
		if (maxc) {
			DO_CPUID1(&family, &model, &stepping, &cpuid, &cpuid2);
		}
		DO_CPUID(0x80000000, &maxe, NULL, NULL, NULL);
		if ((maxe & 0xffff0000) == 0x80000000 && maxe > 0x80000000) {
			DO_CPUID(0x80000001, NULL, NULL, &amd_cpuid2, &amd_cpuid);
			FEATURE_ADD(FEATURE_CPUID_EXT);
		} else maxe = 0;
	}
	if (!strcmp(vendor, "AuthenticAMD")) {
		if (family == 6 && model >= 6 && model <= 10 && maxc && !(cpuid & CPU_FEATURE_SSE)) {
			__asm__ volatile ("			;\
				RDMSR				;\
				ANDL	$~0x00008000, %%EAX	;\
				WRMSR				;\
			"::"c"(MSR_K7_HWCR):"ax","dx","cc");
			DO_CPUID1(&family, &model, &stepping, &cpuid, &cpuid2);
		}
	}
	if (maxc && !(cpuid & CPU_FEATURE_FPU))
		BOOT_FAIL("CPU DOES NOT HAVE FPU");

	/* General instructions */

	if (maxc)
		FEATURE_ADD(FEATURE_CPUID);
	if (maxe)
		FEATURE_ADD(FEATURE_CPUID_EXT);
	if (cpuid & CPU_FEATURE_TSC)
		FEATURE_ADD(FEATURE_TSC);
	if (cpuid & CPU_FEATURE_CX8)
		FEATURE_ADD(FEATURE_CMPXCHG8B);
	if (cpuid & CPU_FEATURE_CMOV)
		FEATURE_ADD(FEATURE_CMOV);
	if (family >= 6 && (!strcmp(vendor, "GenuineIntel") || !strcmp(vendor, "AuthenticAMD")))
		FEATURE_ADD(FEATURE_NOP);
	if (cpuid2 & CPU_FEATURE2_POPCNT)
		FEATURE_ADD(FEATURE_POPCNT);
	if (amd_cpuid2 & CPU_FEATURE2_AMD_LZCNT)
		FEATURE_ADD(FEATURE_LZCNT);
	if (cpuid2 & CPU_FEATURE2_MOVBE)
		FEATURE_ADD(FEATURE_MOVBE);

	/* Prefetch */

	if (amd_cpuid & CPU_FEATURE_AMD_3DNOW || amd_cpuid2 & CPU_FEATURE2_AMD_PREFETCH)
		FEATURE_ADD(FEATURE_3DNOW_PREFETCH);
	if (cpuid & CPU_FEATURE_SSE)
		FEATURE_ADD(FEATURE_SSE_PREFETCH);
	if (!strcmp(vendor, "AuthenticAMD") && (family == 6 || (family == 15 && model <= 5 && stepping <= 8))) {
		/* Prefetch is botched on Athlon */
		FEATURE_DEL(FEATURE_SSE_PREFETCH);
		FEATURE_DEL(FEATURE_3DNOW_PREFETCH);
	}

	/* Monitor */
	if (cpuid2 & CPU_FEATURE2_MONITOR)
		FEATURE_ADD(FEATURE_MONITOR);

	/* FPU */

	FEATURE_ADD(FEATURE_FPU);
	if (cpuid & CPU_FEATURE_CMOV && KERNEL$FEATURE_TEST(FEATURE_FPU))
		FEATURE_ADD(FEATURE_FPU_FCMOV);
	if (family >= 6 && (!strcmp(vendor, "GenuineIntel") || !strcmp(vendor, "AuthenticAMD")))
		FEATURE_ADD(FEATURE_FPU_FCOMI);
	if (cpuid & CPU_FEATURE_FXSR) {
		FEATURE_ADD(FEATURE_FPU_FXSR);
		if (!strcmp(vendor, "GenuineIntel"))
			FEATURE_ADD(FEATURE_FPU_FXSR_NO_LEAK);
	}

	/* MMX/3DNow */

	if (cpuid & CPU_FEATURE_MMX)
		FEATURE_ADD(FEATURE_MMX);
	if (amd_cpuid & CPU_FEATURE_AMD_MMX_EXT)
		FEATURE_ADD(FEATURE_MMX_EXT);
	if (amd_cpuid & CPU_FEATURE_AMD_3DNOW)
		FEATURE_ADD(FEATURE_3DNOW);
	if (amd_cpuid & CPU_FEATURE_AMD_3DNOW_EXT)
		FEATURE_ADD(FEATURE_3DNOW_EXT);

	/* SSE */

	if (cpuid & CPU_FEATURE_SSE)
		FEATURE_ADD(FEATURE_SSE);
	if (cpuid & CPU_FEATURE_SSE2)
		FEATURE_ADD(FEATURE_SSE2);
	if (cpuid2 & CPU_FEATURE2_SSE3)
		FEATURE_ADD(FEATURE_SSE3);
	if (cpuid2 & CPU_FEATURE2_SSSE3)
		FEATURE_ADD(FEATURE_SSSE3);
	if (amd_cpuid2 & CPU_FEATURE2_AMD_MISALIGN_SSE)
		FEATURE_ADD(FEATURE_SSE_MISALIGN);
	if (amd_cpuid2 & CPU_FEATURE2_AMD_SSE4A)
		FEATURE_ADD(FEATURE_SSE4A);
	if (cpuid2 & CPU_FEATURE2_SSE41)
		FEATURE_ADD(FEATURE_SSE41);
	if (cpuid2 & CPU_FEATURE2_SSE42)
		FEATURE_ADD(FEATURE_SSE42);

	/* System features */

	if (cpuid & CPU_FEATURE_MSR)
		FEATURE_ADD(_FEATURE_MSR);
	if (cpuid & CPU_FEATURE_PSE && BOOTCFG->PARAM_PSE != PARAM_PSE_DISABLE) {
		FEATURE_ADD(_FEATURE_4M_PAGES);
		if (family >= 6)
			FEATURE_ADD(_FEATURE_INVLPG_4M_PAGES);
	}
	if (cpuid & CPU_FEATURE_PGE && BOOTCFG->PARAM_PGE != PARAM_PGE_DISABLE)
		FEATURE_ADD(_FEATURE_GLOBAL_PAGES);
	if (!strcmp(vendor, "AuthenticAMD") && family == 5 && model == 0)
		FEATURE_ADD(_FEATURE_GLOBAL_PAGES);
	/* AMD64 has a bug when referring to a page with global and non-global
	   bit for the same address.
	   I create global and non-global pages in the same arena dynamically
	   during kernel operation, so such mixing may happen and trigger
	   the bug.
	   Sadly, AMD has no effort to fix it, it stays in 3 architecture
	   revisions.
	   */
	if (!strcmp(vendor, "AuthenticAMD") && family >= 15 && BOOTCFG->PARAM_PGE != PARAM_PGE_ENABLE)
		FEATURE_DEL(_FEATURE_GLOBAL_PAGES);
	
	if (cpuid & CPU_FEATURE_FGPAT)
		FEATURE_ADD(_FEATURE_PAT);
	if (cpuid & CPU_FEATURE_SS)
		FEATURE_ADD(_FEATURE_SELF_SNOOP);

	/* Syscalls */

/* If the CPU has both AMD SYSCALL and Intel SYSENTER instructions (new AMD CPUs
   have), prefer SYSCALL --- it is slightly faster. Test on Athlon64:
 	SYSCALL: always 148 ticks
	SYSENTER: most time 175 ticks, sometimes 142 or 159 ticks
*/

	if (cpuid & CPU_FEATURE_SEP)
		FEATURE_ADD(_FEATURE_USE_SYSENTER);
	if (!strcmp(vendor, "GenuineIntel") && family == 6 && model < 3 && stepping < 3)
		FEATURE_DEL(_FEATURE_USE_SYSENTER);
	if (amd_cpuid & CPU_FEATURE_AMD_SYSCALL)
		FEATURE_ADD(_FEATURE_USE_SYSCALL);

	switch (BOOTCFG->PARAM_SYSCALL) {
		case PARAM_SYSCALL_INT:
			FEATURE_DEL(_FEATURE_USE_SYSENTER);
			FEATURE_DEL(_FEATURE_USE_SYSCALL);
			break;
		case PARAM_SYSCALL_AMD:
			FEATURE_DEL(_FEATURE_USE_SYSENTER);
			break;
		case PARAM_SYSCALL_INTEL:
			FEATURE_DEL(_FEATURE_USE_SYSCALL);
			break;
		default:
			if (KERNEL$FEATURE_TEST(_FEATURE_USE_SYSCALL))
				FEATURE_DEL(_FEATURE_USE_SYSENTER);
			break;
	}

	if (!strcmp(vendor, "GenuineIntel") && family == 5 && model == 2 && (stepping < 6 || stepping == 11))
		FEATURE_ADD(_FEATURE_LOCAL_APIC_LOCK_WRITE);

	/* Big memory support */

	if (cpuid & CPU_FEATURE_PAE && KERNEL$FEATURE_TEST(_FEATURE_4M_PAGES) && BOOTCFG->PARAM_PAE != PARAM_PAE_DISABLE) {
		if (BOOTCFG->PARAM_PAE != PARAM_PAE_ENABLE || PREPARE_FOR_HIGHMEM) {
			FEATURE_ADD(_FEATURE_USE_PAE);
			if (PREPARE_FOR_HIGHMEM) {
#if KERNEL_BOUNCE_COMPILE
				FEATURE_ADD(_FEATURE_USE_BOUNCE);
#endif
			}
		}
	}
#if __DEBUG_BOUNCE && KERNEL_BOUNCE_COMPILE
	FEATURE_ADD(_FEATURE_USE_BOUNCE);
#endif
	if (KERNEL$FEATURE_TEST(FEATURE_FPU_FXSR) ||
	    KERNEL$FEATURE_TEST(FEATURE_SSE) ||
	    KERNEL$FEATURE_TEST(_FEATURE_4M_PAGES) ||
	    KERNEL$FEATURE_TEST(_FEATURE_GLOBAL_PAGES) ||
	    KERNEL$FEATURE_TEST(_FEATURE_PAT) ||
	    KERNEL$FEATURE_TEST(_FEATURE_USE_PAE))
		FEATURE_ADD(FEATURE_CR4);
}

static __finline__ __u32 READ_CR4(void)
{
	__u32 reg;
	__asm__ volatile ("MOVL %%CR4, %0":"=r"(reg));
	return reg;
}

static __finline__ void WRITE_CR4(__u32 val)
{
	__asm__ volatile ("MOVL %0, %%CR4"::"r"(val));
}

static void PREPARE_CPU_REGISTERS(void)
{
	if (KERNEL$FEATURE_TEST(_FEATURE_4M_PAGES))
		WRITE_CR4(READ_CR4() | CR4_PSE);
	if (KERNEL$FEATURE_TEST(_FEATURE_GLOBAL_PAGES))
		WRITE_CR4(READ_CR4() | CR4_PGE);
	if (KERNEL$FEATURE_TEST(_FEATURE_USE_PAE))
		WRITE_CR4(READ_CR4() | CR4_PAE);
	if (KERNEL$FEATURE_TEST(_FEATURE_PAT))
		KERNEL$WRITE_MSR(IA32_CR_PAT, IA32_MEMORY_TYPE_WB | (IA32_MEMORY_TYPE_WT << 8) | (IA32_MEMORY_TYPE_UC_MINUS << 16) | (IA32_MEMORY_TYPE_UC << 24) | ((__u64)IA32_MEMORY_TYPE_WB << 32) | ((__u64)IA32_MEMORY_TYPE_WT << 40) | ((__u64)IA32_MEMORY_TYPE_WC << 48) | ((__u64)IA32_MEMORY_TYPE_WP << 56));
	if (KERNEL$FEATURE_TEST(FEATURE_FPU_FXSR))
		WRITE_CR4(READ_CR4() | CR4_OSFXSR);
	if (KERNEL$FEATURE_TEST(FEATURE_SSE))
		WRITE_CR4(READ_CR4() | CR4_OSXMMEXCPT);
}

/*static void debug_val(int line, unsigned long val)
{
	unsigned short int *base = (void *)((*(char *)0x449 == 7 ? 0xb0000 : 0xb8000) + line * 0xa0);
	int i;
	for (i = 28; i >= 0; i -= 4) {
		unsigned n = (val >> i) & 0x0f;
		if (n >= 10) n += 7;
		*base++ = 0x7030 + n;
	}
}*/

static unsigned long FSD;
static unsigned long TOP_OF_CODE;
static unsigned long TOP_OF_KERNEL;
static unsigned long TOP_OF_DATA;

void KERNEL$BOOT(struct link_header *l, unsigned long fsd, unsigned long top_of_heap, int highmem);
__NOINLINE_ATTR__ static void KERNEL_BOOT_RELOCATED(struct link_header *l, unsigned long fsd, unsigned long top_of_heap, int highmem);

void KERNEL$BOOT(struct link_header *l, unsigned long fsd, unsigned long top_of_heap, int highmem)
{
	/* not relocated here, hope that the compiler won't generate global references */
	struct section *secs = (struct section *)((char *)l + l->sections);
	secs[0].ptr = (__f_off)l + secs[0].offset;
	secs[1].ptr = (__f_off)l + secs[1].offset;
	secs[2].ptr = (__f_off)0x400000;
	secs[3].ptr = (__f_off)0x400000 + (secs[3].offset - secs[2].offset);
	secs[4].ptr = (__f_off)0x400000 + (secs[4].offset - secs[2].offset);
	memmove((char *)secs[2].ptr, (char *)l + secs[2].offset, secs[4].ptr - secs[2].ptr);
	memset((char *)secs[4].ptr, 0, secs[4].len);
	if (LINK_RELOC(l, 0))
		RELOC_FAIL();
	KERNEL_BOOT_RELOCATED(l, fsd, top_of_heap, highmem);
}

__NOINLINE_ATTR__ static void KERNEL_BOOT_RELOCATED(struct link_header *l, unsigned long fsd, unsigned long top_of_heap, int highmem)
{
	void *pgtbl;
	struct section *secs = (struct section *)((char *)l + l->sections);

	PREPARE_FOR_HIGHMEM = highmem;

	memmove((char *)l + secs[0].len + secs[1].len, (void *)fsd, top_of_heap - fsd);
	/* and here we are safe... */

	FSD = (long)l + secs[0].len + secs[1].len;
	TOP_OF_CODE = FSD + top_of_heap - fsd;
	TOP_OF_CODE = (TOP_OF_CODE + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	TOP_OF_KERNEL = FSD;
	TOP_OF_KERNEL = (TOP_OF_KERNEL + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	TOP_OF_DATA = 0x400000 + secs[2].len + secs[3].len + secs[4].len;
	TOP_OF_DATA = (TOP_OF_DATA + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);

	FSD_SET_ENTRY((void *)FSD);
	LOAD_CONFIG_KRN();

	MPS_DETECT();
	ACPI_DETECT();

	CHECK_CPU_FEATURES();

	pgtbl = VM_BOOT_INIT();

	PREPARE_CPU_REGISTERS();

	if (LINK_RELOC(l, (__f_off)KMAP_PHYS_2_VIRT(0)))
		RELOC_FAIL();

	/* unsafe again ... */
	VM_BOOT_ENABLE_PAGING(pgtbl);
	/* VM_BOOT_ENABLE_PAGING never returns because it switched stack ... */
}

static void KERNEL_FEATURE_FIXUP(void *code);
void INIT_KERNEL(void);
void KERNEL_BOOT_CONTINUE(void);

void KERNEL_BOOT_CONTINUE(void)
{
	/*__debug_printf("features: %08x\n", KERNEL$CPU_FEATURES);*/
	/*__debug_printf("FSD: %08lx, DATA: %08lx, CODE: %08lx\n", FSD, TOP_OF_DATA, TOP_OF_CODE);*/

	__DL_INIT(KMAP_PHYS_2_VIRT(0x100000), KMAP_PHYS_2_VIRT(TOP_OF_CODE), KMAP_PHYS_2_VIRT(0x400000), KMAP_PHYS_2_VIRT(TOP_OF_DATA));
	KERNEL_FEATURE_FIXUP(KMAP_PHYS_2_VIRT(0x100000));
	VM_BOOT_GET_MEM(0x100000, TOP_OF_KERNEL, TOP_OF_CODE, 0x400000, TOP_OF_DATA, 0x800000);
	SLAB_ZERO_PREINIT();
	SLAB_INIT();
	SLAB_INIT_MALLOC_ARENA();
	FSD_SET_ENTRY(KMAP_PHYS_2_VIRT(FSD));
	MPS_INIT();
	if (APIC_IRQ_INIT())
		I8259_IRQ_INIT();
	IRQ_INIT();
	IPI_INIT();
	TIMER_INIT();
	KERNEL_FEATURE_FIXUP(KMAP_PHYS_2_VIRT(0x100000));	/* fixup again so that fine ticks are changed */
	MACHINE_INIT();
	INIT_KERNEL();
}

static void KERNEL_FEATURE_FIXUP(void *code)
{
	__f_off l_l, l_l_2;
	if (!LINK_GET_SYMBOL(code, "DLL$FIXUP", SYM_SPECIAL, &l_l) &&
	    !LINK_GET_SYMBOL(code, "DLL$FIXUP_END", SYM_SPECIAL, &l_l_2)) {
		if (FEATURE_FIXUP((struct feature_fixup *)l_l, (struct feature_fixup *)l_l_2))
			KERNEL$SUICIDE("KERNEL_FEATURE_FIXUP: FEATURE FIXUP FAILED");
	}

}

__cpu_id_t CPU_ID_LIMIT = 1;
__node_id_t NODE_ID = 0;
__node_id_t NODE_ID_LIMIT = 2;

int FEATURE_FIND_VARIABLE(feature_t feature, __f_off *start, __f_off *end)
{
	switch (feature) {
		case _FEATURE_VALUE_FINE_TICKS_SHIFT:
			*start = (__f_off)&FINE_TICKS_SHIFT;
			*end = *start + sizeof(FINE_TICKS_SHIFT);
			return 0;
		case FEATURE_VALUE_CPU_ID:
			*start = (__f_off)&CPU_ID;
			*end = *start + sizeof(CPU_ID);
			return 0;
		case FEATURE_VALUE_CPU_ID_LIMIT:
			*start = (__f_off)&CPU_ID_LIMIT;
			*end = *start + sizeof(__cpu_id_t);
			return 0;
		case FEATURE_VALUE_NODE_ID:
			*start = (__f_off)&NODE_ID;
			*end = *start + sizeof(NODE_ID);
			return 0;
		case FEATURE_VALUE_NODE_ID_LIMIT:
			*start = (__f_off)&NODE_ID_LIMIT;
			*end = *start + sizeof(__node_id_t);
			return 0;
		default:
			return -ENOENT;
	}
}

#if __KERNEL_SUPPORT_SMP

void KERNEL_FIXUP_SELF_SMP(void *code);
void KERNEL_FIXUP_SELF_SMP(void *code)
{
	PREPARE_FOR_HIGHMEM = *(char *)GET_KERNEL_VAR(&PREPARE_FOR_HIGHMEM, 0);

	CHECK_CPU_FEATURES();
	FEATURE_ADD(FEATURE_SMP);

	KERNEL_FEATURE_FIXUP(code);
}

void KERNEL_BOOT_SMP(void);
void KERNEL_BOOT_SMP(void)
{
	APIC_INIT_AP2();
	FEATURE_SMPSYNC();
	VM_BOOT_INIT_AP();
	IPI_INIT();
	MATRIX_INIT_AP();
	KERNEL$EI();
}

#endif
