#include <SPAD/LIBC.H>
#include <ARCH/IO.H>
#include <ARCH/BITOPS.H>
#include <SPAD/VM.H>
#include <SPAD/CONSOLE.H>
#include <STRING.H>
#include <SYS/TYPES.H>
#include <SPAD/SYSLOG.H>
#include <KERNEL/SMP/SHARE.H>
#include <SPAD/SPINLOCK.H>
#include <KERNEL/ASM.H>
#include <KERNEL/IRQARCH.H>

#include <KERNEL/CONSOLE.H>

/*
#define __BOCHS__
*/

#define CONSOLE_BUFFER_SIZE	16384

#define MAX_COPY_BYTES	64
#define MAX_LOCK_BYTES	16

struct console {
	__u64 buffer_ptr;
	__u16 *addr;
	io_t port;
	unsigned columns;
	unsigned lines;
	unsigned x;
	unsigned y;
	unsigned char gfx;
	SPINLOCK lock;
	__cpu_id_t owner;
	int recurse;
	void (* volatile restore_videomode)(void);
	char buffer[CONSOLE_BUFFER_SIZE];
};

static struct console main_console;
static struct console *console = &main_console;

static KERNEL_SHARED console_share = { "CONSOLE", &main_console };

static IO_RANGE iorange;

int KERNEL_IN_MCE;

#define DECL_LOCK						\
	int intstate = KERNEL$SAVE_INT_STATE();

#define LOCK							\
do {								\
	KERNEL$DI();						\
	if (__likely(!console->recurse) || console->owner != VAL_CPU_ID) {\
		KERNEL$SPINLOCK_LOCK(&console->lock);		\
		console->owner = VAL_CPU_ID;			\
		if (__DEBUG >= 1 && __unlikely(console->recurse))\
			KERNEL$SUICIDE("LOCK: RECURSE == %d", console->recurse);								\
	}							\
	console->recurse++;					\
} while (0)

#define UNLOCK							\
do {								\
	if (__DEBUG >= 1 && __unlikely(!console->recurse))	\
		KERNEL$SUICIDE("UNLOCK: CONSOLE NOT LOCKED");	\
	if (!--console->recurse)				\
		KERNEL$SPINLOCK_UNLOCK(&console->lock);		\
	KERNEL$RESTORE_INT_STATE(intstate);			\
} while (0)

#define TMP_UNLOCK					\
do {							\
	KERNEL$RESTORE_INT_STATE(intstate);		\
} while (0)

#define TMP_LOCK					\
do {							\
	KERNEL$DI();					\
} while (0)

void KERNEL$CONSOLE_LOCK(void)
{
	DECL_LOCK;
	LOCK;
	TMP_UNLOCK;
}

void KERNEL$CONSOLE_UNLOCK(void)
{
#if __DEBUG >= 1
	if (__unlikely(console->owner != VAL_CPU_ID) || __unlikely(!console->recurse))
		KERNEL$SUICIDE("KERNEL$CONSOLE_UNLOCK: CONSOLE OWNED BY CPU %X, RECURSE %u", (unsigned)VAL_CPU_ID, console->recurse);
#endif
	DECL_LOCK;
	TMP_LOCK;
	UNLOCK;
}


static __finline__ void SET_CURSOR(void)
{
#ifndef __BOCHS__
	__u8 orig_port;
	int val = (console->x >= console->columns ? console->columns - 1 : console->x) + console->y * console->columns;
	orig_port = io_inb_p(console->port);
	io_outb_p(console->port, 0x0e);
	io_outb_p(console->port + 1, val >> 8);
	io_outb_p(console->port, 0x0f);
	io_outb_p(console->port + 1, val);
	io_outb_p(console->port, orig_port);
#endif
}

void CONSOLE_WRITE(const char *str, int len)
{
	DECL_LOCK;
	LOCK;
	while (len--) {
		unsigned char c = *str++;
		console->buffer[(unsigned long)console->buffer_ptr++ & (CONSOLE_BUFFER_SIZE - 1)] = c;
#ifdef __BOCHS__
		io_outb(0xe9, *str++);
#else
		if (__unlikely(console->gfx)) continue;
		if (c == '\n' || console->x >= console->columns || console->y >= console->lines) {
			int i;
			int racing = 0;
			console_race:
			console->x = 0;
			if (++console->y >= console->lines) {
				__u16 *f;
				console->y = console->lines - 1;
				if (racing < 2) TMP_UNLOCK;
				memmove(console->addr, console->addr + console->columns, console->columns * (console->lines - 1) * 2);
				f = &console->addr[(console->lines - 1) * console->columns];
				for (i = 0; i < console->columns; i++)
					*f++ = 0x0720;
				if (racing < 2) TMP_LOCK;
	/* test if someone messed with the screen while we were scrolling ...
	   race condition can mess the screen but it shouldn't cause access to
	   invalid memory.
	   Locking interrupts while scrolling causes inacceptable latency
	   (clicks in sound playback) */
				if (console->x >= console->columns || console->y >= console->lines) {
					racing++;
					goto console_race;
				}
			}
		}
		if (c != '\n') console->addr[console->x++ + console->y * console->columns] = c | 0x7000;
#endif
		if (__unlikely(!((unsigned long)str & (MAX_LOCK_BYTES - 1)))) {
			UNLOCK;
			LOCK;
		}
	}
	/*
	 * If there was MCE on AMD Athlon, an access to IO ports will lockup
	 * the machine.
	 * We must not update cursor, so that the message gets actually printed.
	 */
	if (!KERNEL_IN_MCE && !console->gfx) SET_CURSOR();
	UNLOCK;
}

size_t KERNEL$CONSOLE_READ(char *dest, size_t size, __u64 *last_ptr)
{
	DECL_LOCK;
	size_t orig_size = size;
	while (size) {
		unsigned len;
		unsigned to_end;

		LOCK;

		if (console->buffer_ptr - *last_ptr >= CONSOLE_BUFFER_SIZE)
			*last_ptr = console->buffer_ptr - CONSOLE_BUFFER_SIZE;
		
		len = console->buffer_ptr - *last_ptr;
		if (__likely(len >= size))
			len = size;
		to_end = CONSOLE_BUFFER_SIZE - ((unsigned long)*last_ptr & (CONSOLE_BUFFER_SIZE - 1));
		if (__unlikely(len > to_end))
			len = to_end;
		if (__likely(len > MAX_COPY_BYTES))
			len = MAX_COPY_BYTES;
		
		memcpy(dest, console->buffer + ((unsigned long)*last_ptr & (CONSOLE_BUFFER_SIZE - 1)), len);
		dest += len;
		size -= len;
		*last_ptr += len;

		UNLOCK;

		if (!len)
			break;
	}
	return orig_size - size;
}

__p_addr KERNEL$CONSOLE_BASE_PHYS_ADDR(void)
{
	return (__u8 *)console->addr - KERNEL$ZERO_BANK;
}

void KERNEL$CONSOLE_SET_CURSOR(unsigned x, unsigned y)
{
	DECL_LOCK;
	LOCK;

	console->x = x;
	console->y = y;

	UNLOCK;
}

void KERNEL$CONSOLE_SET_GRAPHICS(void (*restore)(void))
{
	DECL_LOCK;
	LOCK;

	console->gfx = !!restore;
	console->restore_videomode = restore;

	UNLOCK;
}

void KERNEL$CONSOLE_RESTORE_TEXT(void)
{
	/*
	 * We can't disable interrupts here because otherwise VESA would
	 * have problem calling BIOS.
	 *
	 * On crash it doesn't matter that we have disabled interrupts
	 * --- they are never reenabled.
	 */

	void (*restore)(void) = console->restore_videomode;
	int intstate = KERNEL$SAVE_INT_STATE();
	int spl = KERNEL$SPL;
	if (intstate) RAISE_SPL(SPL_TOP);

	if (restore)
		restore();

	if (intstate) LOWER_SPLX(spl);
}

void PREPARE_FOR_CRASH(void)
{
	static char crash_recurse = 0;

	if (__CMPXCHGC(&crash_recurse, 0, 1))
		return;

	STOP_OTHER_CPUS();

	KERNEL$CONSOLE_RESTORE_TEXT();
}

void CONSOLE_INIT(void)
{
	int r;
	unsigned val;

	KERNEL$SPINLOCK_INIT(&console->lock);
	console->recurse = 0;

	memset((char *)console->buffer, 0, CONSOLE_BUFFER_SIZE);
	console->buffer_ptr = 0;
	console->addr = (void *)(*(__u8 *)(KERNEL$ZERO_BANK + 0x449) == 7 ? KERNEL$ZERO_BANK + 0xb0000 : KERNEL$ZERO_BANK + 0xb8000);
	console->port = *(__u8 *)(KERNEL$ZERO_BANK + 0x449) == 7 ? 0x3b4 : 0x3d4;
	console->columns = *(__u16 *)(KERNEL$ZERO_BANK + 0x44a);
	console->lines = *(__u8 *)(KERNEL$ZERO_BANK + 0x484) + 1;
	if (console->columns <= 1) console->columns = 80;
	if (console->lines <= 1) console->lines = 25;
	console->x = 0;
	console->y = 1;
	iorange.start = console->port - 4;
	iorange.len = 8;
	iorange.name = "CONSOLE";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&iorange))) {
		__critical_printf("CAN'T REGISTER CONSOLE IO RANGE "IO_FORMAT" - "IO_FORMAT": %s\n", iorange.start, iorange.start + iorange.len - 1, strerror(-r));
		HALT_KERNEL();
	}
	io_outb_p(console->port, 0x0e);
	val = io_inb_p(console->port + 1) << 8;
	io_outb_p(console->port, 0x0f);
	val |= io_inb_p(console->port + 1);
	console->x = 0;
	console->y = val / console->columns % console->lines;
	if (!console->y) console->y = 1;
	SET_CURSOR();

	REGISTER_SHARED_POINTER(&console_share);
}

#if __KERNEL_SUPPORT_SMP

void CONSOLE_INIT_AP(void)
{
	console = FIND_SHARED_POINTER("CONSOLE");
}

#endif
