#include <STRING.H>
#include <STDLIB.H>
#include <SETJMP.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/VM.H>
#include <ARCH/BITOPS.H>
#include <ARCH/IO.H>
#include <ARCH/CPU.H>
#include <ARCH/TIME.H>
#include <KERNEL/CONSOLE.H>
#include <KERNEL/SMP/SHARE.H>
#include <KERNEL/SMP/MATRIX.H>
#include <KERNEL/ASM.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/SEG.H>
#include <KERNEL/BOOTCFG.H>
#include <SPAD/DL.H>
#include <SPAD/LINK.H>

#include <KERNEL/MPS.H>
#include <KERNEL/ACPI.H>

#include <ARCH/IRQ.H>
#include <SPAD/IRQ.H>
#include <KERNEL/IRQ.H>
#include <KERNEL/IRQARCH.H>
#include <KERNEL/SMP/IPI.H>

#include "APICREG.H"

/*
 APIC architecture is even worse than 8259.

 Interrupt priorities are basically useless, because to use them, OS would
 have to know priorities of all interrupts in advance, when allocating
 interrupt vectors. If the priorities are determined at a time when driver
 requests an interrupt, there is nothing we can do. (maybe we could change
 numbers of existing active interrupt vectors? --- rather not)

 Local APIC can't mask specific interrupts. It can only mask interrupts up to
 a specific priority; but priorities are useless, so this masking is useless
 as well (I use it only to mask all interrupts when entering real mode BIOS).
 The possibility to mask individual interrupts was added to local APIC on
 newer AMD processors.

 So I must mask interrupts on IO APIC and send an immediate EOI to local APIC
 (to get rid of the effect of those priorities). But IO APIC doesn't have flat
 register space, it uses index/value memory mapped registers. This is stupid
 and it basically means that IO APIC needs to be locked on any access. Why
 can't they map registers linearly to memory? There's plenty of memory space.

 Most other OSes use those useless priorities --- and don't (and can't) assign
 them w.r.t. device importance; so their priorities will be random. I get the
 priorities right, but I stress the APIC architecture in an unusual way and I
 wonder how many bugs will that uncover.

 I've found one bug already. When VIA IO APIC receives an EOI from local APIC,
 it rewrites the index register with interrupt vector of that EOI. The
 workaround is to not allocate interrupt vectors to conflict with valid IO
 APIC registers (if invalid index is selected, version register is accessed,
 that is harmless) and re-check the index after every access to IO APIC data
 register.
*/

#define MAX_IRQS	(int)((long)MAX_IRQ_INT - (long)MIN_IRQ_INT + 1)

typedef __u8 apic_id_t;	/* One day we may support x2apic */

#define APIC_ID_LIMIT	0xff

struct processor {
	__u8 valid;
	apic_id_t local_apic_id;
	__node_id_t node;
	void *kernel_handle;
	void *code[2];
	unsigned long code_size[2];
} __ALIGN_ATTR__(32);

struct bus {
	__u8 type;
	char string[7];
};

#define BUS_NONE	0
#define BUS_ISA		1
#define BUS_PCI		2
#define BUS_UNKNOWN	3

struct io_apic_redtbl {
	__u32 redtbl0;
	__u32 redtbl1;
};

struct io_apic {
	void *virt_addr;
	int regwin_cache;
	apic_id_t io_apic_id;
	__u8 flags;
	__u8 version;
	__u16 pins;
	__p_addr phys_addr;
	struct io_apic_redtbl redtbl[1];
};

#define IO_APIC_FLAGS_INDEX_BUG	0x80

struct __irq_handle {
	__s8 mask_count;
	__u8 irq_flags;
	__u8 io_apic_pin;
	struct io_apic *io_apic;
	void *data;
	IRQ_STUB *call;
};

struct irq2 {
	int irq_num;
	__u8 intr;
};

struct irq_routing {
	__u8 bus_id;
	__u8 bus_irq;
	unsigned irq_idx;
};

struct smp {
	struct processor *processors;
	unsigned n_processors;
	struct bus *buses;
	unsigned n_buses;
	struct io_apic **io_apics;
	unsigned n_io_apics;
	struct __irq_handle *irqs;
	unsigned n_irqs;
	struct irq2 *irq2s;
	int *irq_num_to_idx;
	unsigned n_irq_num_to_idx;
	struct irq_routing *irq_routings;
	unsigned n_irq_routings;
	int elcr;
};

static struct smp main_smp;
static struct smp *smp = &main_smp;

static KERNEL_SHARED smp_share = { "IRQAPIC", &main_smp };

#define processor_valid(i)	(smp->processors[i].valid != 0)
#define bus_valid(i)		(smp->buses[i].type != BUS_NONE)
#define io_apic_valid(i)	(smp->io_apics[i] != NULL)
#define irq_valid(i)		(smp->irqs[i].io_apic != NULL)
#define irq_routing_valid(i)	(1)

static __p_addr local_apic_phys_addr;
static void *local_apic_virt_addr = NULL;
static __u32 local_apic_version;
#define local_apic_integrated()		(local_apic_version & 0xf0)

__COLD_ATTR__ static void *mallc(unsigned size)
{
	void *ptr = malloc(size);
	if (!ptr) {
		__critical_printf("CAN'T ALLOC %u BYTES FOR MP INFO", size);
		HALT_KERNEL();
	}
	memset(ptr, 0, size);
	return ptr;
}

__COLD_ATTR__ static void *reallc(void *ptr, unsigned *entries, unsigned sizeof_entry, unsigned idx)
{
	if (idx < *entries)
		return ptr;
	ptr = realloc(ptr, (idx + 1) * sizeof_entry);
	if (!ptr) {
		__critical_printf("CAN'T ALLOC %u BYTES FOR MP INFO", (idx + 1) * sizeof_entry);
		HALT_KERNEL();
	}
	memset((char *)ptr + (*entries * sizeof_entry), 0, (idx + 1 - *entries) * sizeof_entry);
	*entries = idx + 1;
	return ptr;
}

#define resize(ptr, idx) (smp->ptr = reallc(smp->ptr, &smp->n_##ptr, sizeof(*smp->ptr), idx))

#define FAILURE_CORRECTABLE	1
#define FAILURE_INCORRECTABLE	2

static jmp_buf failure_jmp;

__COLD_ATTR__ static void failure(int type)
{
	if (type == FAILURE_INCORRECTABLE || (!FEATURE_TEST(FEATURE_SMP) && BOOTCFG->PARAM_UP_APIC == PARAM_UP_APIC_AUTO))
		longjmp(failure_jmp, type);
}

__COLD_ATTR__ static struct io_apic *find_io_apic(apic_id_t io_apic_id)
{
	unsigned idx;
	if (io_apic_id < smp->n_io_apics)
		if (io_apic_valid(io_apic_id) && smp->io_apics[io_apic_id]->io_apic_id == io_apic_id)
			return smp->io_apics[io_apic_id];
	for (idx = 0; idx < smp->n_io_apics; idx++)
		if (io_apic_valid(idx) && smp->io_apics[idx]->io_apic_id == io_apic_id)
			return smp->io_apics[idx];
	return NULL;
}

__COLD_ATTR__ static void read_elcr(void)
{
	smp->elcr = io_inb(0x4d0) + (io_inb(0x4d1) << 8);
	if (smp->elcr & ((1 << 0) | (1 << 1) | (1 << 2) | (1 << 13)))
		smp->elcr = -1;
}

__COLD_ATTR__ static void add_processor(apic_id_t local_apic_id)
{
	int i;

	for (i = 0; i < smp->n_processors; i++)
		if (processor_valid(i) && smp->processors[i].local_apic_id == local_apic_id) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "PROCESSOR ID %d PRESENT MULTIPLE TIMES", local_apic_id);
			failure(FAILURE_CORRECTABLE);
			return;
		}

	resize(processors, smp->n_processors);
	smp->processors[smp->n_processors - 1].valid = 1;
	smp->processors[smp->n_processors - 1].local_apic_id = local_apic_id;
	smp->processors[smp->n_processors - 1].node = 0;
}

__COLD_ATTR__ static void scan_processors(void)
{
	int nboot;
	MP_PROCESSOR *proc;
	const ACPI_MADT *madt;
	unsigned madt_length;

	madt = ACPI_FIND_TABLE(ACPI_MADT_SIGNATURE, ACPI_MADT_LENGTH, &madt_length);
	if (!__IS_ERR(madt)) {
		const ACPI_MADT_LOCAL_APIC *lapic;
		ACPI_FIND_RESET(madt->entries, (const __u8 *)madt + madt_length, ACPI_MADT_TYPE_LOCAL_APIC, ACPI_MADT_LOCAL_APIC_LENGTH);
		while ((lapic = ACPI_FIND_ENTRY())) {
			if (__32LE2CPU(lapic->flags) & ACPI_MADT_LOCAL_APIC_FLAGS_ENABLED)
				add_processor(lapic->local_apic_id);
		}
		ACPI_UNMAP(madt, madt_length);
	}

	if (!smp->n_processors) {
		MPS_FIND_RESET(MP_ENTRY_TYPE_PROCESSOR);
		while ((proc = MPS_FIND_ENTRY())) {
			if (!(proc->flags & MP_PROCESSOR_FLAGS_EN))
				continue;
			if (!(proc->flags & MP_PROCESSOR_FLAGS_BP))
				continue;
			add_processor(proc->local_apic_id);
			break;
		}
		nboot = 0;
		MPS_FIND_RESET(MP_ENTRY_TYPE_PROCESSOR);
		while ((proc = MPS_FIND_ENTRY())) {
			if (!(proc->flags & MP_PROCESSOR_FLAGS_EN))
				continue;
			if ((proc->flags & MP_PROCESSOR_FLAGS_BP)) {
				nboot++;
				continue;
			}
			add_processor(proc->local_apic_id);
		}
		if (!nboot) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "NO BOOT PROCESSOR");
			failure(FAILURE_CORRECTABLE);
		}
		if (nboot > 1) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "MULTIPLE BOOT PROCESSORS");
			failure(FAILURE_CORRECTABLE);
		}
	}
	if (!smp->n_processors) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "NO PROCESSORS");
		failure(FAILURE_INCORRECTABLE);
	}
	smp->processors[0].kernel_handle = __DL_GET_SELF(&smp->processors[0].code[0], &smp->processors[0].code_size[0], &smp->processors[0].code[1], &smp->processors[0].code_size[1]);
}

__COLD_ATTR__ static void scan_numa(void)
{
	const ACPI_SRAT *srat;
	unsigned real_length;
	const ACPI_SRAT_PROCESSOR_LOCAL_APIC *lapic;

	srat = ACPI_FIND_TABLE(ACPI_SRAT_SIGNATURE, ACPI_SRAT_LENGTH, &real_length);
	if (__IS_ERR(srat))
		return;

	ACPI_FIND_RESET(srat->entries, (const __u8 *)srat + real_length, ACPI_SRAT_TYPE_PROCESSOR_LOCAL_APIC, ACPI_SRAT_PROCESSOR_LOCAL_APIC_LENGTH);
	while ((lapic = ACPI_FIND_ENTRY())) {
		int i;

		if (!(__32LE2CPU(lapic->flags) & ACPI_SRAT_PROCESSOR_LOCAL_APIC_FLAGS_ENABLED))
			continue;

		for (i = 0; i < smp->n_processors; i++)
			if (processor_valid(i) && smp->processors[i].local_apic_id == lapic->local_apic_id) {
				__u32 proximity_domain = acpi_srat_processor_local_apic_proximity_domain(lapic);
				smp->processors[i].node = ACPI_PROXIMITY_DOMAIN_TO_NODE_ID(proximity_domain);
				break;
			}
	}
	ACPI_UNMAP(srat, real_length);
}

static char had_no_isa_bus = 0;

__COLD_ATTR__ static void scan_buses(void)
{
	int have_isa = 0;
	MP_BUS *bus;
	MPS_FIND_RESET(MP_ENTRY_TYPE_BUS);
	while ((bus = MPS_FIND_ENTRY())) {
		__u8 type;
		char *p, *q;
		if (!memcmp(bus->string, "ISA   ", sizeof bus->string)) {
			type = BUS_ISA;
		} else if (!memcmp(bus->string, "PCI   ", sizeof bus->string)) {
			type = BUS_PCI;
		} else {
			type = BUS_UNKNOWN;
		}
		resize(buses, bus->bus_id);
		if (bus_valid(bus->bus_id)) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "BUS ID %d PRESENT MULTIPLE TIMES", bus->bus_id);
			failure(FAILURE_CORRECTABLE);
			continue;
		}
		smp->buses[bus->bus_id].type = type;
		*(__u8 *)mempcpy(smp->buses[bus->bus_id].string, bus->string, sizeof bus->string) = 0;
		for (q = p = smp->buses[bus->bus_id].string; *p; p++) {
			if (*p != ' ') q = p + 1;
		}
		*q = 0;
		if (type == BUS_ISA)
			have_isa = 1;
	}
	if (!have_isa) {
		if (!MPS_DEFAULT()) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "NO ISA BUS IN MP TABLE");
			failure(FAILURE_CORRECTABLE);
			had_no_isa_bus = 1;
		}
		resize(buses, smp->n_buses + 1);
		smp->buses[smp->n_buses - 1].type = BUS_ISA;
		strcpy(smp->buses[smp->n_buses - 1].string, "ISA");
	}
}

__COLD_ATTR__ static void scan_io_apics(void)
{
	MP_IO_APIC *apic;
	unsigned i;
	MPS_FIND_RESET(MP_ENTRY_TYPE_IO_APIC);
	while ((apic = MPS_FIND_ENTRY())) {
		struct io_apic *io_apic;
		if (!(apic->flags & MP_IO_APIC_FLAGS_EN))
			continue;
		resize(io_apics, apic->io_apic_id);
		if (io_apic_valid(apic->io_apic_id)) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "IO APIC ID %d PRESENT MULTIPLE TIMES", apic->io_apic_id);
			failure(FAILURE_CORRECTABLE);
			continue;
		}
		io_apic = mallc(sizeof(struct io_apic));
		io_apic->phys_addr = __32LE2CPU(apic->io_apic_addr);
		io_apic->regwin_cache = -1;
		io_apic->io_apic_id = apic->io_apic_id;
		smp->io_apics[apic->io_apic_id] = io_apic;
	}
	for (i = 0; i < smp->n_io_apics; i++)
		if (io_apic_valid(i)) goto have_some_io_apic;
	KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "NO IO APICS");
	failure(FAILURE_INCORRECTABLE);
	have_some_io_apic:;
}

__COLD_ATTR__ static void add_interrupt(MP_IO_INTERRUPT *io_int)
{
	if (io_int->interrupt_type == MP_INTERRUPT_TYPE_INT) {
		unsigned irq_flags;
		struct io_apic *io_apic;
		unsigned i;
		if (io_int->source_bus_id >= smp->n_buses) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "INVALID BUS ID %u IN IO INTERRUPT TABLE (%u BUSES)", io_int->source_bus_id, smp->n_buses);
			failure(FAILURE_CORRECTABLE);
			return;
		}
		if (smp->buses[io_int->source_bus_id].type == BUS_ISA) {
			if (io_int->source_bus_irq >= 16) {
				KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "INVALID ISA IRQ %u IN IO INTERRUPT TABLE", io_int->source_bus_irq);
				failure(FAILURE_CORRECTABLE);
				return;
			}
			if (io_int->source_bus_irq == 2 ||
			    io_int->source_bus_irq == 13) {
				/* These have no use */
				return;
			}
			irq_flags = 0;
		} else if (smp->buses[io_int->source_bus_id].type == BUS_PCI) {
			irq_flags = IRQ_FLAGS_POLARITY_LOW | IRQ_FLAGS_TRIGGER_LEVEL;
		} else {
			return;
		}
		if ((io_int->flags & MP_INTERRUPT_FLAGS_PO) == MP_INTERRUPT_FLAGS_PO_INVALID || (io_int->flags & MP_INTERRUPT_FLAGS_TRIGGER) == MP_INTERRUPT_FLAGS_TRIGGER_INVALID) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "INVALID FLAGS %02X IN IO INTERRUPT TABLE", io_int->flags);
			failure(FAILURE_CORRECTABLE);
			return;
		}
		if ((io_int->flags & MP_INTERRUPT_FLAGS_PO) == MP_INTERRUPT_FLAGS_PO_HIGH)
			irq_flags &= ~IRQ_FLAGS_POLARITY_LOW;
		if ((io_int->flags & MP_INTERRUPT_FLAGS_PO) == MP_INTERRUPT_FLAGS_PO_LOW)
			irq_flags |= IRQ_FLAGS_POLARITY_LOW;
		if ((io_int->flags & MP_INTERRUPT_FLAGS_TRIGGER) == MP_INTERRUPT_FLAGS_TRIGGER_EDGE)
			irq_flags &= ~IRQ_FLAGS_TRIGGER_LEVEL;
		if ((io_int->flags & MP_INTERRUPT_FLAGS_TRIGGER) == MP_INTERRUPT_FLAGS_TRIGGER_LEVEL)
			irq_flags |= IRQ_FLAGS_TRIGGER_LEVEL;
		io_apic = find_io_apic(io_int->dest_io_apic_id);
		if (!io_apic) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "INVALID IO APIC ID %02X IN IO INTERRUPT TABLE", io_int->dest_io_apic_id);
			failure(FAILURE_CORRECTABLE);
			return;
		}
		for (i = 0; i < smp->n_irq_routings; i++)
			if (irq_routing_valid(i) && smp->irq_routings[i].bus_id == io_int->source_bus_id && smp->irq_routings[i].bus_irq == io_int->source_bus_irq)
				return;
		resize(irq_routings, smp->n_irq_routings + 1);
		smp->irq_routings[smp->n_irq_routings - 1].bus_id = io_int->source_bus_id;
		smp->irq_routings[smp->n_irq_routings - 1].bus_irq = io_int->source_bus_irq;
		for (i = 0; i < smp->n_irqs; i++)
			if (irq_valid(i) && smp->irqs[i].io_apic == io_apic && smp->irqs[i].io_apic_pin == io_int->dest_io_apic_pin) {
				smp->irq_routings[smp->n_irq_routings - 1].irq_idx = i;
				if (smp->irqs[i].irq_flags != irq_flags) {
					KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "INCONSISTENT IRQ FLAGS FOR IO APIC %X, PIN %X: %X != %X", io_apic->io_apic_id, io_int->dest_io_apic_pin, smp->irqs[i].irq_flags, irq_flags);
					failure(FAILURE_CORRECTABLE);
					return;
				}
				return;
			}
		if (smp->n_irqs == MAX_IRQS) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "TOO MANY IRQS, MAX %d", MAX_IRQS);
			failure(FAILURE_CORRECTABLE);
			return;
		}
		resize(irqs, smp->n_irqs);
		smp->irq_routings[smp->n_irq_routings - 1].irq_idx = smp->n_irqs - 1;
		smp->irqs[smp->n_irqs - 1].io_apic = io_apic;
		smp->irqs[smp->n_irqs - 1].io_apic_pin = io_int->dest_io_apic_pin;
		smp->irqs[smp->n_irqs - 1].irq_flags = irq_flags;
	}
}

__COLD_ATTR__ static void add_isa_interrupts(void);

__COLD_ATTR__ static void scan_irqs(void)
{
	unsigned i;
	MP_IO_INTERRUPT *io_int;
	MPS_FIND_RESET(MP_ENTRY_TYPE_IO_INTERRUPT);
	while ((io_int = MPS_FIND_ENTRY())) {
		add_interrupt(io_int);
	}
	for (i = 0; i < smp->n_irq_routings; i++) if (irq_routing_valid(i)) {
		if (smp->buses[smp->irq_routings[i].bus_id].type == BUS_ISA)
			goto have_isa_bus;
	}
	add_isa_interrupts();
	have_isa_bus:;
}

__COLD_ATTR__ static void add_isa_interrupts(void)
{
	/* If we have no IRQs for ISA, connect them to the first 16
	   pins of the first IO APIC. Maybe they belong there. */
	unsigned bus, io_apic, irq;
	if (!MPS_DEFAULT() && !had_no_isa_bus) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "NO IRQ ROUTING FOR ISA BUS");
		failure(FAILURE_CORRECTABLE);
	}
	for (bus = 0; bus < smp->n_buses; bus++) if (bus_valid(bus)) {
		if (smp->buses[bus].type == BUS_ISA) goto have_isa_bus;
	}
	KERNEL$SUICIDE("scan_irqs: NO ISA BUS");
	have_isa_bus:
	for (io_apic = 0; io_apic < smp->n_io_apics; io_apic++) if (io_apic_valid(io_apic)) goto have_io_apic;
	KERNEL$SUICIDE("scan_irqs: NO IO APIC");
	have_io_apic:
	for (irq = 0; irq < 16; irq++) if (irq != 2) {
		MP_IO_INTERRUPT io_int_val;
		io_int_val.type = MP_ENTRY_TYPE_IO_INTERRUPT;
		io_int_val.interrupt_type = MP_INTERRUPT_TYPE_INT;
		io_int_val.flags = MP_INTERRUPT_FLAGS_PO_DEFAULT | MP_INTERRUPT_FLAGS_TRIGGER_DEFAULT;
		if (smp->elcr >= 0 && smp->elcr & (1 << irq)) io_int_val.flags = MP_INTERRUPT_FLAGS_PO_LOW | MP_INTERRUPT_FLAGS_TRIGGER_LEVEL;
		io_int_val.pad = 0;
		io_int_val.source_bus_id = bus;
		io_int_val.source_bus_irq = irq;
		io_int_val.dest_io_apic_id = io_apic;
		io_int_val.dest_io_apic_pin = !irq ? 2 : irq;
		add_interrupt(&io_int_val);
	}
}

/* MPS 1.4 says that the OS is responsible for renumbering io apics if conflict
   is found. */
__COLD_ATTR__ static void renumber_io_apics(void)
{
	unsigned i;
	__u32 new_id = 0;
	for (i = 0; i < smp->n_io_apics; i++) if (io_apic_valid(i)) {
		unsigned j;
		__u32 io_apic_id = smp->io_apics[i]->io_apic_id;
		int changed;
		for (j = 0; j < smp->n_processors; j++)
			if (processor_valid(j) && smp->processors[j].local_apic_id == io_apic_id) goto conflict;
		for (j = 0; j < i; j++)
			if (io_apic_valid(j) && smp->io_apics[j]->io_apic_id == io_apic_id) goto conflict;
		continue;

		conflict:
		if (new_id >= APIC_ID_LIMIT)
			goto skip_it;
		do {
			changed = 0;
			for (j = 0; j < smp->n_processors; j++)
				if (processor_valid(j) && smp->processors[j].local_apic_id == new_id) new_id++, changed = 1;
			for (j = 0; j < smp->n_io_apics; j++)
				if (io_apic_valid(j) && smp->io_apics[j]->io_apic_id == new_id) new_id++, changed = 1;
		} while (changed);
		if (new_id >= APIC_ID_LIMIT) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "NO FREE APIC ID");
			skip_it:
			free(smp->io_apics[i]);
			smp->io_apics[i] = NULL;
			continue;
		}
		smp->io_apics[i]->io_apic_id = new_id;
	}
}

static __u8 io_apic_minimum_intr = 0;

#define INTR_STEP	16

__COLD_ATTR__ static void allocate_interrupts(void)
{
	int offset;
	unsigned irq;
	smp->irq2s = mallc(smp->n_irqs * sizeof(struct irq2));
	for (irq = 0; irq < smp->n_irqs; irq++) smp->irq2s[irq].irq_num = -1;
	irq = 0;
	for (offset = 0; offset < INTR_STEP; offset++) {
		int intr;
		for (intr = (long)MAX_IRQ_INT - offset; intr >= (long)MIN_IRQ_INT && intr >= io_apic_minimum_intr; intr -= INTR_STEP) {
			if (irq == smp->n_irqs)
				goto processed_all;
			smp->irq2s[irq++].intr = intr;
			while (irq < smp->n_irqs && !irq_valid(irq)) irq++;
		}
	}
	processed_all:
	if (irq != smp->n_irqs) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "TOO MANY IRQS: %u, MAX %u", smp->n_irqs, irq);
		failure(FAILURE_CORRECTABLE);
		for (; irq < smp->n_irqs; irq++) {
			memset(&smp->irqs[irq], 0, sizeof(IRQ_HANDLE));
			memset(&smp->irq2s[irq], 0, sizeof(struct irq2));
		}
	}
}

__COLD_ATTR__ static void allocate_irq_numbers(void)
{
	unsigned i;
	int irq_num = 0;
	for (i = 0; i < smp->n_buses; i++) if (bus_valid(i) && smp->buses[i].type == BUS_ISA) {
		unsigned j;
		int isa_irq_num = -1;
		for (j = 0; j < smp->n_irq_routings; j++) if (irq_routing_valid(j) && smp->irq_routings[j].bus_id == i) {
			unsigned irq_idx = smp->irq_routings[j].irq_idx;
			if (!irq_valid(irq_idx) || smp->irq2s[irq_idx].irq_num >= 0)
				continue;
			if (isa_irq_num == -1) {
				isa_irq_num = irq_num;
				irq_num += 16;
			}
			smp->irq2s[irq_idx].irq_num = isa_irq_num + smp->irq_routings[j].bus_irq;
		}
	}
	for (i = 0; i < smp->n_io_apics; i++) if (io_apic_valid(i)) {
		unsigned j;
		unsigned irq_num_shift = MAXUINT;
		for (j = 0; j < smp->n_irqs; j++) if (irq_valid(j) && smp->irqs[j].io_apic == smp->io_apics[i]) {
			if (smp->irq2s[j].irq_num >= 0)
				continue;
			if (smp->irqs[j].io_apic_pin < irq_num_shift)
				irq_num_shift = smp->irqs[j].io_apic_pin;
		}
		if (irq_num_shift == MAXUINT) continue;
		irq_num_shift &= ~15;
		for (j = 0; j < smp->n_irqs; j++) if (irq_valid(j) && smp->irqs[j].io_apic == smp->io_apics[i]) {
			if (smp->irq2s[j].irq_num >= 0)
				continue;
			if (smp->irqs[j].io_apic_pin < irq_num_shift)
				continue;
			smp->irq2s[j].irq_num = irq_num + smp->irqs[j].io_apic_pin - irq_num_shift;
		}
		irq_num += smp->io_apics[i]->pins - irq_num_shift;
	}
	smp->n_irq_num_to_idx = irq_num;
	smp->irq_num_to_idx = mallc(irq_num * sizeof(int));
	for (i = 0; i < smp->n_irq_num_to_idx; i++)
		smp->irq_num_to_idx[i] = -1;
	for (i = 0; i < smp->n_irqs; i++) if (irq_valid(i)) {
		if (smp->irq2s[i].irq_num < 0) {
			memset(&smp->irqs[i], 0, sizeof smp->irqs[i]);
			memset(&smp->irq2s[i], 0, sizeof smp->irq2s[i]);
			continue;
		}
		smp->irq_num_to_idx[smp->irq2s[i].irq_num] = i;
	}
}

__COLD_ATTR__ static void map_local_apic(void)
{
	local_apic_virt_addr = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(local_apic_phys_addr, LOCAL_APIC_REGSPACE, PAT_UC);
	if (__IS_ERR(local_apic_virt_addr)) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "UNABLE TO MAP LOCAL APIC AT %"__64_format"X: %s", (__u64)local_apic_phys_addr, strerror(-__PTR_ERR(local_apic_virt_addr)));
		local_apic_virt_addr = NULL;
		failure(FAILURE_INCORRECTABLE);
	}
}

__COLD_ATTR__ static void map_io_apics(void)
{
	unsigned idx;
	for (idx = 0; idx < smp->n_io_apics; idx++) if (io_apic_valid(idx)) {
		smp->io_apics[idx]->virt_addr = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(smp->io_apics[idx]->phys_addr, IO_APIC_REGSPACE, PAT_UC);
		if (__IS_ERR(smp->io_apics[idx]->virt_addr)) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "UNABLE TO MAP IO APIC AT %"__64_format"X: %s", (__u64)smp->io_apics[idx]->phys_addr, strerror(-__PTR_ERR(smp->io_apics[idx]->virt_addr)));
			smp->io_apics[idx]->virt_addr = NULL;
			failure(FAILURE_INCORRECTABLE);
		}
	}
}

static __finline__ __u32 local_apic_read(unsigned reg)
{
	return mmio_inl((__u8 *)local_apic_virt_addr + reg);
}

static __finline__ void local_apic_write(unsigned reg, __u32 val)
{
	__asm__ volatile ("		\n\
	41:				\n\
		MOVL	%0, %2		\n\
	42:				\n\
		.SECTION .rodata.end	\n\
	43:				\n\
		XCHGL	%0, %2		\n\
	44:				\n\
		.PREVIOUS		\n\
		.SECTION .FEATURE_FIXUP	\n\
		.LONG	41b, 42b, 43b, 44b, "__stringify(_FEATURE_LOCAL_APIC_LOCK_WRITE)"				\n\
		.PREVIOUS		\n\
	":"=r"(val):"0"(val),"m"(*((__u8 *)local_apic_virt_addr + reg)));
}

static __finline__ void local_apic_write_icr(__u32 lo, __u32 hi)
{
	local_apic_write(LOCAL_APIC_ICR1, hi);
	local_apic_write(LOCAL_APIC_ICR0, lo);
}

static __finline__ void local_apic_write_icr_shorthand(__u32 lo)
{
	local_apic_write(LOCAL_APIC_ICR0, lo);
}

__COLD_ATTR__ static int local_apic_wait_for_icr_timeout(void)
{
	int i;
	for (i = 0; i < 1000; i++) {
		__u32 icr = local_apic_read(LOCAL_APIC_ICR0);
		if (__likely(!(icr & LOCAL_APIC_ICR0_DELIVS)))
			return 0;
		KERNEL$UDELAY(100);
	}
	return -ETIMEDOUT;
}

static void local_apic_wait_for_icr(int splx)
{
	while (1) {
		__u32 icr;
		icr = local_apic_read(LOCAL_APIC_ICR0);
		if (__likely(!(icr & LOCAL_APIC_ICR0_DELIVS)))
			return;
		LOWER_SPLX(splx);
		DO_NOP();
		RAISE_SPL(SPL_TOP);
	}
}

static __finline__ void io_apic_index(struct io_apic *apic, unsigned idx)
{
	if (__unlikely(idx != apic->regwin_cache))
		mmio_outb((__u8 *)apic->virt_addr + IO_APIC_IOREGSEL, apic->regwin_cache = idx);
}

__NOINLINE_ATTR__ static __u32 io_apic_read_index_bug(struct io_apic *apic, unsigned idx);
__NOINLINE_ATTR__ static void io_apic_write_index_bug(struct io_apic *apic, unsigned idx, __u32 val);

static __u32 io_apic_read(struct io_apic *apic, unsigned idx)
{
	__u32 val;
	if (__unlikely(apic->flags & IO_APIC_FLAGS_INDEX_BUG))
		return io_apic_read_index_bug(apic, idx);
	io_apic_index(apic, idx);
	val = mmio_inl((__u8 *)apic->virt_addr + IO_APIC_IOREGWIN);
	return val;
}

static void io_apic_write(struct io_apic *apic, unsigned idx, __u32 val)
{
	if (__unlikely(apic->flags & IO_APIC_FLAGS_INDEX_BUG))
		return io_apic_write_index_bug(apic, idx, val);
	io_apic_index(apic, idx);
	mmio_outl((__u8 *)apic->virt_addr + IO_APIC_IOREGWIN, val);
}

__NOINLINE_ATTR__ static __u32 io_apic_read_index_bug(struct io_apic *apic, unsigned idx)
{
	__u32 val;
	retry:
	mmio_outb((__u8 *)apic->virt_addr + IO_APIC_IOREGSEL, idx);
	val = mmio_inl((__u8 *)apic->virt_addr + IO_APIC_IOREGWIN);
	if (mmio_inb((__u8 *)apic->virt_addr + IO_APIC_IOREGSEL) != idx) {
		goto retry;
	}
	return val;
}

__NOINLINE_ATTR__ static void io_apic_write_index_bug(struct io_apic *apic, unsigned idx, __u32 val)
{
	mmio_outb((__u8 *)apic->virt_addr + IO_APIC_IOREGSEL, idx);
	retry:
	mmio_outl((__u8 *)apic->virt_addr + IO_APIC_IOREGWIN, val);
	if (mmio_inb((__u8 *)apic->virt_addr + IO_APIC_IOREGSEL) != idx) {
		goto retry;
	}
}

static const unsigned short local_apic_lvt[6] = {
	LOCAL_APIC_LVT_TIMER,
	LOCAL_APIC_LVT_LINT0,
	LOCAL_APIC_LVT_LINT1,
	LOCAL_APIC_LVT_ERROR,
	LOCAL_APIC_LVT_PERF_COUNT,
	LOCAL_APIC_LVT_THERMAL,
};

__COLD_ATTR__ static unsigned local_apic_max_lvt(void)
{
	unsigned max_lvt;
	if (!local_apic_integrated()) {
		max_lvt = 2;
	} else {
		max_lvt = __GET_FIELD(local_apic_version, LOCAL_APIC_VERSION_MAX_LVT_ENTRY);
	}
	if (max_lvt >= sizeof(local_apic_lvt) / sizeof(*local_apic_lvt))
		max_lvt = sizeof(local_apic_lvt) / sizeof(*local_apic_lvt) - 1;
	return max_lvt;
}

__COLD_ATTR__ static int local_apic_read_version(void)
{
	local_apic_version = local_apic_read(LOCAL_APIC_VERSION);
	if (local_apic_version == 0xffffffff || local_apic_max_lvt() < 2) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "LOCAL APIC FOR CPU %X NOT PRESENT AT %"__64_format"X", VAL_CPU_ID, (__u64)local_apic_phys_addr);
		return -1;
	}
	return 0;
}

__COLD_ATTR__ static void local_apic_detect(void)
{
	if (local_apic_read_version()) {
		failure(FAILURE_INCORRECTABLE);
	}
}

__COLD_ATTR__ static void io_apics_init_ids(void)
{
	unsigned idx;
	for (idx = 0; idx < smp->n_io_apics; idx++) if (io_apic_valid(idx)) {
		__u32 version;
		smp->io_apics[idx]->flags |= IO_APIC_FLAGS_INDEX_BUG;
		version = io_apic_read(smp->io_apics[idx], IO_APIC_VER);
		if (__unlikely(version == 0xffffffff)) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "IO APIC NOT PRESENT AT %"__64_format"X", (__u64)smp->io_apics[idx]->phys_addr);
			failure(FAILURE_INCORRECTABLE);
		}
		smp->io_apics[idx]->version = version & IO_APIC_VER_VERSION;
		smp->io_apics[idx]->pins = __GET_FIELD(version, IO_APIC_VER_MAX_ENTRY) + 1;
		/*if (version & IO_APIC_VER_PCI_IRQ) {*/
			/*__u32 boot_cfg = io_apic_read(smp->io_apics[idx], IO_EXT_APIC_BOOT_CFG);
			if (boot_cfg && boot_cfg != 0xffffffff && boot_cfg != version && boot_cfg & IO_EXT_APIC_BOOT_CFG_FSB) {*/
				smp->io_apics[idx]->regwin_cache = -1;
				mmio_outb((__u8 *)smp->io_apics[idx]->virt_addr + IO_APIC_IOREGSEL, 0x01);
				mmio_outb((__u8 *)smp->io_apics[idx]->virt_addr + IO_EXT_APIC_EOI, 0x02);
				if (mmio_inb((__u8 *)smp->io_apics[idx]->virt_addr + IO_APIC_IOREGSEL) != 0x01) {
			/* VIA IO APIC trashes index when it receives EOI */
					unsigned minimum_intr;
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, "KERNEL", "IO APIC WITH INDEX REGISTER BUG AT %"__64_format"X, VERSION %08X, WORKAROUNG APPLIED", (__u64)smp->io_apics[idx]->phys_addr, (unsigned)version);
					minimum_intr = IO_APIC_REDTBL0 + smp->io_apics[idx]->pins * IO_APIC_REDTBL_STEP;
					if (minimum_intr > io_apic_minimum_intr) io_apic_minimum_intr = minimum_intr;
					goto apic_is_buggy;
				}
			/*}*/
		/*}*/
		smp->io_apics[idx]->flags &= ~IO_APIC_FLAGS_INDEX_BUG;
		apic_is_buggy:
		if ((version & IO_APIC_VER_VERSION) < IO_APIC_VER_VERSION_XAPIC) {
			__u32 id = io_apic_read(smp->io_apics[idx], IO_APIC_ID);
			if (__GET_FIELD(id, IO_APIC_ID_ID) != smp->io_apics[idx]->io_apic_id) {
				io_apic_write(smp->io_apics[idx], IO_APIC_ID, smp->io_apics[idx]->io_apic_id << __BSF_CONST(IO_APIC_ID_ID));
				id = io_apic_read(smp->io_apics[idx], IO_APIC_ID);
				if (__GET_FIELD(id, IO_APIC_ID_ID) != smp->io_apics[idx]->io_apic_id) {
					KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "IO APIC NOT WRITEABLE AT %"__64_format"X", (__u64)smp->io_apics[idx]->phys_addr);
					failure(FAILURE_INCORRECTABLE);
				}
			}
		}
	}
	for (idx = 0; idx < smp->n_irqs; idx++) if (irq_valid(idx)) {
		if (smp->irqs[idx].io_apic_pin >= smp->irqs[idx].io_apic->pins) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "IRQ %u CONNECTED TO NONEXISTENT IO APIC PIN %u (MAX %u)", idx, smp->irqs[idx].io_apic_pin, smp->irqs[idx].io_apic->pins);
			failure(FAILURE_CORRECTABLE);
			memset(&smp->irqs[idx], 0, sizeof(smp->irqs[idx]));
			memset(&smp->irq2s[idx], 0, sizeof(smp->irq2s[idx]));
		}
	}
}

__COLD_ATTR__ static __u32 local_apic_get_error(void)
{
	if (!local_apic_integrated())
		return 0;
	if (local_apic_max_lvt() > 3)
		local_apic_write(LOCAL_APIC_ESR, 0);
	return local_apic_read(LOCAL_APIC_ESR);
}

__COLD_ATTR__ static int local_apic_leave(unsigned lvt, __u32 val)
{
	val &= LOCAL_APIC_LVT_DELMOD;
	if (lvt == LOCAL_APIC_LVT_THERMAL && val == LOCAL_APIC_LVT_DELMOD_SMI)
		return 1;
	if ((lvt == LOCAL_APIC_LVT_LINT0 || lvt == LOCAL_APIC_LVT_LINT1) && (val == LOCAL_APIC_LVT_DELMOD_SMI || val == LOCAL_APIC_LVT_DELMOD_NMI))
		return 1;
	return 0;
}

__COLD_ATTR__ static void local_apic_reset(void)
{
	__u32 v;
	unsigned i;
	unsigned max_lvt = local_apic_max_lvt();

	if (max_lvt >= 3) {
		v = local_apic_read(LOCAL_APIC_LVT_ERROR);
		local_apic_write(LOCAL_APIC_LVT_ERROR, v | LOCAL_APIC_LVT_MASK);
	}
	for (i = 0; i <= max_lvt; i++) {
		v = local_apic_read(local_apic_lvt[i]);
		if (local_apic_leave(local_apic_lvt[i], v)) continue;
		local_apic_write(local_apic_lvt[i], v | LOCAL_APIC_LVT_MASK);
	}
	for (i = 0; i <= max_lvt; i++) {
		v = local_apic_read(local_apic_lvt[i]);
		if (local_apic_leave(local_apic_lvt[i], v)) continue;
		local_apic_write(local_apic_lvt[i], LOCAL_APIC_LVT_MASK);
	}
	local_apic_get_error();

	local_apic_wait_for_icr_timeout();
}

__COLD_ATTR__ static void grab_imcrp(void)
{
	if (MPS_FP_SIZE && __32LE2CPU(MPS_FP->features) & MP_FP_FEATURES_IMCRP) {
		io_outb(0x22, 0x70);
		io_outb(0x23, 0x01);
	}
}

__COLD_ATTR__ static void io_apics_reset(void)
{
	unsigned idx;
	for (idx = 0; idx < smp->n_io_apics; idx++) if (io_apic_valid(idx)) {
		unsigned i;
		struct io_apic *apic = realloc(smp->io_apics[idx], sizeof(struct io_apic) + (smp->io_apics[idx]->pins - 1) * sizeof(struct io_apic_redtbl));
		if (!apic) {
			__critical_printf("CAN'T ALLOC %u BYTES FOR IO APIC", (unsigned)(sizeof(struct io_apic) + (smp->io_apics[idx]->pins - 1) * sizeof(struct io_apic_redtbl)));
			HALT_KERNEL();
		}
		for (i = 0; i < smp->n_irqs; i++) if (irq_valid(i))
			if (smp->irqs[i].io_apic == smp->io_apics[idx])
				smp->irqs[i].io_apic = apic;
		smp->io_apics[idx] = apic;
		for (i = 0; i < apic->pins; i++) {
			__u32 redtbl0 = io_apic_read(apic, IO_APIC_REDTBL0 + i * IO_APIC_REDTBL_STEP);
			switch (redtbl0 & IO_APIC_REDTBL0_DELMOD) {
				case IO_APIC_REDTBL0_DELMOD_SMI:
				case IO_APIC_REDTBL0_DELMOD_NMI:
					break;
				default:
					redtbl0 = IO_APIC_REDTBL0_INTVEC | IO_APIC_REDTBL0_DELMOD_FIXED | IO_APIC_REDTBL0_MASK;
					io_apic_write(apic, IO_APIC_REDTBL0 + i * IO_APIC_REDTBL_STEP, redtbl0);
			}
			apic->redtbl[i].redtbl0 = redtbl0;
			apic->redtbl[i].redtbl1 = 0;
		}
	}
}

#define IRQ_STUB_PTR(i)	APIC_IRQ_STUBS[MAX_IRQS - smp->n_irqs + i].c
#define IPI_STUB_PTR(i)	APIC_IPI_STUBS[i - MIN_IPI_INT].c

__COLD_ATTR__ static void patch_assembler(void)
{
	unsigned i;
	write_4bytes(APIC_EOI_PATCH_1, (unsigned long)((__u8 *)local_apic_virt_addr + LOCAL_APIC_EOI));
	write_4bytes(APIC_EOI_PATCH_2, (unsigned long)((__u8 *)local_apic_virt_addr + LOCAL_APIC_EOI));
	for (i = 0; i < smp->n_irqs; i++) if (irq_valid(i)) {
		/*__debug_printf("irq(%d)->int(0x%x)  ", i, smp->irqs[i].intr);*/
		set_idt_gate(smp->irq2s[i].intr, IRQ_STUB_PTR(i));
	}
	/*__debug_printf("\n");*/
}

__COLD_ATTR__ static void local_apic_init(void)
{
	local_apic_write(LOCAL_APIC_SVR, LOCAL_APIC_SVR_ENABLE | (long)SPURIOUS_INT);
	local_apic_write(LOCAL_APIC_TPR, 0);
}

__COLD_ATTR__ static void irqs_setup_routing(void)
{
	unsigned idx;
	for (idx = 0; idx < smp->n_irqs; idx++) if (irq_valid(idx)) {
		struct io_apic *apic = smp->irqs[idx].io_apic;
		__u32 redtbl0, redtbl1;
		unsigned pin;
		redtbl0 = IO_APIC_REDTBL0_DELMOD_FIXED | IO_APIC_REDTBL0_MASK;
		if (smp->irqs[idx].irq_flags & IRQ_FLAGS_POLARITY_LOW) redtbl0 |= IO_APIC_REDTBL0_INTPOL;
		if (smp->irqs[idx].irq_flags & IRQ_FLAGS_TRIGGER_LEVEL) redtbl0 |= IO_APIC_REDTBL0_TRIGGER;
		redtbl0 |= smp->irq2s[idx].intr << __BSF_CONST(IO_APIC_REDTBL0_INTVEC);
		redtbl1 = smp->processors[0].local_apic_id << __BSF_CONST(IO_APIC_REDTBL1_DESTINATION_FIELD);
		pin = smp->irqs[idx].io_apic_pin;
		/*__debug_printf("routing(%x,%x,%x,%x)\n", idx, smp->irq2s[idx].irq_num, pin, smp->irq2s[idx].intr);*/
		apic->redtbl[pin].redtbl0 = redtbl0;
		apic->redtbl[pin].redtbl1 = redtbl1;
		smp->irqs[idx].mask_count = 0;
		io_apic_write(apic, IO_APIC_REDTBL0 + pin * IO_APIC_REDTBL_STEP, redtbl0);
		io_apic_write(apic, IO_APIC_REDTBL1 + pin * IO_APIC_REDTBL_STEP, redtbl1);
	}
}

__COLD_ATTR__ static void free_this(void)
{
	unsigned idx;
	if (local_apic_virt_addr) {
		KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(local_apic_virt_addr, LOCAL_APIC_REGSPACE);
		local_apic_virt_addr = NULL;
	}
	for (idx = 0; idx < smp->n_io_apics; idx++) if (io_apic_valid(idx)) {
		if (smp->io_apics[idx]->virt_addr && !__IS_ERR(smp->io_apics[idx]->virt_addr))
			KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(smp->io_apics[idx]->virt_addr, IO_APIC_REGSPACE);
		free(smp->io_apics[idx]);
	}
	free(smp->processors), smp->processors = NULL, smp->n_processors = 0;
	free(smp->buses), smp->buses = NULL, smp->n_buses = 0;
	free(smp->io_apics), smp->io_apics = NULL, smp->n_io_apics = 0;
	free(smp->irqs), smp->irqs = NULL, smp->n_irqs = 0;
	free(smp->irq2s); smp->irq2s = NULL;
	free(smp->irq_num_to_idx), smp->irq_num_to_idx = NULL, smp->n_irq_num_to_idx = 0;
	free(smp->irq_routings), smp->irq_routings = NULL, smp->n_irq_routings = 0;
}

__COLD_ATTR__ static IRQ_HANDLE *APIC_GET_HANDLE(int irq_num)
{
	int irq;
	if ((unsigned)irq_num >= smp->n_irq_num_to_idx) return NULL;
	irq = smp->irq_num_to_idx[irq_num];
	if (irq < 0) return NULL;
	return &smp->irqs[irq];
}

#define APIC_MAP_IRQ(n)							\
	if (irq_num >= smp->n_irq_num_to_idx)				\
		KERNEL$SUICIDE(n ": INVALID IRQ %d (MAX %u)", irq_num, smp->n_irq_num_to_idx);								\
	irq = smp->irq_num_to_idx[irq_num];				\
	if (irq < 0)							\
		KERNEL$SUICIDE(n ": INVALID IRQ %d", irq_num);

__COLD_ATTR__ static void APIC_IRQ_STARTUP(int irq_num)
{
	int irq;
	APIC_MAP_IRQ("APIC_IRQ_STARTUP");
	if (smp->irqs[irq].mask_count != 0)
		KERNEL$SUICIDE("APIC_IRQ_STARTUP: WRONG IRQ %d MASK, COUNT %d", irq, smp->irqs[irq].mask_count);
	smp->irqs[irq].irq_flags &= ~IRQ_FLAGS_MISSED_EDGE;
	/*__debug_printf("startup %x (apic %x, pin %x)\n", irq, smp->irqs[irq].io_apic->io_apic_id, smp->irqs[irq].io_apic_pin);*/
	if (!(smp->irqs[irq].irq_flags & IRQ_FLAGS_TRIGGER_LEVEL)) {
		struct io_apic *apic = smp->irqs[irq].io_apic;
		unsigned pin = smp->irqs[irq].io_apic_pin;
	/*__debug_printf("enable %d\n", irq);*/
		io_apic_write(apic, IO_APIC_REDTBL0 + pin * IO_APIC_REDTBL_STEP, apic->redtbl[pin].redtbl0 &= ~IO_APIC_REDTBL0_MASK);
	}
}

__COLD_ATTR__ static void APIC_IRQ_SHUTDOWN(int irq_num, int masked)
{
	int irq;
	APIC_MAP_IRQ("APIC_IRQ_SHUTDOWN");
	if (!masked) {
		if (smp->irqs[irq].mask_count >= 0)
			KERNEL$SUICIDE("APIC_IRQ_SHUTDOWN: IRQ %d NOT UNMASKED BEFORE RELEASE, COUNT %d", irq, smp->irqs[irq].mask_count);
		KERNEL$MASK_IRQ(&smp->irqs[irq]);
	} else {
		if (smp->irqs[irq].mask_count != 0)
			KERNEL$SUICIDE("APIC_IRQ_SHUTDOWN: IRQ %d UNMASKED BEFORE RELEASE, COUNT %d", irq, smp->irqs[irq].mask_count);
	}
	if (!(smp->irqs[irq].irq_flags & IRQ_FLAGS_TRIGGER_LEVEL)) {
		struct io_apic *apic = smp->irqs[irq].io_apic;
		unsigned pin = smp->irqs[irq].io_apic_pin;
		io_apic_write(apic, IO_APIC_REDTBL0 + pin * IO_APIC_REDTBL_STEP, apic->redtbl[pin].redtbl0 |= IO_APIC_REDTBL0_MASK);
	}
}

__COLD_ATTR__ static void copy_irq_template(char *stub, __u32 val, void *jmp)
{
	memcpy(stub, &APIC_STUB_TEMPLATE, SIZEOF_APIC_IRQ_STUB);
	write_4bytes(stub + (long)APIC_STUB_PATCH_IRQ, val);
	write_4bytes(stub + (long)APIC_STUB_PATCH_JMP, (char *)jmp - (stub + (long)APIC_STUB_PATCH_JMP + 4));
}

__COLD_ATTR__ static void APIC_REQUEST_RT_IRQ(int irq_num, IRQ_STUB *call, void *data)
{
	int irq;
	if (((unsigned long)call & 31) != 0)
		KERNEL$SUICIDE("APIC_REQUEST_RT_IRQ: UNALIGNED CALL %p", call);
	APIC_MAP_IRQ("APIC_REQUEST_RT_IRQ");
	copy_irq_template(IRQ_STUB_PTR(irq), (__u32)&smp->irqs[irq], APIC_COMMON_RT);
	smp->irqs[irq].call = call;
	smp->irqs[irq].data = data;
}

__COLD_ATTR__ static void APIC_REQUEST_AST_IRQ(int irq_num, AST *ast)
{
	int irq;
	APIC_MAP_IRQ("APIC_REQUEST_AST_IRQ");
	copy_irq_template(IRQ_STUB_PTR(irq), (__u32)&smp->irqs[irq], APIC_COMMON_AST);
	smp->irqs[irq].data = ast;
}

#if __KERNEL_SUPPORT_SMP

__COLD_ATTR__ static void APIC_REQUEST_IPI(int ipi, IRQ_STUB *call, unsigned data)
{
	__u32 val;
	char *stub;
	if (((unsigned long)call & 31) != 0)
		KERNEL$SUICIDE("APIC_REQUEST_IPI: UNALIGNED CALL %p", call);
	if (data >= 32)
		KERNEL$SUICIDE("APIC_REQUEST_IPI: INVALID DATA %u", data);
	if ((unsigned)ipi < MIN_IPI_INT || (unsigned)ipi > MAX_IPI_INT)
		KERNEL$SUICIDE("APIC_REQUEST_IPI: INVALID IPI %u", ipi);
	val = (__u32)call | data;
	stub = IPI_STUB_PTR(ipi);
	copy_irq_template(stub, val, APIC_COMMON_IPI);
	set_idt_gate(ipi, stub);
}

#endif

static char di_hw = 0;

__COLD_ATTR__ static void APIC_DI_HW(void)
{
	local_apic_write(LOCAL_APIC_TPR, LOCAL_APIC_TPR_MASK);
	if (di_hw)
		KERNEL$SUICIDE("APIC_DI_HW: INTERRUPTS ALREADY DISABLED");
	di_hw = 1;
}

__COLD_ATTR__ static void APIC_EI_HW(void)
{
	if (!di_hw)
		KERNEL$SUICIDE("APIC_DI_HW: INTERRUPTS WERE NOT ENABLED");
	di_hw = 0;
	local_apic_write(LOCAL_APIC_TPR, 0);
}

__COLD_ATTR__ static void INTR(unsigned int_no)
{
	__u32 code = 0xc300cd | (int_no << 8);
	/*((void (*)(void))&code)();*/	/* GCC compiles it as call *%esp and it fails on Intel P6 due to a CPU BUG */
	__asm__ volatile ("call *%0"::"abcdSD"(&code):"memory");
}

void KERNEL$MASK_IRQ(IRQ_HANDLE *irq)
{
	__u32 flags;
	__asm__ volatile ("PUSHFL; POPL %0":"=r"(flags):);
	if (__likely(!(flags & EFLAGS_IF))) {
		KERNEL_MASK_IRQ_DI(irq);
		return;
	}
	__asm__ volatile ("CLI":::"memory");
	KERNEL_MASK_IRQ_DI(irq);
	__asm__ volatile ("STI":::"memory");
}

IRQ_HANDLE *KERNEL_MASK_IRQ_DI(IRQ_HANDLE *irq)
{
#if __DEBUG >= 2
	{
		__u32 flags;
		__asm__ volatile ("PUSHFL; POPL %0":"=r"(flags):);
		if (__unlikely(flags & EFLAGS_IF))
			KERNEL$SUICIDE("KERNEL_MASK_IRQ_DI: INTERRUPTS ARE ENABLED, IRQ %d", (int)(irq - smp->irqs));
	}
#endif

	irq->mask_count++;
#if __DEBUG >= 1
	if (__unlikely(irq->mask_count < 0))
		KERNEL$SUICIDE("KERNEL_MASK_IRQ_DI: MASK COUNT OVERFLOW: %d, IRQ %d", irq->mask_count, (int)(irq - smp->irqs));
#endif

	if (__likely(!irq->mask_count) && __likely(irq->irq_flags & IRQ_FLAGS_TRIGGER_LEVEL)) {
		struct io_apic *apic = irq->io_apic;
		unsigned pin = irq->io_apic_pin;
		io_apic_write(apic, IO_APIC_REDTBL0 + pin * IO_APIC_REDTBL_STEP, apic->redtbl[pin].redtbl0 |= IO_APIC_REDTBL0_MASK);
	}
	return irq;
}

void KERNEL$UNMASK_IRQ(IRQ_HANDLE *irq)
{
#if __DEBUG >= 2
	{
		__u32 flags;
		__asm__ volatile ("PUSHFL; POPL %0":"=r"(flags):);
		if (__unlikely(!(flags & EFLAGS_IF)))
			KERNEL$SUICIDE("KERNEL$UNMASK_IRQ: INTERRUPTS ARE DISABLED, IRQ %d", (int)(irq - smp->irqs));
	}
#endif

	__asm__ volatile ("CLI":::"memory");

#if __DEBUG >= 1
	if (__unlikely(irq->mask_count < 0))
		KERNEL$SUICIDE("KERNEL$UNMASK_IRQ: UNMASKING ALREADY UNMASKED IRQ, COUNT %d, IRQ %d", irq->mask_count, (int)(irq - smp->irqs));
#endif
	if (__unlikely(--irq->mask_count >= 0))
		goto still_masked;

	if (__likely(irq->irq_flags & IRQ_FLAGS_TRIGGER_LEVEL)) {
		struct io_apic *apic = irq->io_apic;
		unsigned pin = irq->io_apic_pin;
		io_apic_write(apic, IO_APIC_REDTBL0 + pin * IO_APIC_REDTBL_STEP, apic->redtbl[pin].redtbl0 &= ~IO_APIC_REDTBL0_MASK);
	} else if (__unlikely(irq->irq_flags & IRQ_FLAGS_MISSED_EDGE)) {
		irq->irq_flags &= ~IRQ_FLAGS_MISSED_EDGE;
		INTR(smp->irq2s[irq - smp->irqs].intr);
	}

	still_masked:
	__asm__ volatile ("STI":::"memory");
}

#if __KERNEL_SUPPORT_SMP

__COLD_ATTR__ static void SEND_IPI_ALL_EXCL_SELF_TIMEOUT(unsigned ipi)
{
	local_apic_wait_for_icr_timeout();
	local_apic_write_icr_shorthand(ipi + LOCAL_APIC_ICR0_DELMOD_FIXED + LOCAL_APIC_ICR0_SHORTHAND_ALL_EXCL_SELF);
}

void SEND_IPI(__cpu_id_t target, unsigned ipi, int splx)
{
	local_apic_wait_for_icr(splx);
	/*__debug_printf("send ipi %x to target %x, lapic id %x\n", ipi, target, smp->processors[target].local_apic_id);*/
	/*local_apic_write_icr_shorthand(ipi + LOCAL_APIC_ICR0_DELMOD_FIXED + LOCAL_APIC_ICR0_SHORTHAND_ALL_EXCL_SELF);*/
	local_apic_write_icr(ipi + LOCAL_APIC_ICR0_DELMOD_FIXED + LOCAL_APIC_ICR0_SHORTHAND_NO, smp->processors[target].local_apic_id << __BSF_CONST(LOCAL_APIC_ICR1_DESTINATION_FIELD));
}

#endif

__COLD_ATTR__ int KERNEL$QUERY_IRQ_ROUTING(__const__ char *bus_string, unsigned bus_id, unsigned bus_irq)
{
	int i;
	struct bus *bus;
	unsigned bus_idx;
	/*__debug_printf("query routing: %s, %x, %x -> ", bus_string, bus_id, bus_irq);*/
	if (__likely(bus_id < smp->n_buses) && __likely(bus_valid(bus_id))) {
		bus = &smp->buses[bus_id];
		if (!strcmp(bus->string, bus_string))
			goto have_bus;
	}
	bus = NULL;
	for (i = 0; i < smp->n_buses; i++) if (bus_valid(i)) {
		if (!strcmp(smp->buses[i].string, bus_string)) {
			if (bus)
				return -ENOENT;
			bus = &smp->buses[i];
		}
	}
	if (!bus)
		return -ENOENT;

	have_bus:
	bus_idx = bus - smp->buses;
	for (i = 0; i < smp->n_irq_routings; i++) if (irq_routing_valid(i) && irq_valid(smp->irq_routings[i].irq_idx)) {
		if (smp->irq_routings[i].bus_id == bus_idx && smp->irq_routings[i].bus_irq == bus_irq && irq_valid(smp->irq_routings[i].irq_idx)) {
			/*__debug_printf("%x,%x\n", i, smp->irq2s[smp->irq_routings[i].irq_idx].irq_num);*/
			return smp->irq2s[smp->irq_routings[i].irq_idx].irq_num;
		}
	}
	if (!strcmp(bus_string, "PCI")) {
		/* Try pin 0. Linux does it too */
		for (i = 0; i < smp->n_irq_routings; i++) if (irq_routing_valid(i) && irq_valid(smp->irq_routings[i].irq_idx)) {
			if (smp->irq_routings[i].bus_id == bus_idx && smp->irq_routings[i].bus_irq == (bus_irq & ~3) && irq_valid(smp->irq_routings[i].irq_idx)) {
				/*__debug_printf("%x,%x (xxx)\n", i, smp->irq2s[smp->irq_routings[i].irq_idx].irq_num);*/
				return smp->irq2s[smp->irq_routings[i].irq_idx].irq_num;
			}
		}
	}
	return -ENOENT;
}

__COLD_ATTR__ static void set_irqc(void)
{
	IRQC.n_irqs = smp->n_irq_num_to_idx;
	IRQC.irq_get_handle = APIC_GET_HANDLE;
	IRQC.irq_request_rt = APIC_REQUEST_RT_IRQ;
	IRQC.irq_request_ast = APIC_REQUEST_AST_IRQ;
	IRQC.irq_startup = APIC_IRQ_STARTUP;
	IRQC.irq_shutdown = APIC_IRQ_SHUTDOWN;
	IRQC.di_hw = APIC_DI_HW;
	IRQC.ei_hw = APIC_EI_HW;
#if __KERNEL_SUPPORT_SMP
	IRQC.ipi_request = APIC_REQUEST_IPI;
#endif
}

static char apic_active = 0;

__COLD_ATTR__ int APIC_IRQ_INIT(void)
{
	REGISTER_SHARED_POINTER(&smp_share);
	memset(smp, 0, sizeof(struct smp));

	if (OFF___irq_handle_mask_count != __offsetof(IRQ_HANDLE, mask_count))
		KERNEL$SUICIDE("APIC_IRQ_INIT: BAD mask_count OFFSET");
	if (OFF___irq_handle_irq_flags != __offsetof(IRQ_HANDLE, irq_flags))
		KERNEL$SUICIDE("APIC_IRQ_INIT: BAD irq_flags OFFSET");
	if (OFF___irq_handle_data != __offsetof(IRQ_HANDLE, data))
		KERNEL$SUICIDE("APIC_IRQ_INIT: BAD data OFFSET");
	if (OFF___irq_handle_call != __offsetof(IRQ_HANDLE, call))
		KERNEL$SUICIDE("APIC_IRQ_INIT: BAD call OFFSET");
	if (BOOTCFG->PARAM_UP_APIC == PARAM_UP_APIC_DISABLE && !FEATURE_TEST(FEATURE_SMP))
		return -ENOENT;
	if (!MPS_TABLE_SIZE)
		return -ENOENT;
	local_apic_phys_addr = __32LE2CPU(MPS_TABLE->local_apic_addr);
	if (setjmp(failure_jmp)) {
		free_this();
		return -EIO;
	}
	read_elcr();
	scan_processors();
	scan_numa();
	scan_buses();
	scan_io_apics();
	scan_irqs();
	renumber_io_apics();

	map_local_apic();
	map_io_apics();

	local_apic_detect();
	io_apics_init_ids();

	allocate_interrupts();
	allocate_irq_numbers();

	/* Point of no return */

	I8259_HW_INIT((long)SPURIOUS_INT - 7, (long)SPURIOUS_INT - 7);

	local_apic_reset();
	grab_imcrp();
	io_apics_reset();
	patch_assembler();
	local_apic_init();
	irqs_setup_routing();

	set_irqc();

	apic_active = 1;

	return 0;
}

#if __KERNEL_SUPPORT_SMP

static char aps_booted = 0;

extern char SMP_TRAMPOLINE[];
extern char SMP_TRAMPOLINE_END[];

#define SMP_BOOT_TIMEOUT	(JIFFIES_PER_SECOND * 5)
#define SMP_STARTUP_TRIES	3

__COLD_ATTR__ static int wait_for_semaphore(__u8 val, u_jiffies_lo_t timeout)
{
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	while (1) {
		__u8 c;
		u_jiffies_lo_t jj = KERNEL$GET_JIFFIES_LO();
		c = *(volatile __u8 *)(KERNEL$ZERO_BANK + SMP_SEMAPHORE_1_PHYS);
		if (c == val)
			return 0;
		if (c == SMP_SEMAPHORE_1_VAL_PANIC) {
			PREPARE_FOR_CRASH();
			__critical_printf("CPU CRASHED IN BOOT\n");
			HALT_KERNEL();
		}
		if (c == SMP_SEMAPHORE_1_VAL_STOPPED) {
			return -EIO;
		}
		if (c > val) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "KERNEL", "SMP BOOT SEMAPHORE CORRUPTED, READS AS %02X", c);
			return -EINVAL;
		}
		if (jj - j > timeout)
			return -ETIMEDOUT;
	}
}

__COLD_ATTR__ static void halt_cpu(void)
{
	int r;
	*(KERNEL$ZERO_BANK + SMP_SEMAPHORE_2_PHYS) = SMP_SEMAPHORE_2_VAL_HALT;
	r = wait_for_semaphore(SMP_SEMAPHORE_1_VAL_HALTED, SMP_BOOT_TIMEOUT);
	if (r) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "KERNEL", "UNABLE TO HALT CPU");
	}
}

__COLD_ATTR__ static int boot_cpu(__cpu_id_t i)
{
	int r;

	*(__u32 *)(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_CR0) = CR_0;
	*(__u32 *)(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_CR3) = KERNEL$VIRT_2_PHYS(VM_BOOT_MAKE_DIRECT_MAPPING());
	*(__u32 *)(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_CR4) = CR_4;
	*(__u64 *)(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_GDTR) = GDTR;
	*(unsigned long *)(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_JMP) = (unsigned long)((char *)smp->processors[i].code[0] + JMP_BASE_OFFSET + JMP_SIZE);
	*(__u16 *)(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_JMP + 4) = SEG_KCODE;
	*(unsigned long *)(KERNEL$ZERO_BANK + SMP_FIND_SHARED) = (unsigned long)BSP_FIND_SHARED_POINTER;
	*(__u32 *)(KERNEL$ZERO_BANK + SMP_CPU_ID) = i;
	*(unsigned long *)(KERNEL$ZERO_BANK + SMP_CODE) = (unsigned long)smp->processors[i].code[0];

	__rw_barrier();

	*(KERNEL$ZERO_BANK + SMP_SEMAPHORE_2_PHYS) = SMP_SEMAPHORE_2_VAL_GO;

	r = wait_for_semaphore(SMP_SEMAPHORE_1_VAL_RUNNING, SMP_BOOT_TIMEOUT);
	if (r) {
		__critical_printf("UNABLE TO START CPU %X\n", i);
		/* !!! FIXME: enable: HALT_KERNEL(); */
	}

	VM_BOOT_CLEAR_DIRECT_MAPPING();

	return 0;
}

DECL_AST(RET, 4, IORQ)
{
	unsigned t = RDTSC_LO();
	__debug_printf("double x-ast latency: %u (%x / %x)\n", t - (unsigned)RQ->tmp3, t, (unsigned)RQ->tmp3);
	RETURN;
}

/* !!! FIXME: testme: toto je test */
DECL_AST(FN, 4, IORQ)
{
	RQ->fn = (AST_STUB *)RQ->tmp2;
	RETURN_AST(RQ);
}

IORQ ast = { NULL, 0 };

static void test()
{
	int i, j;
	for (j = 0; j < 3; j++) for (i = 0; i < 8; i++) {
		IORQ *a = GET_KERNEL_VAR(&ast, i);
		a->fn = GET_KERNEL_VAR(FN, i);
		a->tmp2 = (unsigned long)RET;
		a->tmp3 = RDTSC_LO();
		CALL_AST(a);
		KERNEL$SLEEP(JIFFIES_PER_SECOND);
		/*__asm__ volatile ("xorl %%eax, %%eax; 1: decl %%eax; jnz 1b":::"ax","cc");*/
	}
	__debug_printf("asts posted\n");
}

__COLD_ATTR__ void BOOT_APS(void)
{
	int i;

	if (KERNEL$SPL != SPL_X(SPL_ZERO))
		KERNEL$SUICIDE("BOOT_APS AT SPL %08X", KERNEL$SPL);

	if (!apic_active)
		return;

	if (!FEATURE_TEST(FEATURE_SMP))
		return;

	if (VAL_CPU_ID)
		return;

	if (__CMPXCHGC(&aps_booted, 0, 1))
		return;

	if (MATRIX_INIT())
		return;

	memcpy(KERNEL$ZERO_BANK + SMP_TRAMPOLINE_PHYS, SMP_TRAMPOLINE, SMP_TRAMPOLINE_END - SMP_TRAMPOLINE);

	for (i = 0; i < smp->n_processors; i++) if (processor_valid(i)) {
		int r;
		__u32 e;
		__u32 dest;
		__u8 reg70;
		void *ld;

		if (!i)		/* BOOT processor */
			continue;
		dest = smp->processors[i].local_apic_id << __BSF_CONST(LOCAL_APIC_ICR1_DESTINATION_FIELD);

		reg70 = io_inb_p(0x70);
		reg70 &= 0x80;
		reg70 |= 0x0f;
		io_outb_p(0x70, reg70);
		io_outb_p(0x71, 0x0a);

		*(__u16 *)(KERNEL$ZERO_BANK + 0x467) = 0;
		*(__u16 *)(KERNEL$ZERO_BANK + 0x469) = SMP_TRAMPOLINE_PHYS >> 4;

		*(KERNEL$ZERO_BANK + SMP_SEMAPHORE_1_PHYS) = 0;
		*(KERNEL$ZERO_BANK + SMP_SEMAPHORE_2_PHYS) = 0;

		local_apic_get_error();
		RAISE_SPL(SPL_TOP);
		local_apic_wait_for_icr_timeout();
		local_apic_write_icr(LOCAL_APIC_ICR0_DELMOD_INIT | LOCAL_APIC_ICR0_LEVEL | LOCAL_APIC_ICR0_TRIGGER | LOCAL_APIC_ICR0_SHORTHAND_NO, dest);
		local_apic_wait_for_icr_timeout();
		LOWER_SPL(SPL_ZERO);
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 100 + 1);
		RAISE_SPL(SPL_TOP);
		local_apic_write_icr(LOCAL_APIC_ICR0_DELMOD_INIT | LOCAL_APIC_ICR0_TRIGGER | LOCAL_APIC_ICR0_SHORTHAND_NO, dest);
		local_apic_wait_for_icr_timeout();
		LOWER_SPL(SPL_ZERO);
		if (local_apic_integrated()) {
			int startup_tries = 0;
			local_apic_get_error();
			retry_startup:
			RAISE_SPL(SPL_TOP);
			local_apic_write_icr(SMP_TRAMPOLINE_PHYS >> 12 << __BSF_CONST(LOCAL_APIC_ICR0_INTVEC) | LOCAL_APIC_ICR0_DELMOD_STARTUP | LOCAL_APIC_ICR0_SHORTHAND_NO, dest);
			KERNEL$UDELAY(300);
			local_apic_wait_for_icr_timeout();
			LOWER_SPL(SPL_ZERO);
			r = wait_for_semaphore(SMP_SEMAPHORE_1_VAL_STARTED, SMP_BOOT_TIMEOUT);
			e = local_apic_get_error();
			if (r == -ETIMEDOUT && !(e & (LOCAL_APIC_ESR_SEND_CHECKSUM_ERROR | LOCAL_APIC_ESR_RECEIVE_CHECKSUM_ERROR | LOCAL_APIC_ESR_SEND_ACCEPT_ERROR | LOCAL_APIC_ESR_RECEIVE_ACCEPT_ERROR | LOCAL_APIC_ESR_SEND_ILLEGAL_VECTOR)) && ++startup_tries < SMP_STARTUP_TRIES)
				goto retry_startup;
		} else {
			r = wait_for_semaphore(SMP_SEMAPHORE_1_VAL_STARTED, SMP_BOOT_TIMEOUT);
			e = local_apic_get_error();
		}

		io_outb_p(0x70, reg70);
		io_outb_p(0x71, 0x00);

		if (r) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "KERNEL", "UNABLE TO START CPU %X, APIC ERROR %08X", smp->processors[i].local_apic_id, e);
			cpu_failed:
			smp->processors[i].valid = 0;
			continue;
		}

		ld = __DL_LOAD_SMP_KERNEL(&smp->processors[i].code[0], &smp->processors[i].code_size[0], &smp->processors[i].code[1], &smp->processors[i].code_size[1]);
		if (__unlikely(__IS_ERR(ld))) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "KERNEL", "UNABLE TO LOAD KERNEL FOR CPU %X: %s", smp->processors[i].local_apic_id, strerror(-__PTR_ERR(ld)));
			halt_failed:
			halt_cpu();
			goto cpu_failed;
		}
		smp->processors[i].kernel_handle = ld;

		__debug_printf("kernel loaded: %p/%p(%ld)/%p(%ld)\n", ld, smp->processors[i].code[0], smp->processors[i].code_size[0], smp->processors[i].code[1], smp->processors[i].code_size[1]);

		if (boot_cpu(i))
			goto halt_failed;
	}
	test();
}

__COLD_ATTR__ static void STOP_ME(void)
{
	*(volatile __u8 *)(KERNEL$ZERO_BANK + SMP_SEMAPHORE_1_PHYS) = SMP_SEMAPHORE_1_VAL_STOPPED;
	HALT_CPU();
}

/* This is called before the console goes up */
__COLD_ATTR__ void APIC_INIT_AP1(void);
__COLD_ATTR__ void APIC_INIT_AP1(void)
{
	smp = FIND_SHARED_POINTER("IRQAPIC");
	NODE_ID = CPU_NODE(CPU_ID);
	CPU_ID_LIMIT = *(__cpu_id_t *)GET_KERNEL_VAR(&CPU_ID_LIMIT, 0);
	NODE_ID_LIMIT = *(__node_id_t *)GET_KERNEL_VAR(&NODE_ID_LIMIT, 0);
	__DL_INIT(smp->processors[CPU_ID].code[0],
		  (char *)smp->processors[CPU_ID].code[0] + smp->processors[CPU_ID].code_size[0],
		  smp->processors[CPU_ID].code[1],
		  (char *)smp->processors[CPU_ID].code[1] + smp->processors[CPU_ID].code_size[1]);
}

/* This is called after the console goes up */
__COLD_ATTR__ void APIC_INIT_AP2(void);
__COLD_ATTR__ void APIC_INIT_AP2(void)
{
	local_apic_phys_addr = *(__p_addr *)GET_KERNEL_VAR(&local_apic_phys_addr, 0);
	local_apic_virt_addr = *(void **)GET_KERNEL_VAR(&local_apic_virt_addr, 0);
	if (local_apic_read_version()) {
		STOP_ME();
	}
	local_apic_reset();
	patch_assembler();
	local_apic_init();

	set_irqc();

	apic_active = 1;
	aps_booted = 1;
}

__COLD_ATTR__ void STOP_OTHER_CPUS(void)
{
	if (!apic_active || !aps_booted) {
		if (CPU_ID)
			*(volatile __u8 *)(KERNEL$ZERO_BANK + SMP_SEMAPHORE_1_PHYS) = SMP_SEMAPHORE_1_VAL_PANIC;
		return;
	}

	SEND_IPI_ALL_EXCL_SELF_TIMEOUT(IPI_HALT);
}

int VALID_CPU_ID(__cpu_id_t cpu)
{
	return cpu < smp->n_processors && processor_valid(cpu) && smp->processors[cpu].kernel_handle;
}

__node_id_t CPU_NODE(__cpu_id_t cpu)
{
	return smp->processors[cpu].node;
}

__COLD_ATTR__ void *GET_KERNEL_VAR(void *ptr, __cpu_id_t cpu)
{
	struct processor *self;
	int i;
	unsigned long diff;

	if (__unlikely(cpu >= smp->n_processors) || __unlikely(!processor_valid(cpu)))
		KERNEL$SUICIDE("GET_KERNEL_VAR: INVALID CPU ID: %X (%d %d) %p %p\n", cpu, smp->n_processors, processor_valid(cpu), smp, &main_smp);

	self = &smp->processors[CPU_ID];
	for (i = 0; i < 2; i++) {
		if ((char *)ptr >= (char *)self->code[i] && (char *)ptr < (char *)self->code[i] + self->code_size[i])
			goto got_it;
	}
	KERNEL$SUICIDE("GET_KERNEL_VAR: INVALID POINTER %p, KERNEL AT %p/%lX, %p/%lX", ptr, self->code[0], self->code_size[0], self->code[1], self->code_size[1]);

	got_it:
	diff = (char *)ptr - (char *)self->code[i];
	return (char *)smp->processors[cpu].code[i] + diff;
}

#endif
