#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <ARCH/IO.H>
#include <ARCH/TIME.H>
#include <KERNEL/ASM.H>
#include <KERNEL/SEG.H>

#include <ARCH/IRQ.H>
#include <SPAD/IRQ.H>
#include <KERNEL/IRQ.H>
#include <KERNEL/IRQARCH.H>

#define I8259_VALID_IRQ(n)	((unsigned)(n) != 2 && (unsigned)(n) < N_I8259_IRQS)
#define I8259_IRQ_2_IDX(n)	((n) - ((n) > 2))

unsigned read_4bytes(char *p)
{
	return (unsigned char)p[0] + ((unsigned char)p[1] << 8) + ((unsigned char)p[2] << 16) + ((unsigned char)p[3] << 24);
}

void write_4bytes(char *p, unsigned v)
{
	p[0] = v & 0xff;
	p[1] = (v >> 8) & 0xff;
	p[2] = (v >> 16) & 0xff;
	p[3] = (v >> 24) & 0xff;
}

void set_idt_gate(unsigned intr, void *addr)
{
	__u64 entry = ((unsigned long)addr & 0xffff) | (SEG_KCODE << 16) | ((__u64)0x8e00 << 32) | (__u64)((unsigned long)addr & 0xffff0000) << 32;
	IDT[intr] = entry;
}

static void I8259_REQUEST_RT_IRQ(int irq, IRQ_STUB *call, void *data)
{
	int intstate;
	if (((unsigned long)call & 31) != 0)
		KERNEL$SUICIDE("I8259_REQUEST_RT_IRQ: UNALIGNED CALL %p", call);
	if (!I8259_VALID_IRQ(irq)) KERNEL$SUICIDE("I8259_REQUEST_RT_IRQ: UNACCEPTABLE IRQ NUMBER: %d", irq);
	irq = I8259_IRQ_2_IDX(irq);
	intstate = KERNEL$SAVE_INT_STATE();
	KERNEL$DI();
	memcpy(I8259_IRQ_STUBS + irq, I8259_RT_STUB_TEMPLATES + irq, SIZEOF_I8259_IRQ_STUB);
	write_4bytes(I8259_IRQ_STUBS[irq].c + I8259_RT_CALL_PATCHES[irq], read_4bytes(I8259_IRQ_STUBS[irq].c + I8259_RT_CALL_PATCHES[irq]) + (unsigned)call);
	write_4bytes(I8259_IRQ_STUBS[irq].c + I8259_RT_PTR_PATCHES[irq], (unsigned)data);
	KERNEL$RESTORE_INT_STATE(intstate);
}

static void I8259_REQUEST_AST_IRQ(int irq, AST *ast)
{
	int intstate;
	if (!I8259_VALID_IRQ(irq)) KERNEL$SUICIDE("I8259_REQUEST_AST_IRQ: UNACCEPTABLE IRQ NUMBER: %d", irq);
	irq = I8259_IRQ_2_IDX(irq);
	intstate = KERNEL$SAVE_INT_STATE();
	KERNEL$DI();
	memcpy(I8259_IRQ_STUBS + irq, I8259_AST_STUB_TEMPLATES + irq, SIZEOF_I8259_IRQ_STUB);
	write_4bytes(I8259_IRQ_STUBS[irq].c + I8259_AST_PATCHES[irq], (unsigned)ast);
	KERNEL$RESTORE_INT_STATE(intstate);
}

static void (*I8259_DISABLE_VECTOR(int irq))(void)
{
	if (!I8259_VALID_IRQ(irq)) KERNEL$SUICIDE("I8259_DISABLE_VECTOR: UNACCEPTABLE IRQ NUMBER: %d", irq);
	return (void (*)(void))(I8259_DISABLE_VECTORS + I8259_IRQ_2_IDX(irq) * (unsigned long)I8259_DISABLE_ENABLE_STRIDE);
}

static void (*I8259_ENABLE_VECTOR(int irq))(void)
{
	if (!I8259_VALID_IRQ(irq)) KERNEL$SUICIDE("I8259_ENABLE_VECTOR: UNACCEPTABLE IRQ NUMBER: %d", irq);
	return (void (*)(void))(I8259_ENABLE_VECTORS + I8259_IRQ_2_IDX(irq) * (unsigned long)I8259_DISABLE_ENABLE_STRIDE);
}

static char * const IRQ_PATCHES[N_I8259_IRQS - 1] = {
	I8259_PATCH_0,
	I8259_PATCH_1,
	I8259_PATCH_3,
	I8259_PATCH_4,
	I8259_PATCH_5,
	I8259_PATCH_6,
	I8259_PATCH_7,
	I8259_PATCH_8,
	I8259_PATCH_9,
	I8259_PATCH_10,
	I8259_PATCH_11,
	I8259_PATCH_12,
	I8259_PATCH_13,
	I8259_PATCH_14,
	I8259_PATCH_15,
};

static void I8259_IRQ_STARTUP(int irq)
{
	if (!I8259_VALID_IRQ(irq)) KERNEL$SUICIDE("IRQC_STARTUP: UNACCEPTABLE IRQ NUMBER: %d", irq);
	irq = I8259_IRQ_2_IDX(irq);
	IRQ_PATCHES[irq][0] = 0xEB;
	IRQ_PATCHES[irq][1] = 0xFB;
}

static void I8259_IRQ_SHUTDOWN(int irq, int masked)
{
	if (!I8259_VALID_IRQ(irq)) KERNEL$SUICIDE("IRQC_SHUTDOWN: UNACCEPTABLE IRQ NUMBER: %d", irq);
	if (!masked) {
		if (irq < 8 ? I8259MASK_1 & (1 << irq) : I8259MASK_2 & (1 << (irq - 8))) KERNEL$SUICIDE("IRQC_SHUTDOWN: IRQ %d NOT UNMASKED BEFORE RELEASE", irq);
		I8259_DISABLE_VECTOR(irq)();
	}
}

static __s8 intstate = -1;

static void I8259_DI_HW(void)
{
	int i;
	if (__unlikely(intstate == -2)) return;
	if (__unlikely(intstate != -1))
		KERNEL$SUICIDE("I8259_DI_HW: INTERRUPTS NOT ENABLED, INTSTATE %d", intstate);
	for (i = 0; i < N_I8259_IRQS; i++) if (I8259_VALID_IRQ(i))
		I8259_DISABLE_VECTOR(i)();
	/* This is not as easy as it seems: We must first mask interrupts on
	   the PIC and then disable processor interrupts, otherwise spurious
	   int 7 would be called in real mode (and that means crash, because
	   interrupt vectors differ).

	   If the interrupts were disabled on processor (that only happens
	   during crash screen restore), we must not reflect spurious 7 to
	   real mode. So we reinitialize the PIC with spurious vector ---
	   we are never ever going to enable interrupts again, so resetting
	   the PIC won't matter.
	*/
	intstate = KERNEL$SAVE_INT_STATE();
	if (__unlikely(!intstate)) {
		I8259_HW_INIT((long)SPURIOUS_INT - 7, (long)SPURIOUS_INT - 7);
		intstate = -2;
	}
	KERNEL$DI();
}

static void I8259_EI_HW(void)
{
	int i;
	if (__unlikely(intstate == -2)) return;
	if (__unlikely(intstate != 1))
		KERNEL$SUICIDE("I8259_EI_HW: INTERRUPTS NOT DISABLED, INTSTATE %d", intstate);
	intstate = -1;
	KERNEL$EI();
	for (i = 0; i < N_I8259_IRQS; i++) if (I8259_VALID_IRQ(i))
		I8259_ENABLE_VECTOR(i)();
}

static IRQ_HANDLE *I8259_GET_HANDLE(int irq)
{
	if (!I8259_VALID_IRQ(irq)) return NULL;
	return (IRQ_HANDLE *)I8259_ENABLE_VECTOR(irq);
}

static void I8259_GET_RESOURCES(void)
{
	static IO_RANGE iorange1;
	static IO_RANGE iorange2;
	int r;
	iorange1.start = 0x20;
	iorange1.len = 2;
	iorange1.name = "IRQ MASTER";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&iorange1))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR IRQ MASTER: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	iorange2.start = 0xa0;
	iorange2.len = 2;
	iorange2.name = "IRQ MASTER";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&iorange2))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR IRQ SLAVE: %s\n", strerror(-r));
		HALT_KERNEL();
	}
}

/* Copied from Linux. Copyright (c) I don't know who. */

void I8259_HW_INIT(unsigned vec1, unsigned vec2)
{
	static char got_resources = 0;
	if (!__CMPXCHGC(&got_resources, 0, 1)) {
		I8259_GET_RESOURCES();
	}
	/*
	 * io_outb_p - this has to work on a wide range of PC hardware.
	 */
	io_outb_p(0x20, 0x11);	/* ICW1: select 8259A-1 init */
	io_outb_p(0x21, vec1);	/* ICW2: 8259A-1 IR0-7 mapped to I8259_IRQ_INT */
	io_outb_p(0x21, 0x04);	/* 8259A-1 (the master) has a slave on IR2 */
	io_outb_p(0x21, 0x01);	/* master expects normal EOI */

	io_outb_p(0xa0, 0x11);	/* ICW1: select 8259A-2 init */
	io_outb_p(0xa1, vec2);	/* ICW2: 8259A-2 IR0-7 mapped to I8259_IRQ_INT + 8 */
	io_outb_p(0xa1, 0x02);	/* 8259A-2 is a slave on master's IR2 */
	io_outb_p(0xa1, 0x01);	/* (slave's support for AEOI in flat mode
				    is to be investigated) */

	KERNEL$UDELAY(100);	/* wait for 8259A to initialize */

	io_outb_p(0x21, 0xff);
	io_outb_p(0xa1, 0xff);
}

void I8259_IRQ_INIT(void)
{
	int i;

	I8259_HW_INIT((long)I8259_IRQ_INT, (long)I8259_IRQ_INT + 8);

	IRQC.n_irqs = N_I8259_IRQS;
	IRQC.irq_get_handle = I8259_GET_HANDLE;
	IRQC.irq_request_rt = I8259_REQUEST_RT_IRQ;
	IRQC.irq_request_ast = I8259_REQUEST_AST_IRQ;
	IRQC.irq_startup = I8259_IRQ_STARTUP;
	IRQC.irq_shutdown = I8259_IRQ_SHUTDOWN;
	IRQC.di_hw = I8259_DI_HW;
	IRQC.ei_hw = I8259_EI_HW;

	for (i = 0; i < N_I8259_IRQS; i++) if (I8259_VALID_IRQ(i)) {
		I8259_DISABLE_VECTOR(i)();
		set_idt_gate((long)I8259_IRQ_INT + i, I8259_IRQ_STUBS[I8259_IRQ_2_IDX(i)].c);
	}

	memcpy(KERNEL$MASK_IRQ, I8259_MASK_IRQ_STUB, I8259_MASK_IRQ_STUB_END - I8259_MASK_IRQ_STUB);
	memcpy(KERNEL_MASK_IRQ_DI, I8259_MASK_IRQ_STUB, I8259_MASK_IRQ_STUB_END - I8259_MASK_IRQ_STUB);
	memcpy(KERNEL$UNMASK_IRQ, I8259_UNMASK_IRQ_STUB, I8259_UNMASK_IRQ_STUB_END - I8259_UNMASK_IRQ_STUB);
}
