#include <ARCH/CPU.H>
#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <SPAD/TIMER.H>
#include <SYS/TYPES.H>
#include <SPAD/SYSLOG.H>
#include <KERNEL/CONSOLE.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/BOOTCFG.H>
#include <KERNEL/ASM.H>
#include <KERNEL/MTRR.H>

#define MCA_CHECK_TIME		(10 * JIFFIES_PER_SECOND)
#define MCA_READ_RETRIES	5
#define MAX_MCES		5

int MCE_HANDLER(int kernel);

static char memory_limit_k6;
static char cpu_has_mca;
static int mce_first_read_bank;
static int mce_first_write_bank;
static int mce_banks;
static int mce_extended_msrs;

static TIMER MCA_TIMER;
static char addr[__MAX_STR_LEN];
static char misc[__MAX_STR_LEN];

static int MCA_CHECK_FOR_PCC(void)
{
	int i;
	for (i = mce_first_read_bank; i < mce_banks; i++) {
		__u64 st;
		if (__unlikely(KERNEL$READ_MSR(IA32_MC0_STATUS + i * 4, &st)))
			continue;
		if (__likely(!(st & IA32_MC_STATUS_VAL)))
			continue;
		if (__unlikely((st & IA32_MC_STATUS_PCC) != 0))
			return 1;
	}
	return 0;
}

static void MCA_CLEAR_BANKS(void)
{
	int i;
	for (i = mce_first_read_bank; i < mce_banks; i++) {
		__u64 st;
		if (__unlikely(KERNEL$READ_MSR(IA32_MC0_STATUS + i * 4, &st)))
			continue;
		if (__likely(!(st & IA32_MC_STATUS_VAL)))
			continue;
		KERNEL$WRITE_MSR(IA32_MC0_STATUS + i * 4, 0x0000000000000000ULL);
		if (FEATURE_TEST(FEATURE_CPUID)) {
			__asm__ volatile ("			;\
				PUSHL	%%EBX			;\
				XORL	%%EAX, %%EAX		;\
				CPUID				;\
				POPL	%%EBX			;\
			":::"ax","cx","dx","cc");
		}
	}
}

static void MCA_SCAN_BANKS(int exc)
{
	int i;
	for (i = mce_first_read_bank; i < mce_banks; i++) {
		__u64 st;
		unsigned retries = MCA_READ_RETRIES;
		retry:
		if (__unlikely(KERNEL$READ_MSR(IA32_MC0_STATUS + i * 4, &st)))
			continue;
		if (__likely(!(st & IA32_MC_STATUS_VAL)))
			continue;
		addr[0] = 0;
		if (st & IA32_MC_STATUS_ADDRV) {
			__u64 ad;
			if (!KERNEL$READ_MSR(IA32_MC0_ADDR + i * 4, &ad))
				_snprintf(addr, sizeof addr, ", ADDR: %016LX", ad);
		}
		misc[0] = 0;
		if (st & IA32_MC_STATUS_MISCV) {
			__u64 ad;
			if (!KERNEL$READ_MSR(IA32_MC0_MISC + i * 4, &ad))
				_snprintf(misc, sizeof misc, ", MISC: %016LX", ad);
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "%s, BANK %d: STATUS: %016LX%s%s", !exc ? "NON FATAL CORRECTABLE ERROR" : st & IA32_MC_STATUS_PCC ? "FATAL ERROR" : exc == 1 ? "NON FATAL CORRECTABLE ERROR" : "ERROR", i, st, addr, misc);
		if (!exc) {
			KERNEL$WRITE_MSR(IA32_MC0_STATUS + i * 4, 0x0000000000000000ULL);
			if (FEATURE_TEST(FEATURE_CPUID)) {
				__asm__ volatile ("			;\
					PUSHL	%%EBX			;\
					XORL	%%EAX, %%EAX		;\
					CPUID				;\
					POPL	%%EBX			;\
				":::"ax","cx","dx","cc");
			}
			if (--retries > 0) goto retry;
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "ERROR CANNOT BE CLEARED IN %d RETRIES", MCA_READ_RETRIES);
		}
	}
}

static jiffies_t last_mce = 0;
static int n_mces = 0;

int MCE_HANDLER(int kernel)
{
	if (!cpu_has_mca) {
		__u64 ad, ty;
		int will_halt = 0;
		jiffies_t j = KERNEL$GET_JIFFIES();
		if (j == last_mce) {
			if (++n_mces >= MAX_MCES) {
				will_halt = 1;
			}
		} else {
			last_mce = j;
			n_mces = 0;
		}
		if (will_halt) {
			PREPARE_FOR_CRASH();
		}
		if (KERNEL$READ_MSR(IA32_P5_MC_ADDR, &ad) || KERNEL$READ_MSR(IA32_P5_MC_TYPE, &ty)) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION");
		else KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION, ADDR %016LX, TYPE %016LX", ad, ty);
		if (will_halt) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "TOO MANY MACHINE CHECK EXCEPTIONS - HALTING");
			return 2;
		}
		TLB_INVD_G();
		return 0;
	} else {
		__u64 st;
		int will_halt = 0;
		if (KERNEL$READ_MSR(IA32_MCG_STATUS, &st)) {
			PREPARE_FOR_CRASH();
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION, UNABLE TO READ STATUS");
			return 2;
		}
		if (MCA_CHECK_FOR_PCC()) will_halt = 2;
		else if (kernel && !(st & IA32_MCG_STATUS_RIPV)) will_halt = 1;
		if (will_halt) {
			PREPARE_FOR_CRASH();
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION, STATUS %016LX", st);
		MCA_SCAN_BANKS(st & IA32_MCG_STATUS_RIPV ? 1 : 2);
		if (will_halt) {
			if (will_halt >= 2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "CPU CONTEXT CORRUPT - HALTING");
			else KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "UNRESTARTABLE ERROR IN KERNEL MODE - HALTING");
			return 2;
		}
		MCA_CLEAR_BANKS();
		st &= ~IA32_MCG_STATUS_MCIP;
		if (KERNEL$WRITE_MSR(IA32_MCG_STATUS, st)) {
			KERNEL$WRITE_MSR(IA32_MCG_STATUS, 0x0000000000000000ULL);
		}
		if (st & IA32_MCG_STATUS_RIPV) {
			TLB_INVD_G();
			return 0;
		} else {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "KILLING CURRENT PROCESS AND TRYING TO CONTINUE");
			TLB_INVD_G();
			return 1;
		}
	}
}

static void MCA_NONFATAL(TIMER *t)
{
	LOWER_SPL(SPL_TIMER);
	MCA_SCAN_BANKS(0);
	KERNEL$SET_TIMER(MCA_CHECK_TIME, &MCA_TIMER);
}

void MACHINE_INIT(void)
{
	unsigned family = 4;
	unsigned model = 0;
	unsigned stepping = 0;
	__u32 cpuid = 0;
	__u32 cpuid2 = 0;
	char vendor[13] = "";
	if (FEATURE_TEST(FEATURE_CPUID)) {
/* do not change EBX, because it can't be used when compiling with -fPIC */
		DO_CPUID0(NULL, vendor);
		DO_CPUID1(&family, &model, &stepping, &cpuid, &cpuid2);
	}

	if (BOOTCFG->PARAM_CACHE_PROBE == PARAM_CACHE_PROBE_AUTO) {
		if (family <= 5)
			BOOTCFG->PARAM_CACHE_PROBE = PARAM_CACHE_PROBE_ENABLE;
		else
			BOOTCFG->PARAM_CACHE_PROBE = PARAM_CACHE_PROBE_DISABLE;
	}

	if (cpuid & CPU_FEATURE_MTRR) MTRR_INIT_GENERIC();

	if (!strcmp(vendor, "GenuineIntel")) {
		/* Pentium 1 F00F bug */
		if (family == 5) {
			void *idt = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(KERNEL$VIRT_2_PHYS(IDT), (char *)IDT_END - (char *)IDT, PAT_RO);
			if (__unlikely(__IS_ERR(idt))) {
				__critical_printf("UNABLE TO CREATE MAPPING FOR IDT TO WORK AROUND PENTIUM BUG: %s", strerror(-__PTR_ERR(idt)));
				HALT_KERNEL();
			}
			IDTR.limit = (char *)IDT_END - (char *)IDT - 1;
			IDTR.base_lo = (__u32)idt;
			IDTR.base_hi = (__u32)idt >> 16;
			__asm__ volatile ("LIDTL IDTR");
		}

		/* XEON errata */
		if (family == 15 && model == 1 && stepping == 1) {
			__u64 misc;
			if (!KERNEL$READ_MSR(IA32_MISC_ENABLE, &misc)) {
				misc |= 1ULL << 9;
				KERNEL$WRITE_MSR(IA32_MISC_ENABLE, misc);
			}
		}

		/* Pentium Pro errata */
		if (family == 6 && model == 1 && stepping < 8) {
			VM_ADD_MEMORY_HOLE(0x70000000, 0x7003ffff);
		}
	}

	memory_limit_k6 = 0;
	if (!strcmp(vendor, "AuthenticAMD")) {
		if (family == 5) {
			if ((model >= 6 && model < 8) || (model == 8 && stepping < 8)) {
				memory_limit_k6 = 1;
			} else if (model == 8 || model == 9 || model == 13) {
				memory_limit_k6 = 2;
				MTRR_INIT_K6();
			}
		} 
		if (family == 6 && ((model == 8 && stepping >= 1) || model > 8)) {
			__u64 clk;
			if (!KERNEL$READ_MSR(MSR_K7_CLK_CTL, &clk)) {
				clk &= ~0xfff00000ULL;
				clk |=  0x20000000ULL;
				KERNEL$WRITE_MSR(MSR_K7_CLK_CTL, clk);
			}
		}
		/* Disable flush filter */
		if (family >= 15 && family <= 17 && FEATURE_TEST(FEATURE_SMP)) {
			__u64 hwcr;
			if (!KERNEL$READ_MSR(MSR_K7_HWCR, &hwcr)) {
				hwcr |= 1ULL << 6;
				KERNEL$WRITE_MSR(MSR_K7_HWCR, hwcr);
			}
		}
	}

	/* enable MCE / MCA */
	cpu_has_mca = 0;
	if (cpuid & CPU_FEATURE_MCE) {
		if (cpuid & CPU_FEATURE_MCA) {
			__u64 cap;
			int i;
			if (KERNEL$READ_MSR(IA32_MCG_CAP, &cap)) {
				goto skip_mca;
			}
			if (cap & IA32_MCG_CAP_MCG_CTL_P) {
				KERNEL$WRITE_MSR(IA32_MCG_CTL, 0xffffffffffffffffULL);
			}
			mce_banks = cap & IA32_MCG_CAP_COUNT;
			if (cap & IA32_MCG_CAP_MCG_EXT_P) {
				mce_extended_msrs = (cap & IA32_MCG_CAP_MCG_EXT_CNT) >> P_IA32_MCG_CAP_MCG_EXT_CNT;
			}
			mce_first_read_bank = 0;
			mce_first_write_bank = 1;
			if (!strcmp(vendor, "GenuineIntel") && family >= 0xf) mce_first_write_bank = 0;
			if (!strcmp(vendor, "AuthenticAMD")) mce_first_read_bank = 1;
			if (mce_first_write_bank)
				KERNEL$WRITE_MSR(IA32_MC0_STATUS + 0 * 4, 0x0000000000000000ULL);
			for (i = mce_first_write_bank; i < mce_banks; i++) {
				KERNEL$WRITE_MSR(IA32_MC0_CTL + i * 4, 0xffffffffffffffffULL);
				KERNEL$WRITE_MSR(IA32_MC0_STATUS + i * 4, 0x0000000000000000ULL);
			}
			cpu_has_mca = 1;
			/*__debug_printf("first: %d, banks: %d, ext: %d\n", mce_first_bank, mce_banks, mce_extended_msrs);*/
			MCA_TIMER.fn = MCA_NONFATAL;
			INIT_TIMER(&MCA_TIMER);
			KERNEL$SET_TIMER(MCA_CHECK_TIME, &MCA_TIMER);
		}
		skip_mca:
		if (!strcmp(vendor, "CentaurHauls")) {
			__u64 fcr1;
			if (!KERNEL$READ_MSR(MSR_IDT_FCR1, &fcr1)) {
				fcr1 |= 1 << 2;
				fcr1 &= ~(1 << 4);
				KERNEL$WRITE_MSR(MSR_IDT_FCR1, fcr1);
			}
		}
		CR_4 |= CR4_MCE;
		__asm__ volatile ("MOVL	%0, %%CR4"::"r"(CR_4));
	}

	/* disable processor serial number */
	if (cpuid & CPU_FEATURE_PN) {
		__u64 sn;
		if (!KERNEL$READ_MSR(MSR_BBL_CR_CTL, &sn)) {
			sn |= MSR_BBL_CR_CTL_NUMBER_DISABLE;
			KERNEL$WRITE_MSR(MSR_BBL_CR_CTL, sn);
		}
	}
}

__u64 MACHINE_SET_MEMORY_LIMIT(__u64 memsize, __u64 *holes, int n_holes)
{
	/* Some BIOSes don't set write allocate. Not setting it has disastrous
	   performance effects (especially on K6-3), compiling is more than
	   twice slower */
	if (memory_limit_k6) {
		int i;

		__u64 efer, whcr;
		__u32 mask_15, mask_limit, mask_zero;
		__u32 whcr_memsize;

		int allow_15 = memsize >= 0xf00000;

		for (i = 0; i < n_holes; i++) {
			__u64 start = holes[i * 2];
			__u64 end = holes[i * 2 + 1];
			if (start == end) continue;
			if (end <= 0x100000) continue;
			if (start >= 0xf00000 && end <= 0x1000000)
				allow_15 = 0;
			else
				if (start < memsize)
					memsize = start;
		}
		memsize >>= 22;

		if (memory_limit_k6 == 1) {
			mask_15 = MSR_K6_WHCR_1_15M;
			mask_limit = MSR_K6_WHCR_1_LIMIT;
			mask_zero = MSR_K6_WHCR_1_ZERO;
		} else {
			mask_15 = MSR_K6_WHCR_2_15M;
			mask_limit = MSR_K6_WHCR_2_LIMIT;
			mask_zero = 0;
		}
		if (memsize > mask_limit >> __BSF(mask_limit))
			memsize = mask_limit >> __BSF(mask_limit);

		if (!KERNEL$READ_MSR(MSR_K6_WHCR, &whcr)) {
			whcr_memsize = ((__u32)whcr & mask_limit) >> __BSF(mask_limit);
			if (whcr_memsize < (__u32)memsize) {
				whcr &= ~(__u64)mask_limit;
				whcr |= (__u32)memsize << __BSF(mask_limit);
			}
			if (allow_15)
				whcr |= mask_15;
			else
				whcr &= ~(__u64)mask_15;
			START_CACHEMODE_MODIFY();
			KERNEL$WRITE_MSR(MSR_K6_WHCR, whcr);
			END_CACHEMODE_MODIFY();
			if (whcr_memsize > memsize)
				memsize = whcr_memsize;
		}
		if (memory_limit_k6 == 2 && !KERNEL$READ_MSR(IA32_EFER, &efer)) {
			efer |= MSR_K6_EFER_DPE;
			/* efer &= ~MSR_K6_EFER_L2D; */
			START_CACHEMODE_MODIFY();
			KERNEL$WRITE_MSR(IA32_EFER, efer);
			END_CACHEMODE_MODIFY();
		}
		memsize <<= 22;
	}
	return memsize;
}
