#include <SYS/TYPES.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/VM.H>
#include <KERNEL/VMDEF.H>
#include <KERNEL/ASM.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/FEATURE.H>
#include <KERNEL/BOOTCFG.H>
#include <STDLIB.H>
#include <STRING.H>

#include <KERNEL/MPS.H>

/*#define FORCE_DEFAULT	MP_FP_DEFAULT_CFG_INTEGRATED_ISA_PCI*/

static int mp_log;
static const char *mp_error = "";

__u8 MPS_CHECKSUM(const void *ptrv, size_t len)
{
	const __u8 *ptr = ptrv;
	__u8 chs = 0;
	while (len) {
		chs += *ptr;
		ptr++;
		len--;
	}
	return chs;
}

MP_FLOATING_POINTER *MPS_FP = NULL;
unsigned MPS_FP_SIZE = 0;
MP_TABLE *MPS_TABLE = NULL;
unsigned MPS_TABLE_SIZE = 0;

static struct {
	MP_TABLE mp;
	MP_PROCESSOR proc0;
	MP_PROCESSOR proc1;
	MP_IO_APIC io_apic;
} default_mptable;

static int find_fp(unsigned long base, size_t len)
{
	MP_FLOATING_POINTER *ptr = (MP_FLOATING_POINTER *)base;
	while (len >= sizeof(MP_FLOATING_POINTER)) {
		if (!memcmp(ptr->signature, MP_FP_SIGNATURE, 4)) {
			if (ptr->fp_length < 1) goto bad_fp;
			if (ptr->fp_length * 0x10 > len) goto bad_fp;
			if (MPS_CHECKSUM(ptr, ptr->fp_length * 0x10)) goto bad_fp;
			MPS_FP = ptr;
			MPS_FP_SIZE = ptr->fp_length * 0x10;
			return 0;
		}
		bad_fp:
		ptr++;
		len -= sizeof(MP_FLOATING_POINTER);
	}
	return -1;
}

static void make_default_mptable(int version)
{
	memset(&default_mptable, 0, sizeof default_mptable);
	memcpy(default_mptable.mp.signature, MP_T_SIGNATURE, 4);
	default_mptable.mp.base_length = __16CPU2LE(sizeof default_mptable);
	default_mptable.mp.entry_count = __16CPU2LE(3);
	default_mptable.mp.local_apic_addr = __32CPU2LE(0xfee00000);
	default_mptable.proc0.type = MP_ENTRY_TYPE_PROCESSOR;
	default_mptable.proc0.local_apic_id = 0;
	default_mptable.proc0.local_apic_version = version < 5 ? 0x01 : 0x10;
	default_mptable.proc0.flags = MP_PROCESSOR_FLAGS_EN | MP_PROCESSOR_FLAGS_BP;
	default_mptable.proc1.type = MP_ENTRY_TYPE_PROCESSOR;
	default_mptable.proc1.local_apic_id = 1;
	default_mptable.proc1.local_apic_version = version < 5 ? 0x01 : 0x10;
	default_mptable.proc1.flags = MP_PROCESSOR_FLAGS_EN;
	default_mptable.io_apic.type = MP_ENTRY_TYPE_IO_APIC;
	default_mptable.io_apic.io_apic_id = 2;
	default_mptable.io_apic.io_apic_version = version < 5 ? 0x01 : 0x10;
	default_mptable.io_apic.flags = MP_IO_APIC_FLAGS_EN;
	default_mptable.io_apic.io_apic_addr = __32CPU2LE(0xfec00000);

	default_mptable.mp.checksum = -MPS_CHECKSUM(&default_mptable, sizeof default_mptable);
}

static void FIND_MP(void)
{
	if (*(__u16 *)0x40e)
		if (!find_fp(*(__u16 *)0x40e << 4, 1024))
			goto got_fp;

	if (!find_fp(0x9fc00, 0x400))
		goto got_fp;
	if (!find_fp(0xf0000, 0x10000))
		goto got_fp;
	if (!find_fp(0, 0x400))
		goto got_fp;
	
	return;

	got_fp:

	VM_ADD_MEMORY_HOLE((unsigned long)MPS_FP, (unsigned long)MPS_FP + MPS_FP_SIZE);

#ifdef FORCE_DEFAULT
	make_default_mptable(FORCE_DEFAULT);
	MPS_TABLE = &default_mptable.mp;
#else
	if (!__32LE2CPU(MPS_FP->mp_table)) {
		if (!MPS_FP->default_cfg) {
			mp_log = __SYSLOG_SW_ERROR;
			mp_error = "NO MP TABLE IN FLOATING POINTER AND NO DEFAULT";
			return;
		}
		make_default_mptable(MPS_FP->default_cfg);
		MPS_TABLE = &default_mptable.mp;
	} else {
		MPS_TABLE = (void *)__32LE2CPU(MPS_FP->mp_table);
	}
#endif

	if (memcmp(MPS_TABLE->signature, MP_T_SIGNATURE, 4)) {
		mp_log = __SYSLOG_SW_ERROR;
		mp_error = "BAD MP TABLE SIGNATURE";
		return;
	}
	if (__16LE2CPU(MPS_TABLE->base_length) < sizeof(MP_TABLE)) {
		mp_log = __SYSLOG_SW_ERROR;
		mp_error = "BAD MP TABLE BASE LENGTH";
		return;
	}
	if (MPS_CHECKSUM(MPS_TABLE, __16LE2CPU(MPS_TABLE->base_length))) {
		mp_log = __SYSLOG_SW_ERROR;
		mp_error = "BAD MP TABLE CHECKSUM";
		return;
	}

	if ((__u8)(MPS_CHECKSUM((__u8 *)MPS_TABLE + __16LE2CPU(MPS_TABLE->base_length), __16LE2CPU(MPS_TABLE->ext_length)) + MPS_TABLE->ext_checksum)) {
		mp_log = __SYSLOG_SW_WARNING;
		mp_error = "BAD EXTENDED MP TABLE CHECKSUM";
	}

	MPS_TABLE_SIZE = __16LE2CPU(MPS_TABLE->base_length);

	if (MPS_TABLE_SIZE > PAGE_CLUSTER_SIZE) {
		mp_log = __SYSLOG_SW_INCOMPATIBILITY;
		mp_error = "TOO BIG MP TABLE";
		MPS_TABLE_SIZE = 0;
		return;
	}
	if (MPS_FP_SIZE > PAGE_CLUSTER_SIZE)
		MPS_FP_SIZE = sizeof(MP_FLOATING_POINTER);

	if (MPS_TABLE != &default_mptable.mp)
		VM_ADD_MEMORY_HOLE((unsigned long)MPS_TABLE, (unsigned long)MPS_TABLE + MPS_TABLE_SIZE);

	return;
}

static const unsigned char mp_entry_sizes[5] = {
	sizeof(MP_PROCESSOR),
	sizeof(MP_BUS),
	sizeof(MP_IO_APIC),
	sizeof(MP_IO_INTERRUPT),
	sizeof(MP_LOCAL_INTERRUPT),
};

static __u8 *mp_find_last;
static __u16 mp_find_remaining;
static __u8 mp_find_type;
static char *mp_find_error;

void MPS_FIND_RESET(__u8 type)
{
	mp_find_last = (__u8 *)(MPS_TABLE + 1);
	mp_find_remaining = __16LE2CPU(MPS_TABLE->entry_count);
	mp_find_type = type;
	mp_find_error = NULL;
}

void *MPS_FIND_ENTRY(void)
{
	while (mp_find_last < (__u8 *)MPS_TABLE + __16LE2CPU(MPS_TABLE->base_length) && mp_find_remaining) {
		__u8 *ptr = mp_find_last;
		unsigned char type = *ptr;
		if (type >= sizeof(mp_entry_sizes) / sizeof(*mp_entry_sizes)) {
			mp_find_error = "INVALID TYPE IN MP TABLE";
			return NULL;
		}
		mp_find_last += mp_entry_sizes[type];
		mp_find_remaining--;
		if (mp_find_last > (__u8 *)MPS_TABLE + __16LE2CPU(MPS_TABLE->base_length)) {
			mp_find_error = "MP ENTRY PAST TABLE END";
			return NULL;
		}
		if (type == mp_find_type) {
			return ptr;
		}
	}
	return NULL;
}

int MPS_DEFAULT(void)
{
	if (!MPS_TABLE_SIZE)
		return 0;
#ifdef FORCE_DEFAULT
	return FORCE_DEFAULT;
#else
	if (!__32LE2CPU(MPS_FP->mp_table))
		return MPS_FP->default_cfg;
	return 0;
#endif
}

static void MPS_PREINIT(void)
{
	if (MPS_TABLE_SIZE) {
		MP_PROCESSOR *proc;
		int procs = 0;
		MPS_FIND_RESET(MP_ENTRY_TYPE_PROCESSOR);
		while ((proc = MPS_FIND_ENTRY())) {
			if (proc->flags & MP_PROCESSOR_FLAGS_EN) {
				procs++;
			}
		}
#if __KERNEL_SUPPORT_SMP
		if (procs >= 2 && BOOTCFG->PARAM_SMP != PARAM_SMP_DISABLE) {
			FEATURE_ADD(FEATURE_SMP);
			if (procs >= CPU_ID_LIMIT) CPU_ID_LIMIT = procs;
		}
#endif
		if (mp_find_error) {
			mp_log = __SYSLOG_SW_ERROR;
			mp_error = mp_find_error;
		}
	}
}

void MPS_DETECT(void)
{
	FIND_MP();
	MPS_PREINIT();
}

static void copy_from_phys(void *dest, __p_addr src, size_t len)
{
	while (len) {
		void *p;
		size_t l = len;
		if (__unlikely(((unsigned long)src & (PG_SIZE * PG_BANK - 1)) + l > PG_SIZE * PG_BANK))
			l = PG_SIZE * PG_BANK - ((unsigned long)src & (PG_SIZE * PG_BANK - 1));
		p = KERNEL$MAP_PHYSICAL_BANK(src);
		memcpy(dest, p, l);
		KERNEL$UNMAP_PHYSICAL_BANK(p);
		dest = (char *)dest + l;
		src += l;
		len -= l;
	}
}

void MPS_INIT(void)
{
	unsigned long fp_phys, mp_phys;
	if (*mp_error)
		KERNEL$SYSLOG(mp_log, "KERNEL", mp_error);
	/*__debug_printf("MPS_FP @ %p, MPS_TABLE @ %p\n", MPS_FP, MPS_TABLE);*/
	if (!MPS_TABLE_SIZE)
		return;
	fp_phys = (unsigned long)MPS_FP;
	mp_phys = (unsigned long)MPS_TABLE;
	MPS_FP = malloc(MPS_FP_SIZE);
	if (!MPS_FP) {
		__critical_printf("CAN'T ALLOCATE MEMORY FOR MP FLOATING POINTER\n");
		HALT_KERNEL();
	}
	copy_from_phys(MPS_FP, fp_phys, MPS_FP_SIZE);
	MPS_TABLE = malloc(MPS_TABLE_SIZE);
	if (!MPS_TABLE) {
		__critical_printf("CAN'T ALLOCATE MEMORY FOR MP TABLE\n");
		HALT_KERNEL();
	}
	copy_from_phys(MPS_TABLE, mp_phys, MPS_TABLE_SIZE);
	/*{
		int i;
		for (i = 0; i < MPS_TABLE_SIZE; i++) {
			__debug_printf("%02x ", ((unsigned char *)MPS_TABLE)[i]);
		}
		__debug_printf("\n");
	}*/
}
