#include <SYS/TYPES.H>
#include <ARCH/IO.H>
#include <ARCH/CPU.H>
#include <ARCH/BSF.H>
#include <KERNEL/VM_ARCH.H>
#include <SPAD/SYSLOG.H>

#include <KERNEL/MTRR.H>

static unsigned MTRR_NUM = 0;
static unsigned MTRR_MIN_SIZE = 0;
static __u64 MTRR_PHYS_SIZE = 0;
static __u8 MTRR_DEFAULT_PAT = PAT_WB;
static unsigned MTRR_MASK = 0;
static int (*MTRR_READ)(int, __u64 *, __u64 *, __u8 *);
static int (*MTRR_WRITE)(int, __u64 , __u64, __u8);
static int (*MTRR_DISABLE_ENABLE)(unsigned) = NULL;

/* --- Cache flusher --- */

static __u32 orig_cr0 = 0;

void START_CACHEMODE_MODIFY(void)
{
	KERNEL$DI();
	if (orig_cr0)
		KERNEL$SUICIDE("START_CACHEMODE_MODIFY: CALLED TWICE (SAVED CR0 == %08X)", (unsigned)orig_cr0);
	__asm__ volatile ("MOVL %%CR0, %0":"=r"(orig_cr0)::"memory");
	__asm__ volatile ("MOVL %0, %%CR0"::"r"((orig_cr0 | CR0_CD) & ~CR0_NW):"memory");
	CACHE_WBINVD_WHEN_NO_SELFSNOOP();
	TLB_INVD_G();
	if (MTRR_DISABLE_ENABLE) MTRR_DISABLE_ENABLE(0);
}

void END_CACHEMODE_MODIFY(void)
{
	if (!orig_cr0)
		KERNEL$SUICIDE("END_CACHEMODE_MODIFY: CALLED WITHOUT START_CACHEMODE_MODIFY");
	if (MTRR_DISABLE_ENABLE) MTRR_DISABLE_ENABLE(1);
	/*CACHE_WBINVD_WHEN_NO_SELFSNOOP();*/
	__asm__ volatile ("WBINVD":::"memory");
	TLB_INVD_G();
	__asm__ volatile ("MOVL %0, %%CR0"::"r"(orig_cr0):"memory");
	orig_cr0 = 0;
	KERNEL$EI();
}

/* --- INTEL --- */

static __u64 ia32_mtrr_cap;
static __u64 ia32_mtrr_def_type;

static const __u8 ia32_mtrr2pat[8] = { PAT_UC, PAT_WC, 0xff, 0xff, PAT_WT, PAT_WP, PAT_WB, 0xff };
static const __u8 ia32_pat2mtrr[8] = { IA32_MEMORY_TYPE_WB, IA32_MEMORY_TYPE_WT, IA32_MEMORY_TYPE_UC, IA32_MEMORY_TYPE_UC, IA32_MEMORY_TYPE_WB, IA32_MEMORY_TYPE_WT, IA32_MEMORY_TYPE_WC, IA32_MEMORY_TYPE_WP };

static int GENERIC_MTRR_READ(int idx, __u64 *start, __u64 *len, __u8 *pat)
{
	int r;
	__u64 base, mask;
	if ((r = KERNEL$READ_MSR(IA32_MTRR_PHYSBASE0 + idx * 2, &base))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ MTRR %X BASE: %s", idx, strerror(-r));
		return r;
	}
	if ((r = KERNEL$READ_MSR(IA32_MTRR_PHYSMASK0 + idx * 2, &mask))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ MTRR %X MASK: %s", idx, strerror(-r));
		return r;
	}
	if (!(mask & IA32_MTRR_PHYSMASK_V) || !(mask & IA32_MTRR_PHYSMASK_MASK)) return -ENOENT;
	if ((base & IA32_MTRR_PHYSBASE_TYPE) >= 8) {
		invl:
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "INVALID MTRR %X: %"__64_format"X, %"__64_format"X", idx, base, mask);
		return -EINVAL;
	}
	*pat = ia32_mtrr2pat[base & IA32_MTRR_PHYSBASE_TYPE];
	if (*pat == 0xff) goto invl;
	*start = base & mask & IA32_MTRR_PHYSBASE_BASE;
	*len = 4096;
	while (!(*len & mask)) *len *= 2;
	return 0;
}

static int GENERIC_MTRR_WRITE(int idx, __u64 start, __u64 len, __u8 pat)
{
	int r;
	__u64 base, mask;
	if (!len) {
		if ((r = KERNEL$WRITE_MSR(IA32_MTRR_PHYSMASK0 + idx * 2, 0))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO CLEAR MTRR %X: %s", idx, strerror(-r));
			return r;
		}
		return 0;
	}
	base = start | ia32_pat2mtrr[pat];
	mask = (MTRR_PHYS_SIZE - len) | IA32_MTRR_PHYSMASK_V;
	if ((r = KERNEL$WRITE_MSR(IA32_MTRR_PHYSBASE0 + idx * 2, base))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO WRITE MTRR %X BASE (%"__64_format"X, %"__64_format"X): %s", idx, base, mask, strerror(-r));
		return r;
	}
	if ((r = KERNEL$WRITE_MSR(IA32_MTRR_PHYSMASK0 + idx * 2, mask))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO WRITE MTRR %X MASK (%"__64_format"X, %"__64_format"X): %s", idx, base, mask, strerror(-r));
		return r;
	}
	return 0;
}

static int GENERIC_MTRR_DISABLE_ENABLE(unsigned en)
{
	int r;
	ia32_mtrr_def_type = (ia32_mtrr_def_type & ~(__u64)IA32_MTRR_DEF_TYPE_E) | (en * IA32_MTRR_DEF_TYPE_E);
	if ((r = KERNEL$WRITE_MSR(IA32_MTRR_DEF_TYPE, ia32_mtrr_def_type)))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO WRITE MTRR DEFAULT TYPE %"__64_format"X: %s", ia32_mtrr_def_type, strerror(-r));
	return r;
}

void MTRR_INIT_GENERIC(void)
{
	int i, r;
	if (MTRR_NUM) return;
	if ((r = KERNEL$READ_MSR(IA32_MTRRCAP, &ia32_mtrr_cap))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ IA32_MTRRCAP: %s", strerror(-r));
		return;
	}
	if ((r = KERNEL$READ_MSR(IA32_MTRR_DEF_TYPE, &ia32_mtrr_def_type))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ IA32_MTRR_DEF_TYPE: %s", strerror(-r));
		return;
	}
	if (!(ia32_mtrr_def_type & IA32_MTRR_DEF_TYPE_E)) return;
	if ((ia32_mtrr_def_type & IA32_MTRR_DEF_TYPE_TYPE) >= 8) {
		inval:
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "INVALID DEFAULT MTRR TYPE: %"__64_format"X", ia32_mtrr_def_type);
		return;
	}
	MTRR_DEFAULT_PAT = ia32_mtrr2pat[ia32_mtrr_def_type & IA32_MTRR_DEF_TYPE_TYPE];
	if (MTRR_DEFAULT_PAT == 0xff) goto inval;
	MTRR_MIN_SIZE = 4096;
	MTRR_NUM = (ia32_mtrr_cap & IA32_MTRRCAP_VCNT) >> __BSF_CONST(IA32_MTRRCAP_VCNT);
	MTRR_MASK = (1 << PAT_WB) | (1 << PAT_WT) | (1 << PAT_UC) | (1 << PAT_WP);
	if (ia32_mtrr_cap & IA32_MTRRCAP_WC) MTRR_MASK |= 1 << PAT_WC;
	MTRR_PHYS_SIZE = (__u64)1 << 36;
	for (i = 0; i < MTRR_NUM; i++) {
		__u64 base, mask;
		if ((r = KERNEL$READ_MSR(IA32_MTRR_PHYSBASE0 + i * 2, &base))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ MTRR %X BASE: %s", i, strerror(-r));
			bad_mtr:
			MTRR_NUM = i;
			break;
		}
		if ((r = KERNEL$READ_MSR(IA32_MTRR_PHYSMASK0 + i * 2, &mask))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ MTRR %X MASK: %s", i, strerror(-r));
			goto bad_mtr;
		}
		if (mask & IA32_MTRR_PHYSMASK_V && mask & IA32_MTRR_PHYSMASK_MASK) {
			if (mask >= MTRR_PHYS_SIZE) {
				MTRR_PHYS_SIZE = (__u64)2 << __BSR(mask);
			}
		}
	}
	MTRR_READ = GENERIC_MTRR_READ;
	MTRR_WRITE = GENERIC_MTRR_WRITE;
	MTRR_DISABLE_ENABLE = GENERIC_MTRR_DISABLE_ENABLE;
}

/* --- AMD K6 --- */

static __u64 k6_uwccr, k6_efer;

static int K6_MTRR_READ(int idx, __u64 *start, __u64 *len, __u8 *pat)
{
	__u32 val = k6_uwccr >> (32 * idx);
	if (val & MSR_K6_UWCCR_UC) *pat = PAT_UC;
	else if (val & MSR_K6_UWCCR_WC) *pat = PAT_WC;
	else return -ENOENT;
	*start = val & MSR_K6_UWCCR_BASE;
	*len = (val & MSR_K6_UWCCR_MASK) << 15;
	*start &= *len;
	*len = ((__u64)1 << 32) - *len;
	if (*len & (*len - 1)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "INVALID K6 MTRR CONTENT: %"__64_format"X", k6_uwccr);
		return -EINVAL;
	}
	return 0;
}

static int K6_MTRR_WRITE(int idx, __u64 start, __u64 len, __u8 pat)
{
	int r;
	__u32 val;
	__u64 new_k6_uwccr;
	if (!len) {
		val = 0;
	} else {
		val = start | ((-len >> 15) & MSR_K6_UWCCR_MASK);
		if (pat == PAT_UC && !(k6_efer & MSR_K6_EFER_SEWBED)) val |= MSR_K6_UWCCR_UC;
		else if (pat == PAT_WC) val |= MSR_K6_UWCCR_WC;
		else return -EINVAL;
	}
	new_k6_uwccr = k6_uwccr;
	new_k6_uwccr &= ~((__u64)0xffffffff << (32 * idx));
	new_k6_uwccr |= (__u64)val << (32 * idx);
	r = KERNEL$WRITE_MSR(MSR_K6_UWCCR, new_k6_uwccr);
	if (r) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO WRITE K6 MTRR %"__64_format"X: %s", new_k6_uwccr, strerror(-r));
		return r;
	}
	k6_uwccr = new_k6_uwccr;
	return 0;
}

void MTRR_INIT_K6(void)
{
	int r;
	if (MTRR_NUM) return;
	if ((r = KERNEL$READ_MSR(MSR_K6_UWCCR, &k6_uwccr))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ K6 MTRR: %s", strerror(-r));
		return;
	}
	if ((r = KERNEL$READ_MSR(IA32_EFER, &k6_efer))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO READ K6 EFER: %s", strerror(-r));
		return;
	}
	MTRR_NUM = 2;
	MTRR_MIN_SIZE = 131072;
	MTRR_PHYS_SIZE = (__u64)1 << 32;
	MTRR_READ = K6_MTRR_READ;
	MTRR_WRITE = K6_MTRR_WRITE;
	k6_efer &= ~MSR_K6_EFER_GEWBED;
	/* If BIOS set some MTRR as UC, don't enable SEWBED, this would break
	   BIOS' assumption about ordering on that memory area */
	if (!(k6_uwccr & (MSR_K6_UWCCR_UC | ((__u64)MSR_K6_UWCCR_UC << 32))))
		k6_efer |= MSR_K6_EFER_SEWBED;
	MTRR_MASK = 1 << PAT_WC;
	if (!(k6_efer & MSR_K6_EFER_SEWBED))
		MTRR_MASK |= 1 << PAT_UC;
	if ((r = KERNEL$WRITE_MSR(IA32_EFER, k6_efer))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO WRITE K6 EFER %"__64_format"X: %s", k6_efer, strerror(-r));
		return;
	}
}

/* --- arch-independent code --- */

static unsigned GET_MASK(__u64 start, __u64 len, int *overlay, int *span, int *fre)
{
	__u64 s, l;
	__u8 p;
	int i, r;
	unsigned mask = 0;
	*overlay = 0;
	*span = 0;
	*fre = 0;
	for (i = 0; i < MTRR_NUM; i++) {
		if ((r = MTRR_READ(i, &s, &l, &p))) {
			if (r == -ENOENT) (*fre)++;
			continue;
		}
		if (start + len > s && start < s + l) {
			if (s < start || s + l > start + len) (*overlay)++;
			else (*span)++;
			mask |= 1 << p;
		}
	}
	while (len) {
		for (i = 0; i < MTRR_NUM; i++) {
			if (MTRR_READ(i, &s, &l, &p)) continue;
			if (start >= s && start < s + l) {
				__u64 diff = s + l - start;
				start += diff;
				len -= diff;
				goto cont;
			}
		}
		mask |= 1 << MTRR_DEFAULT_PAT;
		break;
		cont:;
	}
	return mask;
}

unsigned MTRR_GET_MASK(__u64 start, __u64 len)
{
	int overlay, span, fre;
	return GET_MASK(start, len, &overlay, &span, &fre);
}

int MTRR_TRY_TO_SET(__u64 start, __u64 len, unsigned pat_mask, int pat_pref)
{
	int make_new = 0;
	int r, i, new;
	unsigned current_mask;
	int overlay, span, fre;
	current_mask = GET_MASK(start, len, &overlay, &span, &fre);
	if (!(current_mask & ~pat_mask)) {
		if (pat_mask & (1 << MTRR_DEFAULT_PAT) && span) {
			/* we can free MTRRs */
			goto free_mtrrs;
		}
		return 0;
	}
	if ((len | start) & (len - 1) || len < MTRR_MIN_SIZE || start >= MTRR_PHYS_SIZE) return -EINVAL;
	if (overlay) return -EBUSY;
	if (pat_pref == -1) return 0;
	if (!(MTRR_MASK & (1 << pat_pref))) return -EOPNOTSUPP;
	if (!(pat_mask & (1 << MTRR_DEFAULT_PAT))) {
		make_new = 1;
	}
	if (make_new && !span && !fre) return -ENOSPC;

	if (make_new && (len & (PG_SIZE * PG_BANK - 1))) {
		if ((r = VM_SPLIT_BIGPAGE(start))) return r;
	}

	free_mtrrs:
	START_CACHEMODE_MODIFY();
	new = -1;
	for (i = 0; i < MTRR_NUM; i++) {
		__u64 s, l;
		__u8 p;
		if ((r = MTRR_READ(i, &s, &l, &p))) {
			if (r == -ENOENT && new == -1) new = i;
			continue;
		}
		if (s >= start && s + l <= start + len) {
			if (!MTRR_WRITE(i, 0, 0, 0))
				if (new == -1) new = i;
		}
	}
	if (!make_new) r = 0;
	else if (new != -1) r = MTRR_WRITE(new, start, len, pat_pref);
	else {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CPU", "UNABLE TO FIND PLACE FOR MTRR %"__64_format"X, %"__64_format"X, %X", start, len, (unsigned)pat_pref);
		r = -EIO;
	}
	END_CACHEMODE_MODIFY();

	return r;
}

int MTRR_CHECK_SPLIT(__u64 start, __u64 len)
{
	int i;
	for (i = 0; i < MTRR_NUM; i++) {
		__u64 s, l;
		__u8 p;
		if (MTRR_READ(i, &s, &l, &p)) continue;
		if ((s > start && s < start + len) ||
		    (s + l > start && s + l < start + len))
			return 1;
	}
	return 0;
}
