#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <KERNEL/ASM.H>
#include <KERNEL/TIMER.H>

#include <KERNEL/BOOTCFG.H>
#include <KERNEL/TIME.H>
#include <ARCH/TIME.H>

#include "TIMEREG.H"

#define I8254_FREQ	1193181

#define TIMER_DIV	((I8254_FREQ + BOOTCFG->PARAM_TICK / 2) / BOOTCFG->PARAM_TICK)

volatile jiffies_t __JIFFIES = 0;

static IRQ_HANDLE *TIMER_IRQ_CONTROL;

extern __u32 TIME_MAX_FIXUP_1, TIME_MAX_FIXUP_2, TIME_MUL_FIXUP;
extern __u8 TIME_SHR_FIXUP;

static void WAIT_LOOP(int c)
{
	/* nops are for jmp prediction anomaly on Pentium... we won't get as cool bogomips values as on Linux :-) */
	__asm__ volatile(".ALIGN 16; 1: DECL %%EAX; NOP; NOP; NOP; NOP; NOP; NOP; NOP; NOP; JNS 1b":"=a"(c):"a"(c):"cc");
}

/* default values ... they are used before timer interrupt is enabled and real values calibrated */

static int cpu_mhz = 100;
static int bogomips = 160;

#define UDELAY_LIMIT	10000

void KERNEL$UDELAY(unsigned u)
{
	while (__unlikely(u > UDELAY_LIMIT)) {
		KERNEL$UDELAY(UDELAY_LIMIT);
		u -= UDELAY_LIMIT;
	}
	if (__likely(FEATURE_TEST(FEATURE_TSC))) {
		unsigned t = u * cpu_mhz;
		unsigned a = RDTSC_LO();
		do {
			DO_NOP();
		} while ((RDTSC_LO() - a) <= t);
	} else {
		WAIT_LOOP((u * bogomips) >> 3);
	}
}

void KERNEL$UDELAY_PREPARE(udelay_cookie_t *cookie)
{
	if (FEATURE_TEST(FEATURE_TSC))
		*cookie = RDTSC_LO();
}

void KERNEL$UDELAY_WAIT(udelay_cookie_t *cookie, unsigned u)
{
	unsigned a, t;
	if (__unlikely(u > UDELAY_LIMIT) ||
	    __unlikely(!FEATURE_TEST(FEATURE_TSC))) {
		return KERNEL$UDELAY(u);
	}
	a = *cookie;
	t = u * cpu_mhz;
	do {
		DO_NOP();
	} while ((RDTSC_LO() - a) <= t);
}

int GET_TIME_OF_DAY(struct timeval *tv)
{
	int r;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	__asm__ volatile ("CALL %c1":"=a"(r):"i"(APLACE(APAGE_GET_TIME_OF_DAY)),"0"(tv):"dx","cx","cc","memory");
	LOWER_SPLX(spl);
	return r;
}

static volatile int timer_init_cnt;

static volatile __u64 rdtsc_calibrate;
__u8 FINE_TICKS_SHIFT;
sched_unsigned FINE_TICKS_PER_MINUTE;

static DECL_RT_IRQ_HANDLER(TIMER_INIT_IRQ)
{
	/*__debug_printf("timer init irq\n");*/
	if (__likely(FEATURE_TEST(FEATURE_TSC))) {
		__u64 r = RDTSC();
		if (timer_init_cnt == 1) rdtsc_calibrate = r;
		if (timer_init_cnt == 2) rdtsc_calibrate = r - rdtsc_calibrate;
	}
	timer_init_cnt++;
	IRQ_RETURN;
}

void FIXUP_TIMER_ASM(unsigned jiffies_step);

void TIMER_INIT(void)
{
	static IO_RANGE iorange1;
	static IO_RANGE iorange2;
	int r;
	if (BOOTCFG->PARAM_TICK & (BOOTCFG->PARAM_TICK - 1)) {
		__critical_printf("INVALID TICK VALUE %d, MUST BE POWER OF 2\n", BOOTCFG->PARAM_TICK);
		HALT_KERNEL();
	}
	FIXUP_TIMER_ASM(JIFFIES_PER_SECOND / BOOTCFG->PARAM_TICK);
	iorange1.start = TIMER_IO_BASE;
	iorange1.len = TIMER_IO_RANGE;
	iorange1.name = "TIMER";
	if ((r = KERNEL$REGISTER_IO_RANGE(&iorange1))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR TIMER: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	iorange2.start = PPI_PORT_B;
	iorange2.len = 1;
	iorange2.name = "BEEP";
	if ((r = KERNEL$REGISTER_IO_RANGE(&iorange2))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR BEEP: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	io_outb_p(TIMER_CTL_IO, TIMER_CTL_TIMER_0 | TIMER_CTL_ACTION_RW_LSB_MSB | TIMER_CTL_MODE_2);
	io_outb_p(TIMER_0_IO, TIMER_DIV & 0xff);
	io_outb_p(TIMER_0_IO, TIMER_DIV >> 8);
	timer_init_cnt = 0;
	if ((r = KERNEL$REQUEST_IRQ(0, &TIMER_IRQ_CONTROL, IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_EXCLUSIVE, &TIMER_INIT_IRQ, NULL, "KERNEL$TIMER_TEST")) < 0) {
		__critical_printf("CAN'T REGISTER TIMER INTERRUPT: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	if (__likely(FEATURE_TEST(FEATURE_TSC))) {
/* FPU may be used here, because it's init code and no user processes
   run. Otherwise it would have to be saved and restored */
		long double cpu_mhz_fp;
		while (timer_init_cnt < 3) DO_NOP();
		if (rdtsc_calibrate < 1100000 / BOOTCFG->PARAM_TICK) {
			__critical_printf("BAD TIMER (%uHZ)\n", (unsigned)rdtsc_calibrate * BOOTCFG->PARAM_TICK);
			HALT_KERNEL();
		}
		cpu_mhz_fp = rdtsc_calibrate / ((long double)1000000 / BOOTCFG->PARAM_TICK);
		cpu_mhz = cpu_mhz_fp + 0.5;

			/* wrap around in about 1/2 - 1 hour */
		FINE_TICKS_SHIFT = 0;
		again_fine_ticks:
		if (cpu_mhz_fp / (1 << FINE_TICKS_SHIFT) >= 2) { /* this is needed so that we get smooth tv_usec */
			FINE_TICKS_SHIFT++;
			goto again_fine_ticks;
		}
		FINE_TICKS_PER_MINUTE = cpu_mhz_fp * 60000000. / (1 << FINE_TICKS_SHIFT);

		if (TIME_MAX_FIXUP_1 != 0x90909090 ||
		    TIME_MAX_FIXUP_2 != 0x90909090 ||
		    TIME_MUL_FIXUP != 0x90909090 ||
		    TIME_SHR_FIXUP != 0x90) {
			KERNEL$SUICIDE("TIMER_INIT: INVALID INITIAL FIXUPS: TIME_MAX_FIXUP_1 == %08X, TIME_MAX_FIXUP_2 == %08X, TIME_MUL_FIXUP == %08X, TIME_SHR_FIXUP == %02X", TIME_MAX_FIXUP_1, TIME_MAX_FIXUP_2, TIME_MUL_FIXUP, TIME_SHR_FIXUP);
		}
		
		TIME_SHR_FIXUP = 31 - __BSR(1000000 / (JIFFIES_PER_SECOND >> KERNEL$JIFFIES_STEP_BITS) - 1);
		TIME_MUL_FIXUP = 60000000. * (1 << TIME_SHR_FIXUP) / FINE_TICKS_PER_MINUTE + .5;
		TIME_MAX_FIXUP_1 = TIME_MAX_FIXUP_2 = ((__u64)(1000000 / (JIFFIES_PER_SECOND >> KERNEL$JIFFIES_STEP_BITS) + 1) << TIME_SHR_FIXUP) / TIME_MUL_FIXUP - 1;
	} else {
#define SYNC_TIMER	\
do { cnt = timer_init_cnt; while (timer_init_cnt == cnt) DO_NOP(); } while (0)
		int cnt;
		int i;
		while (timer_init_cnt < 2) DO_NOP();
		for (i = 0; i < 17; i++) {
			bogomips = 1 << i;
			KERNEL$UDELAY(1000000 / BOOTCFG->PARAM_TICK);
			if (timer_init_cnt > 2) goto ool;
		}
		__critical_printf("TIMER STUCK\n");
		HALT_KERNEL();
		ool:
		while (1) {
			SYNC_TIMER;
			KERNEL$UDELAY(1000000 / BOOTCFG->PARAM_TICK);
			if (timer_init_cnt > cnt + 1) break;
			bogomips = 1 << ++i;
		}
		bogomips = 1 << --i;
		while (--i >= 0) {
			bogomips += 1 << i;
			SYNC_TIMER;
			KERNEL$UDELAY(1000000 / BOOTCFG->PARAM_TICK);
			if (timer_init_cnt > cnt + 1) bogomips -= 1 << i;
		}
		FINE_TICKS_PER_MINUTE = (JIFFIES_PER_SECOND * 60) << _FINE_TICKS_SHIFT_WITHOUT_TSC;
	}
	KERNEL$RELEASE_IRQ(TIMER_IRQ_CONTROL, IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_EXCLUSIVE, &TIMER_INIT_IRQ, NULL);
	KTIMER_INIT();
	if ((r = KERNEL$REQUEST_IRQ(0, &TIMER_IRQ_CONTROL, IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_EXCLUSIVE, &TIMER_IRQ, NULL, "KERNEL$TIMER")) < 0) {
		__critical_printf("CAN'T REGISTER TIMER INTERRUPT: %s\n", strerror(-r));
		HALT_KERNEL();
	}
}

void KERNEL$BEEP(unsigned freq)
{
	__u8 port_b;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	port_b = io_inb(PPI_PORT_B);
	if (!freq) {
		io_outb(PPI_PORT_B, port_b & ~(PPI_PORT_B_ENABLE_TIMER_2 | PPI_PORT_B_ENABLE_SPEAKER));
	} else {
		unsigned count;
		count = I8254_FREQ / freq;
		if (__unlikely(count < 2)) count = 2;
		if (__unlikely(count > 65536)) count = 65536;
		io_outb(PPI_PORT_B, port_b | (PPI_PORT_B_ENABLE_TIMER_2 | PPI_PORT_B_ENABLE_SPEAKER));
		io_outb(TIMER_CTL_IO, TIMER_CTL_TIMER_2 | TIMER_CTL_ACTION_RW_LSB_MSB | TIMER_CTL_MODE_3);
		io_outb(TIMER_2_IO, count);
		io_outb(TIMER_2_IO, count >> 8);
	}
	LOWER_SPLX(spl);
}
