#include <SPAD/SYSLOG.H>
#include <ARCH/CPU.H>
#include <KERNEL/SMP/SHARE.H>
#include <KERNEL/ACPI.H>
#include <KERNEL/IRQARCH.H>

#include <KERNEL/VM_ARCH.H>

char FOLD_DL;

unsigned long KERNEL$STACKPAGE = KUVMBASE;

static struct vm_arch MAIN_VM_ARCH = { 0 };
struct vm_arch *VM_ARCH = &MAIN_VM_ARCH;

static KERNEL_SHARED VM_ARCH_SHARE = { "VM_COM", &MAIN_VM_ARCH };

int VM_ADD_MEMORY_HOLE(__u64 h0, __u64 h1)
{
	int i;
	if (h0 == h1)
		return 0;
	h0 &= ~(__u64)(PAGE_CLUSTER_SIZE - 1);
	h1 = (h1 + PAGE_CLUSTER_SIZE - 1) & ~(__u64)(PAGE_CLUSTER_SIZE - 1);
	for (i = 0; i < VM_ARCH->VM_N_HOLES; i++) {
		if (VM_ARCH->VM_HOLES[i * 2 + 1] >= h0 && VM_ARCH->VM_HOLES[i * 2] <= h1) {
			if (h0 < VM_ARCH->VM_HOLES[i * 2]) VM_ARCH->VM_HOLES[i * 2] = h0;
			if (h1 > VM_ARCH->VM_HOLES[i * 2 + 1]) VM_ARCH->VM_HOLES[i * 2 + 1] = h1;
			goto brk;
		}
	}
	if (VM_ARCH->VM_N_HOLES == N_HOLES) return -ENFILE;
	VM_ARCH->VM_HOLES[VM_ARCH->VM_N_HOLES * 2] = h0;
	VM_ARCH->VM_HOLES[VM_ARCH->VM_N_HOLES * 2 + 1] = h1;
	VM_ARCH->VM_N_HOLES++;
	brk:
	return 0;
}

__node_id_t VM_GET_NODE(__p_addr addr)
{
	static __p_addr cache_start = 0;
	static __p_addr cache_end = 0;
	static __node_id_t cache_node;
	__p_addr bottom, top;

	const ACPI_SRAT *srat;
	unsigned real_length;
	const ACPI_SRAT_MEMORY *mem;

	if (__likely(addr >= cache_start) && __likely(addr < cache_end))
		return cache_node;

	srat = ACPI_FIND_TABLE(ACPI_SRAT_SIGNATURE, ACPI_SRAT_LENGTH, &real_length);
	if (__IS_ERR(srat)) {
		cache_start = 0;
		cache_end = ~(__p_addr)0;
		cache_node = 0;
		return 0;
	}

	ACPI_FIND_RESET(srat->entries, (const __u8 *)srat + real_length, ACPI_SRAT_TYPE_MEMORY, ACPI_SRAT_MEMORY_LENGTH);

	bottom = 0;
	top = ~(__p_addr)0;
	while ((mem = ACPI_FIND_ENTRY())) {
		__u64 start, end;
		if (!(__32LE2CPU(mem->flags) & ACPI_SRAT_MEMORY_FLAGS_ENABLED))
			continue;
		start = __64LE2CPU(mem->base_addr);
		end = start + __64LE2CPU(mem->length);
		if (__unlikely(start >= end))
			continue;
		if (addr >= start && addr < end) {
			cache_start = start;
			cache_end = end;
			cache_node = ACPI_PROXIMITY_DOMAIN_TO_NODE_ID(acpi_srat_memory_proximity_domain(mem));
			/*__debug_printf("range: %Lx - %Lx: %d\n", cache_start, cache_end, cache_node);*/
			goto unmap_ret_cache_node;
		}
		if (end <= addr && end > bottom) bottom = end;
		if (start > addr && start < top) top = start;
	}

	cache_start = bottom;
	cache_end = top;
	cache_node = 0;
	/*__debug_printf("no mapping for %Lx - %Lx\n", cache_start, cache_end);*/

	unmap_ret_cache_node:
	ACPI_UNMAP(srat, real_length);
	return cache_node;
}

int VM_ARCH_CHECK_HW_MEM(PAGE *p)
{
	char *e;
	__u32 *v;
	v = KERNEL$MAP_PHYSICAL_PAGE(p);
	v[0] = 0x33343536;
	v[PAGE_CLUSTER_SIZE / 4 - 1] = 0xCCCDCECF;
	if ((e = getenv("@KERNEL$MEMORY_CHECK")) && __likely(e[0] >= '1') && __likely(!e[1]))
		CACHE_WBINVD();
	if (__unlikely(v[0] != 0x33343536) || __unlikely(v[PAGE_CLUSTER_SIZE / 4 - 1] != 0xCCCDCECF)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "MEMORY", "MEMORY NOT PRESENT AT PHYSICAL ADDRESS %"__64_format"X", (__u64)PAGE_2_PHYS(p));
		KERNEL$UNMAP_PHYSICAL_BANK(v);
		return -ENODEV;
	}
	if ((e = getenv("@KERNEL$MEMORY_CHECK")) && __likely(e[0] >= '2') && __likely(!e[1])) {
		__u32 i, j;
		for (j = 0; j < 2; j++) {
			for (i = 0; i < PAGE_CLUSTER_SIZE / 4; i++) v[i] = 0x12345678 ^ (j - 1) ^ i ^ i << 16;
			CACHE_WBINVD();
			for (i = 0; i < PAGE_CLUSTER_SIZE / 4; i++) if (v[i] != (0x12345678 ^ (j - 1) ^ i ^ i << 16)) {
				KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "MEMORY", "MEMORY ERROR AT %"__64_format"X", (__u64)PAGE_2_PHYS(p) + i);
				KERNEL$UNMAP_PHYSICAL_BANK(v);
				return -EIO;
			}
		}
	}
	KERNEL$UNMAP_PHYSICAL_BANK(v);
	return 0;
}

int KERNEL$SET_MEMORY_LIMIT(__u64 memory, __u64 *holes, int n_holes)
{
	int r = (!FEATURE_TEST(_FEATURE_USE_PAE) ? KERNEL_SET_MEMORY_LIMIT_NOPAE : KERNEL_SET_MEMORY_LIMIT_PAE)(memory, holes, n_holes);
/* Kick the other CPUs. Maybe one day I may find a better place for this. */
	if (!r) BOOT_APS();
	return r;
}

int KERNEL$SET_MEMORY_PAT(__p_addr addr, __p_addr len, int pat)
{
	return (!FEATURE_TEST(_FEATURE_USE_PAE) ? KERNEL_SET_MEMORY_PAT_NOPAE : KERNEL_SET_MEMORY_PAT_PAE)(addr, len, pat);
}

int KERNEL$SET_MAPPED_MEMORY(__p_addr addr, __p_addr len, int pat)
{
	return (!FEATURE_TEST(_FEATURE_USE_PAE) ? KERNEL_SET_MAPPED_MEMORY_NOPAE : KERNEL_SET_MAPPED_MEMORY_PAE)(addr, len, pat);
}

__u64 KERNEL$GET_MEMORY_SIZE(int type)
{
	return (!FEATURE_TEST(_FEATURE_USE_PAE) ? KERNEL_GET_MEMORY_SIZE_NOPAE : KERNEL_GET_MEMORY_SIZE_PAE)(type);
}

#define PAE_FIXUP(nopae, pae)				\
__asm__ ("						\n\
	.SECTION .FEATURE_FIXUP				\n\
	.LONG	"#nopae"				\n\
	.LONG	"#nopae" + 6				\n\
	.LONG	1f					\n\
	.LONG	2f					\n\
	.LONG	"__stringify(_FEATURE_USE_PAE)"		\n\
	.PREVIOUS					\n\
	.SECTION .rodata.end				\n\
1:	JMPL	*3f					\n\
2:							\n\
	.PREVIOUS					\n\
	.SECTION .rodata				\n\
	.ALIGN	4					\n\
3:	.LONG	"#pae"					\n\
	.PREVIOUS					\n\
");

PAE_FIXUP(KERNEL$MAP_PHYSICAL_BANK,		KERNEL_MAP_PHYSICAL_BANK_PAE)
PAE_FIXUP(KERNEL$UNMAP_PHYSICAL_BANK,		KERNEL_UNMAP_PHYSICAL_BANK_PAE)
PAE_FIXUP(KERNEL$UNMAP_PHYSICAL_BANK_ADDR,	KERNEL_UNMAP_PHYSICAL_BANK_ADDR_PAE)
PAE_FIXUP(KERNEL$UNMAP_PHYSICAL_BANK_PAGE,	KERNEL_UNMAP_PHYSICAL_BANK_PAGE_PAE)
PAE_FIXUP(KERNEL$MAP_PHYSICAL_PAGE,		KERNEL_MAP_PHYSICAL_PAGE_PAE)
PAE_FIXUP(KERNEL$PAGE_2_PHYS,			KERNEL_PAGE_2_PHYS_PAE)
PAE_FIXUP(KERNEL$PHYS_2_PAGE,			KERNEL_PHYS_2_PAGE_PAE)
PAE_FIXUP(KERNEL$VIRT_2_PHYS,			KERNEL_VIRT_2_PHYS_PAE)
PAE_FIXUP(KERNEL$PHYS_2_VIRT,			KERNEL_PHYS_2_VIRT_PAE)
PAE_FIXUP(KERNEL$DMA_2_VIRT,			KERNEL_DMA_2_VIRT_PAE)
PAE_FIXUP(VM_ARCH_MAP_PAGE,			VM_ARCH_MAP_PAGE_PAE)
PAE_FIXUP(VM_ARCH_UNMAP_RANGE,			VM_ARCH_UNMAP_RANGE_PAE)
PAE_FIXUP(VM_ARCH_READ_ONLY,			VM_ARCH_READ_ONLY_PAE)
PAE_FIXUP(VM_ARCH_UNMAP_MAPPING,		VM_ARCH_UNMAP_MAPPING_PAE)
PAE_FIXUP(VM_ARCH_CHECK_MAPPING,		VM_ARCH_CHECK_MAPPING_PAE)
PAE_FIXUP(VM_ARCH_GET_PAGE,			VM_ARCH_GET_PAGE_PAE)
PAE_FIXUP(VM_KERNEL_FAULT_EXCEPTION,		VM_KERNEL_FAULT_EXCEPTION_PAE)
PAE_FIXUP(SYSCALL_INVD_EXTD_PAGE,		SYSCALL_INVD_EXTD_PAGE_PAE)
PAE_FIXUP(ARCH_PROC_CTOR,			ARCH_PROC_CTOR_PAE)
PAE_FIXUP(ARCH_PROC_INIT,			ARCH_PROC_INIT_PAE)
PAE_FIXUP(ARCH_PROC_DESTROY,			ARCH_PROC_DESTROY_PAE)
PAE_FIXUP(ARCH_PROC_FREE,			ARCH_PROC_FREE_PAE)
PAE_FIXUP(VM_ARCH_AFTER_FORK,			VM_ARCH_AFTER_FORK_PAE)
PAE_FIXUP(SAVE_FPU,				SAVE_FPU_PAE)
PAE_FIXUP(FLUSH_FPU,				FLUSH_FPU_PAE)
PAE_FIXUP(KERNEL$MAP_PHYSICAL_REGION_LONG_TERM,	KERNEL_MAP_PHYSICAL_REGION_LONG_TERM_PAE)
PAE_FIXUP(KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM,KERNEL_UNMAP_PHYSICAL_REGION_LONG_TERM_PAE)

void VM_BOOT_GET_MEM(unsigned long code_bottom, unsigned long free_later, unsigned long code_top, unsigned long data_bottom, unsigned long data_top, unsigned long mem_top)
{
	REGISTER_SHARED_POINTER(&VM_ARCH_SHARE);
	(!FEATURE_TEST(_FEATURE_USE_PAE) ? VM_BOOT_GET_MEM_NOPAE : VM_BOOT_GET_MEM_PAE)(code_bottom, free_later, code_top, data_bottom, data_top, mem_top);
}

#if __DEBUG >= 2
void *VSPACE_LOWER_SPL(void *ptr, int spl);
void *VSPACE_LOWER_SPL(void *ptr, int spl)
{
	LOWER_SPLX(spl);
	return ptr;
}
#endif

#if __KERNEL_SUPPORT_SMP
void VM_BOOT_INIT_AP(void)
{
	VM_ARCH = FIND_SHARED_POINTER("VM_COM");

	if (!FEATURE_TEST(_FEATURE_USE_PAE)) {
		VM_BOOT_INIT_AP_NOPAE();
	} else {
		VM_BOOT_INIT_AP_PAE();
	}
}
#endif
