#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/DL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <SYS/TYPES.H>
#include <ARCH/IO.H>

#include <SPAD/ISADMA.H>

#include "DMAREGS.H"

#define MAX_TRIES	10000

#define N_CHANS		8

static struct dma_state global_dma_state;

/* !!! SMPFIX: use locking & shared data */

static struct dma_state {
	char used[N_CHANS];
	char enabled[N_CHANS];
	char paused[N_CHANS];
	IO_RANGE range1;
	IO_RANGE range2;
	IO_RANGE rangep;
} * const dma_state = &global_dma_state;

static const unsigned char dma_page_regs[N_CHANS] = { DMA_PAGE_0, DMA_PAGE_1, DMA_PAGE_2, DMA_PAGE_3, 0, DMA_PAGE_5, DMA_PAGE_6, DMA_PAGE_7 };

#define BAD_NUMBER(channel)	(((unsigned)(channel) ^ 3) >= 7)

#define ASSERT_SPL_ALWAYS(fn)					\
do {								\
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl)))	\
		KERNEL$SUICIDE(fn" AT SPL %08X", spl);		\
} while (0)

static __NORET_ATTR__ void ASSERT_FAILED(char *fn, int channel, int en)
{
	if (BAD_NUMBER(channel)) KERNEL$SUICIDE("%s: INVALID CHANEL %d", fn, channel);
	if (!dma_state->used[channel]) KERNEL$SUICIDE("%s: CHANNEL %d NOT REGISTERED", fn, channel);
	else if (dma_state->used[channel] != en) KERNEL$SUICIDE("%s: CHANNEL %d HAS%s TRANSFER IN PROGRESS", fn, channel, en ? " NOT" : "");
	else KERNEL$SUICIDE("%s: JAMMED MEMORY, CHANNEL %d, USED %02X, ENABLED %02X, PAUSED %02X", fn, channel, (unsigned char)dma_state->used[channel], (unsigned char)dma_state->enabled[channel], (unsigned char)dma_state->paused[channel]);
}

#define ASSERT_CHANNEL_ALWAYS(fn, en)					\
do {									\
	if (__unlikely(BAD_NUMBER(channel)) || __unlikely((dma_state->used[channel] ^ 1) | (dma_state->enabled[channel] ^ (en)))) {			\
		ASSERT_FAILED(fn, channel, (en));			\
	}								\
} while (0)

#if __DEBUG >= 1
#define ASSERT_CHANNEL(fn, en) ASSERT_CHANNEL_ALWAYS(fn, en)
#else
#define ASSERT_CHANNEL(fn, en) do { } while (0)
#endif

void ISADMA$SETUP_TRANSFER(int channel, int wr, int autoinit, unsigned long addr, unsigned long size)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL("ISADMA$SETUP_TRANSFER", 0);
	dma_state->enabled[channel] = 1;
	dma_state->paused[channel] = 1;
	io_outb(channel < 4 ? DMA1_MODE_REG : DMA2_MODE_REG, DMA_MODE_READ + (wr << 2) + (autoinit << 4) + (channel & 3));
	ISADMA$SETUP_AUTOINIT(channel, addr, size);
	ISADMA$RESUME_TRANSFER(channel);
	LOWER_SPLX(spl);
	return;
}

void ISADMA$SETUP_AUTOINIT(int channel, unsigned long addr, unsigned long size)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL("ISADMA$SETUP_AUTOINIT", 1);
	if (channel < 4) {
		size--;
#if __DEBUG >= 1
		if (__unlikely(((addr & ~0x00FFFFFFUL) | (size & ~0x0000FFFFUL)) != 0)) goto invl;
		if (__unlikely((addr & ~0xFFFFUL) != ((addr + size) & ~0xFFFFUL))) goto invl;
#endif
		io_outb(dma_page_regs[channel], addr >> 16);
		io_outb((channel << 1) + DMA_ADDR_0, addr & 0xff);
		io_outb((channel << 1) + DMA_ADDR_0, (addr >> 8) & 0xff);
		io_outb((channel << 1) + DMA_CNT_0, size & 0xff);
		io_outb((channel << 1) + DMA_CNT_0, size >> 8);
	} else {
		size -= 2;
#if __DEBUG >= 1
		if (__unlikely(((addr & ~0x00FFFFFEUL) | (size & ~0x0001FFFEUL)) != 0)) goto invl;
		if (__unlikely((addr & ~0x1FFFFUL) != ((addr + size) & ~0x1FFFFUL))) goto invl;
#endif
		io_outb(dma_page_regs[channel], (addr >> 16) & ~1);
		io_outb(((channel - 4) << 2) + DMA_ADDR_4, (addr >> 1) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_ADDR_4, (addr >> 9) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_CNT_4, (size >> 1) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_CNT_4, size >> 9);
	}
	LOWER_SPLX(spl);
	return;

#if __DEBUG >= 1
	invl:
	KERNEL$SUICIDE("ISADMA$SETUP_AUTOINIT: CHANNEL %d: INVALID ADDRESS/SIZE: %08lX, %08lX", channel, addr, size);
#endif
}

unsigned long ISADMA$GET_ADDRESS(int channel)
{
	unsigned change_counter;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL("ISADMA$GET_ADDRESS", 1);
	change_counter = 0;
	again:
	if (__unlikely(++change_counter == MAX_TRIES)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "ISADMA", "ADDRESS IS CONSTANTLY CHANGING, DMA CHANNEL %d", channel);
	}
	/*
	 * Note:
	 * on my VIA 586 mainboard, address is constantly changing under some
	 * workloads (sound card + disk simultaneously).
	 * We must compare only high byte --- low byte is unstable.
	 *	1. read low byte and discard it
	 *	2. read high byte and remember it
	 *	3. read low byte and remember it
	 *	4. read high byte --- if it matches, we know that low byte
	 *	   didn't wrap around. If it doesn't match, repeat
	 */
	if (channel < 4) {
		unsigned long address = io_inb(dma_page_regs[channel]) << 16;
		io_inb((channel << 1) + DMA_ADDR_0);
		address |= io_inb((channel << 1) + DMA_ADDR_0) << 8;
		address |= io_inb((channel << 1) + DMA_ADDR_0);
		if (__unlikely(((address >> 8) & 0xff) != io_inb((channel << 1) + DMA_ADDR_0)) && __likely(change_counter < MAX_TRIES)) goto again;
		LOWER_SPLX(spl);
		/*if (change_counter != 1) __debug_printf("changes: %d\n", change_counter);*/
		return address;
	} else {
		unsigned long address = (io_inb(dma_page_regs[channel]) << 16) & ~1;
		io_inb(((channel - 4) << 2) + DMA_ADDR_4);
		address |= io_inb(((channel - 4) << 2) + DMA_ADDR_4) << 9;
		address |= io_inb(((channel - 4) << 2) + DMA_ADDR_4) << 1;
		if (__unlikely(((address >> 9) & 0xff) != io_inb(((channel - 4) << 2) + DMA_ADDR_4)) && __likely(change_counter < MAX_TRIES)) goto again;
		LOWER_SPLX(spl);
		return address;
	}
}

void ISADMA$END_TRANSFER(int channel)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL("ISADMA$GET_ADDRESS", 1);
	ISADMA$PAUSE_TRANSFER(channel);
	dma_state->enabled[channel] = 0;
	LOWER_SPLX(spl);
}

void ISADMA$PAUSE_TRANSFER(int channel)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL("ISADMA$PAUSE_TRANSFER", 1);
	if (__unlikely(dma_state->paused[channel])) goto already_paused;
	dma_state->paused[channel] = 1;
	if (channel < 4) io_outb(DMA1_MASK_REG, channel + 4);
	else io_outb(DMA2_MASK_REG, channel - 4 + 4);
	already_paused:
	LOWER_SPLX(spl);
}

void ISADMA$RESUME_TRANSFER(int channel)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL("ISADMA$RESUME_TRANSFER", 1);
	if (__unlikely(!dma_state->paused[channel])) goto already_running;
	dma_state->paused[channel] = 0;
	if (channel < 4) io_outb(DMA1_MASK_REG, channel);
	else io_outb(DMA2_MASK_REG, channel - 4);
	already_running:
	LOWER_SPLX(spl);
}

int ISADMA$REQUEST_CHANNEL(int channel)
{
	int spl = KERNEL$SPL;
	ASSERT_SPL_ALWAYS("ISADMA$REQUEST_CHANNEL");
	RAISE_SPL(SPL_DMA);
	if (__unlikely(BAD_NUMBER(channel))) {
		LOWER_SPLX(spl);
		return -ERANGE;
	}
	if (__unlikely(dma_state->used[channel])) {
		LOWER_SPLX(spl);
		return -EBUSY;
	}
	dma_state->used[channel] = 1;
	LOWER_SPLX(spl);
	return 0;
}

void ISADMA$RELEASE_CHANNEL(int channel)
{
	int spl = KERNEL$SPL;
	ASSERT_SPL_ALWAYS("ISADMA$RELEASE_CHANNEL");
	RAISE_SPL(SPL_DMA);
	ASSERT_CHANNEL_ALWAYS("ISADMA$RELEASE_CHANNEL", 0);
	dma_state->used[channel] = 0;
	LOWER_SPLX(spl);
	return;
}

DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	int r;
	memset(dma_state, 0, sizeof(struct dma_state));
	dma_state->range1.start = IO_DMA1_BASE;
	dma_state->range1.len = IO_DMA1_LEN;
	dma_state->range1.name = "DMA 0-3";
	dma_state->range2.start = IO_DMA2_BASE;
	dma_state->range2.len = IO_DMA2_LEN;
	dma_state->range2.name = "DMA 4-7";
	dma_state->rangep.start = IO_DMAP_BASE;
	dma_state->rangep.len = IO_DMAP_LEN;
	dma_state->rangep.name = "DMA PAGE";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&dma_state->range1))) {
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER DMA 0-3 PORT RANGE: %s", strerror(-r));
		err0:
		RQ->status = r;
		RETURN_AST(RQ);
	}
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&dma_state->range2))) {
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER DMA 4-7 PORT RANGE: %s", strerror(-r));
		err1:
		KERNEL$UNREGISTER_IO_RANGE(&dma_state->range1);
		goto err0;
	}
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&dma_state->rangep))) {
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER DMA PAGE PORT RANGE: %s", strerror(-r));
		KERNEL$UNREGISTER_IO_RANGE(&dma_state->range2);
		goto err1;
	}
	io_outb(DMA1_CLEAR_FF_REG, 0);
	io_outb(DMA2_CLEAR_FF_REG, 0);
	io_outb(DMA2_MODE_REG, DMA_MODE_CASCADE);
	io_outb(DMA1_MASK_ALL_REG, 0);
	io_outb(DMA2_MASK_ALL_REG, 1);
	io_outb(DMA2_MASK_REG, 0);
	RQ->status = 0;
	RETURN_AST(RQ);
}

DECL_IOCALL(DLL$UNLOAD, SPL_DEV, DLINITRQ)
{
	int i;
	for (i = 0; i < 8; i++) if (dma_state->used[i]) KERNEL$SUICIDE("ISADMA: CHANNEL %d WAS NOT RELEASED BY THE DRIVER", i);
	KERNEL$UNREGISTER_IO_RANGE(&dma_state->range1);
	KERNEL$UNREGISTER_IO_RANGE(&dma_state->range2);
	KERNEL$UNREGISTER_IO_RANGE(&dma_state->rangep);
	RQ->status = 0;
	RETURN_AST(RQ);
}



