#ifndef __LIB_KERNEL_SIGSTACK_H
#define __LIB_KERNEL_SIGSTACK_H

#include <ARCH/SETUP.H>
#include <ARCH/CPU.H>
#include <SETJMP.H>
#include <LIB/KERNEL/UASM.H>

static __finline__ int on_altstack(void)
{
	register unsigned long sp __asm__("esp");
	return ((unsigned long)altstack - sp) < altstack_size;
}

static __finline__ int ptr_on_altstack(unsigned long p)
{
	return __unlikely(((unsigned long)altstack - p) < altstack_size);
}

static __finline__ void get_altstack(stack_t *oss)
{
	if (__unlikely(!altstack)) {
		oss->ss_sp = NULL;
		oss->ss_flags = SS_DISABLE;
		oss->ss_size = 0;
	} else {
		oss->ss_sp = (void *)(altstack - altstack_size);
		oss->ss_flags = on_altstack() * SS_ONSTACK;
		oss->ss_size = altstack_size;
	}
}

static __finline__ void set_altstack(const stack_t *ss)
{
	if (ss->ss_flags & SS_DISABLE) {
		altstack = 0;
		altstack_size = 0;
	} else {
		unsigned long d;
		altstack = (unsigned long)ss->ss_sp + ss->ss_size;
		d = altstack & 3UL;
		altstack -= d;
		altstack_size = ss->ss_size - d;
	}
	THREAD_UPDATE_ALTSTACK();
}

static __finline__ unsigned long has_altstack(void)
{
	return altstack;
}

#define emit_sigaltstack(name, call)		\
extern void name(void *, unsigned);		\
__asm__ ("					\n\
	.SECTION .text				\n\
	.ALIGN	"__stringify(__CPU_CALL_ALIGN)"	\n\
"#name":					\n\
	PUSHL	%EBP				\n\
	MOVL	%ESP, %EBP			\n\
	MOVL	altstack, %ESP			\n\
	CALL	"#call"				\n\
	LEAVE					\n\
	RET					\n\
	.PREVIOUS				\n\
");

#define def_save_fpu							\
	char fpu_state[512 + FPU_ALIGN - 1];				\
	char fpu_saved;

#define save_fpu							\
do {									\
	fpu_saved = 0;							\
	if (*(int *)UPLACE(UDATA_COPROCESSOR_USED)) {			\
		unsigned long p;					\
		p = ((unsigned long)fpu_state + FPU_ALIGN - 1) & ~(unsigned long)(FPU_ALIGN - 1);							\
		if (__likely(KERNEL$FEATURE_TEST(FEATURE_FPU_FXSR))) {	\
			fpu_saved = 2;					\
			__asm__ volatile("FXSAVE (%0);FINIT"::"r"(p):"memory");\
		} else {						\
			fpu_saved = 1;					\
			__asm__ volatile("FNSAVE (%0);FWAIT"::"r"(p):"memory");\
		}							\
	}								\
} while (0)

#define restore_fpu							\
do {									\
	if (fpu_saved) {						\
		unsigned long p;					\
		p = ((unsigned long)fpu_state + FPU_ALIGN - 1) & ~(unsigned long)(FPU_ALIGN - 1);							\
		if (fpu_saved == 2) {					\
			__asm__ volatile ("FXRSTOR (%0)"::"r"(p):"memory");\
		} else {						\
			__asm__ volatile ("FRSTOR (%0)"::"r"(p):"memory");\
		}							\
	}								\
} while (0)

static __finline__ int __PTR_BELOW_SP(void *io, jmp_buf j)
{
	int p1 = ptr_on_altstack((unsigned long)io);
	int p2 = ptr_on_altstack(j->__esp);
	if (__unlikely(p1 != p2)) return p1 > p2;
	return (unsigned long)io < j->__esp;
}

#endif
