#include <SPAD/LIBC.H>
#include <STRING.H>
#include <ARCH/BIOS16.H>
#include <ARCH/BIOS32.H>
#include <SPAD/IRQ.H>
#include <ARCH/BITOPS.H>

#include <SPAD/PCI.H>

/* based on linux/arch/i386/kernel/pci-pc.c (c) 1999--2000 Martin Mares <mj@suse.cz> */

#define BIOS32_PCI_SERVICE	(('$' << 0) + ('P' << 8) + ('C' << 16) + ('I' << 24))

#define PCI_SIGNATURE		(('P' << 0) + ('C' << 8) + ('I' << 16) + (' ' << 24))

#define PCIBIOS_PCI_BIOS_PRESENT 	0xb101
#define PCIBIOS_FIND_PCI_DEVICE		0xb102
#define PCIBIOS_FIND_PCI_CLASS_CODE	0xb103
#define PCIBIOS_GENERATE_SPECIAL_CYCLE	0xb106
#define PCIBIOS_READ_CONFIG_BYTE	0xb108
#define PCIBIOS_READ_CONFIG_WORD	0xb109
#define PCIBIOS_READ_CONFIG_DWORD	0xb10a
#define PCIBIOS_WRITE_CONFIG_BYTE	0xb10b
#define PCIBIOS_WRITE_CONFIG_WORD	0xb10c
#define PCIBIOS_WRITE_CONFIG_DWORD	0xb10d
#define PCIBIOS_GET_ROUTING_OPTIONS	0xb10e
#define PCIBIOS_SET_PCI_HW_INT		0xb10f

static struct pci_info {
	unsigned long bios_entry;
	unsigned bios_version;
	int n_buses;
	long mapped_amount;
	__u32 devices[2048];
} *pci_info;

static struct pci_info global_pci_info;

static void pci_ctor(void *p_)
{
	struct pci_info *p = p_;
	p->bios_entry = -1UL;
	memset(p->devices, 0, sizeof p->devices);
	p->mapped_amount = 0;
}

#define PCI_DEVICE_IS_USED(i)	(pci_info->devices[(i)/32] & (1 << ((i) & 31)))
#define PCI_DEVICE_SET_USED(i)	(pci_info->devices[(i)/32] |= (1 << ((i) & 31)))
#define PCI_DEVICE_SET_FREE(i)	(pci_info->devices[(i)/32] &= ~(1 << ((i) & 31)))

static int pci_call_bios(struct bios_regs *regs)
{
/* GCC is brain-damaged and does not allow more that 10 parameters to asm */
/* Furthermore, we can't change EBX if compiled with -fpic */
	int status;
	/*int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);*/
	__asm__ volatile ("				\n\
		PUSHL	%%EBX				\n\
		MOVL	%%EDX, %%EBX			\n\
		PUSHFL					\n\
		CLI					\n\
		PUSHL	%%CS				\n\
		CALL	*%%ESI				\n\
		SBBL	%%ESI, %%ESI			\n\
		POPFL					\n\
		MOVL	%%EBX, %%EDI			\n\
		POPL	%%EBX				\n\
	":"=a"(regs->eax),"=c"(regs->ecx),"=d"(regs->edx),/*"=b"(regs->ebx),*/
	"=D"(regs->ebx),"=S"(status)
	:"a"(regs->eax),"c"(regs->ecx),"d"(regs->ebx),/*"b"(regs->ebx),*/
	"D"(regs->edi),"S"(pci_info->bios_entry)
	);
	/*LOWER_SPLX(spl);*/
	return status;
}

__u8 PCI$READ_CONFIG_BYTE(pci_id_t id, int off)
{
	struct bios_regs r;
	memset(&r, 0, sizeof r);
	r.eax = PCIBIOS_READ_CONFIG_BYTE;
	r.ebx = id;
	r.edi = off;
	if (pci_call_bios(&r)) return -1;
	return r.ecx;
}

__u16 PCI$READ_CONFIG_WORD(pci_id_t id, int off)
{
	struct bios_regs r;
	memset(&r, 0, sizeof r);
	r.eax = PCIBIOS_READ_CONFIG_WORD;
	r.ebx = id;
	r.edi = off;
	if (pci_call_bios(&r)) return -1;
	return r.ecx;
}

__u32 PCI$READ_CONFIG_DWORD(pci_id_t id, int off)
{
	struct bios_regs r;
	memset(&r, 0, sizeof r);
	r.eax = PCIBIOS_READ_CONFIG_DWORD;
	r.ebx = id;
	r.edi = off;
	if (pci_call_bios(&r)) return -1;
	return r.ecx;
}

void PCI$WRITE_CONFIG_BYTE(pci_id_t id, int off, __u8 val)
{
	struct bios_regs r;
	memset(&r, 0, sizeof r);
	r.eax = PCIBIOS_WRITE_CONFIG_BYTE;
	r.ebx = id;
	r.edi = off;
	r.ecx = val;
	pci_call_bios(&r);
}

void PCI$WRITE_CONFIG_WORD(pci_id_t id, int off, __u16 val)
{
	struct bios_regs r;
	memset(&r, 0, sizeof r);
	r.eax = PCIBIOS_WRITE_CONFIG_WORD;
	r.ebx = id;
	r.edi = off;
	r.ecx = val;
	pci_call_bios(&r);
}

void PCI$WRITE_CONFIG_DWORD(pci_id_t id, int off, __u32 val)
{
	struct bios_regs r;
	memset(&r, 0, sizeof r);
	r.eax = PCIBIOS_WRITE_CONFIG_DWORD;
	r.ebx = id;
	r.edi = off;
	r.ecx = val;
	pci_call_bios(&r);
}

int PCI$PARSE_PARAMS(const char *opt, const char *optend, const char *str, pci_id_t *id, pci_id_t *mask, int *skip)
{
	static const struct __param_table params[5] = {
		"BUS", __PARAM_INT, 0, 256,
		"DEV", __PARAM_INT, 0, 32,
		"FN", __PARAM_INT, 0, 8,
		"SKIP", __PARAM_INT, 0, 65536,
		NULL, 0, 0, 0,
	};
	void *vars[5];
	int state = 0;
	int bus, dev, fn;
	if (__unlikely(__strcasexcmp("PCI", opt, optend))) return 1;
	bus = dev = fn = -1;
	vars[0] = &bus;
	vars[1] = &dev;
	vars[2] = &fn;
	vars[3] = skip;
	vars[4] = NULL;
	if (__parse_extd_param(&str, &state, params, vars, NULL, NULL, NULL, NULL)) return 1;
	if (bus != -1) {
		*id = (*id & 0xff) | (bus << 8);
		*mask |= 0xff00;
	}
	if (dev != -1) {
		*id = (*id & 0xff07) | (dev << 3);
		*mask |= 0x00F8;
	}
	if (fn != -1) {
		*id = (*id & 0xfff8) | (fn);
		*mask |= 0x0007;
	}
	return 0;
}

char *PCI$ID(char str[__MAX_STR_LEN], pci_id_t id)
{
	_snprintf(str, __MAX_STR_LEN, "BUS %d, DEV %d, FN %d", (id >> 8) & 255, (id >> 3) & 31, id & 7);
	return str;
}

#include <LIB/PCI/PCICOMM.I>
