#include <SYS/TYPES.H>
#include <STRING.H>
#include <VALUES.H>
#include <STDARG.H>
#include <STDLIB.H>
#include <SPAD/LIBC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TIMER.H>
#include <SPAD/RANDOM.H>
#include <SPAD/IRQ.H>
#include <ARCH/IO.H>
#include <SPAD/BIO_KRNL.H>

#include <SPAD/SCSI.H>
#include <SPAD/ATA.H>
#include "ACTRL.H"
#include "ACTRLREG.H"

static int (*const special_controllers[])(pci_id_t id, int (**special_init)(ACTRL *a)) = {
	DETECT_VT6421,
	DETECT_ULI_4PORT,
};

static int (*const controllers[])(ACTRL *a) = {
	DETECT_RZ1000,
	DETECT_VIA,
	DETECT_INTEL,
	DETECT_SVW,
	DETECT_ALI,
	DETECT_IT821,
	DETECT_TRIFLEX,
	DETECT_CMD,
	DETECT_SII,
	DETECT_PDC,
	DETECT_PDC2,
	DETECT_HPT,
	DETECT_JMB,
};

static const struct pci_id_s ata_dev[] = {
 	{ 0x1103, 0x0004, PCI_ANY_ID, PCI_ANY_ID, 0x03, 0xff, "HIGHPOINT HPT370", ACTRL_ALWAYS_NATIVE },
 	{ 0x1103, 0x0004, PCI_ANY_ID, PCI_ANY_ID, 0x04, 0xff, "HIGHPOINT HPT370A", ACTRL_ALWAYS_NATIVE },
 	{ 0x1103, 0x0004, PCI_ANY_ID, PCI_ANY_ID, 0x05, 0xff, "HIGHPOINT HPT372", ACTRL_ALWAYS_NATIVE },
 	{ 0x1103, 0x0005, PCI_ANY_ID, PCI_ANY_ID, 0x00, 0xfe, "HIGHPOINT HPT372A", ACTRL_ALWAYS_NATIVE },
 	{ 0x1103, 0x0006, PCI_ANY_ID, PCI_ANY_ID, 0x00, 0xfe, "HIGHPOINT HPT302", ACTRL_ALWAYS_NATIVE },
 	{ 0x1103, 0x0007, PCI_ANY_ID, PCI_ANY_ID, 0x00, 0xfe, "HIGHPOINT HPT371", ACTRL_ALWAYS_NATIVE },
 	{ 0x1103, 0x0008, PCI_ANY_ID, PCI_ANY_ID, 0x00, 0x00, "HIGHPOINT HPT374", ACTRL_ALWAYS_NATIVE },
 
	{ 0x105a, 0x4d33, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20246", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x4d38, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20262", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x0d38, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20263", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x0d30, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20265", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x4d30, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20267", ACTRL_ALWAYS_NATIVE },

	{ 0x105a, 0x4d68, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20268", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x4d69, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20269", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x6268, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20270", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x6269, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20271", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x1275, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20275", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x5275, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20276", ACTRL_ALWAYS_NATIVE },
	{ 0x105a, 0x7275, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PROMISE PDC20277", ACTRL_ALWAYS_NATIVE },

	{ 0x1095, 0x0640, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "CMD 640", 0 },
	{ 0x1095, 0x0643, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "CMD 643", ACTRL_TRY_NONSIMPLEX },
	{ 0x1095, 0x0646, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "CMD 646", 0 },
	{ 0x1095, 0x0648, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "CMD 648", 0 },
	{ 0x1095, 0x0649, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "CMD 649", 0 },
	{ 0x1095, 0x0680, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 680", ACTRL_ALWAYS_NATIVE },
	{ 0x1095, 0x3112, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3112", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1095, 0x0249, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3112", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1095, 0x3512, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3112", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1002, 0x436e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3112", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1002, 0x4379, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3112", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1002, 0x437a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3112", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1095, 0x3114, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SII 3114", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },

	{ 0x1283, 0x8211, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ITE IT8211", ACTRL_ALWAYS_NATIVE },
	{ 0x1283, 0x8212, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ITE IT8212", ACTRL_ALWAYS_NATIVE },

	{ 0x10b9, 0x5228, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ALI", ACTRL_TRY_NONSIMPLEX },
	{ 0x10b9, 0x5229, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ALI", ACTRL_TRY_NONSIMPLEX },
	{ 0x10b9, 0x5289, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ULI 5289", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_IGNORE_SIMPLEX },
	{ 0x10b9, 0x5287, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ULI 5287", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_IGNORE_SIMPLEX },
	{ 0x10b9, 0x5281, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ULI 5281", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_IGNORE_SIMPLEX },

	{ 0x1106, 0x5337, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY },
	{ 0x1106, 0x0591, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY },
	{ 0x1106, 0x3149, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY },
	{ 0x1106, 0x5287, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY },
	{ 0x1106, 0x5372, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY },
	{ 0x1106, 0x7372, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY },
	{ 0x1106, 0x3249, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA VT6421", ACTRL_ALWAYS_NATIVE | ACTRL_LIMIT_PROBE | ACTRL_BOTCHED_IDENTIFY | ACTRL_SPECIAL_PORTS },

	{ 0x197b, 0x2361, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "JMB 361", 0 },
	{ 0x197b, 0x2363, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "JMB 363", 0 },
	{ 0x197b, 0x2365, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "JMB 365", 0 },
	{ 0x197b, 0x2366, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "JMB 366", 0 },
	{ 0x197b, 0x2368, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "JMB 368", 0 },

	{ PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_CLASS_STORAGE_IDE << 16, 0xffff0000, NULL, 0 },

	{ 0 }
};


static int ATA_UNLOAD(void *p, void **dlrq, const char * const argv[]);

static const __u16 dummy_id[512];

#define APORT_DEVICE_ARRAY_SIZE		2

#define CODE_FIXUP_FLAGS						\
	if (a->actrl_flags & ACTRL_FORCE_MASKIRQ) dev_flags |= ACTRL_FORCE_MASKIRQ;

#define CODE_DISK_NAME(result, len, prefix, a, portn, devn)		\
{									\
	int n = (portn) * 2 + (devn);					\
	if (!strcmp((a)->dev_name, "ATA$ACTRL"))			\
		_snprintf(result, len, "%s%d", prefix, n);		\
	else								\
		_snprintf(result, len, "%s%d@%s", prefix, n, a->dev_name);\
}

#define CODE_PROBE_LIMIT(p)		((p)->probe_limit)
#define CODE_WAIT_FOR_OTHER_DEVICE(p)	1
#define CODE_INIT_CANT_RESET(p)		(!(p)->get_avail_xfer && !((p)->aport_flags & (APORT_SATA | APORT_IMPLICIT_DMA)) && !atapi)

#define CODE_INIT_XFER(p, dev)						\
	if ((p)->set_xfer)						\
		(p)->set_xfer(p, dev, IDE_XFER_PIO_0, dummy_id);

#define CODE_DETACH_XFER(p, dev)					\
	if ((p)->set_xfer) {						\
		(p)->set_xfer(p, dev, IDE_XFER_PIO_NONE, dummy_id);	\
		(p)->set_xfer(p, dev, IDE_XFER_DMA_NONE, dummy_id);	\
	}

#include "FN_ATTCH.I"

__COLD_ATTR__ static int ACTRL_LOCK_SET_XFER(APORT *p, int drive, unsigned mode, __const__ __u16 id[256])
{
	int r;
#if __DEBUG >= 2
	KERNEL$SLEEP(1);	/* test block */
#endif
	ATA_LOCK(p);
	if (!p->set_xfer) {
		r = -EOPNOTSUPP;
	} else {
		r = p->set_xfer(p, drive, mode, id);
	}
	ATA_UNLOCK(p);
	return r;
}

__COLD_ATTR__ static int ACTRL_AUX_CMD(ATA_ATTACH_PARAM *ap, int cmd, ...)
{
	APORT *p;
	int r;
	unsigned avail, avail_unsupp;
	va_list args;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ACTRL_AUX_CMD AT SPL %08X", KERNEL$SPL);
	p = ap->port;
	va_start(args, cmd);
	switch (cmd) {
		case ATA_AUX_GET_PROTOCOL_MASK: {
			__u16 *ident = va_arg(args, __u16 *);
			unsigned xfer_mask = va_arg(args, unsigned);
			r = (1 << ATA_PROTOCOL_NODATA) | (1 << ATA_PROTOCOL_PIO);
			if (!p->dma_io) break;
			if (!(p->aport_flags & (APORT_IMPLICIT_DMA | APORT_SATA))) {
				__u8 dmast = io_inb(p->dma_io + DMAPORT_STATUS);
				__u8 mask = !ap->device ? DMASTATUS_MASTER : DMASTATUS_SLAVE;
				if (!(dmast & mask)) {
					avail = 0, avail_unsupp = 0;
					if (p->get_avail_xfer) p->get_avail_xfer(p, ap->device, ident, &avail, &avail_unsupp);
					if (!((avail | avail_unsupp) & xfer_mask & IDE_XFER_MASK_DMA)) {
						break;
					}
				}
			}
			r |= 1 << ATA_PROTOCOL_DMA;
			break;
		}
		case ATA_AUX_SETUP_PIO_XFER:
		case ATA_AUX_SETUP_DMA_XFER: {
			__u16 *ident = va_arg(args, __u16 *);
			unsigned xfer_mask = va_arg(args, unsigned);
			u_jiffies_lo_t timeout = va_arg(args, u_jiffies_lo_t);
			avail = 0, avail_unsupp = 0;
			if (p->get_avail_xfer) p->get_avail_xfer(p, ap->device, ident, &avail, &avail_unsupp);
			if (!(p->aport_flags & APORT_SATA)) {
				ATA$CHECK_DEVICE_CABLE(ident, &avail, &avail_unsupp);
			}
			r = ATA$SETUP_XFER(ap, cmd == ATA_AUX_SETUP_DMA_XFER, ident, p->device[ap->device].dev_flags & IDE_XFER_MASK, xfer_mask, avail, avail_unsupp, ACTRL_LOCK_SET_XFER, timeout);
			break;
		}
		case ATA_AUX_FIXUP_IDENT: {
			__u16 *ident = va_arg(args, __u16 *);
			r = 0;
			if (p->fixup_ident) r = p->fixup_ident(p, ap->device, ident);
			break;
		}
		case ATA_AUX_GET_TRANSFER_SIZE: {
			int dma = va_arg(args, int);
			r = ATA$SGLIST_RECOMMENDED_SIZE(p->sglist, dma);
			if (p->recommended_length && p->recommended_length < r)
				r = p->recommended_length;
			break;
		}
		case ATA_AUX_SET_ATAPI_FLAGS: {
#include "FN_AFLAG.I"
			break;
		}
		default: {
			r = -ENOOP;
			break;
		}
	}
	va_end(args);
	return r;
}

static void ATA_INS(APORT *p, int bit32, void *data, unsigned len)
{
	if (bit32 && __likely(!(len & 1))) {
		io_insl(p->io + PORT_DATA, data, len >> 1);
	} else {
		io_insw(p->io + PORT_DATA, data, len);
	}
}

static void ATA_OUTS(APORT *p, int bit32, void *data, unsigned len)
{
	if (bit32 && __likely(!(len & 1))) {
		io_outsl(p->io + PORT_DATA, data, len >> 1);
	} else {
		io_outsw(p->io + PORT_DATA, data, len);
	}
}

#define CODE_SETUP_XFER(p, rq)					\
	if (__unlikely((p)->aport_flags & APORT_SETUP_XFER)) {	\
		(p)->setup_xfer(p, rq);				\
	}

#define CODE_SETUP_DMA(p, rq)					\
	setup_dma_generic(p, rq);

#define CODE_START_DMA(p, rq)					\
	if (__unlikely((p)->aport_flags & APORT_EXTRA_DMA)) {	\
		(p)->start_dma(p, rq);				\
	} else {						\
		start_dma_generic(p, rq);			\
	}

#define CODE_CHECK_ATAPI_DMA(p, rq)				\
	(__unlikely((p)->check_atapi_dma != NULL) && __unlikely((p)->check_atapi_dma(p, rq)))

#define CODE_MANGLE_REQUEST						\
	if (__unlikely(p->aport_flags & APORT_MANGLE_REQUEST))		\
		p->mangle(p, rq);

#define CODE_ERROR_CONDITION						\
	if (!(p->ctrl->actrl_flags & ACTRL_BOTCHED_IDENTIFY && (rq->fis.command == ATA_CMD_IDENTIFY_DEVICE || rq->fis.command == ATA_CMD_IDENTIFY_PACKET_DEVICE)))

#include "FN_CORE.I"

#include "FN_ATAPI.I"

static DECL_IRQ_AST(ACTRL_IRQ, SPL_ATA_SCSI, AST)
{
	ATARQ *rq;
	APORT *p = GET_STRUCT(RQ, APORT, irq_ast);
	__u8 dmastatus, status;
	dmastatus = 0;	/* against warning */
	if (__unlikely(p->aport_flags & APORT_EXTRA_DMA)) {
		dmastatus = p->stop_dma(p, DMASTATUS_IRQ | DMASTATUS_ERROR);
		if (__unlikely(!(dmastatus & (DMASTATUS_IRQ | DMASTATUS_ERROR)))) goto not_for_me;
	} else if (__likely(p->dma_io != 0)) {
/* ICH SATA (and maybe others) need to clear DMA IRQ even on non-dma commands */
		dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		if (__unlikely(!(dmastatus & (DMASTATUS_IRQ | DMASTATUS_ERROR)))) {
			not_for_me:
			rq = p->current_rq;
			if (rq && rq->atarq_flags & ATA_PROTOCOL_DMA) goto done;
	/* Maybe some controllers don't set DMASTATUS_IRQ on non-DMA transfers.
	   All controllers that I have set it. */
			goto test_pci_int;
		}
		stop_dma_generic(p, dmastatus);
	} else if (p->ctrl->on_pci) {
	/* The interrupt may be shared and caused by something else...
	   I'm not sure if reading STATUS is atomic to test-and-clear
	   interrupt (i.e. may it return BSY bit and clear ongoing interrupt?)
	   ... Linux and BSD also read ALTSTATUS here.
	*/
		test_pci_int:
		status = ATA_IN_ALTSTATUS(p);
		if (__unlikely(status & ATA_STATUS_BSY)) goto done;
	}

	status = ATA_IN_STATUS(p);
	rq = p->current_rq;
	if (__unlikely(!rq)) {
		/*
		This may happen because one of these:
		- int 15 is asseted by 2nd PIC if it has to send masked irq
		- interrupt line is dropped too slowly
		- during disk probing/reset
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, p->ctrl->dev_name, "INTERRUPT WITH NO ACTIVE REQUEST. DMA STATUS %02X, DRIVE STATUS %02X", dmastatus, status);
		*/
		goto done;
	}
	if (__unlikely(status & ATA_STATUS_BSY)) {
		if (__unlikely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
			/* some broken disks do this --- poll the disk */
			status = AD_POLL_SLOW_DISK(p, dmastatus, status);
			if (__unlikely(status & ATA_STATUS_BSY)) goto ad_error;
		} else {
			goto done;
		}
	}
	if (__unlikely(rq == &p->atapi_placeholder)) {
		ATAPI_IRQ(p, dmastatus, status);
	} else if (__likely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
		if (__unlikely(dmastatus & DMASTATUS_ERROR)) {
			ad_error:
			AD_ERROR(p, dmastatus, status, 0);
			goto done;
		}
		if (__unlikely((status & (ATA_STATUS_ERROR | ATA_STATUS_DRQ | ATA_STATUS_DF | ATA_STATUS_DRDY)) != ATA_STATUS_DRDY)) {
			goto ad_error;
		}
		ATA$UNMAP_DMA(p->sglist);
		ATA_END_REQUEST(p, 0);
	} else {
		ATA_NONDMA_IRQ(p, status);
	}

	done:
	KERNEL$ADD_RANDOMNESS(&p->random_ctx, NULL, 0);
	KERNEL$UNMASK_IRQ(p->irq_ctrl);
	RETURN;
}

__COLD_ATTR__ static void ATA_TIMEOUT(TIMER *t)
{
	int dmastatus;
	__u8 status;
	APORT *p = GET_STRUCT(t, APORT, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	SET_TIMER_NEVER(&p->timeout);
	dmastatus = -1;
	if (p->aport_flags & APORT_EXTRA_DMA) {
		dmastatus = p->stop_dma(p, 0);
	} else if (__likely(p->dma_io != 0)) {
		dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		stop_dma_generic(p, dmastatus);
	}
	status = ATA_IN_STATUS(p);
	AD_ERROR(p, dmastatus, status, 1);
}



__COLD_ATTR__ static char *ATA_FIND_TEST(const void *id_, pci_id_t id, unsigned long *rev_)
{
	__u32 dev_id = (__u32)(unsigned long)id_;
	__u8 rev_min = (__u8)(unsigned long)rev_;
	__u8 rev_max = (__u8)((unsigned long)rev_ >> 8);
	__u8 rv;
	if (dev_id != PCI$READ_CONFIG_DWORD(id, PCI_VENDOR_ID)) return NULL;
	rv = PCI$READ_CONFIG_BYTE(id, PCI_REVISION_ID);
	if (rv < rev_min || rv > rev_max) return NULL;
	return "";
}

__COLD_ATTR__ int ATA_FIND_DEV(ACTRL *a, __u32 dev_id, __u8 rev_min, __u8 rev_max)
{
	return !PCI$FIND_DEVICE((const void *)(unsigned long)dev_id, 0, 0, 0, ATA_FIND_TEST, NULL, NULL, (void *)(unsigned long)(rev_min + (rev_max << 8)), 1);
}

__COLD_ATTR__ static int ATA_IS_NATIVE(pci_id_t id, unsigned long flags, int port)
{
	__u16 class;
	__u8 prg_info;
	if (flags & ACTRL_ALWAYS_NATIVE)
		return 1;
	class = PCI$READ_CONFIG_WORD(id, PCI_CLASS_DEVICE);
	if (class != 0x0101)
		return 1;
	prg_info = PCI$READ_CONFIG_BYTE(id, PCI_CLASS_PROG);
	if (prg_info & (1 << (port << 1)))
		return 1;
	return 0;
}

/* type: 1 - std (0x1f0, 0x3f4), 2 - nonstd, 3 - both */

#define ATA_PCI_TEST_STD	1
#define ATA_PCI_TEST_NON_STD	2

__COLD_ATTR__ static char *ATA_PCI_TEST(const void *type_p, pci_id_t id, unsigned long *actrl_flags)
{
	char *name;
	__u16 command;
	int native;
	int type = (int)(unsigned long)type_p;
	if (!(name = PCI$TEST_LIST(ata_dev, id, actrl_flags))) return NULL;
	command = PCI$READ_CONFIG_WORD(id, PCI_COMMAND);
		/* Linux ata-generic.c: do not enable PCI_COMMAND_IO */
	if (!(command & PCI_COMMAND_IO)) return NULL;
	native = ATA_IS_NATIVE(id, *actrl_flags, 0) || ATA_IS_NATIVE(id, *actrl_flags, 1);
	if (!(type & (native ? ATA_PCI_TEST_NON_STD : ATA_PCI_TEST_STD))) return NULL;
	return name;
}

__COLD_ATTR__ void SATA_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned  *avail_unsupported)
{
	*avail = IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2 | IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4 | IDE_XFER_UDMA_5 | IDE_XFER_UDMA_6;
	*avail_unsupported = IDE_XFER_MASK & ~IDE_XFER_PIO_NONE & ~IDE_XFER_DMA_NONE & ~*avail;
}

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	int r;
	union {
		MALLOC_REQUEST mrq;
		DLLZRQ lz;
		char gstr[__MAX_STR_LEN];
	} u;
	ACTRL *a;
	char pci_dll;
	char on_pci = 0;
	pci_id_t pci_id = 0;
	char *chipset_name = NULL;
	unsigned long actrl_flags;
	pci_id_t id = 0, id_mask = 0;
	int order;
	char type;
	char some_enabled;
	int i;
	int nochipsetinit;
	int (*special_init)(ACTRL *a) = NULL;
	int bus_clock = -1;
	const char * const *arg;
	int state;
	const char *opt, *optend, *str;
		/* min SPL --- SPL_TIMER + 1 */
	static const struct __param_table params[3] = {
		"NO_CHIPSET_INIT", __PARAM_BOOL, ~0, 1,
		"BUS_CLOCK", __PARAM_INT, 20, 67,
		NULL, 0, 0, 0,
	};
	void *vars[3];
	vars[0] = &nochipsetinit;
	vars[1] = &bus_clock;
	vars[2] = NULL;
	if (DMACMD_READ != ATARQ_TO_DEVICE)
		KERNEL$SUICIDE("ACTRL: BITS DON'T MATCH: DMACMD_READ %02X, ATARQ_TO_DEVICE %02X", DMACMD_READ, ATARQ_TO_DEVICE);
	u.lz.handle = KERNEL$DLRQ();
	u.lz.caller = NULL;
	strcpy(u.lz.modname, "PCI");
	SYNC_IO_CANCELABLE(&u.lz, KERNEL$DL_LOAD_LAZY);
	pci_dll = u.lz.status >= 0;
	order = 0;
	type = 0;
	nochipsetinit = 0;
	arg = argv;
	state = 0;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (!pci_dll) {
			if (__strcasexcmp("PCI", opt, optend)) goto bads;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: PCI NOT ENABLED: %s", u.lz.error);
			r = -ENODEV;
			goto ret0;
		}
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			bads:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		} else type = 1;
	}
	if (type == 1) {
		if (pci_dll && !PCI$FIND_DEVICE((const void *)(id_mask ? ATA_PCI_TEST_STD | ATA_PCI_TEST_NON_STD : ATA_PCI_TEST_NON_STD), id, id_mask, order, ATA_PCI_TEST, &pci_id, &chipset_name, &actrl_flags, 0)) {
			on_pci = 1;
		} else {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: NO PCI DEVICE FOUND");
			r = -ENODEV;
			goto ret0;
		}
	} else {
		if (pci_dll && !PCI$FIND_DEVICE((const void *)ATA_PCI_TEST_STD, id, id_mask, order, ATA_PCI_TEST, &pci_id, &chipset_name, &actrl_flags, 0)) {
			on_pci = 1;
		}
	}
	if (on_pci) {
		PCI$ENABLE_DEVICE(pci_id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	} else {
		actrl_flags = 0;
	}

	if (on_pci && !nochipsetinit) {
		for (i = 0; i < sizeof(special_controllers) / sizeof(*special_controllers); i++) {
			r = special_controllers[i](pci_id, &special_init);
			if (r) {
				goto special_controller;
			}
		}
	}
	if (actrl_flags & ACTRL_SPECIAL_PORTS) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: CAN'T INITIALIZE NONSTANDARD CONTROLLER WITHOUT CHIPSET DRIVER");
		r = -ENXIO;
		goto ret1;
	}

	r = 2;
	special_controller:
	u.mrq.size = sizeof(ACTRL) + (r - 1) * sizeof(APORT);
	SYNC_IO_CANCELABLE(&u.mrq, &KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret1;
	}
	a = u.mrq.ptr;
	memset(a, 0, u.mrq.size);

	a->n_ports = r;
	a->pci_dll = pci_dll;
	a->pci_id = pci_id;
	a->on_pci = on_pci;
	if (chipset_name && !chipset_name) chipset_name = NULL;
	a->chipset_name = chipset_name;
	a->actrl_flags = actrl_flags;

	if (a->on_pci) {
		a->dev_id = PCI$READ_CONFIG_DWORD(a->pci_id, PCI_VENDOR_ID);
		a->dev_rev = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_REVISION_ID);
		if (bus_clock == -1) bus_clock = 33;
	}
	if (bus_clock == -1) bus_clock = 50;
	if (bus_clock == 33) a->bus_clock = 33333;
	else if (bus_clock == 37) a->bus_clock = 37500;
	else if (bus_clock == 41) a->bus_clock = 41666;
	else a->bus_clock = bus_clock * 1000;

	/* Initialize ports */
	for (i = 0; i < a->n_ports; i++) {
		a->port[i].n = i;
		if (a->actrl_flags & ACTRL_SATA) a->port[i].aport_flags |= APORT_SATA;
		if (a->actrl_flags & ACTRL_LIMIT_PROBE) a->port[i].probe_limit = PROBE_LIMIT_JIFFIES;
		a->port[i].ctrl = a;
		INIT_TIMER(&a->port[i].timeout);
	}
	if (special_init) {
		r = special_init(a);
		if (r < 0) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CHIPSET DRIVER REFUSED TO INITIALIZE THE CONTROLLER", a->dev_name);
			goto ret2;
		}
	} else for (i = 0; i < 2; i++) {
		io_t base_port, alt_port, dma_port;
		int irq;
		if (a->on_pci) {
			dma_port = PCI$READ_IO_RESOURCE(a->pci_id, 4);
			if (dma_port) dma_port += i * 8;
			if (ATA_IS_NATIVE(a->pci_id, a->actrl_flags, i)) {
				base_port = PCI$READ_IO_RESOURCE(a->pci_id, !i ? 0 : 2);
				alt_port = PCI$READ_IO_RESOURCE(a->pci_id, !i ? 1 : 3);
				irq = PCI$READ_INTERRUPT_LINE(a->pci_id);
			} else {
				goto legacy;
			}
			/*__debug_printf("pci read %d: base = %08x, alt = %08x, dma = %08x, irq=%d\n", i, base_port, alt_port, dma_port, irq);*/
		} else {
			dma_port = 0;
			legacy:
			base_port = !i ? 0x1f0 : 0x170;
			alt_port = base_port + 0x204;
			irq = !i ? 14 : 15;
		}
		if (!base_port || !alt_port) {
			a->port[i].aport_flags = APORT_DISABLED;
			continue;
		}
		a->port[i].io = base_port;
		a->port[i].alt_io = alt_port;
		a->port[i].dma_io = dma_port;
		a->port[i].irq = irq;
	}

	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED))
		goto at_least_one_enabled_port;
	r = -ENOENT;
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: NO PORT IS ENABLED");
	goto ret2;
	at_least_one_enabled_port:

	if (a->n_ports == 2 && (a->port[0].io == 0x1f0 || a->port[1].io == 0x170)) _snprintf(a->dev_name, __MAX_STR_LEN, "ATA$ACTRL");
	else if (a->on_pci) _snprintf(a->dev_name, __MAX_STR_LEN, "ATA$ACTRL@" PCI_ID_FORMAT, a->pci_id);
	else _snprintf(a->dev_name, __MAX_STR_LEN, "ATA$ACTRL@" IO_ID, !(a->port[0].aport_flags & APORT_DISABLED) ? a->port[0].io : a->port[1].io);

	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		a->port[i].irq_ast.fn = &ACTRL_IRQ;
		a->port[i].atapi_placeholder.port = &a->port[i];
	}

	if (!nochipsetinit) {
		for (i = 0; i < sizeof(controllers) / sizeof(*controllers); i++) {
			r = controllers[i](a);
			if (r < 0) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CHIPSET DRIVER REFUSED TO INITIALIZE THE CONTROLLER", a->dev_name);
				goto ret2;
			}
			if (r > 0) break;
		}
	}

	for (i = 0; i < a->n_ports; i++) {
		ASGLIST *sg = ATA$ALLOC_SGLIST(a->dev_name, &a->port[i].sglist_dev);
		if (__unlikely(__IS_ERR(sg))) {
			if (__PTR_ERR(sg) != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", a->dev_name, strerror(-__PTR_ERR(sg)));
			r = __PTR_ERR(sg);
			goto ret2;
		}
		a->port[i].sglist = sg;
	}

	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		a->port[i].io_range.start = a->port[i].io;
		a->port[i].io_range.len = PORT_RANGE;
		a->port[i].io_range.name = a->dev_name;
		if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&a->port[i].io_range))) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR PORT %d: %s", a->port[i].io_range.start, a->port[i].io_range.start + a->port[i].io_range.len - 1, i, strerror(-r));
			goto disable0;
		}
		a->port[i].alt_io_range.start = a->port[i].alt_io + ALTPORT_OFFSET;
		a->port[i].alt_io_range.len = ALTPORT_RANGE;
		a->port[i].alt_io_range.name = a->dev_name;
		if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&a->port[i].alt_io_range))) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET ALT IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR PORT %d: %s", a->port[i].alt_io_range.start, a->port[i].alt_io_range.start + a->port[i].alt_io_range.len - 1, i, strerror(-r));
			goto disable1;
		}
		if (a->port[i].dma_io) {
			__u8 dmast;
			if (i && a->port[0].dma_io && (dmast = io_inb(a->port[0].dma_io + DMAPORT_STATUS)) & DMASTATUS_SIMPLEX && !(a->actrl_flags & ACTRL_IGNORE_SIMPLEX)) {
				if (a->actrl_flags & ACTRL_TRY_NONSIMPLEX) {
					dmast &= ~DMASTATUS_SIMPLEX;
					io_outb(a->port[0].dma_io + DMAPORT_STATUS, dmast);
					dmast = io_inb(a->port[0].dma_io + DMAPORT_STATUS);
					if (!(dmast & DMASTATUS_SIMPLEX))
						goto dma_ok;
				}
				goto disable_dma;
			}
			dma_ok:
			a->port[i].dma_io_range.start = a->port[i].dma_io;
			a->port[i].dma_io_range.len = DMAPORT_RANGE;
			a->port[i].dma_io_range.name = a->dev_name;
			if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&a->port[i].dma_io_range))) {
				KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET DMA IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR PORT %d: %s", a->port[i].dma_io_range.start, a->port[i].dma_io_range.start + a->port[i].dma_io_range.len - 1, i, strerror(-r));
				a->port[i].dma_io = 0;
				goto disable_dma;
			}
			if (__unlikely(io_inb(a->port[0].dma_io + DMAPORT_STATUS) == 0xff)) {
				KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
				disable_dma:
				a->port[i].dma_io = 0;
			}
		}
		if (a->port[i].dma_io) {
				/* clear possible DMA errors */
			if (a->port[i].aport_flags & APORT_EXTRA_DMA) {
				a->port[i].stop_dma(&a->port[i], 0);
			} else if (a->port[i].dma_io) {
				stop_dma_generic(&a->port[i], io_inb(a->port[i].dma_io + DMAPORT_STATUS));
			}
		}
		if (0) {
			disable1:
			KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
			disable0:
			a->port[i].aport_flags |= APORT_DISABLED;
		}
	}
	some_enabled = 0;
	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		if ((r = KERNEL$REQUEST_IRQ(a->port[i].irq, &a->port[i].irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &a->port[i].irq_ast, a->dev_name)) < 0) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET IRQ %d FOR PORT %d: %s", a->port[i].irq, i, strerror(-r));

			if (a->port[i].dma_io) KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
			KERNEL$UNREGISTER_IO_RANGE(&a->port[i].alt_io_range);
			KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
			a->port[i].aport_flags |= APORT_DISABLED;
		} else {
			some_enabled = 1;
		}
	}

	if (!some_enabled) {
		r = -ENXIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ENABLE ANY PORT", a->dev_name);
		goto ret3;
	}

	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		if ((a->port[i].aport_flags & (APORT_SATA | APORT_IMPLICIT_DMA)) == APORT_SATA) {
			if (!a->port[i].set_xfer) {
				a->port[i].set_xfer = ATA$SET_XFER_EMPTY;
			}
			if (!a->port[i].get_avail_xfer) {
				a->port[i].get_avail_xfer = SATA_GET_AVAIL_XFER;
			}
		}
	}

	if (a->on_pci) {
		char *str = "";
		for (i = 0; i < a->n_ports; i++) {
			if (a->port[i].aport_flags & APORT_DISABLED) continue;
			if (a->port[i].aport_flags & APORT_SATA) str = "S";
		}
		if (*str) for (i = 0; i < a->n_ports; i++) {
			if (a->port[i].aport_flags & APORT_DISABLED) continue;
			if (!(a->port[i].aport_flags & APORT_SATA)) str = "ATA/S";
		}
		_printf("%s: %sATA CONTROLLER ON PCI: %s\n", a->dev_name, str, PCI$ID(u.gstr, a->pci_id));
	}
	if (a->chipset_name) _printf("%s: %s\n", a->dev_name, a->chipset_name);
	for (i = 0; i < a->n_ports; i++) {
		if (a->port[i].aport_flags & APORT_DISABLED) _printf("%s: PORT %d DISABLED\n", a->dev_name, i);
		else if (!a->port[i].dma_io) _printf("%s: PORT %d: IO "IO_FORMAT", ALTIO "IO_FORMAT", IRQ %d\n", a->dev_name, i, a->port[i].io, a->port[i].alt_io, a->port[i].irq);
		else _printf("%s: PORT %d: IO "IO_FORMAT", ALTIO "IO_FORMAT", DMAIO "IO_FORMAT", IRQ %d\n", a->dev_name, i, a->port[i].io, a->port[i].alt_io, a->port[i].dma_io, a->port[i].irq);
	}

	r = KERNEL$REGISTER_DEVICE(a->dev_name, "ACTRL.SYS", 0, a, NULL, ATA_DCALL, "ATA,SCSI", NULL, ATA_UNLOAD, &a->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", a->dev_name, strerror(-r));
		goto ret3;
	}
	a->dlrq = KERNEL$TSR_IMAGE();
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s", a->dev_name);
	return 0;

	ret3:
	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		KERNEL$RELEASE_IRQ(a->port[i].irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &a->port[i].irq_ast);
		if (a->port[i].dma_io) KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].alt_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
	}
	ret2:
	for (i = 0; i < a->n_ports; i++) if (a->port[i].sglist) ATA$FREE_SGLIST(a->dev_name, a->port[i].sglist);
	if (a->chipset_dtor) a->chipset_dtor(a);
	free(a);
	ret1:
	if (on_pci) PCI$FREE_DEVICE(pci_id);
	ret0:
	return r;
}

__COLD_ATTR__ static int ATA_UNLOAD(void *p, void **dlrq, const char * const argv[])
{
	int r;
	int i, j;
	ACTRL *a = p;
	RAISE_SPL(SPL_DEV);
	while (a->dcall_users) KERNEL$SLEEP(1);
	for (i = 0; i < a->n_ports; i++) for (j = 0; j < 2; j++) if (a->port[i].device[j].attached) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS CONTROLLER", a->dev_name);
		return -EBUSY;
	}
	if ((r = KERNEL$DEVICE_UNLOAD(a->lnte, argv))) {
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	LOWER_SPL(SPL_ZERO);
	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		if (__unlikely(a->port[i].setup_xfer != NULL)) {
			int j;
			for (j = 0; j < 2; j++) {
				ATARQ rq;
				memset(&rq, 0, sizeof rq);
				rq.atarq_flags = ATA_PROTOCOL_PIO;
				rq.port = &a->port[i];
				rq.device = j;
				a->port[i].setup_xfer(&a->port[i], &rq);
			}
		}
		KERNEL$RELEASE_IRQ(a->port[i].irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &a->port[i].irq_ast);
		if (a->port[i].dma_io) KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].alt_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
	}
	if (a->chipset_dtor) a->chipset_dtor(a);
	if (a->on_pci) PCI$FREE_DEVICE(a->pci_id);
	for (i = 0; i < a->n_ports; i++) if (a->port[i].sglist) ATA$FREE_SGLIST(a->dev_name, a->port[i].sglist);
	*dlrq = a->dlrq;
	free(a);
	return 0;
}

