#include "ACTRL.H"

#define ALI_PCI_CABLE		0x4a
#define ALI_PCI_FLAGS_4b	0x4b
#define ALI_PCI_CDROM		0x53
#define ALI_PCI_FIFO		0x54
#define ALI_PCI_UDMA		0x56
#define ALI_PCI_SETUP		0x58
#define ALI_PCI_8B		0x59
#define ALI_PCI_16B		0x5a

#define ALI_EARLY	0
#define ALI_20		1
#define ALI_20_UDMA	2
#define ALI_C2		3
#define ALI_C3		4
#define ALI_C4		5
#define ALI_C5		6
#define ALI_MASK	7

static void ALI_LBA48_MANGLE(APORT *p, ATARQ *rq);
static __u8 ALI_STOP_DMA(APORT *p, __u8 mask);
static int ALI_FIXUP_IDENT(APORT *p, int drive, __u16 ident[256]);
static void ALI_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported);
static int ALI_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256]);

static const struct pci_id_s southbridge_table[] = {
	{ 0x10b9, 0x1533, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

static const struct pci_id_s northbridge_table[] = {
	{ 0x10b9, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

__COLD_ATTR__ int DETECT_ALI(ACTRL *a)
{
	pci_id_t southbridge_id;
	__u8 southbridge_rev = 0;
	int southbridge_present = 0;
	pci_id_t north_id, north_mask;
	int north_skip;
	const char *b, *bb;
	int northbridge_present = 0;
	int i;
	if (a->dev_id != 0x522810b9 &&
	    a->dev_id != 0x522910b9) return 0;

	if (!PCI$FIND_DEVICE(southbridge_table, 0, 0, 0, PCI$TEST_LIST, &southbridge_id, NULL, NULL, 1)) {
		southbridge_rev = PCI$READ_CONFIG_BYTE(southbridge_id, PCI_REVISION_ID);
		southbridge_present = 1;
	}
	north_id = 0;
	north_mask = 0;
	north_skip = 0;
	b = "PCI";
	bb = strchr(b, 0);
	PCI$PARSE_PARAMS(b, bb, "BUS=0", &north_id, &north_mask, &north_skip);
	PCI$PARSE_PARAMS(b, bb, "DEV=0", &north_id, &north_mask, &north_skip);
	PCI$PARSE_PARAMS(b, bb, "FN=0", &north_id, &north_mask, &north_skip);
	if (!PCI$FIND_DEVICE(northbridge_table, north_id, north_mask, north_skip, PCI$TEST_LIST, NULL, NULL, NULL, 1)) northbridge_present = 1;

	if (a->dev_rev < 0x20) {
		a->chipset_name = "ALI (EARLY)";
		a->chipset_flags = ALI_EARLY;
		for (i = 0; i < a->n_ports; i++) {
			a->port[i].dma_io = 0;
		}
	} else if (a->dev_rev < 0xc2) {
		a->chipset_name = "ALI (REV 20)";
		a->chipset_flags = ALI_20;
		if (southbridge_present && southbridge_rev >= 0x20 && southbridge_rev < 0xc2) {
			__u8 udma_cap = PCI$READ_CONFIG_BYTE(southbridge_id, 0x5e);
			if ((udma_cap & 0x1e) == 0x12) {
				a->chipset_name = "ALI (REV 20 UDMA)";
				a->chipset_flags = ALI_20_UDMA;
			}
		}
	} else if (a->dev_rev == 0xc2) {
		a->chipset_name = "ALI (REV C2)";
		a->chipset_flags = ALI_C2;
	} else if (a->dev_rev == 0xc3) {
		a->chipset_name = "ALI (REV C3)";
		a->chipset_flags = ALI_C3;
	} else if (a->dev_rev == 0xc4) {
		a->chipset_name = "ALI (REV C4)";
		a->chipset_flags = ALI_C4;
	} else {
		a->chipset_name = "ALI (REV C5)";
		a->chipset_flags = ALI_C5;
	}

	if ((a->chipset_flags & ALI_MASK) >= ALI_C2) {
		__u8 flags = PCI$READ_CONFIG_BYTE(a->pci_id, ALI_PCI_FLAGS_4b);
		flags |= 0x08;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_FLAGS_4b, flags);
	} else if ((a->chipset_flags & ALI_MASK) >= ALI_20) {
		__u8 flags = PCI$READ_CONFIG_BYTE(a->pci_id, ALI_PCI_FLAGS_4b);
		flags |= ~0x80;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_FLAGS_4b, flags);
	}

	if (northbridge_present && southbridge_present) {
		__u8 reg79 = PCI$READ_CONFIG_BYTE(southbridge_id, 0x79);
		if (a->dev_rev == 0xc2) {
			PCI$WRITE_CONFIG_BYTE(southbridge_id, 0x79, reg79 | 0x04);
		} else if (a->dev_rev > 0xc2 && a->dev_rev < 0xc5) {
			PCI$WRITE_CONFIG_BYTE(southbridge_id, 0x79, reg79 | 0x02);
		}
	}

	if (a->dev_rev >= 0x20) {
		__u8 cd = PCI$READ_CONFIG_BYTE(a->pci_id, ALI_PCI_CDROM);
		if (a->dev_rev == 0x20) {
			cd &= ~0x02;
		} else if (a->dev_rev < 0xc7) {
			cd |= 0x01;
		} else {
			cd |= 0x03;
		}
		PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_CDROM, cd);
	}

	if ((a->chipset_flags & ALI_MASK) >= ALI_20 && 
	    (a->chipset_flags & ALI_MASK) <= ALI_C4) {
		for (i = 0; i < a->n_ports; i++) {
			a->port[i].aport_flags |= APORT_MANGLE_REQUEST;
			a->port[i].mangle = ALI_LBA48_MANGLE;
			a->port[i].recommended_length = 255 << BIO_SECTOR_SIZE_BITS;
		}
	}

	for (i = 0; i < a->n_ports; i++) {
		a->port[i].fixup_ident = ALI_FIXUP_IDENT;
		a->port[i].get_avail_xfer = ALI_GET_AVAIL_XFER;
		a->port[i].set_xfer = ALI_SET_XFER;
		a->port[i].xfer_mode[0][0] = 0xff;
		a->port[i].xfer_mode[0][1] = 0xff;
		a->port[i].xfer_mode[1][0] = 0xff;
		a->port[i].xfer_mode[1][1] = 0xff;
		a->port[i].aport_flags |= APORT_EXTRA_DMA;
		a->port[i].start_dma = start_dma_generic;
		a->port[i].stop_dma = ALI_STOP_DMA;
	}

	return 1;
}

/*
 * The controller hates LBA48 DMA commands.
 * --- turn them into LBA28 DMA or LBA48 PIO.
 */

static void ALI_LBA48_MANGLE(APORT *p, ATARQ *rq)
{
	static const __u8 pio_commands[4] = {
		ATA_CMD_READ_EXT, ATA_CMD_WRITE_EXT,
		ATA_CMD_READ_MULTIPLE_EXT, ATA_CMD_WRITE_MULTIPLE_EXT,
	};
	static const __u8 dma_commands[2] = {
		ATA_CMD_READ_DMA, ATA_CMD_WRITE_DMA,
	};
	int rw;
	if (__likely((rq->atarq_flags & (ATA_PROTOCOL_DMA | ATARQ_VALID_48BIT | ATARQ_SET_SIZE)) != (ATA_PROTOCOL_DMA | ATARQ_VALID_48BIT | ATARQ_SET_SIZE)))
		return;
	if (__likely(rq->fis.command == ATA_CMD_READ_DMA_EXT)) {
		rw = 0;
	} else if (__likely(rq->fis.command == ATA_CMD_WRITE_DMA_EXT)) {
		rw = 1;
	} else {
		return;
	}
	if (__unlikely(rq->fis.lba24 & 0xf0) || __unlikely(rq->fis.lba32) || __unlikely(rq->fis.lba40) || __unlikely(!(rq->atarq_flags & ATARQ_SET_SIZE))) {
		rw += (rq->multicount != 1) << 1;
		rq->fis.command = pio_commands[rw];
		rq->atarq_flags ^= ATA_PROTOCOL_DMA ^ ATA_PROTOCOL_PIO;
	} else {
		rq->fis.command = dma_commands[rw];
		rq->atarq_flags &= ~ATARQ_VALID_48BIT;
		if (__likely(rq->len > 255 << BIO_SECTOR_SIZE_BITS)) rq->len = 255 << BIO_SECTOR_SIZE_BITS;
		rq->fis.device = rq->fis.lba24 | ATA_DEVICE_LBA;
	}
}

/*
 * On ServerWorks mainboard, the card will lockup on any write to dma port.
 * To at least not crash when probing hardware and allow to use disk in
 * PIO mode, write to the ports only if there's some active dma.
 */

static __u8 ALI_STOP_DMA(APORT *p, __u8 mask)
{
	__u8 dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
	if (__unlikely(!(dmastatus & mask)) && __likely(mask)) return dmastatus;
	if (__likely(io_inb(p->dma_io + DMAPORT_CMD))) {
		stop_dma_generic(p, dmastatus);
	}
	return dmastatus;
}

__COLD_ATTR__ static int ALI_FIXUP_IDENT(APORT *p, int drive, __u16 ident[256])
{
	int atapi = (ident[0] & 0xc000) == 0x8000;
	if (atapi) return 0;
	ident[83] &= ~0x0002;	/* Turn off QUEUED DMA */
	ident[84] &= ~0x00c0;	/* Turn off FUA */
	ident[86] &= ~0x0002;	/* Turn off QUEUED DMA */
	ident[87] &= ~0x00c0;	/* Turn off FUA */
	return 0;
}

static const char * const BAD_UDMA_20[] = {
	"WDC*",
	NULL
};

#if 0
static const char * const BAD_DMA[] = {
	"TS*",
	NULL
};
#endif

__COLD_ATTR__ static void ALI_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	int atapi = ATA$IS_ATAPI(ident);
	char model[41];
	ATA$GET_MODEL(ident, &model, NULL, NULL);
	*avail = 0;
	*avail_unsupported = 0;
	switch (p->ctrl->chipset_flags & ALI_MASK) {
		case ALI_C5:
			*avail |= IDE_XFER_UDMA_6;
		case ALI_C4:
			*avail |= IDE_XFER_UDMA_5;
		case ALI_C3:
		case ALI_C2:
			*avail |= IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4;
			if (PCI$READ_CONFIG_BYTE(p->ctrl->pci_id, ALI_PCI_CABLE) & (1 << p->n)) {
				ATA$CABLE40_RESTRICT(avail, avail_unsupported);
			}
		case ALI_20_UDMA:
			*avail |= /*IDE_XFER_UDMA_0 |*/ IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2;
			*avail_unsupported |= IDE_XFER_UDMA_0;
		case ALI_20:
			*avail |= IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2 | IDE_XFER_WDMA_3 | IDE_XFER_WDMA_4;
		default:
			*avail |= IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_PIO_5 | IDE_XFER_PIO_6;
	};
	switch (p->ctrl->chipset_flags & ALI_MASK) {
		case ALI_20_UDMA:
		case ALI_20:
			ATA$BLACKLIST_RESTRICT(BAD_UDMA_20, ident, IDE_XFER_MASK_UDMA, avail, avail_unsupported);
			if (atapi) {
				*avail &= ~IDE_XFER_MASK_DMA;
				*avail_unsupported &= ~IDE_XFER_MASK_DMA;
			}
	}
	if (atapi) {
		*avail_unsupported |= *avail & IDE_XFER_MASK_DMA;
		*avail &= ~IDE_XFER_MASK_DMA;
	}
#if 0
	ATA$BLACKLIST_RESTRICT(BAD_DMA, ident, IDE_XFER_MASK_DMA, avail, avail_unsupported);
#endif
}

__COLD_ATTR__ static void ALI_SET_FIFO(APORT *p, int drive, int fifo)
{
	pci_id_t id = p->ctrl->pci_id;
	unsigned port = ALI_PCI_FIFO + p->n;
	unsigned shift = 4 * drive;
	__u8 val = PCI$READ_CONFIG_BYTE(id, port);
	val &= ~(0xf << shift);
	val |= fifo << shift;
	PCI$WRITE_CONFIG_BYTE(id, port, val);
}

__COLD_ATTR__ static int ALI_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	struct ata_timing t;
	int udma_idx;
	ACTRL *a = p->ctrl;
	if (mode & IDE_XFER_MASK_PIO) {
		p->xfer_mode[0][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
	} else {
		p->xfer_mode[1][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_DMA);
	}
	ATA$INIT_TIMING(&t);
	ATA$MERGE_PIO_TIMING(&t, p->xfer_mode[0][drive], ATA_MERGE_ALL);
	ATA$MERGE_DMA_TIMING(&t, p->xfer_mode[1][drive], ATA_MERGE_ALL);
	ATA$MERGE_PIO_TIMING(&t, p->xfer_mode[0][drive ^ 1], ATA_MERGE_SETUP | ATA_MERGE_8BIT);
	ATA$MERGE_DMA_TIMING(&t, p->xfer_mode[1][drive ^ 1], ATA_MERGE_SETUP | ATA_MERGE_8BIT);

	ATA$QUANTIZE_TIMING(&t, a->bus_clock, 0);


	if (ATA$IS_ATAPI(id)) ALI_SET_FIFO(p, drive, 0);

	if (t.setup)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_SETUP + 4 * p->n, ATA$FIT_TIMING(t.setup, 1, 8, 0) & 7);
	if (t.act8b && t.rec8b)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_8B + 4 * p->n, ((ATA$FIT_TIMING(t.act8b, 1, 8, 0) & 7) << 4) | (ATA$FIT_TIMING(t.rec8b, 1, 16, 0) & 15));
	if (t.active && t.recover)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_16B + 4 * p->n + drive, ((ATA$FIT_TIMING(t.active, 1, 8, 0) & 7) << 4) | (ATA$FIT_TIMING(t.recover, 1, 16, 0) & 15));
	udma_idx = (__s8)p->xfer_mode[1][drive] - __BSF_CONST(IDE_XFER_UDMA_0 / IDE_XFER_SDMA_0);
	if ((a->chipset_flags & ALI_MASK) >= ALI_20_UDMA) {
		unsigned udma_port, udma_shift, udma, ut;
		if (udma_idx >= 0) {
			static const __u8 ali_udma_timing[7] = {
				0xC, 0xB, 0xA, 0x9, 0x8, 0xF, 0xD
			};
			ut = ali_udma_timing[udma_idx];
		} else {
			ut = 0;
		}
/*__debug_printf("udma timing: %02x\n", ut);*/
		udma_port = ALI_PCI_UDMA + p->n;
		udma_shift = drive * 4;
		udma = PCI$READ_CONFIG_BYTE(a->pci_id, udma_port);
		udma &= ~(0xf << udma_shift);
		udma |= ut << udma_shift;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, udma_port, udma);
		if (udma_idx >= 3) {
			__u8 flags = PCI$READ_CONFIG_BYTE(a->pci_id, ALI_PCI_FLAGS_4b);
			flags |= 0x01;
			PCI$WRITE_CONFIG_BYTE(a->pci_id, ALI_PCI_FLAGS_4b, flags);
		}
	}

	if (!ATA$IS_ATAPI(id)) ALI_SET_FIFO(p, drive, udma_idx >= 0 ? 8 : 5);

	return 0;
}

