#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ATA.H>

#include "ACTRLREG.H"
#include "ACTRL.H"

#define HPT_370		0x00000001
#define HPT_370A	0x00000002
#define HPT_372		0x00000004
#define HPT_372A	0x00000008
#define HPT_302		0x00000010
#define HPT_371		0x00000020
#define HPT_374		0x00000040
#define HPT_FN1		0x00000080

struct hpt_timing {
	__u32 reg[N_PIO_MODES + N_DMA_MODES];
};

static const struct hpt_timing hpt37x_timing_33 = {{
	0x06814ea7,
	0x06814e93,
	0x06414e53,
	0x06414e42,
	0x06414e31,
	0, 0, 0, 0, 0, 0,
	0x22406c97,
	0x22406c33,
	0x22406c31,
	0, 0,
	0x12506297,
	0x124c6233,
	0x12486231,
	0x126c6231,
	0x12446231,
	0x12446231,
	0x12446231,
}};

static const struct hpt_timing hpt37x_timing_50 = {{
	0x0ac1f48a,
	0x0ac1f465,
	0x0a81f454,
	0x0a81f443,
	0x0a81f442,
	0, 0, 0, 0, 0, 0,
	0x228082ea,
	0x22808254,
	0x22808242,
	0, 0,
	0x121882ea,
	0x12148254,
	0x120c8242,
	0x128c8242,
	0x12ac8242,
	0x12848242,
	0x12848242,
}};

static const struct hpt_timing hpt37x_timing_66 = {{
	0x0d029d5e,
	0x0d029d26,
	0x0c829ca6,
	0x0c829c84,
	0x0c829c62,
	0, 0, 0, 0, 0, 0,
	0x2c829d2e,
	0x2c829c66,
	0x2c829c62,
	0, 0,
	0x1c829c62,
	0x1c9a9c62,
	0x1c929c62,
	0x1c8e9c62,
	0x1c8a9c62,
	0x1cae9c62,
	0x1c869c62,
}};

static const struct hpt_chip {
	unsigned char base;
	unsigned char udma;
	const struct hpt_timing *timing[4];
} hpt_chips[7] = {
  { 48, 5, { &hpt37x_timing_33, NULL, NULL, NULL } },
  { 48, 5, { &hpt37x_timing_33, NULL, &hpt37x_timing_50, NULL } },
  { 55, 6, { &hpt37x_timing_33, NULL, &hpt37x_timing_50, &hpt37x_timing_66 } },
  { 66, 6, { &hpt37x_timing_33, NULL, &hpt37x_timing_50, &hpt37x_timing_66 } },
  { 66, 6, { &hpt37x_timing_33, NULL, &hpt37x_timing_50, &hpt37x_timing_66 } },
  { 66, 6, { &hpt37x_timing_33, NULL, &hpt37x_timing_50, &hpt37x_timing_66 } },
  { 48, 5, { &hpt37x_timing_33, NULL, NULL, NULL } },
};

__COLD_ATTR__ static const struct hpt_chip *GET_HPT_CHIP(unsigned chipset_flags)
{
	return &hpt_chips[__BSF(chipset_flags)];
}

static const char * const BAD_DMA[] = {
	"IC35L*",
	"TS*",
	NULL
};

static const char * const BAD_UDMA_33[] = {
	"Maxtor*",
	NULL
};

static const char * const BAD_UDMA_100[] = {
	"IBM-DTLA-*",
	"WDC AC310200R",
	NULL
};

__COLD_ATTR__ static void HPT_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	int max_udma;
	const struct hpt_chip *hpt_chip;
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2;
	*avail_unsupported = 0;
	hpt_chip = GET_HPT_CHIP(p->ctrl->chipset_flags);
	max_udma = hpt_chip->udma;
	*avail |= (IDE_XFER_UDMA_0 << (max_udma + 1)) - IDE_XFER_UDMA_0;
	if (p->ctrl->chipset_flags & HPT_370) ATA$BLACKLIST_RESTRICT(BAD_UDMA_33, ident, IDE_XFER_MASK_UDMA, avail, avail_unsupported);
	if (p->ctrl->chipset_flags & (HPT_370 | HPT_370A)) ATA$BLACKLIST_RESTRICT(BAD_UDMA_100, ident, IDE_XFER_UDMA_5 | IDE_XFER_UDMA_6, avail, avail_unsupported);
	ATA$BLACKLIST_RESTRICT(BAD_DMA, ident, IDE_XFER_MASK_DMA, avail, avail_unsupported);
}

__COLD_ATTR__ static int HPT_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	ACTRL *a = p->ctrl;
	unsigned addr1, addr2;
	__u8 fast;
	unsigned idx;
	struct hpt_timing *timing;
	if (mode & IDE_XFER_MASK_PIO) {
		p->xfer_mode[0][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
	} else {
		p->xfer_mode[1][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK);
	}
	addr1 = 0x40 + 4 * (drive + p->n * 2);
	addr2 = 0x51 + 4 * p->n;

	fast = PCI$READ_CONFIG_BYTE(a->pci_id, addr2);
	if (a->chipset_flags & (HPT_370 | HPT_370A)) {
		fast &= ~0x02;
		fast |= 0x01;
	} else {
		fast &= ~0x07;
	}
	PCI$WRITE_CONFIG_BYTE(a->pci_id, addr2, fast);

	if (p->xfer_mode[1][drive] < N_PIO_MODES + N_DMA_MODES) {
		idx = p->xfer_mode[1][drive];
	} else {
		idx = p->xfer_mode[0][drive];
		if (idx >= N_PIO_MODES) idx = 0;
	}
	timing = (struct hpt_timing *)a->chipset_flags_2;
	while (!timing->reg[idx]) idx--;
	/* It is very weird in Linux driver, the code doesn't correspond to
	   the bit definitions above. */
	PCI$WRITE_CONFIG_DWORD(a->pci_id, addr1, timing->reg[idx]);
	return 0;
}

static void HPT370_START_DMA(APORT *p, ATARQ *rq)
{
	PCI$WRITE_CONFIG_BYTE(p->ctrl->pci_id, 0x50 + p->n * 4, 0x37);
	KERNEL$UDELAY(10);
	start_dma_generic(p, rq);
}

static __u8 HPT370_STOP_DMA(APORT *p, __u8 mask)
{
	__u8 dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
	if (__unlikely(!(dmastatus & mask)) && __likely(mask)) return dmastatus;
	if (__unlikely(dmastatus & DMASTATUS_ACTIVE)) {
		KERNEL$UDELAY(20);
		dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		if (__unlikely(dmastatus & DMASTATUS_ACTIVE)) {
			PCI$WRITE_CONFIG_BYTE(p->ctrl->pci_id, 0x50 + p->n * 4, 0x37);
			KERNEL$UDELAY(10);
			stop_dma_generic(p, dmastatus);
			PCI$WRITE_CONFIG_BYTE(p->ctrl->pci_id, 0x50 + p->n * 4, 0x37);
			KERNEL$UDELAY(10);
			dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		}
	}
	stop_dma_generic(p, dmastatus);
	return dmastatus;
}

static __u8 HPT372_STOP_DMA(APORT *p, __u8 mask)
{
	__u8 bwsr_stat, msc_stat;
	__u8 dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
	if (__unlikely(!(dmastatus & mask)) && __likely(mask)) return dmastatus;
	bwsr_stat = PCI$READ_CONFIG_BYTE(p->ctrl->pci_id, 0x6a);
	if (bwsr_stat & (1 << p->n)) {
		msc_stat = PCI$READ_CONFIG_BYTE(p->ctrl->pci_id, 0x50 + p->n * 4);
		PCI$WRITE_CONFIG_BYTE(p->ctrl->pci_id, 0x50 + p->n * 4, msc_stat | 0x30);
	}
	stop_dma_generic(p, dmastatus);
	return dmastatus;
}

__COLD_ATTR__ static int HPT37X_CLOCK_SLOT(unsigned freq, unsigned base)
{
	unsigned f = (base * freq) / 192;
	if (f < 40)
		return 0;
	if (f < 45)
		return 1;
	if (f < 55)
		return 2;
	return 3;
}

static int HPT37X_CALIBRATE_DPLL(pci_id_t pci_id)
{
	__u8 reg5b;
	__u32 reg5c;
	int i, j;

	for (i = 0; i < 0x5000; i++) {
		KERNEL$UDELAY(50);
		reg5b = PCI$READ_CONFIG_BYTE(pci_id, 0x5b);
		if (reg5b & 0x80) {
			for (j = 0; j < 0x1000; j++) {
				reg5b = PCI$READ_CONFIG_BYTE(pci_id, 0x5b);
				if (!(reg5b & 0x80))
					return 0;
			}
			reg5c = PCI$READ_CONFIG_DWORD(pci_id, 0x5c);
			reg5c &= ~0x100;
			PCI$WRITE_CONFIG_DWORD(pci_id, 0x5c, reg5c);
			return 1;
		}
	}
	return 0;
}

__COLD_ATTR__ int DETECT_HPT(ACTRL *a)
{
	__u8 rv;
	int r;
	int i;
	unsigned chip = 0;
	const struct hpt_chip *hpt_chip;
	unsigned port;
	char prefer_dpll;
	__u8 irqmask;
	__u32 freq;
	IO_RANGE range;
	unsigned clock_slot;

	if ((a->dev_id & 0xffff) != 0x1103) return 0;
	rv = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_REVISION_ID);
	if (a->dev_id == 0x00041103) {
		if (rv == 3) chip = HPT_370;
		else if (rv == 4) chip = HPT_370A;
		else if (rv == 5) chip = HPT_372;
	} else if (a->dev_id == 0x00051103) {
		if (rv < 2) chip = HPT_372A;
	} else if (a->dev_id == 0x00061103) {
		if (rv < 2) chip = HPT_302;
	} else if (a->dev_id == 0x00071103) {
		if (rv < 2) chip = HPT_371;
	} else if (a->dev_id == 0x00081103) {
		chip = HPT_374;
		if (a->pci_id & __SET_FIELD((pci_id_t)1, PCI_ID_T_FUNCTION))
			chip |= HPT_FN1;
	}
	if (!chip) return 0;

	hpt_chip = GET_HPT_CHIP(chip);

	port = PCI$READ_IO_RESOURCE(!(chip & HPT_374) ? a->pci_id : a->pci_id & ~__SET_FIELD((pci_id_t)1, PCI_ID_T_FUNCTION), 4);
	if (!port) return 0;

	range.start = port;
	range.len = 0x100;
	range.name = a->dev_name;
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&range))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET IO RANGE "IO_FORMAT" - "IO_FORMAT": %s", range.start, range.start + range.len - 1, strerror(-r));
		return 0;
	}

	a->chipset_flags = chip;

	prefer_dpll = !(chip & (HPT_370 | HPT_370A));

	PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_LATENCY_TIMER, 0x78);
	PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_MIN_GNT, 0x08);
	PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_MAX_LAT, 0x08);

	irqmask = PCI$READ_CONFIG_BYTE(a->pci_id, 0x5a);
	irqmask &= ~0x10;
	PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x5a, irqmask);

	PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x5b, 0x23);

	if (chip & HPT_372A)
		io_outb(port + 0x9c, 0x0e);
	
	freq = io_inl(port + 0x90);

	if ((freq >> 12) != 0xabcde) {
		__u32 total = 0;
		for (i = 0; i < 128; i++) {
			__u8 sr = PCI$READ_CONFIG_BYTE(a->pci_id, 0x78);
			total += sr;
			KERNEL$UDELAY(15);
		}
		freq = total / 128;
	}
	freq &= 0x1ff;

	clock_slot = HPT37X_CLOCK_SLOT(freq, hpt_chip->base);

	if (!hpt_chip->timing[clock_slot] || prefer_dpll) {
		unsigned f_low, f_high;
		int dpll, adjust;
		static const unsigned char MHz[4] = { 33, 40, 50, 66 };

		dpll = hpt_chip->udma >= 6 ? 3 : 2;
		f_low = (MHz[clock_slot] * 48) / MHz[dpll];
		f_high = f_low + 2;
		if (clock_slot > 1)
			f_high += 2;

		PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x5b, 0x21);
		for (adjust = 0; adjust < 8; adjust++) {
			PCI$WRITE_CONFIG_DWORD(a->pci_id, 0x5c, (f_high << 16) | f_low | 0x100);
			if (HPT37X_CALIBRATE_DPLL(a->pci_id))
				goto dpll_ok;
			if (adjust & 1)
				f_low -= adjust >> 1;
			else
				f_high += adjust >> 1;
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "DPLL DID NOT STABILIZE");
		goto no_dma;
		dpll_ok:
		if (dpll == 3) a->chipset_flags_2 = (unsigned long)&hpt37x_timing_66;
		else a->chipset_flags_2 = (unsigned long)&hpt37x_timing_50;
	} else {
		a->chipset_flags_2 = (unsigned long)hpt_chip->timing[clock_slot];
	}

	for (i = 0; i < a->n_ports; i++) {
		a->port[i].get_avail_xfer = HPT_GET_AVAIL_XFER;
		a->port[i].set_xfer = HPT_SET_XFER;
		a->port[i].xfer_mode[0][0] = 0xff;
		a->port[i].xfer_mode[0][1] = 0xff;
		a->port[i].xfer_mode[1][0] = 0xff;
		a->port[i].xfer_mode[1][1] = 0xff;
		a->port[i].aport_flags |= APORT_EXTRA_DMA;
		if (chip & (HPT_370 | HPT_370A)) {
			a->port[i].start_dma = HPT370_START_DMA;
			a->port[i].stop_dma = HPT370_STOP_DMA;
		} else {
			a->port[i].start_dma = start_dma_generic;
			a->port[i].stop_dma = HPT372_STOP_DMA;
		}
		PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x50 + i * 4, 0x37);
		KERNEL$UDELAY(100);
	}

	ret:
	KERNEL$UNREGISTER_IO_RANGE(&range);
	return 1;

	no_dma:
	for (i = 0; i < a->n_ports; i++) {
		a->port[i].dma_io = 0;
	}
	goto ret;
}

