#include <SYS/PARAM.H>

#include "ACTRL.H"

#define PCI_IDE_TIM_PRI		0x40
#define PCI_IDE_TIM_SEC		0x42
#define  PCI_IDE_TIM_TIME0		0x0001
#define  PCI_IDE_TIM_IE0		0x0002
#define  PCI_IDE_TIM_PPE0		0x0004
#define  PCI_IDE_TIM_DTE0		0x0008
#define  PCI_IDE_TIM_TIME1		0x0010
#define  PCI_IDE_TIM_IE1		0x0020
#define  PCI_IDE_TIM_PPE1		0x0040
#define  PCI_IDE_TIM_DTE1		0x0080
#define  PCI_IDE_TIM_RCT		0x0300
#define  PCI_IDE_TIM_ISP		0x3000
#define  PCI_IDE_TIM_SITRE		0x4000
#define  PCI_IDE_TIM_IDE		0x8000
#define PCI_SLV_IDE_TIM		0x44
#define  PCI_SLV_IDE_TIM_PRCT1		0x03
#define  PCI_SLV_IDE_TIM_PISP1		0x0c
#define  PCI_SLV_IDE_TIM_SRCT1		0x30
#define  PCI_SLV_IDE_TIM_SISP1		0xc0
#define PCI_SDMA_CNT		0x48
#define  PCI_SDMA_CNT_PSDE0		0x01
#define  PCI_SDMA_CNT_PSDE1		0x02
#define  PCI_SDMA_CNT_SSDE0		0x04
#define  PCI_SDMA_CNT_SSDE1		0x08
#define PCI_SDMA_TIM		0x4a
#define  PCI_SDMA_TIM_PCT0		0x0003
#define  PCI_SDMA_TIM_PCT1		0x0030
#define  PCI_SDMA_TIM_SCT0		0x0300
#define  PCI_SDMA_TIM_SCT1		0x3000
#define PCI_IDE_CONFIG		0x54
#define  PCI_IDE_CONFIG_PCB0		0x00000001
#define  PCI_IDE_CONFIG_PCB1		0x00000002
#define  PCI_IDE_CONFIG_SCB0		0x00000004
#define  PCI_IDE_CONFIG_SCB1		0x00000008
#define  PCI_IDE_CONFIG_PCABLE0		0x00000010
#define  PCI_IDE_CONFIG_PCABLE1		0x00000020
#define  PCI_IDE_CONFIG_SCABLE0		0x00000040
#define  PCI_IDE_CONFIG_SCABLE1		0x00000080
#define  PCI_IDE_CONFIG_FAST_PCB0	0x00001000
#define  PCI_IDE_CONFIG_FAST_PCB1	0x00002000
#define  PCI_IDE_CONFIG_FAST_SCB0	0x00004000
#define  PCI_IDE_CONFIG_FAST_SCB1	0x00008000
#define  PCI_IDE_CONFIG_PRIM_SIG_MODE	0x00030000
#define  PCI_IDE_CONFIG_SEC_SIG_MODE	0x000c0000

#define PIIX_MWDMA	0x00000001
#define PIIX_33		0x00000002
#define ICH_33		0x00000004
#define ICH_66		0x00000008
#define ICH_100		0x00000010
#define SNGL		0x00000020
#define VMWARE		0x00000040
#define HAS_UDMA	(PIIX_33 | ICH_33 | ICH_66 | ICH_100)
#define IS_ICH		(ICH_33 | ICH_66 | ICH_100)

/* Laptop table, copied from Linux */

static const struct {
	__u16 device;
	__u16 subvendor;
	__u16 subdevice;
} laptops[] = {
	{ 0x27DF, 0x0005, 0x0280 },	/* ICH7 on Acer 5602WLMi */
	{ 0x27DF, 0x1025, 0x0102 },	/* ICH7 on Acer 5602aWLMi */
	{ 0x27DF, 0x1025, 0x0110 },	/* ICH7 on Acer 3682WLMi */
	{ 0x27DF, 0x1043, 0x1267 },	/* ICH7 on Asus W5F */
	{ 0x27DF, 0x103C, 0x30A1 },	/* ICH7 on HP Compaq nc2400 */
	{ 0x27DF, 0x1071, 0xD221 },	/* ICH7 on Hercules EC-900 */
	{ 0x24CA, 0x1025, 0x0061 },	/* ICH4 on ACER Aspire 2023WLMi */
	{ 0x24CA, 0x1025, 0x003d },	/* ICH4 on ACER TM290 */
	{ 0x266F, 0x1025, 0x0066 },	/* ICH6 on ACER Aspire 1694WLMi */
	{ 0x2653, 0x1043, 0x82D8 },	/* ICH6M on Asus Eee 701 */
};

__COLD_ATTR__ static void INTEL_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	ACTRL *a = p->ctrl;
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_SDMA_2 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2;
	*avail_unsupported = 0;
	if (a->chipset_flags & HAS_UDMA) {
		*avail |= IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2;
	}
	if (a->chipset_flags & (ICH_66 | ICH_100)) {
		*avail |= IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4;
	}
	if (a->chipset_flags & ICH_100) {
		*avail |= IDE_XFER_UDMA_5;
	}
	if (*avail & (IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4 | IDE_XFER_UDMA_5 | IDE_XFER_UDMA_6)) {
		int i;
		__u8 cfg;
		__u16 device = a->dev_id >> 16;
		__u16 subvendor = PCI$READ_CONFIG_WORD(a->pci_id, PCI_SUBSYSTEM_VENDOR_ID);
		__u16 subdevice = PCI$READ_CONFIG_WORD(a->pci_id, PCI_SUBSYSTEM_ID);
		for (i = 0; i < sizeof(laptops) / sizeof(*laptops); i++) {
			if (laptops[i].device == device &&
			    laptops[i].subvendor == subvendor &&
			    laptops[i].subdevice == subdevice)
				goto skip_cable_test;
		}
		cfg = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_IDE_CONFIG);
		if (!(cfg & ((PCI_IDE_CONFIG_PCABLE0 | PCI_IDE_CONFIG_PCABLE1) << (p->n * 2)))) {
			ATA$CABLE40_RESTRICT(avail, avail_unsupported);
		}
		skip_cable_test:;
	}
}

__COLD_ATTR__ static void SET_PIO_DMA(APORT *p, int drive, int atapi, unsigned pio, int dma)
{
	static const __u8 pio_isp[5] = { 0, 0, 1, 2, 2 };
	static const __u8 pio_rct[5] = { 0, 0, 0, 1, 3 };

	ACTRL *a = p->ctrl;
	__u16 reg_master;
	__u8 reg_slave;
	__u8 control = 0;
	if (pio >= 2 || dma)
		control |= PCI_IDE_TIM_TIME0;
	if (pio >= 3 || dma)	/* IORDY */
		control |= PCI_IDE_TIM_IE0;
	if (!atapi && !dma)
		control |= PCI_IDE_TIM_PPE0;
	if (dma == 2)
		control |= PCI_IDE_TIM_DTE0;
	reg_master = PCI$READ_CONFIG_WORD(a->pci_id, PCI_IDE_TIM_PRI + 2 * p->n);
	if (!drive) {
		reg_master &= ~(PCI_IDE_TIM_TIME0 | PCI_IDE_TIM_IE0 | PCI_IDE_TIM_PPE0 | PCI_IDE_TIM_DTE0 | PCI_IDE_TIM_RCT | PCI_IDE_TIM_ISP);
		reg_master |= control;
		reg_master |= pio_isp[pio] << __BSF_CONST(PCI_IDE_TIM_ISP);
		reg_master |= pio_rct[pio] << __BSF_CONST(PCI_IDE_TIM_RCT);
	} else {
		reg_master &= ~(PCI_IDE_TIM_TIME1 | PCI_IDE_TIM_IE1 | PCI_IDE_TIM_PPE1 | PCI_IDE_TIM_DTE1);
		reg_master |= PCI_IDE_TIM_SITRE;
		reg_master |= control << __BSF_CONST(PCI_IDE_TIM_TIME1);
		reg_slave = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_SLV_IDE_TIM);
		reg_slave &= ~((PCI_SLV_IDE_TIM_PRCT1 | PCI_SLV_IDE_TIM_PISP1) << (p->n * 4));
		reg_slave |= pio_isp[pio] << (__BSF_CONST(PCI_SLV_IDE_TIM_PISP1) + p->n * 4);
		reg_slave |= pio_rct[pio] << (__BSF_CONST(PCI_SLV_IDE_TIM_PRCT1) + p->n * 4);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_SLV_IDE_TIM, reg_slave);
	}
	PCI$WRITE_CONFIG_WORD(a->pci_id, PCI_IDE_TIM_PRI + 2 * p->n, reg_master);
	if (dma && a->chipset_flags & HAS_UDMA) {
		__u8 reg_udma = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_SDMA_CNT);
		reg_udma &= ~(PCI_SDMA_CNT_PSDE0 << (p->n * 2 + drive));
		PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_SDMA_CNT, reg_udma);
	}
}

__COLD_ATTR__ static void SET_UDMA(APORT *p, int drive, unsigned udma)
{
	ACTRL *a = p->ctrl;
	int devid = p->n * 2 + drive;
	__u16 reg_timing, reg_config;
	__u8 reg_udma;
	__u32 u_clock = udma >= 5 ? PCI_IDE_CONFIG_FAST_PCB0 : udma > 2 ? PCI_IDE_CONFIG_PCB0 : 0;
	unsigned u_speed = MIN(2 - (udma & 1), udma);
	reg_timing = PCI$READ_CONFIG_WORD(a->pci_id, PCI_SDMA_TIM);
	reg_timing &= ~(PCI_SDMA_TIM_PCT0 << (devid * 4));
	reg_timing |= u_speed << (devid * 4);
	PCI$WRITE_CONFIG_WORD(a->pci_id, PCI_SDMA_TIM, reg_timing);
	if (a->chipset_flags & IS_ICH) {
		reg_config = PCI$READ_CONFIG_WORD(a->pci_id, PCI_IDE_CONFIG);
		reg_config &= ~((PCI_IDE_CONFIG_PCB0 | PCI_IDE_CONFIG_FAST_PCB0) << devid);
		reg_config |= u_clock << devid;
		PCI$WRITE_CONFIG_WORD(a->pci_id, PCI_IDE_CONFIG, reg_config);
	}
	reg_udma = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_SDMA_CNT);
	reg_udma |= PCI_SDMA_CNT_PSDE0 << devid;
	PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_SDMA_CNT, reg_udma);
}

__COLD_ATTR__ static int INTEL_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	static const __u8 dma2pio[N_SDMA_MODES + N_WDMA_MODES] = { 0, 0, 2, 0, 3, 4 };

	int atapi = ATA$IS_ATAPI(id);
	unsigned pio;
	unsigned dma;
	if (mode & IDE_XFER_MASK_PIO) {
		p->xfer_mode[0][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
	} else {
		p->xfer_mode[1][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_DMA);
	}

	pio = p->xfer_mode[0][drive];
	dma = p->xfer_mode[1][drive];
	if (dma >= N_SDMA_MODES + N_WDMA_MODES && pio < N_PIO_MODES) {
		SET_PIO_DMA(p, drive, atapi, pio, 0);
	}
	if (dma < N_SDMA_MODES + N_WDMA_MODES) {
		unsigned vpio = dma2pio[dma];
		SET_PIO_DMA(p, drive, atapi, vpio, 1 + (vpio > pio));
	} else if ((dma -= N_SDMA_MODES + N_WDMA_MODES) < N_UDMA_MODES) {
		SET_UDMA(p, drive, dma);
	}
	return 0;
}

static __u8 VMWARE_STOP_DMA(APORT *p, __u8 mask)
{
	__u8 dmastatus = io_inb(p->dma_io + DMAPORT_STATUS) & ~DMASTATUS_ERROR;
	if (__unlikely(!(dmastatus & mask)) && __likely(mask)) return dmastatus;
	stop_dma_generic(p, dmastatus);
	return dmastatus;
}

static const struct {
	__u32 chip;
	unsigned flags;
	char *name;
} chipsets[] = {
	0x70108086,	PIIX_MWDMA,	"INTEL PIIX3",
	0x71118086,	PIIX_33,	"INTEL PIIX4",
	0x71998086,	PIIX_33,	"INTEL PIIX4",
	0x76018086,	PIIX_33,	"INTEL PIIX4",
	0x84CA8086,	PIIX_33,	"INTEL PIIX",
	0x24118086,	ICH_66,		"INTEL ICH",
	0x24218086,	ICH_33,		"INTEL ICH0",
	0x244A8086,	ICH_100,	"INTEL ICH2M",
	0x244B8086,	ICH_100,	"INTEL ICH2",
	0x248A8086,	ICH_100,	"INTEL ICH3M",
	0x248B8086,	ICH_100,	"INTEL ICH3",
	0x24CA8086,	ICH_100,	"INTEL ICH4",
	0x24CB8086,	ICH_100,	"INTEL ICH4",
	0x24DB8086,	ICH_100,	"INTEL ICH5",
	0x245B8086,	ICH_100,	"INTEL C-ICH",
	0x25A28086,	ICH_100,	"INTEL ESB",
	0x266F8086,	ICH_100 | SNGL,	"INTEL ICH6",
	0x27DF8086,	ICH_100 | SNGL,	"INTEL ICH7",
	0x269E8086,	ICH_100 | SNGL,	"INTEL ICH7",
	0x28508086,	ICH_100 | SNGL,	"INTEL ICH8M",
};

__COLD_ATTR__ int DETECT_INTEL(ACTRL *a)
{
	unsigned i;

	for (i = 0; i < sizeof(chipsets) / sizeof(*chipsets); i++) {
		if (a->dev_id == chipsets[i].chip) {
			a->chipset_flags = chipsets[i].flags;
			a->chipset_name = chipsets[i].name;
			goto found;
		}
	}
	return 0;

	found:

	if (a->dev_id == 0x71118086) {
		__u32 subvendor = PCI$READ_CONFIG_DWORD(a->pci_id, PCI_SUBSYSTEM_VENDOR_ID);
		if (subvendor == 0x197615AD) {
			a->chipset_name = "VMWARE IDE";
			a->chipset_flags |= VMWARE;
		}
	}

	for (i = 0; i < a->n_ports; i++) {
		if (i && a->chipset_flags & SNGL) {
			a->port[i].aport_flags |= APORT_DISABLED;
			continue;
		}
		a->port[i].get_avail_xfer = INTEL_GET_AVAIL_XFER;
		a->port[i].set_xfer = INTEL_SET_XFER;
		a->port[i].xfer_mode[0][0] = 0xff;
		a->port[i].xfer_mode[0][1] = 0xff;
		a->port[i].xfer_mode[1][0] = 0xff;
		a->port[i].xfer_mode[1][1] = 0xff;
	
		if (a->chipset_flags & VMWARE) {
			a->port[i].aport_flags |= APORT_EXTRA_DMA;
			a->port[i].start_dma = start_dma_generic;
			a->port[i].stop_dma = VMWARE_STOP_DMA;
		}
	}

	return 1;
}


