#include <ARCH/BSF.H>
#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ATA.H>

#include "ACTRLREG.H"
#include "ACTRL.H"

static int IT8212_FIXUP_IDENT(APORT *p, int drive, __u16 ident[256]);
static void IT_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported);
static int IT_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 ident[256]);
static void IT_MANGLE_REQUEST(APORT *p, ATARQ *rq);
static void IT_SETUP_XFER(APORT *p, ATARQ *rq);

/* chipset_flags */
#define IT_FLAGS_REG_50			0x000000ff
#define IT_FLAGS_FIRMWARE_MODE		0x00008000
#define IT_FLAGS_PRIMARY_TIMING		0x00ff0000
#define IT_FLAGS_SECONDARY_TIMING	0xff000000

#define IT_PCI_MODE			0x50
#define  IT_PCI_MODE_FIRMWARE			0x01
#define  IT_PCI_MODE_PRIMARY_CLK_50		0x02
#define  IT_PCI_MODE_SECONDARY_CLK_50		0x04
#define  IT_PCI_MODE_PRIMARY_MASTER_MWDMA	0x08
#define  IT_PCI_MODE_PRIMARY_SLAVE_MWDMA	0x10
#define  IT_PCI_MODE_SECONDARY_MASTER_MWDMA	0x20
#define  IT_PCI_MODE_SECONDARY_SLAVE_MWDMA	0x40
#define  IT_PCI_MODE_RESET			0x80
#define IT_PRIMARY_PIO_MWDMA_TIMING	0x54
#define IT_PRIMARY_MASTER_UDMA_TIMING	0x56
#define IT_PRIMARY_SLAVE_UDMA_TIMING	0x57
#define IT_SECONDARY_PIO_MWDMA_TIMING	0x58
#define IT_SECONDARY_MASTER_UDMA_TIMING	0x5a
#define IT_SECONDARY_SLAVE_UDMA_TIMING	0x5b

static const __u8 udma_timing[N_UDMA_MODES][2] = {
	{ 0x44, 0x33 },
	{ 0x42, 0x31 },
	{ 0x31, 0x21 },
	{ 0x21, 0x21 },
	{ 0x11, 0x11 },
	{ 0xa2, 0x91 },
	{ 0x91, 0x91 },
};

static const __u8 pio_wdma_timing[N_PIO_MODES + N_SDMA_MODES + N_WDMA_MODES][2] = {
	{ 0xAA, 0x88 },
	{ 0xA3, 0x82 },
	{ 0xA1, 0x81 },
	{ 0x33, 0x32 },
	{ 0x32, 0x32 },
	{ 0, 0 },
	{ 0, 0 },
	{ 0, 0 },

	{ 0, 0 },
	{ 0, 0 },
	{ 0, 0 },

	{ 0x88, 0x66 },
	{ 0x32, 0x22 },
	{ 0x31, 0x21 },
	{ 0, 0 },
	{ 0, 0 },
};

static const __s8 preferred_clock[N_PIO_MODES + N_DMA_MODES] = {
	0, 0, 0, 0, -1, -1, -1, -1,
	-1, -1, -1,
	-1, 0, -1, -1, -1,
	-1, 1, -1, 0, 0, 1, 0
};

__COLD_ATTR__ int DETECT_IT821(ACTRL *a)
{
	int i;
	__u8 mode;
	if (a->dev_id != 0x82111283 && 
	    a->dev_id != 0x82121283) return 0;
	mode = PCI$READ_CONFIG_BYTE(a->pci_id, IT_PCI_MODE);
	if (mode & IT_PCI_MODE_FIRMWARE) {
		a->chipset_name = "ITE IT8212 (FIRMWARE MODE)";
		a->chipset_flags = IT_FLAGS_FIRMWARE_MODE;
		for (i = 0; i < a->n_ports; i++) {
			a->port[i].aport_flags |= APORT_IMPLICIT_DMA | APORT_MANGLE_REQUEST;
			a->port[i].mangle = IT_MANGLE_REQUEST;
			a->port[i].recommended_length = 255 << BIO_SECTOR_SIZE_BITS;
		}
	} else {
		/* chipset_name already set from PCI table */
		a->chipset_flags = 0;
		a->chipset_flags_2 = 0;
		for (i = 0; i < a->n_ports; i++) {
			a->port[i].get_avail_xfer = IT_GET_AVAIL_XFER;
			a->port[i].set_xfer = IT_SET_XFER;
			a->port[i].aport_flags |= APORT_SETUP_XFER;
			a->port[i].setup_xfer = IT_SETUP_XFER;
			a->port[i].xfer_mode[0][0] = 0xff;
			a->port[i].xfer_mode[0][1] = 0xff;
			a->port[i].xfer_mode[1][0] = 0xff;
			a->port[i].xfer_mode[1][1] = 0xff;
		}
		mode &= ~(IT_PCI_MODE_PRIMARY_MASTER_MWDMA | IT_PCI_MODE_PRIMARY_SLAVE_MWDMA | IT_PCI_MODE_SECONDARY_MASTER_MWDMA | IT_PCI_MODE_SECONDARY_SLAVE_MWDMA);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PCI_MODE, mode);
		a->chipset_flags |= mode << __BSF_CONST(IT_FLAGS_REG_50);
		a->chipset_flags |= PCI$READ_CONFIG_BYTE(a->pci_id, IT_PRIMARY_PIO_MWDMA_TIMING) << __BSF_CONST(IT_FLAGS_PRIMARY_TIMING);
		a->chipset_flags |= PCI$READ_CONFIG_BYTE(a->pci_id, IT_SECONDARY_PIO_MWDMA_TIMING) << __BSF_CONST(IT_FLAGS_SECONDARY_TIMING);
	}
	for (i = 0; i < a->n_ports; i++) {
		a->port[i].fixup_ident = IT8212_FIXUP_IDENT;
	}
	return 1;
}

__COLD_ATTR__ static int IT8212_FIXUP_IDENT(APORT *p, int drive, __u16 ident[256])
{
	ACTRL *a = p->ctrl;
	char model[41];
	/*
	int atapi = ATA$IS_ATAPI(ident);
	if (atapi && (a->dev_rev == 0x10 || a->chipset_flags & IT_FLAGS_FIRMWARE_MODE)) {
		ident[49] &= ~0x0100;
		ident[62] = 0;
	}
	*/
	if (!(a->chipset_flags & IT_FLAGS_FIRMWARE_MODE)) return 0;
	ATA$GET_MODEL(ident, &model, NULL, NULL);
	if (strstr(model, "Integrated Technology Express")) {
		ident[10] = 0x0000; /* firmware returns invalid serial number */
		ident[49] |= 0x0300;
		ident[83] |= 0x0400;
		ident[86] |= 0x0400;
	} else {
		ident[53] &= 0x0001;
		ident[75] = 0x0000;
		ident[82] = 0x0000;
		ident[83] &= 0xc400;
		ident[84] &= 0xc000;
		ident[85] = 0x0000;
		ident[86] &= 0xc400;
		ident[87] &= 0xc000;
		ident[127] = 0x0000;
		ident[128] = 0x0000;
		ident[129] = 0x0000;
		ident[160] = 0x0000;
	}
	return 0;
}

__COLD_ATTR__ static void IT_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	int atapi = ATA$IS_ATAPI(ident);
	*avail =
		IDE_XFER_PIO_0 |
		IDE_XFER_PIO_1 |
		IDE_XFER_PIO_2 |
		IDE_XFER_PIO_3 |
		IDE_XFER_PIO_4 |
		IDE_XFER_PIO_5 |
		IDE_XFER_WDMA_0 |
		IDE_XFER_WDMA_1 |
		IDE_XFER_WDMA_2 |
		IDE_XFER_UDMA_0 |
		IDE_XFER_UDMA_1 |
		IDE_XFER_UDMA_2 |
		IDE_XFER_UDMA_3 |
		IDE_XFER_UDMA_4 |
		IDE_XFER_UDMA_5 |
		IDE_XFER_UDMA_6;
	*avail_unsupported = 0;
	if (atapi && p->ctrl->dev_rev <= 0x10)
		*avail &= ~IDE_XFER_MASK_WDMA;
}

__COLD_ATTR__ static int IT_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 ident[256])
{
	p->xfer_mode[!(mode & IDE_XFER_MASK_PIO)][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK);
	return 0;
}

static void IT_MANGLE_REQUEST(APORT *p, ATARQ *rq)
{
	if (__likely(rq->atarq_flags & ATARQ_SET_SIZE))
		if (__unlikely(rq->len > 255 << BIO_SECTOR_SIZE_BITS)) rq->len = 255 << BIO_SECTOR_SIZE_BITS;
}

__COLD_ATTR__ static void IT_SETUP_XFER(APORT *p, ATARQ *rq)
{
	ACTRL *a = p->ctrl;
	int clock;
	__u8 timing, old_timing;
	unsigned xfer;
	int timing_shift = (p->n << 1) | rq->device;
	if (__unlikely((rq->atarq_flags & ATARQ_PROTOCOL) == ATA_PROTOCOL_NODATA)) return;
	xfer = p->xfer_mode[(rq->atarq_flags >> __BSF_CONST(ATA_PROTOCOL_DMA)) & 1][rq->device];
	if (__unlikely(xfer > __BSF_CONST(IDE_XFER_UDMA_6))) goto straight;
#define clock_shift	(__BSF_CONST(IT_PCI_MODE_PRIMARY_CLK_50) + __BSF_CONST(IT_FLAGS_REG_50) + p->n)
		/*
		a->chipset_flags ^= 1 << clock_shift;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PCI_MODE, a->chipset_flags >> __BSF_CONST(IT_FLAGS_REG_50));
		*/
	clock = preferred_clock[xfer];
	if (__unlikely(clock < 0)) {
		clock = (a->chipset_flags >> clock_shift) & 1;
	}
	timing = a->chipset_flags >> __BSF_CONST(IT_FLAGS_REG_50);
	old_timing = timing;
	timing &= ~(IT_PCI_MODE_PRIMARY_CLK_50 << p->n);
	timing |= clock << (__BSF_CONST(IT_PCI_MODE_PRIMARY_CLK_50) + p->n);
	if (__likely(xfer >= __BSF_CONST(IDE_XFER_SDMA_0))) {
		if (__likely(xfer >= __BSF_CONST(IDE_XFER_UDMA_0))) {
			timing &= ~(IT_PCI_MODE_PRIMARY_MASTER_MWDMA << timing_shift);
		} else {
			timing |= IT_PCI_MODE_PRIMARY_MASTER_MWDMA << timing_shift;
		}
	}
	if (__unlikely(timing != old_timing)) {
		a->chipset_flags = (a->chipset_flags & ~IT_FLAGS_REG_50) | (timing << __BSF_CONST(IT_FLAGS_REG_50));
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PCI_MODE, timing);
	}
#undef clock_shift
	if (__likely(xfer >= __BSF_CONST(IDE_XFER_UDMA_0))) {
		timing = udma_timing[xfer - __BSF_CONST(IDE_XFER_UDMA_0)][clock];
		timing_shift *= 8;
		if (__unlikely(timing != ((a->chipset_flags_2 >> timing_shift) & 0xff))) {
			a->chipset_flags_2 = (a->chipset_flags_2 & ~(0xff << timing_shift)) | (timing << timing_shift);
			PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PRIMARY_MASTER_UDMA_TIMING + (4 * p->n) + rq->device, timing);
			if (__unlikely(a->dev_rev <= 0x10)) {
				a->chipset_flags_2 = (a->chipset_flags_2 & ~(0xff << (timing_shift ^ 8))) | (timing << (timing_shift ^ 8));
				PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PRIMARY_MASTER_UDMA_TIMING + (4 * p->n) + (rq->device ^ 1), timing);
			}
		}
	} else {
		timing = pio_wdma_timing[xfer][clock];
		timing_shift >>= 1;
		timing_shift *= 8;
		if (__unlikely(((a->chipset_flags >> (timing_shift + __BSF_CONST(IT_FLAGS_PRIMARY_TIMING))) & 0xff) != timing)) {
			a->chipset_flags = (a->chipset_flags & ~(IT_FLAGS_PRIMARY_TIMING << timing_shift)) | (timing << (timing_shift + __BSF_CONST(IT_FLAGS_PRIMARY_TIMING)));
			PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PRIMARY_PIO_MWDMA_TIMING + (timing_shift >> 1), timing);
		}
	}

	straight:;
}

