#include "ACTRL.H"

#define PDC_PORT_11		0x000000ff
#define  PDC_PORT_11_CLK_0		0x00000002
#define  PDC_PORT_11_CLK_1		0x00000008
#define PDC_VERSION_0		0x00000100
#define PDC_VERSION_1		0x00000200
#define PDC_VERSION_2		0x00000400

#define PDC_PORT_SET_CLOCK	0x00000001
#define PDC_PORT_SET_LENGTH	0x00000002

/* Maybe it could be increased to 512 */
#define PDC_MAX_SECTORS		256

static void PDC_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported);
static int PDC_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256]);
static void PDC_MANGLE_REQUEST(APORT *p, ATARQ *rq);
static void PDC_START_DMA(APORT *p, ATARQ *rq);
static __u8 PDC_STOP_DMA(APORT *p, __u8 mask);

__COLD_ATTR__ int DETECT_PDC(ACTRL *a)
{
	int i, dma_set;
	switch (a->dev_id) {
		case 0x4d33105a:
			a->chipset_flags = PDC_VERSION_0;
			break;
		case 0x4d38105a:
		case 0x0d38105a:
			a->chipset_flags = PDC_VERSION_1;
			break;
		case 0x0d30105a:
		case 0x4d30105a:
			a->chipset_flags = PDC_VERSION_2;
			break;
		default:
			return 0;
	}
	dma_set = 0;
	for (i = 0; i < a->n_ports; i++) {
		a->port[i].get_avail_xfer = PDC_GET_AVAIL_XFER;
		a->port[i].set_xfer = PDC_SET_XFER;
		a->port[i].port_chipset_flags = 0;
		if (a->port[i].dma_io && !(a->chipset_flags & PDC_VERSION_0)) {
			io_t dma_base;
			a->port[i].recommended_length = PDC_MAX_SECTORS << BIO_SECTOR_SIZE_BITS;
			a->port[i].atapi_length = PDC_MAX_SECTORS << BIO_SECTOR_SIZE_BITS;
			a->port[i].aport_flags |= APORT_MANGLE_REQUEST | APORT_EXTRA_DMA;
			a->port[i].mangle = PDC_MANGLE_REQUEST;
			a->port[i].start_dma = PDC_START_DMA;
			a->port[i].stop_dma = PDC_STOP_DMA;
			dma_base = a->port[i].dma_io & ~15;
			io_outl(dma_base + 0x20 + i * 4, 0);
			if (!dma_set) {
				a->chipset_flags |= io_inb(dma_base + 0x11) & ~(PDC_PORT_11_CLK_0 | PDC_PORT_11_CLK_1);;
				io_outb(dma_base + 0x11, a->chipset_flags);
				io_outb(dma_base + 0x1f, io_inb(dma_base + 0x1f) | 0x01);
				dma_set = 1;
			}
		}
	}
	return 1;
}

__COLD_ATTR__ static void PDC_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	ACTRL *a = p->ctrl;
	__u16 cbl;
	*avail =
		IDE_XFER_PIO_0 |
		IDE_XFER_PIO_1 |
		IDE_XFER_PIO_2 |
		IDE_XFER_PIO_3 |
		IDE_XFER_PIO_4 |
		IDE_XFER_WDMA_0 |
		IDE_XFER_WDMA_1 |
		IDE_XFER_WDMA_2 |
		IDE_XFER_UDMA_0 |
		IDE_XFER_UDMA_1 |
		IDE_XFER_UDMA_2;
	*avail_unsupported = 0;
	if (a->chipset_flags & (PDC_VERSION_1 | PDC_VERSION_2)) {
		*avail |= IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4;
	}
	if (a->chipset_flags & PDC_VERSION_2) {
		*avail |= IDE_XFER_UDMA_5;
	}
	cbl = PCI$READ_CONFIG_WORD(a->pci_id, 0x50);
	if (cbl & (0x400 << p->n))
		ATA$CABLE40_RESTRICT(avail, avail_unsupported);
}

__COLD_ATTR__ static int PDC_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	ACTRL *a = p->ctrl;
	__u8 port0, port1, port2;
	unsigned port = 0x60 + p->n * 8 + drive * 4;
	if (mode & IDE_XFER_MASK_PIO) {
		static const __u8 pio0[5] = { 0x09, 0x05, 0x03, 0x02, 0x01 };
		static const __u8 pio1[5] = { 0x13, 0x0c, 0x08, 0x06, 0x04 };
		unsigned pio = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
		if (pio > 4)
			return 0;
		port0 = PCI$READ_CONFIG_BYTE(a->pci_id, port);
		port1 = PCI$READ_CONFIG_BYTE(a->pci_id, port + 1);
		port0 &= 0xc0;
		port1 &= 0xe0;
		port0 |= pio0[pio];
		port1 |= pio1[pio];
		if (pio >= 3)	/* IORDY */
			port0 |= 0x20;
		if (!ATA$IS_ATAPI(id))
			port0 |= 0x10;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, port, port0);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, port + 1, port1);
	} else {
		static const __u8 dma1[N_WDMA_MODES + N_UDMA_MODES - 1] = { 0xe0, 0x60, 0x60, 0x60, 0x60, 0x60, 0x40, 0x20, 0x40, 0x20, 0x20 };
		static const __u8 dma2[N_WDMA_MODES + N_UDMA_MODES - 1] = { 0x0f, 0x04, 0x03, 0x03, 0x03, 0x03, 0x02, 0x01, 0x02, 0x01, 0x01 };
		unsigned dma = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_WDMA | IDE_XFER_MASK_UDMA | IDE_XFER_DMA_NONE);
		if (dma >= sizeof dma1)
			return 0;
		port1 = PCI$READ_CONFIG_BYTE(a->pci_id, port + 1);
		port2 = PCI$READ_CONFIG_BYTE(a->pci_id, port + 2);
		port1 &= 0x1f;
		port2 &= 0xf0;
		port1 |= dma1[dma];
		port2 |= dma2[dma];
		PCI$WRITE_CONFIG_BYTE(a->pci_id, port + 1, port1);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, port + 2, port2);
		p->port_chipset_flags &= ~PDC_PORT_SET_CLOCK;
		if (dma >= N_WDMA_MODES + 3)
			p->port_chipset_flags |= PDC_PORT_SET_CLOCK;
	}
	return 0;
}

static void PDC_MANGLE_REQUEST(APORT *p, ATARQ *rq)
{
	if (__likely(rq->atarq_flags & ATARQ_SET_SIZE))
		if (__unlikely(rq->len > PDC_MAX_SECTORS << BIO_SECTOR_SIZE_BITS)) rq->len = PDC_MAX_SECTORS << BIO_SECTOR_SIZE_BITS;
}

static void PDC_START_DMA(APORT *p, ATARQ *rq)
{
	if (__unlikely((rq->fis.command & ~(ATA_CMD_READ_DMA ^ ATA_CMD_WRITE_DMA)) != (ATA_CMD_READ_DMA & ATA_CMD_WRITE_DMA))) {
		unsigned len = (rq->len >> 1) + 0x05000000 + (rq->atarq_flags & ATARQ_TO_DEVICE) * ((0x06000000 - 0x05000000) / ATARQ_TO_DEVICE);
		p->port_chipset_flags |= PDC_PORT_SET_LENGTH;
		io_outl((p->dma_io & ~15) + 0x20 + p->n * 4, len);
	}
	if (__likely((p->port_chipset_flags & PDC_PORT_SET_CLOCK) != 0)) {
		p->ctrl->chipset_flags |= PDC_PORT_11_CLK_0 << (p->n * __BSF_CONST(PDC_PORT_11_CLK_1 / PDC_PORT_11_CLK_0));
#ifdef __i386__
		/* lower pressure on register allocator */
		__barrier();
#endif
		io_outb((p->dma_io & ~15) + 0x11, p->ctrl->chipset_flags);
	}
	start_dma_generic(p, rq);
}

static __u8 PDC_STOP_DMA(APORT *p, __u8 mask)
{
	__u8 dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
	if (__unlikely(!(dmastatus & mask)) && __likely(mask)) return dmastatus;
	if (__unlikely((p->port_chipset_flags & PDC_PORT_SET_LENGTH) != 0)) {
		p->port_chipset_flags &= ~PDC_PORT_SET_LENGTH;
		io_outl((p->dma_io & ~15) + 0x20 + p->n * 4, 0);
	}
	if (__likely((p->port_chipset_flags & PDC_PORT_SET_CLOCK) != 0)) {
		p->ctrl->chipset_flags &= ~(PDC_PORT_11_CLK_0 << (p->n * __BSF_CONST(PDC_PORT_11_CLK_1 / PDC_PORT_11_CLK_0)));
		io_outb((p->dma_io & ~15) + 0x11, p->ctrl->chipset_flags);
	}
	stop_dma_generic(p, dmastatus);
	return dmastatus;
}
