#include <SYS/TIME.H>
#include <SPAD/SYSLOG.H>

#include "ACTRL.H"

#define PDC_IDX			5
#define PDC_REGSPACE		0x2000

#define PDC_SYS_CTL		0x1100
#define PDC_ATA_CTL		0x1104
#define PDC_GLOBAL_CTL		0x1108
#define PDC_CTCR0		0x110c
#define PDC_CTCR1		0x1110
#define PDC_BYTE_COUNT		0x1120
#define PDC_PPL_CTL		0x1202
#define PDC_BYTE_COUNT2	0x1220

#define PDC_READB(a, idx)	mmio_inb((__u8 *)(a)->chipset_flags + (idx))
#define PDC_WRITEB(a, idx, v)	mmio_outb((__u8 *)(a)->chipset_flags + (idx), v)
#define PDC_READW(a, idx)	mmio_inw((__u8 *)(a)->chipset_flags + (idx))
#define PDC_WRITEW(a, idx, v)	mmio_outw((__u8 *)(a)->chipset_flags + (idx), v)
#define PDC_READL(a, idx)	mmio_inl((__u8 *)(a)->chipset_flags + (idx))
#define PDC_WRITEL(a, idx, v)	mmio_outl((__u8 *)(a)->chipset_flags + (idx), v)

static int PDC_DETECT_PLL(ACTRL *a);
static unsigned PDC_READ_COUNTER(ACTRL *a);
static void PDC_ADJUST_PLL(ACTRL *a, unsigned pll);
static void PDC_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported);
static int PDC_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256]);
static int PDC_CHECK_ATAPI_DMA(APORT *p, SCSIRQ *rq);
static void PDC_DTOR(ACTRL *a);

#define PDC_133(a)	((a)->dev_id != 0x4d68105a && (a)->dev_id != 0x6268105a)

int DETECT_PDC2(ACTRL *a)
{
	void *res;
	int i;
	unsigned pll;

	switch (a->dev_id) {
		case 0x4d68105a:
		case 0x6268105a:
			break;
		case 0x4d69105a:
		case 0x6269105a:
		case 0x1275105a:
		case 0x5275105a:
		case 0x7275105a:
			break;
		default:
			return 0;
	}
	res = PCI$MAP_MEM_RESOURCE(a->pci_id, PDC_IDX, PDC_REGSPACE);
	if (!res || __IS_ERR(res))
		return 0;
	a->chipset_flags = (unsigned long)res;
	a->chipset_dtor = PDC_DTOR;

	pll = PDC_DETECT_PLL(a);
	PDC_ADJUST_PLL(a, pll);

	for (i = 0; i < a->n_ports; i++) {
		if (!(PDC_READB(a, PDC_ATA_CTL + i * 0x100) & 0x02)) {
			a->port[i].aport_flags |= APORT_DISABLED;
			continue;
		}
		a->port[i].get_avail_xfer = PDC_GET_AVAIL_XFER;
		a->port[i].set_xfer = PDC_SET_XFER;
		a->port[i].check_atapi_dma = PDC_CHECK_ATAPI_DMA;
	}
	return 1;
}

static int PDC_DETECT_PLL(ACTRL *a)
{
	__u32 sys_ctl;
	int spl = KERNEL$SPL;
	unsigned count1, count2, count_diff, usec_diff;
	struct timeval time1, time2;
	unsigned pll;

	sys_ctl = PDC_READL(a, PDC_SYS_CTL);
	PDC_WRITEL(a, PDC_SYS_CTL, sys_ctl | 0x00004000);
	PDC_READL(a, PDC_SYS_CTL);

	RAISE_SPL(SPL_ALMOST_TOP);
	count1 = PDC_READ_COUNTER(a);
	gettimeofday(&time1, NULL);
	LOWER_SPLX(spl);

	KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);

	RAISE_SPL(SPL_ALMOST_TOP);
	count2 = PDC_READ_COUNTER(a);
	gettimeofday(&time2, NULL);
	LOWER_SPLX(spl);

	PDC_WRITEL(a, PDC_SYS_CTL, sys_ctl & ~0x00004000);

	count_diff = (count1 - count2) & 0x3fffffff;
	usec_diff = (time2.tv_sec * 1000000 + time2.tv_usec) - (time1.tv_sec * 1000000 + time1.tv_usec);
	pll =  count_diff / 100 * (100000000 / usec_diff);
	/*__debug_printf("count diff: %x, usec diff %x\n", count_diff, usec_diff);
	__debug_printf("pll: %d\n", pll);*/
	return pll;
}

static unsigned PDC_READ_COUNTER(ACTRL *a)
{
	int retries = 2;
	unsigned lo, hi, hi2;
	read_again:
	hi = PDC_READL(a, PDC_BYTE_COUNT2) & 0x7fff;
	lo = PDC_READL(a, PDC_BYTE_COUNT) & 0x7fff;
	hi2 = PDC_READL(a, PDC_BYTE_COUNT2) & 0x7fff;
	if (hi != hi2 && retries--)
		goto read_again;
	return (hi << 15) | lo;
}

static void PDC_ADJUST_PLL(ACTRL *a, unsigned pll)
{
	unsigned ratio, required, r, f;
	if (pll < 5000000 || pll > 70000000) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "INVALID PLL %u, NOT SETTING", pll);
		return;
	}
	required = PDC_133(a) ? 133333333 : 100000000;
	ratio = required / (pll / 1000);

	if (ratio < 8600) {
		r = 0x0d;
	} else if (ratio < 12900) {
		r = 0x08;
	} else if (ratio < 16100) {
		r = 0x06;
	} else {
		r = 0x00;
	}
	f = (ratio * (r + 2)) / 1000 - 2;
	PDC_WRITEW(a, PDC_PPL_CTL, (r << 8) | f);
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 30 + 1);
}

static const char * const BAD_UDMA_133[] = {
	"Maxtor*",
	NULL,
};

static void PDC_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	__u32 gctl;
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2 | IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2 | IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4 | IDE_XFER_UDMA_5;
	if (PDC_133(p->ctrl)) *avail |= IDE_XFER_UDMA_6;
	*avail_unsupported = 0;
	gctl = PDC_READL(p->ctrl, PDC_GLOBAL_CTL + p->n * 0x100);
	if (gctl & 0x04000000)
		ATA$CABLE40_RESTRICT(avail, avail_unsupported);
	if (!drive)
		ATA$BLACKLIST_RESTRICT(BAD_UDMA_133, ident, IDE_XFER_UDMA_6, avail, avail_unsupported);
}

static const struct {
	__u8 v0, v1, v2;
} PDC_PIO_TIMING[5] = {
	{ 0xfb, 0x2b, 0xac },
	{ 0x46, 0x29, 0xa4 },
	{ 0x23, 0x26, 0x64 },
	{ 0x27, 0x0d, 0x35 },
	{ 0x23, 0x09, 0x25 },
};

static const struct {
	__u8 v0, v1;
} PDC_WDMA_TIMING[3] = {
	{ 0xdf, 0x5f },
	{ 0x6b, 0x27 },
	{ 0x69, 0x25 },
};

static const struct {
	__u8 v0, v1, v2;
} PDC_UDMA_TIMING[7] = {
	{ 0x4a, 0x0f, 0xd5 },
	{ 0x3a, 0x0a, 0xd0 },
	{ 0x2a, 0x07, 0xcd },
	{ 0x1a, 0x05, 0xcd },
	{ 0x1a, 0x03, 0xcd },
	{ 0x1a, 0x02, 0xcb },
	{ 0x1a, 0x01, 0xcb },
};

static int PDC_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	unsigned n;
	unsigned ctcr0_off = PDC_CTCR0 + p->n * 0x100 + drive * 8;
	unsigned ctcr1_off = PDC_CTCR1 + p->n * 0x100 + drive * 8;
	__u32 ctcr0 = PDC_READL(p->ctrl, ctcr0_off);
	__u32 ctcr1 = PDC_READL(p->ctrl, ctcr1_off);
	if (mode & IDE_XFER_MASK_PIO) {
		n = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
		if (n > 4)
			return 0;
		ctcr0 &= ~0x0000ffff;
		ctcr0 |= PDC_PIO_TIMING[n].v0;
		ctcr0 |= PDC_PIO_TIMING[n].v1 << 8;
		ctcr1 &= ~0xff000000;
		ctcr1 |= PDC_PIO_TIMING[n].v2 << 24;
	} else if (mode & IDE_XFER_MASK_WDMA) {
		n = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_WDMA);
		if (n > 2)
			return 0;
		ctcr0 &= ~0xffff0000;
		ctcr0 |= PDC_WDMA_TIMING[n].v0 << 16;
		ctcr0 |= PDC_WDMA_TIMING[n].v1 << 24;
	} else if (mode & IDE_XFER_MASK_UDMA) {
		n = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_UDMA);
		if (n > 6)
			return 0;
		ctcr1 &= ~0x00ffffff;
		ctcr1 |= PDC_UDMA_TIMING[n].v0;
		ctcr1 |= PDC_UDMA_TIMING[n].v1 << 8;
		ctcr1 |= PDC_UDMA_TIMING[n].v2 << 16;
	}
	PDC_WRITEL(p->ctrl, ctcr0_off, ctcr0);
	PDC_WRITEL(p->ctrl, ctcr1_off, ctcr1);
	return 0;
}

static int PDC_CHECK_ATAPI_DMA(APORT *p, SCSIRQ *rq)
{
	switch (rq->cmd[0]) {
		case SCMD_READ_6:
		case SCMD_WRITE_6:
		case SCMD_READ_10:
		case SCMD_WRITE_10:
		case SCMD_READ_12:
		case SCMD_WRITE_12:
		case SCMD_READ_CD:
		case SCMD_READ_DVD_STRUCTURE:
			return 0;
		default:
			return 1;
	}
}

static void PDC_DTOR(ACTRL *a)
{
	PCI$UNMAP_MEM_RESOURCE(a->pci_id, (void *)a->chipset_flags, PDC_REGSPACE);
}
