#include "ACTRL.H"

#define SVW_OSB4	0x00000001
#define SVW_CSB5	0x00000002
#define SVW_CSB6	0x00000004
#define SVW_HT1000	0x00000008
#define SVW_NO_UDMA	0x00000010

static const struct pci_id_s SOUTHBRIDGE_OSB4[] = {
	{ 0x1166, 0x0200, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

static void SETUP_OSB(ACTRL *a)
{
	pci_id_t id;
	if (!PCI$FIND_DEVICE(SOUTHBRIDGE_OSB4, 0, 0, 0, PCI$TEST_LIST, &id, NULL, NULL, 1)) {
		__u32 cfg64 = PCI$READ_CONFIG_DWORD(id, 0x64);
		cfg64 &= ~0x00002000;
		cfg64 |= 0x00004000;
		PCI$WRITE_CONFIG_DWORD(id, 0x64, cfg64);
	} else {
		a->chipset_flags |= SVW_NO_UDMA;
	}
}

static const struct pci_id_s SOUTHBRIDGE_CSB5[] = {
	{ 0x1166, 0x0201, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

static const struct pci_id_s SOUTHBRIDGE_CSB6[] = {
	{ 0x1166, 0x0203, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

static void SETUP_CSB(ACTRL *a)
{
	pci_id_t id;
	__u8 cfg5a;
	if (!(__GET_FIELD(a->pci_id, PCI_ID_T_FUNCTION) & 1)) {
		
		if (!PCI$FIND_DEVICE(SOUTHBRIDGE_CSB5, 0, 0, 0, PCI$TEST_LIST, &id, NULL, NULL, 1)) {
			__u32 cfg4c = PCI$READ_CONFIG_DWORD(id, 0x4c);
			cfg4c &= ~0x000007FF;
			cfg4c |= 0x00000060;
			PCI$WRITE_CONFIG_DWORD(id, 0x4c, cfg4c);
		}
	} else {
		if (!PCI$FIND_DEVICE(SOUTHBRIDGE_CSB6, 0, 0, 0, PCI$TEST_LIST, &id, NULL, NULL, 1)) {
			__u8 cfg41 = PCI$READ_CONFIG_BYTE(id, 0x41);
			cfg41 &= ~0x40;
			PCI$WRITE_CONFIG_BYTE(id, 0x41, cfg41);
		}
	}
	cfg5a = PCI$READ_CONFIG_BYTE(a->pci_id, 0x5a);
	cfg5a &= ~0x40;
	if (!(__GET_FIELD(a->pci_id, PCI_ID_T_FUNCTION) & 1))
		cfg5a |= 0x02;
	else if (PCI$READ_CONFIG_BYTE(a->pci_id, PCI_REVISION_ID) >= 0x92)
		cfg5a |= 0x03;
	else
		cfg5a |= 0x02;
	PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x5a, cfg5a);
}

static void SETUP_HT1000(ACTRL *a)
{
	__u8 cfg5a;
	cfg5a = PCI$READ_CONFIG_BYTE(a->pci_id, 0x5a);
	cfg5a &= ~0x40;
	cfg5a |= 0x03;
	PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x5a, cfg5a);
}

static const char * const BAD_UDMA_5[] = {
	"ST320011A",
	"ST340016A",
	"ST360021A",
	"ST380021A",
	NULL
};

static void SVW_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	ACTRL *a = p->ctrl;
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_SDMA_2 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2 | IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2;
	*avail_unsupported = 0;
	if (a->chipset_flags & (SVW_CSB5 | SVW_CSB6 | SVW_HT1000)) {
		*avail |= IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4;
	}
	if (a->chipset_flags & (SVW_CSB5 | SVW_CSB6)) {
		__u8 cfg5a = PCI$READ_CONFIG_BYTE(a->pci_id, 0x5a);
		if ((cfg5a & 0x03) == 0x03)
			*avail |= IDE_XFER_UDMA_5;
	}
	if (a->chipset_flags & SVW_HT1000) {
		*avail |= IDE_XFER_UDMA_5;
	}
	if (a->chipset_flags & SVW_NO_UDMA) {
		*avail_unsupported |= *avail & IDE_XFER_MASK_UDMA;
		*avail &= ~IDE_XFER_MASK_UDMA;
	}
	ATA$BLACKLIST_RESTRICT(BAD_UDMA_5, ident, IDE_XFER_UDMA_5, avail, avail_unsupported);
}

static const __u8 PIO_MODES[5] = { 0x5d, 0x47, 0x34, 0x22, 0x20 };
static const __u8 WDMA_MODES[3] = { 0x77, 0x21, 0x20 };

static int SVW_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	ACTRL *a = p->ctrl;
	unsigned o = p->n * 2 + 1 - drive;
	unsigned b = p->n * 2 + drive;
	unsigned b4 = b * 4;
	if (mode & (IDE_XFER_PIO_NONE | IDE_XFER_DMA_NONE))
		return 0;
	else if (mode & IDE_XFER_MASK_PIO) {
		unsigned pio = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
		if (pio >= 5)
			KERNEL$SUICIDE("SVW_SET_XFER: BAD PIO MODE: %08X", mode);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x40 + o, PIO_MODES[pio]);
		if (a->chipset_flags & (SVW_CSB5 | SVW_CSB6 | SVW_HT1000)) {
			__u16 cfg4a = PCI$READ_CONFIG_WORD(a->pci_id, 0x4a);
			cfg4a &= ~(0xf << b4);
			cfg4a |= pio << b4;
			PCI$WRITE_CONFIG_WORD(a->pci_id, 0x4a, cfg4a);
		}
	} else if (mode & IDE_XFER_MASK_DMA) {
		__u8 cfg54 = PCI$READ_CONFIG_BYTE(a->pci_id, 0x54);
		__u16 cfg56 = PCI$READ_CONFIG_BYTE(a->pci_id, 0x56);
		cfg54 &= ~(1 << b);
		cfg56 &= ~(0xf << b4);
		if (mode & IDE_XFER_MASK_WDMA) {
			unsigned wdma = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_WDMA);
			if (wdma >= 3)
				KERNEL$SUICIDE("SVW_SET_XFER: BAD WDMA MODE: %08X", mode);
			PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x44 + o, WDMA_MODES[wdma]);
		} else if (mode & IDE_XFER_MASK_UDMA) {
			unsigned udma = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_UDMA);
			if (udma >= 7)
				KERNEL$SUICIDE("SVW_SET_XFER: BAD UDMA MODE: %08X", mode);
			PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x44 + o, 0x20);
			cfg54 |= 1 << b;
			cfg56 |= udma << b4;
		} else
			KERNEL$SUICIDE("SVW_SET_XFER: BAD DMA MODE: %08X", mode);
		PCI$WRITE_CONFIG_WORD(a->pci_id, 0x56, cfg56);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, 0x54, cfg54);
	} else
		KERNEL$SUICIDE("SVW_SET_XFER: BAD MODE: %08X", mode);
	return 0;
}

int DETECT_SVW(ACTRL *a)
{
	int i;

	if (a->dev_id == 0x02111166) {
		a->chipset_name = "SERVERWORKS OSB4 ATA";
		a->chipset_flags = SVW_OSB4;
	} else if (a->dev_id == 0x02121166) {
		a->chipset_name = "SERVERWORKS CSB5 ATA";
		a->chipset_flags = SVW_CSB5;
	} else if (a->dev_id == 0x02131166) {
		a->chipset_name = "SERVERWORKS CSB6 ATA";
		a->chipset_flags = SVW_CSB6;
	} else if (a->dev_id == 0x02171166) {
		a->chipset_name = "SERVERWORKS CSB6 ATA";
		a->chipset_flags = SVW_CSB6;
	} else if (a->dev_id == 0x02141166) {
		a->chipset_name = "SERVERWORKS HT1000 ATA";
		a->chipset_flags = SVW_HT1000;
	} else return 0;

	PCI$WRITE_CONFIG_BYTE(a->pci_id, PCI_LATENCY_TIMER, 64);

	if (a->chipset_flags & SVW_OSB4) {
		SETUP_OSB(a);
	}
	if (a->chipset_flags & (SVW_CSB5 | SVW_CSB6)) {
		if (a->chipset_flags & SVW_CSB5)
			a->actrl_flags |= ACTRL_TRY_NONSIMPLEX;
		SETUP_CSB(a);
	}
	if (a->chipset_flags & SVW_HT1000) {
		SETUP_HT1000(a);
	}

	for (i = 0; i < a->n_ports; i++) {
		if (i && ((a->dev_id == 0x02171166) || a->chipset_flags & SVW_HT1000)) {
			a->port[i].aport_flags |= APORT_DISABLED;
			continue;
		}
		a->port[i].get_avail_xfer = SVW_GET_AVAIL_XFER;
		a->port[i].set_xfer = SVW_SET_XFER;
	}

	return 1;
}
