#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ATA.H>

#include "ACTRLREG.H"
#include "ACTRL.H"

static void TRIFLEX_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported);
static int TRIFLEX_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 ident[256]);
static void TRIFLEX_SETUP_XFER(APORT *p, ATARQ *rq);

#define PCI_TRIFLEX_TIMING	0x70
#define PCI_TRIFLEX_TIMING_NEXT	0x04

__COLD_ATTR__ int DETECT_TRIFLEX(ACTRL *a)
{
	int i;
	if (a->dev_id != 0xae330e11) return 0;
	a->chipset_name = "COMPAQ TRIFLEX";
	for (i = 0; i < 2; i++) {
		a->port[i].get_avail_xfer = TRIFLEX_GET_AVAIL_XFER;
		a->port[i].set_xfer = TRIFLEX_SET_XFER;
		a->port[i].aport_flags |= APORT_SETUP_XFER;
		a->port[i].setup_xfer = TRIFLEX_SETUP_XFER;
		a->port[i].xfer_mode[0][0] = 0xff;
		a->port[i].xfer_mode[0][1] = 0xff;
		a->port[i].xfer_mode[1][0] = 0xff;
		a->port[i].xfer_mode[1][1] = 0xff;
		a->port[i].port_chipset_flags = PCI$READ_CONFIG_DWORD(a->pci_id, PCI_TRIFLEX_TIMING + PCI_TRIFLEX_TIMING_NEXT * i);
	}
	return 1;
}

__COLD_ATTR__ static void TRIFLEX_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_SDMA_0 | IDE_XFER_SDMA_1 | IDE_XFER_SDMA_2 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2;
	*avail_unsupported = 0;
}

__COLD_ATTR__ static int TRIFLEX_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 ident[256])
{
	p->xfer_mode[!(mode & IDE_XFER_MASK_PIO)][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK);
	return 0;
}

static const __u16 xfer_timing[N_PIO_MODES + N_SDMA_MODES + N_WDMA_MODES - 2] = {
	0x0808,
	0x0508,
	0x0404,
	0x0204,
	0x0202,
	0x0202,
	0x0202,
	0x0202,
	0x0f0f,
	0x0f0f,
	0x0f0f,
	0x0808,
	0x0203,
	0x0103,
};

static void TRIFLEX_SETUP_XFER(APORT *p, ATARQ *rq)
{
	unsigned offset, tim;
	unsigned xfer;
	/* Compaq BIOS leaves IRQ pending after APM suspend/resume. Clear it */
	if (__likely(p->dma_io != 0)) {
		__u8 dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		if (__unlikely(dmastatus & (DMASTATUS_ERROR | DMASTATUS_IRQ)))
			io_outb(p->dma_io + DMAPORT_STATUS, dmastatus | DMASTATUS_IRQ | DMASTATUS_ERROR);
	}
	if (__unlikely((rq->atarq_flags & ATARQ_PROTOCOL) == ATA_PROTOCOL_NODATA)) return;
	xfer = p->xfer_mode[(rq->atarq_flags >> __BSF_CONST(ATA_PROTOCOL_DMA)) & 1][rq->device];
	if (__unlikely(xfer > __BSF_CONST(IDE_XFER_WDMA_2))) return;
	tim = xfer_timing[xfer];
	offset = rq->device * 16;
	__barrier();	/* turn off bogus optimization */
	if (__likely(((p->port_chipset_flags >> offset) & 0xffff) == tim)) return;
	p->port_chipset_flags = (p->port_chipset_flags & ~(0xffff << offset)) | (tim << offset);
	PCI$WRITE_CONFIG_DWORD(p->ctrl->pci_id, PCI_TRIFLEX_TIMING + p->n * PCI_TRIFLEX_TIMING_NEXT, p->port_chipset_flags);
}


