#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>

#include "ACTRL.H"

#define VIA_IDE_ENABLE		0x40
#define VIA_IDE_CONFIG		0x41
#define VIA_FIFO_CONFIG		0x43
#define VIA_MISC_1		0x44
#define VIA_MISC_2		0x45
#define VIA_MISC_3		0x46
#define VIA_DRIVE_TIMING	0x48
#define VIA_ADDRESS_SETUP	0x4c
#define VIA_8BIT_TIMING		0x4e
#define VIA_UDMA_TIMING		0x50

#define VIA_UDMA_33	0x0001
#define VIA_UDMA_66	0x0002
#define VIA_UDMA_100	0x0004
#define VIA_UDMA_133	0x0008
#define VIA_SET_FIFO	0x0010
#define VIA_NO_UNMASK	0x0020
#define VIA_BAD_PREQ	0x0040
#define VIA_BAD_CLK66	0x0080
#define VIA_BAD_AST	0x0100
#define VIA_BRIDGE_BUG	0x0200
#define VIA_UDMA_MASK	(VIA_UDMA_33 | VIA_UDMA_66 | VIA_UDMA_100 | VIA_UDMA_133)

static const struct {
	char *name;
	__u32 id;
	__u8 rev_min;
	__u8 rev_max;
	unsigned short flags;
} via_bridges[] = {
{ "VIA VX800",		0x83531106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA CX700",		0x83241106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8237S",	0x33721106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT6410",		0x31641106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8251",		0x32871106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8237",		0x32271106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8237A",	0x33371106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8235",		0x31771106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8233A",	0x31471106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
{ "VIA VT8233C",	0x31091106, 0x00, 0x2f,	VIA_UDMA_100 },
{ "VIA VT8233",		0x30741106, 0x00, 0x2f,	VIA_UDMA_100 },
{ "VIA VT8231",		0x82311106, 0x00, 0x2f,	VIA_UDMA_100 | VIA_BRIDGE_BUG },
{ "VIA VT82C686B",	0x06861106, 0x40, 0x4f,	VIA_UDMA_100 | VIA_BRIDGE_BUG },
{ "VIA VT82C686A",	0x06861106, 0x10, 0x2f,	VIA_UDMA_66 },
{ "VIA VT82C686",	0x06861106, 0x00, 0x0f,	VIA_UDMA_33 | VIA_BAD_CLK66 },
{ "VIA VT82C596B",	0x05961106, 0x10, 0x2f,	VIA_UDMA_66 },
{ "VIA VT82C596A",	0x05961106, 0x00, 0x0f,	VIA_UDMA_33 | VIA_BAD_CLK66 },
{ "VIA VT82C586B",	0x05861106, 0x47, 0x4f,	VIA_UDMA_33 | VIA_SET_FIFO },
{ "VIA VT82C586B",	0x05861106, 0x40, 0x46,	VIA_UDMA_33 | VIA_SET_FIFO | VIA_BAD_PREQ },
{ "VIA VT82C586B",	0x05861106, 0x30, 0x3f,	VIA_UDMA_33 | VIA_SET_FIFO },
{ "VIA VT82C586A",	0x05861106, 0x20, 0x2f,	VIA_UDMA_33 | VIA_SET_FIFO },
{ "VIA VT82C586",	0x05861106, 0x00, 0x0f,	VIA_SET_FIFO },
{ "VIA VT82C576",	0x05761106, 0x00, 0x2f,	VIA_SET_FIFO | VIA_NO_UNMASK },
{ "VIA VT82C576",	0x05761107, 0x00, 0x2f,	VIA_SET_FIFO | VIA_NO_UNMASK },
};

static const char * const BAD_UDMA_33[] = {
	"TS*",
	NULL,
};

static void VIA_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	ACTRL *a = p->ctrl;
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_PIO_5 | IDE_XFER_PIO_6 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2 | IDE_XFER_WDMA_3 | IDE_XFER_WDMA_4;
	*avail_unsupported = 0;
	switch (a->chipset_flags & VIA_UDMA_MASK) {
	case VIA_UDMA_133:
		*avail |= IDE_XFER_UDMA_6;
	case VIA_UDMA_100:
		*avail |= IDE_XFER_UDMA_5;
	case VIA_UDMA_66:
		*avail |= IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4;
	case VIA_UDMA_33:
		*avail |= IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2;
		*avail_unsupported |= IDE_XFER_UDMA_3;
	}
	*avail_unsupported &= ~*avail;
	if (a->chipset_flags & VIA_UDMA_33) {
		ATA$BLACKLIST_RESTRICT(BAD_UDMA_33, ident, IDE_XFER_MASK_UDMA, avail, avail_unsupported);
	}
#if 0
/* Cable detection doesn't work if the drive was not recognised by BIOS
   --- the purpose of this code is to allow operation of devices in case of
   BIOS failure, so it is pointless to rely on BIOS here */
	if (ident && ident[93]) switch (a->chipset_flags & VIA_UDMA_MASK)
	case VIA_UDMA_133:
	case VIA_UDMA_100: {
		__u32 cblid = PCI$READ_CONFIG_DWORD(a->pci_id, 0x50);
		if (!(cblid & (0x10100000 >> (16 * p->n))))
			ATA$CABLE40_RESTRICT(avail, avail_unsupported);
	}
#endif
}

static int VIA_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	struct ata_timing t;
	unsigned udma_clock;
	unsigned val;
	unsigned reg_offs;
	ACTRL *a = p->ctrl;

	if (mode & IDE_XFER_MASK_PIO) {
		p->xfer_mode[0][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
	} else {
		p->xfer_mode[1][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_DMA);
	}

	ATA$INIT_TIMING(&t);
	ATA$MERGE_PIO_TIMING(&t, p->xfer_mode[0][drive], ATA_MERGE_ALL);
	ATA$MERGE_DMA_TIMING(&t, p->xfer_mode[1][drive], ATA_MERGE_ALL);
	ATA$MERGE_PIO_TIMING(&t, p->xfer_mode[0][drive ^ 1], ATA_MERGE_8BIT);

	udma_clock = a->bus_clock;
	switch (a->chipset_flags & VIA_UDMA_MASK) {
	case VIA_UDMA_133:
		udma_clock *= 4;
		break;
	case VIA_UDMA_100:
		udma_clock *= 3;
		break;
	case VIA_UDMA_66:
		udma_clock *= 2;
		break;
	}

	ATA$QUANTIZE_TIMING(&t, a->bus_clock, udma_clock);

	reg_offs = 3 - drive - (p->n << 1);

	if (t.setup && !(a->chipset_flags & VIA_BAD_AST)) {
		__u8 ast = PCI$READ_CONFIG_BYTE(a->pci_id, VIA_ADDRESS_SETUP);
		int shift = 2 * reg_offs;
		ast &= ~(3 << shift);
		ast |= ATA$FIT_TIMING(t.setup, 1, 4, 1) << shift;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_ADDRESS_SETUP, ast);
	}

	/*__debug_printf("%d: 8: %d %d\n", p->n, t.act8b, t.rec8b);*/
	if (t.act8b && t.rec8b)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_8BIT_TIMING + 1 - p->n, (ATA$FIT_TIMING(t.act8b, 1, 16, 1) << 4) | ATA$FIT_TIMING(t.rec8b, 1, 16, 1));
	/*__debug_printf("%d: 16: %d %d\n", p->n, t.active, t.recover);*/
	if (t.active && t.recover)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_DRIVE_TIMING + reg_offs, (ATA$FIT_TIMING(t.active, 1, 16, 1) << 4) | ATA$FIT_TIMING(t.recover, 1, 16, 1));

	if (a->chipset_flags & VIA_UDMA_MASK) {
		if (t.udma) {
			val = ATA$FIT_TIMING(t.udma, 2, a->chipset_flags & VIA_UDMA_33 ? 5 : 9, 2);
			if (a->chipset_flags & VIA_UDMA_66) val |= 0xe8;
			else val |= 0xe0;
			/*__debug_printf("mode: %x, udma %02x, timing: %02x, port: %02x\n", mode, t.udma, val, VIA_UDMA_TIMING + reg_offs);*/
		} else {
			switch (a->chipset_flags & VIA_UDMA_MASK) {
			default:
				val = 0x07;
				break;
			case VIA_UDMA_66:
				val = 0x0f;
				break;
			case VIA_UDMA_33:
				val = 0x03;
				break;
			}
		}
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_UDMA_TIMING + reg_offs, val);
	}

	return 0;
}

static const struct pci_id_s southbridge_table[] = {
	{ 0x1106, 0x0305, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0x1106, 0x0391, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0x1106, 0x3102, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0x1106, 0x3112, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

static void VIA_SOUTHBRIDGE_FIXUP(ACTRL *a)
{
	pci_id_t id;
	if (!PCI$FIND_DEVICE(southbridge_table, 0, 0, 0, PCI$TEST_LIST, &id, NULL, NULL, 1)) {
		__u8 reg76 = PCI$READ_CONFIG_BYTE(id, 0x76);
		if ((reg76 & 0xf0) != 0xd0) {
			PCI$WRITE_CONFIG_BYTE(id, 0x75, 0x80);
			PCI$WRITE_CONFIG_BYTE(id, 0x76, (reg76 & 0x0f) | 0xd0);
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, a->dev_name, "APPLYING VIA SOUTHBRIDGE DATA CORRUPTION FIX");
		}
	}
}

int DETECT_VIA(ACTRL *a)
{
	unsigned i;
	if (a->dev_id != 0x15711106 &&
	    a->dev_id != 0x05711106 &&
	    a->dev_id != 0x05811106 &&
	    a->dev_id != 0x31641106 &&
	    a->dev_id != 0x53241106) return 0;
	for (i = 0; i < sizeof(via_bridges) / sizeof(*via_bridges); i++) {
		if (ATA_FIND_DEV(a, via_bridges[i].id, via_bridges[i].rev_min, via_bridges[i].rev_max)) goto got_it;
	}
	return 0;
	got_it:
	a->chipset_name = via_bridges[i].name;
	a->chipset_flags = via_bridges[i].flags;

	if (a->chipset_flags & VIA_BRIDGE_BUG) {
		VIA_SOUTHBRIDGE_FIXUP(a);
	}

	if (a->chipset_flags & (VIA_UDMA_66 | VIA_BAD_CLK66)) {
		__u32 u = PCI$READ_CONFIG_DWORD(a->pci_id, VIA_UDMA_TIMING);
		if (a->chipset_flags & VIA_UDMA_66) {
			PCI$WRITE_CONFIG_DWORD(a->pci_id, VIA_UDMA_TIMING, u | 0x80008);
		} else if (a->chipset_flags & VIA_BAD_CLK66) {
			PCI$WRITE_CONFIG_DWORD(a->pci_id, VIA_UDMA_TIMING, u & ~0x80008);
		}
	}
	if (a->chipset_flags & (VIA_SET_FIFO | VIA_BAD_PREQ)) {
		__u8 f = PCI$READ_CONFIG_BYTE(a->pci_id, VIA_FIFO_CONFIG);
		if (a->chipset_flags & VIA_BAD_PREQ) f &= ~0x80;
		if (a->chipset_flags & VIA_SET_FIFO) {
			f = f & ~0x60;
			if (!(a->port[0].aport_flags & APORT_DISABLED) && !(a->port[1].aport_flags & APORT_DISABLED)) f |= 0x20;
			else if (!(a->port[1].aport_flags & APORT_DISABLED)) f |= 0x60;
		}
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_FIFO_CONFIG, f);
	}
	if (a->chipset_flags & VIA_NO_UNMASK) {
		a->actrl_flags |= ACTRL_FORCE_MASKIRQ;
	}
	for (i = 0; i < a->n_ports; i++) {
		a->port[i].get_avail_xfer = VIA_GET_AVAIL_XFER;
		a->port[i].set_xfer = VIA_SET_XFER;
		a->port[i].xfer_mode[0][0] = 0xff;
		a->port[i].xfer_mode[0][1] = 0xff;
		a->port[i].xfer_mode[1][0] = 0xff;
		a->port[i].xfer_mode[1][1] = 0xff;
	}
	return 1;
}
