#include <ARCH/BSF.H>
#include <SPAD/PCI.H>
#include <SYS/PARAM.H>

#include "ACTRL.H"

#define VT_PATA_DRIVE_TIMING		0xa8
#define VT_PATA_8BIT_TIMING		0xae
#define VT_PATA_UDMA_TIMING		0xb0

static int INIT_VT6421(ACTRL *a);
static void VT_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported);
static int VT_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256]);

int DETECT_VT6421(pci_id_t pci_id, int (**special_init)(ACTRL *a))
{
	__u32 dev_id = PCI$READ_CONFIG_DWORD(pci_id, PCI_VENDOR_ID);
	if (dev_id != 0x32491106)
		return 0;
	*special_init = INIT_VT6421;
	/* The card is 3-port, but the controller looks to be 4-port.
	   This can be increased if 4-port card will be seen. */
	return 3;
}

static int INIT_VT6421(ACTRL *a)
{
	int i;
	int irq = PCI$READ_INTERRUPT_LINE(a->pci_id);
	for (i = 0; i < a->n_ports; i++) {
		io_t base_port, dma_port;
		base_port = PCI$READ_IO_RESOURCE(a->pci_id, i);
		dma_port = PCI$READ_IO_RESOURCE(a->pci_id, 4);
		if (!base_port) {
			a->port[i].aport_flags = APORT_DISABLED;
			continue;
		}
		a->port[i].io = base_port;
		a->port[i].alt_io = base_port + 8;
		if (dma_port)
			a->port[i].dma_io = dma_port + i * 8;
		a->port[i].irq = irq;
		if (i >= 2) {
			a->port[i].aport_flags &= ~APORT_SATA;
			a->port[i].get_avail_xfer = VT_GET_AVAIL_XFER;
			a->port[i].set_xfer = VT_SET_XFER;
			a->port[i].xfer_mode[0][0] = 0xff;
			a->port[i].xfer_mode[0][1] = 0xff;
			a->port[i].xfer_mode[1][0] = 0xff;
			a->port[i].xfer_mode[1][1] = 0xff;
		} else {
			a->port[i].aport_flags |= APORT_SATA;
		}
	}
	return 1;
}

static void VT_GET_AVAIL_XFER(APORT *p, int drive, const __u16 ident[256], unsigned *avail, unsigned *avail_unsupported)
{
	unsigned reg_offs = 3 - drive - ((p->n << 1) & 1);
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_PIO_5 | IDE_XFER_PIO_6 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2 | IDE_XFER_WDMA_3 | IDE_XFER_WDMA_4 | IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2 | IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4 | IDE_XFER_UDMA_5 | IDE_XFER_UDMA_6;
	*avail_unsupported = 0;
	if (PCI$READ_CONFIG_BYTE(p->ctrl->pci_id, VT_PATA_UDMA_TIMING + reg_offs) & 0x10)
		ATA$CABLE40_RESTRICT(avail, avail_unsupported);
}

static int VT_SET_XFER(APORT *p, int drive, unsigned mode, const __u16 id[256])
{
	ACTRL *a = p->ctrl;
	unsigned reg_offs = 3 - drive - ((p->n << 1) & 1);
	struct ata_timing t;
	unsigned val;
	__u8 reg;
	if (mode & IDE_XFER_MASK_PIO) {
		p->xfer_mode[0][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_PIO);
	} else {
		p->xfer_mode[1][drive] = ATA$XFER_GET_NUMBER(mode, IDE_XFER_MASK_DMA);
	}
	ATA$INIT_TIMING(&t);
	ATA$MERGE_PIO_TIMING(&t, p->xfer_mode[0][drive], ATA_MERGE_ALL);
	ATA$MERGE_DMA_TIMING(&t, p->xfer_mode[1][drive], ATA_MERGE_ALL);
	ATA$MERGE_PIO_TIMING(&t, p->xfer_mode[0][drive ^ 1], ATA_MERGE_8BIT);
	ATA$QUANTIZE_TIMING(&t, a->bus_clock, a->bus_clock * 4);
	if (t.act8b && t.rec8b)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VT_PATA_8BIT_TIMING + 1 - (p->n & 1), (ATA$FIT_TIMING(t.act8b, 1, 16, 1) << 4) | ATA$FIT_TIMING(t.rec8b, 1, 16, 1));
	if (t.active && t.recover)
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VT_PATA_DRIVE_TIMING + reg_offs, (ATA$FIT_TIMING(t.active, 1, 16, 1) << 4) | ATA$FIT_TIMING(t.recover, 1, 16, 1));
	/*__debug_printf("%02x %02x\n",
		(ATA$FIT_TIMING(t.act8b, 1, 16, 1) << 4) | ATA$FIT_TIMING(t.rec8b, 1, 16, 1),
		(ATA$FIT_TIMING(t.active, 1, 16, 1) << 4) | ATA$FIT_TIMING(t.recover, 1, 16, 1));*/
	if (t.udma) {
		val = ATA$FIT_TIMING(t.udma, 2, 17, 2);
		val |= 0xe0;
	} else {
		val = 0x0f;
	}
	/*__debug_printf("computed value: %02x\n", val);*/
	reg = PCI$READ_CONFIG_BYTE(a->pci_id, VT_PATA_UDMA_TIMING + reg_offs);
	reg &= 0x10;
	reg |= val;
	PCI$WRITE_CONFIG_BYTE(a->pci_id, VT_PATA_UDMA_TIMING + reg_offs, reg);

	return 0;
}

