#ifndef CODE_ATAPI_QUEUE_DEPTH
#define CODE_ATAPI_QUEUE_DEPTH		1
#endif

#ifndef CODE_PREPARE_FOR_IRQ
#define CODE_PREPARE_FOR_IRQ(p)
#endif

#ifndef CODE_SETUP_XFER
#define CODE_SETUP_XFER(p, rq)
#endif

#ifndef CODE_CHECK_ATAPI_DMA
#define CODE_CHECK_ATAPI_DMA(p, rq)	0
#endif

#define ATAPI_XFER_LEN		(0x10000 - BIO_SECTOR_SIZE)

#define ATAPI_AUTOSENSE_LENGTH	18

#define	PHASE_COMMAND		0x01
#define PHASE_MAPPED_DMA	0x02
#define PHASE_MAPPED_PIO	0x04
#define PHASE_RESET		0x08

#define atapi_phase		atapi_placeholder.status
#define atapi_device		atapi_placeholder.device
#define atapi_len		atapi_placeholder.len
#define atapi_sec_word		atapi_placeholder.retries
#define atapi_error_register	atapi_placeholder.fis.feature8
#define atapi_atarq_flags	atapi_placeholder.atarq_flags

#define IS_SENSE(p, rq)		((rq) == (SCSIRQ *)(void *)&(p)->sense_rq)

static int ATAPI_POST(SCSIRQ *rq);
static void ATAPI_STATUS_ERROR(APORT *p, __u8 status);
static void ATAPI_END_SENSE_REQUEST(APORT *p);
static void ATAPI_ERROR_RESET_DRIVE(APORT *p);
static void ATAPI_CANCEL(SCSIRQ *rq);

__COLD_ATTR__ static void ATAPI_INIT(SCSI_ATTACH_PARAM *sa)
{
	ATA_ATTACH_PARAM *ap = sa->adapter;
	APORT *p = ap->port;
	unsigned al = ATA$SGLIST_GUARANTEED_SIZE(p->sglist, !!(p->device[ap->device].dev_flags & DEV_F_ATAPI_DMA));
	if (p->atapi_length && p->atapi_length < al)
		al = p->atapi_length;
	sa->max_sectors = al >> BIO_SECTOR_SIZE_BITS;
	sa->max_untagged_requests = CODE_ATAPI_QUEUE_DEPTH;
	sa->max_tagged_requests = 0;
	sa->max_fragments = 0;

	sa->post = ATAPI_POST;
	sa->cancel = ATAPI_CANCEL;
}

__COLD_ATTR__ static char *ATAPI_NAME(APORT *p)
{
	return *p->device[p->atapi_device].attached->dev_name;
}

static int ATAPI_POLL_DRQ(APORT *p)
{
	__u8 status;
	u_jiffies_lo_t j, jj, timeout;
	timeout = __unlikely(p->device[p->atapi_device].dev_flags & (DEV_F_ATAPI_DRQ_INT | DEV_F_ATAPI_LONG_DRQ)) ? ATAPI_DRQ_LONG : ATAPI_DRQ_SHORT;
	timeout = TIMEOUT_JIFFIES(timeout);
	j = KERNEL$GET_JIFFIES_LO();
	do {
		jj = KERNEL$GET_JIFFIES_LO();
		status = ATA_IN_STATUS(p);
		if (__unlikely(status & ATA_STATUS_BSY)) continue;
		if (__unlikely(status & ATA_STATUS_ERROR)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "ERROR POLLING FOR DRQ, STATUS %02X, ERROR %02X", status, ATA_IN_ERROR(p));
			return -1;
		}
		if (__likely(status & ATA_STATUS_DRQ)) return 0;
	} while (jj - j <= timeout);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "TIMEOUT ON POLLING FOR DRQ, STATUS %02X", status);
	return -1;
}

static int ATAPI_COMMAND(APORT *p);
static void ATAPI_END_REQUEST(APORT *p);
static void ATAPI_DONE(ATARQ *rq);
static void ATAPI_UNMAP(APORT *p);
static void ATAPI_NONDMA_IRQ(APORT *p, __u8 status);
static void ATAPI_RESET_POLL(TIMER *t);

static int ATAPI_POST(SCSIRQ *rq)
{
	ATA_ATTACH_PARAM *ap = rq->adapter;
	APORT *p = ap->port;
	int r;
	BIODESC *desc;
	unsigned dev_flags;
	__u8 status;
	if (__unlikely(p->aport_flags & APORT_LOCK)) {
		r = -EAGAIN;
		goto ret0;
	}
	if (__unlikely(p->current_rq != NULL)) {
		if (__unlikely(IS_SENSE(p, rq))) goto ok;
		r = -EAGAIN;
		goto ret0;
	}
	ok:
	ATA_OUT_DRIVE(p, (ap->device * ATA_DEVICE_DRIVE) | ATA_DEVICE_ATAPI_OR_BITS);
	status = ATA_IN_STATUS(p);
	if (__unlikely(status & (ATA_STATUS_BSY | ATA_STATUS_DRQ))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "DEVICE NOT READY: STATUS %02X", status);
		r = -EIO;
		goto ret0;
	}
	p->current_rq = &p->atapi_placeholder;
	p->atapi_phase = PHASE_COMMAND;
	p->atapi_device = ap->device;
	p->atapi_placeholder.done = ATAPI_DONE;

	p->atapi_placeholder.atarq_flags = ATA_PROTOCOL_PIO | ATARQ_TO_DEVICE;
	p->scsirq = rq;
	dev_flags = p->device[p->atapi_device].dev_flags;
#if __DEBUG >= 1
	if (!(dev_flags & DEV_F_ATAPI))
		KERNEL$SUICIDE("ATAPI_POST: INVALID DEVICE FLAGS: %X", dev_flags);
#endif
	if (__unlikely(rq->cmdlen > 12 + ((dev_flags / (DEV_F_ATAPI_CMD_16 / 4)) & 4))) {
		r = -EOPNOTSUPP;
		goto ret1;
	}
	p->atapi_len = 0;
	if (__unlikely(!rq->direction)) {
		goto no_data;
	}
	desc = rq->desc;
	do {
		p->atapi_len += desc->v.len;
	} while ((desc = desc->next));
	CODE_SETUP_XFER(p, &p->atapi_placeholder);
	if (__likely(dev_flags & DEV_F_ATAPI_DMA)) {
		if (__unlikely(CODE_CHECK_ATAPI_DMA(p, rq)))
			goto unaligned__pio;
		if (__unlikely((((unsigned long)rq->desc->v.ptr | rq->desc->v.len) & (BIO_SECTOR_SIZE - 1)) != 0)) {
			goto unaligned__pio;
		}
		p->atapi_phase |= PHASE_MAPPED_DMA;
		r = ATA$MAP_DMA(p->sglist, &rq->desc, p->atapi_len, rq->direction, 1);
		if (__unlikely(r != p->atapi_len)) {
			map_failed:
			if (__unlikely(r >= 0)) {
				r = -EVSPACEFAULT;
				goto ret2;
			}
			goto ret1;
		}
		p->atapi_atarq_flags = (rq->direction & PF_WRITE) * ATARQ_TO_DEVICE;
		if (__unlikely(dev_flags & DEV_F_ATAPI_DMADIR)) {
			ATA_OUT_FEATURES(p, ATA_FEATURE_PACKET_DMA | ((rq->direction & PF_WRITE) * ATA_FEATURE_PACKET_DMADIR));
		} else {
			ATA_OUT_FEATURES(p, ATA_FEATURE_PACKET_DMA);
		}
	} else {
		unaligned__pio:
		if (__unlikely(((unsigned long)rq->desc->v.ptr & (BIO_SECTOR_SIZE - 1)) != 0)) {
			if (__unlikely((((unsigned long)rq->desc->v.ptr & __PAGE_CLUSTER_SIZE_MINUS_1) + rq->desc->v.len) > __PAGE_CLUSTER_SIZE)) {
				r = -ERANGE;
				goto ret1;
			}
		}
		p->atapi_phase |= PHASE_MAPPED_PIO;
		r = ATA$MAP_PIO(p->sglist, &rq->desc, p->atapi_len, rq->direction, 1);
		if (__unlikely(r != p->atapi_len)) goto map_failed;
		p->pio_sg_pos = 0;
		p->atapi_sec_word = 0;
		no_data:
		ATA_OUT_FEATURES(p, 0);
	}
	ATA_OUT_LBA_M(p, ATAPI_XFER_LEN & 0xff);
	ATA_OUT_LBA_H(p, ATAPI_XFER_LEN >> 8);
	if (__unlikely(dev_flags & DEV_F_ATAPI_DRQ_INT)) {
		CODE_PREPARE_FOR_IRQ(p);
	}
	ATA_OUT_CMD(p, ATA_CMD_PACKET);

	rq->status = p->atapi_len;
	rq->scsi_status = SCSI_GOOD;

	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		CODE_SETUP_DMA(p, &p->atapi_placeholder);
	}

	SET_TIMER_NEVER(&p->timeout);
	if (__unlikely(dev_flags & DEV_F_ATAPI_DRQ_INT)) {
		return 0;
	}
	if (__unlikely(ATAPI_POLL_DRQ(p))) {
		ATAPI_ERROR_RESET_DRIVE(p);
		return 0;
	}
	return ATAPI_COMMAND(p);

	ret2:
	ATAPI_UNMAP(p);
	ret1:
	p->current_rq = NULL;
	ret0:
	return r;
}

static int ATAPI_COMMAND(APORT *p)
{
	unsigned dev_flags;
	char atapi_packet[16];
	CODE_PREPARE_FOR_IRQ(p);
	memset(atapi_packet, 0, sizeof atapi_packet);
	memcpy(atapi_packet, p->scsirq->cmd, p->scsirq->cmdlen);
	ATA_DI(p, p->atapi_device);
	dev_flags = p->device[p->atapi_device].dev_flags;
	ATA_OUTS(p, dev_flags & DEV_F_IO32, atapi_packet, 6 + ((dev_flags / (DEV_F_ATAPI_CMD_16 / 2)) & 2));
	ATA_EI(p, p->atapi_device);
	p->atapi_phase &= ~PHASE_COMMAND;
	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		p->atapi_placeholder.atarq_flags = ATA_PROTOCOL_DMA;
		CODE_SETUP_XFER(p, &p->atapi_placeholder);
		CODE_START_DMA(p, &p->atapi_placeholder);
	}
	return 0;
}

static void ATAPI_IRQ(APORT *p, __u8 dmastatus, __u8 status)
{
	if (__unlikely(p->atapi_phase & (PHASE_COMMAND | PHASE_RESET))) {
		if (__unlikely(p->atapi_phase & PHASE_RESET)) {
			KERNEL$DEL_TIMER(&p->timeout);
			ATAPI_RESET_POLL(&p->timeout);
			return;
		}
		if (__unlikely(ATAPI_POLL_DRQ(p))) {
			p->scsirq->status = -EIO;
			ATAPI_END_REQUEST(p);
			return;
		}
		ATAPI_COMMAND(p);
		return;
	}
	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		if (__unlikely(dmastatus & DMASTATUS_ERROR)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "DMA ERROR ON ATAPI COMMAND %02X, DMA STATUS %02X, DRIVE STATUS %02X, ERROR %02X", p->scsirq->cmd[0], dmastatus, status, ATA_IN_ERROR(p));
			p->scsirq->status = -EIO;
			ATAPI_END_REQUEST(p);
			return;
		}
		if (__unlikely(status & ATA_STATUS_ERROR)) {
			ATAPI_STATUS_ERROR(p, status);
			return;
		}
		ATAPI_END_REQUEST(p);
	} else {
		ATAPI_NONDMA_IRQ(p, status);
	}
}

static void ATAPI_END_REQUEST(APORT *p)
{
	if (__unlikely(IS_SENSE(p, p->scsirq))) {
		ATAPI_END_SENSE_REQUEST(p);
		return;
	}
	ATA_END_REQUEST(p, p->atapi_phase);
}

static void ATAPI_DONE(ATARQ *rq)
{
	APORT *p = GET_STRUCT(rq, APORT, atapi_placeholder);
	SCSIRQ *scsirq;
	ATAPI_UNMAP(p);
	scsirq = p->scsirq;
	/*__debug_printf("done(%02x) -> %d\n", scsirq->cmd[0], scsirq->status);*/
#if __DEBUG >= 1
	p->scsirq = NULL;
#endif
	scsirq->done(scsirq);
}

static void ATAPI_UNMAP(APORT *p)
{
	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		ATA$UNMAP_DMA(p->sglist);
	} else if (p->atapi_phase & PHASE_MAPPED_PIO) {
		ATA$UNMAP_PIO(p->sglist);
	}
}

static void ATAPI_NONDMA_IRQ(APORT *p, __u8 status)
{
	CODE_PREPARE_FOR_IRQ(p);
	if (status & ATA_STATUS_DRQ) {
		unsigned devlen;
		__u8 ireason = ATA_IN_COUNT(p);
		__u8 lo = ATA_IN_LBA_M(p);
		__u8 hi = ATA_IN_LBA_H(p);
		devlen = (hi << 8) | lo;
		if (__unlikely(!(p->atapi_phase & PHASE_MAPPED_PIO))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TRANSFER FOR NON-DATA COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X", p->scsirq->cmd[0], status, ireason, devlen);
			end_reset:
			ATAPI_ERROR_RESET_DRIVE(p);
			return;
		}
		if (__unlikely((ireason & (ATA_QSTAT_CD | ATA_QSTAT_IO)) != (p->scsirq->direction & PF_WRITE) * ATA_QSTAT_IO)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TRANSFER DATA IN BAD DIRECTION, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X", p->scsirq->cmd[0], status, ireason, devlen);
			goto end_reset;
		}
		if (__unlikely(devlen - 1 >= ATAPI_XFER_LEN)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TO TRANSFER INVALID LENGTH, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X", p->scsirq->cmd[0], status, ireason, devlen);
			goto end_reset;
		}
		if (__unlikely(devlen > p->atapi_len)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TO TRANSFER MORE DATA THAN EXPECTED, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X (EXPECTED %X)", p->scsirq->cmd[0], status, ireason, devlen, p->atapi_len);
			goto end_reset;
		}
		if (__unlikely(devlen & 1) && __unlikely(devlen != p->atapi_len)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TO TRANSFER ODD NUMBER OF BYTES IN THE MIDDLE OF THE STREAM, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X (REMAINING %X)", p->scsirq->cmd[0], status, ireason, devlen, p->atapi_len);
			goto end_reset;
		}
		p->atapi_len -= devlen;
		CODE_PREPARE_FOR_IRQ(p);
		ATA_DI(p, p->atapi_device);
		do {
			__u16 *data = ATA$MAP_PIO_SECTOR(p->sglist, p->pio_sg_pos);
			unsigned words = (BIO_SECTOR_SIZE / 2) - p->atapi_sec_word;
			if (__unlikely(devlen == 1)) {
				if (__likely(p->scsirq->direction & PF_WRITE)) {
					*(__u8 *)(data + p->atapi_sec_word) = ATA_IN_DATA16(p);
				} else {
					ATA_OUT_DATA16(p, *(__u8 *)(data + p->atapi_sec_word));
				}
				ATA$UNMAP_PIO_SECTOR(p->sglist, data);
				break;
			}
			if (__likely(words > (devlen >> 1))) words = devlen >> 1;
			if (__likely(p->scsirq->direction & PF_WRITE)) {
				ATA_INS(p, p->device[p->atapi_device].dev_flags & DEV_F_IO32, data + p->atapi_sec_word, words);
			} else {
				ATA_OUTS(p, p->device[p->atapi_device].dev_flags & DEV_F_IO32, data + p->atapi_sec_word, words);
			}
			p->atapi_sec_word = (p->atapi_sec_word + words) & (BIO_SECTOR_SIZE / 2 - 1);
			if (__unlikely(!p->atapi_sec_word)) ATA$SET_NEXT_PIO_SECTOR(p->sglist, &p->pio_sg_pos);
			devlen -= words << 1;
			ATA$UNMAP_PIO_SECTOR(p->sglist, data);
		} while (devlen);
		ATA_EI(p, p->atapi_device);
		if (__unlikely(status & ATA_STATUS_ERROR)) goto status_error;
		return;
	}
	p->scsirq->status -= p->atapi_len;
	if (__unlikely(status & ATA_STATUS_ERROR)) {
		status_error:
		ATAPI_STATUS_ERROR(p, status);
		return;
	}
	ATAPI_END_REQUEST(p);
}

__COLD_ATTR__ static void ATAPI_STATUS_ERROR(APORT *p, __u8 status)
{
	__u8 error;
	if (__unlikely(IS_SENSE(p, p->scsirq))) {
		/* some CD-ROMs reportedly do not have sense */
		goto ret_eio;
	}
	error = ATA_IN_ERROR(p);
	if (__likely(error & ATA_ERROR_SENSE_KEY)) {
		int r;
		KERNEL$DEL_TIMER(&p->timeout);
		p->error_register = error;
		p->orig_scsirq = p->scsirq;
		p->scsirq = (SCSIRQ *)(void *)&p->sense_rq;
		p->sense_rq.status = 0;
		p->sense_rq.ch_id = p->orig_scsirq->ch_id;
		p->sense_rq.lun = p->orig_scsirq->lun;
		p->sense_rq.adapter = p->orig_scsirq->adapter;
		p->sense_rq.desc = &p->sense_desc;
		p->sense_rq.internal = 0;
		p->sense_rq.direction = PF_WRITE;
		p->sense_rq.cmdlen = 6;
		p->sense_rq.tagging = SCSI_TAGGING_NONE;
		p->sense_rq.cmd[0] = SCMD_REQUEST_SENSE;
		p->sense_rq.cmd[1] = p->orig_scsirq->cmd[1] & 0xe0;
		p->sense_rq.cmd[4] = p->orig_scsirq->sense_size;
		if (p->orig_scsirq->sense_size > ATAPI_AUTOSENSE_LENGTH) {
			p->sense_rq.cmd[4] = ATAPI_AUTOSENSE_LENGTH;
		}
		p->sense_desc.v.ptr = (unsigned long)p->orig_scsirq->sense;
		p->sense_desc.v.len = p->sense_rq.cmd[4];
		p->sense_desc.v.vspace = &KERNEL$VIRTUAL;
		p->sense_desc.next = NULL;
		ATAPI_UNMAP(p);
		if (__unlikely(r = ATAPI_POST((SCSIRQ *)(void *)&p->sense_rq))) {
			KERNEL$SUICIDE("ATAPI_STATUS_ERROR: CANNOT POST REQUEST SENSE: %d", r);
		}
		return;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "DEVICE ERROR: COMMAND %02X, STATUS %02X, ERROR %02X", p->scsirq->cmd[0], status, error);
	ret_eio:
	p->scsirq->status = -EIO;
	ATAPI_END_REQUEST(p);
}

__COLD_ATTR__ static void ATAPI_END_SENSE_REQUEST(APORT *p)
{
	SCSIRQ *scsirq = p->scsirq = p->orig_scsirq;
	__u8 *sense = scsirq->sense;
	unsigned sense_size = scsirq->sense_size;
	p->orig_scsirq = NULL;
	scsirq->scsi_status = SCSI_CHECK_CONDITION;
	if (__unlikely(p->sense_rq.status < 0)) {
		memset(sense, 0, sense_size);
		if (sense_size > 0) sense[0] = SENSE_RESPONSE_CODE_CURRENT;
		if (sense_size > 2) {
			sense[2] = p->error_register >> __BSF_CONST(ATA_ERROR_SENSE_KEY);
			if (p->error_register & ATA_ERROR_ILI) sense[2] |= SENSE_FLAGS_ILI;
			if (p->error_register & ATA_ERROR_EOM) sense[2] |= SENSE_FLAGS_EOM;
		}
	} else {
		memset(sense + p->sense_rq.status, 0, sense_size - p->sense_rq.status);
	}
	ATAPI_END_REQUEST(p);
}

__COLD_ATTR__ static void ATAPI_ERROR_RESET_DRIVE(APORT *p)
{
	if (IS_SENSE(p, p->scsirq)) {
		p->scsirq = p->orig_scsirq;
		p->orig_scsirq = NULL;
	}
	p->scsirq->status = -EIO;
	ATAPI_CANCEL(p->scsirq);
}

__COLD_ATTR__ static void ATAPI_CANCEL(SCSIRQ *rq)
{
	ATA_ATTACH_PARAM *ap = rq->adapter;
	APORT *p = ap->port;
	/*__debug_printf("atapi cancel\n");*/
	if (IS_SENSE(p, p->scsirq)) {
		p->scsirq = p->orig_scsirq;
		p->orig_scsirq = NULL;
	}
	if (__unlikely(rq != p->scsirq))
		KERNEL$SUICIDE("ATAPI_CANCEL: CANCELING WRONG REQUEST, %p != %p", rq, p->scsirq);
	if (p->atapi_phase & PHASE_RESET) return;
	ATA_OUT_CMD(p, ATA_CMD_DEVICE_RESET);
	p->atapi_phase |= PHASE_RESET;
	KERNEL$DEL_TIMER(&p->timeout);
	p->pio_timeout = KERNEL$GET_JIFFIES_LO();
	p->timeout.fn = ATAPI_RESET_POLL;
	KERNEL$SET_TIMER(JIFFIES_PER_SECOND, &p->timeout);
}

__COLD_ATTR__ static void ATAPI_RESET_POLL(TIMER *t)
{
	__u8 status;
	APORT *p = GET_STRUCT(t, APORT, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	status = ATA_IN_STATUS(p);
	if (!(status & (ATA_STATUS_BSY | ATA_STATUS_DRQ))) {
		SET_TIMER_NEVER(&p->timeout);
		if (p->scsirq->status >= 0) p->scsirq->status = -EINTR;
		ATAPI_END_REQUEST(p);
		return;
	}
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - p->pio_timeout > ATA$PI_TIMEOUT(p->device[p->atapi_device].attached))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "RESET TIMEOUT, STATUS %02X", status);
		SET_TIMER_NEVER(&p->timeout);
		p->scsirq->status = -EIO;
		ATAPI_END_REQUEST(p);
		return;
	}
	KERNEL$SET_TIMER(JIFFIES_PER_SECOND, &p->timeout);
}

