#include <SPAD/LIBC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/SYNC.H>
#include <SPAD/PCI.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/RANDOM.H>
#include <ARCH/BSF.H>
#include <ARCH/MOV.H>
#include <SPAD/TIMER.H>
#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <VALUES.H>
#include <SPAD/BIO_KRNL.H>

#include <SPAD/ATA.H>
#include "PROMREG.H"

typedef struct {
	unsigned char pdc_flags;
} PDC_BOARD;

#define BOARD_2037x			0
#define BOARD_20319			1
#define BOARD_20619			2
#define BOARD_2057x			3
#define BOARD_40518			4

#define PDC_4_PORTS			1
#define PDC_ADDITIONAL_PATA_PORT	2
#define PDC_GEN_II			4
#define PDC_SATA2_TX4			8

static const PDC_BOARD boards[7] = {
	{ PDC_ADDITIONAL_PATA_PORT },
	{ PDC_4_PORTS },
	{ PDC_4_PORTS },
	{ PDC_ADDITIONAL_PATA_PORT | PDC_GEN_II },
	{ PDC_4_PORTS | PDC_GEN_II | PDC_SATA2_TX4 },
};

static const struct pci_id_s pci_cards[] = {
{ 0x105a, 0x3371, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20371", BOARD_2037x },
{ 0x105a, 0x3373, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20378", BOARD_2037x },
{ 0x105a, 0x3375, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20375", BOARD_2037x },
{ 0x105a, 0x3376, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20376", BOARD_2037x },

{ 0x105a, 0x3570, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20771", BOARD_2057x },
{ 0x105a, 0x3571, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20571", BOARD_2057x },
{ 0x105a, 0x3574, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20579", BOARD_2057x },
{ 0x105a, 0x3577, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC40779", BOARD_2057x },
{ 0x105a, 0x3d73, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC40775", BOARD_2057x },
{ 0x105a, 0x3d75, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20575", BOARD_2057x },

{ 0x105a, 0x3318, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20318", BOARD_20319 },
{ 0x105a, 0x3319, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20319", BOARD_20319 },
{ 0x105a, 0x3515, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC40719", BOARD_20319 },
{ 0x105a, 0x3519, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC40519", BOARD_20319 },

{ 0x105a, 0x3d17, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC40718", BOARD_40518 },
{ 0x105a, 0x3d18, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20518/PDC40518", BOARD_40518 },

{ 0x105a, 0x6629, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PDC20619", BOARD_20619 },
{ 0, }
};

typedef struct __actrl ACTRL;
typedef struct __adevice ADEVICE;

struct __adevice {
	ATA_ATTACH_PARAM *attached;
	unsigned dev_flags;
};

/* 0x1 - 0x00020000  --- IDE_XFER_* */
#define DEV_F_MASKIRQ		0x01000000U
#define DEV_F_IO32		0	/* this chip can't do 32-bit PIO */
#define DEV_F_ATAPI		0x04000000U
#define DEV_F_ATAPI_DMADIR	0x08000000U
#define DEV_F_ATAPI_DRQ_INT	0x10000000U
#define DEV_F_ATAPI_LONG_DRQ	0x20000000U
#define DEV_F_ATAPI_CMD_16	0x40000000U
#define DEV_F_ATAPI_DMA		0x80000000U


struct __aport {
	__u8 *bar;
	unsigned char n;
	unsigned char aport_flags;
	ATARQ *pending_rq[2];
	CMD_PACKET *packet[2];
	__u32 packet_pciaddr[2];
	ASGLIST *sg_list[2];
	TIMER timeout;
	ACTRL *ctrl;

	unsigned pio_n_sect;
	unsigned pio_sg_pos;
	u_jiffies_lo_t pio_timeout;

	__u8 *bar_scr;
	ADEVICE device[2];

	unsigned atapi_length;
	SCSIRQ *scsirq;
	ATARQ atapi_placeholder;
	__u8 error_register;
	SCSIRQ *orig_scsirq;
	DECL_SCSIRQ(6) sense_rq;
	BIODESC sense_desc;
};

#define APORT_DISABLED		0
#define APORT_SATA		0x01
#define APORT_LOCK		0x80

struct __actrl {
	__u8 *bar3;
	AST irq_ast;
	IRQ_HANDLE *irq_ctrl;
	RANDOM_CTX random_ctx;

	unsigned char n_ports;
	unsigned char hotplug_io;

	__u32 ctl_stat_error_mask;
	APORT *pp[4];

	CMD_PACKET *cmd_packets;
	__u32 packet_pciaddr;
	vspace_dmaunlock_t *packet_pciunlock;

	const PDC_BOARD *board;

	int dcall_users;

	char dev_name[__MAX_STR_LEN];
	void *lnte, *dlrq;
	pci_id_t id;
	APORT port[1];
};

#define CMD_PACKET_SPACE	(sizeof(CMD_PACKET) * 2 * a->n_ports)

#define PDC_READL(a, reg)		mmio_inl((a)->bar3 + (reg))
#define PDC_WRITEL(a, reg, val)		mmio_outl((a)->bar3 + (reg), (val))

#define PORT_READ_8(p, reg)		mmio_inb((p)->bar + (reg))
#define PORT_READ_16(p, reg)		mmio_inw((p)->bar + (reg))
#define PORT_READ_32(p, reg)		mmio_inl((p)->bar + (reg))
#define PORT_WRITE_8(p, reg, val)	mmio_outb((p)->bar + (reg), (val))
#define PORT_WRITE_16(p, reg, val)	mmio_outw((p)->bar + (reg), (val))
#define PORT_WRITE_32(p, reg, val)	mmio_outl((p)->bar + (reg), (val))

#define ATA_OUT_DATA16(p, v)		PORT_WRITE_16(p, PORT_DATA, v)
#define ATA_OUT_FEATURES(p, v)		PORT_WRITE_8(p, PORT_FEATURES, v)
#define ATA_OUT_COUNT(p, v)		PORT_WRITE_8(p, PORT_COUNT, v)
#define ATA_OUT_LBA_L(p, v)		PORT_WRITE_8(p, PORT_LBA_L, v)
#define ATA_OUT_LBA_M(p, v)		PORT_WRITE_8(p, PORT_LBA_M, v)
#define ATA_OUT_LBA_H(p, v)		PORT_WRITE_8(p, PORT_LBA_H, v)
#define ATA_OUT_DRIVE(p, v)		PORT_WRITE_8(p, PORT_DRIVE, v)
#define ATA_OUT_CMD(p, v)		PORT_WRITE_8(p, PORT_CMD, v)
#define ATA_IN_DATA16(p)		PORT_READ_16(p, PORT_DATA)
#define ATA_IN_ERROR(p)			PORT_READ_8(p, PORT_ERROR)
#define ATA_IN_FEATURES(p)		PORT_READ_8(p, PORT_FEATURES)
#define ATA_IN_COUNT(p)			PORT_READ_8(p, PORT_COUNT)
#define ATA_IN_LBA_L(p)			PORT_READ_8(p, PORT_LBA_L)
#define ATA_IN_LBA_M(p)			PORT_READ_8(p, PORT_LBA_M)
#define ATA_IN_LBA_H(p)			PORT_READ_8(p, PORT_LBA_H)
#define ATA_IN_DRIVE(p)			PORT_READ_8(p, PORT_DRIVE)
#define ATA_IN_STATUS(p)		PORT_READ_8(p, PORT_STATUS)

#define ATA_OUT_CTRL(p, v)		PORT_WRITE_8(p, PORT_CTRL, v)
#define ATA_IN_ALTSTATUS(p)		PORT_READ_8(p, PORT_ALTSTATUS)

static void ATA_INS(APORT *p, int bit32, void *data, unsigned len)
{
	__u16 *d = data;
	while (len--)
		*d++ = __16CPU2LE(ATA_IN_DATA16(p));
}

static void ATA_OUTS(APORT *p, int bit32, void *data, unsigned len)
{
	__u16 *d = data;
	while (len--)
		ATA_OUT_DATA16(p, __16LE2CPU(*d++));
}

static int AD_POST(ATARQ *rq);
static int PDC_POST(ATARQ *rq);
static int PORT_SWAP(APORT *p);
static void PDC_DIRECT_IRQ(APORT *p);
static int PDC_REPORT_PORT_ERROR(APORT *p, __u32 ctl_stat);
static void PDC_PORT_ERROR(APORT *p, __u32 ctl_stat, int timeout);
static void PDC_RESET_SEQUENCER(APORT *p);

#define APORT_DEVICE_ARRAY_SIZE	2
#define APORT_DEVICES(p)	((p)->aport_flags & APORT_SATA ? 1 : 2)

#define CODE_POST		PDC_POST
#define CODE_QUEUE_DEPTH	2

#define CODE_DISK_NAME(result, len, prefix, a, portn, devn)		\
	_snprintf(result, len, "%s%d@%s", prefix, portn + devn, a->dev_name);

#define CODE_LOCK_WAIT(p)						\
	while (__unlikely((p)->pending_rq[0] != NULL) ||		\
	       __unlikely((p)->pending_rq[1] != NULL))			\
		KERNEL$SLEEP(1);

#define CODE_WAIT_FOR_OTHER_DEVICE(p)	(!((p)->aport_flags & APORT_SATA))
#define CODE_INIT_CANT_RESET(p)		0
#define CODE_INIT_XFER(p, dev)		/* !!! FIXME: reset pio */

#include "FN_ATTCH.I"

#define IS_SEQUENCER_COMMAND(p, rq)	(__likely((rq) != &p->atapi_placeholder) && __likely((rq)->atarq_flags & ATA_PROTOCOL_DMA))

#define current_rq	pending_rq[0]
#define sglist		sg_list[0]

#define direct_port_sequence(port_no)		((port_no) * 4 + 1)
#define packet_port_sequence(port_no, q)	((port_no) * 4 + 2 + (q))

#define CODE_PREPARE_FOR_IRQ(p)						\
do {									\
	PDC_WRITEL((p)->ctrl, PDC_SEQ_COUNTER + direct_port_sequence((p)->n) * 4, 1);									\
	PORT_WRITE_8(p, PORT_CTL_STAT, direct_port_sequence((p)->n));	\
} while (0)

#define CODE_TEST_LOCKED						\
	if (__unlikely(p->aport_flags & APORT_LOCK) ||			\
	    __unlikely(p->pending_rq[0] != NULL) ||			\
	    __unlikely(p->pending_rq[1] != NULL)) return -EAGAIN;

#define POST_ONLY_PIO
#define NO_AD_POLL_SLOW_DISK

#define CODE_END_REQUEST_EXTRA						\
	if (p->pending_rq[1]) {						\
		p->timeout.fn = ATA_TIMEOUT;				\
		KERNEL$SET_TIMER(p->pending_rq[1]->timeout, &p->timeout);\
		PORT_SWAP(p);						\
	}

#include "FN_CORE.I"

#define CODE_SETUP_DMA(p, rq)						\
	PORT_WRITE_32(p, PORT_DESC_TABLE, __32LE2CPU((p)->packet[0]->psg));\
	PORT_WRITE_8(p, PORT_CTL_STAT, direct_port_sequence((p)->n) | (((rq)->atarq_flags & ATARQ_TO_DEVICE) << __BSF_CONST(PORT_CTL_STAT_DMADIR_TO_DEVICE / ATARQ_TO_DEVICE)));

#define CODE_START_DMA(p, rq)						\
	PORT_WRITE_8(p, PORT_CTL_STAT, direct_port_sequence((p)->n) | (((rq)->atarq_flags & ATARQ_TO_DEVICE) << __BSF_CONST(PORT_CTL_STAT_DMADIR_TO_DEVICE / ATARQ_TO_DEVICE)) | PORT_CTL_STAT_DMA_ENABLE);

#define DMASTATUS_ERROR		0x80

#include "FN_ATAPI.I"

#undef current_rq
#undef sglist

static int PDC_POST(ATARQ *rq)
{
	APORT *p = rq->port;
	int r;
	CMD_PACKET *pkt;
	if (!IS_SEQUENCER_COMMAND(p, rq)) {
		return AD_POST(rq);
	}
	__PREFETCHW(p->packet[1]);
	if (__unlikely(rq->atarq_flags & ATARQ_RETRIED)) {
#if __DEBUG >= 1
		if (__unlikely(rq != p->pending_rq[0]))
			KERNEL$SUICIDE("PDC_POST: RETRYING NON-CURRENT REQUEST %p, CURRENT %p, %p", rq, p->pending_rq[0], p->pending_rq[1]);
#endif
		p->timeout.fn = ATA_TIMEOUT;
		KERNEL$SET_TIMER(rq->timeout, &p->timeout);
		/* sequencer will be restarted in PDC_PORT_ERROR */
		return 0;
	}
	/*__debug_printf("command: %s\n", ATA$COMMAND_NAME(rq->fis.command, rq->fis.feature0));*/
	if (__unlikely(p->aport_flags & APORT_LOCK) ||
	    __unlikely(p->pending_rq[1] != NULL)) return -EAGAIN;
	if (p->pending_rq[0] && !IS_SEQUENCER_COMMAND(p, p->pending_rq[0])) return -EAGAIN;
	r = ATA$MAP_DMA(p->sg_list[1], &rq->desc, rq->len, PF_WRITE + ((rq->atarq_flags / ATARQ_TO_DEVICE) & 1), 0);
	if (__unlikely(r < 0)) return r;
	if (__likely(rq->atarq_flags & ATARQ_SET_SIZE)) {
		rq->len = r;
		rq->fis.nsect0 = r >> BIO_SECTOR_SIZE_BITS;
		rq->fis.nsect8 = r >> (BIO_SECTOR_SIZE_BITS + 8);
	} else if (__unlikely(r != rq->len)) {
		ATA$UNMAP_DMA(p->sg_list[1]);
		return -EVSPACEFAULT;
	}
	pkt = p->packet[1];
	pkt->control = (unsigned)(~rq->atarq_flags & ATARQ_TO_DEVICE) / (ATARQ_TO_DEVICE / CONTROL_DMA_READ);
	pkt->nca = __32CPU2LE(0);
	pkt->cmd[0] = CMD_N(1) | CMD_CONDITION_UNCOND | CMD_REG_DRIVE;
	pkt->cmd[1] = rq->fis.device | (rq->device * ATA_DEVICE_DRIVE);
	if (rq->atarq_flags & ATARQ_VALID_48BIT) {
		pkt->cmd[2] = CMD_N(2) | CMD_CONDITION_UNCOND | CMD_REG_COUNT;
		pkt->cmd[3] = rq->fis.nsect8;
		pkt->cmd[4] = rq->fis.nsect0;
		pkt->cmd[5] = CMD_N(2) | CMD_CONDITION_UNCOND | CMD_REG_LBA_L;
		pkt->cmd[6] = rq->fis.lba24;
		pkt->cmd[7] = rq->fis.lba0;
		pkt->cmd[8] = CMD_N(2) | CMD_CONDITION_UNCOND | CMD_REG_LBA_M;
		pkt->cmd[9] = rq->fis.lba32;
		pkt->cmd[10] = rq->fis.lba8;
		pkt->cmd[11] = CMD_N(2) | CMD_CONDITION_UNCOND | CMD_REG_LBA_H;
		pkt->cmd[12] = rq->fis.lba40;
		pkt->cmd[13] = rq->fis.lba16;
		if (__likely(!(rq->atarq_flags & ATARQ_VALID_FEATURE))) {
			pkt->cmd[14] = CMD_N(1) | CMD_CONDITION_EOP | CMD_REG_CMD;
			pkt->cmd[15] = rq->fis.command;
		} else {
			pkt->cmd[14] = CMD_N(2) | CMD_CONDITION_UNCOND | CMD_REG_FEATURES;
			pkt->cmd[15] = rq->fis.feature8;
			pkt->cmd[16] = rq->fis.feature0;
			pkt->cmd[17] = CMD_N(1) | CMD_CONDITION_EOP | CMD_REG_CMD;
			pkt->cmd[18] = rq->fis.command;
		}
	} else {
		pkt->cmd[2] = CMD_N(1) | CMD_CONDITION_UNCOND | CMD_REG_COUNT;
		pkt->cmd[3] = rq->fis.nsect0;
		pkt->cmd[4] = CMD_N(1) | CMD_CONDITION_UNCOND | CMD_REG_LBA_L;
		pkt->cmd[5] = rq->fis.lba0;
		pkt->cmd[6] = CMD_N(1) | CMD_CONDITION_UNCOND | CMD_REG_LBA_M;
		pkt->cmd[7] = rq->fis.lba8;
		pkt->cmd[8] = CMD_N(1) | CMD_CONDITION_UNCOND | CMD_REG_LBA_H;
		pkt->cmd[9] = rq->fis.lba16;
		if (__likely(!(rq->atarq_flags & ATARQ_VALID_FEATURE))) {
			pkt->cmd[10] = CMD_N(1) | CMD_CONDITION_EOP | CMD_REG_CMD;
			pkt->cmd[11] = rq->fis.command;
		} else {
			pkt->cmd[10] = CMD_N(1) | CMD_CONDITION_UNCOND | CMD_REG_FEATURES;
			pkt->cmd[11] = rq->fis.feature0;
			pkt->cmd[12] = CMD_N(1) | CMD_CONDITION_EOP | CMD_REG_CMD;
			pkt->cmd[13] = rq->fis.command;
		}
	}
	p->pending_rq[1] = rq;
	PDC_WRITEL(p->ctrl, PDC_SEQ_COUNTER + pkt->sync_seq * 4, 1);
	PDC_READL(p->ctrl, PDC_SEQ_COUNTER);
	__barrier();
	if (!p->pending_rq[0]) {
		PORT_WRITE_32(p, PORT_PKT_SUBMIT, p->packet_pciaddr[1]);
		p->timeout.fn = ATA_TIMEOUT;
		KERNEL$SET_TIMER(rq->timeout, &p->timeout);
		return PORT_SWAP(p);
	} else {
		PORT_WRITE_8(p, PORT_G_CTL_STAT, PORT_G_CTL_STAT_NCA_PAUSE);
		PORT_READ_32(p, PORT_G_CTL_STAT);
		if (__likely(PORT_READ_32(p, PORT_G_CTL_STAT) & PORT_G_CTL_STAT_COMMAND_CYCLE)) {
			p->packet[0]->nca = __32CPU2LE(p->packet_pciaddr[1]);
			__barrier();
			PORT_WRITE_32(p, PORT_SG_NEXT_COMMAND, p->packet_pciaddr[1]);
			PORT_WRITE_8(p, PORT_G_CTL_STAT, 0);
		} else {
			PORT_WRITE_8(p, PORT_G_CTL_STAT, 0);
			PORT_WRITE_32(p, PORT_PKT_SUBMIT, p->packet_pciaddr[1]);
		}
		return 0;
	}
}

static int PORT_SWAP(APORT *p)
{
	CMD_PACKET *pkt;
	__u32 pci;
	ASGLIST *sgl;
#if __DEBUG >= 1
	if (__unlikely(p->pending_rq[0] != NULL))
		KERNEL$SUICIDE("PORT_SWAP: FIRST REQUEST NOT DONE");
#endif
	p->pending_rq[0] = p->pending_rq[1], p->pending_rq[1] = NULL;
	pkt = p->packet[0], p->packet[0] = p->packet[1], p->packet[1] = pkt;
	pci = p->packet_pciaddr[0], p->packet_pciaddr[0] = p->packet_pciaddr[1], p->packet_pciaddr[1] = pci;
	sgl = p->sg_list[0], p->sg_list[0] = p->sg_list[1], p->sg_list[1] = sgl;
	return 0;
}

static DECL_IRQ_AST(PDC_IRQ, SPL_ATA_SCSI, AST)
{
	ACTRL *a = GET_STRUCT(RQ, ACTRL, irq_ast);
	__u32 irq_mask;
	__u32 hotplug_status;

	hotplug_status = PDC_READL(a, a->hotplug_io);
	if (__unlikely(hotplug_status & 0xff)) {
		PDC_WRITEL(a, a->hotplug_io, hotplug_status);
	}

	irq_mask = PDC_READL(a, PDC_INT_SEQMASK);
	/*__debug_printf("irq: mask %08x, hotplug %08x\n", irq_mask, hotplug_status);*/

	irq_mask &= 0xffff;
	if (__unlikely(!irq_mask)) goto not_mine;
	PDC_WRITEL(a, PDC_INT_SEQMASK, irq_mask);

	do {
		APORT *p;
		unsigned bit = __BSF(irq_mask);
		irq_mask &= -2 << bit;
		p = a->pp[bit / 4];
		if (__unlikely(!p)) {
			bad_bit:
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, a->dev_name, "UNEXPECTED INTERRUPT BIT %u", bit);
			continue;
		}
		if (__likely(bit & 2)) {
			ATARQ *atarq;
			__u32 ctl_stat = PORT_READ_32(p, PORT_G_CTL_STAT);
			if (__unlikely(ctl_stat & a->ctl_stat_error_mask)) {
				PDC_PORT_ERROR(p, ctl_stat, 0);
				continue;
			}
			atarq = p->pending_rq[0];
			if (__unlikely(!atarq) || !IS_SEQUENCER_COMMAND(p, atarq)) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, a->dev_name, "UNEXPECTED INTERRUPT ON PORT %d", p->n);
				continue;
			}
			ATA$UNMAP_DMA(p->sg_list[0]);
			ATA_END_REQUEST(p, 0);
		} else if (__likely(bit & 1)) {
			PDC_DIRECT_IRQ(p);
		} else {
			goto bad_bit;
		}
	} while (irq_mask);

	KERNEL$ADD_RANDOMNESS(&a->random_ctx, NULL, 0);
	not_mine:
	KERNEL$UNMASK_IRQ(a->irq_ctrl);
	RETURN;
}

static void PDC_DIRECT_IRQ(APORT *p)
{
	ATARQ *atarq;
	__u8 status, dmastatus;
	__u32 ctl_stat;
	atarq = p->pending_rq[0];
	if (__unlikely(!atarq) || __unlikely(IS_SEQUENCER_COMMAND(p, atarq)))
	    	return;
	dmastatus = 0;
	ctl_stat = PORT_READ_32(p, PORT_G_CTL_STAT);
	if (__unlikely(ctl_stat & p->ctrl->ctl_stat_error_mask)) {
		if (PDC_REPORT_PORT_ERROR(p, ctl_stat))
			dmastatus = DMASTATUS_ERROR;
		PDC_RESET_SEQUENCER(p);
	}
	status = ATA_IN_ALTSTATUS(p);
	if (__unlikely(status & ATA_STATUS_BSY) && !dmastatus) return;
	status = ATA_IN_STATUS(p);
	if (__likely(atarq == &p->atapi_placeholder)) {
		ATAPI_IRQ(p, dmastatus, status);
	} else {
		ATA_NONDMA_IRQ(p, status);
	}
}

__COLD_ATTR__ static void ATA_TIMEOUT(TIMER *t)
{
	APORT *p = GET_STRUCT(t, APORT, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	SET_TIMER_NEVER(&p->timeout);
	PDC_PORT_ERROR(p, 0, 1);
}

__COLD_ATTR__ static int PDC_REPORT_PORT_ERROR(APORT *p, __u32 ctl_stat)
{
	int r = 0;
	ACTRL *a = p->ctrl;
	ctl_stat &= a->ctl_stat_error_mask;
	if (ctl_stat & PORT_G_CTL_STAT_PH_ERR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: PACKET PCI BUS-MASTER ERROR", p->n), r = 1;
	if (ctl_stat & PORT_G_CTL_STAT_SH_ERR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: DESCRIPTOR PCI BUS-MASTER ERROR", p->n), r = 1;
	if (ctl_stat & PORT_G_CTL_STAT_DH_ERR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: DATA PCI BUS-MASTER ERROR", p->n), r = 1;
	if (ctl_stat & PORT_G_CTL_STAT_OVERRUN && p->pending_rq[0] != &p->atapi_placeholder)		/* this can happen on ATAPI */
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: DMA OVERRUN", p->n), r = 1;
	if (ctl_stat & PORT_G_CTL_STAT_UNDERRUN)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: DMA UNDERRUN", p->n), r = 1;
	if (ctl_stat & PORT_G_CTL_STAT_PCI_ERROR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: PCI ERROR", p->n), r = 1;
	if (ctl_stat & PORT_G_CTL_STAT_PARITY_ERROR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: PARITY ERROR", p->n), r = 1;
	if (ctl_stat & PORT5_G_CTL_STAT_HTO_ERR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: BUS TIMEOUT", p->n), r = 1;
	if (ctl_stat & PORT5_G_CTL_STAT_HB_ERR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: DATA FIS ERROR", p->n), r = 1;
	if (ctl_stat & PORT5_G_CTL_STAT_DMACNT_ERR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, a->dev_name, "PORT %d: DMA COUNT ERROR", p->n), r = 1;
	return r;
}

__COLD_ATTR__ static void PDC_PORT_ERROR(APORT *p, __u32 ctl_stat, int timeout)
{
	ACTRL *a = p->ctrl;
	ATARQ *rq;
	__u32 current_addr;
	__u8 status;
	/*__debug_printf("pdc port error (%x)\n", ctl_stat);*/
	PDC_REPORT_PORT_ERROR(p, ctl_stat);
	current_addr = PORT_READ_32(p, PORT_PKT_SUBMIT);
	PDC_RESET_SEQUENCER(p);
	status = ATA_IN_STATUS(p);

	if ((rq = p->pending_rq[0])) {
		int old_flags = -1;
		if (IS_SEQUENCER_COMMAND(p, rq)) {
			old_flags = p->aport_flags;
			p->aport_flags |= APORT_LOCK;
		}
		if (p->pending_rq[1] && current_addr >= p->packet_pciaddr[1] && current_addr < p->packet_pciaddr[1] + sizeof(CMD_PACKET)) {
			ATA$UNMAP_DMA(p->sg_list[0]);
			ATA_END_REQUEST(p, 0);
		}
		AD_ERROR(p, -1, status, timeout);
		if (old_flags != -1) {
			p->aport_flags = old_flags;
		}
		if ((rq = p->pending_rq[0]) && IS_SEQUENCER_COMMAND(p, rq)) {
			PDC_WRITEL(a, PDC_SEQ_COUNTER + p->packet[0]->sync_seq * 4, 1);
			PDC_READL(p->ctrl, PDC_SEQ_COUNTER);
			__barrier();
			PORT_WRITE_32(p, PORT_PKT_SUBMIT, p->packet_pciaddr[0]);
		}
	}
	ATA_DEQUEUE(p);
}

__COLD_ATTR__ static void PDC_RESET_SEQUENCER(APORT *p)
{
	__u32 val;
	int i;
	for (i = 0; i < 11; i++) {
		val = PORT_READ_32(p, PORT_CTL_STAT);
		if (val & PORT_CTL_STAT_ATA_RESET) break;
		KERNEL$UDELAY(100);
		val |= PORT_CTL_STAT_ATA_RESET;
		val &= ~PORT_CTL_STAT_SEQID;
		PORT_WRITE_32(p, PORT_CTL_STAT, val);
	}
	val &= ~(PORT_CTL_STAT_ATA_RESET | PORT_CTL_STAT_SEQID);
	PORT_WRITE_32(p, PORT_CTL_STAT, val);
	PORT_READ_32(p, PORT_CTL_STAT);
	KERNEL$UDELAY(100);
}

__COLD_ATTR__ static int ACTRL_AUX_CMD(ATA_ATTACH_PARAM *ap, int cmd, ...)
{
	APORT *p;
	int r;
	va_list args;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ACTRL_AUX_CMD AT SPL %08X", KERNEL$SPL);
	p = ap->port;
	va_start(args, cmd);
	switch (cmd) {
		case ATA_AUX_GET_PROTOCOL_MASK: {
			__u16 *ident = va_arg(args, __u16 *);
			r = (1 << ATA_PROTOCOL_NODATA) | (1 << ATA_PROTOCOL_PIO) | (1 << ATA_PROTOCOL_DMA);
			if (p->aport_flags & APORT_SATA && !(p->ctrl->board->pdc_flags & PDC_GEN_II) && ATA$IS_ATAPI(ident))
				r &= ~(1 << ATA_PROTOCOL_DMA);
			break;
		}
		case ATA_AUX_SETUP_PIO_XFER:
		case ATA_AUX_SETUP_DMA_XFER: {
			unsigned avail, avail_unsupp;
			__u16 *ident = va_arg(args, __u16 *);
			unsigned xfer_mask = va_arg(args, unsigned);
			u_jiffies_lo_t timeout = va_arg(args, u_jiffies_lo_t);
			avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2 | IDE_XFER_MASK_UDMA;
			avail_unsupp = 0;
			if (!(p->aport_flags & APORT_SATA)) {
				ATA$CHECK_DEVICE_CABLE(ident, &avail, &avail_unsupp);
				if (PORT_READ_32(p, PORT_CTL_STAT) & PORT_CTL_STAT_CABLE_40)
					ATA$CABLE40_RESTRICT(&avail, &avail_unsupp);
			}
			r = ATA$SETUP_XFER(ap, cmd == ATA_AUX_SETUP_DMA_XFER, ident, p->device[ap->device].dev_flags & IDE_XFER_MASK, xfer_mask, avail, avail_unsupp, ATA$SET_XFER_EMPTY, timeout);
			break;
		}
		case ATA_AUX_SET_ATAPI_FLAGS: {
#include "FN_AFLAG.I"
			break;
		}
		default: {
			r = -ENOOP;
			break;
		}
	}
	va_end(args);
	return r;
}

static int PDC_UNLOAD(void *p, void **dlrq, const char * const argv[]);

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	union {
		MALLOC_REQUEST mrq;
		struct {
			VDESC desc;
			VDMA dma;
		} s;
	} u;
	int r;
	char *chip_name;
	unsigned long flags;
	unsigned char n_ports;
	union {
		char dev_name[__MAX_STR_LEN];
		char pci_id_str[__MAX_STR_LEN];
	} uu;
	ACTRL *a;
	const PDC_BOARD *board;
	int irq;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	int i, q;
	const char *opt, *optend, *str;
	static const struct __param_table params[1] = {
		NULL, 0, 0, 0,
	};
	const char * const *arg = argv;
	int state = 0;
	__u8 *bar3;
	__u32 tmpreg;
	void *vars[1];
	vars[0] = NULL;

	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PROMISE: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}

	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PROMISE: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}

	_snprintf(uu.dev_name, sizeof uu.dev_name, "ATA$PROMISE@" PCI_ID_FORMAT, id);

	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	bar3 = PCI$MAP_MEM_RESOURCE(id, 3, PDC_REGSPACE);
	if (!bar3) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uu.dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", uu.dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(bar3)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", uu.dev_name, strerror(-__PTR_ERR(bar3)));
		r = __PTR_ERR(bar3);
		goto ret1;
	}

	if (__unlikely(flags >= sizeof boards))
		KERNEL$SUICIDE("PROMISE: INVALID BOARD ID %lX", flags);

	board = &boards[flags];
	n_ports = 2;
	if (board->pdc_flags & PDC_4_PORTS) n_ports = 4;
	else if (board->pdc_flags & PDC_ADDITIONAL_PATA_PORT) {
		if (!(mmio_inb(bar3 + PDC_FLASH_CTL + 1) & 0x80))
			n_ports++;
	}

	u.mrq.size = sizeof(ACTRL) + (n_ports - 1) * sizeof(APORT);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PROMISE: CAN'T ALLOCATE MEMORY");
		r = u.mrq.status;
		goto ret2;
	}
	a = u.mrq.ptr;
	memset(a, 0, u.mrq.size);
	a->bar3 = bar3;
	a->n_ports = n_ports;
	a->board = board;
	a->id = id;
	strcpy(a->dev_name, uu.dev_name);

	a->cmd_packets = KERNEL$ALLOC_CONTIG_AREA(CMD_PACKET_SPACE, AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA | AREA_ALIGN, (unsigned long)CMD_PACKET_ALIGN);
	if (__IS_ERR(a->cmd_packets)) {
		r = __PTR_ERR(a->cmd_packets);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PROMISE: CAN'T ALLOCATE PCI MEMORY");
		goto ret25;
	}
	memset(a->cmd_packets, 0, CMD_PACKET_SPACE);
	u.s.desc.ptr = (unsigned long)a->cmd_packets;
	u.s.desc.len = CMD_PACKET_SPACE;
	u.s.desc.vspace = &KERNEL$VIRTUAL;
	u.s.dma.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	a->packet_pciunlock = KERNEL$VIRTUAL.op->vspace_dmalock(&u.s.desc, PF_RW, &u.s.dma);
	if (u.s.dma.len != u.s.desc.len)
		KERNEL$SUICIDE("PROMISE: CAN'T DMALOCK DESCRIPTOR LIST");
	a->packet_pciaddr = u.s.dma.ptr;

	for (i = 0; i < a->n_ports; i++) {
		unsigned char port_no = i;
		if (board->pdc_flags & PDC_SATA2_TX4) {
			const unsigned char remap[4] = { 3, 1, 0, 2 };
			port_no = remap[i];
		}
		a->pp[port_no] = &a->port[i];
		a->port[i].ctrl = a;
		a->port[i].n = port_no;
		a->port[i].bar = bar3 + 0x200 + port_no * 0x80;
		a->port[i].bar_scr = bar3 + 0x400 + port_no * 0x100;
		if (!(i == a->n_ports - 1 && a->n_ports & 1))
			a->port[i].aport_flags |= APORT_SATA;
		INIT_TIMER(&a->port[i].timeout);
		for (q = 0; q < 2; q++) {
			__u32 psg;
			ASGLIST *sg = ATA$ALLOC_SGLIST(a->dev_name, &psg);
			if (__unlikely(__IS_ERR(sg))) {
				if (__PTR_ERR(sg) != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", a->dev_name, strerror(-__PTR_ERR(sg)));
				r = __PTR_ERR(sg);
				goto ret3;
			}
			a->port[i].sg_list[q] = sg;
			a->port[i].packet[q] = &a->cmd_packets[i * 2 + q];
			a->port[i].packet_pciaddr[q] = a->packet_pciaddr + (i * 2 + q) * sizeof(CMD_PACKET);
			a->port[i].packet[q]->sync_seq = packet_port_sequence(port_no, q);
			a->port[i].packet[q]->psg = __32CPU2LE(psg);
		}
	}
	a->ctl_stat_error_mask = PORT_G_CTL_STAT_PH_ERR | PORT_G_CTL_STAT_SH_ERR | PORT_G_CTL_STAT_DH_ERR | PORT_G_CTL_STAT_OVERRUN | PORT_G_CTL_STAT_UNDERRUN | PORT_G_CTL_STAT_DRIVE_ERROR | PORT_G_CTL_STAT_PCI_ERROR;
	if (!(board->pdc_flags & PDC_GEN_II)) {
		a->ctl_stat_error_mask |= PORT_G_CTL_STAT_PARITY_ERROR;
		a->hotplug_io = PDC_SATA_PLUG_CSR;
	} else {
		a->ctl_stat_error_mask |= PORT5_G_CTL_STAT_HTO_ERR | PORT5_G_CTL_STAT_HB_ERR | PORT5_G_CTL_STAT_DMACNT_ERR;
		a->hotplug_io = PDC5_SATA_PLUG_CSR;
	}

	irq = PCI$READ_INTERRUPT_LINE(id);

	_printf("%s: %sSATA CONTROLLER ON PCI: %s\n", a->dev_name, board->pdc_flags & PDC_ADDITIONAL_PATA_PORT ? "ATA/" : "", PCI$ID(uu.pci_id_str, a->id));
	_printf("%s: %s\n", a->dev_name, chip_name);
	_printf("%s: %d PORTS, MEM %"__64_format"X, IRQ %d\n", a->dev_name, a->n_ports, PCI$READ_MEM_RESOURCE(a->id, 3, NULL), irq);

	a->irq_ast.fn = PDC_IRQ;

	if ((r = KERNEL$REQUEST_IRQ(irq, &a->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &a->irq_ast, a->dev_name)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", a->dev_name);
		goto ret3;
	}

	tmpreg = PDC_READL(a, PDC_FLASH_CTL);
	tmpreg |= 0x00002000;
	if (!(board->pdc_flags & PDC_GEN_II)) tmpreg |= 0x00010000;
	PDC_WRITEL(a, PDC_FLASH_CTL, tmpreg);

	tmpreg = PDC_READL(a, a->hotplug_io);
	PDC_WRITEL(a, a->hotplug_io, tmpreg | 0xff);

	tmpreg = PDC_READL(a, a->hotplug_io);
	tmpreg &= ~0x00ff0000;
	PDC_WRITEL(a, a->hotplug_io, tmpreg);

	if (!(board->pdc_flags & PDC_GEN_II)) {
		tmpreg = PDC_READL(a, PDC_TBG_MODE);
		tmpreg &= ~0x00030000;
		tmpreg |= 0x00010000;
		PDC_WRITEL(a, PDC_TBG_MODE, tmpreg);
		PDC_READL(a, PDC_TBG_MODE);
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 100 + 1);

		tmpreg = PDC_READL(a, PDC_SLEW_CTL);
		tmpreg &= ~0x00000fc0;
		tmpreg |= 0x00000900;
		PDC_WRITEL(a, PDC_SLEW_CTL, tmpreg);
	}

	for (i = 0; i < a->n_ports; i++) {
		PDC_RESET_SEQUENCER(&a->port[i]);
	}

	KERNEL$UNMASK_IRQ(a->irq_ctrl);

	r = KERNEL$REGISTER_DEVICE(a->dev_name, "PROMISE.SYS", 0, a, NULL, ATA_DCALL, "ATA,SCSI", NULL, PDC_UNLOAD, &a->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", a->dev_name, strerror(-r));
		goto ret4;
	}
	a->dlrq = KERNEL$TSR_IMAGE();
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s", a->dev_name);
	return 0;

	ret4:
	KERNEL$MASK_IRQ(a->irq_ctrl);
	tmpreg = PDC_READL(a, a->hotplug_io);
	tmpreg |= 0x00ff0000;
	PDC_WRITEL(a, a->hotplug_io, tmpreg);
	tmpreg = PDC_READL(a, a->hotplug_io);
	PDC_WRITEL(a, a->hotplug_io, tmpreg & ~0xff);
	KERNEL$RELEASE_IRQ(a->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &a->irq_ast);
	ret3:
	for (i = 0; i < a->n_ports; i++) for (q = 0; q < 2; q++) if (a->port[i].sg_list[q]) ATA$FREE_SGLIST(a->dev_name, a->port[i].sg_list[q]);
	a->packet_pciunlock(a->packet_pciaddr);
	KERNEL$FREE_CONTIG_AREA(a->cmd_packets, CMD_PACKET_SPACE);
	ret25:
	free(a);
	ret2:
	PCI$UNMAP_MEM_RESOURCE(id, bar3, PDC_REGSPACE);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

__COLD_ATTR__ static int PDC_UNLOAD(void *p, void **dlrq, const char * const argv[])
{
	int r;
	int i, j, q;
	__u32 tmpreg;
	ACTRL *a = p;
	RAISE_SPL(SPL_DEV);
	while (a->dcall_users) KERNEL$SLEEP(1);
	for (i = 0; i < a->n_ports; i++) for (j = 0; j < 2; j++) if (a->port[i].device[j].attached) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS CONTROLLER", a->dev_name);
		return -EBUSY;
	}
	if ((r = KERNEL$DEVICE_UNLOAD(a->lnte, argv))) {
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	LOWER_SPL(SPL_ZERO);
	KERNEL$MASK_IRQ(a->irq_ctrl);
	tmpreg = PDC_READL(a, a->hotplug_io);
	tmpreg |= 0x00ff0000;
	PDC_WRITEL(a, a->hotplug_io, tmpreg);
	tmpreg = PDC_READL(a, a->hotplug_io);
	PDC_WRITEL(a, a->hotplug_io, tmpreg & ~0xff);
	KERNEL$RELEASE_IRQ(a->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &a->irq_ast);
	PCI$UNMAP_MEM_RESOURCE(a->id, a->bar3, PDC_REGSPACE);
	*dlrq = a->dlrq;
	for (i = 0; i < a->n_ports; i++) for (q = 0; q < 2; q++) ATA$FREE_SGLIST(a->dev_name, a->port[i].sg_list[q]);
	a->packet_pciunlock(a->packet_pciaddr);
	KERNEL$FREE_CONTIG_AREA(a->cmd_packets, CMD_PACKET_SPACE);
	PCI$FREE_DEVICE(a->id);
	free(a);
	return 0;
}
