#include <SYS/PARAM.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <STDLIB.H>

#include <SPAD/ATA.H>
#include <SPAD/SCSI.H>

#define ATAPI_TIMEOUT			35
#define SPINUP_TIMEOUT			35
#define ATAPI_INQUIRY_MAX_LENGTH	36

/* Can there be multilun ATAPI devices? I think no, but I am not sure.
   If you know about such device, report it to me. */

typedef struct {
	ATA_ATTACH_PARAM ap;
	SCSI_ATTACH_PARAM *sa;
	u_jiffies_lo_t timeout;
	unsigned at_flags;
	unsigned at_avail_flags;
	unsigned at_avail_xfer;
	int pio_xfer;
	int dma_xfer;
	unsigned ctrl_flags;
	__u16 *ident;
	char model[41];
	char serial[21];
	char revision[9];
} ATAPI;

#define AT_F_TRANSFER_PIO	0x01
#define AT_F_TRANSFER_DMA	0x02
#define AT_F_TRANSFER_MASK	(AT_F_TRANSFER_PIO | AT_F_TRANSFER_DMA)

static int ATAPI_INIT_DEVICE(ATAPI *at);
static int ATAPI_DEQUEUE(ATA_ATTACH_PARAM *ap);
static BIORQ *ATAPI_PROBE_QUEUE(ATA_ATTACH_PARAM *ap);

static int d_call(__DCALL *dcall, void *a, const char *type, int cmd, ...)
{
	int r;
	va_list list;
	va_start(list, cmd);
	r = dcall(a, type, cmd, list);
	va_end(list);
	return r;
}

int ATA$PI_ATTACH(void *a, __DCALL *dcall, SCSI_ATTACH_PARAM *sa, int ports, int devices)
{
	ATAPI *at;
	char *param = NULL;
	int r;
	int fixup, protocol_mask;
	int port = -1, device = -1;
	const char *arg;
	int state;
	const char *opt, *optend, *val, *valend;
	int my_port, my_device;
	union {
		MALLOC_REQUEST mrq;
		ATARQ rq;
	} u;
	static const struct __param_table transfer_params[3] = {
		"PIO", __PARAM_BOOL, AT_F_TRANSFER_MASK, AT_F_TRANSFER_PIO,
		"DMA", __PARAM_BOOL, AT_F_TRANSFER_MASK, AT_F_TRANSFER_DMA,
		NULL, 0, 0, 0,
	};
	void *transfer_vars[3];
	struct __param_table params[4] = {
		"TRANSFER", __PARAM_EXTD_ONE, 0, 0,
		"PORT", __PARAM_INT, 0, 0,
		"DEVICE", __PARAM_INT, 0, 0,
		NULL, 0, 0, 0,
	};
	void *vars[4];
	if (KERNEL$SPL != SPL_X(SPL_ATA_SCSI))
		KERNEL$SUICIDE("ATA$PI_ATTACH AT SPL %08X", KERNEL$SPL);
	if (sa->u.p.version != SCSI_VERSION) {
		sa->msg = "INCORRECT VERSION";
		r = -EINVAL;
		goto ret0;
	}
	if (!sa->probe_timeout) sa->probe_timeout = ATAPI_TIMEOUT * JIFFIES_PER_SECOND;
	if (!sa->timeout) sa->timeout = ATAPI_TIMEOUT * JIFFIES_PER_SECOND;
	u.mrq.size = sizeof(ATAPI);
	SYNC_IO_CANCELABLE(&u.mrq, &KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		sa->msg = "OUT OF MEMORY";
		r = u.mrq.status;
		goto ret0;
	}
	at = u.mrq.ptr;
	memset(at, 0, sizeof(ATAPI));
	u.mrq.size = 512;
	SYNC_IO_CANCELABLE(&u.mrq, &KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		sa->msg = "OUT OF MEMORY";
		r = u.mrq.status;
		goto ret1;
	}
	at->ident = u.mrq.ptr;

	at->timeout = sa->timeout;
	at->at_flags |= AT_F_TRANSFER_DMA;

	vars[0] = transfer_vars;
	params[0].__min = (long)&transfer_params;
	vars[1] = &port;
	params[1].__max = ports;
	vars[2] = &device;
	params[2].__max = devices;
	vars[3] = NULL;
	transfer_vars[0] = &at->at_flags;
	transfer_vars[1] = &at->at_flags;
	transfer_vars[2] = NULL;

	arg = sa->u.p.param;
	state = 0;
	while (__parse_extd_param(&arg, &state, params, vars, &opt, &optend, &val, &valend)) {
		/*__debug_printf("%s/%s/%s/%s.\n", opt, optend, val, valend);*/
		if (opt) {
			r = __accumulate_params(&param, opt, optend, val, valend);
			if (__unlikely(r)) {
				goto ret2;
			}
			continue;
		}
		sa->msg = "BAD SYNTAX";
		r = -EBADSYN;
		goto ret2;
	}

	if (sa->adapter) {	/* not the first call */
		try_another:
		if (++sa->ch_id >= ports * devices) {
			r = -ENODEV;
			goto ret2;
		}
	} else {
		sa->adapter = (void *)1L;
	}

	my_port = sa->ch_id / devices;
	my_device = sa->ch_id % devices;
	sa->lun = LUN_NONE;

	if (port >= 0 && port != my_port) goto try_another;
	if (device >= 0 && device != my_device) goto try_another;

	u.mrq.size = (param ? strlen(param) : 0) + __MAX_STR_LEN;
	SYNC_IO_CANCELABLE(&u.mrq, &KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		sa->msg = "OUT OF MEMORY";
		r = u.mrq.status;
		goto ret2;
	}
	_snprintf(u.mrq.ptr, u.mrq.size, "%s%sPORT=%u,DEVICE=%u", param ? param : "", param && *param ? "," : "", my_port, my_device);

	/* __debug_printf("arg: \"%s\"\n", u.mrq.ptr); */

	memset(&at->ap, 0, sizeof at->ap);
	at->ap.version = ATA_VERSION;
	at->ap.probe_timeout = sa->probe_timeout;
	at->ap.param = u.mrq.ptr;
	at->ap.dev_prefix = sa->dev_prefix;
	at->ap.dev_name = sa->dev_name;
	at->ap.dequeue = ATA$DUMMY_DEQUEUE;
	at->ap.probe_queue = ATA$DUMMY_PROBE_QUEUE;
	at->ap.biosched = sa->biosched;

	r = d_call(dcall, a, "ATAPI", ATA_CMD_GET, &at->ap);
	free(u.mrq.ptr);
	if (r) {
		if (at->ap.msg && (r != -EBUSY || (port >= 0 && device >= 0))) sa->msg = at->ap.msg;
		goto try_another;
	}

	memset(&u.rq.fis, 0, sizeof u.rq.fis);
	u.rq.fis.command = ATA_CMD_IDENTIFY_PACKET_DEVICE;
	u.rq.atarq_flags = ATA_PROTOCOL_PIO | ATARQ_ERROR_ALLOW_ABORT;
	u.rq.multicount = 1;
	ATA$SYNC_RQ(&u.rq, &at->ap, at->ident, 512, at->ap.probe_timeout, 0);
	if (u.rq.status) {
		detach_try_another:
		d_call(dcall, a, "ATAPI", ATA_CMD_FREE, &at->ap);
		goto try_another;
	}
	ATA$BSWAP_IDENTIFY(at->ident);
	if ((at->ident[0] & 0xc000) != 0x8000) {
		goto detach_try_another;	/* not an ATAPI device */
	}
	if (at->ident[0] & 0x0004 || at->ident[2] == 0x37C8 || at->ident[2] == 0x738C) {
		memset(&u.rq.fis, 0, sizeof u.rq.fis);
		u.rq.fis.feature0 = ATA_FEATURE_SET_FEATURES_SPIN_UP;
		u.rq.fis.command = ATA_CMD_SET_FEATURES;
		u.rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_FEATURE;
		ATA$SYNC_RQ(&u.rq, &at->ap, NULL, 0, MAX(at->timeout, SPINUP_TIMEOUT * JIFFIES_PER_SECOND), 0);
		memset(&u.rq.fis, 0, sizeof u.rq.fis);
		u.rq.fis.command = ATA_CMD_IDENTIFY_PACKET_DEVICE;
		u.rq.atarq_flags = ATA_PROTOCOL_PIO;
		u.rq.multicount = 1;
		ATA$SYNC_RQ(&u.rq, &at->ap, at->ident, 512, at->timeout, 0);
		if (u.rq.status) {
			r = u.rq.status;
			goto ret3;
		}
		ATA$BSWAP_IDENTIFY(at->ident);
		if ((at->ident[0] & 0xc000) != 0x8000) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, *at->ap.dev_name, "DEVICE REPORTS WRONG SIGNATURE AFTER SPIN-UP: %04X", at->ident[0]);
			r = -EIO;
			goto ret3;
		}
	}

	/*{
		int i;
		for (i = 0; i < 256; i++) {
			if (!(i & 7)) __debug_printf(" %3d:", i);
			__debug_printf("  %04x", at->ident[i]);
			if ((i & 7) == 7) __debug_printf("\n");
		}
	}*/

	fixup = at->ap.aux_cmd(&at->ap, ATA_AUX_FIXUP_IDENT, at->ident);
	if (fixup < 0) fixup = 0;

	ATA$GET_MODEL(at->ident, &at->model, &at->serial, &at->revision);

	switch (at->ident[0] & 0x0003) {
		case 0x0000:
			break;
		case 0x0001:
			at->ctrl_flags |= ATA_AUX_ATAPI_FLAGS_CMD_16;
			break;
		default:
			reserved_0:
			KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, *at->ap.dev_name, "DEVICE HAS UNKNOWN BITS IN WORD 0: %04X", at->ident[0]);
			r = -EIO;
			goto ret3;
	}
	switch (at->ident[0] & 0x0060) {
		case 0x0000:
			at->ctrl_flags |= ATA_AUX_ATAPI_FLAGS_LONG_DRQ;
			break;
		case 0x0020:
			at->ctrl_flags |= ATA_AUX_ATAPI_FLAGS_DRQ_INT;
			break;
		case 0x0040:
			break;
		case 0x0060:
			goto reserved_0;
	}

	at->at_avail_xfer = ATA$GET_XFER_MODE(at->ident);

	protocol_mask = at->ap.aux_cmd(&at->ap, ATA_AUX_GET_PROTOCOL_MASK, at->ident, at->at_avail_xfer);
	if (protocol_mask < 0) protocol_mask = 0;

	if (at->ident[49] & 0x0100 && protocol_mask & (1 << ATA_PROTOCOL_DMA)) at->at_avail_flags |= AT_F_TRANSFER_DMA;
	if (at->ident[62] & 0x8000 && protocol_mask & (1 << ATA_PROTOCOL_DMA)) {
		at->at_avail_flags |= AT_F_TRANSFER_DMA;
		at->ctrl_flags |= ATA_AUX_ATAPI_FLAGS_DMADIR;
	}

	r = ATAPI_INIT_DEVICE(at);
	if (__unlikely(r)) {
		goto ret3;
	}

	at->sa = sa;
	sa->adapter = &at->ap;
	sa->scsi_flags = SCSI_FLAG_CAN_USE_MODE_10 | SCSI_FLAG_SHORT_INQUIRY;
	at->ap.dequeue = ATAPI_DEQUEUE;
	at->ap.probe_queue = ATAPI_PROBE_QUEUE;

	return 0;

	ret3:
	d_call(dcall, a, "ATAPI", ATA_CMD_FREE, &at->ap);
	ret2:
	free(at->ident);
	ret1:
	free(at);
	ret0:
	free(param);
	return r;
}

static int ATAPI_INIT_DEVICE(ATAPI *at)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATAPI_INIT_DEVICE AT SPL %08X", KERNEL$SPL);

	if (at->at_avail_flags & AT_F_TRANSFER_DMA && at->at_flags & AT_F_TRANSFER_DMA) at->at_flags = (at->at_flags & ~AT_F_TRANSFER_MASK) | AT_F_TRANSFER_DMA;
	else at->at_flags = (at->at_flags & ~AT_F_TRANSFER_MASK) | AT_F_TRANSFER_PIO;

	at->pio_xfer = at->ap.aux_cmd(&at->ap, ATA_AUX_SETUP_PIO_XFER, at->ident, at->at_avail_xfer, at->timeout);
	if (at->pio_xfer < 0)
		at->pio_xfer = 0;
	if (at->at_flags & AT_F_TRANSFER_DMA) {
		at->dma_xfer = at->ap.aux_cmd(&at->ap, ATA_AUX_SETUP_DMA_XFER, at->ident, at->at_avail_xfer, at->timeout);
		if (at->dma_xfer < 0) {
			at->dma_xfer = 0;
			at->at_flags = (at->at_flags & ~AT_F_TRANSFER_MASK) | AT_F_TRANSFER_PIO;
		}
	} else {
		at->dma_xfer = 0;
	}

	return at->ap.aux_cmd(&at->ap, ATA_AUX_SET_ATAPI_FLAGS, at->ctrl_flags | (at->at_flags & AT_F_TRANSFER_DMA ? ATA_PROTOCOL_DMA : ATA_PROTOCOL_PIO));
}

void ATA$PI_DETACH(void *a, __DCALL *dcall, SCSI_ATTACH_PARAM *sa)
{
	ATAPI *at = GET_STRUCT(sa->adapter, ATAPI, ap);
	d_call(dcall, a, "ATAPI", ATA_CMD_FREE, &at->ap);
	free(at->ident);
	free(at);
}

int ATA$PI_SCSI_DCALL(int cmd, va_list params)
{
	SCSI_ATTACH_PARAM *sa = va_arg(params, SCSI_ATTACH_PARAM *);
	switch (cmd) {
		case SCSI_CMD_SET_LUN:
			if (!sa->lun) return 0;
			sa->lun = LUN_NONE;
			return -ENXIO;
		case SCSI_CMD_N_LUNS:
			return 1;
		default:
			return -ENOOP;
	}
}

static int ATAPI_DEQUEUE(ATA_ATTACH_PARAM *ap)
{
	ATAPI *at = GET_STRUCT(ap, ATAPI, ap);
	SCSI_ATTACH_PARAM *sa = at->sa;
	return sa->dequeue(sa);
}

static BIORQ *ATAPI_PROBE_QUEUE(ATA_ATTACH_PARAM *ap)
{
	ATAPI *at = GET_STRUCT(ap, ATAPI, ap);
	SCSI_ATTACH_PARAM *sa = at->sa;
	return sa->probe_queue(sa);
}

u_jiffies_lo_t ATA$PI_TIMEOUT(ATA_ATTACH_PARAM *ap)
{
	ATAPI *at = GET_STRUCT(ap, ATAPI, ap);
	return at->timeout;
}

