#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <VALUES.H>
#include <STDLIB.H>
#include <SYS/PARAM.H>
#include <SPAD/TIMER.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/BSF.H>
#include <ARCH/MOV.H>
#include <SPAD/BIO.H>
#include <SPAD/BIO_KRNL.H>
#include <SPAD/IOCTL.H>

#include <SPAD/ATA.H>

#define DEFAULT_TIMEOUT		35
#define DEFAULT_PROBE_TIMEOUT	35
#define DEFAULT_FLUSH_TIMEOUT	(2 * 60)   /* the standard doesn't specify it */
#define SPINUP_TIMEOUT		35

typedef struct __ad AD;
typedef struct __ad_tag AD_TAG;

struct __ad_tag {
	ATARQ atarq;
	AD *ad;
	__u8 tag_n;
};

struct __ad {
	ATA_ATTACH_PARAM ap;
	unsigned ad_flags;
	unsigned timeout;
	unsigned max_transfer;

	BIOQUEUE *q;
	PARTITIONS *p;

	AD_TAG *tag_freelist;
	AD_TAG implicit_tag;

	unsigned flush_timeout;
	unsigned probe_timeout;
	__sec_t capacity;
	unsigned cyls, heads, secs;
	__u8 multiple;
	__u8 ata_version;

	char ctrl[__MAX_STR_LEN];
	char dev_name[__MAX_STR_LEN];
	void *lnte, *dlrq;
	__u16 *ident;
	AD_TAG *tag_memory;
	__u8 queue_depth;

	unsigned ad_available_flags;
	unsigned ad_available_xfer;
	int pio_xfer;
	int dma_xfer;
	unsigned physical_sector_size;
	unsigned short cache_size;
	char model[41];
	char serial[21];
	char revision[9];
};

/* ad_flags */
#define AD_F_TRANSFER_PIO	0x00000001
#define AD_F_TRANSFER_MPIO	0x00000002
#define AD_F_TRANSFER_DMA	0x00000004
#define AD_F_TRANSFER_MASK	(AD_F_TRANSFER_PIO | AD_F_TRANSFER_MPIO | AD_F_TRANSFER_DMA)
#define AD_F_ADDRESSING_CHS	0x00000100
#define AD_F_ADDRESSING_LBA	0x00000200
#define AD_F_ADDRESSING_LBA48	0x00000400
#define AD_F_ADDRESSING_MASK	(AD_F_ADDRESSING_CHS | AD_F_ADDRESSING_LBA | AD_F_ADDRESSING_LBA48)
#define AD_F_WCACHE		0x00001000
#define AD_F_RCACHE		0x00002000

#define AD_F_HAS_FLUSH_EXT	0x00010000
#define AD_F_DO_FLUSH		0x00020000

#define AD_F_SKIPID		0x01000000

#define AD_F_SOMETHING_TO_FLUSH	0x80000000U

static int AD_UNLOAD(void *p, void **release, const char * const argv[]);
static void AD_INIT_ROOT(HANDLE *h, void *data);
static void AD_DETACH(AD *ad);
int AD_PARSE_IDENTIFY(AD *ad);
static void AD_INIT_DISK(AD *ad);
static void AD_FLUSH_CACHE(AD *ad);
static int AD_IOCTL(IOCTLRQ *rq, PARTITION *pa, IORQ *rq_to_queue);
static IO_STUB AD_REQUEST;
static int AD_POST_REQUEST(AD_TAG *tag, BIORQ *rq);
static AD_TAG *AD_CONTINUE_REQUEST(AD_TAG *old_tag);
static int AD_DEQUEUE(ATA_ATTACH_PARAM *ap);
static BIORQ *AD_PROBE_QUEUE(ATA_ATTACH_PARAM *ap);
static void AD_REQUEST_DONE(ATARQ *arq);
static void AD_ERROR(AD_TAG *tag, int err, __sec_t lba);

static const HANDLE_OPERATIONS AD_OPERATIONS = {
	SPL_X(SPL_ATA_SCSI),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,				/* clone */
	BIO$LOOKUP_PARTITION,		/* lookup */
	NULL,				/* create */
	NULL,				/* delete */
	NULL,				/* rename */
	NULL,				/* lookup_io */
	BIO$INSTANTIATE_PARTITION,	/* instantiate */
	NULL,				/* leave */
	BIO$DETACH,			/* detach */
	NULL,				/* close */
	BIO$READ,
	BIO$WRITE,
	BIO$AREAD,
	BIO$AWRITE,
	BIO$IOCTL,
	AD_REQUEST,
	KERNEL$NO_OPERATION,
};

static void TAG_CTOR(AD *ad, AD_TAG *tag, int n)
{
	memset(&tag->atarq.fis, 0, sizeof tag->atarq.fis);
	tag->atarq.port = ad->ap.port;
	tag->atarq.done = AD_REQUEST_DONE;
	tag->atarq.device = ad->ap.device;
	tag->atarq.multicount = ad->ad_flags & AD_F_TRANSFER_PIO ? 1 : ad->multiple;
	tag->ad = ad;
	tag->tag_n = n;
}

#define TAG_GET_NEXT_FREE(tag)		((AD_TAG *)((tag)->atarq.desc))
#define TAG_SET_NEXT_FREE(tag, val)	((tag)->atarq.desc = (void *)(val))

__COLD_ATTR__ static void AD_INIT_ROOT(HANDLE *h, void *data)
{
	AD *ad = data;
	BIO$ROOT_PARTITION(h, ad->p);
	h->flags = 0;
	h->flags2 = 0;
	h->op = &AD_OPERATIONS;
	BIO$SET_CACHE(ad->p, h);
}

__COLD_ATTR__ static int AD_TRY_CONTROLLER(AD *ad, char *param)
{
	int r;
	ATARQ rq;
	memset(&ad->ap, 0, sizeof(ATA_ATTACH_PARAM));
	ad->ap.version = ATA_VERSION;
	ad->ap.param = param;
	ad->ap.dev_prefix = "BIO$AD";
	ad->ap.dev_name = &ad->dev_name;
	ad->ap.dequeue = ATA$DUMMY_DEQUEUE;
	ad->ap.probe_queue = ATA$DUMMY_PROBE_QUEUE;
	ad->ap.biosched = BIOQUE$IOSCHED(ad->q);
	try_another:
	ad->ap.probe_timeout = ad->probe_timeout;
	if ((r = KERNEL$DCALL(ad->ctrl, "ATA", ATA_CMD_GET, &ad->ap))) {
		if (ad->ap.msg) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: %s", ad->ap.msg);
		else if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: CAN'T ATTACH TO HOST CONTROLLER DRIVER %s: %s", ad->ctrl, strerror(-r));
		return r;
	}

	if (ad->ad_flags & AD_F_SKIPID) {
		memset(ad->ident, 0, 512);
		goto skip_identify;
	}

	memset(&rq.fis, 0, sizeof rq.fis);
	rq.fis.device = 0xa0;
	rq.fis.command = ATA_CMD_IDENTIFY_DEVICE;
	rq.atarq_flags = ATA_PROTOCOL_PIO | ATARQ_ERROR_ALLOW_ABORT;
	rq.multicount = 1;
	ATA$SYNC_RQ(&rq, &ad->ap, ad->ident, 512, ad->ap.probe_timeout, 0);
	if (rq.status) {
		detach_try_another:
		AD_DETACH(ad);
		goto try_another;
	}
	ATA$BSWAP_IDENTIFY(ad->ident);
	if (ad->ident[0] & 0x8000) {
		goto detach_try_another;	/* not a disk */
	}

	skip_identify:

	return 0;
}

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	union {
		MALLOC_REQUEST mrq;
		LOGICAL_LIST_REQUEST llr;
	} u;
	const char * const *arg;
	int state;
	const char *opt, *optend, *val;
	char *ctrl = NULL;
	char *param = NULL;
	int i, r;
	AD *ad;
	AD_TAG *last_tag;
	unsigned pio_xfer, dma_xfer;
	static const struct __param_table skipid_params[4] = {
		"CYLS", __PARAM_UNSIGNED_INT, 1, 65536+1,
		"HEADS", __PARAM_UNSIGNED_INT, 1, 16+1,
		"SECS", __PARAM_UNSIGNED_INT, 1, 255+1,
		NULL, 0, 0, 0,
	};
	void *skipid_vars[4];
	static const struct __param_table transfer_params[4] = {
		"PIO", __PARAM_BOOL, AD_F_TRANSFER_MASK, AD_F_TRANSFER_PIO,
		"MPIO", __PARAM_BOOL, AD_F_TRANSFER_MASK, AD_F_TRANSFER_MPIO,
		"DMA", __PARAM_BOOL, AD_F_TRANSFER_MASK, AD_F_TRANSFER_DMA,
		NULL, 0, 0, 0,
	};
	void *transfer_vars[4];
	static const struct __param_table addressing_params[4] = {
		"CHS", __PARAM_BOOL, AD_F_ADDRESSING_MASK, AD_F_ADDRESSING_CHS,
		"LBA", __PARAM_BOOL, AD_F_ADDRESSING_MASK, AD_F_ADDRESSING_LBA,
		"LBA48", __PARAM_BOOL, AD_F_ADDRESSING_MASK, AD_F_ADDRESSING_LBA48,
		NULL, 0, 0, 0,
	};
	void *addressing_vars[4];
	static const struct __param_table params[14] = {
		"CTRL", __PARAM_STRING, 1, __MAX_STR_LEN,
		"TIMEOUT", __PARAM_UNSIGNED_INT, 1, MAXINT / JIFFIES_PER_SECOND,
		"FLUSH_TIMEOUT", __PARAM_UNSIGNED_INT, 1, MAXINT / JIFFIES_PER_SECOND,
		"PROBE_TIMEOUT", __PARAM_UNSIGNED_INT, 1, MAXINT / JIFFIES_PER_SECOND,
		"SKIPID", __PARAM_BOOL_X, AD_F_SKIPID, AD_F_SKIPID,
		"SKIPID", __PARAM_EXTD, (long)&skipid_params, 0,
		"TRANSFER", __PARAM_EXTD_ONE, (long)&transfer_params, 0,
		"ADDRESSING", __PARAM_EXTD_ONE, (long)&addressing_params, 0,
		"RCACHE", __PARAM_BOOL, AD_F_RCACHE, AD_F_RCACHE,
		"NORCACHE", __PARAM_BOOL, AD_F_RCACHE, 0,
		"WCACHE", __PARAM_BOOL, AD_F_WCACHE, AD_F_WCACHE,
		"NOWCACHE", __PARAM_BOOL, AD_F_WCACHE, 0,
		"MAX_TRANSFER", __PARAM_UNSIGNED_INT, 512, 65536 * 512,
		NULL, 0, 0, 0,
	};
	void *vars[14];

	u.mrq.size = sizeof(AD);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret0;
	}
	ad = u.mrq.ptr;
	memset(ad, 0, sizeof(AD));

	u.mrq.size = 512;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret1;
	}
	ad->ident = u.mrq.ptr;

	ad->timeout = DEFAULT_TIMEOUT;
	ad->flush_timeout = DEFAULT_FLUSH_TIMEOUT;
	ad->probe_timeout = DEFAULT_PROBE_TIMEOUT;
	ad->ad_flags |= AD_F_TRANSFER_DMA | AD_F_ADDRESSING_LBA48 | AD_F_WCACHE | AD_F_RCACHE;

	vars[0] = &ctrl;
	vars[1] = &ad->timeout;
	vars[2] = &ad->flush_timeout;
	vars[3] = &ad->probe_timeout;
	vars[4] = &ad->ad_flags;
	vars[5] = &skipid_vars;
	vars[6] = &transfer_vars;
	vars[7] = &addressing_vars;
	vars[8] = &ad->ad_flags;
	vars[9] = &ad->ad_flags;
	vars[10] = &ad->ad_flags;
	vars[11] = &ad->ad_flags;
	vars[12] = &ad->max_transfer;
	vars[13] = NULL;

	skipid_vars[0] = &ad->cyls;
	skipid_vars[1] = &ad->heads;
	skipid_vars[2] = &ad->secs;
	skipid_vars[3] = NULL;

	transfer_vars[0] = &ad->ad_flags;
	transfer_vars[1] = &ad->ad_flags;
	transfer_vars[2] = &ad->ad_flags;
	transfer_vars[3] = NULL;

	addressing_vars[0] = &ad->ad_flags;
	addressing_vars[1] = &ad->ad_flags;
	addressing_vars[2] = &ad->ad_flags;
	addressing_vars[3] = NULL;

	arg = argv;
	state = 0;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &val)) {
		if (opt) {
			r = __accumulate_params(&param, opt, optend, val, val ? strchr(val, 0) : NULL);
			if (r) {
				if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: %s", strerror(-r));
				goto ret2;
			}
			continue;
		}
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret2;
	}
	if (ad->max_transfer & 511) goto bads;
	ad->max_transfer >>= 9;
	ad->timeout *= JIFFIES_PER_SECOND;
	ad->flush_timeout *= JIFFIES_PER_SECOND;
	ad->probe_timeout *= JIFFIES_PER_SECOND;
	if (ad->ad_flags & AD_F_SKIPID && (!ad->cyls || !ad->heads || !ad->secs)) goto bads;

	if ((r = BIOQUE$ALLOC_QUEUE(&ad->q))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: CAN'T ALLOC QUEUE: %s", strerror(-r));
		goto ret2;
	}

	if (ctrl) {
		strcpy(ad->ctrl, ctrl);
		if (__check_logical_name(ad->ctrl, 1)) goto bads;
		r = AD_TRY_CONTROLLER(ad, param);
		if (r) goto ret3;
	} else {
		u.llr.prefix = "ATA$";
		SYNC_IO_CANCELABLE(&u.llr, KERNEL$LIST_LOGICALS);
		if (u.llr.status < 0) {
			if (u.llr.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: CAN'T LIST LOGICAL NAMES: %s", strerror(-u.llr.status));
			r = u.llr.status;
			goto ret3;
		}
		for (i = 0; i < u.llr.n_entries; i++) {
			strlcpy(ad->ctrl, u.llr.entries[i]->name, __MAX_STR_LEN);
			r = AD_TRY_CONTROLLER(ad, param);
			if (!r) {
				KERNEL$FREE_LOGICAL_LIST(&u.llr);
				goto got_it;
			}
		}
		KERNEL$FREE_LOGICAL_LIST(&u.llr);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: NO DISK FOUND");
		r = -ENODEV;
		goto ret3;
	}

	got_it:
	if ((r = AD_PARSE_IDENTIFY(ad))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: DISK ERROR");
		goto ret4;
	}

	AD_INIT_DISK(ad);

	_printf("%s: ATA DISK%s%s%s%s%s%s\n", ad->dev_name, *ad->model || *ad->serial || *ad->revision ? " " : "", ad->model, *ad->model && *ad->serial ? "," : "", ad->serial, *ad->revision && (*ad->model || *ad->serial) ? "," : "", ad->revision);
	_printf("%s: ", ad->dev_name);
	if (ad->ata_version) {
		_printf("ATA %u, ", ad->ata_version);
	}
	if (ad->capacity < 2097152)
		_printf("%ld MiB", (long)(ad->capacity >> 11));
	else if (ad->capacity < (__u64)2097152 * 1024)
		_printf("%ld.%ld GiB", (long)(ad->capacity >> 21), ((long)ad->capacity & 2097151) / 209716);
	else
		_printf("%ld.%ld TiB", (long)(ad->capacity >> 31), (long)((ad->capacity >> 10) & 2097151) / 209716);
	
	if (ad->cache_size)
		_printf(", %d KiB CACHE", ad->cache_size / 2);

	_printf("\n");
	_printf("%s: TRANSFER %s, ADDRESSING %s", ad->dev_name, ad->ad_flags & AD_F_TRANSFER_PIO ? "PIO" : ad->ad_flags & AD_F_TRANSFER_MPIO ? "MPIO" : "DMA", ad->ad_flags & AD_F_ADDRESSING_CHS ? "CHS" : ad->ad_flags & AD_F_ADDRESSING_LBA ? "LBA" : "LBA48");
	if (ad->ad_flags & AD_F_WCACHE) _printf(", WRITE CACHE");
	if (ad->ad_flags & AD_F_RCACHE) _printf(", READ CACHE");
	_printf("\n");

	pio_xfer = ad->pio_xfer & IDE_XFER_MASK_PIO & ~IDE_XFER_PIO_NONE;
	dma_xfer = ad->dma_xfer & (IDE_XFER_MASK_SDMA | IDE_XFER_MASK_WDMA | IDE_XFER_MASK_UDMA);

	if (pio_xfer || dma_xfer) {
		_printf("%s: ", ad->dev_name);
		if (pio_xfer) {
			_printf("PIO%u", ATA$XFER_GET_NUMBER(pio_xfer, IDE_XFER_MASK_PIO & ~IDE_XFER_PIO_NONE));
			if (dma_xfer)
				_printf(", ");
		}
		if (dma_xfer & IDE_XFER_MASK_SDMA)
			_printf("SDMA%u", ATA$XFER_GET_NUMBER(dma_xfer, IDE_XFER_MASK_SDMA));
		if (dma_xfer & IDE_XFER_MASK_WDMA)
			_printf("WDMA%u", ATA$XFER_GET_NUMBER(dma_xfer, IDE_XFER_MASK_WDMA));
		if (dma_xfer & IDE_XFER_MASK_UDMA)
			_printf("UDMA%u", ATA$XFER_GET_NUMBER(dma_xfer, IDE_XFER_MASK_UDMA));
		_printf("\n");
	}

	r = BIO$ALLOC_PARTITIONS(&ad->p, ad, SPL_X(SPL_ATA_SCSI), ad->capacity, AD_IOCTL, ad->dev_name);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC PARTITIONS: %s", ad->dev_name, strerror(-r));
		goto ret4;
	}

	u.mrq.size = (ad->queue_depth - 1) * sizeof(AD_TAG);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "AD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret5;
	}
	ad->tag_memory = u.mrq.ptr;
	last_tag = NULL;
	for (i = ad->queue_depth - 1; i >= 1; i--) {
		AD_TAG *tag = &ad->tag_memory[i - 1];
		TAG_CTOR(ad, tag, i);
		TAG_SET_NEXT_FREE(tag, last_tag);
		last_tag = tag;
	}
	TAG_CTOR(ad, &ad->implicit_tag, 0);
	TAG_SET_NEXT_FREE(&ad->implicit_tag, last_tag);
	ad->tag_freelist = &ad->implicit_tag;

	RAISE_SPL(SPL_ATA_SCSI);
	ad->ap.dequeue = AD_DEQUEUE;
	ad->ap.probe_queue = AD_PROBE_QUEUE;
	LOWER_SPL(SPL_ZERO);

	r = KERNEL$REGISTER_DEVICE(ad->dev_name, "AD.SYS", 0, ad, AD_INIT_ROOT, NULL, NULL, NULL, AD_UNLOAD, &ad->lnte, ad->ctrl, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", ad->dev_name, strerror(-r));
		goto ret6;
	}
	ad->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), ad->dev_name);
	free(param);
	return 0;

	ret6:
	free(ad->tag_memory);
	ret5:
	BIO$FREE_PARTITIONS(ad->p);
	ret4:
	AD_DETACH(ad);
	ret3:
	BIOQUE$FREE_QUEUE(ad->q);
	ret2:
	free(param);
	free(ad->ident);
	ret1:
	free(ad);
	ret0:
	return r;
}

__COLD_ATTR__ static int AD_UNLOAD(void *p, void **release, const char * const argv[])
{
	AD *ad = p;
	AD_TAG *tag;
	int r, i;
	BIO$DISABLE_CACHE(ad->p);
	if ((r = KERNEL$DEVICE_UNLOAD(ad->lnte, argv))) return r;
	RAISE_SPL(SPL_ATA_SCSI);
	while (!BIOQUE$QUEUE_EMPTY(ad->q)) {
		slp:
		KERNEL$SLEEP(1);
	}
	i = 0;
	tag = ad->tag_freelist;
	while (tag) tag = TAG_GET_NEXT_FREE(tag), i++;
	if (i != ad->queue_depth) goto slp;
	LOWER_SPL(SPL_ZERO);
	AD_FLUSH_CACHE(ad);
	free(ad->tag_memory);
	BIO$FREE_PARTITIONS(ad->p);
	AD_DETACH(ad);
	BIOQUE$FREE_QUEUE(ad->q);
	free(ad->ident);
	*release = ad->dlrq;
	free(ad);
	return 0;
}

__COLD_ATTR__ static void AD_DETACH(AD *ad)
{
	int r;
	if ((r = KERNEL$DCALL(ad->ctrl, "ATA", ATA_CMD_FREE, &ad->ap)))
		KERNEL$SUICIDE("AD: CAN'T UNREGISTER AT CONTROLLER %s: %s", ad->ctrl, strerror(-r));
}

__COLD_ATTR__ int AD_PARSE_IDENTIFY(AD *ad)
{
	ATARQ rq;
	int fvalid, evalid;
	int fixup, protocol_mask;

	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("AD_PARSE_IDENTIFY AT SPL %08X", KERNEL$SPL);

	ad->ad_available_flags = AD_F_TRANSFER_PIO | AD_F_ADDRESSING_CHS;
	ad->physical_sector_size = 512;
	if (ad->ad_flags & AD_F_SKIPID) {
		ad->capacity = ad->secs * ad->heads * ad->cyls;
		goto skipid;
	}
	if (ad->ident[0] & 0x0004 || ad->ident[2] == 0x37C8 || ad->ident[2] == 0x738C) {
		/* Send spin-up command */
		memset(&rq.fis, 0, sizeof rq.fis);
		rq.fis.feature0 = ATA_FEATURE_SET_FEATURES_SPIN_UP;
		rq.fis.command = ATA_CMD_SET_FEATURES;
		rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_FEATURE;
		ATA$SYNC_RQ(&rq, &ad->ap, NULL, 0, MAX(ad->timeout, SPINUP_TIMEOUT * JIFFIES_PER_SECOND), 0);
		memset(&rq.fis, 0, sizeof rq.fis);
		rq.fis.command = ATA_CMD_IDENTIFY_DEVICE;
		rq.atarq_flags = ATA_PROTOCOL_PIO;
		rq.multicount = 1;
		ATA$SYNC_RQ(&rq, &ad->ap, ad->ident, 512, ad->timeout, 1);
		if (rq.status) {
			return rq.status;
		}
		ATA$BSWAP_IDENTIFY(ad->ident);
		if (ad->ident[0] & 0x8000) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ad->dev_name, "DISK REPORTS WRONG SIGNATURE AFTER SPIN-UP: %04X", ad->ident[0]);
			return -EIO;
		}
	}

	/*{
		int i;
		for (i = 0; i < 256; i++) {
			if (!(i & 7)) __debug_printf(" %3d:", i);
			__debug_printf("  %04x", ad->ident[i]);
			if ((i & 7) == 7) __debug_printf("\n");
		}
	}*/

	RAISE_SPL(SPL_ATA_SCSI);
	fixup = ad->ap.aux_cmd(&ad->ap, ATA_AUX_FIXUP_IDENT, ad->ident);
	LOWER_SPL(SPL_ZERO);
	if (fixup < 0) fixup = 0;

	ad->ata_version = 0;
	if (ad->ident[80] & 0x7ffe && ad->ident[80] != 0xffff) {
		ad->ata_version = __BSR(ad->ident[80] & 0x7ffe);
	}

	fvalid = (ad->ident[83] & 0xC000) == 0x4000;
	evalid = (ad->ident[87] & 0xC000) == 0x4000;

	ATA$GET_MODEL(ad->ident, &ad->model, &ad->serial, &ad->revision);

	/* DMA */

	ad->ad_available_xfer = ATA$GET_XFER_MODE(ad->ident);

	RAISE_SPL(SPL_ATA_SCSI);
	protocol_mask = ad->ap.aux_cmd(&ad->ap, ATA_AUX_GET_PROTOCOL_MASK, ad->ident, ad->ad_available_xfer);
	if (protocol_mask < 0) protocol_mask = 0;
	LOWER_SPL(SPL_ZERO);
	if (ad->ident[49] & 0x0100 && protocol_mask & (1 << ATA_PROTOCOL_DMA)) {
		ad->ad_available_flags |= AD_F_TRANSFER_DMA;
	}

	/* Multiple sector count */

	ad->multiple = 1;
	if ((ad->ident[47] & 0xff) > 1) {
		ad->multiple = ad->ident[47] & 0xff;
		ad->ad_available_flags |= AD_F_TRANSFER_MPIO;
	}

	/* Capacity */
	
	if (ad->ident[53] & 1) {
		ad->cyls = ad->ident[54];
		ad->heads = ad->ident[55];
		ad->secs = ad->ident[56];
	} else {
		ad->cyls = ad->ident[1];
		ad->heads = ad->ident[3];
		ad->secs = ad->ident[6];
	}
	if (!ad->cyls || !ad->heads || !ad->secs || ad->cyls > 65536 || ad->heads > 16 || ad->secs > 255) {
		if (ad->cyls || ad->heads || ad->secs) {
			KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, ad->dev_name, "DISK REPORTS BOGUS CHS GEOMETRY: %u/%u/%u", ad->cyls, ad->heads, ad->secs);
			ad->cyls = 0; ad->heads = 0; ad->secs = 0;
		}
		ad->ad_available_flags &= ~AD_F_ADDRESSING_CHS;
	}
	if (ad->ident[49] & 0x0200) ad->ad_available_flags |= AD_F_ADDRESSING_LBA;
	if (fvalid && ad->ident[83] & 0x0400) {
		ad->ad_available_flags |= AD_F_ADDRESSING_LBA48;
		if (ad->ident[83] & 0x2000) ad->ad_flags |= AD_F_HAS_FLUSH_EXT;
	}

	if (ad->ad_available_flags & AD_F_ADDRESSING_LBA48) {
		__u64 capa = (__u64)ad->ident[100] + ((__u64)ad->ident[101] << 16) + ((__u64)ad->ident[102] << 32) + ((__u64)ad->ident[103] << 48);
		if (!capa || capa == (__s64)-1) {
			KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, ad->dev_name, "DISK DOESN'T HAVE LBA48 CAPACITY");
			ad->ad_available_flags &= ~AD_F_ADDRESSING_LBA48;
			goto skip_lba48;
		}
		if (capa > (__u64)1 << 48) {
			KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, ad->dev_name, "DISK REPORTS BOGUS LBA48 CAPACITY %016"__64_format"X SECTORS", capa);
			capa = (__u64)1 << 48;
		}
		if ((__sec_t)capa != capa || (__sec_t)capa < 0) {
			KERNEL$SYSLOG(__SYSLOG_SW_INCOMPATIBILITY, ad->dev_name, "DISK REPORTS LBA48 CAPACITY %016"__64_format"X SECTORS BUT SYSTEM WAS COMPILED WITH ONLY 31-BIT SECTOR NUMBERS", capa);
			return -ERANGE;
		}
		ad->capacity = capa;
		goto done_capacity;
	}
	skip_lba48:
	if (ad->ad_available_flags & AD_F_ADDRESSING_LBA) {
		__u32 capa = ad->ident[60] + ad->ident[61] * 65536;
		if (!capa || capa == (__s32)-1) {
			KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, ad->dev_name, "DISK DOESN'T HAVE LBA CAPACITY");
			ad->ad_available_flags &= ~AD_F_ADDRESSING_LBA;
			goto skip_lba;
		}
		ad->capacity = capa;
		goto done_capacity;
	}
	skip_lba:
	if (ad->ad_available_flags & AD_F_ADDRESSING_CHS) {
		ad->capacity = ad->cyls * ad->heads * ad->secs;
		goto done_capacity;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, ad->dev_name, "DISK DOESN'T HAVE ANY ADDRESSING MODE");
	return -ENXIO;

	done_capacity:
	if (ad->capacity > 1 << 28) {
		ad->ad_available_flags &= ~AD_F_ADDRESSING_LBA;
	}
	if (ad->capacity != ad->cyls * ad->heads * ad->secs) {
		ad->ad_available_flags &= ~AD_F_ADDRESSING_CHS;
	}

	/* Cache */

	if (ad->ident[21] > 0) ad->cache_size = ad->ident[21];
	if (fvalid) {
		if (ad->ident[82] & 0x0040) ad->ad_available_flags |= AD_F_RCACHE;
		if (ad->ata_version < 4) {
			goto skip_wcache_test;
		}
		if (ad->ident[82] & 0x0020) {
			memset(&rq.fis, 0, sizeof rq.fis);
			rq.fis.command = ATA_CMD_FLUSH_CACHE;
			rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_ERROR_ALLOW_ABORT;
			ATA$SYNC_RQ(&rq, &ad->ap, NULL, 0, ad->flush_timeout, 0);
			if (rq.status != -EOPNOTSUPP)
				ad->ad_available_flags |= AD_F_WCACHE;
		}
		skip_wcache_test:;
	}

	if ((ad->ident[106] & 0xC000) == 0x4000) {
		if (ad->ident[106] & 0x1000) {
			__u64 logical_sector_size = (__u64)(ad->ident[117] + (ad->ident[118] << 16)) << 1;
			if (logical_sector_size != 512) {
				KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, ad->dev_name, "LOGICAL SECTOR SIZE %"__64_format"d NOT SUPPORTED. ONLY 512 BYTES IS ACCEPTABLE", logical_sector_size);
				return -ENXIO;
			}
		}
		ad->physical_sector_size = 512 << (ad->ident[106] & 0xf);
	}

/* Some Western Digital disks have cache consistency bugs when using 256-sector
   transfers (some part of firmware probably treats 256 as zero).
   It is unknown which disks have this bug, so I blacklisted all */
	if (!memcmp(ad->model, "WDC", 3) && !(ad->ad_available_flags & AD_F_ADDRESSING_LBA48)) {
		ad->max_transfer = 255;
	}

/* Transcend flash cards always cache writes, even if cache is disabled. */
	if (!memcmp(ad->model, "TS", 2)) {
		ad->ad_flags |= AD_F_DO_FLUSH;
	}

	skipid:

	if (ad->ad_flags & AD_F_WCACHE);
		ad->ad_flags |= AD_F_DO_FLUSH;

	return 0;
}

__COLD_ATTR__ static void AD_INIT_DISK(AD *ad)
{
	ATARQ rq;
	int fvalid;
	unsigned maxt;

	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("AD_INIT_DISK AT SPL %08X", KERNEL$SPL);

	/* Choose transfer */
	if (ad->ad_available_flags & AD_F_TRANSFER_DMA && ad->ad_flags & AD_F_TRANSFER_DMA) ad->ad_flags = (ad->ad_flags & ~AD_F_TRANSFER_MASK) | AD_F_TRANSFER_DMA;
	else if (ad->ad_available_flags & AD_F_TRANSFER_MPIO && ad->ad_flags & (AD_F_TRANSFER_MPIO | AD_F_TRANSFER_DMA)) ad->ad_flags = (ad->ad_flags & ~AD_F_TRANSFER_MASK) | AD_F_TRANSFER_MPIO;
	else ad->ad_flags = (ad->ad_flags & ~AD_F_TRANSFER_MASK) | AD_F_TRANSFER_PIO;

	/* Choose addressing */
	if (ad->ad_available_flags & AD_F_ADDRESSING_LBA48 && ad->ad_flags & AD_F_ADDRESSING_LBA48) ad->ad_flags = (ad->ad_flags & ~AD_F_ADDRESSING_MASK) | AD_F_ADDRESSING_LBA48;
	else if (ad->ad_available_flags & AD_F_ADDRESSING_LBA && ad->ad_flags & (AD_F_ADDRESSING_LBA | AD_F_ADDRESSING_LBA48)) ad->ad_flags = (ad->ad_flags & ~AD_F_ADDRESSING_MASK) | AD_F_ADDRESSING_LBA;
	else ad->ad_flags = (ad->ad_flags & ~AD_F_ADDRESSING_MASK) | AD_F_ADDRESSING_CHS;

	/* Mask cache */
	ad->ad_flags &= ad->ad_available_flags | ~(AD_F_WCACHE | AD_F_RCACHE);

	if (ad->ad_flags & AD_F_SKIPID) goto ret;

	fvalid = (ad->ident[83] & 0xC000) == 0x4000;

	/* Select transfer */
	RAISE_SPL(SPL_ATA_SCSI);
	ad->pio_xfer = ad->ap.aux_cmd(&ad->ap, ATA_AUX_SETUP_PIO_XFER, ad->ident, ad->ad_available_xfer, (u_jiffies_lo_t)ad->timeout);
	if (ad->pio_xfer < 0)
		ad->pio_xfer = 0;
	LOWER_SPL(SPL_ZERO);
	if (ad->ad_flags & AD_F_TRANSFER_DMA) {
		RAISE_SPL(SPL_ATA_SCSI);
		ad->dma_xfer = ad->ap.aux_cmd(&ad->ap, ATA_AUX_SETUP_DMA_XFER, ad->ident, ad->ad_available_xfer, (u_jiffies_lo_t)ad->timeout);
		LOWER_SPL(SPL_ZERO);
		if (ad->dma_xfer < 0) {
			ad->dma_xfer = 0;
			if (ad->ad_available_flags & AD_F_TRANSFER_MPIO) ad->ad_flags = (ad->ad_flags & ~AD_F_TRANSFER_MASK) | AD_F_TRANSFER_MPIO;
			else ad->ad_flags = (ad->ad_flags & ~AD_F_TRANSFER_MASK) | AD_F_TRANSFER_PIO;
		}
	} else {
		ad->dma_xfer = 0;
	}

	/* Set read/write cache */
	if (fvalid) {
		if (ad->ident[82] & 0x0040) {
			memset(&rq.fis, 0, sizeof rq.fis);
			rq.fis.feature0 = ad->ad_flags & AD_F_WCACHE ? ATA_FEATURE_SET_FEATURES_ENABLE_RCACHE : ATA_FEATURE_SET_FEATURES_DISABLE_RCACHE;
			rq.fis.command = ATA_CMD_SET_FEATURES;
			rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_FEATURE;
			ATA$SYNC_RQ(&rq, &ad->ap, NULL, 0, ad->timeout, 0);
		}
		if (ad->ident[82] & 0x0020) {
			memset(&rq.fis, 0, sizeof rq.fis);
			rq.fis.feature0 = ad->ad_flags & AD_F_WCACHE ? ATA_FEATURE_SET_FEATURES_ENABLE_WCACHE : ATA_FEATURE_SET_FEATURES_DISABLE_WCACHE;
			rq.fis.command = ATA_CMD_SET_FEATURES;
			rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_FEATURE;
			ATA$SYNC_RQ(&rq, &ad->ap, NULL, 0, ad->timeout, 0);
		}
	}
	
	if (ad->multiple > 1 && !(ad->ad_flags & AD_F_TRANSFER_PIO)) {
		memset(&rq.fis, 0, sizeof rq.fis);
		rq.fis.nsect0 = ad->multiple;
		rq.fis.command = ATA_CMD_SET_MULTIPLE_MODE;
		rq.atarq_flags = ATA_PROTOCOL_NODATA;
		ATA$SYNC_RQ(&rq, &ad->ap, NULL, 0, ad->timeout, 0);
		if (rq.status) {
			if (ad->ad_flags & AD_F_TRANSFER_MPIO) {
				ad->ad_flags = (ad->ad_flags & ~AD_F_TRANSFER_MASK) | AD_F_TRANSFER_PIO;
			}
			ad->multiple = 1;
		}
	}
	ret:

	if (!ad->ap.queue_depth)
		KERNEL$SUICIDE("AD_INIT_DISK (%s): THE HOST CONTROLLER REPORTS ZERO QUEUE DEPTH", ad->dev_name);
	ad->queue_depth = ad->ap.queue_depth;
	/* for non-TCQ and non-NCQ modes, there is really no point in having
	   more than 2 queued requests */
	if (ad->queue_depth > 2) ad->queue_depth = 2;

	if (!(ad->ad_flags & AD_F_ADDRESSING_LBA48))
		maxt = 256;
	else
		maxt = 65535;
	
	if (!ad->max_transfer || ad->max_transfer > maxt)
		ad->max_transfer = maxt;
}

__COLD_ATTR__ static void AD_FLUSH_CACHE(AD *ad)
{
	ATARQ rq;
	if (!(ad->ad_flags & AD_F_DO_FLUSH)) {
		return;
	}
	memset(&rq.fis, 0, sizeof rq.fis);
	if (ad->ad_flags & AD_F_HAS_FLUSH_EXT) {
		rq.fis.command = ATA_CMD_FLUSH_CACHE_EXT;
		rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_48BIT;
	} else {
		rq.fis.command = ATA_CMD_FLUSH_CACHE;
		rq.atarq_flags = ATA_PROTOCOL_NODATA;
	}
	ATA$SYNC_RQ(&rq, &ad->ap, NULL, 0, ad->flush_timeout, 0);
}

__COLD_ATTR__ static int AD_IOCTL(IOCTLRQ *rq, PARTITION *pa, IORQ *rq_to_queue)
{
	AD *ad = pa->dev;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI))) KERNEL$SUICIDE("AD_IOCTL AT SPL %08X", KERNEL$SPL);
#endif
	switch (rq->ioctl) {
		case IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE: {
			int ctrl_size;
			rq->status = ad->max_transfer;
			if (rq->status > 2046 && __unlikely(ad->ad_flags & (AD_F_TRANSFER_PIO | AD_F_TRANSFER_MPIO))) rq->status = 2046;
			rq->status <<= BIO_SECTOR_SIZE_BITS;
			/*RAISE_SPL(SPL_ATA_SCSI);*/
			ctrl_size = ad->ap.aux_cmd(&ad->ap, ATA_AUX_GET_TRANSFER_SIZE, ad->ad_flags & (AD_F_TRANSFER_MASK & ~(AD_F_TRANSFER_PIO | AD_F_TRANSFER_MPIO)));
			if (ctrl_size >= 0) {
				if (rq->status > ctrl_size) rq->status = ctrl_size;
			}
			/*LOWER_SPL(SPL_DEV);*/
			return 0;
		}
		case IOCTL_BIO_PHYSICAL_BLOCKSIZE: {
			rq->status = ad->physical_sector_size;
			return 0;
		}
		default: {
			return -1;
		}
	}
}

static void AD_SETUP_CHS(AD_TAG *tag, BIORQ *rq)
{
	unsigned s, q, c, h;
	AD *ad = tag->ad;
	s = rq->sec;
	q = s / ad->secs;
	s = s % ad->secs;
	c = q / ad->heads;
	h = q % ad->heads;
	if (__unlikely(c >= ad->cyls) || __unlikely(h >= 16) || __unlikely(s > 255))
		KERNEL$SUICIDE("AD_SETUP_CHS: INVALID CHS: LBA %"__sec_t_format"X, CYLINDER %d, HEAD %d, SECTOR %d, CYLINDERS %d, HEADS %d, SECTORS %d", rq->sec, c, h, s, ad->cyls, ad->heads, ad->secs);
	tag->atarq.fis.lba0 = s + 1;
	tag->atarq.fis.lba8 = c;
	tag->atarq.fis.lba16 = c >> 8; 
	tag->atarq.fis.device = h;
}


static DECL_IOCALL(AD_REQUEST, SPL_ATA_SCSI, BIORQ)
{
	AD *ad;
	AD_TAG *tag;
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &AD_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_BIO);
	SWITCH_PROC_ACCOUNT(RQ->proc, SPL_X(SPL_ATA_SCSI));
	BIO_TRANSLATE_PARTITION;
	ad = ((PARTITION *)h->fnode)->dev;
	/* do not overwrite random sectors if request is damaged */
	if (__unlikely(RQ->flags & ~BIO_FLAG_MASK))
		KERNEL$SUICIDE("AD_REQUEST: BAD REQUEST: FLAGS %08X", RQ->flags);
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("AD_REQUEST", ad->q, RQ);
#endif

	RQ->status = 0;
	RQ->tmp3 = 1;
	if ((!(tag = ad->tag_freelist)) || !BIOQUE$QUEUE_EMPTY(ad->q)) goto enq;
	ad->tag_freelist = TAG_GET_NEXT_FREE(tag);
	
	AD_POST_REQUEST(tag, RQ);

	ret:
	RETURN;

	enq:
	BIOQUE$ENQUEUE_REQUEST(ad->q, RQ);
	goto ret;

	out:
	out2:
	RQ->status = -ERANGE;
	CALL_BIO(RQ);
	goto ret;
}

static int AD_POST_REQUEST(AD_TAG *tag, BIORQ *rq)
{
	static const __u8 commands[12] = {
		ATA_CMD_READ, ATA_CMD_WRITE,
		ATA_CMD_READ_EXT, ATA_CMD_WRITE_EXT,

		ATA_CMD_READ_MULTIPLE, ATA_CMD_WRITE_MULTIPLE,
		ATA_CMD_READ_MULTIPLE_EXT, ATA_CMD_WRITE_MULTIPLE_EXT,

		ATA_CMD_READ_DMA, ATA_CMD_WRITE_DMA,
		ATA_CMD_READ_DMA_EXT, ATA_CMD_WRITE_DMA_EXT,
	};
	static const __u8 flags[3] = {
		ATA_PROTOCOL_PIO,
		ATA_PROTOCOL_PIO,
		ATA_PROTOCOL_DMA,
	};
	unsigned s, h, xfer;
	int r;
	AD *ad;
	__PREFETCHT0(&commands[8]);
	BIO_CHECK_REQUEST("AD_POST_REQUEST", rq);
	queue_next_part:
	ad = tag->ad;
	tag->atarq.brq = rq;
	tag->atarq.retries = 1;
	tag->atarq.fis.ctl = 0;
	if (__unlikely(rq->flags & BIO_FLUSH)) {
		if ((ad->ad_flags & (AD_F_DO_FLUSH | AD_F_SOMETHING_TO_FLUSH)) != (AD_F_DO_FLUSH | AD_F_SOMETHING_TO_FLUSH)) {
			tag->atarq.status = 0;
			AD_REQUEST_DONE(&tag->atarq);
			return BIO_DEQUEUE_WANT_MORE;
		}
		tag->atarq.timeout = ad->flush_timeout;
		if (!(ad->ad_flags & AD_F_HAS_FLUSH_EXT)) {
			tag->atarq.atarq_flags = ATA_PROTOCOL_NODATA;
			tag->atarq.fis.command = ATA_CMD_FLUSH_CACHE;
		} else {
			tag->atarq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_48BIT;
			tag->atarq.fis.command = ATA_CMD_FLUSH_CACHE_EXT;
		}
		tag->atarq.desc = NULL;
		goto do_post;
	}
	tag->atarq.timeout = ad->timeout;
	tag->atarq.atarq_flags = ((rq->flags & BIO_WRITE) << __BSF_CONST(ATARQ_TO_DEVICE / BIO_WRITE)) | ATARQ_SET_SIZE;
	if (__likely((unsigned)rq->nsec <= ad->max_transfer)) tag->atarq.len = rq->nsec << BIO_SECTOR_SIZE_BITS;
	else tag->atarq.len = ad->max_transfer << BIO_SECTOR_SIZE_BITS;
	if (__likely(ad->ad_flags & AD_F_ADDRESSING_LBA48)) {
		s = rq->sec;
		tag->atarq.fis.lba0 = s;
		s >>= 8;
		tag->atarq.fis.lba8 = s;
		s >>= 8;
		tag->atarq.fis.lba16 = s;
		s >>= 8;
		h = rq->sec >> 31 >> 1;
		if (__likely(!h) && s < 16 && tag->atarq.len < 256 << BIO_SECTOR_SIZE_BITS) goto lba28;
		tag->atarq.fis.lba24 = s;
		tag->atarq.fis.lba32 = h;
		tag->atarq.fis.lba40 = h >> 8;
		tag->atarq.fis.device = ATA_DEVICE_LBA;
		tag->atarq.atarq_flags |= ATARQ_VALID_48BIT;
	} else {
		if (__likely(ad->ad_flags & AD_F_ADDRESSING_LBA)) {
			s = rq->sec;
			tag->atarq.fis.lba0 = s;
			s >>= 8;
			tag->atarq.fis.lba8 = s;
			s >>= 8;
			tag->atarq.fis.lba16 = s;
			s >>= 8;
			lba28:
			tag->atarq.fis.device = s | ATA_DEVICE_LBA;
		} else {
#if __DEBUG >= 1
			if (!(ad->ad_flags & AD_F_ADDRESSING_CHS))
				KERNEL$SUICIDE("AD_POST_REQUEST: INVALID ADDRESSING MODE, FLAGS %08X", ad->ad_flags);
#endif
			AD_SETUP_CHS(tag, rq);
		}
	}

	xfer = __BSF((ad->ad_flags & AD_F_TRANSFER_MASK) / AD_F_TRANSFER_PIO);
	tag->atarq.atarq_flags |= flags[xfer];
	tag->atarq.fis.command = commands[(xfer << 2) | ((tag->atarq.atarq_flags & ATARQ_VALID_48BIT) >> __BSF_CONST(ATARQ_VALID_48BIT / 2)) | (rq->flags & BIO_WRITE)];
	tag->atarq.desc = rq->desc;
	do_post:
	r = ad->ap.post(&tag->atarq);
	if (__unlikely(r < 0)) {
		if (r == -EAGAIN) {
			TAG_SET_NEXT_FREE(tag, ad->tag_freelist);
			ad->tag_freelist = tag;
			BIO_CHECK_REQUEST("AD_POST_REQUEST - REQUEUE", rq);
			BIOQUE$REQUEUE_DEQUEUED_REQUEST(ad->q, rq);
			return BIO_DEQUEUE_FULL;
		}
		AD_ERROR(tag, r, !(rq->flags & BIO_FLUSH) ? rq->sec : -1);
		return BIO_DEQUEUE_WANT_MORE;
	}
	if (__unlikely((tag->atarq.len >> BIO_SECTOR_SIZE_BITS) != rq->nsec) && !(rq->flags & BIO_FLUSH)) {
		if (__unlikely((tag = AD_CONTINUE_REQUEST(tag)) != NULL)) goto queue_next_part;
	}
	ad->ad_flags &= ~((rq->flags & BIO_FLUSH) << __BSF_CONST(AD_F_SOMETHING_TO_FLUSH / BIO_FLUSH));
	BIOQUE$ACK_DEQUEUE(ad->q);
	return ad->tag_freelist ? BIO_DEQUEUE_WANT_MORE : BIO_DEQUEUE_EMPTY;
}

static int AD_DEQUEUE(ATA_ATTACH_PARAM *ap)
{
	AD *ad;
	AD_TAG *tag;
	BIORQ *rq;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("AD_DEQUEUE AT SPL %08X", KERNEL$SPL);
#endif
	ad = GET_STRUCT(ap, AD, ap);
	if (__likely((tag = ad->tag_freelist) != NULL) && !BIOQUE$QUEUE_EMPTY(ad->q)) {
		rq = BIOQUE$DEQUEUE(ad->q);
		ad->tag_freelist = TAG_GET_NEXT_FREE(tag);
		
		return AD_POST_REQUEST(tag, rq);
	}
	return BIO_DEQUEUE_EMPTY;
}

static BIORQ *AD_PROBE_QUEUE(ATA_ATTACH_PARAM *ap)
{
	AD *ad;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("AD_PROBE_QUEUE AT SPL %08X", KERNEL$SPL);
#endif
	ad = GET_STRUCT(ap, AD, ap);
	return BIOQUE$PROBE_QUEUE(ad->q);
}

static void AD_REQUEST_DONE(ATARQ *arq)
{
	AD *ad;
	AD_TAG *tag = GET_STRUCT(arq, AD_TAG, atarq);
	BIORQ *rq;
	if (__unlikely(tag->atarq.status)) {
		AD_ERROR(tag, tag->atarq.status, -1);
		return;
	}
	rq = tag->atarq.brq;
	ad = tag->ad;
	TAG_SET_NEXT_FREE(tag, ad->tag_freelist);
	ad->tag_freelist = tag;
	ad->ad_flags |= (rq->flags & BIO_WRITE) << __BSF_CONST(AD_F_SOMETHING_TO_FLUSH / BIO_WRITE);
#if __DEBUG >= 1
	if (__unlikely(!rq->tmp3))
		KERNEL$SUICIDE("AD_REQUEST_DONE: REFERENCE COUNT UNDERFLOW");
#endif
	if (__likely(!--rq->tmp3))
		CALL_BIO(rq);
}

static AD_TAG *AD_CONTINUE_REQUEST(AD_TAG *old_tag)
{
	AD *ad;
	AD_TAG *tag;
	BIORQ *rq = old_tag->atarq.brq;
	rq->desc = old_tag->atarq.desc;
#if __DEBUG >= 1
	if (__unlikely(old_tag->atarq.len & (BIO_SECTOR_SIZE - 1)))
		KERNEL$SUICIDE("AD_CONTINUE_REQUEST: NUMBER OF SECTORS NOT ALIGNED: %X", old_tag->atarq.len);
#endif
	rq->sec += old_tag->atarq.len >> BIO_SECTOR_SIZE_BITS;
	rq->nsec -= old_tag->atarq.len >> BIO_SECTOR_SIZE_BITS;
	BIO_CHECK_REQUEST("AD_CONTINUE_REQUEST", rq);
	if (__unlikely(rq->status != 0)) {
		return NULL;
	}
	rq->tmp3++;
	ad = old_tag->ad;
	if (__likely(!(tag = ad->tag_freelist))) {
		BIOQUE$ENQUEUE_REQUEST(ad->q, rq);
		return NULL;
	}
	ad->tag_freelist = TAG_GET_NEXT_FREE(tag);
	return tag;
}

__COLD_ATTR__ static void AD_ERROR(AD_TAG *tag, int err, __sec_t lba)
{
	AD *ad = tag->ad;
	BIORQ *rq = tag->atarq.brq;
	if (__likely(err == -EVSPACEFAULT)) {
		BIO$FAULT(rq, lba);
	} else {
		if (rq->flags & (BIO_WRITE | BIO_FLUSH)) ad->ad_flags |= AD_F_SOMETHING_TO_FLUSH;
		if (err == -EIO) {
			BIO$ERROR(rq, lba);
		} else if (__likely(!rq->status)) {
			rq->status = err;
		}
	}
	TAG_SET_NEXT_FREE(tag, ad->tag_freelist);
	ad->tag_freelist = tag;
	if (__unlikely(!rq->tmp3))
		KERNEL$SUICIDE("AD_ERROR: REFERENCE COUNT UNDERFLOW (ERROR %d)", err);
	if (__likely(!--rq->tmp3))
		CALL_BIO(rq);
}
