#include <SPAD/SYNC.H>
#include <SPAD/AC.H>
#include <SPAD/BIO_KRNL.H>
#include <SPAD/LIST.H>
#include <SPAD/LIBC.H>
#include <SPAD/ISADMA.H>
#include <SPAD/VM.H>
#include <ARCH/IO.H>
#include <ARCH/TIME.H>
#include <SPAD/IRQ.H>
#include <SPAD/TIMER.H>
#include <SYS/TYPES.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYSLOG.H>
#include <STRING.H>
#include <STDLIB.H>
#include <TIME.H>
#include <FCNTL.H>
#include <UNISTD.H>
#include <SPAD/RANDOM.H>
#include <SPAD/CONSOLE.H>
#include <SPAD/SCHED.H>

#include <LINUX/FD.H>

/*
#define VERBOSE_ERRORS
*/

#include "FDREG.H"

#define FD_NAME			"BIO$FD"

#define CRASHDUMP_FILE		"LOG:/CRASHDUMP"

#define SPINDOWN_TIME		(3 * JIFFIES_PER_SECOND)
#define RETRIES			5
#define TIMEOUT_RETRIES		2
#define FLOPPY_TIMEOUT		(3 * JIFFIES_PER_SECOND)
#define DEFAULT_THRESH		8
#define RECOVERY_TRACK		1

#define N_FDC			2	/* amount of controllers */
#define N_FD			2	/* drives per controller */
#define FLOPPY_DMA		2
#define FLOPPY_IRQ		6
#define FLOPPY_DMA_SIZE		65536
#define FLOPPY_DMA_ALIGN	FLOPPY_DMA_SIZE
static const io_t FLOPPY_IO[N_FDC] = { 0x3f0, 0x370 };

#define N_FLOPPIES	(N_FDC * N_FD)

#define FD_RESET_DELAY 		100
#define FD_POLL_DELAY 		20000
#define FD_POLL_DELAY_EXTD	(FLOPPY_TIMEOUT * (1000000 / JIFFIES_PER_SECOND + 1))
#define FD_SEEK_WAIT		1000000
#define RATE_SELECT_DELAY	2

#define FD_NAME_LEN		16

#define FLOPPY_SECTORS(x)	((x) & 0x3f)
#define FLOPPY_SIDES(x)		((((x) >> 6) & 1) + 1)
#define FLOPPY_TRACKS(x)	(((x) >> 8) & 0xff)
#define FLOPPY_STRETCH(x)	(((x) >> 7) & 1)
#define FLOPPY_GAP1(x)		(((x) >> 16) & 0xff)
#define FLOPPY_STEP(x)		(((x) >> 24) & 0x0f)
#define FLOPPY_RATE(x)		(((x) >> 28) & 3)
#define FLOPPY_PERPENDICULAR(x)	(((x) >> 30) & 1)
#define FLOPPY_UNLOCKED_FD(x)	(((x) >> 31) & 1)
#define FLOPPY_UNLOCKED_FD_CONST (1UL << 31)

#define FLOPPY_ENCODE(sectors, sides, tracks, stretch, gap1, step, rate, perpendicular) ((unsigned long)sectors + ((unsigned long)(sides - 1) << 6) + ((unsigned long)tracks << 8) + ((unsigned long)stretch << 7) + ((unsigned long)gap1 << 16) + ((unsigned long)step << 24) + ((unsigned long)rate << 28) + ((unsigned long)perpendicular << 30))

static const struct drive_type {
	unsigned long default_format;
	int spinup_time;
	int head_load_time;
	int head_unload_time;
	const char *name;
} unknown_drive = { FLOPPY_ENCODE(18, 2, 80, 0, 0x1b, 0xc, 0, 0), JIFFIES_PER_SECOND, 16, 15, "UNKNOWN DRIVE (3.5\" 1.44MB ASSUMED)" };


static struct floppy {
	int used;
	const struct drive_type *default_type;
	int i;
	io_t io;
	int drive;
	struct fdc *fdc;
	int track;
	char name[FD_NAME_LEN];		/* BIO$FD:/xx */
} floppies[N_FLOPPIES];

static struct fdc {
	int data_rate;
	int perpendicular;
	int spec1;
	int spec2;
	int used;
	int need_configure;
} fdcs[N_FDC];

static IO_RANGE ranges1[N_FDC];
static IO_RANGE ranges2[N_FDC];

static PARTITIONS *part;	/* needed for emulation of read/write --- although floppies don't have partitions */

static int thresh = DEFAULT_THRESH;

static int unexp_irq;

static __u8 *virtual_dma_buffer;
static unsigned long physical_dma_buffer;

#define CACHE_BLOCKS	(FLOPPY_DMA_SIZE / BIO_SECTOR_SIZE)

static char cache_map[CACHE_BLOCKS];	/* 0 -- free, or floppy->i + 1 */
static unsigned cache[CACHE_BLOCKS];
static int cache_ptr;

static struct floppy *spinning_floppy;
static jiffies_t spinup_start;

static int timeout_pending;
static int crash_dumping = 0;

static struct {
	BIORQ *current_request;
	struct floppy *current_floppy;
	int tries;
	int only_one_sector;
	int command;
	int cache_read_ptr, cache_read_len;
	int cache_written;
	int seeked_track;
	int dskchg;
	sched_unsigned t;
} rq_state;

#define C_SPINUP_WAIT	1
#define C_RECALIBRATE	2
#define C_RATE_WAIT	3
#define C_SEEK		4
#define C_RW		5

static BIOQUEUE *queue;

static void *dlrq;
static void *lnte;

static RANDOM_CTX random_ctx;

static AST floppy_irq_ast;
static IRQ_HANDLE *floppy_irq_ctrl = NULL;
static AST_STUB floppy_irq;

static void zap_cache(int ii);
static void fd_init_root(HANDLE *h, void *f_);
static const HANDLE_OPERATIONS floppy_operations;
static IO_STUB floppy_request;
static void floppy_dequeue(void);
static int floppy_unload(void *p, void **release, const char * const argv[]);
static int floppy_dctl(void *p, void **release, const char * const argv[]);
static void floppy_read_dump(char *dev_name, char *path);
static int floppy_dump(void);

static int poll_fdc_ready(io_t io, int extd_timeo)
{
	int r;
	unsigned t = 1;
	do {
		if (__likely((r = io_inb(io + FD_STATUS)) & STATUS_READY)) return r;
		KERNEL$UDELAY(t);
		t *= 2;
	} while (t < (!extd_timeo ? FD_POLL_DELAY : FD_POLL_DELAY_EXTD));
	return -1;
}

static int reset_fdc(int fdc)
{
	int intstate = KERNEL$SAVE_INT_STATE();
	int r;
	int c;
	if (intstate && floppy_irq_ctrl) KERNEL$MASK_IRQ(floppy_irq_ctrl);
	fdcs[fdc].data_rate = -1;
	fdcs[fdc].spec1 = -1;
	fdcs[fdc].spec2 = -1;
	fdcs[fdc].perpendicular = -1;
	fdcs[fdc].need_configure = 1;
	io_outb(FLOPPY_IO[fdc] + FD_DOR, DOR_RESET);
	KERNEL$UDELAY(FD_RESET_DELAY);
	io_outb(FLOPPY_IO[fdc] + FD_DOR, DOR_DEFAULT);
	KERNEL$UDELAY(FD_RESET_DELAY);
	if (poll_fdc_ready(FLOPPY_IO[fdc], 0) != STATUS_READY) goto eoi_err;

	io_outb(FLOPPY_IO[fdc] + FD_COMMAND, FD_SENSEI);
	c = 0;
	next:
	if ((r = poll_fdc_ready(FLOPPY_IO[fdc], 0)) < 0) {
		if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, "FD", "SENSEI TIMEOUT ON " IO_FORMAT, FLOPPY_IO[fdc]);
		goto eoi_err;
	}
	if ((r & (STATUS_DIR | STATUS_READY | STATUS_DMA)) == STATUS_READY) {
		fdcs[fdc].used = 1;
		if (intstate && floppy_irq_ctrl) KERNEL$UNMASK_IRQ(floppy_irq_ctrl);
		unexp_irq = 1;
		return 0;
	}
	if (__unlikely((r & (STATUS_DIR | STATUS_READY | STATUS_BUSY)) != (STATUS_DIR | STATUS_READY | STATUS_BUSY))) {
		if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, "FD", "BOGUS STATUS %02X ON " IO_FORMAT, r, FLOPPY_IO[fdc]);
		goto eoi_err;
	}
	io_inb(FLOPPY_IO[fdc] + FD_DATA);
	if (c++ < 16) goto next;
	if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, "FD", "TOO MANY REPLIES ON " IO_FORMAT, FLOPPY_IO[fdc]);
	eoi_err:
	if (intstate && floppy_irq_ctrl) KERNEL$UNMASK_IRQ(floppy_irq_ctrl);
	return -1;
}

static int output_byte(int b)
{
	io_t io = rq_state.current_floppy->io;
	int r;
	if (__unlikely((r = poll_fdc_ready(io, 0)) < 0)) goto timeo;
	if (__likely((r & (STATUS_READY|STATUS_DIR|STATUS_DMA))) == STATUS_READY) {
		io_outb(io + FD_COMMAND, b);
		/*__debug_printf("%02x.", b);*/
		return 0;
	}
	if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "FDC RETURNS BOGUS STATUS %02X, SENDING BYTE %02X", r, b);
	return -1;
	timeo:
	if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "FDC TIMEOUT, STATUS %02X, SENDING BYTE %02X", io_inb(io + FD_STATUS), b);
	return -1;
}

#define MAX_REPLIES 16

static __u8 st[MAX_REPLIES];
static int stlen;

static int result(void)
{
	io_t io = rq_state.current_floppy->io;
	int r;
	stlen = 0;
	while (1) {
		if (__unlikely(r = poll_fdc_ready(io, 0)) < 0) goto timeo;
		if ((r & (STATUS_DIR | STATUS_READY | STATUS_DMA)) == STATUS_READY)
			return stlen;
		if (__unlikely((r & (STATUS_DIR | STATUS_READY | STATUS_BUSY)) != (STATUS_DIR | STATUS_READY | STATUS_BUSY))) goto invl;
		if (stlen == MAX_REPLIES) goto over;
		st[stlen++] = io_inb(io + FD_DATA);
	}
	timeo:
	if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "FDC TIMEOUT, STATUS %02X, RECEIVING", io_inb(io + FD_STATUS));
	return -1;
	invl:
	if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "FDC RETURNS BOGUS STATUS %02X, RECEIVING", r);
	return -1;
	over:
	if (!crash_dumping) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "RECEIVE OVERFLOW, STATUS %02X", r);
	return -1;
}

static int need_more_output(void)
{
	int r;
	r = poll_fdc_ready(rq_state.current_floppy->io, 0);
	if (r < 0) return 0;
	if ((r & (STATUS_READY | STATUS_DIR | STATUS_DMA)) == STATUS_READY) return 1;
	result();
	return 0;
}

static int disk_changed(void)
{
	int ch = io_inb(rq_state.current_floppy->io + FD_DIR) & DIR_DSKCHG;
	if (__unlikely(ch)) zap_cache(-1);
	return ch;
}

#if defined(__IA32__)

#include <SPAD/CMOS.H>

static const struct drive_type bios_drives[6] = {
	{ FLOPPY_ENCODE(18, 2, 80, 0, 0x1b, 0xc, 0, 0), JIFFIES_PER_SECOND, 16, 15, "unknown drive (3.5\" 1.44MB assumed)" },
	{ FLOPPY_ENCODE( 9, 2, 40, 0, 0x2a, 0xd, 2, 0), JIFFIES_PER_SECOND, 16, 15, "5.25\" 360kB" },
	{ FLOPPY_ENCODE(15, 2, 80, 0, 0x1b, 0xd, 0, 0), JIFFIES_PER_SECOND * 4/10, 16, 15, "5.25\" 1.2MB" },
	{ FLOPPY_ENCODE( 9, 2, 80, 0, 0x2a, 0xd, 2, 0), JIFFIES_PER_SECOND, 16, 15, "3.5\" 720kB" },
	{ FLOPPY_ENCODE(18, 2, 80, 0, 0x1b, 0xc, 0, 0), JIFFIES_PER_SECOND * 4/10, 16, 15, "3.5\" 1.44MB" },
	{ FLOPPY_ENCODE(18, 2, 80, 0, 0x1b, 0xa, 3, 1), JIFFIES_PER_SECOND * 4/10, 15, 8, "3.5\" 2.88MB" },
};

static const struct drive_type *drive_type(unsigned i)
{
	int cm;
	unsigned type;
	if (i >= 2) return &unknown_drive;
	cm = CMOS$READ(0x10);
	if (cm < 0) return &unknown_drive;
	type = (cm >> ((i ^ 1) * 4)) & 0x0f;
	if (!type) return NULL;
	if (type > 6) return &unknown_drive;
	if (type == 6) type = 5;
	return &bios_drives[type];
}

#else

static struct const drive_type *drive_type(unsigned i)
{
	return &unknown_drive;
}

#endif

static int floppy_ioctl(IOCTLRQ *rq, PARTITION *pa, IORQ *rq_to_queue)
{
	struct floppy_struct fs;
	unsigned long fl = rq->handle->flags2;
	int r;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_FLOPPY))) KERNEL$SUICIDE("floppy_ioctl AT SPL %08X", KERNEL$SPL);
#endif
	switch (rq->ioctl) {
		case IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE:
			rq->status = FLOPPY_SECTORS(fl) << BIO_SECTOR_SIZE_BITS;
			return 0;
		case IOCTL_BIO_FDGETPRM:
			memset(&fs, 0, sizeof fs);
			fs.sect = FLOPPY_SECTORS(fl);
			fs.head = FLOPPY_SIDES(fl);
			fs.track = FLOPPY_TRACKS(fl);
			fs.size = fs.sect * fs.head * fs.track;
			fs.stretch = FLOPPY_STRETCH(fl);
			fs.gap = FLOPPY_GAP1(fl);
			fs.rate = FLOPPY_RATE(fl);
			fs.spec1 = (FLOPPY_STEP(fl) << 4) | 0x0f;
			fs.fmt_gap = 0;
			r = KERNEL$PUT_IOCTL_STRUCT(rq, &fs, sizeof fs);
			if (__unlikely(r > 0)) {
				DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);
				return 1;
			}
			rq->status = r;
			return 0;
		default:
			return -1;
	}
}

int main(int argc, const char * const argv[])
{
	void *dev_ptr;
	int r, f, i;
	const char * const *arg = argv;
	int state = 0;
	char *name;
	static const struct __param_table params[1] = {
		NULL, 0, 0, 0,
	};
	void *vars[1];
	vars[0] = NULL;

	memset(floppies, 0, sizeof floppies);
	memset(fdcs, 0, sizeof fdcs);

	unexp_irq = 0;

	memset(cache_map, -1, sizeof cache_map);
	cache_ptr = 0;
	timeout_pending = 0;
	memset(&rq_state, 0, sizeof rq_state);
	spinning_floppy = NULL;

	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FD: SYNTAX ERROR");
		r = -EBADSYN;
		goto err0;
	}

	if ((r = ISADMA$REQUEST_CHANNEL(FLOPPY_DMA))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, FD_NAME, "COULD NOT GET DMA %d: %s", FLOPPY_DMA, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, FD_NAME ": COULD NOT GET DMA");
		goto err0;
	}

	f = 0;
	for (i = 0; i < N_FDC; i++) {
		int ff;
		ranges1[i].start = FLOPPY_IO[i] + REG_OFFSET_1;
		ranges1[i].len = REG_LEN_1;
		ranges1[i].name = FD_NAME;
		if (__unlikely(KERNEL$REGISTER_IO_RANGE(&ranges1[i]))) {
			continue;
		}
		ranges2[i].start = FLOPPY_IO[i] + REG_OFFSET_2;
		ranges2[i].len = REG_LEN_2;
		ranges2[i].name = FD_NAME;
		if (__unlikely(KERNEL$REGISTER_IO_RANGE(&ranges2[i]))) {
			KERNEL$UNREGISTER_IO_RANGE(&ranges1[i]);
			continue;
		}
		ff = !reset_fdc(i);
		if (!ff) {
			KERNEL$UNREGISTER_IO_RANGE(&ranges1[i]);
			KERNEL$UNREGISTER_IO_RANGE(&ranges2[i]);
		}
		f |= ff;
	}
	if (!f) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FD: NO FD CONTROLLERS FOUND");
		r = -ENODEV;
		goto err1;
	}
	name = NULL;
	f = 0;
	for (i = 0; i < N_FLOPPIES; i++) if (fdcs[i / N_FD].used) {
		struct floppy *floppy = &floppies[i];
		if ((floppy->default_type = drive_type(i))) {
			floppy->used = 1;
			if (i) _snprintf(floppy->name, FD_NAME_LEN, FD_NAME ":/%d", i);
			else _snprintf(floppy->name, FD_NAME_LEN, FD_NAME);
			floppy->i = i;
			floppy->io = FLOPPY_IO[i / N_FD];
			floppy->drive = i % N_FD;
			floppy->fdc = &fdcs[i / N_FD];
			floppy->track = -1;
			f |= 1;
			_printf("%s: %s\n", floppy->name, floppy->default_type->name);
			if (!name) name = floppy->name;
		}
	}
	if (!f) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FD: NO FLOPPY DRIVES FOUND");
		r = -ENODEV;
		goto err1;
	}

	virtual_dma_buffer = KERNEL$ALLOC_CONTIG_AREA(FLOPPY_DMA_SIZE, AREA_ISADMA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)FLOPPY_DMA_ALIGN);
	if (__IS_ERR(virtual_dma_buffer)) {
		r = __PTR_ERR(virtual_dma_buffer);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FD: CAN'T ALLOCATE DMA BUFFER: %s", strerror(-r));
		goto err1;
	}
	physical_dma_buffer = KERNEL$VIRT_2_PHYS(virtual_dma_buffer);

	r = BIOQUE$ALLOC_QUEUE(&queue);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FD: CAN'T ALLOCATE REQUEST QUEUE");
		goto err2;
	}

	r = BIO$ALLOC_PARTITIONS(&part, NULL, SPL_X(SPL_FLOPPY), -1, floppy_ioctl, floppies[i].name);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FD: CAN'T ALLOCATE PARTITION TABLE");
		goto err3;
	}

	dev_ptr = NULL;	/* against warning */
	for (i = 0; i < N_FLOPPIES; i++) if (floppies[i].used) {
		dev_ptr = &floppies[i];
		break;
	}

	floppy_irq_ast.fn = floppy_irq;
	if ((r = KERNEL$REQUEST_IRQ(FLOPPY_IRQ, &floppy_irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_EXCLUSIVE, NULL, &floppy_irq_ast, FD_NAME)) < 0) {
		floppy_irq_ctrl = NULL;
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, FD_NAME, "COULD NOT GET IRQ %d: %s", FLOPPY_IRQ, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, FD_NAME ": COULD NOT GET IRQ");
		goto err35;
	}

	r = KERNEL$REGISTER_DEVICE(FD_NAME, "FD.SYS", 0, dev_ptr, fd_init_root, NULL, NULL, floppy_dctl, floppy_unload, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, FD_NAME ": COULD NOT REGISTER DEVICE: %s", strerror(-r));
		goto err4;
	}

	dlrq = KERNEL$TSR_IMAGE();
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s", name);
	return 0;

	err4:
	KERNEL$RELEASE_IRQ(floppy_irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_EXCLUSIVE, NULL, &floppy_irq_ast);
	err35:
	BIO$FREE_PARTITIONS(part);
	err3:
	BIOQUE$FREE_QUEUE(queue);
	err2:
	KERNEL$FREE_CONTIG_AREA(virtual_dma_buffer, FLOPPY_DMA_SIZE);
	err1:
	ISADMA$RELEASE_CHANNEL(FLOPPY_DMA);
	for (i = 0; i < N_FDC; i++) if (fdcs[i].used) {
		KERNEL$UNREGISTER_IO_RANGE(&ranges1[i]);
		KERNEL$UNREGISTER_IO_RANGE(&ranges2[i]);
	}
	err0:
	return r;
}

static void set_floppy(HANDLE *h, struct floppy *f)
{
	h->flags = (unsigned long)f;
	h->flags2 = f->default_type->default_format;
}

static void fd_init_root(HANDLE *h, void *f_)
{
	struct floppy *f = f_;
	BIO$ROOT_PARTITION(h, part);
	set_floppy(h, f);
	h->flags2 |= FLOPPY_UNLOCKED_FD_CONST;
	h->op = &floppy_operations;
}

static void *fd_lookup(HANDLE *h, char *strr, int open_flags)
{
	char *str;
	char *e;
	unsigned long fl;
	long num;
	int t;
	if (!__get_number(strr, strchr(strr, 0), 0, &num)) {
		if (__unlikely(!FLOPPY_UNLOCKED_FD(h->flags2))) return __ERR_PTR(-EACCES);
		if (__unlikely((unsigned long)num >= N_FLOPPIES)) return __ERR_PTR(-ERANGE);
		if (__unlikely(!floppies[num].used)) return __ERR_PTR(-ENODEV);
		set_floppy(h, &floppies[num]);
		return NULL;
	}
	if (__unlikely(*(str = strr) != '^')) goto def;
	str++;
	next:
	if (!_memcasecmp(str, "SECTORS=", 8)) str += 8, t = 1;
	else if (!_memcasecmp(str, "SIDES=", 6)) str += 6, t = 2;
	else if (!_memcasecmp(str, "TRACKS=", 7)) str += 7, t = 3;
	else if (!_memcasecmp(str, "STRETCH=", 8)) str += 8, t = 4;
	else if (!_memcasecmp(str, "GAP=", 4)) str += 4, t = 5;
	else if (!_memcasecmp(str, "STEP=", 5)) str += 5, t = 6;
	else if (!_memcasecmp(str, "RATE=", 5)) str += 5, t = 7;
	else if (!_memcasecmp(str, "PERPENDICULAR=", 14)) str += 14, t = 8;
	else {
		if (str != strr + 1) return __ERR_PTR(-EBADMOD);
		def:
		return BIO$LOOKUP_PARTITION(h, strr, open_flags);
	}
	e = strchr(str, ',');
	if (!e) e = str + strlen(str);
	if (__get_number(str, e, 0, &num)) return __ERR_PTR(-EBADSYN);
	if (num < 0) rng:return __ERR_PTR(-ERANGE);
	fl = h->flags2;
	switch (t) {
		case 1:
			if (num < 1 || num > 0x3f) goto rng;
			fl = FLOPPY_ENCODE(num, FLOPPY_SIDES(fl), FLOPPY_TRACKS(fl), FLOPPY_STRETCH(fl), FLOPPY_GAP1(fl), FLOPPY_STEP(fl), FLOPPY_RATE(fl), FLOPPY_PERPENDICULAR(fl));
			break;
		case 2:
			if (num < 1 || num > 2) goto rng;
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), num, FLOPPY_TRACKS(fl), FLOPPY_STRETCH(fl), FLOPPY_GAP1(fl), FLOPPY_STEP(fl), FLOPPY_RATE(fl), FLOPPY_PERPENDICULAR(fl));
			break;
		case 3:
			if (num < 1 || num > 0xff) goto rng;
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), FLOPPY_SIDES(fl), num, FLOPPY_STRETCH(fl), FLOPPY_GAP1(fl), FLOPPY_STEP(fl), FLOPPY_RATE(fl), FLOPPY_PERPENDICULAR(fl));
			break;
		case 4:
			if (num > 1) goto rng;
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), FLOPPY_SIDES(fl), FLOPPY_TRACKS(fl), num, FLOPPY_GAP1(fl), FLOPPY_STEP(fl), FLOPPY_RATE(fl), FLOPPY_PERPENDICULAR(fl));
			break;
		case 5:
			if (num > 0xff) goto rng;
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), FLOPPY_SIDES(fl), FLOPPY_TRACKS(fl), FLOPPY_STRETCH(fl), num, FLOPPY_STEP(fl), FLOPPY_RATE(fl), FLOPPY_PERPENDICULAR(fl));
			break;
		case 6:
			if (num > 0xf) goto rng;
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), FLOPPY_SIDES(fl), FLOPPY_TRACKS(fl), FLOPPY_STRETCH(fl), FLOPPY_GAP1(fl), num, FLOPPY_RATE(fl), FLOPPY_PERPENDICULAR(fl));
			break;
		case 7:
			switch (num) {
				case 250: num = 2; break;
				case 300: num = 1; break;
				case 500: num = 0; break;
				case 1000: num = 3; break;
				default: goto rng;
			}
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), FLOPPY_SIDES(fl), FLOPPY_TRACKS(fl), FLOPPY_STRETCH(fl), FLOPPY_GAP1(fl), FLOPPY_STEP(fl), num, FLOPPY_PERPENDICULAR(fl));
			break;
		case 8:
			if (num > 1) goto rng;
			fl = FLOPPY_ENCODE(FLOPPY_SECTORS(fl), FLOPPY_SIDES(fl), FLOPPY_TRACKS(fl), FLOPPY_STRETCH(fl), FLOPPY_GAP1(fl), FLOPPY_STEP(fl), FLOPPY_RATE(fl), num);
			break;
		default:
			KERNEL$SUICIDE("fd_lookup: t == %d", t);
	}
	h->flags2 = fl;
	if (*e) {
		str = e + 1;
		goto next;
	}
	return NULL;
}

static const HANDLE_OPERATIONS floppy_operations = {
	SPL_X(SPL_FLOPPY),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	fd_lookup,
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	BIO$INSTANTIATE_PARTITION,	/* instantiate */
	NULL,			/* leave */
	BIO$DETACH,		/* detach */
	NULL,			/* close */
	BIO$READ,		/* READ */
	BIO$WRITE,		/* WRITE */
	BIO$AREAD,		/* AREAD */
	BIO$AWRITE,		/* AWRITE */
	BIO$IOCTL,		/* IOCTL */
	floppy_request,		/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static VBUF vbuf = { SPL_X(SPL_FLOPPY), BIO_SECTOR_SIZE, NULL };

static void floppy_read_cached(BIORQ *rq)
{
	BIODESC *desc;
	int i, ii;
	int found;
	struct floppy *f;
	if (__unlikely((rq->flags & BIO_WRITE) != 0)) return;
	f = rq->p;
	desc = rq->desc;
	ii = f->i;
	again:
	found = 0;
	for (i = 0; i < CACHE_BLOCKS; i++) if (cache_map[i] == ii && cache[i] == (unsigned)rq->sec) {
		vbuf.ptr = virtual_dma_buffer + i * BIO_SECTOR_SIZE;
		RAISE_SPL(SPL_VSPACE);
		/*__debug_printf("vspio->%Lx\n", desc->v.ptr);*/
		if (__unlikely(desc->v.vspace->op->vspace_put(&desc->v, &vbuf) < BIO_SECTOR_SIZE)) goto fault;
		if (!desc->v.len) rq->desc = desc = desc->next;
		rq->sec++;
		if (!--rq->nsec) return;
		found = 1;
	}
	if (found) goto again;
	return;

	fault:
	/*__debug_printf("fault.\n");*/
	LOWER_SPL(SPL_FLOPPY);
	rq->nsec = 0;
	BIO$FAULT(rq, rq->sec);
}

static void floppy_write_cached(BIORQ *rq, int cache_sec, int *nsec, int fsec)
{
	int n = *nsec;
	BIODESC *desc;
	desc = rq->desc;
	while (n) {
		vbuf.ptr = virtual_dma_buffer + cache_sec * BIO_SECTOR_SIZE;
		RAISE_SPL(SPL_VSPACE);
		if (__unlikely(desc->v.vspace->op->vspace_get(&desc->v, &vbuf) < BIO_SECTOR_SIZE)) goto fault;
		if (!desc->v.len) rq->desc = desc = desc->next;
		n--;
		cache_sec++;
	}
	return;
	fault:
	LOWER_SPL(SPL_FLOPPY);
	*nsec -= n;
	BIO$FAULT(rq, fsec + *nsec);
}

static void zap_cache(int ii)
{
	int i;
	for (i = 0; i < CACHE_BLOCKS; i++) if (ii == -1 || cache_map[i] == ii) {
		cache_map[i] = -1;
	}
}

static DECL_TIMER(spindown_timer);

static void floppy_spindown_timer(TIMER *t)
{
	LOWER_SPL(SPL_FLOPPY);
	if (__unlikely(rq_state.current_request != NULL)) {
		KERNEL$SET_TIMER(SPINDOWN_TIME, &spindown_timer);
		return;
	}
	io_outb(spinning_floppy->io + FD_DOR, DOR_DEFAULT);
	spinning_floppy = NULL;
	/*zap_cache(-1);*/
}

static void spin_down(void)
{
	if (spinning_floppy) {
		KERNEL$DEL_TIMER(&spindown_timer);
		io_outb(spinning_floppy->io + FD_DOR, DOR_DEFAULT);
		spinning_floppy = NULL;
		/*zap_cache(-1);*/
	}
}

static DECL_TIMER(wait_timer);

static void floppy_do_req(BIORQ *rq);

static void floppy_timeout(TIMER *t);

static void timewait_end(TIMER *t)
{
	LOWER_SPL(SPL_FLOPPY);
	floppy_do_req(rq_state.current_request);
}

static BIORQ *end_request(void)
{
	sched_unsigned t = KERNEL$GET_SCHED_TICKS();
	BIORQ *rq = rq_state.current_request;
	PROC *proc;
	RAISE_SPL(SPL_VSPACE);
	proc = rq->proc;
	SWITCH_PROC_ACCOUNT_TICKS(proc, SPL_X(SPL_VSPACE), t);
	BIOQUE$ACCOUNT_IOSCHED(proc, queue, rq_state.t, t);
	LOWER_SPL(SPL_FLOPPY);
	memset(&rq_state, 0, sizeof rq_state);
	return rq;
}

static void floppy_do_req(BIORQ *rq)
{
	int spt;
	int nsec;
	int asec;
	int sec;
	int side;
	int track;
	int i, j;
	int wspec1, wspec2, wperp;
	int dskchg;
	jiffies_t ji;
	again:
	rq_state.t = KERNEL$GET_SCHED_TICKS();
	rq_state.current_request = rq;
	rq_state.current_floppy = rq->p;

	if (__unlikely(rq_state.current_floppy != spinning_floppy)) {
			/* SPIN UP */
		spin_down();

		io_outb(rq_state.current_floppy->io + FD_DOR, DOR_DEFAULT | DOR_SPINUP << rq_state.current_floppy->drive | rq_state.current_floppy->drive);

		spindown_timer.fn = floppy_spindown_timer;
		spinning_floppy = rq_state.current_floppy;
		spinup_start = KERNEL$GET_JIFFIES();
		KERNEL$SET_TIMER(SPINDOWN_TIME, &spindown_timer);
	}

	dskchg = 0;
	if (__unlikely(disk_changed())) {
#ifdef VERBOSE_ERRORS
		__debug_printf("DISK CHANGED, CACHE ZAPPED\n");
#endif
		dskchg = 1;
	}
	nsec = rq->nsec;
	floppy_read_cached(rq);
	if (rq->nsec != nsec) rq_state.tries = 0;
	if (!rq->nsec) goto err;

	ji = (jiffies_t)rq_state.current_floppy->default_type->spinup_time - KERNEL$GET_JIFFIES() + spinup_start;
	if (__unlikely(ji >= 0)) {
		rq_state.command = C_SPINUP_WAIT;
		wait_timer.fn = timewait_end;
		KERNEL$SET_TIMER(ji, &wait_timer);
		return;
	}

	KERNEL$DEL_TIMER(&spindown_timer);
	KERNEL$SET_TIMER(SPINDOWN_TIME, &spindown_timer);

	if (__unlikely(poll_fdc_ready(rq_state.current_floppy->io, 0) != STATUS_READY)) goto not_ready;

	if (rq_state.current_floppy->fdc->need_configure) {
		output_byte(FD_CONFIGURE);
		if (need_more_output()) {
			output_byte(0);
			output_byte(0x10 | thresh);
			output_byte(0);
		}
		rq_state.current_floppy->fdc->need_configure = 0;
	}

	if (__unlikely(rq_state.current_floppy->fdc->data_rate != FLOPPY_RATE(rq->tmp3))) {
		if (FLOPPY_RATE(rq->tmp3) == 1) {
			/* DRIVESPEC COMMAND */
			output_byte(FD_DRIVESPEC);
			if (need_more_output()) {
				output_byte(rq_state.current_floppy->drive);
				output_byte(0xc0);
			}
		}
		/* SET RATE & WAIT */
		io_outb(rq_state.current_floppy->io + FD_DCR, FLOPPY_RATE(rq->tmp3));
		rq_state.current_floppy->fdc->data_rate = FLOPPY_RATE(rq->tmp3);
		rq_state.current_floppy->fdc->spec1 = -1;
		rq_state.current_floppy->fdc->spec2 = -1;
		rq_state.command = C_RATE_WAIT;
		wait_timer.fn = timewait_end;
		KERNEL$SET_TIMER(RATE_SELECT_DELAY, &wait_timer);
		return;
	}

	wperp = 0;
	if (FLOPPY_PERPENDICULAR(rq->tmp3)) {
		/* SET PERPENDICULAR */
		switch (rq_state.current_floppy->fdc->data_rate) {
			case 0:
				wperp = 2;
				break;
			case 2:
				wperp = 3;
				break;
		}
	}
	if (__unlikely(rq_state.current_floppy->fdc->perpendicular != wperp)) {
		/* PERPENDICULAR COMMAND */
		output_byte(FD_PERPENDICULAR);
		if (need_more_output()) {
			output_byte(wperp);
		}
		rq_state.current_floppy->fdc->perpendicular = wperp;
	}

	wspec1 = (FLOPPY_STEP(rq->tmp3) << 4) + rq_state.current_floppy->default_type->head_unload_time;
	wspec2 = rq_state.current_floppy->default_type->head_load_time << 1;
	if (__unlikely(rq_state.current_floppy->fdc->spec1 != wspec1) || __unlikely(rq_state.current_floppy->fdc->spec2 != wspec2)) {
		/* SPECIFY COMMAND */
		output_byte(FD_SPECIFY);
		output_byte(wspec1);
		output_byte(wspec2);
		rq_state.current_floppy->fdc->spec1 = wspec1;
		rq_state.current_floppy->fdc->spec2 = wspec2;
	}

	asec = rq->sec;

	spt = FLOPPY_SECTORS(rq->tmp3) << (FLOPPY_SIDES(rq->tmp3) - 1);
	sec = (unsigned)asec % spt;
	track = (unsigned)asec / spt;
	if (__unlikely(track >= FLOPPY_TRACKS(rq->tmp3))) goto oot;

	BIOQUE$SET_HEAD(queue, track * spt);

	if (__unlikely(rq_state.only_one_sector)) nsec = 1;
	else if (!(rq->flags & BIO_WRITE)) {
		nsec = spt - sec;
	} else {
		nsec = rq->nsec;
		if (sec + nsec > spt) nsec = spt - sec;
	}
	if (nsec > CACHE_BLOCKS) nsec = CACHE_BLOCKS;

	i = -1;
	if (rq_state.cache_written) i = rq_state.cache_read_ptr;

	for (j = 0; j < CACHE_BLOCKS; j++) if (cache_map[j] == rq_state.current_floppy->i && cache[j] >= asec && cache[j] < asec + nsec) {
		if (rq->flags & BIO_WRITE) {
			if (i == -1 && j + nsec <= CACHE_BLOCKS) i = j;
			cache_map[j] = -1;
		} else {
			if (__unlikely(cache_ptr + nsec > CACHE_BLOCKS)) cache_ptr = 0;
			if (cache[j] > asec && (j < cache_ptr || j >= cache_ptr + nsec)) nsec = cache[j] - asec;
		}
	}
	if (__unlikely(i != -1)) goto ii;
	i = cache_ptr;
	if (i + nsec > CACHE_BLOCKS) i = 0;
	if (nsec > rq->nsec) cache_ptr = i + nsec;
	ii:
	for (j = 0; j < nsec; j++) cache_map[i + j] = -1, cache[i + j] = asec + j;

	if (!rq_state.cache_written && rq->flags & BIO_WRITE) {
		floppy_write_cached(rq, i, &nsec, asec);
		if (!nsec) goto err;
		rq_state.cache_written = 1;
	}

	rq_state.cache_read_ptr = i;
	rq_state.cache_read_len = nsec;

	if (__likely(FLOPPY_SIDES(rq->tmp3) > 1) && sec >= (spt >> 1)) {
		side = 1;
		sec -= spt >> 1;
	} else side = 0;
	if (__unlikely(FLOPPY_STRETCH(rq->tmp3) != 0)) track <<= 1;
	sec++;

	if (dskchg && !rq_state.dskchg) {
	/* We must perform a seek command to clear the disk-change condition */
		rq_state.current_floppy->track = -3;
		rq_state.dskchg = 1;
	}
	if (track != rq_state.current_floppy->track) {
		/*if (rq_state.current_floppy->track < -1) rq_state.current_floppy->track = -1;*/
		if ((rq_state.current_floppy->track == -1 || rq_state.current_floppy->track == -3)) {
			/*__debug_printf("calibrate\n");*/
			/* RECALIBRATE COMMAND */
			output_byte(FD_RECALIBRATE);
			output_byte(rq_state.current_floppy->drive);
			rq_state.command = C_RECALIBRATE;
			goto timeout;
		}
		/* SEEK COMMAND */
		if (rq_state.current_floppy->track == -2) track = RECOVERY_TRACK;
		/*__debug_printf("seek: %d\n", track);*/
		output_byte(FD_SEEK);
		output_byte(rq_state.current_floppy->drive);
		output_byte(track);
		rq_state.seeked_track = track;
		rq_state.command = C_SEEK;
		goto timeout;
	}

	/* R/W COMMAND */

	/*__debug_printf("rw: %x, %x\n", asec, nsec);*/

	ISADMA$SETUP_TRANSFER(FLOPPY_DMA, rq->flags & BIO_WRITE ? 1 : 0, 0, physical_dma_buffer + i * BIO_SECTOR_SIZE, nsec * BIO_SECTOR_SIZE);

	/*
	output_byte(FD_READID);
	output_byte((rq_state.current_floppy->drive & 3) | (side << 2));
	*/
	output_byte(rq->flags & BIO_WRITE ? FD_WRITE : FD_READ);
	output_byte((rq_state.current_floppy->drive & 3) | (side << 2));
	output_byte(track);
	output_byte(side);
	output_byte(sec);
	output_byte(2);
	output_byte(FLOPPY_SECTORS(rq->tmp3));
	output_byte(FLOPPY_GAP1(rq->tmp3));
	output_byte(0xff);
	rq_state.tries++;
	rq_state.command = C_RW;
	/*__debug_printf("command written: %02x/%02x/%02x/%02x/%02x/%02x/%02x/%02x/%02x\n", rq->flags & BIO_WRITE ? FD_WRITE : FD_READ, (rq_state.current_floppy->drive & 3) | (side << 2), track, side, sec, 2, (int)FLOPPY_SECTORS(rq->tmp3), (int)FLOPPY_GAP1(rq->tmp3), 0xff);*/

	timeout:
	timeout_pending = 1;
	wait_timer.fn = floppy_timeout;
	KERNEL$SET_TIMER(FLOPPY_TIMEOUT, &wait_timer);
	return;

	not_ready:
	KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, rq_state.current_floppy->name, "CONTROLLER IS NOT READY, STATUS %02X, RESETTING AND RETRYING", io_inb(rq_state.current_floppy->io + FD_STATUS));
	spin_down();
	zap_cache(-1);
	reset_fdc(rq_state.current_floppy->i / N_FD);
	rq_state.tries++;
	if (rq_state.tries < RETRIES) goto again;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "CONTROLLER PERMANENTLY NOT READY, ABORTING REQUEST");
	BIO$ERROR(rq, -1);
	goto err;

	oot:
	rq->status = -ERANGE;

	err:
	CALL_BIO(rq);
	end_request();
	if (!BIOQUE$QUEUE_EMPTY(queue)) {
		rq = BIOQUE$DEQUEUE(queue);
		goto again;
	}
}

static const struct {
	int reg;
	int mask;
	int not_reg;
	int not_mask;
	const char *msg;
	int ign;
} errors[] = {
	0, ST0_NR, 0, 0, "NOT READY", 0,
	0, ST0_ECE, 0, 0, "EQUIPMENT CHECK", 0,
	0, ST0_INTR_CMD, 0, ST0_INTR_ERR, "BAD COMMAND", 0,
	0, ST0_INTR_ABORT, 0, 0, "ABORTED", 0,
	1, ST1_OR, 0, 0, "OVERRUN", 0,
	1, ST1_MAM, 0, 0, "ADDRESS MARK NOT FOUND", 1,
	1, ST1_WP, 0, 0, "WRITE PROTECTED", -EWP,
	1, ST1_ND, 0, 0, "SECTOR NOT FOUND", 1,
	1, ST1_CRC, 2, ST2_CRC, "CRC ERROR IN ADDRESS MARK", 1,
	1, ST1_EOC, 0, 0, "END OF TRACK", 1,
	2, ST2_MAM, 1, ST1_MAM, "ADDRESS MARK NOT FOUND", 1,
	2, ST2_BC, 0, 0, "BAD TRACK", 1,
	2, ST2_WC, 0, 0, "TRACK DOES NOT MATCH", 1,
	2, ST2_CRC, 0, 0, "CRC ERROR", 1,
	2, ST2_CM, 0, 0, "DELETED SECTOR", 1,
	0, 0, 0, 0, NULL,
};

#define MSG_LEN	1024

static char error_msg[MSG_LEN];

static DECL_IRQ_AST(floppy_irq, SPL_FLOPPY, AST)
{
	int r;
	int i;
	io_t io;
	int set_st = 0;
	KERNEL$ADD_RANDOMNESS(&random_ctx, NULL, 0);
	/*__debug_printf("got intr\n");*/
	if (timeout_pending) KERNEL$DEL_TIMER(&wait_timer), timeout_pending = 0;
	switch (rq_state.command) {
	case C_RW:
		ISADMA$END_TRANSFER(FLOPPY_DMA);
		io = rq_state.current_floppy->io + FD_DATA;
		if (__unlikely((r = result()) < 3)) {
			if (r >= 0) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "ONLY %d BYTES RESULT ON R/W COMMAND", r);
			goto error;
		}
		if (__unlikely((st[0] & ST0_DS) != rq_state.current_floppy->drive)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "INTERRUPT ON UNEXPECTED DRIVE, STATUS %02X-%02X-%02X", st[0], st[1], st[2]);
			goto error;
		}
		if (__likely(!(st[0] & ST0_INTR))) goto ok;
		if (__unlikely(st[1] & ST1_OR) && __likely(thresh < 0xf)) {
			/*thresh++;*/
			thresh = 0xf;
			rq_state.current_floppy->fdc->need_configure = 1;
			rq_state.tries = 0;
#ifdef VERBOSE_ERRORS
			__debug_printf("overrun, adjusting thresh to %d\n", thresh);
#endif
			goto redo;
		}
		if (!(rq_state.current_request->flags & BIO_WRITE) && rq_state.cache_read_len != 1) {
#ifdef VERBOSE_ERRORS
			__debug_printf("error, fallback to 1-sector: %02X-%02X-%02X\n", st[0], st[1], st[2]);
#endif
			rq_state.only_one_sector = 1;
			rq_state.tries = 0;
			goto redo;
		}
		if (rq_state.tries < RETRIES) {
#ifdef VERBOSE_ERRORS
			__debug_printf("error, retry %d\n", rq_state.tries);
#endif
			rq_state.current_floppy->track = rq_state.tries == RETRIES - 1 ? -3 : -1;
			goto redo;
		}

		*error_msg = 0;
		for (i = 0; errors[i].msg; i++) if ((st[errors[i].reg] & errors[i].mask) == errors[i].mask && !(st[errors[i].not_reg] & errors[i].not_mask)) {
			_snprintf(error_msg + strlen(error_msg), MSG_LEN - strlen(error_msg), *error_msg ? ", %s" : "%s", errors[i].msg);
			if (__likely(errors[i].ign)) {
				if (__unlikely(errors[i].ign < 0)) {
					rq_state.current_request->status = errors[i].ign;
					set_st = 1;
				}
#ifndef VERBOSE_ERRORS
				goto error;
#endif
			}
		}
		if (!*error_msg) _snprintf(error_msg, MSG_LEN, "UNKNOWN ERROR");
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, rq_state.current_floppy->name, "%s: STATUS %02X-%02X-%02X: %s", rq_state.current_request->flags & BIO_WRITE ? "WRITE" : "READ", st[0], st[1], st[2], error_msg);
		error:
		if (__likely(!set_st)) BIO$ERROR(rq_state.current_request, -1);
		goto end_request;

		ok:
		for (i = rq_state.cache_read_ptr; i < rq_state.cache_read_ptr + rq_state.cache_read_len; i++) cache_map[i] = rq_state.current_floppy->i;
		floppy_read_cached(rq_state.current_request);
		if (rq_state.current_request->flags & BIO_WRITE) {
			rq_state.current_request->nsec -= rq_state.cache_read_len;
			rq_state.current_request->sec += rq_state.cache_read_len;
			rq_state.cache_written = 0;
		}
		if (__likely(!rq_state.current_request->nsec)) {
			BIORQ *rq;
			end_request:
			rq = end_request();
			KERNEL$UNMASK_IRQ(floppy_irq_ctrl);
			floppy_dequeue();
			RETURN_BIO(rq);
		}
		rq_state.tries = 0;
		redo:
		KERNEL$UNMASK_IRQ(floppy_irq_ctrl);
		floppy_do_req(rq_state.current_request);
		RETURN;

	case C_RECALIBRATE:
		output_byte(FD_SENSEI);
		if (__unlikely((r = result()) < 2)) {
			if (r >= 0) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "ONLY %d BYTES RESULT ON SENSEI / RECALIBRATE COMMAND", r);
			goto error;
		}
		if (__likely(!(st[0] & ST0_INTR)) || rq_state.current_floppy->track == -3) {
			rq_state.current_floppy->track++;
			goto redo;
		}
		/* 5.25" drives do this when they don't have media */
		if ((st[0] & (ST0_NR | ST0_ECE | ST0_SE | ST0_INTR)) == (ST0_ECE | ST0_SE | ST0_INTR_ERR)) {
			rq_state.current_request->status = -ENOMEDIUM;
			goto end_request;
		}
		goto seek_err;
	case C_SEEK:
		output_byte(FD_SENSEI);
		if (__unlikely((r = result()) < 2)) {
			if (r >= 0) KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "ONLY %d BYTES RESULT ON SENSEI / SEEK COMMAND", r);
			goto error;
		}
		if (__likely(!(st[0] & ST0_INTR))) {
			if (__unlikely(rq_state.current_floppy->track <= -2))
				rq_state.current_floppy->track++;
			else
				rq_state.current_floppy->track = rq_state.seeked_track;
			goto redo;
		}
		seek_err:
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, rq_state.current_floppy->name, "%s ERROR, STATUS %02X-%02X", rq_state.command == C_RECALIBRATE ? "RECALIBRATE" : "SEEK", st[0], st[1]);
		goto error;
	}
	if (!unexp_irq)
		KERNEL$SYSLOG(__SYSLOG_HW_NONFATAL_BUG, rq_state.current_floppy ? rq_state.current_floppy->name : "FD", "UNEXPECTED INTERRUPT");
	unexp_irq = 0;
	/*
	{
		rq_state.current_floppy = &floppies[0];
		output_byte(FD_SENSEI);
		r = result();
		__debug_printf("result: %d--", r);
		for (i = 0; i < r; i++) __debug_printf("%02X,", st[i]);
		__debug_printf("\n");
	}
	*/
	/*for (i = 0; i < N_FDC; i++) if (fdcs[i].used) io_inb(FLOPPY_IO[i] + FD_DATA);*/
	KERNEL$UNMASK_IRQ(floppy_irq_ctrl);
	RETURN;
}

static void floppy_timeout(TIMER *t)
{
	int dskchg = 0;
	/*__debug_printf("timeout\n");*/
	LOWER_SPL(SPL_FLOPPY);
	timeout_pending = 0;
	if (rq_state.command == C_RW) {
		ISADMA$END_TRANSFER(FLOPPY_DMA);
		dskchg = disk_changed();
	}
	if (rq_state.command != C_RW) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, rq_state.current_floppy->name, "TIMEOUT WHEN WAITING FOR %s COMMAND", rq_state.command == C_RECALIBRATE ? "RECALIBRATE" : rq_state.command == C_SEEK ? "SEEK" : "UNKNOWN");
	}
	io_outb(rq_state.current_floppy->io + FD_COMMAND, 0);
	spin_down();
	zap_cache(-1);
	reset_fdc(rq_state.current_floppy->i / N_FD);
	if (rq_state.tries < TIMEOUT_RETRIES) {
#ifdef VERBOSE_ERRORS
		__debug_printf("timeout, retrying.\n");
#endif
		rq_state.current_floppy->track = -3;
		floppy_do_req(rq_state.current_request);
		return;
	}
#ifdef VERBOSE_ERRORS
	__debug_printf("timeout, aborting.\n");
#endif
	if (__unlikely(!dskchg)) BIO$ERROR(rq_state.current_request, -1);
	else rq_state.current_request->status = -ENOMEDIUM;
	CALL_BIO(rq_state.current_request);
	end_request();
	floppy_dequeue();
}

static DECL_IOCALL(floppy_request, SPL_FLOPPY, BIORQ)
{
	HANDLE *h = RQ->handle;
	struct floppy *f;
	unsigned limit;

	if (__unlikely(h->op != &floppy_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_BIO);
	SWITCH_PROC_ACCOUNT(RQ->proc, SPL_X(SPL_FLOPPY));

	f = (struct floppy *)h->flags;

	BIO_TRANSLATE_PARTITION;

	RQ->p = f;
	
	if (__unlikely(RQ->flags & ~BIO_FLAG_MASK)) goto mask;

#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("floppy_request", queue, RQ);
#endif

	RQ->status = 0;
	RQ->tmp3 = h->flags2;

	/*if (RQ->flags & BIO_WRITE) __debug_printf("fd: write(%d,%d)", (int)RQ->sec, RQ->nsec);*/
	/*else if (!(RQ->flags & BIO_FLUSH)) __debug_printf("fd: read(%d,%d)", (int)RQ->sec, RQ->nsec);*/

	if (__unlikely((RQ->flags & BIO_FLUSH) != 0)) RETURN_BIO(RQ);

	limit = (FLOPPY_SECTORS(RQ->tmp3) * FLOPPY_TRACKS(RQ->tmp3)) << (FLOPPY_SIDES(RQ->tmp3) - 1);
	if (__unlikely(RQ->sec >= limit) || __unlikely(RQ->sec + RQ->nsec > limit))
		goto out;

		/* if there is pending request on current floppy, we can bypass
		   queue and return cached value directly. If the diskette has
		   been changed, current_request would invalidate cache */
	if (rq_state.current_floppy == f) {
		floppy_read_cached(RQ);
		if (!RQ->nsec) RETURN_BIO(RQ);
	}

	/*if (__unlikely(RQ->flags & BIO_DELAY_BEFORE)) goto enq;*/
	if (rq_state.current_request) goto enq;
	if (!BIOQUE$QUEUE_EMPTY(queue)) goto enqdeq;
	floppy_do_req(RQ);
	RETURN;

	enq:
	BIOQUE$ENQUEUE_REQUEST(queue, RQ);
	RETURN;

	enqdeq:
	BIOQUE$ENQUEUE_REQUEST(queue, RQ);
	floppy_dequeue();
	RETURN;

	out:
	out2:
	RQ->status = -ERANGE;
	RETURN_BIO(RQ);

	mask:
	KERNEL$SUICIDE("%s: BAD REQUEST: FLAGS = %08X", f->name, RQ->flags);
}

static void floppy_dequeue(void)
{
	/*__debug_printf("dequeue\n");*/
	if (!(BIOQUE$QUEUE_EMPTY(queue) | (unsigned long)rq_state.current_request)) {
		floppy_do_req(BIOQUE$DEQUEUE(queue));
	}
}

static int floppy_unload(void *p, void **release, const char * const argv[])
{
	int i;
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	RAISE_SPL(SPL_TOP);
	if (KERNEL$SUICIDE_DUMP == floppy_dump) {
		KERNEL$SUICIDE_DUMP = NULL;
		__barrier();
		KERNEL$SUICIDE_DUMP_PARAM = NULL;
	}
	LOWER_SPL(SPL_FLOPPY);
	lnte = NULL;
	while (!BIOQUE$QUEUE_EMPTY(queue) || rq_state.current_request) KERNEL$SLEEP(1);
	spin_down();
	LOWER_SPL(SPL_ZERO);
	BIO$FREE_PARTITIONS(part);
	BIOQUE$FREE_QUEUE(queue);
	KERNEL$FREE_CONTIG_AREA(virtual_dma_buffer, FLOPPY_DMA_SIZE);
	KERNEL$RELEASE_IRQ(floppy_irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_EXCLUSIVE, NULL, &floppy_irq_ast);
	ISADMA$RELEASE_CHANNEL(FLOPPY_DMA);
	for (i = 0; i < N_FDC; i++) if (fdcs[i].used) {
		KERNEL$UNREGISTER_IO_RANGE(&ranges1[i]);
		KERNEL$UNREGISTER_IO_RANGE(&ranges2[i]);
	}
	*release = dlrq;
	return 0;
}

static int floppy_dctl(void *p, void **release, const char * const argv[])
{
	char n[FD_NAME_LEN + 2];
	struct floppy *f = p;
	if (!argv[0] || !argv[1] || argv[2] || _strcasecmp(argv[1], "CRASHDUMP")) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "DCTL %s: SYNTAX ERROR", f->name);
		return -EBADSYN;
	}
	stpcpy(stpcpy(n, f->name), ":/");
	floppy_read_dump(f->name, n);
	RAISE_SPL(SPL_TOP);
	if (KERNEL$SUICIDE_DUMP) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "DCTL %s: CRASHDUMP ALREADY INSTALLED", f->name);
		return -EEXIST;
	}
	KERNEL$SUICIDE_DUMP_PARAM = f;
	__barrier();
	KERNEL$SUICIDE_DUMP = floppy_dump;
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static void floppy_read_dump(char *dev_name, char *path)
{
	__u8 *buf;
	unsigned len;
	__u64 time;
	int h, hf;
	int pos;
	int i;
	char logn[32];
	buf = __sync_malloc(512);
	if (__unlikely(!buf)) goto ret0;
	h = open(path, O_RDWR);
	if (__unlikely(h == -1)) goto ret1;
	if (__unlikely(pread(h, buf, 512, 0) != 512)) goto ret2;
	if (memcmp(buf, "SPADUSED", 8)) goto ret2;
	len = __32LE2CPU(*(__u32 *)&buf[12]);
	time = __64LE2CPU(*(__u64 *)&buf[16]);
	i = 0;
	create_again:
	_snprintf(logn, sizeof(logn), CRASHDUMP_FILE ".%04d", i);
	hf = open(logn, O_WRONLY | O_CREAT | O_EXCL, 0600);
	if (__unlikely(hf == -1)) {
		i++;
		if (i >= 10000) goto ret2;
		goto create_again;
	}
	pos = 0;
	next_block:
	errno = 0;
	if (__unlikely(pread(h, buf, 512, 512 + pos) != 512)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "UNABLE TO READ CRASH DUMP: %s", strerror(errno ? errno : EEOF));
		goto ret3;
	}
	errno = 0;
	if (pos + 512 < len) {
		if (__unlikely(write(hf, buf, 512) != 512)) {
			wr_err:
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "UNABLE TO WRITE CRASH DUMP TO %s: %s", logn, strerror(errno ? errno : EEOF));
			goto ret3;
		}
	} else {
		if (__unlikely(write(hf, buf, len - pos) != len - pos)) goto wr_err;
	}
	pos += 512;
	if (pos < len) goto next_block;
	if (__unlikely(fsync(hf))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "UNABLE TO FSYNC CRASH DUMP FILE %s: %s", logn, strerror(errno));
		goto ret3;
	}
	memset(buf, 0, 512);
	memcpy(buf, "SPADDUMP", 8);
	errno = 0;
	if (__unlikely(pwrite(h, buf, 512, 0) != 512)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "UNABLE TO WRITE HEADER FOR NEXT DUMP: %s", strerror(errno ? errno : EEOF));
	}
	if (__unlikely(fsync(h))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "UNABLE TO FSYNC HEADER FOR NEXT DUMP: %s", strerror(errno));
	}
	KERNEL$SYSLOG(__SYSLOG_SW_INFO, dev_name, "WROTE CRASH DUMP TO %s (TIME OF DUMP WAS %"__64_format"X)", logn, time);
	ret3:
	close(hf);
	ret2:
	close(h);
	ret1:
	free(buf);
	ret0:
	return;
}

static int wait_for_sense(void)
{
	KERNEL$UDELAY(FD_SEEK_WAIT);
	output_byte(FD_SENSEI);
	if (result() < 2 || st[0] & ST0_INTR) {
		return -1;
	}
	return 0;
}

static int sync_reset(void)
{
	if (reset_fdc(rq_state.current_floppy->i / N_FD)) {
		__critical_printf("%s: COULD NOT RESET FLOPPY CONTROLLER AT "IO_FORMAT"\n", rq_state.current_floppy->name, FLOPPY_IO[rq_state.current_floppy->i / N_FD]);
		return -EIO;
	}
	rq_state.current_floppy->track = -1;
	io_outb(rq_state.current_floppy->io + FD_DOR, DOR_DEFAULT | DOR_SPINUP << rq_state.current_floppy->drive | rq_state.current_floppy->drive);
	KERNEL$UDELAY(rq_state.current_floppy->default_type->spinup_time * (1000000 / JIFFIES_PER_SECOND + 1));
	output_byte(FD_CONFIGURE);
	if (need_more_output()) {
		output_byte(0);
		output_byte(0x10 | 0x0f);
		output_byte(0);
	}
	if (FLOPPY_RATE(rq_state.current_floppy->default_type->default_format) == 1) {
		output_byte(FD_DRIVESPEC);
		if (need_more_output()) {
			output_byte(rq_state.current_floppy->drive);
			output_byte(0xc0);
		}
	}
	io_outb(rq_state.current_floppy->io + FD_DCR, FLOPPY_RATE(rq_state.current_floppy->default_type->default_format));
	KERNEL$UDELAY(100000);
	if (FLOPPY_PERPENDICULAR(rq_state.current_floppy->default_type->default_format)) {
		int wperp = 0;
		if (FLOPPY_RATE(rq_state.current_floppy->default_type->default_format) == 0) wperp = 2;
		if (FLOPPY_RATE(rq_state.current_floppy->default_type->default_format) == 2) wperp = 3;
		if (wperp) {
			output_byte(FD_PERPENDICULAR);
			if (need_more_output()) {
				output_byte(wperp);
			}
		}
	}
	output_byte(FD_SPECIFY);
	output_byte((FLOPPY_STEP(rq_state.current_floppy->default_type->default_format) << 4) + rq_state.current_floppy->default_type->head_unload_time);
	output_byte(rq_state.current_floppy->default_type->head_load_time << 1);
	return 0;
}

static int dump_io_noretry(unsigned asec, int rw)
{
	int r;
	unsigned format = rq_state.current_floppy->default_type->default_format;
	unsigned spt = FLOPPY_SECTORS(format) << (FLOPPY_SIDES(format) - 1);
	unsigned sec = asec % spt;
	unsigned track = asec / spt;
	unsigned side = 0;
	if (FLOPPY_SIDES(format) > 1 && sec >= (spt >> 1)) {
		side = 1;
		sec -= spt >> 1;
	}
	if (__unlikely(FLOPPY_STRETCH(format) != 0)) track <<= 1;
	sec++;
	if (rq_state.current_floppy->track < 0) {
		output_byte(FD_RECALIBRATE);
		output_byte(rq_state.current_floppy->drive);
		if (wait_for_sense()) return -1;
		rq_state.current_floppy->track = 0;
	}
	if (rq_state.current_floppy->track != track) {
		output_byte(FD_SEEK);
		output_byte(rq_state.current_floppy->drive);
		output_byte(track);
		if (wait_for_sense()) {
			rq_state.current_floppy->track = -1;
			return -1;
		}
		rq_state.current_floppy->track = track;
	}
	ISADMA$SETUP_TRANSFER(FLOPPY_DMA, rw ? 1 : 0, 0, physical_dma_buffer, BIO_SECTOR_SIZE);
	output_byte(rw ? FD_WRITE : FD_READ);
	output_byte((rq_state.current_floppy->drive & 3) | (side << 2));
	output_byte(track);
	output_byte(side);
	output_byte(sec);
	output_byte(2);
	output_byte(FLOPPY_SECTORS(format));
	output_byte(FLOPPY_GAP1(format));
	output_byte(0xff);
	r = poll_fdc_ready(rq_state.current_floppy->io, 1);
	ISADMA$END_TRANSFER(FLOPPY_DMA);
	if (r < 0) {
		sync_reset();
		return -1;
	}
	if (result() < 3) {
		return -1;
	}
	if (st[0] & ST0_INTR) {
		return -1;
	}
	return 0;
}

static int dump_io(unsigned asec, int rw)
{
	int c = 0;
	do {
		if (!dump_io_noretry(asec, rw)) return 0;
	} while (++c < RETRIES);
	return -1;
}

static int floppy_dump(void)
{
	unsigned sec, l, total_l;
	int r;
	time_t t;
	__u64 pos;
	KERNEL$PPL = 0;
	KERNEL$SPL = SPL_X(SPL_DMA);
	crash_dumping = 1;
	rq_state.current_floppy = KERNEL$SUICIDE_DUMP_PARAM;
	if ((r = sync_reset())) return r;
	if (dump_io(0, 0)) {
		__critical_printf("%s: HEADER READ ERROR\n", rq_state.current_floppy->name);
		r = -EIO;
		goto ret;
	}
	if (_memcasecmp(virtual_dma_buffer, "SPADDUMP", 8)) {
		if (!_memcasecmp(virtual_dma_buffer, "SPADUSED", 8)) {
			__critical_printf("%s: FLOPPY DISK ALREADY CONTAINS DUMP\n", rq_state.current_floppy->name);
			r = -EEXIST;
		} else {
			__critical_printf("%s: FLOPPY DISK DOESN'T CONTAIN DUMP HEADER\n", rq_state.current_floppy->name);
			r = -ENOENT;
		}
		goto ret;
	}


	pos = 0;
	sec = 1;
	total_l = 0;
	get_next:
	memset(virtual_dma_buffer, 0, 512);
	l = KERNEL$CONSOLE_READ((char *)virtual_dma_buffer, 512, &pos);
	if (l) {
		total_l += l;
		r |= dump_io(sec++, 1);
		if (l == 512)
			goto get_next;
	}

	memset(virtual_dma_buffer, 0, 512);
	memcpy(virtual_dma_buffer, "SPADUSED", 8);
	*(__u32 *)&virtual_dma_buffer[12] = __32CPU2LE(total_l);
	t = time(NULL);
	*(__u64 *)&virtual_dma_buffer[16] = __64CPU2LE((__u64)t);
	if (dump_io(0, 1)) {
		__critical_printf("%s: HEADER WRITE ERROR\n", rq_state.current_floppy->name);
		r = -EIO;
		goto ret;
	}
	if (r) {
		__critical_printf("%s: DATA WRITE ERROR\n", rq_state.current_floppy->name);
		r = -EIO;
	}
	__critical_printf("%s: LOG DUMPED OK\n", rq_state.current_floppy->name);
	ret:
	io_outb(rq_state.current_floppy->io + FD_DOR, DOR_DEFAULT);
	return r;
}

