#include <SYS/TYPES.H>
#include <SPAD/BIO.H>
#include <SPAD/AC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/LIBC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <ARCH/SETUP.H>
#include <SPAD/SLAB.H>
#include <SPAD/TIMER.H>
#include <SPAD/VM.H>
#include <SPAD/DL.H>
#include <SPAD/IOCTL.H>
#include <SYS/PARAM.H>
#include <SYS/STAT.H>
#include <TIME.H>
#include <VALUES.H>
#include <SPAD/HASH.H>

#include <SPAD/BIO_KRNL.H>

/* just a soft limit --- handles are closed when you overflow it */
#define N_PARTITIONS		32

#define SEQUENTIAL_ACCESSES	8
#define SEQUENTIAL_BYTES	131072

#define BUFFER_SECTORS		(PG_SIZE / BIO_SECTOR_SIZE - 2)
#define CACHE_SECTORS		960

#define SBIORQ_ENTRIES		\
	BIORQ b;		\
	BIODESC desc;		\
	__sec_t sec;		\
	unsigned nsec;		\
	_u_off_t off;		\
	unsigned written;	\
	int flags;		\
	int file_flags;		\
	PARTITIONS *p;		\
	__sec_t part_start;	\
	PROC_TAG tag;		\
	void *sio

typedef struct {
	SBIORQ_ENTRIES;
	__u8 pad[1];
} SBIORQ_FOR_ALIGN;

typedef struct {
	SBIORQ_ENTRIES;
	__u8 pad[BIO_SECTOR_SIZE - (__offsetof(SBIORQ_FOR_ALIGN, pad) & (BIO_SECTOR_SIZE - 1))];
	__u8 buffer[BIO_SECTOR_SIZE * BUFFER_SECTORS];
} SBIORQ;

typedef struct {
	LIST_HEAD page_list;
	_u_off_t off;
	unsigned len;
	VMENTITY e;	/* inserted when len != 0 */
	BIORQ b;
	BIODESC *desc;	/* cache is loading if this is non-zero */
	WQ wq;
	PROC_TAG tag;
	PARTITIONS *p;
} CACHE;

#define N_CACHES	2

#define SBIORQ_LOCK	1

struct __partitions {
	int (*ioctl)(IOCTLRQ *, PARTITION *, IORQ *);
	struct __slhead iorqs;
	WQ iorqs_wait;
	int lock;
	int n;
	CACHE c[N_CACHES];
	HANDLE *cache_handle;
	int splx;
	int clock;
	_u_off_t last_access;
	unsigned sequential_bytes;
	unsigned char sequential_accesses;
	unsigned char previous_vspacefault;
	void *dummy_rq;
	blksize_t tmp_blksize;
	char *dev_name;
	PARTITION p[1];
};

static int cache_vm_entity;

static void BIO_CACHE_FREE(CACHE *c)
{
#if __DEBUG >= 1
	if (__unlikely(c->desc != NULL))
		KERNEL$SUICIDE("BIO_CACHE_FREE: FREEING ACTIVE CACHE");
#endif
	if (!c->len)
		return;
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_REMOVE_VM_ENTITY(&c->e);
	LOWER_SPLX(c->p->splx);
	while (!LIST_EMPTY(&c->page_list)) {
		PAGE *page = LIST_STRUCT(c->page_list.next, PAGE, hash_entry);
		DEL_FROM_LIST(&page->hash_entry);
		KERNEL$FREE_USER_PAGE(page, VM_TYPE_CACHED_UNMAPPED);
	}
	c->len = 0;
}

static DECL_AST(BIO_CACHE_LOADED, SPL_DEV, BIORQ)
{
	CACHE *c = GET_STRUCT(RQ, CACHE, b);
	RAISE_SPLX(c->p->splx);
	free(c->desc);
	c->desc = NULL;
	RAISE_SPL(SPL_VSPACE);
	KERNEL$RELEASE_IO_TAG(&c->tag);
	LOWER_SPLX(c->p->splx);
	WQ_WAKE_ALL(&c->wq);
	if (__unlikely(c->b.status < 0) || __unlikely(c->off == (_u_off_t)-1)) {
		BIO_CACHE_FREE(c);
		RETURN;
	}
	RETURN;
}

static int BIO_CACHE_SEQUENTIAL(PARTITIONS *p, unsigned long add)
{
	p->sequential_accesses++;
	p->sequential_bytes += add;
	p->last_access += add;
	if (__unlikely(p->sequential_accesses >= SEQUENTIAL_ACCESSES) &&
	    __unlikely(p->sequential_bytes >= SEQUENTIAL_BYTES)) {
		p->sequential_accesses = 0;
		p->sequential_bytes = 0;
		return 1;
	}
	return 0;
}

static void BIO_CACHE_LOAD(PARTITIONS *p, __sec_t sec, PROC *proc, CACHE *no_pageout)
{
	int i, cache_sectors;
	CACHE *c = NULL, *d;
	WQ *wq;
	if (__unlikely(!p->cache_handle))
		return;
	for (i = 0, d = p->c; i < N_CACHES; i++, d++) {
		if (__unlikely(d->desc))
			continue;
		if (d == no_pageout)
			continue;
		if (__likely(!c) || (c->len && (!d->len || c->off > d->off)))
			c = d;
	}
	if (__unlikely(!c))
		return;
	cache_sectors = CACHE_SECTORS;
#if 0
	{
		IOCTLRQ io;
		io.status = RQS_PROCESSING;
		io.h = -1;
		io.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
		io.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ;
		io.v.ptr = 0;
		io.v.len = 0;
		io.v.vspace = &KERNEL$VIRTUAL;
		io.handle = p->cache_handle;
		if (!p->ioctl(&io, &p->p[0], &KERNEL$DUMMY_IORQ) && io.status >= BIO_SECTOR_SIZE) {
			cache_sectors = io.status >> BIO_SECTOR_SIZE_BITS;
		}
		/*__debug_printf("cache sectors: %d\n", cache_sectors);*/
	}
#endif
	if (__unlikely(KERNEL$MAY_ALLOC(proc, cache_sectors << BIO_SECTOR_SIZE_BITS) != NULL))
		return;
	BIO_CACHE_FREE(c);
	RAISE_SPL(SPL_VSPACE);
	wq = KERNEL$ACQUIRE_IO_TAG(&c->tag, proc);
	LOWER_SPLX(p->splx);
	if (__unlikely(wq != NULL))
		return;
	c->desc = malloc(sizeof(BIODESC) * (cache_sectors + __SECTORS_PER_PAGE_CLUSTER_MINUS_1) >> __SECTORS_PER_PAGE_CLUSTER_BITS);
	for (i = 0; i << __SECTORS_PER_PAGE_CLUSTER_BITS < cache_sectors; i++) {
		PAGE *page = KERNEL$ALLOC_IO_PAGE(VM_TYPE_CACHED_UNMAPPED);
		if (__unlikely(!page)) {
			cache_sectors = i << __SECTORS_PER_PAGE_CLUSTER_BITS;
			break;
		}
		ADD_TO_LIST_END(&c->page_list, &page->hash_entry);
		c->desc[i].v.ptr = KERNEL$PAGE_2_PHYS(page);
		c->desc[i].v.len = __PAGE_CLUSTER_SIZE;
		c->desc[i].v.vspace = &KERNEL$PHYSICAL;
		c->desc[i].next = &c->desc[i + 1];
	}
	if (__unlikely(!i)) {
		free(c->desc);
		c->desc = NULL;
		RAISE_SPL(SPL_VSPACE);
		KERNEL$RELEASE_IO_TAG(&c->tag);
		LOWER_SPLX(p->splx);
		return;
	}
	c->desc[i - 1].next = NULL;
	c->desc[i - 1].v.len = (((cache_sectors - 1) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1) + 1) << BIO_SECTOR_SIZE_BITS;
	c->off = (_u_off_t)sec << BIO_SECTOR_SIZE_BITS;
	c->len = cache_sectors << BIO_SECTOR_SIZE_BITS;
	c->b.fn = BIO_CACHE_LOADED;
	c->b.h = p->cache_handle->handle_num & HANDLE_NUM_MASK;
	c->b.handle = p->cache_handle;
	c->b.desc = c->desc;
	c->b.sec = sec;
	c->b.nsec = cache_sectors;
	c->b.flags = BIO_READ;
	c->b.fault_sec = -1;
	c->b.proc = proc;
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_INSERT_VM_ENTITY(&c->e, proc, 0);
	LOWER_SPLX(p->splx);
	/*__debug_printf("cache load: %Lx, %x\n", sec, cache_sectors);*/
	CALL_IORQ(&c->b, KERNEL$WAKE_BIO);
}

static unsigned long BIO_CACHE_GET(PARTITION *pa, VDESC *v, _u_off_t pos, PROC *proc, WQ **wq)
{
	_u_off_t xpos = pos + v->len;
	int i;
	CACHE *c;
	CACHE *ldc;
	unsigned coff, cleft, s, total;
	PAGE *page;
	char *map;
	VBUF vbuf;
	if (__unlikely(xpos <= pos)) {
		return 0;
	}
	if (__unlikely((xpos >> BIO_SECTOR_SIZE_BITS) >= (__usec_t)pa->a.len)) {
		return 0;
	}
	total = 0;
	next_cache:
	xpos = pos + (pa->a.start << BIO_SECTOR_SIZE_BITS);
	ldc = NULL;
	for (i = 0, c = pa->p->c; i < N_CACHES; i++, c++) {
		_u_off_t lcoff = xpos - c->off;
		if (lcoff < c->len) {
			if (c->desc) {
				ldc = c;
				continue;
			}
			coff = lcoff;
			goto found;
		}
	}
	if (ldc) {
		/*__debug_printf("cache wait\n");*/
		*wq = &ldc->wq;
	}
	return total;
	found:
	RAISE_SPL(SPL_CACHE);
	KERNEL$CACHE_TOUCH_VM_ENTITY(&c->e, proc);
	LOWER_SPLX(pa->p->splx);
	if (__unlikely(!coff)) {
		BIO_CACHE_LOAD(pa->p, (c->off + c->len) >> BIO_SECTOR_SIZE_BITS, proc, c);
	}
	page = LIST_STRUCT(c->page_list.next, PAGE, hash_entry);
	cleft = c->len - coff;
	while (coff >= __PAGE_CLUSTER_SIZE) {
		coff -= __PAGE_CLUSTER_SIZE;
		next_page:
		page = LIST_STRUCT(page->hash_entry.next, PAGE, hash_entry);
#if __DEBUG >= 1
		if (page == LIST_STRUCT(&c->page_list, PAGE, hash_entry))
			KERNEL$SUICIDE("BIO_CACHE_GET: TOO FEW PAGES IN CACHE, OFFSET %"__64_format"X, LEN %X", (__u64)c->off, c->len);
#endif
	}
	map = KERNEL$MAP_PHYSICAL_PAGE(page);
	vbuf.ptr = map + coff;
	vbuf.len = MIN(__PAGE_CLUSTER_SIZE - coff, cleft);
	/* There are max. two mappings allowed per spl.
	   So we must perform this on a higher spl --- spl_fs */
	vbuf.spl = pa->p->splx;
	next_xfer:
	RAISE_SPL(SPL_VSPACE);
	s = v->vspace->op->vspace_put(v, &vbuf);
	pos += s;
	total += s;
	cleft -= s;
	if (__likely(s != 0) && __likely(v->len != 0) && __likely(KERNEL$LOCKUP_LEVEL < LOCKUP_LEVEL_ONE_PASS)) {
		if (__unlikely(s < vbuf.len)) {
			vbuf.ptr = (char *)vbuf.ptr + s;
			vbuf.len -= s;
			goto next_xfer;
		}
		KERNEL$UNMAP_PHYSICAL_BANK(map);
		if (__likely(cleft)) {
			coff = 0;
			goto next_page;
		} else {
			goto next_cache;
		}
	}
	KERNEL$UNMAP_PHYSICAL_BANK(map);
	return total;
}

static int CACHE_CHECKMAP(VMENTITY *e)
{
	CACHE *c = GET_STRUCT(e, CACHE, e);
	LOWER_SPLX(c->p->splx);
	return 0;
}

static void CACHE_WRITE(VMENTITY *e, PROC *p, int trashing)
{
	CACHE *c = GET_STRUCT(e, CACHE, e);
	LOWER_SPLX(c->p->splx);
}

static int BIO_CACHE_INVALIDATE(CACHE *c);

static int CACHE_SWAPOUT(VMENTITY *e)
{
	CACHE *c = GET_STRUCT(e, CACHE, e);
	LOWER_SPLX(c->p->splx);
	return BIO_CACHE_INVALIDATE(c);
}

static int BIO_CACHE_INVALIDATE(CACHE *c)
{
	if (__likely(!c->desc)) {
		BIO_CACHE_FREE(c);
		return 0;
	} else {
		c->off = (_u_off_t)-1;
		return 1;
	}
}

__COLD_ATTR__ static void BIO_CACHE_INVALIDATE_ALL(PARTITIONS *p)
{
	int i;
	for (i = 0; i < N_CACHES; i++) {
		BIO_CACHE_INVALIDATE(&p->c[i]);
	}
}

__COLD_ATTR__ void BIO$SET_CACHE(PARTITIONS *p, HANDLE *h)
{
	p->cache_handle = h;
}

__COLD_ATTR__ void BIO$DISABLE_CACHE(PARTITIONS *p)
{
	int i;
	int spl = KERNEL$SPL;
	p->cache_handle = NULL;
	RAISE_SPL(SPL_DEV);
	for (i = 0; i < N_CACHES; i++) {
		if (p->c[i].desc) {
			KERNEL$SLEEP(1);
			i = -1;
		}
	}
	for (i = 0; i < N_CACHES; i++) {
		BIO_CACHE_FREE(&p->c[i]);
	}
	LOWER_SPLX(spl);
}

static const VMENTITY_T cache_calls = { CACHE_CHECKMAP, CACHE_WRITE, CACHE_SWAPOUT, "BIOSTREAM" };

__COLD_ATTR__ static void KILL_PARTITION_HANDLES(PARTITION *pa)
{
	while (!XLIST_EMPTY(&pa->list)) {
		HANDLE *h = LIST_STRUCT(pa->list.next, HANDLE, fnode_entry);
		KERNEL$DETACH_HANDLE(h);
	}
	BIO_CACHE_INVALIDATE_ALL(pa->p);
}

__COLD_ATTR__ static int DUMMY_IOCTL(IOCTLRQ *rq, PARTITION *part, IORQ *rq_to_queue)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != part->p->splx)) KERNEL$SUICIDE("DUMMY_IOCTL AT SPL %08X (WANTED %08X)", KERNEL$SPL, part->p->splx);
#endif
	return -1;
}

__COLD_ATTR__ int BIO$ALLOC_PARTITIONS(PARTITIONS **pp, void *dev, int splx, __sec_t dev_blocx, int (*ioctl)(IOCTLRQ *, PARTITION *, IORQ *), char *dev_name)
{
	int i;
	PARTITIONS *p;
	MALLOC_REQUEST mrq;
	int r;
	mrq.size = sizeof(PARTITIONS) + N_PARTITIONS * sizeof(PARTITION);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) return mrq.status;
	p = mrq.ptr;
	p->ioctl = ioctl ? ioctl : DUMMY_IOCTL;
	p->n = N_PARTITIONS + 1;
	p->splx = splx;
	p->clock = 1;
	p->last_access = 0;
	p->sequential_accesses = 0;
	p->sequential_bytes = 0;
	p->previous_vspacefault = 0;
	p->dev_name = dev_name;
	for (i = 0; i < p->n; i++) {
		p->p[i].dev = dev;
		memset(&p->p[i].a, 0, sizeof(PARTITION_ATTR));
		p->p[i].p = p;
		INIT_XLIST(&p->p[i].list);
	}
	for (i = 0; i < N_CACHES; i++) {
		CACHE *c = &p->c[i];
		INIT_LIST(&c->page_list);
		c->len = 0;
		c->desc = NULL;
		WQ_INIT(&c->wq, "BIO$CACHE_WQ");
		c->p = p;
		CACHE_CONSTRUCT_VM_ENTITY(&c->e);
		c->e.type = cache_vm_entity;
	}
	p->p[0].a.len = dev_blocx;
	KERNEL$SLAB_INIT(&p->iorqs, sizeof(SBIORQ), BIO_SECTOR_SIZE, VM_TYPE_WIRED_MAPPED, NULL, NULL, "BIO$SBIORQ");
	if (__unlikely(r = KERNEL$SLAB_RESERVE(&p->iorqs, 2))) {
		KERNEL$SLAB_DESTROY(&p->iorqs);
		free(p);
		return r;
	}
	p->dummy_rq = __slalloc(&p->iorqs);
	if (__unlikely(!p->dummy_rq))
		KERNEL$SUICIDE("BIO$ALLOC_PARTITIONS: CAN'T ALLOCATE DUMMY IO REQUEST");
	WQ_INIT(&p->iorqs_wait, "BIO$IORQS_WAIT");
	p->lock = 0;
	*pp = p;
	return 0;
}

	/* must not block & must be called at SPL_BIO */
__COLD_ATTR__ void BIO$SET_PARTITIONS_SIZE(PARTITIONS *p, __sec_t dev_blocx)
{
	int i;
	p->p[0].a.len = dev_blocx;
	for (i = 0; i < p->n; i++) KILL_PARTITION_HANDLES(&p->p[i]);
}

__COLD_ATTR__ void BIO$FREE_PARTITIONS(PARTITIONS *p)
{
	int i;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_DEV);
	BIO$DISABLE_CACHE(p);
	if (__unlikely(p->lock))
		KERNEL$SUICIDE("BIO$FREE_PARTITIONS: IO LOCK LEAKED: %d", p->lock);
	for (i = 0; i < p->n; i++) if (__unlikely(!XLIST_EMPTY(&p->p[i].list)))
		KERNEL$SUICIDE("BIO$FREE_PARTITIONS: PARTITION (%016"__sec_t_format"X, %016"__sec_t_format"X) IS STILL USED", p->p[i].a.start, p->p[i].a.len);
	__slow_slfree(p->dummy_rq);
	while (__unlikely(!KERNEL$SLAB_EMPTY(&p->iorqs))) KERNEL$SLEEP(1);
	KERNEL$SLAB_DESTROY(&p->iorqs);
	WQ_WAKE_ALL(&p->iorqs_wait);
	free(p);
	LOWER_SPLX(spl);
}

int BIO$IS_ROOT_PARTITION(PARTITION *p)
{
	return p == &p->p->p[0];
}

__COLD_ATTR__ void BIO$ROOT_PARTITION(HANDLE *h, PARTITIONS *p)
{
	h->fnode = &p->p[0];
}

__COLD_ATTR__ void *BIO$LOOKUP_PARTITION(HANDLE *h, char *str, int open_flags)
{
	PARTITION *p = (PARTITION *)h->fnode;
	PARTITIONS *pp;
	PARTITION *ppp;
	PARTITION_ATTR a;
	long val;
	char *mod;
	memcpy(&a, &p->a, sizeof(PARTITION_ATTR));
	if (__unlikely(str[0] != '^')) return __ERR_PTR(-ENOTDIR);
	str++;
	if (__unlikely(!_memcasecmp(str, "PART=", 5))) {
		str += 5;
		if (__unlikely(!(mod = strchr(str, ',')))) return __ERR_PTR(-EBADSYN);
		if (__unlikely(__get_64_number(str, mod, 0, (__s64 *)&a.start))) return __ERR_PTR(-EBADSYN);
		if (__unlikely(__get_64_number(mod + 1, strchr(mod + 1, 0), 0, (__s64 *)&a.len))) return __ERR_PTR(-EBADSYN);
		if (__unlikely(a.start + a.len > p->a.len) || __unlikely(a.start + a.len < a.start)) return __ERR_PTR(-ERANGE);
		a.start += p->a.start;
	} else if (__unlikely(!_memcasecmp(str, "FATOFFSET=", 10))) {
		str += 10;
		if (__unlikely(__get_number(str, strchr(str, 0), 0, &val))) return __ERR_PTR(-EBADSYN);
		a.fat_offset = val;
	} else if (__unlikely(!_memcasecmp(str, "TYPE=", 5))) {
		str += 5;
		if (__unlikely(__get_number(str, strchr(str, 0), 0, &val))) return __ERR_PTR(-EBADSYN);
		if ((unsigned long)val >= 256) return __ERR_PTR(-ERANGE);
		a.type = val;
	} else {
		return __ERR_PTR(-EBADMOD);
	}
	pp = p->p;
	for (ppp = pp->p; ppp < pp->p + pp->n; ppp++)
		if (!memcmp(&ppp->a, &a, sizeof(PARTITION_ATTR))) goto ok2;
	for (ppp = pp->p + 1; ppp < pp->p + pp->n; ppp++)
		if (!ppp->a.start && !ppp->a.len && XLIST_EMPTY(&ppp->list)) goto ok;
	for (ppp = pp->p + 1; ppp < pp->p + pp->n; ppp++)
		if (XLIST_EMPTY(&ppp->list)) goto ok;
	ppp = pp->p + pp->clock;
	if (__unlikely(++pp->clock >= pp->n)) pp->clock = 1;
	KILL_PARTITION_HANDLES(ppp);
	ok:
	memcpy(&ppp->a, &a, sizeof(PARTITION_ATTR));
	ok2:
	h->fnode = ppp;
	return NULL;
}

__COLD_ATTR__ void *BIO$INSTANTIATE_PARTITION(HANDLE *h, IORQ *rq, int open_flags)
{
	PARTITION *p = h->fnode;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != p->p->splx))
		KERNEL$SUICIDE("BIO$INSTANTIATE_PARTITION AT SPL %08X (WANTED %08X)", KERNEL$SPL, p->p->splx);
#endif
	if (__likely(h->file_addrspace != NULL)) ADD_TO_XLIST(&p->list, &h->fnode_entry);
	else VOID_LIST_ENTRY(&h->fnode_entry);
	BIO_CACHE_INVALIDATE_ALL(p->p);
	return NULL;
}

__COLD_ATTR__ void BIO$DETACH(HANDLE *h)
{
	DEL_FROM_LIST(&h->fnode_entry);
}

__COLD_ATTR__ void BIO$FAULT(BIORQ *rq, __sec_t sec)
{
	if (__likely(sec != -1)) {
		if (__unlikely(sec < rq->partition)) KERNEL$SUICIDE("BIO$FAULT: SECTOR %016"__64_format"X, PARTITION %016"__64_format"X", (__u64)sec, (__u64)rq->partition);
		sec -= rq->partition;
	}
	if (__likely(!rq->status)) {
		rq->status = -EVSPACEFAULT;
		f: rq->fault_sec = sec;
	} else {
		if (__likely(rq->status == -EVSPACEFAULT) && __unlikely((__s64)sec < (__s64)rq->fault_sec)) goto f;
	}
}

__COLD_ATTR__ void BIO$ERROR(BIORQ *rq, __sec_t sec)
{
	if (__likely(sec != -1)) {
		if (__unlikely(sec < rq->partition)) sec = -1;
		else sec -= rq->partition;
	}
	if (__likely(rq->status != -EIO) || __unlikely((__s64)sec < (__s64)rq->fault_sec)) rq->fault_sec = sec;
	rq->status = -EIO;
}

static void FREE_SB(SBIORQ *sb, PARTITIONS *p)
{
	WQ_WAKE_ALL(&p->iorqs_wait);
	RAISE_SPL(SPL_VSPACE);
	KERNEL$RELEASE_IO_TAG(&sb->tag);
	LOWER_SPLX(p->splx);
	__slfree(sb);
}

#define RW	0
#define AS	0
#include "SBIO.I"
#undef AS
#define AS	1
#include "SBIO.I"
#undef AS
#undef RW
#define RW	1
#define AS	0
#include "SBIO.I"
#undef AS
#define AS	1
#include "SBIO.I"
#undef AS
#undef RW

__COLD_ATTR__ void BIO$NO_MEDIA_STAT(struct stat *stat)
{
	stat->st_mode = 0600 | S_IFBLK;
	stat->st_nlink = 1;
	stat->st_atime = stat->st_mtime = stat->st_ctime = time(NULL);
	stat->st_blksize = 2048;
}

static __finline__ void VFS_FILLSTAT(struct stat *stat, PARTITION *pa)
{
#if 0
	char *c;
	ino_t hash;
#endif
	stat->st_mode = 0600 | S_IFBLK;
	stat->st_nlink = 1;
	stat->st_atime = stat->st_mtime = stat->st_ctime = time(NULL);
	if (pa->a.len != -1) {
		stat->st_size = pa->a.len << BIO_SECTOR_SIZE_BITS;
		stat->st_blocks = pa->a.len;
	}
#if 0
	hash = 0;
	c = pa->p->dev_name;
	quickcasehash(c, *c, hash);
	stat->st_dev = hash;
	c = (char *)&pa->start;
	quickcasehash(c, c < (char *)(&pa->start + 1), hash);
	c = (char *)&pa->len;
	quickcasehash(c, c < (char *)(&pa->len + 1), hash);
	stat->st_ino = stat->st_rdev = hash;
#endif
	stat->st_blksize = 1 << __BSR(pa->p->tmp_blksize);
}

#define VFS_FN_SPL	SPL_DEV

#include <SPAD/VFS_FN.H>

static AST_STUB FLUSH_FINISHED;

DECL_IOCALL(BIO$IOCTL, SPL_DEV, IOCTLRQ)
{
	PARTITION *pa;
	PARTITIONS *p;
	int r;
	HANDLE *h = RQ->handle;
	RAISE_SPL(SPL_TOP);
	if (__unlikely(h->op->ioctl != BIO$IOCTL)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	pa = h->fnode;
	p = pa->p;
	LOWER_SPLX(p->splx);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, p->splx);
	r = p->ioctl(RQ, pa, (IORQ *)RQ);
	if (__likely(!r)) {
		switch (RQ->ioctl) {
			case IOCTL_BIO_PHYSICAL_BLOCKSIZE: {
				unsigned n;
				if (__unlikely(RQ->status < 0)) break;
				if (!(unsigned)pa->a.start) break;
				n = BIO_SECTOR_SIZE << __BSF((unsigned)pa->a.start);
				if (__unlikely(!n)) break;
				if (n < RQ->status) RQ->status = n;
				break;
			}
		}
		RETURN_AST(RQ);
	}
	if (__unlikely(r >= 0)) {
		RETURN;
	}
	switch (RQ->ioctl) {
		case IOCTL_STAT: {
			IOCTLRQ io;
			io.status = RQS_PROCESSING;
			io.h = -1;
			io.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
			io.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ;
			io.v.ptr = 0;
			io.v.len = 0;
			io.v.vspace = &KERNEL$VIRTUAL;
			io.handle = h;
			r = pa->p->ioctl(&io, pa, (IORQ *)RQ);
			if (__likely(!r) && __likely(io.status > 0)) p->tmp_blksize = io.status;
			else if (__unlikely(r > 0)) RETURN;
			else p->tmp_blksize = OPTIMAL_IO_SIZE;
			LOWER_SPL(SPL_DEV);
			r = VFS_STAT(RQ, pa);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
		}
		case IOCTL_LSEEK: {
			LOWER_SPL(SPL_DEV);
			r = VFS_LSEEK(RQ, 0);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (r == 2) RETURN;
			DO_PAGEIN(RQ, &RQ->v, PF_RW);
		}
		case IOCTL_FSYNC: {
			SBIORQ *sb;
			WQ *wq;
			BIO_CACHE_INVALIDATE_ALL(p);
			if (RQ->param & PARAM_FSYNC_ASYNC) {
				/* disks sync their cache automatically, no
				   need to ask them */
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(sb = __slalloc(&p->iorqs)))) {
				WQ_WAIT_F(&p->iorqs_wait, RQ);
				RETURN;
			}
			RAISE_SPL(SPL_VSPACE);
			if (__unlikely((wq = KERNEL$ACQUIRE_IO_TAG(&sb->tag, h->name_addrspace)) != NULL)) {
				WQ_WAIT_F(wq, RQ);
				LOWER_SPLX(p->splx);
				__slow_slfree(sb);
				RETURN;
			}
			LOWER_SPLX(p->splx);
			sb->flags = 0;
			sb->sio = RQ;
			sb->p = p;
			sb->b.h = h->handle_num & HANDLE_NUM_MASK;
			sb->b.handle = h;
			sb->b.flags = BIO_FLUSH;
			sb->b.sec = 0;
			sb->b.nsec = 1;
			sb->b.proc = sb->tag.proc;
			sb->b.fault_sec = -1;
			sb->b.fn = FLUSH_FINISHED;
			RETURN_IORQ_CANCELABLE(&sb->b, KERNEL$WAKE_BIO, RQ);
		}
		case IOCTL_FCNTL_GETFL:
		case IOCTL_FCNTL_SETFL: {
			LOWER_SPL(SPL_DEV);
			if (__likely(!VFS_GETSETFL(RQ))) RETURN_AST(RQ);
			RETURN;
		}
		case IOCTL_BIO_PHYSICAL_BLOCKSIZE:
		case IOCTL_BIO_PARTITION_BLOCKSIZE: {
			RQ->status = 512;
			RETURN_AST(RQ);
		}
		case IOCTL_BIO_GET_FAT_OFFSET: {
			r = KERNEL$PUT_IOCTL_STRUCT(RQ, &pa->a.fat_offset, sizeof(__u32));
			if (__unlikely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static DECL_AST(FLUSH_FINISHED, SPL_DEV, BIORQ)
{
	SBIORQ *sb = LIST_STRUCT(RQ, SBIORQ, b);
	IOCTLRQ *sio = sb->sio;
	PARTITIONS *p = sb->p;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), sio);
	RAISE_SPLX(p->splx);
	sio->status = sb->b.status;
	RAISE_SPL(SPL_VSPACE);
	KERNEL$RELEASE_IO_TAG(&sb->tag);
	LOWER_SPLX(p->splx);
	__slfree(sb);
	WQ_WAKE_ALL(&p->iorqs_wait);
	RETURN_AST(sio);
}

void BIO$CHECK_REQUEST(const char *msg, BIORQ *rq)
{
	BIODESC *desc;
	unsigned long len;
	if (__unlikely(rq->flags & BIO_FLUSH)) {
		return;
	}
	if (__unlikely(rq->nsec <= 0)) {
		unsigned long off = 0;
		const char *str = KERNEL$DL_GET_SYMBOL_NAME(rq->fn, &off, 0);
		KERNEL$SUICIDE("BIO$CHECK_REQUEST(%s): INVALID NUMBER OF SECTORS %X, SECTOR %"__sec_t_format"X, AST %s+%lX", msg, rq->nsec, rq->sec, str ? str : "?", off);
	}
	len = 0;
	desc = rq->desc;
	do {
		if (__unlikely(((unsigned)desc->v.ptr | (unsigned)desc->v.len) & (BIO_SECTOR_SIZE - 1)) || __unlikely(!desc->v.len)) {
			unsigned long off = 0;
			const char *str = KERNEL$DL_GET_SYMBOL_NAME(rq->fn, &off, 0);
			KERNEL$SUICIDE("BIO$CHECK_REQUEST(%s): UNALIGNED DESCRIPTOR: %"__64_format"X, %lX, AST %s+%lX", msg, (__u64)desc->v.ptr, desc->v.len, str ? str : "?", off);
		}
		len += desc->v.len;
	} while ((desc = desc->next));
	if (__unlikely(len != ((unsigned long)rq->nsec << BIO_SECTOR_SIZE_BITS))) {
		unsigned long off = 0;
		const char *str = KERNEL$DL_GET_SYMBOL_NAME(rq->fn, &off, 0);
		KERNEL$SUICIDE("BIO$CHECK_REQUEST(%s): LENGTH MISMATCH, LEN %lX, SECTORS %X, SECTOR %"__sec_t_format"X, AST %s+%lX", msg, len, rq->nsec, rq->sec, str ? str : "?", off);
	}
}

__COLD_ATTR__ DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	int r;
	r = KERNEL$CACHE_REGISTER_VM_TYPE(&cache_vm_entity, &cache_calls);
	if (r) {
		RQ->status = r;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER CACHE VM ENTITY");
		RETURN_AST(RQ);
	}
	RQ->status = 0;
	RETURN_AST(RQ);
}

__COLD_ATTR__ DECL_IOCALL(DLL$UNLOAD, SPL_DEV, DLINITRQ)
{
	KERNEL$CACHE_UNREGISTER_VM_TYPE(cache_vm_entity);
	RQ->status = 0;
	RETURN_AST(RQ);
}
