#include <SYS/TYPES.H>
#include <SPAD/BIO.H>
#include <SPAD/AC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/LIBC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <ARCH/SETUP.H>
#include <SPAD/TIMER.H>
#include <SPAD/VM.H>
#include <TIME.H>
#include <SPAD/SCHED.H>
#include <SPAD/HASH.H>

#include <SPAD/BIO_KRNL.H>

#define IOSCHED_LATENCY		(JIFFIES_PER_SECOND)

struct __bioqueue {
	LIST_HEAD queue;
	BIORQ *next;
	__usec_t head;
	int iosched;
};

void BIOQUE$ACCOUNT_IOSCHED(PROC *p, BIOQUEUE *q, sched_unsigned start, sched_unsigned now)
{
	KERNEL$ACCOUNT_IOSCHED(p, q->iosched, start, now);
}

__COLD_ATTR__ int BIOQUE$ALLOC_QUEUE(BIOQUEUE **qq)
{
	int r;
	BIOQUEUE *b;
	MALLOC_REQUEST mrq;
	mrq.size = sizeof(BIOQUEUE);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		return mrq.status;
	}
	b = mrq.ptr;
	INIT_LIST(&b->queue);
	b->next = NULL;
	b->head = 0;
	if (__unlikely((r = KERNEL$ALLOC_IOSCHED()) < 0)) {
		free(b);
		return r;
	}
	b->iosched = r;
	*qq = b;
	return 0;
}

__COLD_ATTR__ void BIOQUE$FREE_QUEUE(BIOQUEUE *q)
{
	if (__unlikely(!LIST_EMPTY(&q->queue))) KERNEL$SUICIDE("BIOQUE$FREE_QUEUE: QUEUE IS NOT EMPTY");
	if (__unlikely(q->next != NULL)) KERNEL$SUICIDE("BIOQUE$FREE_QUEUE: QUEUE HAS NEXT REQUEST FIELD NON-EMPTY");
	KERNEL$FREE_IOSCHED(q->iosched);
	free(q);
}

#define Q_ENTRY(RQ) ((LIST_ENTRY *)(void *)&RQ->tmp1)
#define Q_RQ(LIST_ENT) LIST_STRUCT(LIST_ENT, BIORQ, tmp1)

static sched_signed COMPARE_SEC(BIORQ *rq1, BIORQ *rq2, __usec_t *head)
{
	/*sched_signed h = (rq1->flags & BIO_WRITE) - (rq2->flags & BIO_WRITE);
	if (__unlikely(h)) return h;*/
	if ((__usec_t)rq1->sec - *head < (__usec_t)rq2->sec - *head) return -1;
	else return 1;
}

#define CAN_PREEMPT(b, rq1, rq2, label1, label2)			\
do {									\
	sched_signed s_ = KERNEL$COMPARE_IOSCHED((rq1)->proc, (b)->iosched, (rq2)->proc, (b)->iosched);							\
	if (!s_) {							\
		if (__unlikely((rq1)->enq_time - (rq2)->enq_time + IOSCHED_LATENCY >= 2 * IOSCHED_LATENCY)) {						\
			if ((jiffies_lo_t)((rq1)->enq_time - (rq2)->enq_time) < 0) goto label1;								\
			else goto label2;				\
		}							\
		s_ = COMPARE_SEC(rq1, rq2, &(b)->head);			\
	}								\
	if (s_ >= 0) goto label2;					\
	goto label1;							\
} while (0)

void BIOQUE$ENQUEUE_REQUEST(BIOQUEUE *b, BIORQ *rq)
{
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$ENQUEUE_REQUEST 1", b, rq);
#endif
	if (!b->next && __likely(LIST_EMPTY(&b->queue))) {
		b->next = rq;
	} else {
		if (__unlikely(!b->next)) goto label2;
		CAN_PREEMPT(b, rq, b->next, label1, label2);
		if (0) {
			label1:
			ADD_TO_LIST(&b->queue, Q_ENTRY(b->next));
			b->next = rq;
		} else {
			label2:
			ADD_TO_LIST_END(&b->queue, Q_ENTRY(rq));
		}
	}
	rq->enq_time = KERNEL$GET_JIFFIES_LO();
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$ENQUEUE_REQUEST 2", b, NULL);
#endif
}

int BIOQUE$QUEUE_EMPTY(BIOQUEUE *b)
{
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$QUEUE_EMPTY", b, NULL);
#endif
	return (!b->next) & LIST_EMPTY(&b->queue);
}

BIORQ *BIOQUE$DEQUEUE(BIOQUEUE *q)
{
	BIORQ *RQ;
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$DEQUEUE 1", q, NULL);
#endif
	if (__likely(q->next != NULL)) {
		deq_next:
		RQ = q->next;
		q->next = NULL;
		return RQ;
	}
#if __DEBUG >= 1
	if (__unlikely(LIST_EMPTY(&q->queue)))
		KERNEL$SUICIDE("BIOQUE$DEQUEUE: QUEUE EMPTY");
#endif
	BIOQUE$ACK_DEQUEUE(q);
	goto deq_next;
}

void BIOQUE$SET_HEAD(BIOQUEUE *b, __usec_t sec)
{
	b->head = sec;
}

void BIOQUE$ACK_DEQUEUE(BIOQUEUE *b)
{
	LIST_ENTRY *le;
	BIORQ *best;
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$ACK_DEQUEUE 1", b, NULL);
#endif
	if (__unlikely(b->next != NULL)) return;
	le = b->queue.next;
	if (le == (LIST_ENTRY *)(void *)&b->queue) return;
	best = Q_RQ(le);
	for (le = le->next; le != (LIST_ENTRY *)(void *)&b->queue; le = le->next) {
		CAN_PREEMPT(b, Q_RQ(le), best, label1, label2);
		label1:
		best = Q_RQ(le);
		label2:;
	}
	DEL_FROM_LIST(Q_ENTRY(best));
	b->next = best;
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$ACK_DEQUEUE 2", b, NULL);
#endif
}

void BIOQUE$REQUEUE_DEQUEUED_REQUEST(BIOQUEUE *b, BIORQ *rq)
{
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$REQUEUE_DEQUEUED_REQUEST 1", b, rq);
#endif
	if (__unlikely(b->next != NULL)) {
		BIOQUE$ENQUEUE_REQUEST(b, rq);
		return;
	}
	b->next = rq;
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$REQUEUE_DEQUEUED_REQUEST 2", b, NULL);
#endif
}

BIORQ *BIOQUE$DEQUEUE_FLUSHES(BIOQUEUE *b)
{
	unsigned long ret = 0, *last = &ret;
	LIST_ENTRY *le;
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$DEQUEUE_FLUSHES 1", b, NULL);
#endif
	for (le = b->queue.next; le != (LIST_ENTRY *)(void *)&b->queue; le = le->next) {
		BIORQ *q = Q_RQ(le);
		if (__unlikely(q->flags & BIO_FLUSH)) {
			LIST_ENTRY *lee = le->prev;
			DEL_FROM_LIST(le);
			le = lee;
			__barrier();	/* warning: ANSI C aliasing */
			*last = (unsigned long)q;
			q->tmp1 = 0;
			last = &q->tmp1;
		}
	}
	if (b->next && b->next->flags & BIO_FLUSH) {
		*last = (unsigned long)b->next;
		b->next->tmp1 = 0;
		last = &b->next->tmp1;
		b->next = NULL;
	}
#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("BIOQUE$DEQUEUE_FLUSHES 2", b, NULL);
#endif
	return (BIORQ *)ret;
}

BIORQ *BIOQUE$PROBE_QUEUE(BIOQUEUE *b)
{
	if (b->next) return b->next;
	if (__likely(LIST_EMPTY(&b->queue))) return NULL;
	BIOQUE$ACK_DEQUEUE(b);
	return b->next;
}

long BIOQUE$COMPARE_REQUESTS(BIORQ *rq1, int s1, BIORQ *rq2, int s2)
{
	sched_signed s;
	if (__unlikely(s1 < 0)) rq1 = BIO_COMPARE_UNSPECIFIED;
	if (__unlikely(s2 < 0)) rq2 = BIO_COMPARE_UNSPECIFIED;
	if (__unlikely(rq1 == BIO_COMPARE_UNSPECIFIED)) {
		if (rq2 == BIO_COMPARE_UNSPECIFIED) return 0;
		return -1;
	}
	if (__unlikely(rq2 == BIO_COMPARE_UNSPECIFIED)) return 1;
	s = KERNEL$COMPARE_IOSCHED(rq1->proc, s1, rq2->proc, s2);
	if (!s) {
		if (__unlikely(rq1->enq_time - rq2->enq_time + IOSCHED_LATENCY >= 2 * IOSCHED_LATENCY)) {
			return (jiffies_lo_t)(rq1->enq_time - rq2->enq_time);
		}
	}
	return s;
}

int BIOQUE$IOSCHED(BIOQUEUE *b)
{
	return b->iosched;
}

void BIOQUE$CHECK_QUEUE(const char *msg, BIOQUEUE *b, BIORQ *rq)
{
	/* !!! FIXME: check request */
	LIST_ENTRY *le;
	if (rq && __unlikely(rq->flags & BIO_FLUSH) && __unlikely(rq->nsec != 1))
		KERNEL$SUICIDE("BIOQUE$CHECK_QUEUE(%s): FLUSH REQUEST HAS INVALID PARAMETERS, FLAGS %X, SEC %"__64_format"X, NSEC %X", msg, rq->flags, (__u64)rq->sec, rq->nsec);
	if (rq && __unlikely(rq == b->next))
		KERNEL$SUICIDE("BIOQUE$CHECK_QUEUE(%s): REQUEST ALREADY PREPARED AS NEXT, FLAGS %08X", msg, rq->flags);
	for (le = b->queue.next; le != (LIST_ENTRY *)(void *)&b->queue; le = le->next) {
		BIORQ *q = Q_RQ(le);
		if (__unlikely(q == rq))
			KERNEL$SUICIDE("BIOQUE$CHECK_QUEUE(%s): REQUEST ALREADY ON QUEUE, FLAGS %08X", msg, rq->flags);
		if (__unlikely(q == b->next))
			KERNEL$SUICIDE("BIOQUE$CHECK_QUEUE(%s): QUEUED REQUEST ALREADY IN NEXT, FLAGS %08X", msg, q->flags);
		if (__unlikely(le->next->prev != le))
			KERNEL$SUICIDE("BIOQUE$CHECK_QUEUE(%s): NEXT->PREV POINTER DAMAGED, %p -> %p -> %p", msg, le, le->next, le->next->prev);
		if (__unlikely(le->prev->next != le))
			KERNEL$SUICIDE("BIOQUE$CHECK_QUEUE(%s): PREV->NEXT POINTER DAMAGED, %p -> %p -> %p", msg, le, le->prev, le->prev->next);
	}
}

