#include <SPAD/SYNC.H>
#include <SPAD/AC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/LIBC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SLAB.H>
#include <SPAD/IOCTL.H>
#include <SPAD/TIMER.H>
#include <STRING.H>
#include <UNISTD.H>
#include <VALUES.H>
#include <STDLIB.H>
#include <SYS/STAT.H>
#include <SPAD/BIO_KRNL.H>

#include <KERNEL/GETHSIZE.I>

#include "RAID.H"

typedef struct {
	unsigned char type;
	char *name;
	__sec_t (*init)(RAID *raid);
	const HANDLE_OPERATIONS *op;
} RAID_TYPE;

static const RAID_TYPE TYPES[] = {
	{ TYPE_LINEAR,	"LINEAR",	RAIDLN_INIT,	&RAIDLN_OPERATIONS },
	{ TYPE_0,	"0",		RAID0_INIT,	&RAID0_OPERATIONS },
	{ 0,		NULL,		NULL,		NULL }
};

static int RAID_UNLOAD(void *p, void **release, const char * const argv[]);

static int RAID_IOCTL(IOCTLRQ *rq, PARTITION *pa, IORQ *rq_to_queue)
{
	RAID *raid = pa->dev;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_RAID)))
		KERNEL$SUICIDE("RAID_IOCTL AT SPL %08X", KERNEL$SPL);
#endif
	switch (rq->ioctl) {
		case IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE:
			switch (rq->param) {
				case PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ:
					rq->status = raid->read_request_size;
					break;
				case PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_WRITE:
					rq->status = raid->write_request_size;
					break;
				default:
					rq->status = -ENOOP;
					break;
			}
			return 0;
		case IOCTL_BIO_PHYSICAL_BLOCKSIZE:
			rq->status = raid->physical_block_size << BIO_SECTOR_SIZE_BITS;
			return 0;
		default:
			return -1;
	}
}

static void RAID_INIT_ROOT(HANDLE *h, void *f_)
{
	const RAID_TYPE *t;
	RAID *raid = f_;
	BIO$ROOT_PARTITION(h, raid->p);
	for (t = TYPES; t->name; t++) if (t->type == raid->type) goto have_type;
	KERNEL$SUICIDE("RAID_INIT_ROOT: INVALID TYPE %d", t->type);
	have_type:
	h->op = t->op;
	BIO$SET_CACHE(raid->p, h);
}

int main(int argc, const char * const argv[])
{
	const char * const *arg = argv;
	int state = 0;
	const char *val;
	char **dep_vec;
	int r;
	int i, j;
	__sec_t sectors;
	RAID *raid;
	int n_devices;
	char *dev_name;
	const RAID_TYPE *t;
	union {
		MALLOC_REQUEST mrq;
		OPENRQ op;
		CHHRQ ch;
	} u;
	char *raid_type = NULL;
	int stripe = 0;
	int ro = 0;
	static const struct __param_table params[4] = {
		"RAID", __PARAM_STRING, 1, __MAX_STR_LEN,
		"STRIPE", __PARAM_INT, BIO_SECTOR_SIZE, MAXINT,
		"RO", __PARAM_BOOL, ~0, BIO_WRITE,
		NULL, 0, 0, 0,
	};
	void *vars[4];
	vars[0] = &raid_type;
	vars[1] = &stripe;
	vars[2] = &ro;
	vars[3] = NULL;
	u.mrq.size = sizeof(RAID);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		r = u.mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: OUT OF MEMORY");
		goto ret0;
	}
	raid = u.mrq.ptr;
	n_devices = 0;
	while (__parse_params(&arg, &state, params, vars, NULL, NULL, &val)) {
		if (!val) goto bads;
		/* we need one more slot than the number of devices */
		u.mrq.size = sizeof(RAID) + n_devices * sizeof(RAID_DEVICE);
		SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
		if (u.mrq.status < 0) {
			r = u.mrq.status;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: OUT OF MEMORY");
			goto ret1;
		}
		memcpy(u.mrq.ptr, raid, sizeof(RAID) + n_devices * sizeof(RAID_DEVICE));
		free(raid);
		raid = u.mrq.ptr;
		*(const char **)&raid->devices[n_devices] = val;
		n_devices++;
	}
	if (!raid_type || n_devices < 2 || stripe & (BIO_SECTOR_SIZE - 1) || stripe >= MAXINT / n_devices) {
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret1;
	}
	stripe >>= BIO_SECTOR_SIZE_BITS;
	dev_name = *(char **)&raid->devices[--n_devices];
	if (strlen(dev_name) >= sizeof(raid->dev_name)) goto bads;
	strcpy(raid->dev_name, dev_name);
	raid->n_devices = n_devices;
	raid->ro = ro;
	raid->read_request_size = 0;
	raid->write_request_size = 0;
	raid->physical_block_size = 0;
	for (t = TYPES; t->name; t++) if (!_strcasecmp(t->name, raid_type)) goto have_type;
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: UNKNOWN TYPE");
	r = -EBADSYN;
	goto ret1;

	have_type:
	raid->type = t->type;
	raid->stripe = stripe;
	raid->bsf_stripe = -1;
	if (raid->stripe && !(raid->stripe & (raid->stripe - 1))) raid->bsf_stripe = __BSF(raid->stripe);
	raid->stripe_mod = stripe * n_devices;
	raid->bsf_stripe_mod = -1;
	if (raid->stripe_mod && !(raid->stripe_mod & (raid->stripe_mod - 1))) raid->bsf_stripe_mod = __BSF(raid->stripe_mod);

	for (i = 0; i < n_devices; i++) {
		__u64 size;
		int h;
		const char *path = *(const char **)&raid->devices[i];

		u.op.path = path;
		u.op.flags = (!ro ? O_RDONLY : O_RDWR) | O_DIRECT;
		u.op.cwd = KERNEL$CWD();
		SYNC_IO_CANCELABLE(&u.op, KERNEL$OPEN);
		if (u.op.status < 0) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: CAN'T OPEN %s: %s", u.op.path, strerror(-u.op.status));
			r = u.op.status;
			cl_2:
			for (j = 0; j < i; j++) close(raid->devices[j].h);
			goto ret1;
		}
		h = u.op.status;

		u.ch.h = h;
		u.ch.option = "DIRECT";
		u.ch.value = "";
		SYNC_IO(&u.ch, KERNEL$CHANGE_HANDLE);

		raid->devices[i].rq = NULL;
		raid->devices[i].h = h;
		size = gethsize(raid->devices[i].h) >> BIO_SECTOR_SIZE_BITS;
		if (raid->stripe)
			size -= size % raid->stripe;
		if (!size) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: DEVICE %s HAS NO SECTORS", path);
			r = -EINVAL;
			cl_1:
			close(raid->devices[i].h);
			goto cl_2;
		}
		if (size != (__sec_t)size || (__sec_t)size < 0) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: DEVICE %s IS TOO LARGE (%"__64_format"d)", path, size);
			r = -EFBIG;
			goto cl_1;
		}
		raid->devices[i].sector = size;
	}

	sectors = t->init(raid);
	if (sectors < 0) {
		r = sectors;
		goto ret2;
	}
	if (!raid->physical_block_size)
		raid->physical_block_size = 1;

	if (__unlikely((r = BIO$ALLOC_PARTITIONS(&raid->p, raid, SPL_X(SPL_RAID), sectors, RAID_IOCTL, raid->dev_name)) < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: OUT OF MEMORY");
		goto ret2;
	}
	KERNEL$SLAB_INIT(&raid->biorq, sizeof(RAID_BIORQ), sizeof(__sec_t), VM_TYPE_WIRED_MAPPED, RAID_CTOR, NULL, "RAID$BIORQ");
	if ((r = KERNEL$SLAB_RESERVE(&raid->biorq, 2))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: OUT OF MEMORY");
		goto ret3;
	}
	raid->dummy_biorq = __slalloc(&raid->biorq);
	if (!raid->dummy_biorq)
		KERNEL$SUICIDE("RAID: CAN'T ALLOCATE DUMMY BIORQ");
	WQ_INIT(&raid->biorq_wait, "RAID$BIORQ_WAIT");

	u.mrq.size = sizeof(char *) * (n_devices + 1);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		r = u.mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RAID: OUT OF MEMORY");
		goto ret4;
	}
	dep_vec = u.mrq.ptr;
	for (i = 0; i < n_devices; i++)
		dep_vec[i] = KERNEL$HANDLE_PATH(raid->devices[i].h);
	dep_vec[i] = NULL;

	r = KERNEL$REGISTER_DEVICE(raid->dev_name, "RAID.SYS", LN_VECTOR_DEPENDENCIES, raid, RAID_INIT_ROOT, NULL, NULL, NULL, RAID_UNLOAD, &raid->lnte, dep_vec);
	free(dep_vec);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", raid->dev_name, strerror(-r));
		goto ret4;
	}
	raid->dlrq = KERNEL$TSR_IMAGE();
	return 0;

	ret4:
	__slow_slfree(raid->dummy_biorq);
	ret3:
	KERNEL$SLAB_DESTROY(&raid->biorq);
	BIO$FREE_PARTITIONS(raid->p);
	ret2:
	for (i = 0; i < raid->n_devices; i++) close(raid->devices[i].h);
	ret1:
	free(raid);
	ret0:
	return r;
}

static int RAID_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	unsigned i;
	RAID *raid = p;
	BIO$DISABLE_CACHE(raid->p);
	if ((r = KERNEL$DEVICE_UNLOAD(raid->lnte, argv))) return r;
	RAISE_SPL(SPL_RAID);
	__slow_slfree(raid->dummy_biorq);
	while (!KERNEL$SLAB_EMPTY(&raid->biorq)) KERNEL$SLEEP(1);
	LOWER_SPL(SPL_ZERO);
	WQ_WAKE_ALL(&raid->biorq_wait);
	KERNEL$SLAB_DESTROY(&raid->biorq);
	for (i = 0; i < raid->n_devices; i++) {
		if (raid->devices[i].rq)
			KERNEL$SUICIDE("RAID_UNLOAD: REQUEST POINTER ON DEVICE %u LEFT: %p", i, raid->devices[i].rq);
		if (!raid->ro)
			fsync(raid->devices[i].h);
		close(raid->devices[i].h);
	}
	*release = raid->dlrq;
	free(raid);
	return 0;
}

void GET_DEVICE_PARAMS(int h, long *read_size, long *write_size, long *block_size)
{
	IOCTLRQ ioctlrq;

	ioctlrq.h = h;
	ioctlrq.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
	ioctlrq.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ;
	ioctlrq.v.ptr = 0;
	ioctlrq.v.len = 0;
	ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&ioctlrq, KERNEL$IOCTL);
	*read_size = ioctlrq.status;

	ioctlrq.h = h;
	ioctlrq.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
	ioctlrq.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_WRITE;
	ioctlrq.v.ptr = 0;
	ioctlrq.v.len = 0;
	ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&ioctlrq, KERNEL$IOCTL);
	*write_size = ioctlrq.status;

	ioctlrq.h = h;
	ioctlrq.ioctl = IOCTL_BIO_PHYSICAL_BLOCKSIZE;
	ioctlrq.param = 0;
	ioctlrq.v.ptr = 0;
	ioctlrq.v.len = 0;
	ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&ioctlrq, KERNEL$IOCTL);
	if (ioctlrq.status >= 0) *block_size = ioctlrq.status >> BIO_SECTOR_SIZE_BITS;
	else *block_size = ioctlrq.status;
}
