#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <SPAD/PCI.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <ARCH/PAGE.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/SCSI.H>
#include <STDLIB.H>

#include "INIAREG.H"

/* This controller is beatiful! It's so simple to program. */

#define IIO_COMMAND_TIMEOUT	(JIFFIES_PER_SECOND * 10)
#define IIO_REGISTER_TIMEOUT	JIFFIES_PER_SECOND
#define IIO_WAIT_STEP		100
#define IIO_WAIT_STEPS_SLEEP	2

#define IIO_TAGS_PER_DEV	64

/*
MMIO is buggy. At first it seems to work, but during heavy usage, the controller
starts to return invalid indices and resets the computer.
Linux developers know why they aren't using it...
#define USE_MMIO
*/

static const struct pci_id_s pci_cards[] = {
	{ 0x1101, 0x1060, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INITIO A100U2W", 0 },
	{ 0, }
};

typedef struct {
#ifndef USE_MMIO
	io_t io;
#else
	__u8 *mem;
#endif
	__u8 max_queue;
	char need_dequeue;
	ORC_SCB *scb_base;

	AST irq_ast;
	IRQ_HANDLE *irq_ctrl;

	ORC_SCB *scb_freelist;

	LIST_HEAD attached_list;

	/* The SCSI standard allows sending mixed tagged/untagged requests to
	   different LUNs, but my disk chokes on it */
	__u8 used_tags[ORC_MAX_IDS];

	TIMER timeout;
	char timeout_posted;

	char sync_sleep;

	__u16 fwrev;

	__u8 nvram[ORC_NVRAM_SIZE];

	pci_id_t pci_id;
	__u32 scb_phys;
	vspace_dmaunlock_t *scb_unlock;
	void *dlrq;
	void *lnte;
#ifndef USE_MMIO
	IO_RANGE range;
#endif
	char dev_name[__MAX_STR_LEN];
} IIO;

#define SIZEOF_SCBs(iio)	(sizeof(ORC_SCB) * (iio)->max_queue)
#define SIZEOF_ESCBs(iio)	(sizeof(ORC_ESCB) * (iio)->max_queue)
#define SCB_ALIGN		0x4000

#ifndef USE_MMIO
#define IIO_INB(iio, port)		(io_inb((iio)->io + (port)))
#define IIO_INW(iio, port)		(io_inw((iio)->io + (port)))
#define IIO_INL(iio, port)		(io_inl((iio)->io + (port)))
#define IIO_OUTB(iio, port, val)	(io_outb((iio)->io + (port), val))
#define IIO_OUTW(iio, port, val)	(io_outw((iio)->io + (port), val))
#define IIO_OUTL(iio, port, val)	(io_outl((iio)->io + (port), val))
#else
#define IIO_INB(iio, port)		(mmio_inb((iio)->mem + (port)))
#define IIO_INW(iio, port)		(mmio_inw((iio)->mem + (port)))
#define IIO_INL(iio, port)		(mmio_inl((iio)->mem + (port)))
#define IIO_OUTB(iio, port, val)	(mmio_outb((iio)->mem + (port), val))
#define IIO_OUTW(iio, port, val)	(mmio_outw((iio)->mem + (port), val))
#define IIO_OUTL(iio, port, val)	(mmio_outl((iio)->mem + (port), val))
#endif

__COLD_ATTR__ static void UDELAY(IIO *iio, unsigned usec, int slp)
{
	if (!slp || !iio->sync_sleep) {
		if (iio->sync_sleep)
			KERNEL$THREAD_MAY_BLOCK();
		KERNEL$UDELAY(usec);
	} else {
		u_jiffies_lo_t j;
		USEC_2_JIFFIES_LO(usec, j);
		KERNEL$SLEEP(j + 1);
	}
}

__COLD_ATTR__ static int IIO_WAIT(IIO *iio, unsigned reg, __u8 mask, __u8 set)
{
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj = j;
	__u8 val;
	unsigned slp = 0;
again:
	val = IIO_INB(iio, reg);
	if ((val & mask) == set)
		return val;
	if (__likely(jj - j <= IIO_REGISTER_TIMEOUT)) {
		UDELAY(iio, IIO_WAIT_STEP, slp >= IIO_WAIT_STEPS_SLEEP);
		slp++;
		jj = KERNEL$GET_JIFFIES_LO();
		goto again;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, iio->dev_name, "TIMEOUT WHEN WAITING FOR BITS %02X OF %02X IN %X SET, VALUE %02X", set, mask, reg, val);
	return -1;
}

__COLD_ATTR__ static int IIO_HDO(IIO *iio, __u8 cmd)
{
	int r;
	IIO_OUTB(iio, ORC_HDATA, cmd);
	IIO_OUTB(iio, ORC_HCTRL, ORC_HCTRL_HDO);
	if (__unlikely((r = IIO_WAIT(iio, ORC_HCTRL, ORC_HCTRL_HDO, 0)) < 0)) return r;
	return 0;
}

__COLD_ATTR__ static int IIO_HDI(IIO *iio)
{
	int r;
	__u8 val;
	if (__unlikely((r = IIO_WAIT(iio, ORC_HSTUS, ORC_HSTUS_HDI, ORC_HSTUS_HDI)) < 0))
		return r;
	val = IIO_INB(iio, ORC_HDATA);
	IIO_OUTB(iio, ORC_HSTUS, r);
	return val;
}

__COLD_ATTR__ static int IIO_READ_FWREV(IIO *iio)
{
	int r;
	int version1, version2;
	if ((r = IIO_HDO(iio, ORC_CMD_VERSION))) return r;
	if ((version1 = IIO_HDI(iio)) < 0) return version1;
	if ((version2 = IIO_HDI(iio)) < 0) return version1;
	return version1 + (version2 << 8);
}

__COLD_ATTR__ static __u8 IIO_EBIOS_READ(IIO *iio, unsigned address)
{
	IIO_OUTW(iio, ORC_EBIOSADDR0, address);
	return IIO_INB(iio, ORC_EBIOSDATA);
}

__COLD_ATTR__ static int IIO_RESET(IIO *iio)
{
	int r;
	IIO_OUTB(iio, ORC_HCTRL, ORC_HCTRL_DEVRST);
	if ((r = IIO_WAIT(iio, ORC_HCTRL, ORC_HCTRL_HOSTSTOP, ORC_HCTRL_HOSTSTOP)) < 0)
		return r;
	return 0;
}

__COLD_ATTR__ static int IIO_RESET_RELOAD(IIO *iio)
{
	int r;
	__u8 gcfg;
	__u32 fwbase;
	unsigned i;
	if ((r = IIO_RESET(iio)))
		return r;

	gcfg = IIO_INB(iio, ORC_GCFG);
	IIO_OUTB(iio, ORC_GCFG, gcfg | ORC_GCFG_EEPRG);
	if ((r = IIO_EBIOS_READ(iio, 0x0000)) != 0x55) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, iio->dev_name, "BAD SIGNATURE AT BIOS ADDRESS 0: %02X", r);
		goto fail;
	}
	if ((r = IIO_EBIOS_READ(iio, 0x0001)) != 0xaa) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, iio->dev_name, "BAD SIGNATURE AT BIOS ADDRESS 1: %02X", r);
		goto fail;
	}

	IIO_OUTB(iio, ORC_RISCCTL, ORC_RISCCTL_DOWNLOAD | ORC_RISCCTL_PRGMRST);
	fwbase = IIO_EBIOS_READ(iio, 0x0010);
	fwbase += IIO_EBIOS_READ(iio, 0x0011) << 8;
	fwbase += IIO_EBIOS_READ(iio, 0x0012) << 16;
	IIO_OUTL(iio, ORC_FWBASEADR, fwbase);
	UDELAY(iio, 1000, 1);
	for (i = 0; i < ORC_FW_SIZE; i += 4) {
		__u32 val = IIO_EBIOS_READ(iio, fwbase + i);
		val |= IIO_EBIOS_READ(iio, fwbase + i + 1) << 8;
		val |= IIO_EBIOS_READ(iio, fwbase + i + 2) << 16;
		val |= IIO_EBIOS_READ(iio, fwbase + i + 3) << 24;
		IIO_OUTL(iio, ORC_RISCRAM, val);
	}
	IIO_OUTB(iio, ORC_RISCCTL, ORC_RISCCTL_DOWNLOAD | ORC_RISCCTL_PRGMRST);
	for (i = 0; i < ORC_FW_SIZE; i += 4) {
		__u32 val2;
		__u32 val = IIO_EBIOS_READ(iio, fwbase + i);
		val |= IIO_EBIOS_READ(iio, fwbase + i + 1) << 8;
		val |= IIO_EBIOS_READ(iio, fwbase + i + 2) << 16;
		val |= IIO_EBIOS_READ(iio, fwbase + i + 3) << 24;
		val2 = IIO_INL(iio, ORC_RISCRAM);
		if (val != val2) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, iio->dev_name, "FIRMWARE BADLY DOWNLOADED AT DWORD %03X, EEPROM %08X, RAM %08X", i, val, val2);
			goto fail2;
		}
	}

	/*__debug_printf("downloaded microcode\n");*/

	IIO_OUTB(iio, ORC_RISCCTL, ORC_RISCCTL_PRGMRST);
	IIO_OUTB(iio, ORC_GCFG, gcfg);
	return 0;

fail2:
	IIO_OUTB(iio, ORC_RISCCTL, ORC_RISCCTL_PRGMRST);
fail:
	IIO_OUTB(iio, ORC_GCFG, gcfg);
	return -1;
}

__COLD_ATTR__ static void IIO_SETUP_SCBS(IIO *iio)
{
	IIO_OUTB(iio, ORC_SCBSIZE, iio->max_queue);
	IIO_OUTL(iio, ORC_SCBBASE0, iio->scb_phys);
	IIO_OUTL(iio, ORC_SCBBASE1, iio->scb_phys);
}

__COLD_ATTR__ static int IIO_NVRAM_READ(IIO *iio, __u8 address)
{
	int r;
	if ((r = IIO_HDO(iio, ORC_CMD_GET_NVM))) return r;
	if ((r = IIO_HDO(iio, address))) return r;
	return IIO_HDI(iio);
}

__COLD_ATTR__ static __u8 NVRAM_CHECKSUM(__u8 *nvram)
{
	unsigned i;
	__u8 checksum = 0;
	for (i = 0; i < ORC_NVRAM_SIZE - 1; i++) checksum += nvram[i];
	return checksum;
}

__COLD_ATTR__ static int IIO_READ_WHOLE_NVRAM(IIO *iio)
{
	unsigned i;
	__u8 checksum;
	for (i = 0; i < ORC_NVRAM_SIZE; i++) {
		int val = IIO_NVRAM_READ(iio, i);
		if (val < 0) return val;
		iio->nvram[i] = val;
		/*__debug_printf("nvram[%02x] = %02x\n", i, val);*/
	}
	checksum = NVRAM_CHECKSUM(iio->nvram);
	if (checksum != iio->nvram[ORC_NVRAM_CHECKSUM]) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, iio->dev_name, "BAD NVRAM CHECKSUM, COUNTED %02X, STORED %02X", checksum, iio->nvram[ORC_NVRAM_CHECKSUM]);
	}
	return 0;
}

__COLD_ATTR__ static int IIO_INIT_HW(IIO *iio, int rst)
{
	int revision;
	IIO_OUTB(iio, ORC_GIMSK, 0xff);
	if (IIO_INB(iio, ORC_HSTUS) & ORC_HSTUS_RREADY && !rst) {
		revision = IIO_READ_FWREV(iio);
		/*__debug_printf("read fw version: %d\n", revision);*/
		if (revision < 0 || revision == 0xffff)
			goto reset;
		IIO_SETUP_SCBS(iio);
	} else {
reset:
		if (IIO_RESET_RELOAD(iio))
			return -EIO;
		IIO_SETUP_SCBS(iio);
		IIO_OUTB(iio, ORC_HCTRL, 0);
		if (IIO_WAIT(iio, ORC_HSTUS, ORC_HSTUS_RREADY, ORC_HSTUS_RREADY) < 0)
			return -EIO;
		revision = IIO_READ_FWREV(iio);
		if (revision < 0)
			return -EIO;
	}

	iio->fwrev = revision;

	if (IIO_READ_WHOLE_NVRAM(iio))
		return -EIO;

	return 0;
}

__COLD_ATTR__ static void IIO_ENABLE_INTERRUPTS(IIO *iio)
{
	IIO_OUTB(iio, ORC_GIMSK, 0xff & ~ORC_GIMSK_RPFIFO);
}

__COLD_ATTR__ static void IIO_INIT_STRUCTS(IIO *iio)
{
	ORC_SCB **last;
	ORC_SCB *scb;
	ORC_ESCB *escb;
	__u32 escb_phys;
	unsigned i;

	memset(iio->scb_base, 0, SIZEOF_SCBs(iio) + SIZEOF_ESCBs(iio));

	scb = iio->scb_base;

	last = &iio->scb_freelist;

	escb = (ORC_ESCB *)(scb + iio->max_queue);
	escb_phys = iio->scb_phys + iio->max_queue * sizeof(ORC_SCB);
	for (i = 0; i < iio->max_queue; i++, scb++, escb++, escb_phys += sizeof(ORC_ESCB)) {
		escb->state = ESCB_STATE_FREE;
		escb->sgent_phys_le = __32CPU2LE(escb_phys + __offsetof(ORC_ESCB, sgent));
		scb->sense_addr = __32CPU2LE(escb_phys + __offsetof(ORC_ESCB, sense));
		scb->scbidx = i;
		scb->u.escb = escb;
		*last = scb;
		last = &FREELIST_PTR(scb);
	}
	*last = NULL;
}

static void UNMAP_SCB(ORC_ESCB *escb)
{
	unsigned idx;
	for (idx = 0; idx < escb->n_sgents; idx++) {
		escb->dma[idx].dmaunlock(escb->dma[idx].dmaaddr);
	}
}

static __finline__ SCSIRQ *FREE_SCB(IIO *iio, ORC_SCB *scb)
{
	ORC_ESCB *escb;
	scb->status = ORC_STATUS_COMPLETE;
	__write_barrier();
	FREELIST_PTR(scb) = iio->scb_freelist;
	iio->scb_freelist = scb;
	escb = scb->u.escb;
	escb->state = ESCB_STATE_FREE;
	if (__unlikely(escb->need_unmap))
		UNMAP_SCB(escb);
	return escb->scsirq;
}

__COLD_ATTR__ static SCSIRQ *FREE_SCB_NOINLINE(IIO *iio, ORC_SCB *scb)
{
	return FREE_SCB(iio, scb);
}

static __finline__ void DONE_RQ(IIO *iio, SCSIRQ *rq)
{
	rq->internal = 0;
#if __DEBUG >= 1
	if (__unlikely(iio->used_tags[rq->ch_id] < rq->host_aux))
		KERNEL$SUICIDE("DONE_RQ: TAG COUNT UNDERFLOW: %d < %d", iio->used_tags[rq->ch_id], rq->host_aux);
#endif
	iio->used_tags[rq->ch_id] -= rq->host_aux;
	if (__unlikely(iio->need_dequeue)) {
		iio->need_dequeue = 0;
		SCSI$HOST_DEQUEUE(&iio->attached_list);
	}
	rq->done(rq);
}

static __finline__ void DONE_RQ_ERROR(IIO *iio, SCSIRQ *rq, int error)
{
	rq->status = error;
	DONE_RQ(iio, rq);
}

static DECL_IRQ_AST(IIO_IRQ, SPL_ATA_SCSI, AST)
{
	IIO *iio = GET_STRUCT(RQ, IIO, irq_ast);

	if (__unlikely(!IIO_INB(iio, ORC_RQCNT)))
		goto not_mine;

	do {
		ORC_SCB *scb;
		ORC_ESCB *escb;
		SCSIRQ *rq;
		__u8 idx = IIO_INB(iio, ORC_RQUEUE);
		if (__unlikely(idx >= iio->max_queue)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, iio->dev_name, "INVALID QUEUE NUMBER RETURNED: %02X", idx);
			break;
		}
		scb = &iio->scb_base[idx];
		escb = scb->u.escb;
		if (__unlikely(escb->state == ESCB_STATE_FREE)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, iio->dev_name, "FREE QUEUE NUMBER RETURNED: %02X", scb->scbidx);
			break;
		}
		scb->status = ORC_STATUS_COMPLETE;
		__write_barrier();
		
		rq = escb->scsirq;
		if (__unlikely(scb->hastat)) {
			switch (scb->hastat) {
				case SCSI_MSG_LINKED_CMD_COMPLETE:
				case SCSI_MSG_LINKED_FLG_CMD_COMPLETE:
					goto status_ok;
				case ORC_HASTAT_SEL_TOUT:
					rq->status = -ENXIO;
					break;
				case ORC_HASTAT_DO_DU:
					/*__debug_printf("do/du: resid %x, wanted %x; status %02x\n", scb->xferlen, rq->status, scb->tastat);
					{
						int i;
						for (i = 0; i < rq->cmdlen; i++)
							__debug_printf(" %02x", rq->cmd[i]);
						__debug_printf("\n");
					}*/
					if (__unlikely(rq->status < __32LE2CPU(scb->xferlen)))
						rq->status = 0;
					else
						rq->status -= __32LE2CPU(scb->xferlen);
					goto status_ok;
				case ORC_HASTAT_BUS_FREE:
				case ORC_HASTAT_BAD_PHAS:
					rq->status = -EPROTO;
					goto print_error;
				case ORC_HASTAT_INV_CMD:
					rq->status = -EBADMSG;
					goto print_error;
				case ORC_HASTAT_ABORTED:
					rq->status = -ECONNABORTED;
					break;
				case ORC_HASTAT_SCSI_RST:
				case ORC_HASTAT_DEV_RST:
					rq->status = -ECONNRESET;
					break;
				default:
					rq->status = -EIO;
				print_error:
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, iio->dev_name, "HOST ADAPTER ERROR %02X", scb->hastat);
					break;
			}
			if (__unlikely(escb->state == ESCB_STATE_ABORTING))
				rq->status = -EINTR;
		} else {
			status_ok:
			rq->scsi_status = scb->tastat;
			if (__unlikely(scb->tastat == SCSI_CHECK_CONDITION)) {
				if (__unlikely(rq->sense_size < ORC_MAX_SENSE))
					memcpy(rq->sense, escb->sense, rq->sense_size);
				else
					memset(mempcpy(rq->sense, escb->sense, ORC_MAX_SENSE), 0, rq->sense_size - ORC_MAX_SENSE);
			}
		}
		DONE_RQ(iio, FREE_SCB(iio, scb));
	} while (__unlikely(IIO_INB(iio, ORC_RQCNT)));

not_mine:
	KERNEL$UNMASK_IRQ(iio->irq_ctrl);
	RETURN;
}

static int IIO_POST(SCSIRQ *rq)
{
	IIO *iio;
	ORC_SCB *scb;
	ORC_ESCB *escb;
	unsigned idx;
	BIODESC *desc;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("IIO_POST AT SPL %08X", KERNEL$SPL);
	if (__unlikely(rq->internal != 0))
		KERNEL$SUICIDE("IIO_POST: UNINITIALIZED REQUEST, INTERNAL %lX", rq->internal);
#endif
	if (__unlikely(rq->cmdlen > ORC_MAX_CDB))
		return -EOPNOTSUPP;

	rq->host_aux = __unlikely(rq->tagging == SCSI_TAGGING_NONE) ? IIO_TAGS_PER_DEV : 1;

	iio = rq->adapter;

	/*{
		int i;
		__debug_printf("post:");
		for (i = 0; i < rq->cmdlen; i++) __debug_printf(" %02x", rq->cmd[i]);
		__debug_printf("\n");
	}*/
	/*__debug_printf("post: %d %d, tagging %02x\n", iio->used_tags[rq->ch_id], rq->host_aux, rq->tagging);*/
	if (__unlikely(iio->used_tags[rq->ch_id] + rq->host_aux > IIO_TAGS_PER_DEV)) {
		eagain:
		iio->need_dequeue = 1;
		return -EAGAIN;
	}

	scb = iio->scb_freelist;
	if (__unlikely(!scb))
		goto eagain;
	iio->scb_freelist = FREELIST_PTR(scb);

	rq->internal = (unsigned long)scb;

	escb = scb->u.escb;
	escb->scsirq = rq;

#if __DEBUG >= 1
	if (__unlikely(escb->state != ESCB_STATE_FREE))
		KERNEL$SUICIDE("IIO_POST: INVALID ESCB STATE %X", escb->state);
#endif
	scb->opcode = ORC_OPCODE_EXECSCSI;
	scb->flags = __likely(rq->direction == PF_WRITE) ? ORC_FLAGS_DIR_IN : __likely(rq->direction == PF_READ) ? ORC_FLAGS_DIR_OUT : ORC_FLAGS_DIR_NO_XF;
	scb->target = rq->ch_id;
	scb->lun = (unsigned long)rq->lun >> 8;
	scb->reserved0 = __32CPU2LE(0);
	scb->xferlen = 0;
	scb->reserved1 = __32CPU2LE(0);
	scb->sg_addr = escb->sgent_phys_le;
	scb->sg_addrhigh = __32CPU2LE(0);
	scb->hastat = 0;
	scb->tastat = 0;
	scb->status = 0;
	scb->link = 0xff;
	scb->sense_len = ORC_MAX_SENSE;
	scb->cdb_len = rq->cmdlen;
	scb->ident = ORC_IDENT_DISC_ALLOW | scb->lun;
	scb->tag_msg = rq->tagging;
	memcpy(scb->cdb, rq->cmd, rq->cmdlen);

	escb->state = ESCB_STATE_POSTED;
	escb->need_unmap = 0;
	memset(escb->sense, 0, ORC_MAX_SENSE);
	idx = 0;
	/*__debug_printf("trying: %Lx : %lx\n", !rq->desc ? 0LL : rq->desc->v.ptr, !rq->desc ? 0L : rq->desc->v.len);*/
	for (desc = rq->desc; desc; desc = desc->next) {
		VDMA dma;
		VDESC vdesc = desc->v;
		dma.spl = SPL_X(SPL_ATA_SCSI);
		again:
		RAISE_SPL(SPL_VSPACE);
		escb->dma[idx].dmaunlock = vdesc.vspace->op->vspace_dmalock(&vdesc, escb->scsirq->direction, &dma);
		if (__unlikely(!escb->dma[idx].dmaunlock)) {
			escb->n_sgents = idx;
			/*__debug_printf("bail on idx %x\n", idx);*/
			FREE_SCB_NOINLINE(iio, scb);
			return -EVSPACEFAULT;
		}
		if (__unlikely(escb->dma[idx].dmaunlock != KERNEL$NULL_VSPACE_DMAUNLOCK))
			escb->need_unmap = 1;
		scb->xferlen += dma.len;
#if __DEBUG >= 1
		if (__unlikely(idx >= ORC_MAX_SGENT))
			KERNEL$SUICIDE("IIO_POST: RAN OUT OF SG SEGMENTS");
#endif
		escb->dma[idx].dmaaddr = dma.ptr;
		escb->sgent[idx].base = __32CPU2LE(dma.ptr);
		escb->sgent[idx].len = __32CPU2LE(dma.len);
		idx++;
		if (__unlikely(vdesc.len != dma.len)) {
			vdesc.len -= dma.len;
			vdesc.ptr += dma.len;
			goto again;
		}
	}
	/*__debug_printf("posting: %x / %x @ %Lx\n", scb->xferlen, idx, !rq->desc ? 0LL : rq->desc->v.ptr);*/
	escb->n_sgents = idx;
	scb->sg_len = __32CPU2LE(idx * sizeof(ORC_SGENT));
	escb->scsirq->status = scb->xferlen;
	scb->xferlen = __32CPU2LE(scb->xferlen);

	__write_barrier();
	scb->status = ORC_STATUS_POST;
	__barrier();
	IIO_OUTB(iio, ORC_PQUEUE, scb->scbidx);

	rq = escb->scsirq;
	iio->used_tags[rq->ch_id] += rq->host_aux;

	return 0;
}

__COLD_ATTR__ static void IIO_CANCEL(SCSIRQ *rq)
{
	IIO *iio;
	ORC_SCB *scb;
	ORC_ESCB *escb;
	if (!rq->internal)
		return;
	iio = rq->adapter;
	scb = (ORC_SCB *)rq->internal;
	escb = scb->u.escb;
	if (__unlikely(escb->state == ESCB_STATE_ABORTING))
		return;
	if (__unlikely(escb->state != ESCB_STATE_POSTED))
		KERNEL$SUICIDE("IIO_CANCEL: INVALID ESCB STATE %X", escb->state);
	escb->state = ESCB_STATE_ABORTING;
	escb->abort_time = KERNEL$GET_JIFFIES_LO();

	if (!iio->timeout_posted) {
		KERNEL$SET_TIMER(IIO_COMMAND_TIMEOUT, &iio->timeout);
		iio->timeout_posted = 1;
	}

	/* This hardly ever works ...
	if (__likely(scb->tag_msg)) {
		if (__unlikely(IIO_HDO(iio, ORC_CMD_ABORT_SCB))) return;
		if (__unlikely(IIO_HDO(iio, scb->scbidx))) return;
		IIO_HDI(iio);
	}
	*/
}

__COLD_ATTR__ static void IIO_FORCE_ABORT_COMMANDS(IIO *iio)
{
	unsigned i;
	int freed_something;
	scan_again:
	freed_something = 0;
	for (i = 0; i < iio->max_queue; i++) {
		ORC_SCB *scb = &iio->scb_base[i];
		ORC_ESCB *escb = scb->u.escb;
		if (escb->state == ESCB_STATE_POSTED ||
		    escb->state == ESCB_STATE_ABORTING) {
			DONE_RQ_ERROR(iio, FREE_SCB_NOINLINE(iio, scb), -EINTR);
			freed_something = 1;
		}
	}
	if (freed_something) goto scan_again;
}

__COLD_ATTR__ static void IIO_ABORT_CONTROLLER(IIO *iio)
{
	IIO_RESET(iio);
	IIO_FORCE_ABORT_COMMANDS(iio);
	IIO_INIT_STRUCTS(iio);
	IIO_INIT_HW(iio, 1);
	IIO_ENABLE_INTERRUPTS(iio);
}

__COLD_ATTR__ static void IIO_TIMEOUT_FN(TIMER *t)
{
	unsigned i;
	u_jiffies_lo_t j, jj;
	IIO *iio = GET_STRUCT(t, IIO, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	iio->timeout_posted = 0;
	j = KERNEL$GET_JIFFIES_LO();
	jj = (u_jiffies_lo_t)-1;
	for (i = 0; i < iio->max_queue; i++) {
		ORC_SCB *scb = &iio->scb_base[i];
		ORC_ESCB *escb = scb->u.escb;
		if (escb->state == ESCB_STATE_ABORTING) {
			u_jiffies_lo_t jjj;
			if (j - escb->abort_time >= IIO_COMMAND_TIMEOUT) {
				KERNEL$SYSLOG(__SYSLOG_HW_ERROR, iio->dev_name, "TIMEOUT WHEN EXECUTING COMMAND %02X ON TARGET %X, LUN %X. RESETTING CONTROLLER", scb->cdb[0], scb->target, scb->lun);
				goto abort_controller;
			}
			jjj = IIO_COMMAND_TIMEOUT - (j - escb->abort_time);
			if (jjj < jj) jj = jjj;
		}
	}
	if (jj != (u_jiffies_lo_t)-1) {
		KERNEL$SET_TIMER(jj, &iio->timeout);
		iio->timeout_posted = 1;
	}
	return;

	abort_controller:
	IIO_ABORT_CONTROLLER(iio);
}

__COLD_ATTR__ static int IIO_DCALL(void *ptr, const char *dcall_type, int cmd, va_list params)
{
	IIO *iio = ptr;
	SCSI_ATTACH_PARAM *ap = va_arg(params, SCSI_ATTACH_PARAM *);
	if (KERNEL$SPL != SPL_X(SPL_DEV))
		KERNEL$SUICIDE("IIO_DCALL AT SPL %08X", KERNEL$SPL);
	switch (cmd) {
	case SCSI_CMD_GET: {
		int id = -1;
		int i;
		const char *arg;
		int state;
		static const struct __param_table params[3] = {
			"ID", __PARAM_INT, 0, ORC_MAX_IDS,
			NULL, 0, 0, 0,
		};
		void *vars[2];
		vars[0] = &id;
		vars[1] = NULL;
		if (__unlikely(ap->u.p.version != SCSI_VERSION)) {
			ap->msg = "INCORRECT VERSION";
			return -EINVAL;
		}
		arg = ap->u.p.param;
		state = 0;
		if (__unlikely(__parse_extd_param(&arg, &state, params, vars, NULL, NULL, NULL, NULL))) {
			ap->msg = "SYNTAX ERROR";
			return -EBADSYN;
		}
		for (i = 0; i < ORC_MAX_IDS; i++) {
			if (id != -1 && i != id) continue;
			if (i == iio->nvram[ORC_NVRAM_CH0_HA_ID]) continue;
			if (ap->adapter) {
				if (ap->ch_id >= i) continue;
			}
			ap->max_sectors = SCSI$HOST_SGENTS_TO_SECTORS(ORC_MAX_SGENT);
			ap->max_fragments = ORC_MAX_SGENT;
			ap->max_untagged_requests = 1;
			ap->max_tagged_requests = IIO_TAGS_PER_DEV;
			ap->scsi_flags = SCSI_FLAG_CAN_USE_MODE_10 | SCSI_FLAG_CAN_USE_MODE_6;
			ap->lun = LUN_NONE;
			ap->ch_id = i;
			ap->adapter = iio;
			ap->post = IIO_POST;
			ap->cancel = IIO_CANCEL;
			SCSI$HOST_PRINT_NAME(ap, iio->dev_name, 0, 1, i, ORC_MAX_IDS);
			RAISE_SPL(SPL_ATA_SCSI);
			ADD_TO_LIST(&iio->attached_list, &ap->u.h.client_list);
			LOWER_SPL(SPL_DEV);
			return 0;
		}
		return -ENODEV;
	}
	case SCSI_CMD_FREE: {
		if (__unlikely(ap->adapter != iio))
			KERNEL$SUICIDE("IIO_DCALL: INVALID ADAPTER %p != %p", ap->adapter, iio);
		RAISE_SPL(SPL_ATA_SCSI);
		DEL_FROM_LIST(&ap->u.h.client_list);
		LOWER_SPL(SPL_DEV);
		return 0;
	}
	case SCSI_CMD_SET_LUN: {
		int r = -EINVAL;
		if (__unlikely((unsigned)SCSI$LUN_CONVERT_INTERNAL_TO_8(ap->lun) >= ORC_MAX_LUNS) || __unlikely(r = SCSI$HOST_IS_LUN_DUPLICATE(&iio->attached_list, ap))) {
			ap->lun = LUN_NONE;
			return r;
		}
		SCSI$HOST_PRINT_NAME(ap, iio->dev_name, 0, 1, ap->ch_id, ORC_MAX_IDS);
		return 0;
	}
	default: {
		return -EINVAL;
	}
	}
}

static int IIO_UNLOAD(void *ptr, void **unload, const char * const argv[]);

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	int r;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	const char * const *arg = argv;
	int state = 0;
	const char *opt, *optend, *str;
	char *chip_name;
	int irq;
	IIO *iio;
	union {
		MALLOC_REQUEST mrq;
		VDESC vdesc;
		char pci_id_str[__MAX_STR_LEN];
	} u;
	VDMA vdma;
	static const struct __param_table params[1] = {
		NULL, 0, 0, 0,
	};
	void *vars[1];
	vars[0] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "INIA100: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}

	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, (unsigned long *)(void *)&KERNEL$LIST_END, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "INIA100: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}

	u.mrq.size = sizeof(IIO);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "INIA100: CAN'T ALLOCATE MEMORY");
		r = u.mrq.status;
		goto ret1;
	}
	iio = u.mrq.ptr;
	memset(iio, 0, sizeof(IIO));
	iio->pci_id = id;

	_snprintf(iio->dev_name, __MAX_STR_LEN, "SCSI$INIA100@" PCI_ID_FORMAT, id);

	INIT_TIMER(&iio->timeout);
	iio->timeout.fn = IIO_TIMEOUT_FN;
	INIT_LIST(&iio->attached_list);
	iio->max_queue = ORC_MAX_QUEUE;
	if (KERNEL$RESERVE_BOUNCE_ACTIVE(0)) {
		if (iio->max_queue > ORC_MAX_QUEUE_LIMITED)
			iio->max_queue = ORC_MAX_QUEUE_LIMITED;
	}

	iio->scb_base = KERNEL$ALLOC_CONTIG_AREA(SIZEOF_SCBs(iio) + SIZEOF_ESCBs(iio), AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA | AREA_ALIGN, (unsigned long)SCB_ALIGN);
	if (__IS_ERR(iio->scb_base)) {
		r = __PTR_ERR(iio->scb_base);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "INIA100: CAN'T ALLOCATE DMA MEMORY");
		goto ret2;
	}
	u.vdesc.ptr = (unsigned long)iio->scb_base;
	u.vdesc.len = SIZEOF_SCBs(iio) + SIZEOF_ESCBs(iio);
	u.vdesc.vspace = &KERNEL$VIRTUAL;
	vdma.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	iio->scb_unlock = KERNEL$VIRTUAL.op->vspace_dmalock(&u.vdesc, PF_RW, &vdma);
	iio->scb_phys = vdma.ptr;

	if ((r = KERNEL$RESERVE_BOUNCE(iio->dev_name, iio->max_queue * ORC_MAX_SGENT, __PAGE_CLUSTER_SIZE, 0))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT RESERVE BOUNCE BUFFERS: %s", iio->dev_name, strerror(-r));
		goto ret3;
	}

#ifndef USE_MMIO
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	iio->io = PCI$READ_IO_RESOURCE(iio->pci_id, 0);
	if (!iio->io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, iio->dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", iio->dev_name);
		r = -ENXIO;
		goto ret4;
	}
	iio->range.start = iio->io;
	iio->range.len = ORC_REGSPACE;
	iio->range.name = iio->dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&iio->range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", iio->dev_name, iio->range.start, iio->range.start + iio->range.len - 1, strerror(-r));
		goto ret4;
	}
#else
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	iio->mem = PCI$MAP_MEM_RESOURCE(id, 1, ORC_REGSPACE);
	if (!iio->mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, iio->dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", iio->dev_name);
		r = -ENXIO;
		goto ret4;
	}
	if (__IS_ERR(iio->mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", iio->dev_name, strerror(-__PTR_ERR(iio->mem)));
		r = __PTR_ERR(iio->mem);
		goto ret4;
	}
#endif

	iio->sync_sleep = 1;
	IIO_INIT_STRUCTS(iio);
	r = IIO_INIT_HW(iio, 0);
	iio->sync_sleep = 0;
	if (r) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE CHIPSET", iio->dev_name);
		goto ret5;
	}

	irq = PCI$READ_INTERRUPT_LINE(iio->pci_id);

	_printf("%s: SCSI CONTROLLER ON PCI: %s\n", iio->dev_name, PCI$ID(u.pci_id_str, iio->pci_id));
	_printf("%s: %s, FIRMWARE REVISION %d.%d\n%s: ", iio->dev_name, chip_name, iio->fwrev >> 8, iio->fwrev & 0xff, iio->dev_name);
#ifndef USE_MMIO
	_printf("IO "IO_FORMAT, iio->io);
#else
	_printf("MEM %"__64_format"X", PCI$READ_MEM_RESOURCE(id, 1, NULL));
#endif
	_printf(", IRQ %d\n", irq);

	iio->irq_ast.fn = IIO_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(irq, &iio->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &iio->irq_ast, iio->dev_name)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, iio->dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", iio->dev_name);
		goto ret5;
	}

	IIO_ENABLE_INTERRUPTS(iio);
	KERNEL$UNMASK_IRQ(iio->irq_ctrl);

	r = KERNEL$REGISTER_DEVICE(iio->dev_name, "INIA100.SYS", 0, iio, NULL, IIO_DCALL, "SCSI", NULL, IIO_UNLOAD, &iio->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", iio->dev_name, strerror(-r));
		goto ret6;
	}
	iio->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), iio->dev_name);
	return 0;

	ret6:
	KERNEL$MASK_IRQ(iio->irq_ctrl);
	RAISE_SPL(SPL_ATA_SCSI);
	if (iio->timeout_posted) KERNEL$DEL_TIMER(&iio->timeout);
	IIO_OUTB(iio, ORC_GIMSK, 0xff);
	LOWER_SPL(SPL_ZERO);
	KERNEL$RELEASE_IRQ(iio->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &iio->irq_ast);
	ret5:
#ifndef USE_MMIO
	KERNEL$UNREGISTER_IO_RANGE(&iio->range);
#else
	PCI$UNMAP_MEM_RESOURCE(id, iio->mem, ORC_REGSPACE);
#endif
	ret4:
	KERNEL$UNRESERVE_BOUNCE(iio->dev_name, iio->max_queue * ORC_MAX_SGENT, __PAGE_CLUSTER_SIZE, 0);
	ret3:
	iio->scb_unlock(iio->scb_phys);
	KERNEL$FREE_CONTIG_AREA(iio->scb_base, SIZEOF_SCBs(iio) + SIZEOF_ESCBs(iio));
	ret2:
	free(iio);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

__COLD_ATTR__ static int IIO_UNLOAD(void *ptr, void **unload, const char * const argv[])
{
	int r;
	IIO *iio = ptr;
	RAISE_SPL(SPL_ATA_SCSI);
	if (!LIST_EMPTY(&iio->attached_list)) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS CONTROLLER", iio->dev_name);
		return -EBUSY;
	}
	LOWER_SPL(SPL_DEV);
	r = KERNEL$DEVICE_UNLOAD(iio->lnte, argv);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r)) return r;
	KERNEL$MASK_IRQ(iio->irq_ctrl);
	RAISE_SPL(SPL_ATA_SCSI);
	if (iio->timeout_posted) KERNEL$DEL_TIMER(&iio->timeout);
	IIO_OUTB(iio, ORC_GIMSK, 0xff);
	LOWER_SPL(SPL_ZERO);
	KERNEL$RELEASE_IRQ(iio->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &iio->irq_ast);
#ifndef USE_MMIO
	KERNEL$UNREGISTER_IO_RANGE(&iio->range);
#else
	PCI$UNMAP_MEM_RESOURCE(iio->pci_id, iio->mem, ORC_REGSPACE);
#endif
	KERNEL$UNRESERVE_BOUNCE(iio->dev_name, iio->max_queue * ORC_MAX_SGENT, __PAGE_CLUSTER_SIZE, 0);
	iio->scb_unlock(iio->scb_phys);
	PCI$FREE_DEVICE(iio->pci_id);
	KERNEL$FREE_CONTIG_AREA(iio->scb_base, SIZEOF_SCBs(iio) + SIZEOF_ESCBs(iio));
	*unload = iio->dlrq;
	free(iio);
	return 0;
}
