#include <ARCH/BSF.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <ARCH/PAGE.H>
#include <SPAD/SYSLOG.H>
#include <ERRNO.H>

#include "MPT.H"
#include "MPTREG.H"

static const struct pci_id_s pci_cards[] = {
	{ 0x1000, 0x0030, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "53C1030", BUS_SPI | FIX_PCIX_CMD },
	{ 0x117c, 0x0030, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "53C1030 (ATTO)", BUS_SPI | FIX_PCIX_CMD },
	{ 0x1000, 0x0040, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "53C1035", BUS_SPI },
	{ 0x1000, 0x0620, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC909", BUS_FC },
	{ 0x1000, 0x0621, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC909A", BUS_FC },
	{ 0x1000, 0x0622, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC929", BUS_FC },
	{ 0x1000, 0x0623, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC929", BUS_FC },
	{ 0x1000, 0x0624, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC919", BUS_FC },
	{ 0x1000, 0x0625, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC919", BUS_FC },
	{ 0x1000, 0x0626, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC929X", BUS_FC | FIX_PCIX_CMD },
	{ 0x1000, 0x0627, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC919X", BUS_FC | FIX_PCIX_CMD},
	{ 0x1000, 0x0640, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC949X", BUS_FC | FIX_DISABLE_IO },
	{ 0x1000, 0x0642, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC939X", BUS_FC | FIX_DISABLE_IO },
	{ 0x1000, 0x0646, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC949E", BUS_FC },
	{ 0x1657, 0x0646, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "FC949E (BROCADE)", BUS_FC },
	{ 0x1000, 0x0050, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1064", BUS_SAS | FIX_DISABLE_IO },
	{ 0x1000, 0x0054, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1068", BUS_SAS | FIX_DISABLE_IO },
	{ 0x1000, 0x0055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1068", BUS_SAS | FIX_DISABLE_IO },
	{ 0x1000, 0x0056, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1064E", BUS_SAS },
	{ 0x1000, 0x0057, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1064E", BUS_SAS },
	{ 0x1000, 0x0058, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1068E", BUS_SAS },
	{ 0x1000, 0x0059, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1068E", BUS_SAS },
	{ 0x1000, 0x005a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1066E", BUS_SAS },
	{ 0x1000, 0x005c, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1064A", BUS_SAS },
	{ 0x1000, 0x005e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1066", BUS_SAS },
	{ 0x1000, 0x0062, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SAS1078", BUS_SAS },
	{ 0, },
};

static void UDELAY(MPT *mpt, unsigned usec, int slp)
{
	if (!slp || !mpt->sync_sleep) {
		if (mpt->sync_sleep)
			KERNEL$THREAD_MAY_BLOCK();
		KERNEL$UDELAY(usec);
	} else {
		u_jiffies_lo_t j;
		USEC_2_JIFFIES_LO(usec, j);
		KERNEL$SLEEP(j + 1);
	}
}

static int MPT_WAIT(MPT *mpt, unsigned reg, __u32 mask, __u32 set, u_jiffies_lo_t timeout, const char *str)
{
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj = j;
	__u32 val;
	unsigned slp = 0;

	again:
	val = MPT_READ(mpt, reg);
	if ((val & mask) == set)
		return 0;
	
	if (__likely(jj - j <= timeout)) {
		UDELAY(mpt, MPT_WAIT_USEC, slp >= MPT_WAIT_STEPS_SLEEP);
		slp++;
		jj = KERNEL$GET_JIFFIES_LO();
		goto again;
	}

	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, mpt->dev_name, "TIMEOUT WHEN WAITING FOR BITS %08X OF %08X IN %X SET, VALUE %08X, STATE %s", set, mask, reg, val, str);
	return -EIO;
}

static int MPT_SYNC_REQUEST(MPT *mpt, void *rq, unsigned rq_len, u_jiffies_lo_t timeout)
{
	int r;
	__u32 db;
	if (__unlikely(rq_len & 3))
		KERNEL$SUICIDE("MPT_SYNC_REQUEST: INVALID REQUEST LENGTH %X", rq_len);
	if (__unlikely((db = MPT_READ(mpt, MPT_DOORBELL)) & MPT_DOORBELL_ACTIVE)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "DOORBELL IS ALREADY ACTIVE: %08X", (unsigned)db);
		return -EIO;
	}
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);
	MPT_WRITE(mpt, MPT_DOORBELL, (MPT_FUNCTION_HANDSHAKE << __BSF_CONST(MPT_DOORBELL_FUNCTION)) | (rq_len << (__BSF_CONST(MPT_DOORBELL_DWORDS) - 2)));
	if ((r = MPT_WAIT(mpt, MPT_INT_STATUS, MPT_INT_STATUS_DOORBELL, MPT_INT_STATUS_DOORBELL, timeout, "DOORBELL HANDSHAKE INTERRUPT"))) return r;
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);
	if ((r = MPT_WAIT(mpt, MPT_INT_STATUS, MPT_INT_STATUS_IOCDOORBELL, 0, timeout, "DOORBELL HANDSHAKE ACK"))) return r;
	while (rq_len) {
		MPT_WRITE(mpt, MPT_DOORBELL, ((__u8 *)rq)[0] | (((__u8 *)rq)[1] << 8) | (((__u8 *)rq)[2] << 16) | (((__u8 *)rq)[3] << 24));
		rq = (__u8 *)rq + 4;
		rq_len -= 4;
		if ((r = MPT_WAIT(mpt, MPT_INT_STATUS, MPT_INT_STATUS_IOCDOORBELL, 0, timeout, "DOORBELL MESSAGE ACK"))) return r;
	}
	return 0;
}

static int MPT_SYNC_REPLY(MPT *mpt, void *rep, unsigned rep_len, u_jiffies_lo_t timeout)
{
	int r;
	unsigned pos;
	if (__unlikely(rep_len & 3) || __unlikely(!rep_len))
		KERNEL$SUICIDE("MPT_SYNC_REPLY: INVALID REPLY LENGTH %X", rep_len);
	((__u8 *)rep)[2] = 1;
	pos = 0;
	do {
		__u16 data;
		if ((r = MPT_WAIT(mpt, MPT_INT_STATUS, MPT_INT_STATUS_DOORBELL, MPT_INT_STATUS_DOORBELL, timeout, "DOORBELL REPLY INTERRUPT"))) return r;
		data = MPT_READ(mpt, MPT_DOORBELL);
		if (__likely(pos < rep_len)) {
			((__u8 *)rep + pos)[0] = data;
			((__u8 *)rep + pos)[1] = data >> 8;
		}
		MPT_WRITE(mpt, MPT_INT_STATUS, 0);
	} while ((pos += 2) < ((__u8 *)rep)[2] * 4);

	while (__unlikely(pos < rep_len)) {
		*(__u32 *)((__u8 *)rep + pos) = 0;
		pos += 4;
	}
	if ((r = MPT_WAIT(mpt, MPT_INT_STATUS, MPT_INT_STATUS_DOORBELL, MPT_INT_STATUS_DOORBELL, timeout, "DOORBELL END INTERRUPT"))) return r;
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);
	return 0;
}

int MPT_SYNC_REQUEST_REPLY(MPT *mpt, void *rq, unsigned rq_len, void *rep, unsigned rep_len, u_jiffies_lo_t timeout)
{
	int r;
	if (__unlikely(r = MPT_SYNC_REQUEST(mpt, rq, rq_len, timeout)))
		return r;
	return MPT_SYNC_REPLY(mpt, rep, rep_len, timeout);
}

#define REPLY_ENDIAN_16(field)	(sizeof(field) != 2 ? KERNEL$SUICIDE("REPLY_ENDIAN_16: BUG AT LINE %d", __LINE__) : 0, (field) = __16LE2CPU(field))
#define REPLY_ENDIAN_32(field)	(sizeof(field) != 4 ? KERNEL$SUICIDE("REPLY_ENDIAN_32: BUG AT LINE %d", __LINE__) : 0, (field) = __32LE2CPU(field))

static int MPT_GET_IOCFACTS(MPT *mpt)
{
	int r;
	MPT_IOCFACTS_REQUEST req;
	memset(&req, 0, sizeof(req));
	req.function = MPT_FUNCTION_IOC_FACTS;
	if ((r = MPT_SYNC_REQUEST_REPLY(mpt, &req, sizeof(req), &mpt->iocfacts, sizeof(mpt->iocfacts), MPT_IOCFACTS_TIMEOUT)))
		return r;
	/*{
		int i;
		__debug_printf("iocfacts:");
		for (i = 0; i < sizeof(MPT_IOCFACTS_REPLY); i++) {
			__debug_printf(" %02x", ((__u8 *)&mpt->iocfacts)[i]);
		}
		__debug_printf("\n");
	}*/
	REPLY_ENDIAN_16(mpt->iocfacts.msg_version);
	REPLY_ENDIAN_16(mpt->iocfacts.header_version);
	REPLY_ENDIAN_32(mpt->iocfacts.msg_context);
	REPLY_ENDIAN_16(mpt->iocfacts.ioc_exceptions);
	REPLY_ENDIAN_16(mpt->iocfacts.ioc_status);
	REPLY_ENDIAN_32(mpt->iocfacts.ioc_log_info);
	REPLY_ENDIAN_16(mpt->iocfacts.reply_queue_depth);
	REPLY_ENDIAN_16(mpt->iocfacts.request_frame_size);
	REPLY_ENDIAN_16(mpt->iocfacts.fw_version_16);
	REPLY_ENDIAN_16(mpt->iocfacts.product_id);
	REPLY_ENDIAN_32(mpt->iocfacts.current_host_mfa_hi_addr);
	REPLY_ENDIAN_16(mpt->iocfacts.global_credits);
	REPLY_ENDIAN_32(mpt->iocfacts.current_sense_buffer_hi_addr);
	REPLY_ENDIAN_16(mpt->iocfacts.current_reply_frame_size);
	REPLY_ENDIAN_32(mpt->iocfacts.fw_image_size);
	REPLY_ENDIAN_32(mpt->iocfacts.ioc_capabilities);
	REPLY_ENDIAN_32(mpt->iocfacts.fw_version);
	REPLY_ENDIAN_16(mpt->iocfacts.high_priority_queue_depth);
	REPLY_ENDIAN_32(mpt->iocfacts.host_page_buffer_sge.sge_hdr);
	REPLY_ENDIAN_32(mpt->iocfacts.host_page_buffer_sge.sge_lo_addr);
	REPLY_ENDIAN_32(mpt->iocfacts.host_page_buffer_sge.sge_hi_addr);
	REPLY_ENDIAN_32(mpt->iocfacts.reply_fifo_host_signaling_addr);
	return 0;
}

/* Linux and OpenBSD assume that there is only one port
   ... so likely no one has manufactured multi-port IOC yet */

static int MPT_GET_PORTFACTS(MPT *mpt, unsigned port, MPT_PORTFACTS_REPLY *portfacts)
{
	int r;
	MPT_PORTFACTS_REQUEST req;
	memset(&req, 0, sizeof(req));
	req.function = MPT_FUNCTION_PORT_FACTS;
	req.port_number = port;
	if ((r = MPT_SYNC_REQUEST_REPLY(mpt, &req, sizeof(req), portfacts, sizeof(*portfacts), MPT_PORTFACTS_TIMEOUT)))
		return r;
	/*{
		int i;
		__debug_printf("portfacts:");
		for (i = 0; i < sizeof(MPT_PORTFACTS_REPLY); i++) {
			__debug_printf(" %02x", ((__u8 *)portfacts)[i]);
		}
		__debug_printf("\n");
	}*/
	REPLY_ENDIAN_32(portfacts->msg_context);
	REPLY_ENDIAN_16(portfacts->ioc_status);
	REPLY_ENDIAN_16(portfacts->max_devices);
	REPLY_ENDIAN_16(portfacts->port_scsi_id);
	REPLY_ENDIAN_16(portfacts->protocol_flags);
	REPLY_ENDIAN_16(portfacts->max_posted_cmd_buffers);
	REPLY_ENDIAN_16(portfacts->max_persistent_ids);
	REPLY_ENDIAN_16(portfacts->max_lan_buckets);
	return 0;
}

static int MPT_IOCINIT(MPT *mpt)
{
	MPT_IOCINIT_REQUEST req;
	MPT_IOCINIT_REPLY rpl;
	memset(&req, 0, sizeof(req));
	req.whoinit = IOCINIT_WHOINIT_HOST_DRIVER;
	req.function = MPT_FUNCTION_IOC_INIT;
	req.max_devices = mpt->iocfacts.max_devices;
	req.max_buses = mpt->iocfacts.max_buses;
	req.reply_frame_size = __16CPU2LE(MPT_REPLY_SIZE);
	req.host_mfa_hi_addr = __32CPU2LE(mpt->cdbs_pa >> 32);
	req.sense_buffer_hi_addr = __32CPU2LE(mpt->senses_pa >> 32);
	req.reply_hi_addr = __32CPU2LE(mpt->replies_pa >> 32);
	req.msg_version_maj = 0x01;
	req.msg_version_min = 0x02;
	req.hdr_version_unit = 0x0d;
	req.hdr_version_dev = 0x00;
	return MPT_SYNC_REQUEST_REPLY(mpt, &req, sizeof(req), &rpl, sizeof(rpl), MPT_IOCINIT_TIMEOUT);
}

static int MPT_PORT_ENABLE(MPT *mpt, unsigned port)
{
	MPT_PORTENABLE_REQUEST req;
	MPT_PORTENABLE_REPLY rpl;
	memset(&req, 0, sizeof(req));
	req.function = MPT_FUNCTION_PORT_ENABLE;
	req.port_number = port;
	return MPT_SYNC_REQUEST_REPLY(mpt, &req, sizeof(req), &rpl, sizeof(rpl), MPT_PORTENABLE_TIMEOUT);
}

static int MPT_WAIT_FOR_OPERATIONAL(MPT *mpt)
{
	return MPT_WAIT(mpt, MPT_DOORBELL, MPT_DOORBELL_IOC_STATE, MPT_DOORBELL_IOC_STATE_MPI_IOC_STATE_OPERATIONAL, MPT_OPERATIONAL_TIMEOUT, "OPERATIONAL STATE");
}

static void MPT_PUSH_REPLY_FRAMES(MPT *mpt)
{
	int i;
	__u32 paddr = (__u32)mpt->replies_pa + mpt->max_replies * MPT_REPLY_SIZE;
	for (i = mpt->max_replies - 1; i >= 0; i--) {
		paddr -= MPT_REPLY_SIZE;
		MPT_WRITE(mpt, MPT_REPLY_FIFO, paddr);
	}
}

static int MPT_SOFT_RESET(MPT *mpt, unsigned fn, int wait)
{
	int r;
	__u32 db;
	if (__unlikely((db = MPT_READ(mpt, MPT_DOORBELL)) & MPT_DOORBELL_ACTIVE)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "DOORBELL IS ALREADY ACTIVE WHEN ATTEMPTING TO RESET: %08X", (unsigned)db);
		return -EIO;
	}
	MPT_WRITE(mpt, MPT_DOORBELL, fn << __BSF_CONST(MPT_DOORBELL_FUNCTION));
	if ((r = MPT_WAIT(mpt, MPT_INT_STATUS, MPT_INT_STATUS_IOCDOORBELL, 0, MPT_RESET_ACK_TIMEOUT, "RESET ACK"))) return r;
	if (!wait) {
		UDELAY(mpt, 100, 1);
		return 0;
	}
	return MPT_WAIT(mpt, MPT_DOORBELL, MPT_DOORBELL_IOC_STATE, MPT_DOORBELL_IOC_STATE_MPI_IOC_STATE_READY, MPT_RESET_READY_TIMEOUT, "READY STATE");
}

static int MPT_ENABLE(MPT *mpt)
{
	int r;
	if ((r = MPT_IOCINIT(mpt))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "CAN'T INIT IOC");
		return r;
	}

	if ((r = MPT_PORT_ENABLE(mpt, 0))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "CAN'T ENABLE PORT");
		return r;
	}

	if ((r = MPT_WAIT_FOR_OPERATIONAL(mpt))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "IOC NOT OPERATIONAL");
		return r;
	}
	KERNEL$UDELAY_PREPARE(&mpt->udelay_cookie);
	return 0;
}

static void MPT_TIMEOUT_FN(TIMER *t)
{
	MPT *mpt = GET_STRUCT(t, MPT, timeout);
	u_jiffies_lo_t j;
	LOWER_SPL(SPL_ATA_SCSI);
	mpt->timeout_posted = 0;
	j = MPT_SCSI_TEST_TIMEOUT(mpt);
	if (!j) {
		if (MPT_SOFT_RESET(mpt, MPT_FUNCTION_IO_UNIT_RESET, 0) ||
		    MPT_SOFT_RESET(mpt, MPT_FUNCTION_IOC_MESSAGE_UNIT_RESET, 1) ||
		    MPT_ENABLE(mpt)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "RESET FAILED, DISABLING THE CONTROLLER. DOORBELL %08X", MPT_READ(mpt, MPT_DOORBELL));
			mpt->chip_flags |= MPT_DEAD;
			MPT_SCSI_KILL_CONTROLLER(mpt);
		}
		if (!(mpt->chip_flags & MPT_DEAD))
			MPT_PUSH_REPLY_FRAMES(mpt);
		MPT_SCSI_BUST_REQUESTS(mpt);
	} else if (j != (u_jiffies_lo_t)-1) {
		KERNEL$SET_TIMER(j, &mpt->timeout);
		mpt->timeout_posted = 1;
	}
}

static int MPT_UNLOAD(void *ptr, void **unload, const char * const argv[]);

int main(int argc, const char * const argv[])
{
	MPT *mpt;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	const char * const *arg = argv;
	int state = 0;
	const char *opt, *optend, *str;
	char *chip_name;
	unsigned long chip_flags;
	int r;
	int res, bar;
	int irq;
	unsigned udelay_val = MPT_DEFAULT_UDELAY_VAL;
	union {
		VDESC vdesc;
		char pci_id_str[__MAX_STR_LEN];
	} u;
	union {
		VDMA64 vdma64;
	} u2;
	static const struct __param_table params[2] = {
		"UDELAY", __PARAM_UNSIGNED_INT, 0, 255,
		NULL, 0, 0, 0,
	};
	void *vars[2];
	vars[0] = &udelay_val;
	vars[1] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MPT: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &chip_flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MPT: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}

	mpt = KERNEL$ALLOC_CONTIG_AREA(sizeof(MPT), AREA_DATA);
	if (__IS_ERR(mpt)) {
		r = __PTR_ERR(mpt);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "MPT: CAN'T ALLOCATE MEMORY");
		goto ret1;
	}
	memset(mpt, 0, sizeof(MPT));
	mpt->pci_id = id;
	mpt->chip_flags = chip_flags;
	_snprintf(mpt->dev_name, __MAX_STR_LEN, "SCSI$MPT@" PCI_ID_FORMAT, id);
	mpt->sync_sleep = 1;
	mpt->udelay_val = udelay_val;
	KERNEL$UDELAY_PREPARE(&mpt->udelay_cookie);
	INIT_LIST(&mpt->attached_list);
	mpt->timeout.fn = MPT_TIMEOUT_FN;
	INIT_TIMER(&mpt->timeout);

	for (res = -1; (bar = PCI$NEXT_RESOURCE(id, &res)) >= 0; ) {
		if ((bar & PCI_BASE_ADDRESS_SPACE) == PCI_BASE_ADDRESS_SPACE_MEMORY) {
			mpt->mem = PCI$MAP_MEM_RESOURCE(id, res, MPT_REGSPACE);
			if (__unlikely(!mpt->mem)) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "NO MEM RESOURCE IN SLOT %d", res);
				no_res:
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", mpt->dev_name);
				r = -ENXIO;
				goto ret2;
			}
			if (__IS_ERR(mpt->mem)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", mpt->dev_name, strerror(-__PTR_ERR(mpt->mem)));
				r = __PTR_ERR(mpt->mem);
				goto ret2;
			}
			goto have_mem;
		}
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "NO MEM RESOURCE");
	goto no_res;

	have_mem:
	if (chip_flags & FIX_DISABLE_IO) {
		__u16 pcicmd = PCI$READ_CONFIG_WORD(id, PCI_COMMAND);
		if (pcicmd & PCI_COMMAND_IO) {
			pcicmd &= ~PCI_COMMAND_IO;
			PCI$WRITE_CONFIG_WORD(id, PCI_COMMAND, pcicmd);
		}
	}
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	if (chip_flags & FIX_PCIX_CMD) {
		__u16 devid = PCI$READ_CONFIG_WORD(id, PCI_DEVICE_ID);
		__u8 revision = PCI$READ_CONFIG_BYTE(id, PCI_REVISION_ID);
		__u8 pcixcmd = PCI$READ_CONFIG_BYTE(id, 0x6a);
		if (devid == 0x0626) {
			if (!revision) {
				pcix_most_off:
				pcixcmd &= ~0x70;
			} else {
				pcixcmd |= 0x08;
			}
			PCI$WRITE_CONFIG_BYTE(id, 0x6a, pcixcmd);
		} else if (devid == 0x0627) {
			goto pcix_most_off;
		} else if (devid == 0x0030) {
			if (revision < 0x08) {
				goto pcix_most_off;
			}
		}
	}

	MPT_WRITE(mpt, MPT_INT_MASK, ~0);
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);

	/*if ((r = MPT_SOFT_RESET(mpt, MPT_FUNCTION_IO_UNIT_RESET, 0))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T RESET THE IO UNIT", mpt->dev_name);
		goto ret3;
	}*/

	if ((r = MPT_SOFT_RESET(mpt, MPT_FUNCTION_IOC_MESSAGE_UNIT_RESET, 1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T RESET THE MESSAGE UNIT", mpt->dev_name);
		goto ret3;
	}

	if ((r = MPT_GET_IOCFACTS(mpt))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET IOCFACTS", mpt->dev_name);
		goto ret3;
	}
	if ((r = MPT_GET_PORTFACTS(mpt, 0, &mpt->portfacts))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET PORTFACTS", mpt->dev_name);
		goto ret3;
	}
	if (mpt->portfacts.port_type == PORTFACTS_PORT_TYPE_INACTIVE) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: PORT IS INACTIVE", mpt->dev_name);
		goto ret3;
	}
	if (!(mpt->portfacts.protocol_flags & PORTFACTS_PROTOCOL_FLAGS_INITIATOR)) {
		r = -EIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: PORT DOESN'T SUPPORT INITIATOR MODE, FLAGS %04X", mpt->dev_name, mpt->portfacts.protocol_flags);
		goto ret3;
	}

	if (__unlikely(!mpt->iocfacts.global_credits) ||
	    __unlikely(!mpt->iocfacts.reply_queue_depth)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "MPT REPORTS BAD NUMBER OF COMMANDS (%u) OR REPLIES (%u)", (unsigned)mpt->iocfacts.global_credits, (unsigned)mpt->iocfacts.reply_queue_depth);
		r = -EIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE", mpt->dev_name);
		goto ret3;
	}

	mpt->max_cdbs = mpt->iocfacts.global_credits;
	if (mpt->max_cdbs * MPT_CDB_SIZE > PG_SIZE * PG_BANK)
		mpt->max_cdbs = PG_SIZE * PG_BANK / MPT_CDB_SIZE;
	if (mpt->max_cdbs * MPT_SENSE_SIZE > PG_SIZE * PG_BANK)
		mpt->max_cdbs = PG_SIZE * PG_BANK / MPT_SENSE_SIZE;
	if (mpt->max_cdbs > MPT_MAX_CDBS_BOUNCE && KERNEL$RESERVE_BOUNCE_ACTIVE(BOUNCE_DMA64))
		mpt->max_cdbs = MPT_MAX_CDBS_BOUNCE;
	mpt->max_replies = mpt->iocfacts.reply_queue_depth;
	if (mpt->max_replies * MPT_REPLY_SIZE > PG_SIZE * PG_BANK)
		mpt->max_replies = PG_SIZE * PG_BANK / MPT_REPLY_SIZE;

	mpt->cdbs = KERNEL$ALLOC_CONTIG_AREA(mpt->max_cdbs * MPT_CDB_SIZE, AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA64 | AREA_ALIGN, (unsigned long)MPT_CDB_ALIGN);
	if (__IS_ERR(mpt->cdbs)) {
		r = __PTR_ERR(mpt->cdbs);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE COMMAND BUFFER: %s", mpt->dev_name, strerror(-r));
		goto ret3;
	}
	memset(mpt->cdbs, 0, mpt->max_cdbs * MPT_CDB_SIZE);

	mpt->senses = KERNEL$ALLOC_CONTIG_AREA(mpt->max_cdbs * MPT_SENSE_SIZE, AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA64 | AREA_ALIGN, (unsigned long)MPT_SENSE_ALIGN);
	if (__IS_ERR(mpt->senses)) {
		r = __PTR_ERR(mpt->senses);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE SENSE BUFFER: %s", mpt->dev_name, strerror(-r));
		goto ret4;
	}
	memset(mpt->senses, 0, mpt->max_cdbs * MPT_SENSE_SIZE);

	mpt->replies = KERNEL$ALLOC_CONTIG_AREA(mpt->max_replies * MPT_REPLY_SIZE, AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA64 | AREA_ALIGN, (unsigned long)MPT_REPLY_ALIGN);
	if (__IS_ERR(mpt->replies)) {
		r = __PTR_ERR(mpt->replies);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE REPLY BUFFER: %s", mpt->dev_name, strerror(-r));
		goto ret5;
	}
	memset(mpt->replies, 0, mpt->max_replies * MPT_REPLY_SIZE);

	u.vdesc.ptr = (unsigned long)mpt->cdbs;
	u.vdesc.len = mpt->max_cdbs * MPT_CDB_SIZE;
	u.vdesc.vspace = &KERNEL$VIRTUAL;
	u2.vdma64.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	mpt->cdbs_unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&u.vdesc, PF_RW, &u2.vdma64);
	if (__unlikely(u2.vdma64.len != u.vdesc.len))
		KERNEL$SUICIDE("MPT: UNABLE TO LOCK DMA FOR CDBS: %u != %lu", u2.vdma64.len, u.vdesc.len);
	mpt->cdbs_pa = u2.vdma64.ptr;
	u.vdesc.ptr = (unsigned long)mpt->senses;
	u.vdesc.len = mpt->max_cdbs * MPT_SENSE_SIZE;
	u.vdesc.vspace = &KERNEL$VIRTUAL;
	u2.vdma64.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	mpt->senses_unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&u.vdesc, PF_RW, &u2.vdma64);
	if (__unlikely(u2.vdma64.len != u.vdesc.len))
		KERNEL$SUICIDE("MPT: UNABLE TO LOCK DMA FOR SENSES: %u != %lu", u2.vdma64.len, u.vdesc.len);
	mpt->senses_pa = u2.vdma64.ptr;
	u.vdesc.ptr = (unsigned long)mpt->replies;
	u.vdesc.len = mpt->max_replies * MPT_REPLY_SIZE;
	u.vdesc.vspace = &KERNEL$VIRTUAL;
	u2.vdma64.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	mpt->replies_unlock = KERNEL$VIRTUAL.op->vspace_dma64lock(&u.vdesc, PF_RW, &u2.vdma64);
	if (__unlikely(u2.vdma64.len != u.vdesc.len))
		KERNEL$SUICIDE("MPT: UNABLE TO LOCK DMA FOR REPLIES: %u != %lu", u2.vdma64.len, u.vdesc.len);
	mpt->replies_pa = u2.vdma64.ptr;

	if ((MPT_ENABLE(mpt))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ENABLE CONTROLLER", mpt->dev_name);
		goto ret6;
	}

	if (mpt->portfacts.port_type == PORTFACTS_PORT_TYPE_SCSI) {
		MPT_INIT_SPI_PORT(mpt);
	}

	MPT_INIT_CDBS(mpt);
	if ((r = KERNEL$RESERVE_BOUNCE(mpt->dev_name, mpt->max_cdbs * mpt->useful_sges, __PAGE_CLUSTER_SIZE, BOUNCE_DMA64))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT RESERVE BOUNCE BUFFERS: %s", mpt->dev_name, strerror(-r));
		goto ret6;
	}

	irq = PCI$READ_INTERRUPT_LINE(mpt->pci_id);

	_printf("%s: SCSI CONTROLLER ON PCI: %s\n", mpt->dev_name, PCI$ID(u.pci_id_str, mpt->pci_id));
	_printf("%s: %s\n", mpt->dev_name, chip_name);
	_printf("%s: MEM %"__64_format"X, IRQ %d\n", mpt->dev_name, PCI$READ_MEM_RESOURCE(id, res, NULL), irq);

	mpt->irq_ast.fn = MPT_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(irq, &mpt->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &mpt->irq_ast, mpt->dev_name)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, mpt->dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", mpt->dev_name);
		goto ret65;
	}
	MPT_WRITE(mpt, MPT_INT_MASK, ~MPT_INT_MASK_REPLY);
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);

	MPT_PUSH_REPLY_FRAMES(mpt);
	UDELAY(mpt, 100, 0);
	if ((MPT_READ(mpt, MPT_DOORBELL) & MPT_DOORBELL_IOC_STATE) != MPT_DOORBELL_IOC_STATE_MPI_IOC_STATE_OPERATIONAL) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "ERROR INITIALIZING CONTROLLER, DOORBELL %08X", MPT_READ(mpt, MPT_DOORBELL));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: IOC NOT OPERATIONAL", mpt->dev_name);
		r = -EIO;
		goto ret7;
	}

	mpt->sync_sleep = 0;
	KERNEL$UNMASK_IRQ(mpt->irq_ctrl);

	r = KERNEL$REGISTER_DEVICE(mpt->dev_name, "MPT.SYS", 0, mpt, NULL, MPT_DCALL, "SCSI", NULL, MPT_UNLOAD, &mpt->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", mpt->dev_name, strerror(-r));
		goto ret8;
	}

	mpt->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), mpt->dev_name);
	return 0;

	ret8:
	KERNEL$MASK_IRQ(mpt->irq_ctrl);
	RAISE_SPL(SPL_ATA_SCSI);
	if (mpt->timeout_posted) KERNEL$DEL_TIMER(&mpt->timeout);
	mpt->sync_sleep = 1;
	LOWER_SPL(SPL_ZERO);
	ret7:
	MPT_WRITE(mpt, MPT_INT_MASK, ~0);
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);
	KERNEL$RELEASE_IRQ(mpt->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &mpt->irq_ast);
	ret65:
	KERNEL$UNRESERVE_BOUNCE(mpt->dev_name, mpt->max_cdbs * mpt->useful_sges, __PAGE_CLUSTER_SIZE, BOUNCE_DMA64);
	ret6:
	if (!(mpt->chip_flags & MPT_DEAD))
		MPT_SOFT_RESET(mpt, MPT_FUNCTION_IOC_MESSAGE_UNIT_RESET, 1);
	mpt->cdbs_unlock(mpt->cdbs_pa);
	mpt->senses_unlock(mpt->senses_pa);
	mpt->replies_unlock(mpt->replies_pa);
	KERNEL$FREE_CONTIG_AREA(mpt->replies, mpt->max_replies * MPT_REPLY_SIZE);
	ret5:
	KERNEL$FREE_CONTIG_AREA(mpt->senses, mpt->max_cdbs * MPT_SENSE_SIZE);
	ret4:
	KERNEL$FREE_CONTIG_AREA(mpt->cdbs, mpt->max_cdbs * MPT_CDB_SIZE);
	ret3:
	PCI$UNMAP_MEM_RESOURCE(id, mpt->mem, MPT_REGSPACE);
	ret2:
	KERNEL$FREE_CONTIG_AREA(mpt, sizeof(MPT));
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int MPT_UNLOAD(void *ptr, void **unload, const char * const argv[])
{
	int r;
	MPT *mpt = ptr;
	RAISE_SPL(SPL_ATA_SCSI);
	if (!LIST_EMPTY(&mpt->attached_list)) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS CONTROLLER", mpt->dev_name);
		return -EBUSY;
	}
	LOWER_SPL(SPL_DEV);
	r = KERNEL$DEVICE_UNLOAD(mpt->lnte, argv);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r)) return r;
	KERNEL$MASK_IRQ(mpt->irq_ctrl);
	RAISE_SPL(SPL_ATA_SCSI);
	if (mpt->timeout_posted) KERNEL$DEL_TIMER(&mpt->timeout);
	mpt->sync_sleep = 1;
	LOWER_SPL(SPL_ZERO);
	MPT_WRITE(mpt, MPT_INT_MASK, ~0);
	MPT_WRITE(mpt, MPT_INT_STATUS, 0);
	KERNEL$RELEASE_IRQ(mpt->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &mpt->irq_ast);
	if (!(mpt->chip_flags & MPT_DEAD))
		MPT_SOFT_RESET(mpt, MPT_FUNCTION_IOC_MESSAGE_UNIT_RESET, 1);
	KERNEL$UNRESERVE_BOUNCE(mpt->dev_name, mpt->max_cdbs * mpt->useful_sges, __PAGE_CLUSTER_SIZE, BOUNCE_DMA64);
	mpt->cdbs_unlock(mpt->cdbs_pa);
	mpt->senses_unlock(mpt->senses_pa);
	mpt->replies_unlock(mpt->replies_pa);
	KERNEL$FREE_CONTIG_AREA(mpt->replies, mpt->max_replies * MPT_REPLY_SIZE);
	KERNEL$FREE_CONTIG_AREA(mpt->senses, mpt->max_cdbs * MPT_SENSE_SIZE);
	KERNEL$FREE_CONTIG_AREA(mpt->cdbs, mpt->max_cdbs * MPT_CDB_SIZE);
	PCI$UNMAP_MEM_RESOURCE(mpt->pci_id, mpt->mem, MPT_REGSPACE);
	PCI$FREE_DEVICE(mpt->pci_id);
	*unload = mpt->dlrq;
	KERNEL$FREE_CONTIG_AREA(mpt, sizeof(MPT));
	return 0;
}
