#ifndef __SCSI_HOST_MPT_H
#define __SCSI_HOST_MPT_H

#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <SPAD/PCI.H>
#include <SPAD/TIMER.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SCSI.H>

#include "MPTREG.H"

#define MPT_MAX_CDBS_BOUNCE	16
#define MPT_MAX_SG		123
#define MPT_SENSE_SIZE		248
#define MPT_IOCFACTS_TIMEOUT	(5 * JIFFIES_PER_SECOND)
#define MPT_PORTFACTS_TIMEOUT	(5 * JIFFIES_PER_SECOND)
#define MPT_IOCINIT_TIMEOUT	(10 * JIFFIES_PER_SECOND)
#define MPT_PORTENABLE_TIMEOUT	(300 * JIFFIES_PER_SECOND)
#define MPT_OPERATIONAL_TIMEOUT	(60 * JIFFIES_PER_SECOND)
#define MPT_CFG_TIMEOUT		(50 * JIFFIES_PER_SECOND)
#define MPT_RESET_ACK_TIMEOUT	(5 * JIFFIES_PER_SECOND)
#define MPT_RESET_READY_TIMEOUT	(15 * JIFFIES_PER_SECOND)
#define MPT_WAIT_USEC		100
#define MPT_WAIT_STEPS_SLEEP	1
#define MPT_TAGS_PER_DEV	64
#define MPT_COMMAND_TIMEOUT	(10 * JIFFIES_PER_SECOND)
#define MPT_DEFAULT_UDELAY_VAL	0

#define BUS_SPI			0x01
#define BUS_FC			0x02
#define BUS_SAS			0x04
#define FIX_DISABLE_IO		0x08
#define FIX_PCIX_CMD		0x10
#define MPT_DEAD		0x80

typedef struct __mpt_cdb MPT_CDB;

struct __mpt_cdb {
	MPT_MSG_SCSI_IO msg;
	MPT_SGE sge[MPT_MAX_SG];
	__u32 paddr;
	char need_unmap;
	char aborting;
	SCSIRQ *scsirq;
	MPT_CDB *next;
	vspace_dma64unlock_t *unlock[MPT_MAX_SG];
	u_jiffies_lo_t abort_time;
};

#define MPT_CDB_ALIGN			16
#define MPT_CDB_SIZE			((sizeof(MPT_CDB) + MPT_CDB_ALIGN - 1) & ~(MPT_CDB_ALIGN - 1))

#define MPT_REPLY_ALIGN			8
#define MPT_REPLY_SIZE			((sizeof(MPT_REPLY_SCSI_IO_ERROR) + MPT_REPLY_ALIGN - 1) & ~(MPT_REPLY_ALIGN - 1))

#define MPT_SENSE_ALIGN			16

typedef struct {
	__u8 *mem;

	AST irq_ast;
	IRQ_HANDLE *irq_ctrl;

	MPT_CDB *free_cdbs;

	LIST_HEAD attached_list;

	MPT_CDB *cdbs;
	unsigned max_cdbs;

	char need_dequeue;
	__u8 udelay_val;

	__u8 used_tags[255 * 256];

	udelay_cookie_t udelay_cookie;

	__u64 cdbs_pa;
	__u64 senses_pa;
	__u64 replies_pa;

	unsigned useful_sges;
	__u8 (*senses)[MPT_SENSE_SIZE];
	MPT_REPLY_SCSI_IO_ERROR *replies;
	unsigned max_replies;

	vspace_dma64unlock_t *cdbs_unlock;
	vspace_dma64unlock_t *senses_unlock;
	vspace_dma64unlock_t *replies_unlock;

	MPT_IOCFACTS_REPLY iocfacts;
	MPT_PORTFACTS_REPLY portfacts;

	TIMER timeout;
	char timeout_posted;

	char sync_sleep;
	__u8 chip_flags;
	pci_id_t pci_id;
	void *dlrq;
	void *lnte;
	char dev_name[__MAX_STR_LEN];
} MPT;

#define MAX_CHANNELS(mpt)	((mpt)->iocfacts.max_buses)
#define MAX_IDS(mpt)		((__u8)((mpt)->iocfacts.max_devices - 1) + 1)

/* must match beginning of MPT_MSG_SCSI_IO */
#define CHANNEL(ch_id)		((ch_id) >> 8)
#define ID(ch_id)		((__u8)(ch_id))
#define CH_ID(ch, id)		(((ch) << 8) | (id))

static __finline__ void FREE_CDB(MPT *mpt, MPT_CDB *cdb)
{
	cdb->scsirq = NULL;
	cdb->next = mpt->free_cdbs;
	mpt->free_cdbs = cdb;
}

static __finline__ void DONE_RQ(MPT *mpt, SCSIRQ *rq)
{
	rq->internal = 0;
#if __DEBUG >= 1
	if (__unlikely(mpt->used_tags[rq->ch_id] < rq->host_aux))
	        KERNEL$SUICIDE("DONE_RQ: TAG COUNT UNDERFLOW: %d < %d", mpt->used_tags[rq->ch_id], rq->host_aux);
#endif
	mpt->used_tags[rq->ch_id] -= rq->host_aux;
	if (__unlikely(mpt->need_dequeue)) {
		mpt->need_dequeue = 0;
		SCSI$HOST_DEQUEUE(&mpt->attached_list);
	}
	rq->done(rq);
}

#define MPT_READ(mpt, reg)		mmio_inl((mpt)->mem + (reg))
#define MPT_WRITE(mpt, reg, val)	mmio_outl((mpt)->mem + (reg), val)

/* MPT.C */
int MPT_SYNC_REQUEST_REPLY(MPT *mpt, void *rq, unsigned rq_len, void *rep, unsigned rep_len, u_jiffies_lo_t timeout);

/* MPTCFG.C */
void MPT_INIT_SPI_PORT(MPT *mpt);

/* MPTSCSI.C */
int MPT_DCALL(void *ptr, const char *dcall_type, int cmd, va_list params);
int MPT_SLOW_REPLY(MPT *mpt, __u32 reply);
void MPT_CANCEL(SCSIRQ *rq);
void MPT_INIT_CDBS(MPT *mpt);
u_jiffies_lo_t MPT_SCSI_TEST_TIMEOUT(MPT *mpt);
void MPT_SCSI_BUST_REQUESTS(MPT *mpt);
void MPT_SCSI_KILL_CONTROLLER(MPT *mpt);

/* MPTCORE.C */
void MPT_UNMAP_CDB(MPT_CDB *cdb);
extern AST_STUB MPT_IRQ;
int MPT_POST(SCSIRQ *rq);

#endif
