#include <SPAD/SCSI.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/BSF.H>
#include <SYS/PARAM.H>
#include <ERRNO.H>

#include "MPT.H"
#include "MPTREG.H"

int MPT_DCALL(void *ptr, const char *dcall_type, int cmd, va_list params)
{
	MPT *mpt = ptr;
	SCSI_ATTACH_PARAM *ap = va_arg(params, SCSI_ATTACH_PARAM *);
	if (KERNEL$SPL != SPL_X(SPL_DEV))
		KERNEL$SUICIDE("MPT_DCALL AT SPL %08X", KERNEL$SPL);
	switch (cmd) {
	case SCSI_CMD_GET: {
		int channel = -1;
		int id = -1;
		unsigned c, i;
		const char *arg;
		int state;
		static const struct __param_table params[3] = {
			"CHANNEL", __PARAM_INT, 0, 256,
			"ID", __PARAM_INT, 0, 256,
			NULL, 0, 0, 0,
		};
		void *vars[3];
		vars[0] = &channel;
		vars[1] = &id;
		vars[2] = NULL;
		if (__unlikely(ap->u.p.version != SCSI_VERSION)) {
			ap->msg = "INCORRECT VERSION";
			return -EINVAL;
		}
		arg = ap->u.p.param;
		state = 0;
		if (__unlikely(__parse_extd_param(&arg, &state, params, vars, NULL, NULL, NULL, NULL))) {
			ap->msg = "SYNTAX ERROR";
			return -EBADSYN;
		}
		if (__unlikely(channel >= MAX_CHANNELS(mpt))) {
			ap->msg = "INVALID BUS NUMBER";
			return -ERANGE;
		}
		if (__unlikely(id >= MAX_IDS(mpt))) {
			ap->msg = "INVALID DEVICE ID";
			return -ERANGE;
		}
		if (mpt->chip_flags & MPT_DEAD) {
			ap->msg = "THE CONTROLLER IS DEAD";
			return -ESHUTDOWN;
		}
		for (c = 0; c < MAX_CHANNELS(mpt); c++) {
			if (ap->adapter && CHANNEL(ap->ch_id) > c) continue;
			for (i = 0; i < MAX_IDS(mpt); i++) {
				if (ap->adapter && CHANNEL(ap->ch_id) == c && ID(ap->ch_id) >= i) continue;
				ap->max_sectors = SCSI$HOST_SGENTS_TO_SECTORS(mpt->useful_sges);
				if (!ap->max_sectors) ap->max_sectors = 1;
				ap->max_fragments = mpt->useful_sges;
				ap->max_untagged_requests = 1;
				ap->max_tagged_requests = MPT_TAGS_PER_DEV;
				ap->scsi_flags = SCSI_FLAG_CAN_USE_MODE_10 | SCSI_FLAG_CAN_USE_MODE_6;
				ap->lun = LUN_NONE;
				ap->ch_id = CH_ID(c, i);
				ap->adapter = mpt;
				ap->post = MPT_POST;
				ap->cancel = MPT_CANCEL;
				SCSI$HOST_PRINT_NAME(ap, mpt->dev_name, c, MAX_CHANNELS(mpt), i, MAX_IDS(mpt));
				RAISE_SPL(SPL_ATA_SCSI);
				ADD_TO_LIST(&mpt->attached_list, &ap->u.h.client_list);
				LOWER_SPL(SPL_DEV);
				return 0;
			}
		}
		return -ENODEV;
	}
	case SCSI_CMD_FREE: {
		if (__unlikely(ap->adapter != mpt))
			KERNEL$SUICIDE("MPT_DCALL: INVALID ADAPTER %p != %p", ap->adapter, mpt);
		RAISE_SPL(SPL_ATA_SCSI);
		DEL_FROM_LIST(&ap->u.h.client_list);
		LOWER_SPL(SPL_DEV);
		return 0;
	}
	case SCSI_CMD_SET_LUN: {
		int r;
		if (__unlikely(r = SCSI$HOST_IS_LUN_DUPLICATE(&mpt->attached_list, ap))) {
			ap->lun = LUN_NONE;
			return r;
		}
		SCSI$HOST_PRINT_NAME(ap, mpt->dev_name, CHANNEL(ap->ch_id), MAX_CHANNELS(mpt), ID(ap->ch_id), MAX_IDS(mpt));
		return 0;
	}
	default: {
		return -EINVAL;
	}
	}
}

int MPT_SLOW_REPLY(MPT *mpt, __u32 reply)
{
	MPT_REPLY_SCSI_IO_ERROR *ioe;
	__u32 idx;
	unsigned xfer_count;
	MPT_CDB *cdb;
	SCSIRQ *scsirq;
	if (__unlikely(!(reply & MPT_REPLY_FIFO_ADDRESS))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "%s %08X (COMMANDS %X)", (reply & (MPT_REPLY_FIFO_TYPE | MPT_REPLY_FIFO_ADDRESS)) == MPT_REPLY_FIFO_TYPE_INITIATOR ? "INVALID REPLY INDEX" : "UNKNOWN REPLY TYPE", reply, mpt->max_cdbs);
		return -1;
	}
	reply <<= 1;
	reply -= (__u32)mpt->replies_pa;
	if (__unlikely(reply % MPT_REPLY_SIZE) ||
	    __unlikely(reply >= mpt->max_replies * MPT_REPLY_SIZE)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "INVALID REPLY ADDRESS, OFFSET %08X, BASE %016"__64_format"X, REPLY SIZE %X, REPLIES %X", reply, mpt->replies_pa, MPT_REPLY_SIZE, mpt->max_replies);
		return -1;
	}
	ioe = (MPT_REPLY_SCSI_IO_ERROR *)((char *)mpt->replies + reply);
	if (__unlikely(ioe->function != MPT_FUNCTION_SCSI_IO_REQUEST)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "UNKNOWN FUNCTION IN REPLY: %02X", ioe->function);
		goto free_reply_err;
	}
	idx = __32LE2CPU(ioe->msg_context);
	if (__unlikely(idx >= mpt->max_cdbs)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "INVALID INDEX IN REPLY BLOCK %08X (COMMANDS %X)", idx, mpt->max_cdbs);
		goto free_reply_err;
	}
	cdb = (MPT_CDB *)((char *)mpt->cdbs + idx * MPT_CDB_SIZE);
	scsirq = cdb->scsirq;
	if (__unlikely(!scsirq)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, mpt->dev_name, "FREE COMMAND BLOCK RETURNED IN REPLY BLOCK: %08X", idx);
		goto free_reply_err;
	}
	MPT_UNMAP_CDB(cdb);

	/*__debug_printf("ioc status %x, scsi status %x, xf_count %x\n", ioe->ioc_status, ioe->scsi_status, ioe->transfer_count);*/
	switch (__16LE2CPU(ioe->ioc_status) & MPT_IOCSTATUS_MASK) {
	case MPT_IOCSTATUS_BUSY: {
		scsirq->status = -EBUSY;
		break;
	}
	case MPT_IOCSTATUS_SCSI_INVALID_BUS:
	case MPT_IOCSTATUS_SCSI_INVALID_TARGETID: {
		scsirq->status = -EINVAL;
		break;
	}
	case MPT_IOCSTATUS_SCSI_DEVICE_NOT_THERE: {
		scsirq->status = -ENXIO;
		break;
	}
	case MPT_IOCSTATUS_SCSI_IOC_TERMINATED:
	case MPT_IOCSTATUS_SCSI_TASK_TERMINATED:
	case MPT_IOCSTATUS_SCSI_EXT_TERMINATED: {
		scsirq->status = -ECONNABORTED;
		break;
	}
	case MPT_IOCSTATUS_SCSI_PROTOCOL_ERROR: {
		scsirq->status = -EPROTO;
		break;
	}
	case MPT_IOCSTATUS_SCSI_IO_DATA_ERROR: {
		scsirq->status = -EIO;
		break;
	}
	case MPT_IOCSTATUS_SCSI_RESIDUAL_MISMATCH:
	case MPT_IOCSTATUS_SCSI_DATA_UNDERRUN:
	case MPT_IOCSTATUS_SCSI_DATA_OVERRUN:
	case MPT_IOCSTATUS_SCSI_RECOVERED_ERROR:
	case MPT_IOCSTATUS_SUCCESS: {
		break;
	}
	default: {
		scsirq->status = -EPROTO;
		break;
	}
	}

	if (__unlikely(scsirq->status < 0))
		goto ret_it;

	xfer_count = __32LE2CPU(ioe->transfer_count);
	if (__likely(scsirq->status > xfer_count)) {
		scsirq->status = xfer_count;
	}

	scsirq->scsi_status = ioe->scsi_status;
	if (scsirq->scsi_status == SCSI_CHECK_CONDITION) {
		unsigned sense_size = ioe->scsi_state & REPLY_SCSI_STATE_AUTOSENSE_VALID ? ioe->sense_count : 0;
		__u8 *sense_ptr = mpt->senses[idx];
		/*{
			int i;
			__debug_printf("command:");
			for (i = 0; i < scsirq->cmdlen; i++)
				__debug_printf(" %02x", scsirq->cmd[i]);
			__debug_printf("\n");
			__debug_printf("sense size %d, sense_ptr %p, pa %08x\n", sense_size, sense_ptr, cdb->msg.sense_lo_addr);
			for (i = 0; i < sense_size; i++)
				__debug_printf("%02x ", sense_ptr[i]);
			__debug_printf("\n");
		}*/
		if (sense_size >= scsirq->sense_size)
			memcpy(scsirq->sense, sense_ptr, scsirq->sense_size);
		else
			memset(mempcpy(scsirq->sense, sense_ptr, sense_size), 0, scsirq->sense_size - sense_size);
	}

	ret_it:
	MPT_WRITE(mpt, MPT_REPLY_FIFO, (__u32)mpt->replies_pa + reply);
	FREE_CDB(mpt, cdb);
	DONE_RQ(mpt, scsirq);
	return 0;

	free_reply_err:
	MPT_WRITE(mpt, MPT_REPLY_FIFO, (__u32)mpt->replies_pa + reply);
	return -1;
}

void MPT_CANCEL(SCSIRQ *rq)
{
	MPT *mpt;
	MPT_CDB *cdb;

	if (!rq->internal)
		return;

	mpt = rq->adapter;
	cdb = (MPT_CDB *)rq->internal;

	if (__unlikely(cdb->aborting))
		return;

	cdb->aborting = 1;
	cdb->abort_time = KERNEL$GET_JIFFIES_LO();

	if (!mpt->timeout_posted) {
		mpt->timeout_posted = 1;
		KERNEL$SET_TIMER(MPT_COMMAND_TIMEOUT, &mpt->timeout);
	}
	/* TODO: attempt to abort it at the controller */
}

void MPT_INIT_CDBS(MPT *mpt)
{
	int i, j;
	int n_first_sge = MAX(((int)mpt->iocfacts.request_frame_size * 4 - (int)sizeof(MPT_MSG_SCSI_IO)) / (int)sizeof(MPT_SGE), 2);
	int n_sge = MAX(mpt->iocfacts.request_frame_size * 4 / sizeof(MPT_SGE), 2);
	MPT_CDB *cdb = (MPT_CDB *)((char *)mpt->cdbs + mpt->max_cdbs * MPT_CDB_SIZE);
	__u32 paddr = mpt->cdbs_pa + mpt->max_cdbs * MPT_CDB_SIZE + __offsetof(MPT_CDB, msg);
	__u32 spaddr = mpt->senses_pa + mpt->max_cdbs * MPT_SENSE_SIZE;
	mpt->free_cdbs = NULL;
	/*__debug_printf("n_sge %d, n_first_sge %d\n", n_sge, n_first_sge);*/
	for (i = mpt->max_cdbs - 1; i >= 0; i--) {
		int skips;
		cdb = (MPT_CDB *)((char *)cdb - MPT_CDB_SIZE);
		paddr -= MPT_CDB_SIZE;
		spaddr -= MPT_SENSE_SIZE;
		memset(cdb, 0, sizeof(MPT_CDB));
		cdb->msg.function = MPT_FUNCTION_SCSI_IO_REQUEST;
		cdb->msg.sense_buf_length = MPT_SENSE_SIZE;
		cdb->msg.msg_flags = MSG_FLAGS_SENSE_BUF_ADDR_WIDTH_64;
		cdb->msg.msg_context = __32CPU2LE(i);
		cdb->msg.sense_lo_addr = __32CPU2LE(spaddr);
		cdb->paddr = paddr + __offsetof(MPT_CDB, msg);
		mpt->useful_sges = 0;
		skips = 0;
		for (j = 0; j < MPT_MAX_SG; j++) {
			if ((j % n_sge) == (n_first_sge - 1) && j < MPT_MAX_SG - 1) {
				__u32 sge_hdr;
				skips++;
				if (skips >= mpt->iocfacts.max_chain_depth) {
					mpt->useful_sges++;
					break;
				}
				cdb->sge[j].sge_hi_addr = __32CPU2LE(mpt->cdbs_pa >> 32);
				cdb->sge[j].sge_lo_addr = __32CPU2LE(cdb->paddr + sizeof(MPT_MSG_SCSI_IO) + (j + 1) * sizeof(MPT_SGE));
				sge_hdr = SGE_HDR_SIZE_64 | SGE_HDR_TYPE_CHAIN;
				if (j + n_sge < MPT_MAX_SG - 1) {
					if (skips < mpt->iocfacts.max_chain_depth - 1)
						sge_hdr |= (n_sge * sizeof(MPT_SGE) / 4) << __BSF_CONST(SGE_HDR_CHAIN_OFFSET);
					sge_hdr |= n_sge * sizeof(MPT_SGE);
				} else {
					sge_hdr |= (MPT_MAX_SG - (j + 1)) * sizeof(MPT_SGE);
				}
				cdb->sge[j].sge_hdr = sge_hdr;
				cdb->unlock[j] = KERNEL$NULL_VSPACE_DMA64UNLOCK;
			} else {
				mpt->useful_sges++;
			}
		}
		FREE_CDB(mpt, cdb);
	}
	/*__debug_printf("useful sges: %d\n", mpt->useful_sges);*/
}

u_jiffies_lo_t MPT_SCSI_TEST_TIMEOUT(MPT *mpt)
{
	int i;
	u_jiffies_lo_t j, jj;
	j = KERNEL$GET_JIFFIES_LO();
	jj = (u_jiffies_lo_t)-1;
	for (i = 0; i < mpt->max_cdbs; i++) {
		MPT_CDB *cdb = (MPT_CDB *)((char *)mpt->cdbs + i * MPT_CDB_SIZE);
		SCSIRQ *scsirq = cdb->scsirq;
		if (scsirq && cdb->aborting) {
			u_jiffies_lo_t jjj;
			if (j - cdb->abort_time >= MPT_COMMAND_TIMEOUT) {
				goto reset;
			}
			jjj = MPT_COMMAND_TIMEOUT - (j - cdb->abort_time);
			if (jjj < jj) jj = jjj;
		}
	}
	return jj;

	reset:
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, mpt->dev_name, "TIMEOUT, RESETTING CONTROLLER. COMMANDS IN PROGRESS FOLLOW");
	for (i = 0; i < mpt->max_cdbs; i++) {
		MPT_CDB *cdb = (MPT_CDB *)((char *)mpt->cdbs + i * MPT_CDB_SIZE);
		SCSIRQ *scsirq = cdb->scsirq;
		if (scsirq)
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, mpt->dev_name, "%s ON CHANNEL %X, ID %X, LUN %"__64_format"X", SCSI$COMMAND_STRING(scsirq), CHANNEL(scsirq->ch_id), ID(scsirq->ch_id), SCSI$LUN_CONVERT_INTERNAL_TO_64(scsirq->lun));
	}
	return 0;
}

void MPT_SCSI_BUST_REQUESTS(MPT *mpt)
{
	int i;
	for (i = 0; i < mpt->max_cdbs; i++) {
		MPT_CDB *cdb = (MPT_CDB *)((char *)mpt->cdbs + i * MPT_CDB_SIZE);
		SCSIRQ *scsirq = cdb->scsirq;
		if (scsirq) {
			MPT_UNMAP_CDB(cdb);
			FREE_CDB(mpt, cdb);
			scsirq->status = -EINTR;
			DONE_RQ(mpt, scsirq);
		}
	}
}
static int MPT_SCSI_POST_DEAD(SCSIRQ *scsirq)
{
	return -ESHUTDOWN;
}

void MPT_SCSI_KILL_CONTROLLER(MPT *mpt)
{
	SCSI_ATTACH_PARAM *ap;
	LIST_FOR_EACH(ap, &mpt->attached_list, SCSI_ATTACH_PARAM, u.h.client_list)
		ap->post = MPT_SCSI_POST_DEAD;
}

