#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SLAB.H>
#include <STDLIB.H>
#include <STRING.H>
#include <VALUES.H>

#include <SPAD/SCSI.H>
#include <SPAD/USB.H>
#include "USBMSD.H"

#define MSD_MAX_HS_SECTORS	(USB_HIGH_SPEED / BIO_SECTOR_SIZE / 32)
#define MSD_MAX_FS_SECTORS	(USB_FULL_SPEED / BIO_SECTOR_SIZE / 16)
#define MSD_MAX_SECTORS(iface)	(__unlikely(USB$GET_DEVICE_SPEED(iface) < USB_HIGH_SPEED) ? MSD_MAX_FS_SECTORS : MSD_MAX_HS_SECTORS)
#define MSD_MAX_TAGS		8
#define MSD_MAX_HS_FRAGMENTS	((MSD_MAX_HS_SECTORS >> __SECTORS_PER_PAGE_CLUSTER_BITS) + 1)
#define MSD_MAX_FS_FRAGMENTS	((MSD_MAX_FS_SECTORS >> __SECTORS_PER_PAGE_CLUSTER_BITS) + 1)
#define MSD_MAX_BOUNCE_FRAGMENTS	8
#define MSD_AUTOSENSE_LENGTH	18
#define MSD_CTRL_TIMEOUT	USB$GET_CTRL_TIMEOUT(msd->iface)
#define MSD_RETRIES		1
#define MSD_PORT_RESET_WAIT	(JIFFIES_PER_SECOND * 5)

typedef struct {
	__u8 proto;
	__u8 subclass;
	__u8 luns;
	__u8 queue_depth;
	__u8 n_queued;
	__u8 need_reset;
	__u8 flags;
	XLIST_HEAD cmd_queue;
	USB_ENDPOINT *bulk_out;
	USB_ENDPOINT *bulk_in;
	__u32 csw_signature;
	struct __slhead msdrq_slab;
	USB_ENDPOINT *ctrl;
	USB_DEV_INTERFACE *iface;
	LIST_HEAD attached_list;
	unsigned max_fragments;

	USB_CTRL_REQUEST ctrlrq;
	USB_RESET_RQ resetrq;
	__u8 ctrlrq_posted;
	TIMER ctrlrq_timer;

	char dev_name[__MAX_STR_LEN];
	void *lnte, *dlrq;

	void *dummy_msdrq;
} MSD;

#define MSD_FLAGS_CSW_SIGNATURE_VALID	0x01
#define MSD_FLAGS_12_BYTE_COMMANDS	0x02
#define MSD_FLAGS_SOFT_RESET_PERFORMED	0x04

typedef struct __msdrq MSDRQ;

typedef struct {
	USB_REQUEST datarq;
	MSDRQ *msdrq;
} MSD_USBDATARQ;

struct __msdrq {
	LIST_ENTRY list;
	MSD *msd;
	SCSIRQ *srq;
	unsigned outstanding;
	unsigned char retry;
#ifdef DEFERED_DATARQ
	char defered_datarq;
#endif
	char data_valid;
	USB_REQUEST cbwrq;
	USB_REQUEST cswrq;
	MSD_CBW cbw;
	MSD_CSW csw;
	USB_REQUEST sense_cbwrq;
	USB_REQUEST sense_datarq;
	USB_REQUEST sense_cswrq;
	MSD_CBW sense_cbw;
	MSD_CSW sense_csw;
	MSD_USBDATARQ *datarq_end;
	MSD_USBDATARQ datarq[1 /* MSD_MAX_*_FRAGMENTS */ ];
};

#define EXTRA_IN_REQUESTS	3
#define EXTRA_OUT_REQUESTS	2

#define RETRY_MASK		15
#define RETRY_SENSE_FLAG	16

#define PENDING_RQ_MARK		MAXINT

static AST_STUB MSD_CBW_AST;
static AST_STUB MSD_CSW_AST;
static AST_STUB MSD_DATA_AST;
static void MSD_COMPLETION(MSDRQ *msdrq);
static void MSD_SEND_SENSE(MSDRQ *msdrq);
static AST_STUB MSD_SENSE_CBW_AST;
static AST_STUB MSD_SENSE_CSW_AST;
static AST_STUB MSD_SENSE_DATA_AST;
static void MSD_TRANSPORT_ERROR(MSDRQ *msdrq);
static void MSD_ERROR(MSDRQ *msdrq, int err, int can_stall);
static int MSD_FAILRQ(int r, MSDRQ *msdrq);
static void MSD_CANCEL_REQUESTS(MSD *msd, int reset_type);
static void MSD_DO_RECOVERY(MSD *msd, int incr_retr);

__COLD_ATTR__ static void MSDRQ_CTOR(struct __slhead *g, void *o)
{
	unsigned i;
	MSD *msd = GET_STRUCT(g, MSD, msdrq_slab);
	MSDRQ *msdrq = o;
	msdrq->msd = msd;
	msdrq->outstanding = 0;
	memset(&msdrq->cbwrq, 0, sizeof msdrq->cbwrq);
	memset(&msdrq->cswrq, 0, sizeof msdrq->cswrq);
	msdrq->cbwrq.v.vspace = &KERNEL$VIRTUAL;
	msdrq->cswrq.v.vspace = &KERNEL$VIRTUAL;
	msdrq->cbwrq.fn = MSD_CBW_AST;
	msdrq->cswrq.fn = MSD_CSW_AST;
	memset(msdrq->datarq, 0, sizeof(MSD_USBDATARQ) * msd->max_fragments);
	for (i = 0; i < msd->max_fragments; i++) {
		msdrq->datarq[i].msdrq = msdrq;
		msdrq->datarq[i].datarq.fn = MSD_DATA_AST;
	}
	msdrq->cbw.dCBWSignature = CBW_SIGNATURE;
	msdrq->cbw.dCBWTag = __32CPU2LE(random());

	memset(&msdrq->sense_cbwrq, 0, sizeof msdrq->sense_cbwrq);
	msdrq->sense_cbwrq.fn = MSD_SENSE_CBW_AST;
	msdrq->sense_cbwrq.v.vspace = &KERNEL$VIRTUAL;
	memset(&msdrq->sense_datarq, 0, sizeof msdrq->sense_cbwrq);
	msdrq->sense_datarq.fn = MSD_SENSE_DATA_AST;
	msdrq->sense_datarq.v.vspace = &KERNEL$VIRTUAL;
	memset(&msdrq->sense_cswrq, 0, sizeof msdrq->sense_cbwrq);
	msdrq->sense_cswrq.fn = MSD_SENSE_CSW_AST;
	msdrq->sense_cswrq.v.vspace = &KERNEL$VIRTUAL;

	msdrq->sense_cbw.dCBWSignature = CBW_SIGNATURE;
	msdrq->sense_cbw.dCBWTag = __32CPU2LE(random());
	msdrq->sense_cbw.bmCBWFlags = CBW_FLAGS_DATA_IN;
	msdrq->sense_cbw.bCBWLength = 6;
	if (__unlikely(msd->flags & MSD_FLAGS_12_BYTE_COMMANDS)) msdrq->sense_cbw.bCBWLength = 12;
	memset(msdrq->sense_cbw.bCBWCB, 0, MSD_CMD_LENGTH);
	msdrq->sense_cbw.bCBWCB[0] = SCMD_REQUEST_SENSE;
}

static int MSD_POST(SCSIRQ *rq)
{
	MSD *msd;
	MSDRQ *msdrq;
	MSD_USBDATARQ *dr;
	BIODESC *bd;
	USB_ENDPOINT *datapipe;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("MSD_POST AT SPL %08X", KERNEL$SPL);
#endif
	msd = rq->adapter;
	if (__unlikely(rq->internal != 0)) {
		msdrq = (MSDRQ *)rq->internal;
		DEL_FROM_LIST(&msdrq->list);
		goto have_msdrq;
	}
	if (__unlikely(rq->cmdlen > 12)) {
		if (__unlikely(msd->flags & MSD_FLAGS_12_BYTE_COMMANDS) || __unlikely(rq->cmdlen > MSD_CMD_LENGTH)) return -EOPNOTSUPP;
	}
	if (__unlikely(msd->n_queued >= msd->queue_depth) || __unlikely(msd->need_reset)) {
		return -EAGAIN;
	}
	msdrq = __slalloc(&msd->msdrq_slab);
	if (__unlikely(!msdrq)) {
#if __DEBUG >= 1
		if (__unlikely(XLIST_EMPTY(&msd->cmd_queue)))
			KERNEL$SUICIDE("MSD_POST: NO SLAB ENTRY AND EMPTY QUEUE");
#endif
		return -EAGAIN;
	}
	msdrq->retry = 0;
#ifdef DEFERED_DATARQ
	msdrq->defered_datarq = 0;
#endif
	msd->n_queued++;

	have_msdrq:
	msdrq->data_valid = 1;
	ADD_TO_XLIST(&msd->cmd_queue, &msdrq->list);
	rq->internal = (unsigned long)msdrq;
	msdrq->srq = rq;
	msdrq->cbw.dCBWDataTransferLength = __32CPU2LE(0);
	msdrq->cbw.bCBWLUN = (unsigned long)rq->lun >> 8;
	msdrq->cbw.bCBWLength = rq->cmdlen;
	if (__unlikely(msd->flags & MSD_FLAGS_12_BYTE_COMMANDS)) msdrq->cbw.bCBWLength = 12;
	memset(msdrq->cbw.bCBWCB, 0, MSD_CMD_LENGTH);
	memcpy(msdrq->cbw.bCBWCB, rq->cmd, rq->cmdlen);
	msdrq->cbwrq.flags = USBRQ_TERMINATE_SHORT | USBRQ_DELAY_INTERRUPT;
	msdrq->cbwrq.v.ptr = (unsigned long)&msdrq->cbw;
	msdrq->cbwrq.v.len = CBW_SIZE;
	msdrq->cswrq.flags = USBRQ_QUIET_STALL;
	msdrq->cswrq.v.ptr = (unsigned long)&msdrq->csw;
	msdrq->cswrq.v.len = CSW_SIZE;
	/* We used to fixup command here */
	if (__likely(rq->direction == PF_WRITE)) {
		msdrq->cbw.bmCBWFlags = CBW_FLAGS_DATA_IN;
		datapipe = msd->bulk_in;
	} else {
#if __DEBUG >= 1
		if (__unlikely(!rq->direction)) {
			if (__unlikely(rq->desc != NULL))
				KERNEL$SUICIDE("MSD_POST: REQUEST HAS NO TRANSFER DIRECTION AND TRANSFER DESCRIPTORS");
		} else {
			if (__unlikely(rq->direction != PF_READ))
				KERNEL$SUICIDE("MSD_POST: INVALID DIRECTION %u", rq->direction);
		}
#endif
#if 0
		if (msd->queue_depth == 1) {
	/* Since fix on UHCI, this is not needed.
	   But I left the code for defered_datarq there, may sometimes
	   I found a buggy USB key that needs it.
	*/
			msdrq->defered_datarq = 1;
			msdrq->cbwrq.flags &= ~USBRQ_DELAY_INTERRUPT;
		}
#endif
		msdrq->cbw.bmCBWFlags = CBW_FLAGS_DATA_OUT;
		datapipe = msd->bulk_out;
	}
	dr = msdrq->datarq_end = msdrq->datarq;
	for (bd = rq->desc; bd; bd = bd->next) {
		int r;
#if __DEBUG >= 1
		if (__unlikely(dr == msdrq->datarq + msd->max_fragments))
			KERNEL$SUICIDE("MSD_POST: TOO MANY FRAGMENTS");
#endif
		dr->datarq.flags = USBRQ_DELAY_INTERRUPT | USBRQ_QUIET_STALL;
		/*
		 * This is hacky. Allow short requests only when one descriptor
		 * is used (they are needed for scsi passthrough), but don't
		 * allow them when more descriptors are submitted --- it would
		 * be complicated to tear them down and they are not needed for
		 * access to disk anyway.
		 *
		 * This should be cleaned up if redesign ever comes up.
		 */
		if (!rq->desc->next) dr->datarq.flags |= USBRQ_ALLOW_SHORT;
		dr->datarq.v.ptr = bd->v.ptr;
		dr->datarq.v.len = bd->v.len;
		msdrq->cbw.dCBWDataTransferLength = __32CPU2LE(__32LE2CPU(msdrq->cbw.dCBWDataTransferLength) + bd->v.len);
		dr->datarq.v.vspace = bd->v.vspace;
		RAISE_SPL(SPL_USB);
		if (__unlikely(r = datapipe->prepare(datapipe, &dr->datarq))) {
			LOWER_SPL(SPL_ATA_SCSI);
			return MSD_FAILRQ(r, msdrq);
		}
		LOWER_SPL(SPL_ATA_SCSI);
		dr++;
		msdrq->datarq_end = dr;
	}
#if __DEBUG >= 1
	if (__unlikely(__32LE2CPU(msdrq->cbw.dCBWDataTransferLength) > MSD_MAX_SECTORS(msd->iface) << BIO_SECTOR_SIZE_BITS))
		KERNEL$SUICIDE("MSD_POST: TOO LONG REQUEST: %08X", __32LE2CPU(msdrq->cbw.dCBWDataTransferLength));
#endif
	msdrq->srq->status = PENDING_RQ_MARK;
	msdrq->outstanding = 2;
	RAISE_SPL(SPL_USB);
	msd->bulk_out->prepare(msd->bulk_out, &msdrq->cbwrq);
	msd->bulk_out->post(&msdrq->cbwrq);
	for (dr = msdrq->datarq; dr != msdrq->datarq_end; dr++) {
		msdrq->outstanding++;
#ifdef DEFERED_DATARQ
		if (__likely(!msdrq->defered_datarq))
#endif
			datapipe->post(&dr->datarq);
		TEST_SPL(SPL_ATA_SCSI, SPL_USB);
	}
	msd->bulk_in->prepare(msd->bulk_in, &msdrq->cswrq);
	msd->bulk_in->post(&msdrq->cswrq);
	LOWER_SPL(SPL_ATA_SCSI);
	if (__unlikely(msdrq->retry & RETRY_SENSE_FLAG)) {
		if (__likely(msd->queue_depth != 1)) {
			MSD_SEND_SENSE(msdrq);
		} else {
			msdrq->retry &= ~RETRY_SENSE_FLAG;
		}
	}
	return 0;
}

static DECL_AST(MSD_CBW_AST, SPL_ATA_SCSI, USB_REQUEST)
{
	MSDRQ *msdrq = GET_STRUCT(RQ, MSDRQ, cbwrq);
	/*__debug_printf("cbw: %d\n", msdrq->cbwrq.status);*/
	if (__unlikely(msdrq->cbwrq.status < 0)) {
		MSD_ERROR(msdrq, msdrq->cbwrq.status, 0);
		RETURN;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_CBW_AST: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
#ifdef DEFERED_DATARQ
	if (__unlikely(msdrq->defered_datarq)) {
		MSD_USBDATARQ *dr;
		USB_ENDPOINT *datapipe = msdrq->cbw.bmCBWFlags & CBW_FLAGS_DATA_IN ? msdrq->msd->bulk_in : msdrq->msd->bulk_out;
		RAISE_SPL(SPL_USB);
		for (dr = msdrq->datarq; dr != msdrq->datarq_end; dr++) {
			datapipe->post(&dr->datarq);
			TEST_SPL(SPL_ATA_SCSI, SPL_USB);
		}
		LOWER_SPL(SPL_ATA_SCSI);
		msdrq->defered_datarq = 0;
	}
#endif
	if (__unlikely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
	RETURN;
}

static DECL_AST(MSD_DATA_AST, SPL_ATA_SCSI, USB_REQUEST)
{
	MSD_USBDATARQ *dr = GET_STRUCT(RQ, MSD_USBDATARQ, datarq);
	MSDRQ *msdrq = dr->msdrq;
	/*__debug_printf("data: %d\n", dr->datarq.status);*/
	if (__unlikely(dr->datarq.status < 0)) {
		msdrq->data_valid = 0;
		MSD_ERROR(msdrq, dr->datarq.status, 1);
		RETURN;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_DATA_AST: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
	if (__unlikely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
	RETURN;
}

static DECL_AST(MSD_CSW_AST, SPL_ATA_SCSI, USB_REQUEST)
{
	MSDRQ *msdrq = GET_STRUCT(RQ, MSDRQ, cswrq);
	/*__debug_printf("csw: %d\n", msdrq->cswrq.status);*/
	if (__unlikely(msdrq->cswrq.status < 0)) {
		MSD_ERROR(msdrq, msdrq->cswrq.status, 1);
		RETURN;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_CSW_AST: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
	if (__likely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
	RETURN;
}

static void MSD_ERROR_SIG(MSD *msd, __u32 sig, __u32 wanted);
static void MSD_ERROR_TAG(MSD *msd, __u32 tag, __u32 wanted);
static void MSD_ERROR_STATUS(MSD *msd, __u8 st);
static void MSD_ERROR_RESIDUE(MSD *msd, int len, __u32 resid);
static void MSD_ERROR_SENSE_SIG(MSD *msd, __u32 sig, __u32 wanted);
static void MSD_ERROR_SENSE_TAG(MSD *msd, __u32 tag, __u32 wanted);
static void MSD_ERROR_SENSE_STATUS(MSD *msd, __u8 st);
static void MSD_ERROR_SENSE_RESIDUE(MSD *msd, int len, __u32 resid);
static void MSD_ERROR_SENSE_FAILED(MSD *msd);

static void MSD_COMPLETION(MSDRQ *msdrq)
{
	SCSIRQ *rq = msdrq->srq;
	MSD *msd;
	if (__likely(rq->status == PENDING_RQ_MARK)) {
		/*__debug_printf("csw: %08x %08x %08x %02x\n", msdrq->csw.dCSWSignature, msdrq->csw.dCSWTag, msdrq->csw.dCSWDataResidue, msdrq->csw.bCSWStatus);
		{
			int i;
			__debug_printf("cbw:");
			for (i = 0; i < 31; i++) __debug_printf(" %02x", ((unsigned char *)&msdrq->cbw)[i]);
			__debug_printf("\n");
		}*/
		if (__unlikely(!(msdrq->msd->flags & MSD_FLAGS_CSW_SIGNATURE_VALID))) {
			msdrq->msd->flags |= MSD_FLAGS_CSW_SIGNATURE_VALID;
			msdrq->msd->csw_signature = msdrq->csw.dCSWSignature;
		}
		if (__unlikely(msdrq->csw.dCSWSignature != msdrq->msd->csw_signature /*CSW_SIGNATURE*/)) {
			MSD_ERROR_SIG(msdrq->msd, __32LE2CPU(msdrq->csw.dCSWSignature), __32LE2CPU(msdrq->msd->csw_signature));
			err:
			rq->status = -EPROTO;
			err2:
			MSD_TRANSPORT_ERROR(msdrq);
			return;
		}
		if (__unlikely(msdrq->csw.dCSWTag != msdrq->cbw.dCBWTag)) {
			MSD_ERROR_TAG(msdrq->msd, __32LE2CPU(msdrq->csw.dCSWTag), __32LE2CPU(msdrq->cbw.dCBWTag));
			goto err;
		}
		if (__unlikely(msdrq->csw.bCSWStatus >= CSW_STATUS_PHASE_ERROR)) {
			MSD_ERROR_STATUS(msdrq->msd, msdrq->csw.bCSWStatus);
			goto err;
		}
		if (__unlikely(__32LE2CPU(msdrq->cbw.dCBWDataTransferLength) < __32LE2CPU(msdrq->csw.dCSWDataResidue))) {
			MSD_ERROR_RESIDUE(msdrq->msd, __32LE2CPU(msdrq->cbw.dCBWDataTransferLength), __32LE2CPU(msdrq->csw.dCSWDataResidue));
			goto err;
		}
		rq->status = __32LE2CPU(msdrq->cbw.dCBWDataTransferLength) - __32LE2CPU(msdrq->csw.dCSWDataResidue);
		if (__unlikely(!msdrq->data_valid)) rq->status = 0; /* prevent bad devices from enabling the user to read unallocated memory */
		rq->scsi_status = SCSI_GOOD;
		if (__likely(!(msdrq->retry & RETRY_SENSE_FLAG))) {
			if (__unlikely(msdrq->csw.bCSWStatus != CSW_STATUS_GOOD)) {
				msdrq->retry |= RETRY_SENSE_FLAG;
				if (msdrq->msd->n_queued != 1) {
					if (!msdrq->msd->need_reset) {
						int r = MSD_POST(rq);
						if (__unlikely(r)) {
							rq->status = r;
							rq->done(rq);
						}
					}
				} else {
					rq->status = PENDING_RQ_MARK;
					MSD_SEND_SENSE(msdrq);
				}
				return;
			}
		} else {
			if (__likely(msdrq->csw.bCSWStatus != CSW_STATUS_GOOD)) rq->scsi_status = SCSI_CHECK_CONDITION;
			if (__unlikely(!(msdrq->msd->flags & MSD_FLAGS_CSW_SIGNATURE_VALID))) {
				msdrq->msd->flags |= MSD_FLAGS_CSW_SIGNATURE_VALID;
				msdrq->msd->csw_signature = msdrq->sense_csw.dCSWSignature;
			}
			if (__unlikely(msdrq->sense_csw.dCSWSignature != msdrq->msd->csw_signature /*CSW_SIGNATURE*/)) {
				MSD_ERROR_SENSE_SIG(msdrq->msd, __32LE2CPU(msdrq->sense_csw.dCSWSignature), __32LE2CPU(msdrq->msd->csw_signature));
				goto err;
			}
			if (__unlikely(msdrq->sense_csw.dCSWTag != msdrq->sense_cbw.dCBWTag)) {
				MSD_ERROR_SENSE_TAG(msdrq->msd, __32LE2CPU(msdrq->sense_csw.dCSWTag), __32LE2CPU(msdrq->sense_cbw.dCBWTag));
				goto err;
			}
			if (__unlikely(msdrq->sense_csw.bCSWStatus != CSW_STATUS_GOOD)) {
				if (__unlikely(msdrq->sense_csw.bCSWStatus >= CSW_STATUS_PHASE_ERROR)) {
					MSD_ERROR_SENSE_STATUS(msdrq->msd, msdrq->sense_csw.bCSWStatus);
					goto err;
				}
				MSD_ERROR_SENSE_FAILED(msdrq->msd);
				rq->status = -EIO;
				goto err2;
			}
			if (__unlikely(__32LE2CPU(msdrq->sense_cbw.dCBWDataTransferLength) < __32LE2CPU(msdrq->sense_csw.dCSWDataResidue))) {
				MSD_ERROR_SENSE_RESIDUE(msdrq->msd, MSD_AUTOSENSE_LENGTH, __32LE2CPU(msdrq->sense_csw.dCSWDataResidue));
				goto err;
			}
			memset(rq->sense + __32LE2CPU(msdrq->sense_cbw.dCBWDataTransferLength) - __32LE2CPU(msdrq->sense_csw.dCSWDataResidue), 0, rq->sense_size - (__32LE2CPU(msdrq->sense_cbw.dCBWDataTransferLength) - __32LE2CPU(msdrq->sense_csw.dCSWDataResidue)));
		}
	} else {
		if (rq->status != -EINTR) {
			/*__debug_printf("do recovery (%d)\n", rq->status);*/
			do_recov:
			MSD_DO_RECOVERY(msdrq->msd, 1);
			return;
		}
	}
	DEL_FROM_LIST(&msdrq->list);
	msd = msdrq->msd;
	msd->n_queued--;
	__slfree(msdrq);
	SCSI$HOST_DEQUEUE(&msd->attached_list);
	rq->done(rq);
	if (__unlikely(msd->need_reset)) goto do_recov;
	msd->flags &= ~MSD_FLAGS_SOFT_RESET_PERFORMED;
}

__COLD_ATTR__ static void MSD_ERROR_SIG(MSD *msd, __u32 sig, __u32 wanted)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID SIGNATURE IN CSW: %08X (WANTED %08X)", (unsigned)sig, (unsigned)wanted);
}

__COLD_ATTR__ static void MSD_ERROR_TAG(MSD *msd, __u32 tag, __u32 wanted)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID TAG IN CSW: %08X (WANTED %08X)", (unsigned)tag, (unsigned)wanted);
}

__COLD_ATTR__ static void MSD_ERROR_STATUS(MSD *msd, __u8 st)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1) {
		if (st == CSW_STATUS_PHASE_ERROR) KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "PHASE ERROR");
		else KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID STATUS IN CSW: %02X", (unsigned)st);
	}
}

__COLD_ATTR__ static void MSD_ERROR_RESIDUE(MSD *msd, int len, __u32 resid)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID RESIDUE IN CSW: %08X > %08X", (unsigned)resid, len);
	}
}

__COLD_ATTR__ static void MSD_ERROR_SENSE_SIG(MSD *msd, __u32 sig, __u32 wanted)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID SIGNATURE IN SENSE CSW: %08X (WANTED %08X)", (unsigned)sig, (unsigned)wanted);
}

__COLD_ATTR__ static void MSD_ERROR_SENSE_TAG(MSD *msd, __u32 tag, __u32 wanted)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID TAG IN SENSE CSW: %08X (WANTED %08X)", (unsigned)tag, (unsigned)wanted);
}

__COLD_ATTR__ static void MSD_ERROR_SENSE_STATUS(MSD *msd, __u8 st)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1) {
		if (st == CSW_STATUS_PHASE_ERROR) KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "PHASE ERROR IN SENSE COMMAND");
		else KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID STATUS IN CSW: %02X", (unsigned)st);
	}
}

__COLD_ATTR__ static void MSD_ERROR_SENSE_RESIDUE(MSD *msd, int len, __u32 resid)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INVALID RESIDUE IN SENSE CSW: %08X > %08X", (unsigned)resid, len);
	}
}

__COLD_ATTR__ static void MSD_ERROR_SENSE_FAILED(MSD *msd)
{
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "REQUEST SENSE COMMAND FAILED");
	}
}


__COLD_ATTR__ static void MSD_SEND_SENSE(MSDRQ *msdrq)
{
	MSD *msd;
	/*__debug_printf("send sense ...\n");*/
	msdrq->outstanding += 3;
	msdrq->sense_cbw.bCBWLUN = msdrq->cbw.bCBWLUN;
	msdrq->sense_cbw.bCBWCB[1] = msdrq->cbw.bCBWCB[1] & 0xe0;
	msdrq->sense_cbwrq.flags = USBRQ_TERMINATE_SHORT | USBRQ_DELAY_INTERRUPT;
	msdrq->sense_cbwrq.v.ptr = (unsigned long)&msdrq->sense_cbw;
	msdrq->sense_cbwrq.v.len = CBW_SIZE;
	msdrq->sense_datarq.flags = USBRQ_DELAY_INTERRUPT;
	msdrq->sense_datarq.v.ptr = (unsigned long)msdrq->srq->sense;
	msdrq->sense_datarq.v.len = msdrq->srq->sense_size;
	if (__unlikely(msdrq->srq->sense_size > MSD_AUTOSENSE_LENGTH)) {
		msdrq->sense_datarq.v.len = MSD_AUTOSENSE_LENGTH;
		memset((__u8 *)msdrq->srq->sense + MSD_AUTOSENSE_LENGTH, 0, msdrq->srq->sense_size - MSD_AUTOSENSE_LENGTH);
	}
	msdrq->sense_cbw.bCBWCB[4] = msdrq->sense_datarq.v.len;
	msdrq->sense_cbw.dCBWDataTransferLength = __32CPU2LE(msdrq->sense_datarq.v.len);
	msdrq->sense_cswrq.flags = 0;
	msdrq->sense_cswrq.v.ptr = (unsigned long)&msdrq->sense_csw;
	msdrq->sense_cswrq.v.len = CSW_SIZE;
	RAISE_SPL(SPL_USB);
	msd = msdrq->msd;
	msd->bulk_out->prepare(msd->bulk_out, &msdrq->sense_cbwrq);
	msd->bulk_out->post(&msdrq->sense_cbwrq);
	msd->bulk_in->prepare(msd->bulk_in, &msdrq->sense_datarq);
	msd->bulk_in->post(&msdrq->sense_datarq);
	msd->bulk_in->prepare(msd->bulk_in, &msdrq->sense_cswrq);
	msd->bulk_in->post(&msdrq->sense_cswrq);
	LOWER_SPL(SPL_ATA_SCSI);
}

__COLD_ATTR__ static DECL_AST(MSD_SENSE_CBW_AST, SPL_ATA_SCSI, USB_REQUEST)
{
	MSDRQ *msdrq = GET_STRUCT(RQ, MSDRQ, sense_cbwrq);
	/*__debug_printf("sense cbw: %d\n", msdrq->sense_cbwrq.status);*/
	if (__unlikely(msdrq->sense_cbwrq.status < 0)) {
		MSD_ERROR(msdrq, msdrq->sense_cbwrq.status, 0);
		RETURN;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_SENSE_CBW_AST: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
	if (__unlikely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
	RETURN;
}

__COLD_ATTR__ static DECL_AST(MSD_SENSE_DATA_AST, SPL_ATA_SCSI, USB_REQUEST)
{
	MSDRQ *msdrq = GET_STRUCT(RQ, MSDRQ, sense_datarq);
	/*
	__debug_printf("sense data: %d\n", msdrq->sense_datarq.status);
	{
		int i;
		__debug_printf("sense on %02x:", msdrq->srq->cmd[0]);
		for (i = 0; i < msdrq->srq->sense_size; i++) {
			__debug_printf(" %02x", msdrq->srq->sense[i]);
		}
		__debug_printf("\n");
	}
	*/
	if (__unlikely(msdrq->sense_datarq.status < 0)) {
		MSD_ERROR(msdrq, msdrq->sense_datarq.status, 0);
		RETURN;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_SENSE_DATA_AST: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
	if (__unlikely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
	RETURN;
}

__COLD_ATTR__ static DECL_AST(MSD_SENSE_CSW_AST, SPL_ATA_SCSI, USB_REQUEST)
{
	MSDRQ *msdrq = GET_STRUCT(RQ, MSDRQ, sense_cswrq);
	/*__debug_printf("sense csw: %d\n", msdrq->sense_cswrq.status);*/
	if (__unlikely(msdrq->sense_cswrq.status < 0)) {
		MSD_ERROR(msdrq, msdrq->sense_cswrq.status, 0);
		RETURN;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_SENSE_CSW_AST: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
	if (__unlikely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
	RETURN;
}

__COLD_ATTR__ static void MSD_TRANSPORT_ERROR(MSDRQ *msdrq)
{
	MSD *msd = msdrq->msd;
	msd->need_reset = 2;
	if (__unlikely((++msdrq->retry & RETRY_MASK) > MSD_RETRIES)) {
		SCSIRQ *rq;
		msd->n_queued--;
		DEL_FROM_LIST(&msdrq->list);
		rq = msdrq->srq;
		__slow_slfree(msdrq);
		rq->done(rq);
	}
	MSD_DO_RECOVERY(msd, 0);
}

__COLD_ATTR__ static void MSD_ERROR(MSDRQ *msdrq, int err, int can_stall)
{
	/*if (err == -ENODATA && can_stall) err = -EPIPE;*/
	if (err == -EPIPE && !can_stall) err = -EIO;
	if (msdrq->srq->status == PENDING_RQ_MARK || msdrq->srq->status == -EPIPE) {
		if (err == -EINTR) msdrq->srq->status = -EPIPE;
		else msdrq->srq->status = err;
	}
	msdrq->outstanding--;
#if __DEBUG >= 1
	if (__unlikely(msdrq->outstanding >= msdrq->msd->max_fragments + EXTRA_IN_REQUESTS + EXTRA_OUT_REQUESTS))
		KERNEL$SUICIDE("MSD_ERROR: OUTSTANDING FIELD CORRUPTED: %d", msdrq->outstanding);
#endif
	MSD_CANCEL_REQUESTS(msdrq->msd, msdrq->srq->status != -EPIPE ? 2 : 1);
	if (__likely(!msdrq->outstanding)) MSD_COMPLETION(msdrq);
}

__COLD_ATTR__ static int MSD_FAILRQ(int r, MSDRQ *msdrq)
{
	MSD_USBDATARQ *dr;
	MSD *msd = msdrq->msd;
	USB_ENDPOINT *datapipe = msdrq->cbw.bmCBWFlags & CBW_FLAGS_DATA_IN ? msd->bulk_in : msd->bulk_out;
	RAISE_SPL(SPL_USB);
	for (dr = msdrq->datarq; dr != msdrq->datarq_end; dr++) {
		datapipe->free(&dr->datarq);
		TEST_SPL(SPL_ATA_SCSI, SPL_USB);
	}
	LOWER_SPL(SPL_ATA_SCSI);
	DEL_FROM_LIST(&msdrq->list);
	msd->n_queued--;
	__slow_slfree(msdrq);
	return r;
}

__COLD_ATTR__ static void MSD_CANCEL_REQUESTS(MSD *msd, int reset_type)
{
	MSDRQ *last_rq = NULL;
	MSDRQ *msdrq;
	XLIST_FOR_EACH(msdrq, &msd->cmd_queue, MSDRQ, list) {
		MSD_USBDATARQ *dr;
		USB_ENDPOINT *datapipe = msdrq->cbw.bmCBWFlags & CBW_FLAGS_DATA_IN ? msd->bulk_in : msd->bulk_out;
		RAISE_SPL(SPL_USB);
		/*__debug_printf("usb cancel\n");*/
		msd->bulk_out->cancel(&msdrq->cbwrq);
		for (dr = msdrq->datarq; dr != msdrq->datarq_end; dr++) {
#ifdef DEFERED_DATARQ
			if (!msdrq->defered_datarq) {
#endif
				datapipe->cancel(&dr->datarq);
#ifdef DEFERED_DATARQ
			} else {
				datapipe->free(&dr->datarq);
				dr->datarq.status = -EINTR;
				CALL_AST(&dr->datarq);
			}
#endif
			TEST_SPL(SPL_ATA_SCSI, SPL_USB);
		}
		msd->bulk_in->cancel(&msdrq->cswrq);
		msd->bulk_out->cancel(&msdrq->sense_cbwrq);
		msd->bulk_in->cancel(&msdrq->sense_datarq);
		msd->bulk_in->cancel(&msdrq->sense_cswrq);
#ifdef DEFERED_DATARQ
		msdrq->defered_datarq = 0;
#endif
		LOWER_SPL(SPL_ATA_SCSI);
		last_rq = msdrq;
	}
	if (!msd->need_reset) {
		if (__likely(last_rq != NULL)) {
			last_rq->retry++;
		}
	}
	if (msd->need_reset < reset_type) msd->need_reset = reset_type;
}

__COLD_ATTR__ static void MSD_CANCEL(SCSIRQ *rq)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("MSD_CANCEL AT SPL %08X", KERNEL$SPL);
#endif
	if (__likely(rq->status == PENDING_RQ_MARK)) {
		MSD *msd;
		/*__debug_printf("cancel active\n");*/
		rq->status = -EINTR;
		msd = ((MSDRQ *)rq->internal)->msd;
		msd->need_reset = 2;
		MSD_CANCEL_REQUESTS(msd, 0);
	}
}

static AST_STUB MSD_RESET_RECEIVED;
static AST_STUB MSD_CLEAR_IN_RECEIVED;
static AST_STUB MSD_CLEAR_OUT_RECEIVED;
static AST_STUB MSD_DONE_PORT_RESET;
static void MSD_RECOVERED(MSD *msd);
static void MSD_CTRL_RESET_TIMEOUT(TIMER *t);
static void MSD_CTRL_CLEAR_IN_TIMEOUT(TIMER *t);
static void MSD_CTRL_CLEAR_OUT_TIMEOUT(TIMER *t);

__COLD_ATTR__ static void MSD_DO_RECOVERY(MSD *msd, int incr_retr)
{
	MSDRQ *last_rq = NULL;
	MSDRQ *msdrq;
	if (__unlikely(!msd->need_reset)) MSD_CANCEL_REQUESTS(msd, 2);
	XLIST_FOR_EACH(msdrq, &msd->cmd_queue, MSDRQ, list) {
		if (msdrq->outstanding) return;
		last_rq = msdrq;
	}
	if (incr_retr && __likely(last_rq != NULL) && __unlikely((last_rq->retry & RETRY_MASK) > MSD_RETRIES)) {
		SCSIRQ *rq;
		msd->n_queued--;
		DEL_FROM_LIST(&last_rq->list);
		rq = last_rq->srq;
		__slow_slfree(last_rq);
		rq->done(rq);
	}
	memset(&msd->ctrlrq, 0, sizeof(USB_REQUEST));
	msd->ctrlrq.fn = MSD_RESET_RECEIVED;
	msd->ctrlrq.v.vspace = &KERNEL$VIRTUAL;
	msd->ctrlrq.setup.request = __16CPU2LE(MSD_CMD_RESET);
	msd->ctrlrq.setup.w_value = __16CPU2LE(0);
	msd->ctrlrq.setup.w_index = __16CPU2LE(USB$GET_INTERFACE_NUMBER(msd->iface));
	msd->ctrlrq.setup.w_length = __16CPU2LE(0);
	msd->ctrlrq_posted = 1;
	msd->ctrlrq_timer.fn = MSD_CTRL_RESET_TIMEOUT;
	KERNEL$SET_TIMER(MSD_CTRL_TIMEOUT, &msd->ctrlrq_timer);
	if (__unlikely(msd->need_reset == 1) || !USB$IS_ATTACHED(msd->iface)) {
		CALL_AST(&msd->ctrlrq);
	} else if (msd->flags & MSD_FLAGS_SOFT_RESET_PERFORMED) {
/* Some devices respond to RESET command but remain in hung state.
   If the second RESET command should be performed and no intermediate
   successful comands completed, reset the whole USB port. */
		msd->flags &= ~MSD_FLAGS_SOFT_RESET_PERFORMED;
		msd->ctrlrq.status = -EIO;
		CALL_AST(&msd->ctrlrq);
	} else {
		msd->flags |= MSD_FLAGS_SOFT_RESET_PERFORMED;
		if (USB$GET_ERRORLEVEL(msd->iface) >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, msd->dev_name, "RESETTING MASS-STORAGE DEVICE");
		RAISE_SPL(SPL_USB);
		msd->ctrl->prepare(msd->ctrl, (USB_REQUEST *)(void *)&msd->ctrlrq);
		msd->ctrl->post((USB_REQUEST *)(void *)&msd->ctrlrq);
		LOWER_SPL(SPL_ATA_SCSI);
	}
}

__COLD_ATTR__ static DECL_AST(MSD_RESET_RECEIVED, SPL_ATA_SCSI, USB_CTRL_REQUEST)
{
	MSD *msd = GET_STRUCT(RQ, MSD, ctrlrq);
	KERNEL$DEL_TIMER(&msd->ctrlrq_timer);
	if (__unlikely(msd->ctrlrq.status < 0) && USB$IS_ATTACHED(msd->iface)) {
		msd->flags &= ~MSD_FLAGS_SOFT_RESET_PERFORMED;
		msd->resetrq.fn = MSD_DONE_PORT_RESET;
		msd->resetrq.iface = msd->iface;
		msd->resetrq.wait_time = MSD_PORT_RESET_WAIT;
		msd->ctrlrq_posted = 2;
		RETURN_IORQ(&msd->resetrq, USB$RESET_PORT);
	}
	msd->ctrlrq_timer.fn = MSD_CTRL_CLEAR_IN_TIMEOUT;
	KERNEL$SET_TIMER(MSD_CTRL_TIMEOUT, &msd->ctrlrq_timer);
	USB$FILL_CLEAR_STALL_REQUEST(msd->iface, msd->bulk_in, &msd->ctrlrq);
	msd->ctrlrq.fn = MSD_CLEAR_IN_RECEIVED;
	RAISE_SPL(SPL_USB);
	msd->ctrl->prepare(msd->ctrl, (USB_REQUEST *)(void *)&msd->ctrlrq);
	msd->ctrl->post((USB_REQUEST *)(void *)&msd->ctrlrq);
	LOWER_SPL(SPL_ATA_SCSI);
	RETURN;
}

__COLD_ATTR__ static DECL_AST(MSD_CLEAR_IN_RECEIVED, SPL_ATA_SCSI, USB_CTRL_REQUEST)
{
	MSD *msd = GET_STRUCT(RQ, MSD, ctrlrq);
	RAISE_SPL(SPL_USB);
	msd->bulk_in->clear_stall(msd->bulk_in);
	LOWER_SPL(SPL_ATA_SCSI);
	KERNEL$DEL_TIMER(&msd->ctrlrq_timer);
	msd->ctrlrq_timer.fn = MSD_CTRL_CLEAR_OUT_TIMEOUT;
	KERNEL$SET_TIMER(MSD_CTRL_TIMEOUT, &msd->ctrlrq_timer);
	USB$FILL_CLEAR_STALL_REQUEST(msd->iface, msd->bulk_out, &msd->ctrlrq);
	msd->ctrlrq.fn = MSD_CLEAR_OUT_RECEIVED;
	RAISE_SPL(SPL_USB);
	msd->ctrl->prepare(msd->ctrl, (USB_REQUEST *)(void *)&msd->ctrlrq);
	msd->ctrl->post((USB_REQUEST *)(void *)&msd->ctrlrq);
	LOWER_SPL(SPL_ATA_SCSI);
	RETURN;
}

__COLD_ATTR__ static DECL_AST(MSD_CLEAR_OUT_RECEIVED, SPL_ATA_SCSI, USB_CTRL_REQUEST)
{
	MSD *msd = GET_STRUCT(RQ, MSD, ctrlrq);
	RAISE_SPL(SPL_USB);
	msd->bulk_out->clear_stall(msd->bulk_out);
	LOWER_SPL(SPL_ATA_SCSI);
	KERNEL$DEL_TIMER(&msd->ctrlrq_timer);
	MSD_RECOVERED(msd);
	RETURN;
}

__COLD_ATTR__ static DECL_AST(MSD_DONE_PORT_RESET, SPL_ATA_SCSI, USB_RESET_RQ)
{
	MSD *msd = GET_STRUCT(RQ, MSD, resetrq);
	MSD_RECOVERED(msd);
	RETURN;
}

__COLD_ATTR__ static void MSD_RECOVERED(MSD *msd)
{
	MSDRQ *last_rq;
	DECL_XLIST(xl);
	int reset_type = msd->need_reset;
	msd->need_reset = 0;
	msd->ctrlrq_posted = 0;
	last_rq = NULL;
	while (!XLIST_EMPTY(&msd->cmd_queue)) {
		MSDRQ *msdrq = LIST_STRUCT(msd->cmd_queue.next, MSDRQ, list);
		DEL_FROM_LIST(&msdrq->list);
		ADD_TO_XLIST(&xl, &msdrq->list);
		last_rq = msdrq;
	}
	if (last_rq && reset_type == 1) {
		DEL_FROM_LIST(&last_rq->list);
		ADD_TO_XLIST(&msd->cmd_queue, &last_rq->list);
		last_rq->srq->status = PENDING_RQ_MARK;
		last_rq->outstanding = 1;
		last_rq->cswrq.flags = USBRQ_QUIET_STALL;
		last_rq->cswrq.v.ptr = (unsigned long)&last_rq->csw;
		last_rq->cswrq.v.len = CSW_SIZE;
		RAISE_SPL(SPL_USB);
		msd->bulk_in->prepare(msd->bulk_in, &last_rq->cswrq);
		msd->bulk_in->post(&last_rq->cswrq);
		LOWER_SPL(SPL_ATA_SCSI);
	}
	while (!XLIST_EMPTY(&xl)) {
		MSDRQ *msdrq = LIST_STRUCT(xl.next, MSDRQ, list);
		SCSIRQ *rq = msdrq->srq;
		int r = MSD_POST(rq);
		if (__unlikely(r)) {
			rq->status = r;
			rq->done(rq);
		}
	}
	SCSI$HOST_DEQUEUE(&msd->attached_list);
}

__COLD_ATTR__ static void MSD_CTRL_TIMEOUT_COMMON(MSD *msd)
{
	SET_TIMER_NEVER(&msd->ctrlrq_timer);
	RAISE_SPL(SPL_USB);
	msd->ctrl->cancel((USB_REQUEST *)(void *)&msd->ctrlrq);
	LOWER_SPL(SPL_ATA_SCSI);
}

__COLD_ATTR__ static void MSD_CTRL_RESET_TIMEOUT(TIMER *t)
{
	MSD *msd = GET_STRUCT(t, MSD, ctrlrq_timer);
	LOWER_SPL(SPL_ATA_SCSI);
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1 && USB$IS_ATTACHED(msd->iface))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "TIMEOUT ON MASS-STORAGE RESET COMMAND");
	MSD_CTRL_TIMEOUT_COMMON(msd);
}

__COLD_ATTR__ static void MSD_CTRL_CLEAR_IN_TIMEOUT(TIMER *t)
{
	MSD *msd = GET_STRUCT(t, MSD, ctrlrq_timer);
	LOWER_SPL(SPL_ATA_SCSI);
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1 && USB$IS_ATTACHED(msd->iface))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "TIMEOUT ON CLEAR STALL BULK IN COMMAND");
	MSD_CTRL_TIMEOUT_COMMON(msd);
}

__COLD_ATTR__ static void MSD_CTRL_CLEAR_OUT_TIMEOUT(TIMER *t)
{
	MSD *msd = GET_STRUCT(t, MSD, ctrlrq_timer);
	LOWER_SPL(SPL_ATA_SCSI);
	if (USB$GET_ERRORLEVEL(msd->iface) >= 1)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "TIMEOUT ON CLEAR STALL BULK OUT COMMAND");
	MSD_CTRL_TIMEOUT_COMMON(msd);
}


__COLD_ATTR__ static int MSD_TEST(USB_DEV_INTERFACE *iface, long proto)
{
	USB_DESCRIPTOR_INTERFACE *desc = USB$GET_INTERFACE_DESCRIPTOR(iface);
	/*{
		USB_DESCRIPTOR_ENDPOINT *ep;
		int i, j;
		__debug_printf("interface: ");
		for (i = 0; i < sizeof(USB_DESCRIPTOR_INTERFACE); i++) {
			__debug_printf("%02x ", ((__u8 *)desc)[i]);
		}
		__debug_printf("\n");
		for (j = 0; (ep = USB$GET_ENDPOINT_DESCRIPTOR(iface, -1, j)); j++) {
			__debug_printf("endpoint %d: ", j);
			for (i = 0; i < sizeof(USB_DESCRIPTOR_ENDPOINT); i++) {
				__debug_printf("%02x ", ((__u8 *)ep)[i]);
			}
			__debug_printf("\n");
		}
	}*/
	if (desc->bInterfaceClass != USB_CLASS_MASS_STORAGE) return USB_SKIP;
	if (__unlikely(desc->bInterfaceProtocol != proto)) return USB_SKIP;
	return USB_ATTACH_INTERFACE;
}

static int MSD_DCALL(void *ptr, const char *dcall_type, int cmd, va_list params);
static int MSD_UNLOAD(void *ptr, void **unload, const char * const argv[]);

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	int r;
	USB_DEV_INTERFACE *iface;
	MSD *msd;
	__u8 proto;
	__u8 max_lun;
	MALLOC_REQUEST mrq;
	USB_ARGS *args = NULL;
	const char *opt, *optend, *str;
	const char * const *arg = argv;
	int state = 0;
	static const struct __param_table params[1] = {
		NULL, 0, 0, 0,
	};
	void *vars[1];
	vars[0] = NULL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_USB), SPL_X(SPL_ATA_SCSI))))
		KERNEL$SUICIDE("USBMSD: SPL_USB %08X, SPL_ATA_SCSI %08X", SPL_X(SPL_USB), SPL_X(SPL_ATA_SCSI));
	if (__unlikely(__parse_params(&arg, &state, params, vars, &opt, &optend, &str))) {
		if (__unlikely(USB$PARSE_PARAMS(opt, optend, str, &args))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBMSD: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (__likely(!__IS_ERR(iface = USB$ATTACH_DRIVER(args, MSD_TEST, proto = MSD_PROTOCOL_BULK)))) goto got_it;
	r = __PTR_ERR(iface);
	/* test remaining protocols ... but I have no device that uses them
	if (__unlikely(!__IS_ERR(iface = USB$ATTACH_DRIVER(args, MSD_TEST, proto = MSD_PROTOCOL_CBI)))) goto got_it;
	if (__likely(r == -ENODEV) && __unlikely(__PTR_ERR(iface) != -ENODEV)) r = __PTR_ERR(iface);
	if (__unlikely(!__IS_ERR(iface = USB$ATTACH_DRIVER(args, MSD_TEST, proto = MSD_PROTOCOL_CB)))) goto got_it;
	if (__likely(r == -ENODEV) && __unlikely(__PTR_ERR(iface) != -ENODEV)) r = __PTR_ERR(iface);
	*/
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBMSD: %s", strerror(-r));
	goto ret0;

	got_it:

	mrq.size = sizeof(MSD);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBMSD: %s", strerror(-mrq.status));
		r = mrq.status;
		goto ret1;
	}
	msd = mrq.ptr;
	memset(msd, 0, sizeof(MSD));
	msd->queue_depth = 1;
	INIT_XLIST(&msd->cmd_queue);
	INIT_TIMER(&msd->ctrlrq_timer);
	INIT_LIST(&msd->attached_list);
	msd->max_fragments = USB$GET_DEVICE_SPEED(iface) <= USB_FULL_SPEED ? MSD_MAX_FS_FRAGMENTS : MSD_MAX_HS_FRAGMENTS;
	if (__unlikely(KERNEL$RESERVE_BOUNCE_ACTIVE(0)) && msd->max_fragments > MSD_MAX_BOUNCE_FRAGMENTS) msd->max_fragments = MSD_MAX_BOUNCE_FRAGMENTS;
	while (__unlikely(KERNEL$SLAB_ENTRIES_PER_PAGE(sizeof(MSDRQ) + msd->max_fragments * sizeof(MSD_USBDATARQ), 0) <= 0)) msd->max_fragments--;
	KERNEL$SLAB_INIT(&msd->msdrq_slab, sizeof(MSDRQ) + msd->max_fragments * sizeof(MSD_USBDATARQ), 0, VM_TYPE_WIRED_MAPPED, MSDRQ_CTOR, NULL, "USBMSD$MSDRQ");

	r = USB$GET_DEVICE_NAME(msd->dev_name, __MAX_STR_LEN, iface, args, "SCSI$USBMSD");
	if (__unlikely(r)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "USBMSD: CAN'T GET DEVICE NAME: %s", strerror(-r));
		goto ret2;
	}
	msd->proto = proto;
	msd->subclass = USB$GET_INTERFACE_DESCRIPTOR(iface)->bInterfaceSubClass;
	/* not according to spec, but Windows do the same ... so this will
	   probably be most compatible */
	if (__unlikely(msd->subclass != MSD_SUBCLASS_SCSI)) msd->flags |= MSD_FLAGS_12_BYTE_COMMANDS;
	msd->iface = iface;

	if (__unlikely(r = KERNEL$SLAB_RESERVE(&msd->msdrq_slab, 2))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", msd->dev_name, strerror(-r));
		goto ret2;
	}

	msd->dummy_msdrq = __slalloc(&msd->msdrq_slab);
	if (__unlikely(!msd->dummy_msdrq))
		KERNEL$SUICIDE("USBMSD: CAN'T ALLOCATE DUMMY MSDRQ");

	msd->ctrl = USB$GET_DEFAULT_ENDPOINT(iface);
	if (__unlikely(__IS_ERR(msd->bulk_out = USB$FIND_ENDPOINT(iface, USB_EP_BULK_OUT, msd->queue_depth * (msd->max_fragments + EXTRA_OUT_REQUESTS), MSD_MAX_SECTORS(iface) << BIO_SECTOR_SIZE_BITS)))) {
		r = __PTR_ERR(msd->bulk_out);
		msd->bulk_out = NULL;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET BULK OUT ENDPOINT: %s", msd->dev_name, strerror(-r));
		goto ret3;
	}
	if (__unlikely(__IS_ERR(msd->bulk_in = USB$FIND_ENDPOINT(iface, USB_EP_BULK_IN, msd->queue_depth * (msd->max_fragments + EXTRA_IN_REQUESTS), MSD_MAX_SECTORS(iface) << BIO_SECTOR_SIZE_BITS)))) {
		r = __PTR_ERR(msd->bulk_in);
		msd->bulk_in = NULL;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T GET BULK IN ENDPOINT: %s", msd->dev_name, strerror(-r));
		goto ret3;
	}
	RAISE_SPL(SPL_USB);
	r = USB$SYNC_CTRL(msd->ctrl, MSD_CMD_RESET, 0, USB$GET_INTERFACE_NUMBER(iface), 0, NULL, 0, USB$GET_CTRL_TIMEOUT(iface), USB$GET_CTRL_RETRIES(iface), iface, NULL);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FAILED TO RESET DEVICE: %s", msd->dev_name, strerror(-r));
		goto ret3;
	}
	RAISE_SPL(SPL_USB);
	r = USB$SYNC_CTRL(msd->ctrl, MSD_CMD_GET_MAX_LUN, 0, USB$GET_INTERFACE_NUMBER(iface), 1, &max_lun, USBRQ_QUIET_STALL, USB$GET_CTRL_TIMEOUT(iface), USB$GET_CTRL_RETRIES(iface), NULL, NULL);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r < 0)) {
		if (__unlikely(r == -EPIPE)) {
			max_lun = 0;
		} else {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FAILED TO GET MAX LUN: %s", msd->dev_name, strerror(-r));
			goto ret3;
		}
	}
	if (__unlikely(max_lun >= MSD_MAX_LUNS)) {
		if (USB$GET_ERRORLEVEL(iface) >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, msd->dev_name, "INAPPROPRIATE NUMBER OF LUNS: %d", max_lun + 1);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INAPPROPRIATE NUMBER OF LUNS: %d", msd->dev_name, max_lun + 1);
		goto ret3;
	}
	msd->luns = max_lun + 1;

	r = KERNEL$REGISTER_DEVICE(msd->dev_name, "USBMSD.SYS", 0, msd, NULL, MSD_DCALL, "SCSI", NULL, MSD_UNLOAD, &msd->lnte, USB$GET_CONTROLLER_NAME(iface), NULL);
	if (__unlikely(r < 0)) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", msd->dev_name, strerror(-r));
		goto ret3;
	}
	msd->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), msd->dev_name);
	USB$FREE_PARAMS(args);
	return 0;

	ret3:
	__slow_slfree(msd->dummy_msdrq);
	ret2:
	KERNEL$SLAB_DESTROY(&msd->msdrq_slab);
	free(msd);
	ret1:
	USB$DETACH_DRIVER(iface);
	ret0:
	USB$FREE_PARAMS(args);
	return r;
}

__COLD_ATTR__ static int MSD_DCALL(void *ptr, const char *dcall_type, int cmd, va_list params)
{
	MSD *msd = ptr;
	SCSI_ATTACH_PARAM *ap = va_arg(params, SCSI_ATTACH_PARAM *);
	if (KERNEL$SPL != SPL_X(SPL_DEV))
		KERNEL$SUICIDE("MSD_DCALL AT SPL %08X", KERNEL$SPL);
	switch (cmd) {
	case SCSI_CMD_GET: {
		const char *arg;
		int state;
		static const struct __param_table params[2] = {
			NULL, 0, 0, 0,
		};
		void *vars[1];
		vars[0] = NULL;
		if (__unlikely(ap->u.p.version != SCSI_VERSION)) {
			ap->msg = "INCORRECT VERSION";
			return -EINVAL;
		}
		arg = ap->u.p.param;
		state = 0;
		if (__unlikely(__parse_extd_param(&arg, &state, params, vars, NULL, NULL, NULL, NULL))) {
			ap->msg = "SYNTAX ERROR";
			return -EBADSYN;
		}
		if (ap->adapter) {
			return -ENODEV;
		}
		ap->max_sectors = MSD_MAX_SECTORS(msd->iface);
		ap->max_untagged_requests = msd->queue_depth;
		ap->max_tagged_requests = 0;
		ap->max_fragments = msd->max_fragments;
		ap->scsi_flags = SCSI_FLAG_CAN_USE_MODE_10 | /*(msd->subclass == MSD_SUBCLASS_SCSI ? SCSI_FLAG_CAN_USE_MODE_6 : 0) |*/ SCSI_FLAG_SHORT_INQUIRY;
		ap->lun = LUN_NONE;
		ap->ch_id = 0;
		ap->adapter = msd;
		SCSI$HOST_PRINT_NAME(ap, msd->dev_name, 0, 1, 0, 1);
		ap->post = MSD_POST;
		ap->cancel = MSD_CANCEL;
		RAISE_SPL(SPL_ATA_SCSI);
		ADD_TO_LIST(&msd->attached_list, &ap->u.h.client_list);
		LOWER_SPL(SPL_DEV);
		return 0;
	}
	case SCSI_CMD_FREE: {
		if (__unlikely(ap->adapter != msd))
			KERNEL$SUICIDE("MSD_DCALL: INVALID ADAPTER %p != %p", ap->adapter, msd);
		RAISE_SPL(SPL_ATA_SCSI);
		DEL_FROM_LIST(&ap->u.h.client_list);
		LOWER_SPL(SPL_DEV);
		return 0;
	}
	case SCSI_CMD_SET_LUN: {
		int r = -EINVAL;
		if (__unlikely((unsigned)SCSI$LUN_CONVERT_INTERNAL_TO_8(ap->lun) >= msd->luns) || __unlikely(r = SCSI$HOST_IS_LUN_DUPLICATE(&msd->attached_list, ap))) {
			ap->lun = LUN_NONE;
			return r;
		}
		SCSI$HOST_PRINT_NAME(ap, msd->dev_name, 0, 1, 0, 1);
		return 0;
	}
	case SCSI_CMD_N_LUNS: {
		return msd->luns;
	}
	default: {
		return -EINVAL;
	}
	}
}

__COLD_ATTR__ static int MSD_UNLOAD(void *ptr, void **unload, const char * const argv[])
{
	int r;
	MSD *msd = ptr;
	RAISE_SPL(SPL_DEV);
	if (!LIST_EMPTY(&msd->attached_list)) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS CONTROLLER", msd->dev_name);
		return -EBUSY;
	}
	if (__unlikely(msd->n_queued)) {
		if (XLIST_EMPTY(&msd->cmd_queue)) KERNEL$SUICIDE("MSD_UNLOAD: N_QUEUED LEAKED (%d)", (int)msd->n_queued);
		else KERNEL$SUICIDE("MSD_UNLOAD: %d REQUESTS LEAKED", (int)msd->n_queued);
	}
	if (__unlikely(!XLIST_EMPTY(&msd->cmd_queue))) KERNEL$SUICIDE("MSD_UNLOAD: REQUEST LIST DAMAGED");
	r = KERNEL$DEVICE_UNLOAD(msd->lnte, argv);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r)) return r;
	RAISE_SPL(SPL_ATA_SCSI);
	while (__unlikely(msd->ctrlrq_posted)) {
		if (msd->ctrlrq_posted == 1) {
			RAISE_SPL(SPL_USB);
			msd->ctrl->cancel((USB_REQUEST *)(void *)&msd->ctrlrq);
			LOWER_SPL(SPL_ATA_SCSI);
		} else {
			KERNEL$CIO((IORQ *)(void *)&msd->resetrq);
		}
		KERNEL$SLEEP(1);
	}
	LOWER_SPL(SPL_ZERO);
	USB$DETACH_DRIVER(msd->iface);
	*unload = msd->dlrq;
	__slow_slfree(msd->dummy_msdrq);
	KERNEL$SLAB_DESTROY(&msd->msdrq_slab);
	free(msd);
	return 0;
}
