#include <SPAD/TIMER.H>
#include <STDLIB.H>
#include <SPAD/SYSLOG.H>
#include <STRING.H>
#include <SYS/PARAM.H>

#include <SPAD/SCSI.H>

#define SCSI_INQUIRY_TIMEOUT			(5 * JIFFIES_PER_SECOND)
#define SCSI_INQUIRY_EXTENDED_TIMEOUT		(120 * JIFFIES_PER_SECOND)
#define SCSI_INQUIRY_RETRIES			1

static int SCSI_INQUIRY_RECOMMENDED_ACTION(SCSIRQ *srq);

int SCSI$INQUIRY(const char *dev_name, int errorlevel, SCSI_ATTACH_PARAM *ap, scsi_lun_t lun, __u8 lun_byte, void *data, unsigned datalen, jiffies_lo_t timeout, jiffies_lo_t extended_timeout)
{
	DECL_SCSIRQ(6) inq;
	SCSI_SENSE_DATA inq_sense;
	memset(data, 0, datalen);
	if (ap->scsi_flags & SCSI_FLAG_SHORT_INQUIRY && __likely(datalen > SCSI_SHORT_INQUIRY_SIZE)) datalen = SCSI_SHORT_INQUIRY_SIZE;
	inq.sense = (__u8 *)&inq_sense;
	inq.sense_size = SCSI_SIZEOF_SENSE_DATA;
	inq.direction = PF_WRITE;
	inq.cmdlen = 6;
	inq.tagging = SCSI_TAGGING_NONE;
	inq.cmd[0] = SCMD_INQUIRY;
	inq.cmd[1] = lun_byte;
	inq.cmd[2] = 0;
	inq.cmd[3] = datalen >> 8;
	inq.cmd[4] = datalen;
	inq.cmd[5] = 0;
	if (!timeout) timeout = SCSI_INQUIRY_TIMEOUT;
	if (!extended_timeout) extended_timeout = MAX(timeout, SCSI_INQUIRY_EXTENDED_TIMEOUT);
	SCSI$SYNC_RQ(dev_name, errorlevel, (SCSIRQ *)(void *)&inq, ap, lun, data, datalen, timeout, extended_timeout, SCSI_INQUIRY_RETRIES, SCSI_INQUIRY_RECOMMENDED_ACTION);
	if (__unlikely(inq.status < SCSI_MIN_INQUIRY_SIZE)) {
		if (__likely(inq.status < 0)) return inq.status;
		if (inq.scsi_status != SCSI_GOOD) {
			if (inq.scsi_status == SCSI_CHECK_CONDITION) {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INQUIRY: %s", SCSI$SENSE_STRING(0, &inq_sense, SCSI_SIZEOF_SENSE_DATA));
			} else {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INQUIRY: %s", SCSI$STATUS_STRING(inq.scsi_status));
			}
			return SCSI$SENSE_ERROR_CODE(0, &inq_sense, SCSI_SIZEOF_SENSE_DATA);
		} else {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INQUIRY RETURNED ONLY %d BYTES", inq.status);
			return -EPROTO;
		}
	}
	return 0;
}

static int SCSI_INQUIRY_RECOMMENDED_ACTION(SCSIRQ *srq)
{
	if (srq->status == -ENXIO)
		return SCSI_ACTION_FAIL;
	return SCSI$CMD_RECOMMENDED_ACTION(srq);
}

static void GET_MODEL_STRING(__u8 *src, char *str, unsigned len);

void SCSI$GET_MODEL(SCSI_INQUIRY_DATA *inq, char (*vendor)[9], char (*product)[17], char (*revision)[5])
{
	if (vendor) GET_MODEL_STRING(inq->vendor, *vendor, 8);
	if (product) GET_MODEL_STRING(inq->product, *product, 16);
	if (revision) GET_MODEL_STRING(inq->revision, *revision, 4);
}

static void GET_MODEL_STRING(__u8 *src, char *str, unsigned len)
{
	int i, ii;
	strlcpy(str, (char *)src, len + 1);
	for (i = strlen(str) - 1; i >= 0; i--) if (str[i] == ' ') str[i] = 0;
					       else break;
	for (i = 0, ii = 0; str[i]; i++) if (str[i] != ' ' || (ii && str[ii - 1] != ' ')) str[ii++] = str[i];
	str[ii] = 0;
}

void SCSI$MODE_HEADER_10_TO_6(SCSI_MODE_SENSE_6_HEADER *s6h, SCSI_MODE_SENSE_HEADER *s10h)
{
	SCSI_MODE_SENSE_6_HEADER s6h_;
	unsigned dl;

	dl = __16BE2CPU(s10h->data_length);
	if (__likely(dl >= 3))
		dl -= 3;
	if (__unlikely(dl > 255))
		dl = 255;
	s6h_.data_length = dl;

	s6h_.medium_type = s10h->medium_type;
	s6h_.device_specific = s10h->device_specific;

	dl = __16BE2CPU(s10h->block_descriptor_length);
	if (__unlikely(dl > 255))
		dl = 255;
	s6h_.block_descriptor_length = dl;

	memcpy(s6h, &s6h_, sizeof(SCSI_MODE_SENSE_6_HEADER));
}

void SCSI$MODE_HEADER_6_TO_10(SCSI_MODE_SENSE_HEADER *s10h, SCSI_MODE_SENSE_6_HEADER *s6h)
{
	SCSI_MODE_SENSE_6_HEADER s6h_;
	unsigned dl;

	memcpy(&s6h_, s6h, sizeof(SCSI_MODE_SENSE_6_HEADER));

	dl = s6h_.data_length;
	if (dl) dl += 3;
	s10h->data_length = __16CPU2BE(dl);

	s10h->medium_type = s6h_.medium_type;
	s10h->device_specific = s6h_.device_specific;

	dl = s6h_.block_descriptor_length;
	s10h->block_descriptor_length = __16CPU2BE(dl);
}

int SCSI$MODE_SENSE(const char *dev_name, int errorlevel, const SCSI_INQUIRY_DATA *inq, __u8 scsi_version, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, int modepage, SCSI_MODE_SENSE_HEADER *data, unsigned len, jiffies_lo_t timeout, jiffies_lo_t extended_timeout, int retries)
{
	__u32 key;
	DECL_SCSIRQ(10) cmd;
	SCSI_SENSE_DATA sense;
	if (__unlikely(len < sizeof(SCSI_MODE_SENSE_HEADER))) return -EINVAL;
	if (__unlikely(scsi_version == 1) && modepage != 0) return -EOPNOTSUPP;
	memset(data, 0, len);
	cmd.sense = (__u8 *)&sense;
	cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
	if (__likely(ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_10)) {
		cmd.direction = PF_WRITE;
		cmd.cmdlen = 10;
		cmd.tagging = SCSI_TAGGING_NONE;
		cmd.cmd[0] = SCMD_MODE_SENSE_10;
		cmd.cmd[1] = lun_byte | 0x08 | (scsi_version >= 4 ? 0x10 : 0);
		cmd.cmd[2] = modepage;
		cmd.cmd[3] = 0;
		cmd.cmd[4] = 0;
		cmd.cmd[5] = 0;
		cmd.cmd[6] = 0;
		cmd.cmd[7] = len >> 8;
		cmd.cmd[8] = len;
		cmd.cmd[9] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, (SCSIRQ *)(void *)&cmd, ap, LUN_NONE, data, len, timeout, extended_timeout, retries, SCSI$CMD_RECOMMENDED_ACTION);
		if (__likely(SCSI$CMD_RECOMMENDED_ACTION((SCSIRQ *)(void *)&cmd) == SCSI_ACTION_OK)) ok: {
			unsigned data_len, should_transfer;
			data_len = cmd.status >= sizeof(SCSI_MODE_SENSE_HEADER) ? __16BE2CPU(data->data_length) + 2 : sizeof(SCSI_MODE_SENSE_HEADER);
			should_transfer = data_len < len ? data_len : len;
			if (__unlikely(cmd.status < should_transfer)) {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "%s/PAGE %02X RETURNED NOT ENOUGH DATA (TRANSFERRED %d, DESCRIPTOR %d, ALLOCATED %d)", SCSI$COMMAND_NAME((SCSIRQ *)(void *)&cmd), modepage, cmd.status, data_len, len);
				if (cmd.status < sizeof(SCSI_MODE_SENSE_HEADER)) {
					return -EINVAL;
				}
				data->data_length = __16CPU2BE(cmd.status) + 2;
			}
			return 0;
		}
		key = SCSI$GET_CMD_SENSE((SCSIRQ *)(void *)&cmd);
		if (__likely(!SCSI_SENSE_UNKNOWN(key)) && __likely(SCSI_SENSE_KEY(key) == SCSI_KEY_ILLEGAL_REQUEST) && __likely(SCSI_SENSE_ASC(key) == 0x20) && SCSI_SENSE_ASCQ(key) == 0x00) {
			if (ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_6) {
				/*ap->scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;*/
				goto try_mode_6;
			}
		}
	} else try_mode_6: if (__likely(ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) {
		if (__unlikely(len - 4 >= 256)) {
			return -ERANGE;
		}
		cmd.direction = PF_WRITE;
		cmd.cmdlen = 6;
		cmd.tagging = SCSI_TAGGING_NONE;
		cmd.cmd[0] = SCMD_MODE_SENSE_6;
		cmd.cmd[1] = lun_byte | 0x08;
		cmd.cmd[2] = modepage;
		cmd.cmd[3] = 0;
		cmd.cmd[4] = len - 4;
		cmd.cmd[5] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, (SCSIRQ *)(void *)&cmd, ap, LUN_NONE, (__u8 *)data + 4, len - 4, timeout, extended_timeout, retries, SCSI$CMD_RECOMMENDED_ACTION);
		if (__likely(SCSI$CMD_RECOMMENDED_ACTION((SCSIRQ *)(void *)&cmd) == SCSI_ACTION_OK)) {
			SCSI$MODE_HEADER_6_TO_10(data, (SCSI_MODE_SENSE_6_HEADER *)((__u8 *)data + 4));
			cmd.status += 4;
			goto ok;
		}
		key = SCSI$GET_CMD_SENSE((SCSIRQ *)(void *)&cmd);
	} else {
		return -EOPNOTSUPP;
	}
	if (SCSI_SENSE_UNKNOWN(key) ? scsi_version >= 2 || cmd.status < 0 || cmd.scsi_status != SCSI_CHECK_CONDITION : SCSI_SENSE_KEY(key) != SCSI_KEY_ILLEGAL_REQUEST)
		SCSI$LOG_ERROR(dev_name, errorlevel, inq, scsi_version, (SCSIRQ *)(void *)&cmd, "");
	return SCSI$CMD_ERROR_CODE(scsi_version, (SCSIRQ *)(void *)&cmd);
}

int SCSI$MODE_SENSE_CAUTIOUS(const char *dev_name, int errorlevel, const SCSI_INQUIRY_DATA *inq, __u8 scsi_version, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, int modepage, SCSI_MODE_SENSE_HEADER *data, unsigned len, jiffies_lo_t timeout, jiffies_lo_t extended_timeout, int retries)
{
	int r;
	unsigned real_len;
	if (__unlikely(len < sizeof(SCSI_MODE_SENSE_HEADER))) return -EINVAL;
	r = SCSI$MODE_SENSE(dev_name, errorlevel, inq, scsi_version, ap, lun_byte, modepage, data, sizeof(SCSI_MODE_SENSE_HEADER), timeout, extended_timeout, retries);
	if (__unlikely(r)) return r;
	real_len = __16BE2CPU(data->data_length) + 2;
	if (__unlikely(real_len > len)) real_len = len;
	memset((char *)data + real_len, 0, len - real_len);
	return SCSI$MODE_SENSE(dev_name, errorlevel, inq, scsi_version, ap, lun_byte, modepage, data, real_len, timeout, extended_timeout, retries);
}

int SCSI$MODE_SELECT(const char *dev_name, int errorlevel, const SCSI_INQUIRY_DATA *inq, __u8 scsi_version, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, int modepage, SCSI_MODE_SENSE_HEADER *data, unsigned len, jiffies_lo_t timeout, jiffies_lo_t extended_timeout, int retries)
{
	__u32 key;
	DECL_SCSIRQ(10) cmd;
	SCSI_SENSE_DATA sense;
	if (__unlikely(len < sizeof(SCSI_MODE_SENSE_HEADER))) return -EINVAL;
	if (__unlikely(scsi_version == 1) && modepage != 0) return -EOPNOTSUPP;
	cmd.sense = (__u8 *)&sense;
	cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
	if (__likely(ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_10)) {
		cmd.direction = PF_READ;
		cmd.cmdlen = 10;
		cmd.tagging = SCSI_TAGGING_NONE;
		cmd.cmd[0] = SCMD_MODE_SELECT_10;
		cmd.cmd[1] = lun_byte | (scsi_version >= 2 ? 0x10 : 0);
		cmd.cmd[2] = 0;
		cmd.cmd[3] = 0;
		cmd.cmd[4] = 0;
		cmd.cmd[5] = 0;
		cmd.cmd[6] = 0;
		cmd.cmd[7] = len >> 8;
		cmd.cmd[8] = len;
		cmd.cmd[9] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, (SCSIRQ *)(void *)&cmd, ap, LUN_NONE, data, len, timeout, extended_timeout, retries, SCSI$CMD_RECOMMENDED_ACTION);
		if (__likely(SCSI$CMD_RECOMMENDED_ACTION((SCSIRQ *)(void *)&cmd) == SCSI_ACTION_OK)) ok: {
			if (__unlikely(cmd.status < len)) {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "%s/PAGE %02X RETURNED NOT ENOUGH DATA (TRANSFERRED %d, REQUESTED %d)", SCSI$COMMAND_NAME((SCSIRQ *)(void *)&cmd), modepage, cmd.status, len);
				return -EIO;
			}
			return 0;
		}
		key = SCSI$GET_CMD_SENSE((SCSIRQ *)(void *)&cmd);
		if (__likely(!SCSI_SENSE_UNKNOWN(key)) && __likely(SCSI_SENSE_KEY(key) == SCSI_KEY_ILLEGAL_REQUEST) && __likely(SCSI_SENSE_ASC(key) == 0x20) && SCSI_SENSE_ASCQ(key) == 0x00) {
			if (ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_6) {
				/*ap->scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;*/
				goto try_mode_6;
			}
		}
	} else try_mode_6: if (__likely(ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) {
		if (__unlikely(len - 4 >= 256)) {
			return -ERANGE;
		}
		SCSI$MODE_HEADER_10_TO_6((SCSI_MODE_SENSE_6_HEADER *)((__u8 *)data + 4), data);
		cmd.direction = PF_READ;
		cmd.cmdlen = 6;
		cmd.tagging = SCSI_TAGGING_NONE;
		cmd.cmd[0] = SCMD_MODE_SELECT_6;
		cmd.cmd[1] = lun_byte | (scsi_version >= 2 ? 0x10 : 0);
		cmd.cmd[2] = 0;
		cmd.cmd[3] = 0;
		cmd.cmd[4] = len - 4;
		cmd.cmd[5] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, (SCSIRQ *)(void *)&cmd, ap, LUN_NONE, (__u8 *)data + 4, len - 4, timeout, extended_timeout, retries, SCSI$CMD_RECOMMENDED_ACTION);
		SCSI$MODE_HEADER_6_TO_10(data, (SCSI_MODE_SENSE_6_HEADER *)((__u8 *)data + 4));
		if (__likely(SCSI$CMD_RECOMMENDED_ACTION((SCSIRQ *)(void *)&cmd) == SCSI_ACTION_OK)) {
			cmd.status += 4;
			goto ok;
		}
		key = SCSI$GET_CMD_SENSE((SCSIRQ *)(void *)&cmd);
	} else {
		return -EOPNOTSUPP;
	}
	if (SCSI_SENSE_UNKNOWN(key) ? scsi_version >= 2 || cmd.status < 0 || cmd.scsi_status != SCSI_CHECK_CONDITION : SCSI_SENSE_KEY(key) != SCSI_KEY_ILLEGAL_REQUEST)
		SCSI$LOG_ERROR(dev_name, errorlevel, inq, scsi_version, (SCSIRQ *)(void *)&cmd, "");
	return SCSI$CMD_ERROR_CODE(scsi_version, (SCSIRQ *)(void *)&cmd);
}

int SCSI$MODE_CHANGE(const char *dev_name, int errorlevel, const SCSI_INQUIRY_DATA *inq, __u8 scsi_version, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, int modepage, SCSI_MODE_SENSE_HEADER *data, unsigned len, jiffies_lo_t timeout, jiffies_lo_t extended_timeout, int retries)
{
	unsigned orig_len = __16BE2CPU(data->data_length) + 2;
	if (__unlikely(orig_len < sizeof(SCSI_MODE_SENSE_HEADER))) return -EINVAL;
	if (orig_len > len) orig_len = len;
	data->data_length = __16CPU2BE(0);
	data->block_descriptor_length = __16CPU2BE(0);
	if (len > sizeof(SCSI_MODE_SENSE_HEADER)) {
		*(__u8 *)(data + 1) &= ~0xc0;
	}
	SCSI$MODE_SELECT(dev_name, errorlevel, inq, scsi_version, ap, lun_byte, modepage, data, orig_len, timeout, extended_timeout, retries);
	return SCSI$MODE_SENSE(dev_name, errorlevel, inq, scsi_version, ap, lun_byte, modepage, data, orig_len, timeout, extended_timeout, retries);
}

