#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <STDLIB.H>
#include <SYS/PARAM.H>

#include <SPAD/SCSI.H>

#define LUN_LIST_SIZE	__PAGE_CLUSTER_SIZE

#define SCSI_REPORT_LUNS_TIMEOUT		(30 * JIFFIES_PER_SECOND)
#define SCSI_REPORT_LUNS_EXTENDED_TIMEOUT	(120 * JIFFIES_PER_SECOND)
#define SCSI_REPORT_LUNS_RETRIES		1

static int lun_cmp(const void *p1, const void *p2);

int SCSI$FIND_LUN(const char *ctrl, const char *dev_name, int errorlevel, SCSI_ATTACH_PARAM *ap, scsi_lun_t user_lun, jiffies_lo_t timeout, jiffies_lo_t extended_timeout, __u8 *lun_byte)
{
	int r;
	union {
		SCSI_INQUIRY_DATA inq;
		MALLOC_REQUEST mrq;
		DECL_SCSIRQ(12) cmd;
	} u;
	SCSI_SENSE_DATA sense;
	__u8 scsi_version;
	__u64 *lun_list;
	unsigned n_luns, i;
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl)))
		KERNEL$SUICIDE("SCSI$SYNC_RQ AT SPL %08X", spl);
#endif
	RAISE_SPL(SPL_DEV);

	u.mrq.size = LUN_LIST_SIZE;
	SYNC_IO(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		r = u.mrq.status;
		goto lower_ret;
	}
	lun_list = u.mrq.ptr;

	if (__unlikely(r = SCSI$INQUIRY(dev_name, errorlevel, ap, 0, 0, &u.inq, sizeof u.inq, timeout, extended_timeout)))
		goto free_ret;

	if (!timeout) timeout = SCSI_REPORT_LUNS_TIMEOUT;
	if (!extended_timeout) extended_timeout = MAX(timeout, SCSI_REPORT_LUNS_EXTENDED_TIMEOUT);

	scsi_version = SCSI$VERSION(&u.inq);
	
	if (__unlikely(user_lun != LUN_NONE)) {
		lun_list[1] = __64CPU2LE(user_lun);
		n_luns = 1;
		goto got_lun;
	}

	if ((r = KERNEL$DCALL(ctrl, "SCSI", SCSI_CMD_N_LUNS, ap)) > 0) {
		if (r > (LUN_LIST_SIZE - 8) >> 3)
			r = (LUN_LIST_SIZE - 8) >> 3;
		goto fixed_luns;
	}

	if (scsi_version < 3) {
		r = 8;
		fixed_luns:
		for (i = 0; i < r; i++)
			lun_list[i + 1] = __64CPU2LE(i << 8);
		n_luns = r;
	} else {
		u.cmd.sense = (__u8 *)&sense;
		u.cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
		u.cmd.direction = PF_WRITE;
		u.cmd.cmdlen = 12;
		u.cmd.tagging = SCSI_TAGGING_NONE;
		u.cmd.cmd[0] = SCMD_REPORT_LUNS;
		u.cmd.cmd[1] = 0;
		*(__u32 *)&u.cmd.cmd[2] = 0;
		*(__u32 *)&u.cmd.cmd[6] = __32CPU2BE(LUN_LIST_SIZE);
		*(__u16 *)&u.cmd.cmd[10] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, (SCSIRQ *)(void *)&u.cmd, ap, 0, lun_list, LUN_LIST_SIZE, timeout, extended_timeout, SCSI_REPORT_LUNS_RETRIES, SCSI$CMD_RECOMMENDED_ACTION);
		if (__likely(SCSI$CMD_RECOMMENDED_ACTION((SCSIRQ *)(void *)&u.cmd) == SCSI_ACTION_OK) && u.cmd.status >= 16) {
			n_luns = __32BE2CPU(*(__u32 *)lun_list);
			if (__unlikely(n_luns > (u.cmd.status - 8) >> 3))
				n_luns = (u.cmd.status - 8) >> 3;
		} else {
			lun_list[1] = __64CPU2LE(0);
			n_luns = 1;
		}
	}
	got_lun:
	qsort(lun_list + 1, n_luns, 8, lun_cmp);
	for (i = 0; i < n_luns; i++) {
		scsi_lun_t lun = __64LE2CPU(lun_list[i + 1]);
		__u64 ap_lun = __64CPU2LE(ap->lun);
		if (ap->lun == LUN_NONE || lun_cmp(&lun_list[i + 1], &ap_lun) > 0) {
			ap->lun = lun;
			if (!KERNEL$DCALL(ctrl, "SCSI", SCSI_CMD_SET_LUN, ap)) {
				r = 0;
				*lun_byte = 0;
				if (scsi_version < 3) *lun_byte = lun >> 3;
				goto free_ret;
			}
		}
	}
	r = 1;
	free_ret:
	free(lun_list);
	lower_ret:
	LOWER_SPLX(spl);
	return r;
}

static int lun_cmp(const void *p1, const void *p2)
{
	__u64 l1 = __64BE2CPU(*(__u64 *)p1);
	__u64 l2 = __64BE2CPU(*(__u64 *)p2);
	if (l1 < l2) return -1;
	return l1 > l2;
}

int SCSI$LUN_CONVERT_INTERNAL_TO_8(scsi_lun_t lun)
{
	if (__unlikely((lun & ~(scsi_lun_t)0xff00) != 0))
		return -1;
	return ((unsigned)lun >> 8) & 0xff;
}

__u64 SCSI$LUN_CONVERT_INTERNAL_TO_64(scsi_lun_t lun)
{
	return SCSI$LUN_CONVERT_64_TO_INTERNAL(lun);
}

scsi_lun_t SCSI$LUN_CONVERT_64_TO_INTERNAL(__u64 user)
{
	return
		((user >> 8) & (0x00ff00ff | (__u64)0x00ff00ff << 32)) |
		((user << 8) & (0xff00ff00 | (__u64)0xff00ff00 << 32));
}

