#include <SPAD/LIBC.H>
#include <SPAD/SYSLOG.H>
#include <STRING.H>
#include <ARCH/BSF.H>
#include <SPAD/AC.H>
#include <SPAD/DL.H>

#include <SPAD/SCSI.H>

static const struct sense_desc {
	__u8 asc;
	__u8 ascq_from;
	__u8 ascq_to;
	const char *msg;
} senses[] = {

#include "SCSISENS.I"

};

#define N_SENSES	(sizeof(senses) / sizeof(*senses))

const char *SCSI$SENSE_ASC_STRING(__u8 asc, __u8 ascq)
{
	static char string[__MAX_STR_LEN];
	int from = 0;
	int to = N_SENSES - 1;
	while (from <= to) {
		int middle = (from + to) >> 1;
		const struct sense_desc *s = &senses[middle];
		if (s->asc < asc) {
			p2:
			from = middle + 1;
			continue;
		}
		if (s->asc > asc) {
			p1:
			to = middle - 1;
			continue;
		}
		if (s->ascq_to < ascq) {
			goto p2;
		}
		if (s->ascq_from > ascq) {
			goto p1;
		}
		if (__unlikely(s->ascq_from == s->ascq_to)) return s->msg;
		_snprintf(string, sizeof string, "%s%02X", s->msg, (unsigned)ascq);
		return string;
	}
	_snprintf(string, sizeof string, "ADDITIONAL CODE %02X, QUALIFIER %02X", (unsigned)asc, (unsigned)ascq);
	return string;
}

static const char *const sense_keys[16] = {
	"NO SENSE",
	"RECOVERED ERROR",
	"NOT READY",
	"MEDIUM ERROR",
	"HARDWARE ERROR",
	"ILLEGAL REQUEST",
	"UNIT ATTENTION",
	"DATA PROTECT",
	"BLANK CHECK",
	"VENDOR SPECIFIC",
	"COPY ABORTED",
	"ABORTED COMMAND",
	"EQUAL",
	"VOLUME OVERFLOW",
	"MISCOMPARE",
	"RESERVED SENSE KEY 15",
};

__u32 SCSI$GET_SENSE(const SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	if (__unlikely(sense_size <= __offsetof(SCSI_SENSE_DATA, flags))) return -1;
	if (__likely((sense->response_code & (SENSE_RESPONSE_CODE & ~SENSE_RESPONSE_CODE_DEFERRED_BIT)) == SENSE_RESPONSE_CODE_CURRENT)) {
		__u32 ret = (sense->response_code & SENSE_RESPONSE_CODE_DEFERRED_BIT) << 4 | ((sense->flags & SENSE_FLAGS_SENSE_KEY) >> __BSF_CONST(SENSE_FLAGS_SENSE_KEY));
		if (__likely(sense_size > __offsetof(SCSI_SENSE_DATA, ascq)) && __likely(sense->additional_length > __offsetof(SCSI_SENSE_DATA, ascq) - __offsetof(SCSI_SENSE_DATA, additional_length) + 1)) return ret | (__16LE2CPU(*(__u16 *)(void *)&sense->asc) << 16);
		return ret;
	} else {
		return 0xffffffff;
	}
}

__u32 SCSI$GET_CMD_SENSE(const SCSIRQ *rq)
{
	if (__unlikely(rq->status < 0) || __unlikely(rq->scsi_status != SCSI_CHECK_CONDITION)) return 0xffffffff;
	return SCSI$GET_SENSE((SCSI_SENSE_DATA *)rq->sense, rq->sense_size);
}

const char *SCSI$SENSE_STRING(__u8 scsi_version, const SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	static char string[__MAX_STR_LEN];
	if (__unlikely(sense_size < SCSI_MIN_REQUEST_SENSE_SIZE)) return "TOO SMALL SENSE SIZE";
	if (__likely((sense->response_code & (SENSE_RESPONSE_CODE & ~SENSE_RESPONSE_CODE_DEFERRED_BIT)) == SENSE_RESPONSE_CODE_CURRENT)) {
		__u8 key = (sense->flags & SENSE_FLAGS_SENSE_KEY) >> __BSF_CONST(SENSE_FLAGS_SENSE_KEY);
		if (__unlikely(scsi_version == 1) || __unlikely(sense_size <= __offsetof(SCSI_SENSE_DATA, ascq)) || __unlikely(sense->additional_length <= __offsetof(SCSI_SENSE_DATA, ascq) - __offsetof(SCSI_SENSE_DATA, additional_length) + 1) || __unlikely(!(sense->asc | sense->ascq))) return sense_keys[key];
		_snprintf(string, sizeof string, "%s: %s", sense_keys[key], SCSI$SENSE_ASC_STRING(sense->asc, sense->ascq));
		return string;
	}
	if (__unlikely(scsi_version >= 2)) {
		_snprintf(string, sizeof string, "INVALID SENSE CODE %02X", sense->response_code);
		return string;
	}
#define sense1	((SCSI1_SENSE_DATA *)sense)
	if (sense1->flags & SENSE1_FLAGS_ADVALID)
		_snprintf(string, sizeof string, "ERROR CLASS %02X, CODE %02X AT LBA %08X", (sense->flags & SENSE1_FLAGS_ERROR_CLASS) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CLASS), (sense->flags & SENSE1_FLAGS_ERROR_CODE) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CODE), ((sense1->lba3 & SENSE1_LBA3_LBA3) << 16) | (sense1->lba2 << 8) | sense1->lba1);
	else
		_snprintf(string, sizeof string, "ERROR CLASS %02X, CODE %02X", (sense->flags & SENSE1_FLAGS_ERROR_CLASS) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CLASS), (sense->flags & SENSE1_FLAGS_ERROR_CODE) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CODE));
#undef sense1
	return string;
}

const char *SCSI$STATUS_STRING(__u8 status)
{
	static char string[__MAX_STR_LEN];
	switch (status) {
		case SCSI_GOOD:
			return "GOOD";
		case SCSI_CHECK_CONDITION:
			return "CHECK CONDITION";
		case SCSI_CONDITION_MET:
			return "CONDITION MET";
		case SCSI_BUSY:
			return "BUSY";
		case SCSI_INTERMEDIATE:
			return "INTERMEDIATE";
		case SCSI_INTERMEDIATE_CONDITION_MET:
			return "INTERMEDIATE - CONDITION MET";
		case SCSI_RESERVATION_CONFLICT:
			return "RESERVATION CONFLICT";
		case SCSI_COMMAND_TERMINATED:
			return "COMMAND TERMINATED";
		case SCSI_QUEUE_FULL:
			return "QUEUE FULL";
		case SCSI_ACA_ACTIVE:
			return "ACA ACTIVE";
		case SCSI_TASK_ABORTED:
			return "TASK ABORTED";
		default:
			_snprintf(string, sizeof string, "UNKNOWN STATUS %02X", status);
			return string;
	}
}

static const struct command_desc {
	__u8 cmd;
	const char *string;
} commands[] = {
	SCMD_TEST_UNIT_READY, "TEST UNIT READY",
	SCMD_REZERO_UNIT, "REZERO UNIT / REWIND",
	SCMD_REQUEST_SENSE, "REQUEST SENSE",
	SCMD_FORMAT_UNIT, "FORMAT UNIT",
	SCMD_SA_READ_BLOCK_LIMITS, "READ BLOCK LIMITS",
	SCMD_REASSIGN_BLOCKS, "REASSIGN BLOCKS / INITIALIZE ELEMENT STATUS",
	SCMD_READ_6, "READ(6)",
	SCMD_WRITE_6, "WRITE(6)",
	SCMD_SEEK_6, "SEEK(6) / TRACK SELECT / SLEW AND PRINT",
	SCMD_SA_READ_REVERSE, "READ REVERSE",
	SCMD_SA_WRITE_FILEMARKS, "WRITE FILEMARKS / PRINT FLUSH",
	SCMD_SA_SPACE, "SPACE",
	SCMD_INQUIRY, "INQUIRY",
	SCMD_SA_VERIFY, "VERIFY",
	SCMD_SA_RECOVER_BUFFERED_DATA, "RECOVER BUFFERED DATA",
	SCMD_MODE_SELECT_6, "MODE SELECT(6)",
	SCMD_RESERVE_6, "RESERVE(6)",
	SCMD_RELEASE_6, "RELEASE(6)",
	SCMD_COPY, "COPY",
	SCMD_SA_ERASE, "ERASE",
	SCMD_MODE_SENSE_6, "MODE SENSE(6)",
	SCMD_START_STOP_UNIT, "START STOP UNIT",
	SCMD_RECEIVE_DIAGNOSTIC_RESULTS, "RECEIVE DIAGNOSTIC RESULTS",
	SCMD_SEND_DIAGNOSTIC, "SEND DIAGNOSTIC",
	SCMD_PREVENT_ALLOW_MEDIUM_REMOVAL, "PREVENT ALLOW MEDIUM REMOVAL",
	SCMD_READ_FORMAT_CAPACITIES, "READ FORMAT CAPACITIES",
	SCMD_SCN_SET_WINDOW, "SCANNER SET WINDOW",
	SCMD_READ_CAPACITY_10, "READ CAPACITY(10)",
	SCMD_READ_10, "READ(10)",
	SCMD_OPT_READ_GENERATION, "READ GENERATION",
	SCMD_WRITE_10, "WRITE(10)",
	SCMD_SEEK_10, "SEEK(10) / LOCATE",
	SCMD_OPT_ERASE_10, "ERASE(10)",
	SCMD_OPT_READ_UPDATED_BLOCK, "READ UPDATED BLOCK",
	SCMD_WRITE_AND_VERIFY_10, "WRITE AND VERIFY(10)",
	SCMD_VERIFY_10, "VERIFY(10)",
	SCMD_SEARCH_DATA_HIGH_10, "SEARCH DATA HIGH(10)",
	SCMD_SEARCH_DATA_EQUAL_10, "SEARCH DATA EQUAL(10) / OBJECT POSITION",
	SCMD_SEARCH_DATA_LOW_10, "SEARCH DATA LOW(10)",
	SCMD_SET_LIMITS_10, "SET LIMITS(10)",
	SCMD_PREFETCH_10, "PREFETCH(10) / READ POSITION / GET BUFFER STATUS",
	SCMD_SYNCHRONIZE_CACHE_10, "SYNCHRONIZE CACHE(10)",
	SCMD_LOCK_UNLOCK_CACHE, "UNLOCK CACHE(10)",
	SCMD_READ_DEFECT_DATA_10, "READ DEFECT DATA(10)",
	SCMD_OPT_MEDIUM_SCAN, "MEDIUM SCAN",
	SCMD_COMPARE, "COMPARE",
	SCMD_COPY_AND_VERIFY, "COPY AND VERIFY",
	SCMD_WRITE_BUFFER, "WRITE BUFFER",
	SCMD_READ_BUFFER, "READ BUFFER",
	SCMD_OPT_UPDATE_BLOCK, "UPDATE BLOCK",
	SCMD_READ_LONG_10, "READ LONG(10)",
	SCMD_WRITE_LONG_10, "WRITE LONG(10)",
	SCMD_CHANGE_DEFINITION, "CHANGE DEFINITION",
	SCMD_WRITE_SAME_10, "WRITE SAME(10)",
	SCMD_CD_READ_SUB_CHANNEL, "READ SUB CHANNEL",
	SCMD_CD_READ_TOC, "READ TOC",
	SCMD_CD_READ_HEADER, "READ HEADER",
	SCMD_CD_PLAY_AUDIO_10, "PLAY AUDIO(10)",
	SCMD_CD_PLAY_AUDIO_MSF, "PLAY AUDIO MSF(10)",
	SCMD_CD_PLAY_AUDIO_TRACK_INDEX, "PLAY AUDIO TRACK INDEX(10)",
	SCMD_CD_PLAY_TRACK_RELATIVE, "PLAY AUDIO TRACK RELATIVE(10)",
	SCMD_GET_EVENT_STATUS_NOTIFICATION, "GET EVENT STATUS NOTIFICATION",
	SCMD_CD_PAUSE_RESUME, "PAUSE RESUME",
	SCMD_LOG_SELECT, "LOG SELECT",
	SCMD_LOG_SENSE, "LOG SENSE",
	SCMD_XDWRITE_10, "XDWRITE(10)",
	SCMD_XPWRITE_10, "XPWRITE(10)",
	SCMD_XDREAD_10, "XDREAD(10)",
	SCMD_XDWRITEREAD_10, "XDWRITEREAD(10)",
	SCMD_MODE_SELECT_10, "MODE SELECT(10)",
	SCMD_RESERVE_10, "RESERVE(10)",
	SCMD_RELEASE_10, "RELEASE(10)",
	SCMD_MODE_SENSE_10, "MODE SENSE(10)",
	SCMD_PERSISTENT_RESERVE_IN, "PERSISTENT RESERVE IN",
	SCMD_PERSISTENT_RESERVE_OUT, "PERSISTENT RESERVE OUT",
	SCMD_XDWRITE_EXTENDED, "XDWRITE EXTENDED",
	SCMD_REBUILD, "REBUILD",
	SCMD_REGENERATE, "REGENERATE",
	SCMD_EXTENDED_COPY, "EXTENDED COPY",
	SCMD_RECEIVE_COPY_RESULTS, "RECEIVE COPY RESULTS",
	SCMD_ACCESS_CONTROL_IN, "ACCESS CONTROL IN",
	SCMD_ACCESS_CONTROL_OUT, "ACCESS CONTROL OUT",
	SCMD_READ_16, "READ(16)",
	SCMD_WRITE_16, "WRITE(16)",
	SCMD_READ_ATTRIBUTE, "READ ATTRIBUTE",
	SCMD_WRITE_AND_VERIFY_16, "WRITE AND VERIFY(16)",
	SCMD_VERIFY_16, "VERIFY(16)",
	SCMD_PREFETCH_16, "PREFETCH(16)",
	SCMD_SYNCHRONIZE_CACHE_16, "SYNCHRONIZE CACHE(16)",
	SCMD_WRITE_SAME_16, "WRITE SAME(16)",
	SCMD_READ_CAPACITY_16, "READ CAPACITY(16)",
	/*SCMD_READ_LONG_16, "READ LONG(16)",*/
	SCMD_WRITE_LONG_16, "WRITE LONG(16)",
	SCMD_REPORT_LUNS, "REPORT LUNS",
	SCMD_REPORT_DEVICE_IDENTIFIER, "REPORT DEVICE IDENTIFIER",
	SCMD_SET_DEVICE_IDENTIFIER, "SET DEVICE IDENTIFIER",
	SCMD_CD_PLAY_AUDIO_12, "PLAY AUDIO(12)",
	SCMD_MCHG_EXCHANGE_MEDIUM, "EXCHANGE MEDIUM",
	SCMD_MOVE_MEDIUM_ATTACHED, "MOVE MEDIUM ATTACHED",
	SCMD_READ_12, "READ(12)",
	SCMD_CD_PLAY_TRACK_RELATIVE_12, "PLAY AUDIO TRACK RELATIVE(12)",
	SCMD_WRITE_12, "WRITE(12)",
	SCMD_READ_MEDIA_SERIAL_NUMBER, "READ MEDIA SERIAL NUMBER",
	SCMD_OPT_ERASE_12, "ERASE(12)",
	SCMD_WRITE_AND_VERIFY_12, "WRITE AND VERIFY(12)",
	SCMD_VERIFY_12, "VERIFY(12)",
	SCMD_SEARCH_DATA_HIGH_12, "SEARCH DATA HIGH(12)",
	SCMD_SEARCH_DATA_EQUAL_12, "SEARCH DATA EQUAL(12)",
	SCMD_SEARCH_DATA_LOW_12, "SEARCH DATA LOW(12)",
	SCMD_SET_LIMITS_12, "SET LIMITS(12)",
	SCMD_READ_ELEMENT_STATUS_ATTACHED, "READ ELEMENT STATUS ATTACHED",
	SCMD_MCHG_REQUEST_VOLUME_ELEMENT_ADDRESS, "REQUEST VOLUME ELEMENT ADDRESS",
	SCMD_MCHG_SEND_VOLUME_TAG, "SEND VOLUME TAG",
	SCMD_READ_DEFECT_DATA_12, "READ DEFECT DATA(12)",
	SCMD_MCHG_READ_ELEMENT_STATUS, "READ ELEMENT STATUS",
	SCMD_READ_CD_MSF, "READ CD MSF",
	SCMD_REDUNDANCY_GROUP_IN, "REDUNDANCY GROUP IN",
	SCMD_SET_CD_SPEED, "SET CD SPEED / REDUNDANCY GROUP OUT",
	SCMD_SPARE_IN, "SPARE IN",
	SCMD_SPARE_OUT, "SPARE OUT",
	SCMD_READ_CD, "READ CD / VOLUME SET IN",
	SCMD_VOLUME_SET_OUT, "VOLUME SET OUT",
};

#define N_COMMANDS	(sizeof(commands) / sizeof(*commands))

static const char *GET_COMMAND_NAME(const SCSIRQ *srq)
{
	int from = 0;
	int to = N_COMMANDS - 1;
	while (from <= to) {
		int middle = (from + to) >> 1;
		const struct command_desc *d = &commands[middle];
		if (d->cmd < srq->cmd[0]) {
			from = middle + 1;
			continue;
		}
		if (d->cmd > srq->cmd[0]) {
			to = middle - 1;
			continue;
		}
		return d->string;
	}
	return NULL;
}

const char *SCSI$COMMAND_NAME(__const__ SCSIRQ *srq)
{
	static char c[3];
	const char *name = GET_COMMAND_NAME(srq);
	if (name)
		return name;
	_snprintf(c, sizeof c, "%02X", srq->cmd[0]);
	return c;
}

const char *SCSI$COMMAND_STRING(const SCSIRQ *srq)
{
	int i;
	static char string[__MAX_STR_LEN];
	const char *name = GET_COMMAND_NAME(srq);
	if (!name)
		name = "COMMAND";
	strlcpy(string, name, sizeof string);
	for (i = 0; i < srq->cmdlen; i++) {
		_snprintf(strchr(string, 0), sizeof string - strlen(string), " %02X", srq->cmd[i]);
	}
	return string;
}

int SCSI$CMD_RECOMMENDED_ACTION(SCSIRQ *srq)
{
	return SCSI$RECOMMENDED_ACTION(srq->status, srq->scsi_status, srq->sense, srq->sense_size);
}

int SCSI$RECOMMENDED_ACTION(int status, __u8 scsi_status, __u8 *sense, unsigned sense_size)
{
	if (__unlikely(status < 0)) {
		if (status == -EBUSY ||
		    status == -EAGAIN ||
		    status == -ENXIO) return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		return SCSI_ACTION_RETRY_FEW_TIMES;
	}
	if (__likely(scsi_status == SCSI_CHECK_CONDITION)) {
		__u32 key = SCSI$GET_SENSE((SCSI_SENSE_DATA *)sense, sense_size);
		if (__unlikely(SCSI_SENSE_UNKNOWN(key))) return SCSI_ACTION_RETRY_FEW_TIMES;
		if (__unlikely(SCSI_SENSE_DEFERRED(key))) return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		switch (SCSI_SENSE_KEY(key)) {
			case SCSI_KEY_NO_SENSE:
				return SCSI_ACTION_OK;
			case SCSI_KEY_RECOVERED_ERROR:
				return SCSI_ACTION_OK;
			case SCSI_KEY_NOT_READY:
				if (SCSI_SENSE_ASC(key) == 0x3A) return SCSI_ACTION_FAIL;
				if (SCSI_SENSE_ASC(key) == 0x04) {
					switch (SCSI_SENSE_ASCQ(key)) {
						case 0x01:
							return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
						case 0x02:
						case 0x03:
							return SCSI_ACTION_FAIL;
						case 0x04:
						case 0x05:
						case 0x06:
						case 0x07:
						case 0x08:
						case 0x09:
							return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
						case 0x11:
						case 0x12:
							return SCSI_ACTION_FAIL;
					}
				}
				if (SCSI_SENSE_ASC(key) == 0x06) return SCSI_ACTION_RETRY_FEW_TIMES;
				return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
			case SCSI_KEY_MEDIUM_ERROR:
				if (SCSI_SENSE_ASC(key) == 0x3A) return SCSI_ACTION_FAIL;
				return SCSI_ACTION_RETRY_FEW_TIMES;
			case SCSI_KEY_HARDWARE_ERROR:
				if (SCSI_SENSE_ASC(key) == 0x40 && SCSI_SENSE_ASCQ(key) == 0x82)	/* Oh shit. SCSI in its evil glory! Maxtor Atlas disk is sometimes returning this instead of UNIT IS IN PROCESS OF BECOMING READY */
					return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_ILLEGAL_REQUEST:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_UNIT_ATTENTION:
				if (SCSI_SENSE_ASC(key) == 0x3A) return SCSI_ACTION_FAIL;
				return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
			case SCSI_KEY_DATA_PROTECT:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_BLANK_CHECK:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_VENDOR_SPECIFIC:
				return SCSI_ACTION_RETRY_FEW_TIMES;
			case SCSI_KEY_COPY_ABORTED:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_ABORTED_COMMAND:
				return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
			case SCSI_KEY_EQUAL:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_VOLUME_OVERFLOW:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_MISCOMPARE:
				return SCSI_ACTION_FAIL;
			default:
				return SCSI_ACTION_RETRY_FEW_TIMES;
		}
	}
	switch (scsi_status) {
		case SCSI_GOOD:
			return SCSI_ACTION_OK;
		case SCSI_CONDITION_MET:
			return SCSI_ACTION_OK;
		case SCSI_BUSY:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_INTERMEDIATE:
			return SCSI_ACTION_FAIL;
		case SCSI_INTERMEDIATE_CONDITION_MET:
			return SCSI_ACTION_FAIL;
		case SCSI_RESERVATION_CONFLICT:
			return SCSI_ACTION_FAIL;
		case SCSI_COMMAND_TERMINATED:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_QUEUE_FULL:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_ACA_ACTIVE:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_TASK_ABORTED:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		default:
			return SCSI_ACTION_RETRY_FEW_TIMES;
	}
}

int SCSI$SENSE_ERROR_CODE(__u8 scsi_version, const SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	__u32 key = SCSI$GET_SENSE(sense, sense_size);
	if (__unlikely(SCSI_SENSE_UNKNOWN(key))) {
		if (__likely(scsi_version == 1)) return -EIO;
		return -EPROTO;
	}
	switch (SCSI_SENSE_ASC(key)) {
		case 0x27:
			return -EWP;
		case 0x3A:
			return -ENOMEDIUM;
	}
	switch (SCSI_SENSE_KEY(key)) {
		case SCSI_KEY_NO_SENSE:
			return -EPROTO;
		case SCSI_KEY_RECOVERED_ERROR:
			return -EIO;
		case SCSI_KEY_NOT_READY:
			return -EIO;
		case SCSI_KEY_MEDIUM_ERROR:
			return -EIO;
		case SCSI_KEY_HARDWARE_ERROR:
			return -EIO;
		case SCSI_KEY_ILLEGAL_REQUEST:
			return -EOPNOTSUPP;
		case SCSI_KEY_UNIT_ATTENTION:
			return -ENOMEDIUM;
		case SCSI_KEY_DATA_PROTECT:
			return -EWP;
		case SCSI_KEY_BLANK_CHECK:
			return -ENOMEDIUM;
		case SCSI_KEY_VENDOR_SPECIFIC:
			return -EIO;
		case SCSI_KEY_COPY_ABORTED:
			return -ECONNABORTED;
		case SCSI_KEY_ABORTED_COMMAND:
			return -ECONNABORTED;
		case SCSI_KEY_EQUAL:
			return -EPROTO;
		case SCSI_KEY_VOLUME_OVERFLOW:
			return -ENOSPC;
		case SCSI_KEY_MISCOMPARE:
			return -EPROTO;
		default:
			return -EPROTO;
	}
}

int SCSI$STATUS_ERROR_CODE(__u8 scsi_status)
{
	switch (scsi_status) {
		case SCSI_GOOD:
			return -EPROTO;
		case SCSI_CHECK_CONDITION:
			return -EIO;
		case SCSI_CONDITION_MET:
			return -EPROTO;
		case SCSI_BUSY:
			return -EBUSY;
		case SCSI_INTERMEDIATE:
			return -EPROTO;
		case SCSI_INTERMEDIATE_CONDITION_MET:
			return -EPROTO;
		case SCSI_RESERVATION_CONFLICT:
			return -EACCES;
		case SCSI_COMMAND_TERMINATED:
			return -ECONNABORTED;
		case SCSI_QUEUE_FULL:
			return -ENOSR;
		case SCSI_ACA_ACTIVE:
			return -EPERM;
		case SCSI_TASK_ABORTED:
			return -ECONNABORTED;
		default:
			return -EPROTO;
	}
}

int SCSI$CMD_ERROR_CODE(__u8 scsi_version, const SCSIRQ *srq)
{
	if (srq->status < 0) return srq->status;
	if (srq->scsi_status == SCSI_CHECK_CONDITION) return SCSI$SENSE_ERROR_CODE(scsi_version, (SCSI_SENSE_DATA *)srq->sense, srq->sense_size);
	return SCSI$STATUS_ERROR_CODE(srq->scsi_status);
}

__sec_t SCSI$SENSE_LBA(__u8 scsi_version, const SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	if (__unlikely(sense_size < SCSI_MIN_REQUEST_SENSE_SIZE)) return -1;
	if (__likely((sense->response_code & (SENSE_RESPONSE_CODE & ~SENSE_RESPONSE_CODE_DEFERRED_BIT)) == SENSE_RESPONSE_CODE_CURRENT)) {
		if (__unlikely(sense_size < __offsetof(SCSI_SENSE_DATA, information[3]))) return -1;
		if (sense->response_code & SENSE_INFORMATION_VALID) return (sense->information[0] << 3) | (sense->information[1] << 2) | (sense->information[2] << 1) | sense->information[3];
	} else {
#define sense1	((SCSI1_SENSE_DATA *)sense)
		if (sense1->flags & SENSE1_FLAGS_ADVALID) return ((sense1->lba3 & SENSE1_LBA3_LBA3) << 16) | (sense1->lba2 << 8) | sense1->lba1;
	}
#undef sense1
	return -1;
}

__sec_t SCSI$CMD_LBA(__u8 scsi_version, const SCSIRQ *srq)
{
	if (__unlikely(srq->status < 0) || __unlikely(srq->scsi_status != SCSI_CHECK_CONDITION)) return -1;
	return SCSI$SENSE_LBA(scsi_version, (SCSI_SENSE_DATA *)srq->sense, srq->sense_size);
}

void SCSI$LOG_ERROR(const char *dev_name, int errorlevel, const SCSI_INQUIRY_DATA *inq, __u8 scsi_version, const SCSIRQ *rq, const char *append)
{
	int level, clas;
	__u32 key;
	int error = SCSI$CMD_ERROR_CODE(scsi_version, rq);
	if ((__likely(error == -ENOMEDIUM) || __unlikely(error == -EWP)) && __likely(inq->rmb & INQ_RMB_RMB)) return;
	if (__unlikely(error == -EINTR) || __unlikely(error == -ECONNRESET)) return;
	if (__unlikely(rq->status < 0)) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "%s: SCSI TRANSPORT ERROR: %s", SCSI$COMMAND_STRING(rq), strerror(-rq->status));
		return;
	}
	if (__unlikely(rq->scsi_status != SCSI_CHECK_CONDITION)) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "%s: SCSI STATUS: %s", SCSI$COMMAND_STRING(rq), SCSI$STATUS_STRING(rq->scsi_status));
		return;
	}
	key = SCSI$GET_SENSE((SCSI_SENSE_DATA *)rq->sense, rq->sense_size);
	if (SCSI_SENSE_UNKNOWN(key)) {
		level = 1, clas = __SYSLOG_HW_ERROR;
	} else switch (SCSI_SENSE_KEY(key)) {
		case SCSI_KEY_NO_SENSE:
			level = 2, clas = __SYSLOG_HW_ERROR;
		case SCSI_KEY_RECOVERED_ERROR:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_NOT_READY:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_MEDIUM_ERROR:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_HARDWARE_ERROR:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_ILLEGAL_REQUEST:
/* A lot of CDROMs send Illegal Request when reading from non-data parts of
   a CD. So make it a level 2 error. */
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_UNIT_ATTENTION:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_DATA_PROTECT:
			level = 2, clas = __SYSLOG_HW_INCAPABILITY;
			break;
		case SCSI_KEY_BLANK_CHECK:
			level = 2, clas = __SYSLOG_HW_INCAPABILITY;
			break;
		case SCSI_KEY_VENDOR_SPECIFIC:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_COPY_ABORTED:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_ABORTED_COMMAND:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_EQUAL:
			level = 1, clas = __SYSLOG_HW_BUG;
			break;
		case SCSI_KEY_VOLUME_OVERFLOW:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_MISCOMPARE:
			level = 1, clas = __SYSLOG_HW_BUG;
			break;
		default:
			level = 1, clas = __SYSLOG_HW_BUG;
			break;
	}
	if (errorlevel >= level)
		KERNEL$SYSLOG(clas, dev_name, "%s: %s%s%s", SCSI$COMMAND_STRING(rq), SCSI_SENSE_DEFERRED(key) ? "DEFERRED ERROR: " : "", SCSI$SENSE_STRING(scsi_version, (SCSI_SENSE_DATA *)rq->sense, rq->sense_size), append);
}

void SCSI$LOG_RECOVERED_ERROR(const char *dev_name, int errorlevel, const SCSIRQ *rq, const char *append)
{
	__u32 key;
	if (__unlikely(rq->status < 0) || __likely(rq->scsi_status != SCSI_CHECK_CONDITION)) return;
	key = SCSI$GET_SENSE((SCSI_SENSE_DATA *)rq->sense, rq->sense_size);
	if (__unlikely(SCSI_SENSE_UNKNOWN(key))) return;
	if (key == SCSI_KEY_RECOVERED_ERROR) {
		if (errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, dev_name, "%s: %s%s%s", SCSI$COMMAND_STRING(rq), SCSI_SENSE_DEFERRED(key) ? "DEFERRED ERROR: " : "", SCSI$SENSE_STRING(5, (SCSI_SENSE_DATA *)rq->sense, rq->sense_size), append);
	}
}

#if __DEBUG >= 2
DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	unsigned i;
	for (i = 0; i < N_SENSES; i++) {
		if (__unlikely(senses[i].ascq_from > senses[i].ascq_to)) KERNEL$SUICIDE("DLL$LOAD: SCSI SENSE TABLE ASCQ RANGE WRONG AT LINE %u", i);
		if (__unlikely(!i)) continue;
		if (__unlikely(senses[i - 1].asc > senses[i].asc)) KERNEL$SUICIDE("DLL$LOAD: SCSI SENSE TABLE ASC ORDER VIOLATION AT LINE %u", i);
		if (senses[i - 1].asc < senses[i].asc) continue;
		if (__unlikely(senses[i - 1].ascq_to >= senses[i].ascq_from)) KERNEL$SUICIDE("DLL$LOAD: SCSI SENSE TABLE ASCQ ORDER VIOLATION AT LINE %u", i);
	}
	for (i = 0; i < N_COMMANDS; i++) {
		if (__unlikely(!i)) continue;
		if (__unlikely(commands[i - 1].cmd > commands[i].cmd)) KERNEL$SUICIDE("DLL$LOAD: SCSI COMMAND TABLE ORDER VIOLATION AT LINE %u", i);
	}
	RQ->status = 0;
	RETURN_AST(RQ);
}
#endif
