#include <SPAD/BIO.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TIMER.H>
#include <SPAD/BIO_KRNL.H>

#include <SPAD/SCSI.H>

static WQ_DECL(SCSI_SYNC_WQ, "SCSI$SYNC_WQ");
static DECL_XLIST(SCSI_SYNC_LIST);

typedef struct {
	LIST_ENTRY list;
	TIMER t;
	jiffies_lo_t timeout;
	SCSIRQ *rq;
	SCSI_ATTACH_PARAM *ap;
	int (*deq)(SCSI_ATTACH_PARAM *);
	BIORQ *(*pro)(SCSI_ATTACH_PARAM *);
} SCSI_SYNC_STRUCT;

static int SCSI_SYNC_DEQUEUE(SCSI_ATTACH_PARAM *ap);
static BIORQ *SCSI_SYNC_PROBE_QUEUE(SCSI_ATTACH_PARAM *ap);
static void SCSI_SYNC_DONE(SCSIRQ *rq);
static void SCSI_SYNC_TIMEOUT(TIMER *t);

void SCSI$SYNC_RQ(const char *dev_name, int errorlevel, SCSIRQ *rq, SCSI_ATTACH_PARAM *ap, scsi_lun_t lun, void *ptr, unsigned long len, jiffies_lo_t timeout, jiffies_lo_t extended_timeout, int retries, int (*action)(SCSIRQ *rq))
{
	BIODESC desc;
	SCSI_SYNC_STRUCT st;
	int spl;
	int r;
	u_jiffies_lo_t start_time = KERNEL$GET_JIFFIES_LO();
#if __DEBUG >= 1
	*(volatile __u8 *)rq->sense;
#endif
	if (lun == LUN_NONE) lun = ap->lun;
	again:
	rq->ch_id = ap->ch_id;
	rq->lun = lun;
	rq->adapter = ap->adapter;
	rq->internal = 0;
	rq->done = SCSI_SYNC_DONE;
	if (rq->direction) {
		rq->desc = &desc;
		desc.v.ptr = (unsigned long)ptr;
		desc.v.len = len;
		desc.v.vspace = &KERNEL$VIRTUAL;
		desc.next = NULL;
	} else {
		rq->desc = NULL;
#if __DEBUG >= 1
		if (__unlikely(len != 0))
			KERNEL$SUICIDE("SCSI$SYNC_RQ: REQUEST HAS NO TRANSFER DIRECTION AND NON-ZERO TRANSFER LENGTH");
#endif
	}
	st.rq = rq;
	st.ap = ap;
	INIT_TIMER(&st.t);
	st.t.fn = NULL;
	st.timeout = timeout;
	spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_ATA_SCSI), spl)))
		KERNEL$SUICIDE("SCSI$SYNC_RQ AT SPL %08X", spl);
#endif

	RAISE_SPL(SPL_ATA_SCSI);
	ADD_TO_XLIST(&SCSI_SYNC_LIST, &st.list);
	st.deq = ap->dequeue;
	st.pro = ap->probe_queue;
	ap->dequeue = SCSI_SYNC_DEQUEUE;
	ap->probe_queue = SCSI_SYNC_PROBE_QUEUE;
	SCSI_SYNC_DEQUEUE(ap);
	
	notyet:
	if (__likely(rq->done != NULL) && __unlikely(WQ_WAIT_SYNC_CANCELABLE(&SCSI_SYNC_WQ))) {
		while (rq->done != NULL) {
			if (st.t.fn) ap->cancel(rq);
			WQ_WAIT_SYNC(&SCSI_SYNC_WQ);
		}
	}
	if (__unlikely(rq->done != NULL)) goto notyet;
	if (__likely(st.t.fn != NULL)) KERNEL$DEL_TIMER(&st.t);
	DEL_FROM_LIST(&st.list);
	LOWER_SPLX(spl);

	if (__unlikely(!st.t.fn) && __likely(rq->status == -EINTR)) rq->status = -ETIMEDOUT;
	if (__unlikely(rq->status == -EINTR)) return;
	SCSI$LOG_RECOVERED_ERROR(dev_name, errorlevel, rq, ", RECOVERED");
	if (__unlikely(!action)) return;
	r = action(rq);
	switch (r) {
		case SCSI_ACTION_RETRY_UNTIL_TIMEOUT:
			if (__likely(KERNEL$GET_JIFFIES_LO() - start_time <= TIMEOUT_JIFFIES(extended_timeout))) {
				if (__unlikely(r = KERNEL$SLEEP_CANCELABLE(JIFFIES_PER_SECOND / 10))) {
					rq->status = r;
					return;
				}
				goto again;
			}
		case SCSI_ACTION_RETRY_FEW_TIMES:
			if (--retries >= 0) goto again;
		case SCSI_ACTION_OK:
		case SCSI_ACTION_FAIL:
			return;
		default:
			KERNEL$SUICIDE("SCSI$SYNC_RQ: ACTION FUNCTION RETURNED %d", r);
	}
}

static int SCSI_SYNC_DEQUEUE(SCSI_ATTACH_PARAM *ap)
{
	int r;
	SCSI_SYNC_STRUCT *st;
	XLIST_FOR_EACH(st, &SCSI_SYNC_LIST, SCSI_SYNC_STRUCT, list) if (__likely(st->ap == ap)) goto found;
	KERNEL$SUICIDE("SCSI_SYNC_DEQUEUE: REQUEST NOT FOUND");
	found:
	r = ap->post(st->rq);
	if (__unlikely(r)) {
		if (__unlikely(r == -EAGAIN)) return BIO_DEQUEUE_FULL;
		st->rq->status = r;
		st->rq->done(st->rq);
	}
	ap->dequeue = st->deq;
	ap->probe_queue = st->pro;
	st->t.fn = SCSI_SYNC_TIMEOUT;
	KERNEL$SET_TIMER(st->timeout, &st->t);
	WQ_WAKE_ALL_PL(&SCSI_SYNC_WQ);
	return BIO_DEQUEUE_WANT_MORE;
}

static BIORQ *SCSI_SYNC_PROBE_QUEUE(SCSI_ATTACH_PARAM *ap)
{
	return BIO_COMPARE_UNSPECIFIED;
}

static void SCSI_SYNC_DONE(SCSIRQ *rq)
{
	rq->done = NULL;
	WQ_WAKE_ALL_PL(&SCSI_SYNC_WQ);
}

static void SCSI_SYNC_TIMEOUT(TIMER *t)
{
	SCSI_SYNC_STRUCT *st = GET_STRUCT(t, SCSI_SYNC_STRUCT, t);
	LOWER_SPL(SPL_ATA_SCSI);
	st->t.fn = NULL;
	st->ap->cancel(st->rq);
}

