#include <SPAD/DEV.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SCSI.H>
#include <STDLIB.H>
#include <SYS/PARAM.H>
#include <SCSI/SG.H>

#include <LINUX/CDROM.H>
#include <SPAD/IOC.H>

static SCSI_PASS_THROUGH_HEAD *alloc_pass_through(size_t size)
{
	SCSI_PASS_THROUGH_HEAD *pt;
	size += sizeof(SCSI_PASS_THROUGH_HEAD);
	pt = __sync_memalign(BIO_SECTOR_SIZE, size);
	memset(pt, 0, size);
	return pt;
}

static int do_command(int h, const __u8 *cmd, unsigned cmdlen, void *data, size_t datalen, int direction, __u8 *sense)
{
	int repeat_count = 0;
	IOCTLRQ io;
	SCSI_PASS_THROUGH_HEAD *pt = alloc_pass_through(datalen);
	if (__unlikely(!pt)) return -1;
	repeat:
	pt->direction = direction;
	pt->command_len = cmdlen;
	memcpy(pt->command, cmd, cmdlen);
	if (direction == PF_READ) memcpy(pt + 1, data, datalen);
	io.h = h;
	io.ioctl = IOCTL_BIO_SCSI_PASS_THROUGH;
	io.param = PARAM_BIO_SCSI_PASS_THROUGH_COMMAND;
	io.v.ptr = (unsigned long)pt;
	io.v.len = sizeof(SCSI_PASS_THROUGH_HEAD) + datalen;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		errno = -io.status;
		free_ret:
		free(pt);
		return -1;
	}
	if (__unlikely(sense != NULL)) memcpy(sense, pt->sense, SCSI_PASS_THROUGH_SENSE_LEN);
	if (__unlikely(pt->scsi_status != SCSI_GOOD)) {
		int ra;
		ra = SCSI$RECOMMENDED_ACTION(io.status, pt->scsi_status, pt->sense, SCSI_PASS_THROUGH_SENSE_LEN);
		if (ra == SCSI_ACTION_OK) {
			goto ok;
		}
		if (ra == SCSI_ACTION_RETRY_UNTIL_TIMEOUT || ra == SCSI_ACTION_RETRY_FEW_TIMES) {
			if (repeat_count < 1) {
				repeat_count++;
				goto repeat;
			}
		}
		if (pt->scsi_status == SCSI_CHECK_CONDITION) {
			if (sense) {
				free(pt);
				return 1;
			}
			errno = -SCSI$SENSE_ERROR_CODE(2, (SCSI_SENSE_DATA *)pt->sense, SCSI_PASS_THROUGH_SENSE_LEN);
		} else {
			errno = -SCSI$STATUS_ERROR_CODE(pt->scsi_status);
		}
		goto free_ret;
	}
	ok:
	if (direction == PF_WRITE) memcpy(data, pt + 1, datalen);
	free(pt);
	return 0;
}

static void lba_to_msf(unsigned lba, __u8 *m, __u8 *s, __u8 *f)
{
	lba += CD_MSF_OFFSET;
	lba &= 0xffffff;
	*m = lba / (CD_SECS * CD_FRAMES);
	lba %= (CD_SECS * CD_FRAMES);
	*s = lba / CD_FRAMES;
	*f = lba % CD_FRAMES;
}

static unsigned msf_to_lba(unsigned m, unsigned s, unsigned f)
{
	return ((m * CD_SECS) + s) * CD_FRAMES + f - CD_MSF_OFFSET;
}

static int mode_sense(int h, __u8 page, void *data, size_t datalen)
{
	__u8 cmd[10] = { SCMD_MODE_SENSE_10, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	memset(data, 0, datalen);
	if (__unlikely(datalen >= 0x10000)) {
		einval:
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(page >= 0x40))
		goto einval;
	cmd[2] = page;
	cmd[7] = datalen >> 8;
	cmd[8] = datalen;
	return do_command(h, cmd, sizeof cmd, data, datalen, PF_WRITE, NULL);
}

static int mode_select(int h, void *data, size_t datalen)
{
	__u8 cmd[10] = { SCMD_MODE_SELECT_10, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	if (__unlikely(datalen >= 0x10000)) {
		errno = EINVAL;
		return -1;
	}
	memset(data, 0, 2);
	cmd[1] = 0x10;
	cmd[7] = datalen >> 8;
	cmd[8] = datalen;
	return do_command(h, cmd, sizeof cmd, data, datalen, PF_READ, NULL);
}

#define CAP_MODE_SIZE	24

static int cdrom_get_capability(int h)
{
	int ret;
	__u8 cap[CAP_MODE_SIZE];
	if (__unlikely(mode_sense(h, GPMODE_CAPABILITIES_PAGE, cap, CAP_MODE_SIZE)))
		return -1;
	/*{
		int i;
		for (i = 0; i < CAP_MODE_SIZE; i++)
			__debug_printf("%02x ", cap[i]);
		__debug_printf("\n");
	}*/
	ret = 0;
	if (cap[8 + 2] & 0x38)
		ret |= CDC_DVD;
	if (cap[8 + 3] & 0x01)
		ret |= CDC_CD_R;
	if (cap[8 + 3] & 0x02)
		ret |= CDC_CD_RW | CDC_RAM;
	if (cap[8 + 3] & 0x10)
		ret |= CDC_DVD_R;
	if (cap[8 + 3] & 0x20)
		ret |= CDC_DVD_RAM | CDC_RAM;
	if (cap[8 + 4] & 0x01)
		ret |= CDC_PLAY_AUDIO;
	if (cap[8 + 4] & 0x40)
		ret |= CDC_MULTI_SESSION;
	if (cap[8 + 6] & 0x08)
		ret |= CDC_OPEN_TRAY;
	switch (cap[8 + 6] >> 5) {
		case 0:
		case 2:
			ret |= CDC_CLOSE_TRAY;
			break;
	}
	return ret;
}

static int test_cap(int h, int must_have)
{
	int cap = cdrom_get_capability(h);
	if (__unlikely(cap < 0))
		return cap;
	if (__unlikely((cap & must_have) != must_have)) {
		errno = ENOSYS;
		return -1;
	}
	return 0;
}

static int cdrom_readtochdr(int h, struct cdrom_tochdr *tochdr)
{
	__u8 ret[0x0c];
	static const __u8 cmd[10] = { SCMD_CD_READ_TOC, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xc, 0x0 };
	if (__unlikely(do_command(h, cmd, sizeof cmd, &ret, 0x0c, PF_WRITE, NULL)))
		return -1;
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	tochdr->cdth_trk0 = ret[2];
	tochdr->cdth_trk1 = ret[3];
	return 0;
}

static int cdrom_readtocentry(int h, struct cdrom_tocentry *tocentry)
{
	__u8 ret[0x0c];
	__u8 cmd[10] = { SCMD_CD_READ_TOC, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xc, 0x0 };
	if (tocentry->cdte_format != CDROM_LBA && __unlikely(tocentry->cdte_format != CDROM_MSF)) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	cmd[6] = tocentry->cdte_track;
	if (__unlikely(do_command(h, cmd, sizeof cmd, &ret, 0x0c, PF_WRITE, NULL)))
		return -1;
	tocentry->cdte_ctrl = ret[5] & 0x0f;
	tocentry->cdte_adr = ret[5] >> 4;
	tocentry->cdte_datamode = (tocentry->cdte_ctrl & 0x04) ? 1 : 0;
	tocentry->cdte_addr.lba = (ret[8] << 24) | (ret[9] << 16) | (ret[10] << 8) | ret[11];
	if (tocentry->cdte_format == CDROM_MSF) {
		lba_to_msf(tocentry->cdte_addr.lba, &tocentry->cdte_addr.msf.minute, &tocentry->cdte_addr.msf.second, &tocentry->cdte_addr.msf.frame);
	}
	return 0;
}

static int cdrom_playmsf(int h, struct cdrom_msf *msf)
{
	__u8 cmd[10] = { SCMD_CD_PLAY_AUDIO_MSF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	cmd[3] = msf->cdmsf_min0;
	cmd[4] = msf->cdmsf_sec0;
	cmd[5] = msf->cdmsf_frame0;
	cmd[6] = msf->cdmsf_min1;
	cmd[7] = msf->cdmsf_sec1;
	cmd[8] = msf->cdmsf_frame1;
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_playtrkind(int h, struct cdrom_ti *ti)
{
	__u8 cmd[10] = { SCMD_CD_PLAY_AUDIO_MSF, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
	struct cdrom_tochdr tochdr;
	struct cdrom_tocentry tocentry0;
	struct cdrom_tocentry tocentry1;
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	if (__unlikely(cdrom_readtochdr(h, &tochdr)))
		return -1;
	tocentry0.cdte_format = CDROM_LBA;
	tocentry0.cdte_track = ti->cdti_trk0;
	tocentry1.cdte_format = CDROM_LBA;
	tocentry1.cdte_track = ti->cdti_trk1 + 1;
	if (tocentry1.cdte_track > tochdr.cdth_trk1)
		tocentry1.cdte_track = CDROM_LEADOUT;
	if (__unlikely(cdrom_readtocentry(h, &tocentry0))) return -1;
	if (__unlikely(cdrom_readtocentry(h, &tocentry1))) return -1;
	if (__unlikely(tocentry0.cdte_addr.lba >= tocentry1.cdte_addr.lba)) {
		errno = EINVAL;
		return -1;
	}
	lba_to_msf(tocentry0.cdte_addr.lba, &cmd[3], &cmd[4], &cmd[5]);
	lba_to_msf(tocentry1.cdte_addr.lba - 1, &cmd[6], &cmd[7], &cmd[8]);
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_subchnl(int h, struct cdrom_subchnl *sc)
{
	__u8 ret[0x10];
	static const __u8 cmd[10] = { SCMD_CD_READ_SUB_CHANNEL, 0x2, 0x40, 0x1, 0x0, 0x0, 0x0, 0x0, 0x10, 0x0 };
	if (sc->cdsc_format != CDROM_LBA && __unlikely(sc->cdsc_format != CDROM_MSF)) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	if (__unlikely(do_command(h, cmd, sizeof cmd, &ret, 0x10, PF_WRITE, NULL)))
		return -1;
	sc->cdsc_audiostatus = ret[1];
	sc->cdsc_ctrl = ret[5] & 0xf;
	sc->cdsc_trk = ret[6];
	sc->cdsc_ind = ret[7];

	if (sc->cdsc_format == CDROM_MSF) {
		sc->cdsc_absaddr.msf.minute = ret[9];
		sc->cdsc_absaddr.msf.second = ret[10];
		sc->cdsc_absaddr.msf.frame = ret[11];
		sc->cdsc_reladdr.msf.minute = ret[13];
		sc->cdsc_reladdr.msf.second = ret[14];
		sc->cdsc_reladdr.msf.frame = ret[15];
	} else {
		sc->cdsc_absaddr.lba = msf_to_lba(ret[9], ret[10], ret[11]);
		sc->cdsc_reladdr.lba = msf_to_lba(ret[13], ret[14], ret[15]);
	}

	return -ENOSYS;
}

static int cdrom_pause(int h)
{
	static const __u8 cmd[10] = { SCMD_CD_PAUSE_RESUME, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_resume(int h)
{
	static const __u8 cmd[10] = { SCMD_CD_PAUSE_RESUME, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0 };
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_stop(int h)
{
	static const __u8 cmd[6] = { SCMD_START_STOP_UNIT, 0x1, 0x0, 0x0, 0x0, 0x0 };
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_start(int h)
{
	static const __u8 cmd[6] = { SCMD_START_STOP_UNIT, 0x1, 0x0, 0x0, 0x1, 0x0 };
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_eject(int h)
{
	static const __u8 cmd[6] = { SCMD_START_STOP_UNIT, 0x1, 0x0, 0x0, 0x2, 0x0 };
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_closetray(int h)
{
	static const __u8 cmd[6] = { SCMD_START_STOP_UNIT, 0x1, 0x0, 0x0, 0x3, 0x0 };
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_multisession(int h, struct cdrom_multisession *ms)
{
	__u8 ret[0x0c];
	static const __u8 cmd[10] = { SCMD_CD_READ_TOC, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xc, 0x40 };
	if (__unlikely(ms->addr_format != CDROM_LBA) && __unlikely(ms->addr_format != CDROM_MSF)) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(test_cap(h, CDC_MULTI_SESSION)))
		return -1;
	if (__unlikely(do_command(h, cmd, sizeof cmd, &ret, 0x0c, PF_WRITE, NULL)))
		return -1;
	ms->addr.lba = (ret[8] << 24) | (ret[9] << 16) | (ret[10] << 8) | ret[11];
	ms->xa_flag = ms->addr.lba > 0;	/* should do something else ... */
	if (ms->addr_format == CDROM_MSF) {
		lba_to_msf(ms->addr.lba, &ms->addr.msf.minute, &ms->addr.msf.second, &ms->addr.msf.frame);
	}

	return 0;
}

#define VOL_MODE_MIN	24
#define VOL_MODE_MAX	32

static int cdrom_volctrl(int h, unsigned long io, struct cdrom_volctrl *volctrl)
{
	__u8 vol_desc[VOL_MODE_MAX];
	unsigned offset;
	if (__unlikely(test_cap(h, CDC_PLAY_AUDIO)))
		return -1;
	if (__unlikely(mode_sense(h, GPMODE_AUDIO_CTL_PAGE, vol_desc, VOL_MODE_MIN)))
		return -1;
	offset = 8 + (vol_desc[6] << 8) + vol_desc[7];
	if (offset + 16 > sizeof vol_desc) {
		errno = EFBIG;
		return -1;
	}
	if (offset + 16 > VOL_MODE_MIN) {
		if (__unlikely(mode_sense(h, GPMODE_AUDIO_CTL_PAGE, vol_desc, offset + 16)))
			return -1;
	}
	if ((vol_desc[offset] & 0x3f) != GPMODE_AUDIO_CTL_PAGE || vol_desc[offset + 1] < 14) {
		errno = EINVAL;
		return -1;
	}
	if (io == CDROMVOLREAD) {
		volctrl->channel0 = vol_desc[offset+9];
		volctrl->channel1 = vol_desc[offset+11];
		volctrl->channel2 = vol_desc[offset+13];
		volctrl->channel3 = vol_desc[offset+15];
		/*__debug_printf("volume: %d %d %d %d\n", volctrl->channel0, volctrl->channel1, volctrl->channel2, volctrl->channel3);*/
		return 0;
	} else if (io == CDROMVOLCTRL) {
		vol_desc[offset+9] = volctrl->channel0;
		vol_desc[offset+11] = volctrl->channel1;
		vol_desc[offset+13] = volctrl->channel2;
		vol_desc[offset+15] = volctrl->channel3;
		memset(vol_desc + offset - 8, 0, 8);
		return mode_select(h, vol_desc + offset - 8, 24);
	} else KERNEL$SUICIDE("cdrom_volctrl: INVALID IOCTL %ld", io);
}

static int cdrom_select_speed(int h, int speed)
{
	__u8 cmd[12] = { SCMD_SET_CD_SPEED, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
	int cap = cdrom_get_capability(h);
	if (__unlikely(cap < 0))
		return cap;
	if (!speed) speed = 0xffff;
	else speed *= 177;
	cmd[2] = speed >> 8;
	cmd[3] = speed;
	if (cap & (CDC_CD_R | CDC_CD_RW | CDC_DVD_R)) {
		cmd[4] = speed >> 8;
		cmd[5] = speed;
	}
	return do_command(h, cmd, sizeof cmd, NULL, 0, 0, NULL);
}

static int cdrom_drive_status(int h, int slot)
{
	int r;
	union {
		__u8 chr[SCSI_PASS_THROUGH_SENSE_LEN];
		SCSI_SENSE_DATA sense;
	} sense;
	__u32 sense_key;
	__u8 event[8];
	static const __u8 cmd[6] = { SCMD_TEST_UNIT_READY, 0x0, 0x0, 0x0, 0x0, 0x0 };
	static const __u8 cmd2[10] = { SCMD_GET_EVENT_STATUS_NOTIFICATION, 0x1, 0x0, 0x0, 0x10, 0x0, 0x0, 0x0, sizeof event, 0x0 };
	int saved_errno;
	if (__unlikely(slot != CDSL_CURRENT)) {
		errno = EINVAL;
		return -1;
	}
	r = do_command(h, cmd, sizeof cmd, NULL, 0, 0, sense.chr);
	if (__unlikely(r < 0))
		return -1;
	if (__likely(!r))
		return CDS_DISC_OK;

	saved_errno = errno;
	r = do_command(h, cmd2, sizeof cmd2, event, sizeof event, PF_WRITE, NULL);
	errno = saved_errno;
	if (!r) {
		/*__debug_printf("%x %x %x %x %x %x %x %x\n", event[0], event[1], event[2], event[3], event[4], event[5], event[6], event[7]);*/
		if (((event[0] << 8) | event[1]) < 4) goto skip_event;
		if ((event[2] & 0x87) != 0x04) goto skip_event;
		if (event[5] & 0x02)
			return CDS_DISC_OK;
		if (event[5] & 0x01)
			return CDS_TRAY_OPEN;
		return CDS_NO_DISC;
	}
	skip_event:

	sense_key = SCSI$GET_SENSE(&sense.sense, SCSI_PASS_THROUGH_SENSE_LEN);
	if (SCSI_SENSE_KEY(sense_key) == SCSI_KEY_NOT_READY && sense.sense.asc == 0x04)
		return CDS_DISC_OK;
	if (sense.sense.asc == 0x3a)
		return CDS_NO_DISC;
	else
		return CDS_TRAY_OPEN;
}

static unsigned max_cdrom_frames = CD_FRAMES;

static int cdrom_readaudio(int h, struct cdrom_read_audio *ra)
{
	unsigned lba;
	__u8 *buf;
	unsigned frames_to_left;
	if (ra->addr_format == CDROM_LBA) {
		lba = ra->addr.lba;
	} else if (__likely(ra->addr_format == CDROM_MSF)) {
		lba = msf_to_lba(ra->addr.msf.minute, ra->addr.msf.second, ra->addr.msf.frame);
	} else {
		einval:
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(ra->nframes < 0)) goto einval;
	buf = ra->buf;
	frames_to_left = ra->nframes;
	while (frames_to_left) {
		__u8 cmd[12] = { SCMD_READ_CD, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00 };
		int r;
		int saved_errno;
		unsigned nr_frames = MIN(frames_to_left, max_cdrom_frames);
		cmd[2] = lba >> 24;
		cmd[3] = lba >> 16;
		cmd[4] = lba >> 8;
		cmd[5] = lba;
		cmd[6] = nr_frames >> 16;
		cmd[7] = nr_frames >> 8;
		cmd[8] = nr_frames;
		saved_errno = errno;
		/*__debug_printf("do command\n");*/
		r = do_command(h, cmd, sizeof cmd, buf, nr_frames * CD_FRAMESIZE_RAW, PF_WRITE, NULL);
		/*__debug_printf("done command: %d, %d\n", r, errno);*/
		if (__unlikely(r)) {
			if (errno == ERANGE && __likely(nr_frames > 1)) {
				errno = saved_errno;
				max_cdrom_frames = nr_frames - 1;
				continue;
			}
			return r;
		}
		frames_to_left -= nr_frames;
		buf += nr_frames * CD_FRAMESIZE_RAW;
		lba += nr_frames;
	}
	return 0;
}

static int cdrom_reset(int h)
{
	int reset = SG_SCSI_RESET_DEVICE;
	return ioctl(h, SG_SCSI_RESET, &reset);
}

static int cdrom_send_packet(int h, struct cdrom_generic_command *cmd)
{
	int r;
	struct sg_io_hdr sgio;
	sgio.interface_id = SG_INTERFACE_ID_ORIG;
	if (cmd->data_direction == CGC_DATA_WRITE)
		sgio.dxfer_direction = SG_DXFER_TO_DEV;
	else if (cmd->data_direction == CGC_DATA_READ)
		sgio.dxfer_direction = SG_DXFER_FROM_DEV;
	else if (cmd->data_direction == CGC_DATA_NONE)
		sgio.dxfer_direction = SG_DXFER_NONE;
	else if (cmd->data_direction == CGC_DATA_UNKNOWN)
		sgio.dxfer_direction = SG_DXFER_UNKNOWN;
	else {
		errno = EINVAL;
		return -1;
	}
	sgio.cmd_len = CDROM_PACKET_SIZE;
	sgio.mx_sb_len = cmd->sense ? sizeof(struct request_sense) : 0;
	sgio.iovec_count = 0;
	sgio.dxfer_len = cmd->buflen;
	sgio.dxferp = cmd->buffer;
	sgio.sbp = cmd->sense;
	sgio.timeout = cmd->timeout;
	sgio.flags = SG_FLAG_DIRECT_IO;
	if (__unlikely(r = ioctl(h, SG_IO, &sgio))) {
		cmd->stat = -errno;
		cmd->buflen = 0;
		return r;
	}
	cmd->stat = 0;
	cmd->buflen = sgio.resid;
	if (__unlikely(sgio.status)) {
		errno = EIO;
		cmd->stat = -EIO;
		return -1;
	}
	return 0;
}


#define IOC_0_WRAPPER(fn)					\
static int ioc_##fn(int h, unsigned long io, va_list va)	\
{								\
	return fn(h);						\
}

#define IOC_1_WRAPPER(fn, struc)				\
static int ioc_##fn(int h, unsigned long io, va_list va)	\
{								\
	return fn(h, va_arg(va, struc));			\
}

#define IOC_I1_WRAPPER(fn, struc)				\
static int ioc_##fn(int h, unsigned long io, va_list va)	\
{								\
	return fn(h, io, va_arg(va, struc));			\
}

IOC_0_WRAPPER(cdrom_pause);
IOC_0_WRAPPER(cdrom_resume);
IOC_1_WRAPPER(cdrom_playmsf, struct cdrom_msf *);
IOC_1_WRAPPER(cdrom_playtrkind, struct cdrom_ti *);
IOC_1_WRAPPER(cdrom_readtochdr, struct cdrom_tochdr *);
IOC_1_WRAPPER(cdrom_readtocentry, struct cdrom_tocentry *);
IOC_0_WRAPPER(cdrom_stop);
IOC_0_WRAPPER(cdrom_start);
IOC_0_WRAPPER(cdrom_eject);
IOC_1_WRAPPER(cdrom_subchnl, struct cdrom_subchnl *);
IOC_1_WRAPPER(cdrom_readaudio, struct cdrom_read_audio *);
IOC_1_WRAPPER(cdrom_multisession, struct cdrom_multisession *);
IOC_0_WRAPPER(cdrom_reset);
IOC_I1_WRAPPER(cdrom_volctrl, struct cdrom_volctrl *);
IOC_0_WRAPPER(cdrom_closetray);
IOC_1_WRAPPER(cdrom_select_speed, int);
IOC_1_WRAPPER(cdrom_drive_status, int);
IOC_0_WRAPPER(cdrom_get_capability);
IOC_1_WRAPPER(cdrom_send_packet, struct cdrom_generic_command *);

static const struct ioc_entry ioc_table[] = {
	CDROMPAUSE,		1,	ioc_cdrom_pause,
	CDROMRESUME,		1,	ioc_cdrom_resume,
	CDROMPLAYMSF,		1,	ioc_cdrom_playmsf,
	CDROMPLAYTRKIND,	1,	ioc_cdrom_playtrkind,
	CDROMREADTOCHDR,	1,	ioc_cdrom_readtochdr,
	CDROMREADTOCENTRY,	1,	ioc_cdrom_readtocentry,
	CDROMSTOP,		1,	ioc_cdrom_stop,
	CDROMSTART,		1,	ioc_cdrom_start,
	CDROMEJECT,		1,	ioc_cdrom_eject,
	CDROMVOLCTRL,		1,	ioc_cdrom_volctrl,
	CDROMREADAUDIO,		1,	ioc_cdrom_readaudio,
	CDROMSUBCHNL,		1,	ioc_cdrom_subchnl,
	CDROMMULTISESSION,	1,	ioc_cdrom_multisession,
	CDROMRESET,		1,	ioc_cdrom_reset,
	CDROMVOLREAD,		1,	ioc_cdrom_volctrl,
	CDROMCLOSETRAY,		1,	ioc_cdrom_closetray,
	CDROM_SELECT_SPEED,	1,	ioc_cdrom_select_speed,
	CDROM_DRIVE_STATUS,	1,	ioc_cdrom_drive_status,
	CDROM_GET_CAPABILITY,	1,	ioc_cdrom_get_capability,
	CDROM_SEND_PACKET,	1,	ioc_cdrom_send_packet,
	0,	0,	NULL,
};

const struct ioc_entry * const IOC$TABLE = ioc_table;

translate_unx_fn IOC$TRANSLATE_UNX;

const char *IOC$TRANSLATE_UNX(const char *str, const char **to_free)
{
	if (__likely(!strcmp(str, "/dev/cdrom")) ||
	    __unlikely(!strcmp(str, "/dev/dvd")))
		return "BIO$CD:/";
	return NULL;
}
