#include <SPAD/LIBC.H>
#include <SPAD/IOCTL.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <STDLIB.H>
#include <SPAD/TIMER.H>

#include <SYS/IOCTL.H>
#include <SCSI/SCSI.H>
#include <SCSI/SG.H>
#include <SPAD/IOC.H>

#define SG_VERSION	30000

typedef struct scsi_idlun {
	__u32 dev_id;
	__u32 host_unique_id;
} Scsi_Idlun;

static long scsi_get_param(int h, unsigned long param);

__COLD_ATTR__ static int scsi_test(int h)
{
	return scsi_get_param(h, PARAM_BIO_SCSI_PASS_THROUGH_EMULATED_HOST);
}

__COLD_ATTR__ static long scsi_get_param(int h, unsigned long param)
{
	IOCTLRQ ioc;
	ioc.h = h;
	ioc.ioctl = IOCTL_BIO_SCSI_PASS_THROUGH;
	ioc.param = param;
	ioc.v.ptr = 0;
	ioc.v.len = 0;
	ioc.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&ioc, KERNEL$IOCTL);
	if (__unlikely(ioc.status < 0)) {
		if (ioc.status == -ENOOP) errno = ENOTTY;
		else errno = -ioc.status;
		return -1;
	}
	return ioc.status;
}

#define FN0(name)					\
static int name(int h, unsigned long io, va_list va)	\
{							\

#define FN(name, argtype, argname)			\
static int name(int h, unsigned long io, va_list va)	\
{							\
	argtype argname = va_arg(va, argtype);

__COLD_ATTR__ FN(ioc_get_idlun, struct scsi_idlun *, idlun)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	/* It's none of the caller's business */
	memset(idlun, 0, sizeof(struct scsi_idlun));
	return 0;
}

__COLD_ATTR__ FN(ioc_probe_host, void *, ptr)
	size_t len;
	char *path;
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	path = KERNEL$HANDLE_PATH(h);
	if (__unlikely(!path))
		return -1;
	len = strlen(path) + 1;
	if (__unlikely(len >= *(unsigned *)ptr))
		len = *(unsigned *)ptr;
	memcpy(ptr, path, len);
	return 1;
}

__COLD_ATTR__ FN(ioc_get_bus_number, int *, bus)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	/* It's none of the caller's business */
	*bus = 0;
	return 0;
}

__COLD_ATTR__ FN(ioc_get_version_num, int *, vers)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	*vers = SG_VERSION;
	return 0;
}

__COLD_ATTR__ FN(ioc_reserved_size, int *, res)
	long size = scsi_get_param(h, PARAM_BIO_SCSI_PASS_THROUGH_MAX_SIZE);
	if (__unlikely(size < 0)) return size;
	if (io == SG_SET_RESERVED_SIZE) {
		if (__unlikely(*res > size)) {
			errno = ERANGE;
			return -1;
		}
		return 0;
	} else if (__likely(io == SG_GET_RESERVED_SIZE)) {
		*res = size;
		return 0;
	} else KERNEL$SUICIDE("ioc_reserved_size: INVALID IOCTL %ld", io);
}

__COLD_ATTR__ FN(ioc_get_sg_tablesize, int *, tbls)
	/* I don't really know what to return here */
	long size = scsi_get_param(h, PARAM_BIO_SCSI_PASS_THROUGH_MAX_SIZE);
	if (__unlikely(size < 0)) return size;
	size >>= __PAGE_CLUSTER_BITS;
	if (__unlikely(!size)) size = 1;
	*tbls = size;
	return 0;
}

__COLD_ATTR__ FN(ioc_scsi_reset, int *, res)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	switch (*res) {
		case SG_SCSI_RESET_NOTHING:
			return 0;
		case SG_SCSI_RESET_DEVICE:
/* maybe we could pass it down to ATAPI/USBMSD reset one day ... */
			return 0;
		case SG_SCSI_RESET_BUS:
		case SG_SCSI_RESET_HOST:
/* I don't want userspace process to cause system-wide malfunction */
			errno = EPERM;
			return -1;
		default:
			errno = EINVAL;
			return -1;
	}
}

__COLD_ATTR__ FN(ioc_emulated_host, int *, emu)
	int p;
	if (__unlikely((p = scsi_get_param(h, PARAM_BIO_SCSI_PASS_THROUGH_EMULATED_HOST)) < 0)) return p;
	*emu = p;
	return 0;
}

static int transform = 0;

__COLD_ATTR__ FN(ioc_set_transform, int, trans)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	transform = !!trans;
	return 0;
}

__COLD_ATTR__ FN(ioc_get_transform, int *, trans)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	*trans = transform;
	return 0;
}

static int user_timeout = 6000;

__COLD_ATTR__ FN(ioc_set_timeout, int *, tmo)
	int p;
	if (__unlikely(*tmo < 0)) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	user_timeout = *tmo;
	return 0;
}

__COLD_ATTR__ FN0(ioc_get_timeout)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	return user_timeout;
}

__COLD_ATTR__ FN(ioc_get_pack_id, int *, p_id)
	int p;
	if (__unlikely((p = scsi_test(h)) < 0)) return p;
	*p_id = -1;
	return 0;
}

__COLD_ATTR__ FN0(ioc_get_bufsize)
	return scsi_get_param(h, PARAM_BIO_SCSI_PASS_THROUGH_MAX_SIZE);
}


static int copy_data(struct sg_io_hdr *sg, SCSI_PASS_THROUGH_HEAD *pt, size_t xfer_len, void (*memcp)(void *, void *, size_t))
{
	char *mem = (char *)(pt + 1);
	if (__likely(!sg->iovec_count)) {
		memcp(sg->dxferp, mem, xfer_len);
		return 0;
	} else {
		unsigned i;
		struct sg_iovec *iov = sg->dxferp;
		for (i = 0; i < sg->iovec_count; i++, iov++) {
			size_t l = iov->iov_len;
			if (__unlikely(l > xfer_len)) l = xfer_len;
			memcp(iov->iov_base, mem, l);
			xfer_len -= l;
			mem += l;
		}
		if (__unlikely(xfer_len)) {
			errno = EINVAL;
			return -1;
		}
		return 0;
	}
}

static void memcp_verify(void *user, void *pt, size_t len)
{
}

static void memcp_fromuser(void *user, void *pt, size_t len)
{
	memcpy(pt, user, len);
}

static void memcp_touser(void *user, void *pt, size_t len)
{
	memcpy(user, pt, len);
}

FN(ioc_sg_io, struct sg_io_hdr *, sg)
	SCSI_PASS_THROUGH_HEAD *pt;
	unsigned cmd_len;
	size_t xfer_len;
	IOCTLRQ ioc;

	if (__unlikely(sg->interface_id != SG_INTERFACE_ID_ORIG)) {
		einval:
		errno = EINVAL;
		return -1;
	}
	xfer_len = sg->dxfer_len;
	
	if (__unlikely(xfer_len + sizeof(SCSI_PASS_THROUGH_HEAD) < xfer_len))
		goto einval;
	pt = __sync_memalign(BIO_SECTOR_SIZE, sizeof(SCSI_PASS_THROUGH_HEAD) + xfer_len);
	if (__unlikely(!pt))
		return -1;
	memset(pt, 0, sizeof(SCSI_PASS_THROUGH_HEAD));
	if (sg->dxfer_direction == SG_DXFER_NONE) {
		pt->direction = 0;
		if (__unlikely(xfer_len != 0)) goto free_einval;
	} else if (__likely(sg->dxfer_direction == SG_DXFER_TO_DEV)) {
		pt->direction = PF_READ;
		if (__unlikely(!xfer_len)) goto free_einval;
	} else if (__likely(sg->dxfer_direction == SG_DXFER_FROM_DEV) || __likely(sg->dxfer_direction == SG_DXFER_TO_FROM_DEV)) {
		pt->direction = PF_WRITE;
		if (__unlikely(!xfer_len)) goto free_einval;
	} else {
		free_einval:
		errno = EINVAL;
		free_ret:
		free(pt);
		return -1;
	}
	cmd_len = sg->cmd_len;
	if (__unlikely(cmd_len > SCSI_PASS_THROUGH_CMD_LEN))
		goto free_einval;
	pt->command_len = cmd_len;
	memcpy(pt->command, sg->cmdp, cmd_len);
	pt->timeout = (__u64)sg->timeout * JIFFIES_PER_SECOND / 1000;

	if (__unlikely(copy_data(sg, pt, xfer_len, pt->direction == PF_READ ? memcp_fromuser : memcp_verify)))
		goto free_ret;

	ioc.h = h;
	ioc.ioctl = IOCTL_BIO_SCSI_PASS_THROUGH;
	ioc.param = PARAM_BIO_SCSI_PASS_THROUGH_COMMAND;
	ioc.v.ptr = (unsigned long)pt;
	ioc.v.len = sizeof(SCSI_PASS_THROUGH_HEAD) + xfer_len;
	ioc.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&ioc, KERNEL$IOCTL);

	/*__debug_printf("sg return: %ld %d %d\n", ioc.status, pt->resid, xfer_len);*/

	if (__unlikely(ioc.status < 0)) {
		if (ioc.status == -ENOOP) errno = ENOTTY;
		else errno = -ioc.status;
		goto free_ret;
	}

	if (__unlikely(pt->resid > xfer_len)) {
		errno = EPROTO;
		goto free_ret;
	}

	if (pt->direction == PF_WRITE) {
		if (__unlikely(copy_data(sg, pt, xfer_len - pt->resid, memcp_touser))) {
			goto free_ret;
		}
	}

#if 0
	if (pt->scsi_status) {
		__debug_printf("scsi status %d\n", pt->scsi_status);
		{
			int i;
			for (i = 0; i < pt->command_len; i++)
				__debug_printf("%02x ", pt->command[i]);
			__debug_printf("\n");
			__debug_printf("len: %d, dir: %d\n", xfer_len, pt->direction);
		}
	}
#endif

	sg->status = pt->scsi_status;
	sg->masked_status = pt->scsi_status >> 1;
	sg->msg_status = 0;
	sg->sb_len_wr = 0;
	sg->host_status = DID_OK;
	sg->driver_status = DRIVER_OK;
	sg->resid = pt->resid;
	sg->duration = (__u64)pt->duration * 1000 / JIFFIES_PER_SECOND;
	sg->info = 0;
	if (pt->scsi_status == SCSI_CHECK_CONDITION) {
		unsigned sense_len = sg->mx_sb_len;
		if (__unlikely(sense_len > SCSI_PASS_THROUGH_SENSE_LEN))
			sense_len = SCSI_PASS_THROUGH_SENSE_LEN;
		sg->sb_len_wr = sense_len;
		memcpy(sg->sbp, pt->sense, sense_len);
		sg->driver_status = DRIVER_SENSE;
	}
	free(pt);
	return 0;
}

static const struct ioc_entry ioc_table[] = {
	SCSI_IOCTL_GET_IDLUN,		1,	ioc_get_idlun,
	SCSI_IOCTL_PROBE_HOST,		1,	ioc_probe_host,
	SCSI_IOCTL_GET_BUS_NUMBER,	1,	ioc_get_bus_number,
	SG_GET_VERSION_NUM,		1,	ioc_get_version_num,
	SG_IO,				1,	ioc_sg_io,
	SG_SET_RESERVED_SIZE,		2,	ioc_reserved_size,
	/*SG_GET_RESERVED_SIZE,		1,	ioc_reserved_size,*/
	SG_GET_SG_TABLESIZE,		1,	ioc_get_sg_tablesize,
	SG_SCSI_RESET,			1,	ioc_scsi_reset,
	SG_EMULATED_HOST,		1,	ioc_emulated_host,
	SG_SET_TRANSFORM,		1,	ioc_set_transform,
	SG_GET_TRANSFORM,		1,	ioc_get_transform,
	SG_SET_TIMEOUT,			1,	ioc_set_timeout,
	SG_GET_TIMEOUT,			1,	ioc_get_timeout,
	SG_GET_PACK_ID,			1,	ioc_get_pack_id,
	SG_GET_BUFSIZE,			1,	ioc_get_bufsize,
	0,	0,	NULL,
};

const struct ioc_entry * const IOC$TABLE = ioc_table;

