#include <SPAD/LIBC.H>
#include <STRING.H>
#include <VALUES.H>
#include <STDLIB.H>
#include <SPAD/SLAB.H>
#include <SPAD/AC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/BIO.H>
#include <SPAD/BIO_KRNL.H>

#include <SPAD/SCSI.H>
#include <SPAD/SCSISTDS.H>

#include "SD.H"

const HANDLE_OPERATIONS SD_OPERATIONS = {
	SPL_X(SPL_ATA_SCSI),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	BIO$LOOKUP_PARTITION,	/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	BIO$INSTANTIATE_PARTITION,	/* instantiate */
	NULL,			/* leave */
	BIO$DETACH,		/* detach */
	NULL,			/* close */
	BIO$READ,		/* READ */
	BIO$WRITE,		/* WRITE */
	BIO$AREAD,		/* AREAD */
	BIO$AWRITE,		/* AWRITE */
	BIO$IOCTL,		/* IOCTL */
	SD_REQUEST,		/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static void SD_INIT_ROOT(HANDLE *h, void *p)
{
	SD *sd = p;
	BIO$ROOT_PARTITION(h, sd->p);
	h->flags = 0;
	h->flags2 = 0;
	h->op = &SD_OPERATIONS;
	BIO$SET_CACHE(sd->p, h);
}

static int SD_TRY_CONTROLLER(SD *sd, char *param, scsi_lun_t lun);
static int SD_UNLOAD(void *p, void **release, const char * const argv[]);
/*static int SD_DCTL(void *p, void **release, const char * const argv[]);*/

static int SD_DUMMY_DEQUEUE(SCSI_ATTACH_PARAM *ap)
{
	return BIO_DEQUEUE_EMPTY;
}

static BIORQ *SD_DUMMY_PROBE_QUEUE(SCSI_ATTACH_PARAM *ap)
{
	return NULL;
}

int main(int argc, const char * const argv[])
{
	int r, rr;
	int i;
	const char * const *arg;
	int state;
	char *ctrl, *param;
	int lun_set = 0;
	__u64 lun_user;
	scsi_lun_t lun_internal;
	int errorlevel = -1;
	int errorlevel_set = 0;
	SD *sd;
	union {
		MALLOC_REQUEST mrq;
		LOGICAL_LIST_REQUEST llr;
	} u;
	const char *opt, *optend, *val;
	static const struct __param_table params[19] = {
		"CTRL", __PARAM_STRING, 1, __MAX_STR_LEN,
		"LUN", __PARAM_BOOL_X, ~0, 1,
		"LUN", __PARAM_UNSIGNED_INT64, 0, 0,
		"LOG_ALL_ERRORS", __PARAM_BOOL, ~0, 2,
		"LOG_HW_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_NO_ERRORS", __PARAM_BOOL, ~0, 0,
		"TIMEOUT", __PARAM_INT, 1, MAXINT / JIFFIES_PER_SECOND,
		"EXTENDED_TIMEOUT", __PARAM_INT, 1, MAXINT / JIFFIES_PER_SECOND,
		"PROBE_TIMEOUT", __PARAM_INT, 1, MAXINT / JIFFIES_PER_SECOND,
		"RCACHE", __PARAM_BOOL_X, USER_FLAGS_RCACHE_SET, USER_FLAGS_RCACHE_SET,
		"RCACHE", __PARAM_BOOL, USER_FLAGS_RCACHE, USER_FLAGS_RCACHE,
		"NORCACHE", __PARAM_BOOL_X, USER_FLAGS_RCACHE_SET, USER_FLAGS_RCACHE_SET,
		"NORCACHE", __PARAM_BOOL, USER_FLAGS_RCACHE, 0,
		"WCACHE", __PARAM_BOOL_X, USER_FLAGS_WCACHE_SET, USER_FLAGS_WCACHE_SET,
		"WCACHE", __PARAM_BOOL, USER_FLAGS_WCACHE, USER_FLAGS_WCACHE,
		"NOWCACHE", __PARAM_BOOL_X, USER_FLAGS_WCACHE_SET, USER_FLAGS_WCACHE_SET,
		"NOWCACHE", __PARAM_BOOL, USER_FLAGS_WCACHE, 0,
		"MAX_TRANSFER", __PARAM_UNSIGNED_INT, 512, MAXUINT,
		NULL, 0, 0, 0,
	};
	void *vars[19];
	u.mrq.size = sizeof(SD);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err0;
	}
	sd = u.mrq.ptr;
	memset(sd, 0, sizeof(SD));

	sd->user_flags = USER_FLAGS_RCACHE | USER_FLAGS_WCACHE;
	sd->user_max_transfer = MAXUINT;

	ctrl = NULL;
	param = NULL;

	u.mrq.size = SD_MAX_SECSIZE;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err1;
	}
	sd->pad_ptr = u.mrq.ptr;

	u.mrq.size = SD_MAX_SECSIZE;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err1;
	}
	sd->write_buffer = u.mrq.ptr;

	sd->media_thread.thread = KERNEL$ALLOC_THREAD_SYNC();
	if (__unlikely(__IS_ERR(sd->media_thread.thread))) {
		r = __PTR_ERR(sd->media_thread.thread);
		sd->media_thread.thread = NULL;
		goto err1;
	}

	vars[0] = &ctrl;
	vars[1] = &lun_set;
	vars[2] = &lun_user;
	vars[3] = &errorlevel;
	vars[4] = &errorlevel;
	vars[5] = &errorlevel;
	vars[6] = &sd->timeout;
	vars[7] = &sd->extended_timeout;
	vars[8] = &sd->probe_timeout;
	vars[9] = &sd->user_flags;
	vars[10] = &sd->user_flags;
	vars[11] = &sd->user_flags;
	vars[12] = &sd->user_flags;
	vars[13] = &sd->user_flags;
	vars[14] = &sd->user_flags;
	vars[15] = &sd->user_flags;
	vars[16] = &sd->user_flags;
	vars[17] = &sd->user_max_transfer;
	vars[18] = NULL;

	arg = argv;
	state = 0;
	while (__unlikely(__parse_params(&arg, &state, params, vars, &opt, &optend, &val))) {
		if (opt) {
			r = __accumulate_params(&param, opt, optend, val, val ? strchr(val, 0) : NULL);
			if (__unlikely(r)) {
				if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-r));
				goto err1;
			}
			continue;
		}
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: SYNTAX ERROR");
		r = -EBADSYN;
		goto err1;
	}
	if (__unlikely(lun_set)) {
		lun_internal = SCSI$LUN_CONVERT_64_TO_INTERNAL(lun_user);
		if (lun_internal == LUN_NONE)
			goto bads;
	} else {
		lun_internal = LUN_NONE;
	}
	if (errorlevel == -1) errorlevel = 2;
	else errorlevel_set = 1;

	sd->timeout *= JIFFIES_PER_SECOND;
	sd->extended_timeout *= JIFFIES_PER_SECOND;
	sd->probe_timeout *= JIFFIES_PER_SECOND;

	sd->errorlevel = errorlevel;
	sd->errorlevel_set = errorlevel_set;
	INIT_LIST(&sd->waiting_list);
	sd->waiting_tags = 0;
	INIT_XLIST(&sd->in_progress);
	INIT_TIMER(&sd->timer);
	sd->timer.fn = SD_TIMER_FN;
	INIT_TIMER(&sd->unchoke_tags);
	sd->unchoke_tags.fn = SD_UNCHOKE_TAGS;
	SET_TIMER_NEVER(&sd->unchoke_tags);
	WQ_INIT(&sd->pass_through_wait, "SD$PASS_THROUGH_WAIT");
	INIT_TIMER(&sd->pass_through_timer);
	sd->pass_through_timer.fn = SCSI_PASS_THROUGH_TIMER_FN;
	WQ_INIT(&sd->media_wq, "SD$MEDIA_WQ");
	WQ_INIT(&sd->media_wq_noerror, "SD$MEDIA_WQ_NOERROR");

	if (__unlikely(r = BIOQUE$ALLOC_QUEUE(&sd->q))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: CAN'T ALLOC QUEUE: %s", strerror(-r));
		goto err1;
	}

	KERNEL$ERROR_MSG()[0] = 0;
	if (ctrl) {
		strcpy(sd->ctrl, ctrl);
		if (__unlikely(__check_logical_name(sd->ctrl, 1))) goto bads;
		r = SD_TRY_CONTROLLER(sd, param, lun_internal);
		if (__unlikely(r)) {
			goto nodev;
		}
	} else {
		u.llr.prefix = "";
		SYNC_IO_CANCELABLE(&u.llr, KERNEL$LIST_LOGICALS);
		if (__unlikely(u.llr.status < 0)) {
			if (u.llr.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: CAN'T LIST LOGICAL NAMES: %s", strerror(-u.llr.status));
			r = u.llr.status;
			goto err2;
		}
		for (i = 0; i < u.llr.n_entries; i++) {
			char *n = u.llr.entries[i]->name;
			if (_memcasecmp(n, "ATA$", 4) && _memcasecmp(n, "SCSI$", 5)) continue;
			strlcpy(sd->ctrl, n, __MAX_STR_LEN);
			/* __debug_printf("try: %s\n", sd->ctrl); */
			r = SD_TRY_CONTROLLER(sd, param, lun_internal);
			if (!r) {
				KERNEL$FREE_LOGICAL_LIST(&u.llr);
				goto got_it;
			}
		}
		KERNEL$FREE_LOGICAL_LIST(&u.llr);
		r = -ENODEV;
		nodev:
		if (!KERNEL$ERROR_MSG()[0]) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: NO DEVICE FOUND");
		goto err2;
	}

	got_it:

	SCSI$GET_MODEL(&sd->inq, &sd->vendor, &sd->product, &sd->revision);

		/* the following line is confusing ... RBC specification says
		   that the device should support only MODE_SENSE_6 and not need
		   support MODE_SENSE_10.
		   However Windows do exactly the opposite: when the device has
		   something other than transparent SCSI, it uses only
		   MODE_SENSE_10. In that case USBMSD driver have already
		   disabled SCSI_FLAG_CAN_USE_MODE_6.
		*/
	if (sd->std == STD_RBC && __likely(sd->ap.scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) sd->ap.scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;
	if (__unlikely(sd->scsi_version == 1) && __likely(sd->ap.scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) sd->ap.scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;

	if (__unlikely(!_strcasecmp(sd->vendor, "MAXTOR"))) {
		/* Maxtor Atlas 15K2 locks up occasionally under heavy load
		   when larger write requests are used.
		   1152KB is OK. 1280KB and more is failing. */
		if (__likely(sd->user_max_transfer == MAXUINT))
			sd->user_max_transfer = 1048576;

		/*if (!(sd->user_flags & USER_FLAGS_WCACHE_SET))
			sd->user_flags &= ~USER_FLAGS_WCACHE;*/
	}

	if (SD_REMOVABLE_MEDIA(sd)) {
		if (!errorlevel_set) sd->errorlevel = errorlevel = 1;
		sd->flags |= FLAGS_NO_MEDIA;
#if 0
		if (STD_IS_DISK(sd->std)) {
/* USB sticks do not really have removable media, but they report so. */
/* But don't do this, because it slows down USB ZIP driver loading. */
			if (!SD_MEDIA_THREAD(sd)) sd->flags &= ~FLAGS_NO_MEDIA;
		}
#endif
	} else {
		r = SD_MEDIA_THREAD(sd);
		if (__unlikely(r)) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO ACCESS MEDIA: %s", sd->dev_name, strerror(-r));
			goto err3;
		}
		sd->flags &= ~FLAGS_NO_MEDIA;
	}

	if (__unlikely(!sd->ap.max_untagged_requests))
		KERNEL$SUICIDE("SD: ZERO FREE TAGS REPORTED BY SCSI HOST CONTROLLER DRIVER");
	sd->tagging = SCSI_TAGGING_SIMPLE;
	if (__unlikely(sd->scsi_version == 1) || __unlikely(sd->std == STD_SCSI_1) || sd->std == STD_RBC) sd->tagging = SCSI_TAGGING_NONE;
	if ((sd->scsi_version == 0 || sd->scsi_version == 2 || sd->scsi_version == 3) && !(sd->inq.flags3 & INQ_FLAGS3_CMDQUE)) sd->tagging = SCSI_TAGGING_NONE;
	if (__unlikely(sd->scsi_version >= 4) && !(sd->inq.flags3 & INQ_FLAGS3_CMDQUE) && __unlikely(!(sd->inq.flags2 & INQ_FLAGS2_BQUE))) sd->tagging = SCSI_TAGGING_NONE;
	if (__likely(!sd->ap.max_tagged_requests)) sd->tagging = SCSI_TAGGING_NONE;

	if (__likely(sd->tagging == SCSI_TAGGING_NONE)) sd->controller_tags = sd->ap.max_untagged_requests;
	else sd->controller_tags = sd->ap.max_tagged_requests;

	sd->limit_tags = sd->controller_tags;
	sd->used_tags = 0;

	if (__unlikely(r = BIO$ALLOC_PARTITIONS(&sd->p, sd, SPL_X(SPL_ATA_SCSI), sd->flags & FLAGS_NO_MEDIA ? (__sec_t)-1 : sd->capacity, SD_IOCTL, sd->dev_name))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC PARTITIONS: %s", sd->dev_name, strerror(-r));
		goto err3;
	}

	KERNEL$SLAB_INIT(&sd->tag_slab, sizeof(SD_TAG), sizeof(__sec_t), VM_TYPE_WIRED_MAPPED, SD_TAG_CTOR, NULL, "SD$SD_TAG");

	if (__unlikely(r = KERNEL$SLAB_RESERVE(&sd->tag_slab, 2))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", sd->dev_name, strerror(-r));
		goto err5;
	}
	sd->dummy_tag = __slalloc(&sd->tag_slab);
	if (__unlikely(!sd->dummy_tag))
		KERNEL$SUICIDE("SD: CAN'T ALLOCATE DUMMY TAG");

	sd->pass_through_cmd.ch_id = sd->ap.ch_id;
	sd->pass_through_cmd.lun = sd->ap.lun;
	sd->pass_through_cmd.adapter = sd->ap.adapter;
	sd->pass_through_cmd.done = SCSI_PASS_THROUGH_DONE;
	sd->pass_through_cmd.sense = sd->pass_through_sense;
	sd->pass_through_cmd.sense_size = SCSI_PASS_THROUGH_SENSE_LEN;
	sd->pass_through_cmd.tagging = SCSI_TAGGING_NONE;

	_printf("%s: SCSI %s%s%s%s%s%s%s\n", sd->dev_name, sd->std < 0 ? "CD-ROM" : SD_REMOVABLE_MEDIA(sd) ? "REMOVABLE DISK" : "DISK", *sd->vendor || *sd->product || *sd->revision ? " " : "", sd->vendor, *sd->vendor && *sd->product ? "," : "", sd->product, *sd->revision && (*sd->vendor || *sd->product) ? "," : "", sd->revision);
	_printf("%s: ", sd->dev_name);
	/*{
		int i;
		for (i = 0; sd->vendor[i]; i++) __debug_printf("%d: %02x\n", i, sd->vendor[i]);
		for (i = 0; sd->product[i]; i++) __debug_printf("%d: %02x\n", i, sd->product[i]);
		for (i = 0; sd->revision[i]; i++) __debug_printf("%d: %02x\n", i, sd->revision[i]);
	}*/
	if (sd->std == -1) {
		_printf("MMC SCSI 2");
	} else if (sd->std <= -2) {
		_printf("MMC %d", STD_MMC_1 - sd->std + 1);
	} else if (sd->std <= STD_SCSI_2) {
		_printf("SCSI %d", sd->std - STD_SCSI_1 + 1);
	} else if (sd->std == STD_RBC) {
		_printf("RBC");
	} else {
		_printf("SBC %d", sd->std - STD_SBC_1 + 1);
	}
	if (!(sd->flags & FLAGS_NO_MEDIA)) {
		_printf(", ");
		if (sd->capacity < 2097152)
			_printf("%ld MiB", (long)(sd->capacity >> 11));
		else if (sd->capacity < (__u64)2097152 * 1024)
			_printf("%ld.%ld GiB", (long)(sd->capacity >> 21), ((long)sd->capacity & 2097151) / 209716);
		else
			_printf("%ld.%ld TiB", (long)(sd->capacity >> 31), (long)((sd->capacity >> 10) & 2097151) / 209716);
	}
	_printf("\n");

	sd->ap.dequeue = SD_DEQUEUE;
	sd->ap.probe_queue = SD_PROBE_QUEUE;

	r = KERNEL$REGISTER_DEVICE(sd->dev_name, "SD.SYS", 0, sd, SD_INIT_ROOT, NULL, NULL, NULL, SD_UNLOAD, &sd->lnte, sd->ctrl, NULL);
	if (__unlikely(r < 0)) {
		if (__unlikely(r != -EINTR)) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", sd->dev_name, strerror(-r));
		goto err6;
	}
	sd->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), sd->dev_name);
	free(param);
	return 0;

	err6:
	__slow_slfree(sd->dummy_tag);
	err5:
	KERNEL$DEL_TIMER(&sd->unchoke_tags);
	KERNEL$SLAB_DESTROY(&sd->tag_slab);
	BIO$FREE_PARTITIONS(sd->p);
	err3:
	if (__unlikely(rr = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_FREE, &sd->ap)))
		KERNEL$SUICIDE("SD: CAN'T UNREGISTER AT CONTROLLER %s, DEVICE %x, LUN %"__64_format"X: %s", sd->ctrl, sd->ap.ch_id, (__u64)sd->ap.lun, strerror(-rr));
	err2:
	BIOQUE$FREE_QUEUE(sd->q);
	err1:
	free(param);
	free(sd->write_buffer);
	if (sd->media_thread.thread) KERNEL$FREE_THREAD(sd->media_thread.thread);
	free(sd->pad_ptr);
	free(sd);
	err0:
	return r;
}

static int SD_TRY_CONTROLLER(SD *sd, char *param, scsi_lun_t lun)
{
	int r;
	unsigned type;

	memset(&sd->ap, 0, sizeof sd->ap);

	try_another:
	sd->ap.u.p.version = SCSI_VERSION;
	sd->ap.u.p.param = param;
	sd->ap.dev_prefix = "BIO$SD";
	sd->ap.probe_timeout = sd->probe_timeout;
	sd->ap.timeout = sd->timeout;
	sd->ap.dev_name = &sd->dev_name;
	sd->ap.dequeue = SD_DUMMY_DEQUEUE;
	sd->ap.probe_queue = SD_DUMMY_PROBE_QUEUE;
	sd->ap.biosched = BIOQUE$IOSCHED(sd->q);

	/*__debug_printf("attaching...\n");*/
	if (__unlikely(r = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_GET, &sd->ap))) {
		if (sd->ap.msg) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", sd->ap.msg);
		else if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: CAN'T ATTACH TO HOST CONTROLLER DRIVER: %s", strerror(-r));
		return r;
	}

	try_another_lun:
	if (SCSI$FIND_LUN(sd->ctrl, sd->dev_name, !sd->errorlevel_set ? 1 : sd->errorlevel, &sd->ap, lun, sd->probe_timeout, sd->extended_timeout, &sd->lun_byte)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T FIND LUN", sd->dev_name);
		goto try_another_id;
	}

	/*__debug_printf("attached %x\n", sd->ap.ch_id_lun);*/

	if (SCSI$INQUIRY(sd->dev_name, !sd->errorlevel_set ? 1 : sd->errorlevel, &sd->ap, LUN_NONE, sd->lun_byte, &sd->inq, sizeof sd->inq, sd->probe_timeout, sd->extended_timeout)) {
		/*__debug_printf("not me\n");*/
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INQUIRY FAILED", sd->dev_name);
		goto try_another_lun;
	}

	if (__unlikely((sd->inq.type & INQ_TYPE_QUAL) != INQ_TYPE_QUAL_CONNECTED) &&
	    __likely((sd->inq.type & INQ_TYPE_QUAL) != INQ_TYPE_QUAL_DISCONNECTED)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID LUN", sd->dev_name);
		goto try_another_lun;
	}

	/*__debug_printf("ok\n");*/
	/*{
		int i;
		__debug_printf("inquired: %.8s, %.16s, %.4s", sd->inq.vendor, sd->inq.product, sd->inq.revision);
		for (i = 0; i < sizeof(SCSI_INQUIRY_DATA); i++) {
			if (!(i & 15)) __debug_printf("\n%2d: ", i);
			__debug_printf(" %02X", ((unsigned char *)&sd->inq)[i]);
		}
		__debug_printf("\n");
	}*/

	sd->scsi_version = SCSI$VERSION(&sd->inq);

	type = sd->inq.type & INQ_TYPE_TYPE;

	if (!sd->timeout) sd->timeout = TIMEOUT_JIFFIES(__unlikely(type == SCSI_TYPE_OPTICAL_DA) ? SD_OPTICAL_TIMEOUT : SD_TIMEOUT);
	if (!sd->extended_timeout) {
		sd->extended_timeout = TIMEOUT_JIFFIES(SD_EXTENDED_TIMEOUT);
		if (sd->extended_timeout < sd->timeout)
			sd->extended_timeout = sd->timeout;
	}
	
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_SBC_2, SCSI_STD_SBC_2_ANSI_INCITS_405_2005, SCSI_STD_SBC_2_T10_1417_D_revision_16, SCSI_STD_SBC_2_T10_1417_D_revision_5a, SCSI_STD_SBC_2_T10_1417_D_revision_15, 0)) {
		sd->std = STD_SBC_2;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_SBC, SCSI_STD_SBC_ANSI_INCITS_306_1998, SCSI_STD_SBC_T10_0996_D_revision_08c, 0)) {
		sd->std = STD_SBC_1;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_RBC, SCSI_STD_RBC_ANSI_INCITS_330_2000, SCSI_STD_RBC_T10_1240_D_revision_10a, 0)) {
		sd->std = STD_RBC;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_5, 0)) {
		sd->std = STD_MMC_5;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_4, SCSI_STD_MMC_4_ANSI_INCITS_401_2005, SCSI_STD_MMC_4_T10_1545_D_revision_5, SCSI_STD_MMC_4_T10_1545_D_revision_3, SCSI_STD_MMC_4_T10_1545_D_revision_3d, 0)) {
		sd->std = STD_MMC_4;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_3, SCSI_STD_MMC_3_ANSI_INCITS_360_2002, SCSI_STD_MMC_3_T10_1363_D_revision_10g, SCSI_STD_MMC_3_T10_1363_D_revision_9, 0)) {
		sd->std = STD_MMC_3;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_2, SCSI_STD_MMC_2_ANSI_INCITS_333_2000, SCSI_STD_MMC_2_T10_1228_D_revision_11a, SCSI_STD_MMC_2_T10_1228_D_revision_11, 0)) {
		sd->std = STD_MMC_2;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC, SCSI_STD_MMC_ANSI_INCITS_304_1997, SCSI_STD_MMC_T10_1048_D_revision_10a, 0)) {
		sd->std = STD_MMC_1;
		goto mine;
	}

	if (__likely(type == SCSI_TYPE_DA) || __unlikely(type == SCSI_TYPE_WORM) || __unlikely(type == SCSI_TYPE_OPTICAL_DA)) {
		if ((sd->inq.version & 7) == 1) sd->std = STD_SCSI_1;
		else if ((sd->inq.version & 7) == 2) sd->std = STD_SCSI_2;
		else if ((sd->inq.version & 7) == 3) sd->std = STD_SBC_1;
		else if (sd->inq.version == 4) sd->std = STD_SBC_1;
		else if (sd->inq.version >= 5 && sd->inq.version <= 7) sd->std = type == SCSI_TYPE_DA ? STD_SBC_2 : STD_SBC_1;
		else sd->std = STD_SCSI_2;	/* guess ... :-/ */
		goto mine;
	}
	if (__likely(type == SCSI_TYPE_REDUCED_DA)) {
		sd->std = STD_RBC;
		goto mine;
	}
	if (__likely(type == SCSI_TYPE_CD_DVD)) {
		if ((sd->inq.version & 7) < 3) sd->std = STD_MMC_SCSI_2;
		else sd->std = STD_MMC_5;
		goto mine;
	}
	try_another_id:
	if (__unlikely(r = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_FREE, &sd->ap)))
		KERNEL$SUICIDE("SD: CAN'T UNREGISTER AT CONTROLLER %s, DEVICE %x, LUN %"__64_format"X: %s", sd->ctrl, sd->ap.ch_id, (__u64)sd->ap.lun, strerror(-r));
	goto try_another;

	mine:
	if (sd->std < 0) {
		if (!memcmp(sd->dev_name, "BIO$SD", 6))
			memcpy(sd->dev_name, "BIO$CD", 6);
	}
	if (!sd->probe_timeout && sd->ap.probe_timeout) sd->probe_timeout = sd->ap.probe_timeout;
	return 0;
}

static int SD_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r, rr;
	SD *sd = p;
	BIO$DISABLE_CACHE(sd->p);
	if ((r = KERNEL$DEVICE_UNLOAD(sd->lnte, argv))) return r;
	RAISE_SPL(SPL_ATA_SCSI);
	__slow_slfree(sd->dummy_tag);
	while (__unlikely(!BIOQUE$QUEUE_EMPTY(sd->q)) || __unlikely(!KERNEL$SLAB_EMPTY(&sd->tag_slab)) || __unlikely(sd->pass_through != NULL) || __unlikely(sd->flags & FLAGS_MEDIA_PROBING)) {
		if (sd->flags & FLAGS_MEDIA_PROBING) KERNEL$CIO((IORQ *)(void *)&sd->media_thread);
		if (sd->pass_through_posted) sd->ap.cancel((SCSIRQ *)(void *)&sd->pass_through_cmd);
		KERNEL$SLEEP(1);
	}
	LOWER_SPL(SPL_ZERO);
	if (!(sd->flags & (FLAGS_NO_MEDIA | FLAGS_NO_SYNCHRONIZE_CACHE | FLAGS_WP))) {
		DECL_SCSIRQ(16) cmd;
		SCSI_SENSE_DATA sense;
		cmd.sense = (__u8 *)&sense;
		cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
		cmd.direction = 0;
		cmd.tagging = SCSI_TAGGING_NONE;
		if (__likely(!(sd->flags & FLAGS_COMMAND_SIZE_16))) {
			cmd.cmdlen = 10;
			cmd.cmd[0] = SCMD_SYNCHRONIZE_CACHE_10;
			memset(&cmd.cmd[2], 0, 8);
		} else {
			cmd.cmdlen = 16;
			cmd.cmd[0] = SCMD_SYNCHRONIZE_CACHE_16;
			memset(&cmd.cmd[2], 0, 14);
		}
		cmd.cmd[1] = sd->lun_byte;
		SCSI$SYNC_RQ(sd->dev_name, sd->errorlevel, (SCSIRQ *)(void *)&cmd, &sd->ap, LUN_NONE, NULL, 0, sd->timeout, sd->extended_timeout, SD_RETRIES, SCSI$CMD_RECOMMENDED_ACTION);
		if (__unlikely(SCSI$CMD_RECOMMENDED_ACTION((SCSIRQ *)(void *)&cmd) != SCSI_ACTION_OK) && SD_MEDIA_CHANGE_ACTION((SCSIRQ *)(void *)&cmd) < 0) {
			__u32 key = SCSI$GET_CMD_SENSE((SCSIRQ *)(void *)&cmd);
			if (!(!SCSI_SENSE_UNKNOWN(key) && SCSI_SENSE_KEY(key) == SCSI_KEY_ILLEGAL_REQUEST && __likely(SCSI_SENSE_ASC(key) == 0x20) && SCSI_SENSE_ASCQ(key) == 0x00)) {
				SCSI$LOG_ERROR(sd->dev_name, sd->errorlevel, &sd->inq, sd->scsi_version, (SCSIRQ *)(void *)&cmd, "");
			}
		}
	}
	if (__unlikely(sd->used_tags | sd->waiting_tags))
		KERNEL$SUICIDE("SD_UNLOAD: TAGS LEAKED: used_tags %u, waiting_tags %u", (unsigned)sd->used_tags, (unsigned)sd->waiting_tags);
	KERNEL$DEL_TIMER(&sd->unchoke_tags);
	KERNEL$SLAB_DESTROY(&sd->tag_slab);
	BIO$FREE_PARTITIONS(sd->p);
	if (__unlikely(rr = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_FREE, &sd->ap)))
		KERNEL$SUICIDE("SD_UNLOAD: CAN'T UNREGISTER AT CONTROLLER %s, DEVICE %x, LUN %"__64_format"X: %s", sd->ctrl, sd->ap.ch_id, (__u64)sd->ap.lun, strerror(-rr));
	BIOQUE$FREE_QUEUE(sd->q);
	*release = sd->dlrq;
	if (__unlikely(!LIST_EMPTY(&sd->waiting_list)))
		KERNEL$SUICIDE("SD_UNLOAD: WAITING_LIST CORRUPTED");
	if (__unlikely(!XLIST_EMPTY(&sd->in_progress)))
		KERNEL$SUICIDE("SD_UNLOAD: IN_PROGRESS LIST CORRUPTED");
	WQ_WAKE_ALL(&sd->media_wq);
	WQ_WAKE_ALL(&sd->media_wq_noerror);
	WQ_WAKE_ALL(&sd->pass_through_wait);
	free(sd->write_buffer);
	KERNEL$FREE_THREAD(sd->media_thread.thread);
	free(sd->pad_ptr);
	free(sd);
	return 0;
}

#if 0
static int SD_DCTL(void *p, void **release, const char * const argv[])
{
	SD *sd = p;
	RAISE_SPL(SPL_ATA_SCSI);
	__critical_printf("sd: used tags %u, limit tags %u, controller tags %u, waiting tags %u, queue empty %d\n", sd->used_tags, sd->limit_tags, sd->controller_tags, sd->waiting_tags, BIOQUE$QUEUE_EMPTY(sd->q));
	LOWER_SPL(SPL_ZERO);
	return 0;
}
#endif
