#include <SPAD/DEV_KRNL.H>
#include <SPAD/BIO.H>
#include <SPAD/IOCTL.H>

#include "SD.H"

void SCSI_QUEUE_PASS_THROUGH(SD *sd, IOCTLRQ *rq)
{
	VDESC vdesc;
	VPHYS vphys;

	switch (rq->param) {
		case PARAM_BIO_SCSI_PASS_THROUGH_COMMAND:
			goto do_pass_through_command;
		case PARAM_BIO_SCSI_PASS_THROUGH_MAX_SIZE:
			rq->status = sd->ap.max_sectors << BIO_SECTOR_SIZE_BITS;
			call_ret:
			CALL_AST(rq);
			return;
		case PARAM_BIO_SCSI_PASS_THROUGH_EMULATED_HOST:
			rq->status = (sd->ap.scsi_flags & (SCSI_FLAG_CAN_USE_MODE_6 | SCSI_FLAG_CAN_USE_MODE_10)) == SCSI_FLAG_CAN_USE_MODE_10;
			goto call_ret;
		default:
			rq->status = -ENOOP;
			goto call_ret;
	}
	
	do_pass_through_command:

	if (__unlikely(rq->v.len < sizeof(SCSI_PASS_THROUGH_HEAD))) {
		einval_ret:
		rq->status = -EINVAL;
		goto call_ret;
	}

	if (__unlikely((unsigned long)rq->v.ptr & (BIO_SECTOR_SIZE - 1)))
		goto einval_ret;

	if (__unlikely((rq->v.len - sizeof(SCSI_PASS_THROUGH_HEAD) + BIO_SECTOR_SIZE - 1) >> BIO_SECTOR_SIZE_BITS > sd->ap.max_sectors)) {
		rq->status = -ERANGE;
		goto call_ret;
	}
	if (__unlikely(sd->flags & FLAGS_MEDIA_PROBING)) {
		WQ_WAIT_F(&sd->media_wq_noerror, rq);
		return;
	}
	if (__unlikely(sd->pass_through != NULL)) {
		WQ_WAIT_F(&sd->pass_through_wait, rq);
		return;
	}

	vdesc.ptr = rq->v.ptr;
	vdesc.len = sizeof(SCSI_PASS_THROUGH_HEAD);
	vdesc.vspace = rq->v.vspace;
	vphys.spl = SPL_X(SPL_ATA_SCSI);
	RAISE_SPL(SPL_VSPACE);
	sd->pass_through_physunlock = rq->v.vspace->op->vspace_physlock(&vdesc, PF_RW, &vphys);
	if (!sd->pass_through_physunlock) {
		DO_PAGEIN_NORET(rq, &rq->v, PF_RW);
		return;
	}
	sd->pass_through_phys = vphys.ptr;
	if (__unlikely(vphys.len != sizeof(SCSI_PASS_THROUGH_HEAD))) {
		sd->pass_through_physunlock(sd->pass_through_phys);
		goto einval_ret;
	}

	sd->pass_through = rq;
	SD_DEQUEUE_LOOP(sd);
}

#define XFORM_MODE_SENSE_6_TO_10	1
#define XFORM_MODE_SELECT_6_TO_10	2

int SCSI_PROCESS_PASS_THROUGH(SD *sd)
{
	SCSI_PASS_THROUGH_HEAD *pth;
	unsigned timeout;
	int r;
	if (__unlikely(sd->pass_through_posted))
		return 0;
	if (__unlikely(sd->used_tags))
		return 0;

	pth = KERNEL$MAP_PHYSICAL_BANK(sd->pass_through_phys);

	sd->pass_through_cmd.direction = pth->direction;
	if (!sd->pass_through_cmd.direction) {
		if (__unlikely(sd->pass_through->v.len != sizeof(SCSI_PASS_THROUGH_HEAD)))
			goto einval_unmap_ret;
	} else if (sd->pass_through_cmd.direction == PF_READ ||
		   __likely(sd->pass_through_cmd.direction == PF_WRITE)) {
		if (__unlikely(sd->pass_through->v.len == sizeof(SCSI_PASS_THROUGH_HEAD)))
			goto einval_unmap_ret;
	} else goto einval_unmap_ret;

	sd->pass_through_cmd.cmdlen = pth->command_len;
	if (__unlikely(sd->pass_through_cmd.cmdlen > SCSI_PASS_THROUGH_CMD_LEN) || __unlikely(sd->pass_through_cmd.cmdlen < SCSI_MIN_CMD_LEN)) {
		einval_unmap_ret:
		r = -EINVAL;
		KERNEL$UNMAP_PHYSICAL_BANK(pth);
		goto post_r_ret;
	}
	memcpy(sd->pass_through_cmd.cmd, pth->command, sd->pass_through_cmd.cmdlen);
	if (sd->scsi_version < 3) {
		sd->pass_through_cmd.cmd[1] &= ~0xe0;
		sd->pass_through_cmd.cmd[1] |= sd->lun_byte;
	}
	if (!sd->pass_through_cmd.direction) {
		sd->pass_through_cmd.desc = NULL;
	} else {
		sd->pass_through_cmd.desc = &sd->pass_through_desc;
		sd->pass_through_desc.v.ptr = sd->pass_through->v.ptr + sizeof(SCSI_PASS_THROUGH_HEAD);
		sd->pass_through_desc.v.len = sd->pass_through->v.len - sizeof(SCSI_PASS_THROUGH_HEAD);
		sd->pass_through_desc.v.vspace = sd->pass_through->v.vspace;
		sd->pass_through_desc.next = NULL;
	}
	sd->pass_through_cmd.internal = 0;
	timeout = pth->timeout;
	if (!timeout) timeout = sd->timeout;
	else if (timeout > SD_MAX_TIMEOUT) timeout = SD_MAX_TIMEOUT;

	/* patch the command */

	sd->pass_through_xform = 0;

	if (__unlikely(sd->pass_through_cmd.cmd[0] == SCMD_INQUIRY)) {
		if (__likely(sd->ap.scsi_flags & SCSI_FLAG_SHORT_INQUIRY)) {
			if (sd->pass_through_cmd.cmd[4] > SCSI_SHORT_INQUIRY_SIZE) {
				sd->pass_through_cmd.cmd[4] = SCSI_SHORT_INQUIRY_SIZE;
				if (__likely(sd->pass_through_desc.v.len > SCSI_SHORT_INQUIRY_SIZE))
					sd->pass_through_desc.v.len = SCSI_SHORT_INQUIRY_SIZE;
			}
		}
	}
	if (__unlikely(sd->pass_through_cmd.cmd[0] == SCMD_MODE_SENSE_6) && __likely(sd->pass_through_cmd.cmdlen == 6) && (sd->ap.scsi_flags & (SCSI_FLAG_CAN_USE_MODE_6 | SCSI_FLAG_CAN_USE_MODE_10)) == SCSI_FLAG_CAN_USE_MODE_10 && sd->pass_through_desc.v.len >= 4) {
		unsigned len = sd->pass_through_cmd.cmd[4] + 4;
		sd->pass_through_cmd.cmd[0] = SCMD_MODE_SENSE_10;
		sd->pass_through_cmd.cmd[7] = len >> 8;
		sd->pass_through_cmd.cmd[8] = len;
		sd->pass_through_cmd.cmd[9] = sd->pass_through_cmd.cmd[5];
		sd->pass_through_cmd.cmd[3] = sd->pass_through_cmd.cmd[4] = sd->pass_through_cmd.cmd[5] = sd->pass_through_cmd.cmd[6] = 0;
		sd->pass_through_cmd.cmdlen = 10;
		sd->pass_through_desc.v.ptr -= 4;
		sd->pass_through_desc.v.len += 4;
		sd->pass_through_xform = XFORM_MODE_SENSE_6_TO_10;
	}
	if (__unlikely(sd->pass_through_cmd.cmd[0] == SCMD_MODE_SELECT_6) && __likely(sd->pass_through_cmd.cmdlen == 6) && __likely(sd->pass_through_cmd.direction == PF_READ) && (sd->ap.scsi_flags & (SCSI_FLAG_CAN_USE_MODE_6 | SCSI_FLAG_CAN_USE_MODE_10)) == SCSI_FLAG_CAN_USE_MODE_10 && sd->pass_through_desc.v.len >= 4) {
		unsigned len = sd->pass_through_cmd.cmd[4] + 4;
		sd->pass_through_cmd.cmd[0] = SCMD_MODE_SELECT_10;
		sd->pass_through_cmd.cmd[7] = len >> 8;
		sd->pass_through_cmd.cmd[8] = len;
		sd->pass_through_cmd.cmd[9] = sd->pass_through_cmd.cmd[5];
		sd->pass_through_cmd.cmd[3] = sd->pass_through_cmd.cmd[4] = sd->pass_through_cmd.cmd[5] = sd->pass_through_cmd.cmd[6] = 0;
		sd->pass_through_cmd.cmdlen = 10;
		SCSI$MODE_HEADER_6_TO_10((SCSI_MODE_SENSE_HEADER *)(char *)(pth + 1) - 4, (SCSI_MODE_SENSE_6_HEADER *)(pth + 1));
		sd->pass_through_desc.v.ptr -= 4;
		sd->pass_through_desc.v.len += 4;
		sd->pass_through_xform = XFORM_MODE_SELECT_6_TO_10;
	}

	KERNEL$UNMAP_PHYSICAL_BANK(pth);

	memset(&sd->pass_through_sense, 0, SCSI_PASS_THROUGH_SENSE_LEN);
	sd->pass_through_cmd.scsi_status = 0;

	r = sd->ap.post((SCSIRQ *)(void *)&sd->pass_through_cmd);
	if (__unlikely(r)) {
		if (r == -EAGAIN) {
			return 0;
		}
		if (r == -EVSPACEFAULT) {
			sd->pass_through_physunlock(sd->pass_through_phys);
			DO_PAGEIN_NORET(sd->pass_through, &sd->pass_through->v, PF_RW);
		} else {
			post_r_ret:
			sd->pass_through_physunlock(sd->pass_through_phys);
			sd->pass_through->status = r;
			CALL_AST(sd->pass_through);
		}
		sd->pass_through = NULL;
		WQ_WAKE_ALL(&sd->pass_through_wait);
		return 1;
	}
	sd->pass_through_posted = 1;
	sd->pass_through_posted_time = KERNEL$GET_JIFFIES();
	KERNEL$SET_TIMER(timeout, &sd->pass_through_timer);
	return 1;
}

static void SCSI_PASS_THROUGH_UNDO_XFORM(SD *sd, SCSI_PASS_THROUGH_HEAD *pth);

void SCSI_PASS_THROUGH_DONE(SCSIRQ *cmd)
{
	SCSI_PASS_THROUGH_HEAD *pth;
	SD *sd = GET_STRUCT(cmd, SD, pass_through_cmd);
	u_jiffies_t tim;

	sd->pass_through_posted = 0;
	KERNEL$DEL_TIMER(&sd->pass_through_timer);

	pth = KERNEL$MAP_PHYSICAL_BANK(sd->pass_through_phys);
	tim = KERNEL$GET_JIFFIES() - sd->pass_through_posted_time;
	if (__unlikely(tim != (__u32)tim)) tim = (__u32)-1;
	pth->duration = tim;
	if (__likely(sd->pass_through_cmd.status >= 0)) {
		int aa;
		if (__unlikely(sd->pass_through_xform)) {
			SCSI_PASS_THROUGH_UNDO_XFORM(sd, pth);
		}
		pth->resid = sd->pass_through->v.len - sizeof(SCSI_PASS_THROUGH_HEAD) - sd->pass_through_cmd.status;
		pth->scsi_status = sd->pass_through_cmd.scsi_status;
		if (__unlikely(pth->scsi_status != SCSI_GOOD)) {
			if (__likely(pth->scsi_status == SCSI_CHECK_CONDITION)) {
				memcpy(pth->sense, sd->pass_through_sense, SCSI_PASS_THROUGH_SENSE_LEN);
			}
			aa = SD_MEDIA_CHANGE_ACTION((SCSIRQ *)(void *)&sd->pass_through_cmd);
			if (__unlikely(aa >= 0)) {
				sd->flags |= FLAGS_NO_MEDIA;
				SD_RESET_FSTAT(sd);
			}
		}
	} else {
		if (sd->pass_through_cmd.status == -EINTR)
			sd->pass_through_cmd.status = -ETIMEDOUT;
	}
	KERNEL$UNMAP_PHYSICAL_BANK(pth);

	sd->pass_through_physunlock(sd->pass_through_phys);
	sd->pass_through->status = sd->pass_through_cmd.status;
	CALL_AST(sd->pass_through);

	sd->pass_through = NULL;
	WQ_WAKE_ALL(&sd->pass_through_wait);
	SD_DEQUEUE_LOOP(sd);
}

__COLD_ATTR__ static void SCSI_PASS_THROUGH_UNDO_XFORM(SD *sd, SCSI_PASS_THROUGH_HEAD *pth)
{
	switch (sd->pass_through_xform) {
		case XFORM_MODE_SENSE_6_TO_10:
			if (sd->pass_through_cmd.status >= 4) {
				SCSI$MODE_HEADER_10_TO_6((SCSI_MODE_SENSE_6_HEADER *)(pth + 1), (SCSI_MODE_SENSE_HEADER *)(char *)(pth + 1) - 4);
				sd->pass_through_cmd.status -= 4;
			}
			break;
		case XFORM_MODE_SELECT_6_TO_10:
			if (sd->pass_through_cmd.status >= 4) {
				sd->pass_through_cmd.status -= 4;
			}
			break;
	}
}

__COLD_ATTR__ void SCSI_PASS_THROUGH_TIMER_FN(TIMER *t)
{
	SD *sd = GET_STRUCT(t, SD, pass_through_timer);
	LOWER_SPL(SPL_ATA_SCSI);
	SET_TIMER_NEVER(&sd->pass_through_timer);
	sd->ap.cancel((SCSIRQ *)(void *)&sd->pass_through_cmd);
}

