#include <STRING.H>
#include <VALUES.H>
#include <SYS/TYPES.H>
#include <SPAD/AC.H>
#include <SPAD/TIMER.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/CONSOLE.H>
#include <SPAD/IOCTL.H>
#include <SPAD/TTY.H>
#include <SPAD/RANDOM.H>

#include "CONS.H"
#include "CONSCTRL.H"

/* Linux console emulation. Creeping featurism rulez! */

/*
 * linux/drivers/char/console.c
 * Copyright (C) 1991, 1992  Linus Torvalds
 * was used as a reference when writing this file
 */

#define CURSOR_NONE		1
#define CURSOR_UNDERLINE	2
#define CURSOR_LOWER_THIRD	3
#define CURSOR_LOWER_HALF	4
#define CURSOR_TWO_THIRDS	5
#define CURSOR_BLOCK		6
#define CURSOR_SOFT		16

#define CLIPBOARD_SIZE		65536
#define MAX_RING_SIZE   	4096
#define H_FLAGS_VC_MASK		31
#define CURSOR_DEFAULT		CURSOR_UNDERLINE
#define MOUSE_DRAG_BORDER	8

#define BELL_DEFAULT_FREQ	400
#define BELL_DEFAULT_DURATION	100
#define BELL_MAX_DURATION	10000	/* 10 secs */

static void proc_cons_buffer(void);
static __u64 last_cons_update = 0;

#define TAB_STOP_SLOTS		8
#define NPAR			16

#define RING_MASK(x)	((x) & (ring_size - 1))

struct tty_state {
	int color;
	int x, y;
	int attrf;
	int cursor_type;
	int default_cursor_type;

	int def_color;
	int ul_color;
	int half_color;
	int top, bottom;
	unsigned tab_stop[TAB_STOP_SLOTS];

	int decscnm;
	unsigned decom :1;
	unsigned decawm :1;
	unsigned deccm :1;
	unsigned decim :1;

	unsigned disp_ctrl :1;

	int s_color;
	int s_x;
	int s_y;
	int s_attrf;
	unsigned bell_freq;
	unsigned bell_duration;

	unsigned s_charset :1;
	unsigned charset :1;
	unsigned mapping :2;
	unsigned toggle_meta :1;

	unsigned tty_ques :1;
	char tty_state;
	unsigned npar;
	unsigned par[NPAR];
};

#define TTYF_DIM	1
#define TTYF_BRIGHT	2
#define TTYF_BLINK	4
#define TTYF_UNDERLINE	8
#define TTYF_REVERSE	16

static int scrpos[MAX_VCS];

static c_char *lines[MAX_VCS][MAX_RING_SIZE];
static int wvc = -1;
int avc;
WQ_DECL(vc_switch_wait, "CONS$VC_SWITCH_WAIT");
static int scrollback[MAX_VCS] = { 0 };
static int mouse_visible = 0;
static int mouse_x, mouse_y;
static int drag_visible = 0;
static int drag_scrollback = 0;
static int last_drag_dx = 0, last_drag_dy = 0;
static int drag_start_x, drag_start_y;
static int drag_end_x, drag_end_y;

static int clipboard_size = 0;
static char clipboard[CLIPBOARD_SIZE];

int cline;
int chrcolor;
int erasechr;

static struct tty_state tt[MAX_VCS];

static struct tty_state *t;

static int del_upd = 0;
static DECL_TIMER(delayed_update);

#define TTYS_NORMAL	0
#define TTYS_ESC	1
#define TTYS_SQUARE	2
#define TTYS_NONSTD	3
#define TTYS_PERCENT	4
#define TTYS_PALETTE	5
#define TTYS_GETPARS	6
#define TTYS_GOTPARS	7
#define TTYS_FUNCKEY	8
#define TTYS_HASH	9
#define TTYS_SET_G0	10
#define TTYS_SET_G1	11

static const unsigned char color_table[] = { 0, 4, 2, 6, 1, 5, 3, 7, 8,12,10,14, 9,13,11,15 };

static DECL_TIMER(bell_timer);

static void set_tty_color(void)
{
	chrcolor = cons_getattr(t->color, t->attrf ^ t->decscnm) << 8;
	erasechr = (cons_getattr(t->color, (t->attrf & TTYF_BLINK) | t->decscnm) << 8) | 0x20;
}

static void validate_tty_state(void)
{
	if (t->x > cols) t->x = cols;
	if (t->x < 0) t->x = 0;
	if (t->y >= rows) t->y = rows - 1;
	if (t->y < 0) t->y = 0;
	if (t->top < 0) t->top = 0;
	if (t->top > rows - 2) t->top = rows - 2;
	if (t->bottom < 0) t->bottom = 0;
	if (t->bottom > rows) t->bottom = rows;
	if (t->top >= t->bottom - 1) {
		if (t->bottom >= 2) t->top = t->bottom - 2;
		else t->bottom = 0, t->top = 2;
	}
	if (t->npar < 0 || t->npar >= NPAR) t->npar = 0;
	cline = RING_MASK(scrpos[wvc] + t->y);
	set_tty_color();
}

static void save_state(void)
{
	t->s_color = t->color;
	t->s_x = t->x;
	t->s_y = t->y;
	t->s_attrf = t->attrf;
	t->s_charset = t->charset;
}

static void restore_state(void)
{
	t->color = t->s_color;
	t->x = t->s_x;
	t->y = t->s_y;
	t->attrf = t->s_attrf;
	t->charset = t->s_charset;
	t->mapping = t->charset;
	validate_tty_state();
}

static void tty_init(void)
{
	int i, j;
	for (i = 0; i < ring_size; i++)
		for (j = 0; j < cols; j++) lines[wvc][i][j] = 0x0720;
	scrpos[wvc] = 0;

	memset(t, 0, sizeof(struct tty_state));
	t->color = t->def_color = 0x07;
	t->default_cursor_type = CURSOR_DEFAULT;
	t->cursor_type = t->default_cursor_type;
	t->ul_color = 0x0f;
	t->half_color = 0x08;

	t->bottom = rows;
	t->tab_stop[0] = 0x01010100;
	for (i = 1; i < TAB_STOP_SLOTS; i++) t->tab_stop[i] = 0x01010101;

	t->decawm = 1;
	t->deccm = 1;

	t->bell_freq = BELL_DEFAULT_FREQ;
	t->bell_duration = BELL_DEFAULT_DURATION;

	save_state();
	t->tty_state = TTYS_NORMAL;
	scrollback[wvc] = 0;
	validate_tty_state();
}

static void set_write_vc(int vc)
{
	if (__likely(vc == wvc)) return;
	wvc = vc;
	t = &tt[vc];
	validate_tty_state();
}

#define hits_softcursor(x, y)	(__unlikely(tt[avc].cursor_type & CURSOR_SOFT) && __unlikely(!(((x) ^ (tt[avc].x - (tt[avc].x == cols))) | ((y) ^ (tt[avc].y + scrollback[avc])))) && __likely(tt[avc].deccm))
#define hits_mouse(x, y)	(__unlikely(!(((x) ^ mouse_x) | ((y) ^ mouse_y))) && mouse_visible)
#define hits_drag_area(x, y)	(__unlikely(drag_visible) && ((__unlikely((y) == drag_start_y) && (x) >= drag_start_x) || (y) > drag_start_y) && ((__unlikely((y) == drag_end_y) && (x) <= drag_end_x) || (y) < drag_end_y))

static void update_line_slow(int y)
{
	int x;
	c_char *line = lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + y)];
	for (x = 0; x < cols; x++) {
		c_char ch = line[x];
		if (hits_softcursor(x, y)) ch = apply_softcursor(ch);
		if (hits_mouse(x, y)) ch = mouse_invert_char(ch);
		if (hits_drag_area(x, y)) ch = drag_invert_char(ch);
		cons_write(x, y, ch);
	}
}

static void update_pos(int x, int y, int softc);

static void do_update_char(int x, int y, int softc)
{
	int yt;
	yt = y + scrollback[wvc];
	if (__unlikely(yt >= rows)) return;
	update_pos(x, yt, softc);
}

static int soft_last_x = -1, soft_last_y = -1;

static void update_pos(int x, int y, int softc)
{
	c_char ch;
	if (__unlikely(del_upd)) return;
	ch = lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + y)][x];
	if (hits_softcursor(x, y) && (softc || !((x ^ soft_last_x) | (y ^ soft_last_y)))) ch = apply_softcursor(ch);
	if (hits_mouse(x, y)) ch = mouse_invert_char(ch);
	if (hits_drag_area(x, y)) ch = drag_invert_char(ch);
	cons_write(x, y, ch);
}

static __finline__ void update_char(int x, int y)
{
	if (!del_upd && __likely(wvc == avc)) do_update_char(x, y, 0);
}

static void update_softcursor(void)
{
	if (__likely((unsigned)soft_last_x < cols) && __likely((unsigned)soft_last_y < rows)) do_update_char(soft_last_x, soft_last_y, 1);
	if ((soft_last_x ^ (tt[avc].x - (tt[avc].x == cols))) | (soft_last_y ^ (tt[avc].y + scrollback[avc]))) {
		soft_last_x = tt[avc].x - (tt[avc].x == cols);
		soft_last_y = tt[avc].y + scrollback[avc];
		if (__likely((unsigned)soft_last_x < cols) && __likely((unsigned)soft_last_y < rows)) do_update_char(soft_last_x, soft_last_y, 1);
	}
}

static void update_cursor(void)
{
	if (__unlikely(tt[avc].cursor_type & CURSOR_SOFT)) update_softcursor();
	cons_setcursor(tt[avc].x, tt[avc].y + scrollback[avc], __unlikely(!tt[avc].deccm) ? CURSOR_NONE : tt[avc].cursor_type & 0xf);
}

static int kernel_cons_pending(void)
{
	__u64 pos;
	if (__unlikely(!on_kernel_cons)) return 0;
	pos = last_cons_update;
	return KERNEL$CONSOLE_READ((char *)&KERNEL$LIST_END, 1, &pos) != 0;
}

static void delayed_update_fn(TIMER *dummy)
{
	LOWER_SPL(SPL_TTY);
	/*
	 * Warning, we must first process kernel buffer and then clear del_upd.
	 * Otherwise, other CPU hammering on the console could cause livelock.
	 */
	if (__unlikely(kernel_cons_pending())) proc_cons_buffer();
	del_upd = 0;
	cons_update();
	update_cursor();
}

static void set_delayed_update(void)
{
	if (del_upd || __unlikely(avc != wvc)) return;
	del_upd = 1;
	delayed_update.fn = delayed_update_fn;
	KERNEL$SET_TIMER(JIFFIES_PER_SECOND <= DELAYED_UPDATE_FREQ ? 0 : JIFFIES_PER_SECOND / DELAYED_UPDATE_FREQ - 1, &delayed_update);
}

static void clrline(int line, c_char c)
{
	int i;
	for (i = 0; i < cols; i++) lines[wvc][line][i] = c;
}

static void gotoxy(int new_x, int new_y)
{
	int min_y, max_y;
	if (new_x < 0) t->x = 0;
	else if (new_x >= cols) t->x = cols - 1;
	else t->x = new_x;
	if (t->decom) {
		min_y = t->top;
		max_y = t->bottom;
	} else {
		min_y = 0;
		max_y = rows;
	}
	if (new_y < min_y) t->y = min_y;
	else if (new_y >= max_y) t->y = max_y - 1;
	else t->y = new_y;
	cline = RING_MASK(scrpos[wvc] + t->y);
}

static void gotoxay(int new_x, int new_y)
{
	gotoxy(new_x, t->decom ? (t->top + new_y) : new_y);
}

static void bs(void)
{
	if (__unlikely(t->x == cols)) t->x--;
	if (__likely(t->x)) t->x--;
}

#define MAX_SCROLL	16

static c_char *scl[MAX_SCROLL];

static void scroll_reg_down(int top, int bottom, int count);
static void scroll_reg_up(int top, int bottom, int count);

static void scroll_reg_down(int top, int bottom, int count)
{
	int i, j;
	c_char *oln;
	if (__unlikely(!count)) return;
	if (__unlikely(count < 0)) {
		scroll_reg_up(top, bottom, -count);
		return;
	}
	if (__unlikely(bottom - top <= 0)) return;
	if (__unlikely(count > bottom - top)) count = bottom - top;
	while (__unlikely(count > MAX_SCROLL)) {
		scroll_reg_down(top, bottom, MAX_SCROLL);
		count -= MAX_SCROLL;
	}
	for (i = RING_MASK(scrpos[wvc] + bottom - count), j = 0; i != RING_MASK(scrpos[wvc] + bottom); i = RING_MASK(i + 1), j++) {
		scl[j] = lines[wvc][i];
		clrline(i, erasechr);
	}
	if (__unlikely(j > MAX_SCROLL))
		KERNEL$SUICIDE("scroll_reg_down: SHOT OUT OF RING, %d", j);
	for (i = RING_MASK(scrpos[wvc] + top), j = 0; i != RING_MASK(scrpos[wvc] + bottom); i = RING_MASK(i + 1)) {
		oln = lines[wvc][i];
		lines[wvc][i] = scl[j];
		scl[j] = oln;
		if (++j >= count) j = 0;
	}
	set_delayed_update();
}

static void scroll_reg_up(int top, int bottom, int count)
{
	int i, j;
	c_char *oln;
	if (__unlikely(!count)) return;
	if (__unlikely(count < 0)) {
		scroll_reg_down(top, bottom, -count);
		return;
	}
	if (__unlikely(bottom - top <= 0)) return;
	if (__unlikely(count > bottom - top)) count = bottom - top;
	while (__unlikely(count > MAX_SCROLL)) {
		scroll_reg_up(top, bottom, MAX_SCROLL);
		count -= MAX_SCROLL;
	}
	for (i = RING_MASK(scrpos[wvc] + top + count - 1), j = 0; i != RING_MASK(scrpos[wvc] + top - 1); i = RING_MASK(i - 1), j++) {
		scl[j] = lines[wvc][i];
		clrline(i, erasechr);
	}
	if (__unlikely(j > MAX_SCROLL))
		KERNEL$SUICIDE("scroll_reg_up: SHOT OUT OF RING, %d", j);
	for (i = RING_MASK(scrpos[wvc] + bottom - 1), j = 0; i != RING_MASK(scrpos[wvc] + top - 1); i = RING_MASK(i - 1)) {
		oln = lines[wvc][i];
		lines[wvc][i] = scl[j];
		scl[j] = oln;
		if (++j >= count) j = 0;
	}
	set_delayed_update();
}

static void nl(void)
{
	if (t->y < t->bottom - 1) {
		next_line:
		t->y++;
		cline = RING_MASK(cline + 1);
	} else if (__likely(t->y == t->bottom - 1)) {
		if (__likely(!t->top)) {
			scrpos[wvc] = RING_MASK(scrpos[wvc] + 1);
			cline = RING_MASK(cline + 1);
			scroll_reg_down(t->bottom - 1, rows, 1);
			/*clrline(cline, erasechr);*/
			if (del_upd) return;
			set_delayed_update();
		} else {
			scroll_reg_up(t->top, t->bottom, 1);
		}
	} else if (t->y < rows - 1) {
		goto next_line;
	}
}

static __finline__ void cr(void)
{
	t->x = 0;
}

static void ri(void)
{
	if (t->y > t->top) {
		t->y--;
		cline = RING_MASK(cline - 1);
	} else if (t->y == t->top) {
		if (__likely(!t->top)) {
			scrpos[wvc] = RING_MASK(scrpos[wvc] - 1);
			cline = RING_MASK(cline - 1);
			clrline(cline, erasechr);
			scroll_reg_up(t->bottom, rows + 1, 1);
			if (del_upd) return;
			set_delayed_update();
		} else {
			scroll_reg_down(t->top, t->bottom, 1);
		}
	}
}

static void csi_J(int par)
{
	int i;
	switch (par) {
		case 0:
			for (i = t->x; i < cols; i++) lines[wvc][cline][i] = erasechr;
			for (i = t->y + 1; i < rows; i++) clrline(RING_MASK(scrpos[wvc] + i), erasechr);
			set_delayed_update();
			break;
		case 1:
			for (i = 0; i <= t->x && i < cols; i++) lines[wvc][cline][i] = erasechr;
			for (i = 0; i < t->y; i++) clrline(RING_MASK(scrpos[wvc] + i), erasechr);
			set_delayed_update();
			break;
		case 2:
			for (i = 0; i < rows; i++) clrline(RING_MASK(scrpos[wvc] + i), erasechr);
			set_delayed_update();
			break;
	}
}

static void csi_K(int par)
{
	int i;
	switch (par) {
		case 0:
			for (i = t->x; i < cols; i++) {
				lines[wvc][cline][i] = erasechr;
				update_char(i, t->y);
				/*set_delayed_update();*/
			}
			break;
		case 1:
			for (i = 0; i <= t->x && i < cols; i++) {
				lines[wvc][cline][i] = erasechr;
				update_char(i, t->y);
				/*set_delayed_update();*/
			}
			break;
		case 2:
			for (i = 0; i < cols; i++) {
				lines[wvc][cline][i] = erasechr;
				update_char(i, t->y);
				/*set_delayed_update();*/
			}
			break;
	}
}

static void csi_L(int par)
{
	scroll_reg_down(t->y, t->bottom, par ? par : 1);
}

static void csi_M(int par)
{
	scroll_reg_up(t->y, t->bottom, par ? par : 1);
}

static void delete_chars(int count)
{
	int i;
	if (count > cols - t->x) count = cols - t->x;
	memmove(lines[wvc][cline] + t->x, lines[wvc][cline] + t->x + count, (cols - t->x - count) * sizeof(c_char));
	for (i = cols - count; i < cols; i++) lines[wvc][cline][i] = erasechr;
	for (i = t->x; i < cols; i++) update_char(i, t->y);
}

static void csi_mm(void)
{
	int i;
	for (i = 0; i <= t->npar && i < NPAR; i++) {
		switch (t->par[i]) {
			case 0:
				t->color = t->def_color;
				t->attrf = 0;
				break;
			case 1:
				t->attrf |= TTYF_BRIGHT;
				t->attrf &= ~TTYF_DIM;
				break;
			case 2:
				t->attrf |= TTYF_DIM;
				t->attrf &= ~TTYF_BRIGHT;
				break;
			case 4:
				t->attrf |= TTYF_UNDERLINE;
				break;
			case 5:
				t->attrf |= TTYF_BLINK;
				break;
			case 7:
				t->attrf |= TTYF_REVERSE;
				break;
			case 10:
				t->disp_ctrl = 0;
				t->charset = 0;		/* this is nonstandard, but helps to recover console for command prompt */
				t->mapping = t->charset;
				t->toggle_meta = 0;
				break;
			case 11:
				t->disp_ctrl = 1;
				t->mapping = 2;
				t->toggle_meta = 0;
				break;
			case 12:
				t->disp_ctrl = 1;
				t->mapping = 2;
				t->toggle_meta = 1;
				break;
			case 21:
			case 22:
				t->attrf &= ~(TTYF_DIM | TTYF_BRIGHT);
				break;
			case 24:
				t->attrf &= ~TTYF_UNDERLINE;
				break;
			case 25:
				t->attrf &= ~TTYF_BLINK;
				break;
			case 27:
				t->attrf &= ~TTYF_REVERSE;
				break;
			case 30:
			case 31:
			case 32:
			case 33:
			case 34:
			case 35:
			case 36:
			case 37:
				t->color = color_table[t->par[i] - 30] | (t->color & 0xf0);
				break;
			case 38:
				t->color = (t->def_color & 0x0f) | (t->color & 0xf0);
				t->attrf |= TTYF_UNDERLINE;
				break;
			case 39:
				t->color = (t->def_color & 0x0f) | (t->color & 0xf0);
				t->attrf &= ~TTYF_UNDERLINE;
				break;
			case 40:
			case 41:
			case 42:
			case 43:
			case 44:
			case 45:
			case 46:
			case 47:
				t->color = (color_table[t->par[i] - 40] << 4) | (t->color & 0x0f);
				break;
			case 49:
				t->color = (t->def_color & 0xf0) | (t->color & 0x0f);
				break;
		}
	}
	set_tty_color();
}

static void csi_X(unsigned n)
{
	int i;
	if (n > cols - t->x) n = cols - t->x;
	for (i = 0; i < n; i++) {
		lines[wvc][cline][t->x + i] = erasechr;
		update_char(t->x + i, t->y);
	}
}

static void insert_char(unsigned nr)
{
	int i;
	if (nr > cols - t->x) nr = cols - t->x;
	memmove(lines[wvc][cline] + t->x + nr, lines[wvc][cline] + t->x, (cols - t->x - nr) * sizeof(c_char));
	for (i = 0; i < nr; i++) lines[wvc][cline][t->x + i] = erasechr;
	for (i = t->x; i < cols; i++) update_char(i, t->y);
}

static void invert_char_at(int x, int y)
{
	c_char *a = &lines[wvc][RING_MASK(scrpos[wvc] + y)][x];
	*a = invert_char(*a);
}

static void invert_screen(int x1, int y1, int x2, int y2)
{
	int i, j;
	if (y1 != y2) {
		for (i = x1; i < cols; i++) invert_char_at(i, y1);
		for (j = y1 + 1; j < y2; j++)
			for (i = 0; i < cols; i++) invert_char_at(i, j);
		for (i = 0; i <= x2; i++) invert_char_at(i, y2);
	} else {
		for (i = x1; i <= x2; i++) invert_char_at(i, y1);
	}
	set_delayed_update();
}

static void tty_set_mode(int on_off)
{
	int i;
	for (i = 0; i <= t->npar && i < NPAR; i++) {
		if (t->tty_ques) switch (t->par[i]) {
			case 1:
				/* UNIMPL: cursor keys send ESC O */
				break;
			case 3:
				/* UNIMPL: 80/132 col switch */
				break;
			case 5:
				if (t->decscnm != on_off * TTYF_REVERSE) {
					t->decscnm = on_off * TTYF_REVERSE;
					invert_screen(0, 0, cols - 1, rows - 1);
					set_tty_color();
				}
				break;
			case 6:
				t->decom = on_off;
				gotoxay(0, 0);
				break;
			case 7:
				t->decawm = on_off;
				break;
			case 8:
				/* UNIMPL: autorepeat */
				break;
			case 9:
				/* UNIMPL: mouse */
				break;
			case 25:
				t->deccm = on_off;
				break;
			case 1000:
				/* UNIMPL: mouse */
				break;
		} else switch (t->par[i]) {
			case 3:
				t->disp_ctrl = on_off;
				break;
			case 4:
				t->decim = on_off;
				break;
			case 20:
				/* UNIMPL: keyboard CR->CR/NL */
				break;
		}
	}
}

static void linux_specific(void)
{
	switch (t->par[0]) {
		case 1:
			if (can_do_color && t->par[1] < 16)
				t->ul_color = color_table[t->par[1]];
			break;
		case 2:
			if (can_do_color && t->par[1] < 16)
				t->half_color = color_table[t->par[1]];
			break;
		case 8:
			t->def_color = t->color;
			break;
		case 9:
			/* UNIMPL: no blank interval */
			break;
		case 10:
			t->bell_freq = t->par[1];
			break;
		case 11:
			t->bell_duration = t->par[1];
			if (__unlikely(t->bell_duration > BELL_MAX_DURATION))
				t->bell_duration = BELL_MAX_DURATION;
			break;
		case 12:
			/* UNIMPL: no forced console switch */
			break;
		case 13:
			/* UNIMPL: no screen unblank */
			break;
		case 14:
			/* UNIMPL: no VESA powerdown interval */
			break;
	}
	set_tty_color();
}

static void bell(void)
{
	KERNEL$DEL_TIMER(&bell_timer);
	KERNEL$BEEP(t->bell_freq);
	KERNEL$SET_TIMER(t->bell_duration * JIFFIES_PER_SECOND / 1000, &bell_timer);
}

static void bell_timer_fn(TIMER *t)
{
	LOWER_SPL(SPL_TTY);
	SET_TIMER_NEVER(t);
	KERNEL$BEEP(0);
}

static char nlcr = 0;

static void proc_string(unsigned char *p, int s)
{
	int i;
	unsigned char c;
	if ((__unlikely(mouse_visible | drag_visible | scrollback[wvc])) && __likely(wvc == avc)) {
		mouse_visible = 0;
		drag_visible = 0;
		set_delayed_update();
	} /* else if (t->y >= t->bottom - 1 && memchr(p, '\n', s) && !memchr(p, '\e', s)) {
		set_delayed_update();
	} */
	again:
	if (!s) return;
	c = *p++;
	if (__likely(t->tty_state == TTYS_NORMAL)) {
		if (__likely(c >= ' ')) {
			chr:
			if (__likely(t->x < cols)) {
				chb:
				if (__unlikely(t->decim)) goto dcm;
				bdcm:
				lines[wvc][cline][t->x++] = console_charsets[wvc]->xlate[t->mapping][__unlikely(t->toggle_meta) ? c | 0x80 : c] | chrcolor;
				update_char(t->x - 1, t->y);
				if (__unlikely(!t->decawm) && __unlikely(t->x == cols)) t->x = cols - 1;
				s--;
				goto again;

				dcm:
				insert_char(1);
				goto bdcm;
			}
			cr();
			nl();
			goto chb;
		}
		if (__unlikely(!(((t->disp_ctrl ? CTRL_ALWAYS : CTRL_ACTION) >> c) & 1))) goto chr;
	}
	if (__likely(c == 10)) {
		if (__unlikely(nlcr))
			cr();
		nl();
		goto next;
	}
	switch (c) {
		case 0:
			set_delayed_update();
			goto next;
		case 7:
			bell();
			goto next;
		case 8:
			bs();
			goto next;
		case 9:
			while (__likely(t->x < cols - 1)) {
				t->x++;
				if ((t->x >> 5) < TAB_STOP_SLOTS
				    && t->tab_stop[t->x >> 5] & (1 << (t->x & 31)))
					goto next;
			}
			goto next;
		case 11:
		case 12:
			nl();
			goto next;
		case 13:
			cr();
			goto next;
		case 14:
			t->disp_ctrl = 1;
			t->charset = 1;
			t->mapping = 1;
			goto next;
		case 15:
			t->disp_ctrl = 0;
			t->charset = 0;
			t->mapping = 0;
			goto next;
		case 24:
		case 26:
			t->tty_state = TTYS_NORMAL;
			goto next;
		case 27:
			t->tty_state = TTYS_ESC;
			goto next;
#if 0
		case 127:
			goto next;
		case 128 + 27:			/* why ??? */
			t->tty_state = TTYS_SQUARE;
			goto next;
#endif
	}
	switch (t->tty_state) {
		case TTYS_NORMAL:
			goto next;
		case TTYS_ESC:
			t->tty_state = TTYS_NORMAL;
			switch (c) {
				case '[':
					t->tty_state = TTYS_SQUARE;
					goto next;
				case ']':
					t->tty_state = TTYS_NONSTD;
					goto next;
				case '%':
					t->tty_state = TTYS_PERCENT;
					goto next;
				case 'E':
					cr();
					nl();
					goto next;
				case 'M':
					ri();
					goto next;
				case 'D':
					nl();
					goto next;
				case 'H':
					if ((t->x >> 5) < TAB_STOP_SLOTS)
						t->tab_stop[t->x >> 5] |= (1 << (t->x & 31));
					goto next;
				case 'Z':
					/* UNIMPL: should respond. */
					goto next;
				case '7':
					save_state();
					goto next;
				case '8':
					restore_state();
					goto next;
				case '(':
					t->tty_state = TTYS_SET_G0;
					goto next;
				case ')':
					t->tty_state = TTYS_SET_G1;
					goto next;
				case  '#':
					t->tty_state = TTYS_HASH;
					goto next;
				case 'c':
					tty_init();
					set_delayed_update();
					goto next;
				case '>':
				case '=':
					/* UNIMPL: keypad not here now. */
					goto next;
			}
			goto next;
		case TTYS_NONSTD:
			t->tty_state = TTYS_NORMAL;
			switch (c) {
				case 'P':
					t->npar = 0;
					t->tty_state = TTYS_PALETTE;
					goto next;
				case 'R':
					/* UNIMPL: reset palette */
					goto next;
			}
			goto next;
		case TTYS_PALETTE:
			if (((c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f')) && t->npar < NPAR) {
				t->par[t->npar++] = (c > '9' ? (c & 0xDF) - 'A' + 10 : c - '0');
				if (t->npar >= 7) {
					t->tty_state = TTYS_NORMAL;
					/* UNIMPL: we should set palette here ... */
				}
			} else t->tty_state = TTYS_NORMAL;
			goto next;
		case TTYS_SQUARE:
			for (i = 0; i < NPAR; i++) t->par[i] = 0;
			t->npar = 0;
			t->tty_state = TTYS_GETPARS;
			switch (c) {
				case '[':
					t->tty_state = TTYS_FUNCKEY;
					goto next;
				case '?':
					t->tty_ques = 1;
					goto next;
				default:
					t->tty_ques = 0;
			}
			/* fall through */
		case TTYS_GETPARS:
			if (c == ';' && __likely(t->npar < NPAR - 1)) {
				t->npar++;
				goto next;
			}
			if (__likely(c >= '0') && __likely(c <= '9')) {
				if (__likely(t->npar < NPAR)) {
					unsigned x = t->par[t->npar];
					if (__unlikely(x > MAXUINT / 10)) x = MAXUINT;
					else x *= 10;
					x += c - '0';
					if (__unlikely(x < c - '0')) x = MAXUINT;
					t->par[t->npar] = x;
				}
				goto next;
			} else {
				t->tty_state = TTYS_GOTPARS;
			}
			/* fall through */
		case TTYS_GOTPARS:
			t->tty_state = TTYS_NORMAL;
			switch (c) {
				case 'h':
					tty_set_mode(1);
					goto next;
				case 'l':
					tty_set_mode(0);
					goto next;
				case 'c':
				case 'd':
					if (t->tty_ques) {
						if (!t->par[0]) t->cursor_type = __unlikely(c == 'd') ? CURSOR_DEFAULT : t->default_cursor_type;
						else t->cursor_type = t->par[0] | (t->par[1] << 8) | (t->par[2] << 16);
						if (__unlikely(c == 'd')) t->default_cursor_type = t->cursor_type;
						update_char(t->x - (t->x == cols), t->y);
						goto next;
					}
					break;
				case 'm':
					if (t->tty_ques) {
						/* UNIMPL: selection invert */
						goto next;
					}
					break;
				case 'n':
					if (!t->tty_ques) {
						/* UNIMPL: report not here */
						goto next;
					}
					break;
			}
			if (t->tty_ques) {
				t->tty_ques = 0;
				goto next;
			}
			switch (c) {
				case 'G':
				case '`':
					if (t->par[0]) t->par[0]--;
					gotoxy(t->par[0], t->y);
					goto next;
				case 'A':
					if (!t->par[0]) t->par[0] = 1;
					gotoxy(t->x, t->y - t->par[0]);
					goto next;
				case 'B':
				case 'e':
					if (!t->par[0]) t->par[0] = 1;
					gotoxy(t->x, t->y + t->par[0]);
					goto next;
				case 'C':
				case 'a':
					if (!t->par[0]) t->par[0] = 1;
					gotoxy(t->x + t->par[0], t->y);
					goto next;
				case 'D':
					if (!t->par[0]) t->par[0] = 1;
					gotoxy(t->x - t->par[0], t->y);
					goto next;
				case 'E':
					if (!t->par[0]) t->par[0] = 1;
					gotoxy(0, t->y + t->par[0]);
					goto next;
				case 'F':
					if (!t->par[0]) t->par[0] = 1;
					gotoxy(0, t->y - t->par[0]);
					goto next;
				case 'd':
					if (t->par[0]) t->par[0]--;
					gotoxay(t->x, t->par[0]);
					goto next;
				case 'H':
				case 'f':
					if (t->par[0]) t->par[0]--;
					if (t->par[1]) t->par[1]--;
					gotoxay(t->par[1], t->par[0]);
					goto next;
				case 'J':
					csi_J(t->par[0]);
					goto next;
				case 'K':
					csi_K(t->par[0]);
					goto next;
				case 'L':
					csi_L(t->par[0]);
					goto next;
				case 'M':
					csi_M(t->par[0]);
					goto next;
				case 'P':
					delete_chars(t->par[0] ? t->par[0] : 1);
					goto next;
				case 'c':
					if (!t->par[0]) /* UNIMPL: respond */;
					goto next;
				case 'g':
					if (!t->par[0]) {
						if ((t->x >> 5) < TAB_STOP_SLOTS)
							t->tab_stop[t->x >> 5] &= ~(1 << (t->x & 31));
					} else if (t->par[0] == 3) {
						for (i = 0; i < TAB_STOP_SLOTS; i++)
							t->tab_stop[i] = 0;
					}
					goto next;
				case 'm':
					csi_mm();
					goto next;
				case 'q':
					/* UNIMPL: leds not supported */
					goto next;
				case 'r':
					if (!t->par[0]) t->par[0] = 1;
					if (!t->par[1]) t->par[1] = rows;
					if (t->par[0] < t->par[1] &&
					    t->par[1] <= rows) {
						t->top = t->par[0] - 1;
						t->bottom = t->par[1];
						gotoxay(0, 0);
					}
					goto next;
				case 's':
					save_state();
					goto next;
				case 'u':
					restore_state();
					goto next;
				case 'X':
					csi_X(t->par[0] ? t->par[0] : 1);
					goto next;
				case '@':
					insert_char(t->par[0] ? t->par[0] : 1);
					goto next;
				case ']':
					linux_specific();
					goto next;
			}
			goto next;
		case TTYS_PERCENT:
			t->tty_state = TTYS_NORMAL;
			switch (c) {
				case '@':
					/* UNIMPL: no utf-8 */
					goto next;
				case 'G':
				case '8':
					/* UNIMPL: no utf-8 */
					goto next;
			}
			goto next;
		case TTYS_FUNCKEY:
			t->tty_state = TTYS_NORMAL;
			goto next;
		case TTYS_HASH:
			t->tty_state = TTYS_NORMAL;
			switch (c) {
				case '8':
					erasechr = (erasechr & 0xff00) | 'E';
					csi_J(2);
					set_tty_color();
					goto next;
			}
			goto next;
		case TTYS_SET_G0:
		case TTYS_SET_G1:
			t->tty_state = TTYS_NORMAL;
			/* UNIMPL: G0/G1 switching */
			goto next;
		default:
			t->tty_state = TTYS_NORMAL;
	}
	next:
	s--;
	goto again;
}

#define BUF_SIZE	4096

static unsigned char buffer[BUF_SIZE];

static const VBUF vbuf = {  SPL_X(SPL_TTY), BUF_SIZE, buffer };

static const HANDLE_OPERATIONS console_operations;

static DECL_IOCALL(console_write, SPL_TTY, SIORQ)
{
	long s;
	if (__unlikely(RQ->handle->op != &console_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_TTY));
	if (__unlikely(kernel_cons_pending())) proc_cons_buffer();
	if (__unlikely(!RQ->v.len)) goto zlen;
	again:
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_get(&RQ->v, &vbuf);
	if (__unlikely(!s)) goto pf;
	RQ->progress += s;
	KERNEL$ADD_RANDOMNESS(NULL, buffer, s);
	set_write_vc(RQ->handle->flags & H_FLAGS_VC_MASK);
	proc_string(buffer, s);
	if (!del_upd) update_cursor();
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_TTY));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (__unlikely(RQ->v.len != 0)) goto again;
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);

	pf:
	DO_PAGEIN(RQ, &RQ->v, PF_READ);
}

static void proc_cons_buffer(void)
{
	int back_color;
	int back_attrf;
	int back_decscnm;
	int r;
	u_jiffies_lo_t j1, j2;
	int jc;

	cons_clear_cached();
	set_write_vc(avc);
	back_color = t->color;
	back_attrf = t->attrf;
	back_decscnm = t->decscnm;
	t->color = 0x07;
	t->attrf = TTYF_REVERSE;
	t->decscnm = 0;
	set_tty_color();

	nlcr = 1;

	j1 = KERNEL$GET_JIFFIES_LO();
	jc = 0;
	while ((r = KERNEL$CONSOLE_READ((char *)buffer, BUF_SIZE, &last_cons_update))) {
		proc_string(buffer, r);
		j2 = KERNEL$GET_JIFFIES_LO();
		if (j2 != j1) if (++jc == 2) break;
		j1 = j2;
	}

	nlcr = 0;

	t->color = back_color;
	t->attrf = back_attrf;
	t->decscnm = back_decscnm;
	set_tty_color();
	if (!del_upd) update_cursor();
	set_delayed_update();
}

__COLD_ATTR__ static void *console_lookup(HANDLE *h, char *str, int open_flags)
{
	char *e;
	e = strchr(str, '=');
	if (__unlikely(!e)) return __ERR_PTR(-EBADMOD);
	if (__likely(!__strcasexcmp("^VC", str, e))) {
		long vc;
		if (__unlikely(__get_number(e + 1, e + strlen(e), 0, &vc)) || __unlikely((unsigned long)--vc >= vcs)) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~H_FLAGS_VC_MASK) | vc;
	} else return __ERR_PTR(-EBADMOD);
	return NULL;
}

static int is_word(char c, char c1)
{
	return (c >= '0' && c <= '9') || (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || c == '$' || c == '-' || c == '.' || c == '/' || c == '@' || c == '^' || c == '_' || c == '~' || (c == ':' && c1 == '/');
}

static void align_word(int *x, int *y, int dir)
{
	char c, c1, d, d1;
	c = lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + *y)][*x];
	c1 = *x + 1 < cols ? lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + *y)][*x + 1] : 0;
	again:
	*x += dir;
	if (__unlikely(*x < 0)) {
		*x = 0;
		return;
	}
	if (__unlikely(*x >= cols)) {
		*x = cols - 1;
		return;
	}
	d = lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + *y)][*x];
	d1 = *x + 1 < cols ? lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + *y)][*x + 1] : 0;
	if ((!is_word(c, c1) || !is_word(d, d1)) && c != d) {
		*x -= dir;
		return;
	}
	goto again;
}

static __finline__ void msize(int *x, int *y)
{
	*x = 8;
	*y = *x * cols * 3 / 4 / (__likely(rows) ? rows : 1);
}

static DECL_IOCALL(console_ioctl, SPL_TTY, IOCTLRQ)
{
	int os;
	if (__unlikely(RQ->handle->op != &console_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_TTY));
	if (__unlikely(kernel_cons_pending())) proc_cons_buffer();
	switch (RQ->ioctl) {
		case IOCTL_CONSOLE_SWITCH: {
			WQ *wq;
			if (__unlikely((unsigned long)RQ->param >= vcs)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			wq = VESA_SET_VIDEOMODE(RQ->param);
			if (__unlikely(wq != NULL) && __likely(!__IS_ERR(wq))) {
				WQ_WAIT_F(wq, RQ);
				RETURN;
			}
			WQ_WAKE_ALL(&vc_switch_wait);
			/*scrollback[avc] = 0;*/
			drag_scrollback = 0;
			drag_visible = 0;
			mouse_visible = 0;
			CONS_DO_UPDATE();
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_CONSOLE_SCROLL: {
			struct scroll_request sr;
			int r;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &sr, sizeof sr))) {
				if (__likely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(console_gfx)) {
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			os = scrollback[avc];
			scrollback[avc] += (unsigned)sr.n * (__likely(sr.flags & SCROLL_HALFPAGES) ? rows >> 1 : 1);
			set_write_vc(avc);
			if (__unlikely(scrollback[avc] < 0)) scrollback[avc] = 0;
			else if (__unlikely(scrollback[avc] > ring_size - rows)) scrollback[avc] = ring_size - rows;
			validate_tty_state();
			if (__unlikely(sr.flags & SCROLL_DRAGGING)) {
				drag_scrollback += scrollback[avc] - os;
			} else {
				drag_start_y += scrollback[avc] - os;
				drag_end_y += scrollback[avc] - os;
				cons_update();
				update_cursor();
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_CONSOLE_MOUSE: {
			int x, y;
			int dx, dy;
			int msizex, msizey;
			int tl;
			char would_be_newline;
			int space_run, common_space_run;
			int i, j;
			struct console_mouse_request cmr;
			int r;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &cmr, sizeof cmr))) {
				if (__likely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(console_gfx)) {
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			msize(&msizex, &msizey);
			cmr.min_x = 0;
			cmr.min_y = 0;
			cmr.max_x = msizex * cols;
			cmr.max_y = msizey * rows;
			if (__unlikely(cmr.drag)) {
				cmr.min_x -= MOUSE_DRAG_BORDER;
				cmr.min_y -= MOUSE_DRAG_BORDER;
				cmr.max_x += MOUSE_DRAG_BORDER;
				cmr.max_y += MOUSE_DRAG_BORDER;
			}
			if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(RQ, &cmr, sizeof cmr))) {
				if (__likely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			x = cmr.x;
			y = cmr.y;
			x /= __likely(msizex) ? msizex : 1;
			y /= __likely(msizey) ? msizey : 1;
			if (__unlikely(x >= cols)) x = cols - 1;
			if (__unlikely(x < 0)) x = 0;
			if (__unlikely(y >= rows)) y = rows - 1;
			if (__unlikely(y < 0)) y = 0;
			if (__unlikely(cmr.drag)) goto do_drag;
			drag_scrollback = 0;
			if (__likely(mouse_visible)) {
				int lx = mouse_x, ly = mouse_y;
				if (x == lx && y == ly) goto skip_inv;
				mouse_x = x;
				mouse_y = y;
				if (__likely(!cmr.update)) {
					update_pos(x, y, 1);
					update_pos(lx, ly, 1);
				}
			} else {
				mouse_visible = 1;
				mouse_x = x;
				mouse_y = y;
				if (__likely(!cmr.update)) {
					update_pos(x, y, 1);
				}
			}
			skip_inv:
			if (__unlikely(cmr.update)) {
				cons_update();
				update_cursor();
			}
			RQ->status = 1;
			RETURN_AST(RQ);

			do_drag:
			dx = cmr.start_x;
			dy = cmr.start_y;
			dx /= __likely(msizex) ? msizex : 1;
			dy /= __likely(msizey) ? msizey : 1;
			if (__unlikely(dx >= cols)) dx = cols - 1;
			if (__unlikely(dx < 0)) dx = 0;
			if (__unlikely(dy >= rows)) dy = rows - 1;
			if (__unlikely(dy < 0)) dy = 0;

			if (dx != last_drag_dx || dy != last_drag_dy) {
				drag_scrollback = 0;
				last_drag_dx = dx;
				last_drag_dy = dy;
			}
			dy += drag_scrollback;

			if (__unlikely(cmr.y < 0)) {
				x = 0;
			} else if (__unlikely(cmr.y >= rows * msizey)) {
				x = cols - 1;
			} else if (cmr.x < 0 && __likely(y) && __likely(dy < y)) {
				x = cols - 1;
				y--;
			}

			if (cmr.drag == 2) {
				if (dy > y || (dy == y && dx > x)) {
					align_word(&x, &y, -1);
					align_word(&dx, &dy, 1);
				} else {
					align_word(&dx, &dy, -1);
					align_word(&x, &y, 1);
				}
			} else if (__unlikely(cmr.drag == 3)) {
				if (dy > y || (dy == y && dx > x)) {
					x = 0;
					dx = cols - 1;
				} else {
					dx = 0;
					x = cols - 1;
				}
			}
			if (dy > y || (dy == y && dx > x)) {
				int n;
				n = x; x = dx; dx = n;
				n = y; y = dy; dy = n;
			}
			drag_start_x = dx;
			drag_start_y = dy;
			drag_end_x = x;
			drag_end_y = y;
			mouse_visible = 0;
			drag_visible = 1;
			cons_update();
			update_cursor();

			clipboard_size = 0;
			tl = 0;
			would_be_newline = 0;
			common_space_run = MAXINT;
			space_run = 0;
#define scrollback_read(x, y)	(console_charsets[avc]->xlate[3][(unsigned char)lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + (y))][x]])
			for (i = 0; i < dx; i++) {
				if (scrollback_read(i, dy) != ' ') goto no_leading_spaces;
			}
			dx = 0;
			no_leading_spaces:
			while ((dy < y || (dy == y && dx <= x)) && clipboard_size < sizeof clipboard) {
				unsigned char c = scrollback_read(dx, dy);
				if (c == ' ' && would_be_newline && scrollback_read(1, dy) == ' ') {
					would_be_newline = 0;
					clipboard[clipboard_size++] = '\r';
					space_run = 0;
					continue;
				}
				would_be_newline = 0;
				if (c == ' ' && dx >= tl) {
					int i;
					if (dx >= cols - 1) {
						if (dy < rows - 1 + scrollback[avc] && scrollback_read(0, dy + 1) != ' ') goto copy_char;
					}
					for (i = dx + 1; i < cols; i++) if ((unsigned char)lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + dy)][i] > ' ') {
						tl = i;
						goto copy_char;
					}
					clipboard[clipboard_size++] = '\r';
					space_run = 0;
					newl:
					dx = 0;
					dy++;
					tl = 0;
					continue;
				}
				copy_char:
				clipboard[clipboard_size++] = c;
				if (space_run >= 0) {
					if (c == ' ') {
						space_run++;
					} else {
						if (space_run < common_space_run)
							common_space_run = space_run;
						space_run = -1;
					}
				}
				if (__unlikely(++dx == cols)) {
					would_be_newline = 1;
					goto newl;
				}
			}
			if (common_space_run && common_space_run < MAXINT) {
				space_run = common_space_run;
				for (i = 0, j = 0; i < clipboard_size; i++) {
					if (clipboard[i] == '\r') {
						space_run = common_space_run;
						goto copy_ch;
					}
					if (space_run > 0) {
						space_run--;
					} else {
						copy_ch:
						clipboard[j++] = clipboard[i];
					}
				}
				clipboard_size = j;
			}
			RQ->status = 1;
			RETURN_AST(RQ);
		}
		case IOCTL_CONSOLE_GET_CLIPBOARD: {
			int r;
			unsigned long l = RQ->v.len;
			if (__unlikely(!clipboard_size)) {
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			if (__likely(RQ->v.len > clipboard_size)) RQ->v.len = clipboard_size;
			r = KERNEL$PUT_IOCTL_STRUCT(RQ, clipboard, RQ->v.len);
			if (__unlikely(r == 1)) {
				RQ->v.len = l;
				DO_PAGEIN(RQ, &RQ->v, PF_READ | PF_WRITE);
			}
			if (__likely(r >= 0)) {
				if (__likely((long)RQ->v.len >= 0)) RQ->status = RQ->v.len;
				else RQ->status = -EOVERFLOW;
			} else {
				RQ->status = r;
			}
			RQ->v.len = l;
			RETURN_AST(RQ);
		}
		case IOCTL_CONSOLE_ERASE_CLIPBOARD: {
			clipboard_size = 0;
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_WINSIZE: {
			int r;
			int mx, my;
			struct tty_window_size w;
			memset(&w, 0, sizeof w);
			msize(&mx, &my);
			w.ws_row = rows;
			w.ws_col = cols;
			w.ws_xpixel = w.ws_col * mx;
			w.ws_ypixel = w.ws_row * my;
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &w, sizeof w)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_CONSOLE_NUM_VCS: {
			RQ->status = vcs;
			RETURN_AST(RQ);
		}
		case IOCTL_CONSOLE_VC: {
			RQ->status = RQ->handle->flags & H_FLAGS_VC_MASK;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

void CONS_DO_UPDATE(void)
{
	cons_clear_cached();
	cons_update();
	update_cursor();
}

static int COMPARE_CHARSETS(struct console_charset *c1, struct console_charset *c2);
static void UNREF_CHARSET(struct console_charset *c);

__COLD_ATTR__ static long CONS_SET_FONT(int vc, PROC *proc, FBLOB *blob_addr, size_t blob_size, IORQ *rq)
{
	vspace_unmap_t *unmap;
	VDESC vdesc;
	struct font_blob *font;
	unsigned bmp_slots;
	struct console_charset *map;
	unsigned i;
	if (!proc) {
		do_default:
		if (console_charsets[vc] == &default_charset) return 0;
		UNREF_CHARSET(console_charsets[vc]);
		console_charsets[vc] = (struct console_charset *)&default_charset;
		if (vc == avc) cons_hw_set_font(&default_charset);
		return 0;
	}
	if (__unlikely(blob_size < sizeof(struct font_blob))) {
		if (__likely(blob_size == sizeof(FBLOB))) goto do_default;
		return (long)__ERR_PTR(-EINVAL);
	}
	vdesc.ptr = (unsigned long)blob_addr;
	vdesc.len = blob_size;
	vdesc.vspace = KERNEL$PROC_VSPACE(proc);
	RAISE_SPL(SPL_VSPACE);
	font = vdesc.vspace->op->vspace_map(&vdesc, PF_READ, SPL_X(SPL_TTY), &unmap);
	if (__unlikely(!font)) {
		DO_PAGEIN_NORET(rq, &vdesc, PF_READ);
		return 1;
	}
	if (__unlikely(__IS_ERR(font))) {
		return (long)font;
	}
	bmp_slots = LONGS_PER_CHAR(font->w, font->h);
	if (__unlikely(blob_size < sizeof(struct font_blob) + (256 * bmp_slots - 1) * sizeof(unsigned long))) {
		unmap(font);
		return (long)__ERR_PTR(-EINVAL);
	}
	if (!cons_validate_font(font)) {
		unmap(font);
		goto do_default;
	}

	map = malloc(sizeof(struct console_charset) + (256 * bmp_slots - 1) * sizeof(unsigned long));
	if (__unlikely(!map)) {
		unmap(font);
		KERNEL$MEMWAIT(rq, (IO_STUB *)rq->tmp1, sizeof(struct console_charset) + (256 * bmp_slots - 1) * sizeof(unsigned long));
		return 1;
	}
	memcpy(map->xlate, font->xlate, sizeof map->xlate);
	map->width = font->w;
	map->height = font->h;
	map->refcount = 1;
	memcpy(map->data, font->data, 256 * bmp_slots * sizeof(unsigned long));
	unmap(font);
	if (!COMPARE_CHARSETS(console_charsets[vc], map)) {
		free_map_ret:
		free(map);
		return 0;
	}
	for (i = 0; i < MAX_VCS; i++) if (__likely(i != vc)) {
		struct console_charset *prev;
		if (COMPARE_CHARSETS(console_charsets[i], map)) continue;
		console_charsets[i]->refcount++;
		prev = console_charsets[vc];
		console_charsets[vc] = console_charsets[i];
		UNREF_CHARSET(prev);
		if (vc == avc) cons_hw_set_font(console_charsets[vc]);
		goto free_map_ret;
	}
	UNREF_CHARSET(console_charsets[vc]);
	console_charsets[vc] = map;
	if (vc == avc) cons_hw_set_font(map);
	return 0;
}

__COLD_ATTR__ void CONS_SET_FONT_VT_SWITCH(int prev_cons, int new_cons)
{
	if (prev_cons == new_cons) return;
	if (prev_cons != -1 && !COMPARE_CHARSETS(console_charsets[prev_cons], console_charsets[new_cons])) return;
	cons_hw_set_font(console_charsets[new_cons]);
}

static int COMPARE_CHARSETS(struct console_charset *c1, struct console_charset *c2)
{
	if (__unlikely(c1->width != c2->width)) return 1;
	if (__unlikely(c1->height != c2->height)) return 1;
	if (__unlikely(memcmp(c1->xlate, c2->xlate, sizeof c1->xlate))) return 1;
	if (__unlikely(memcmp(c1->data, c2->data, 256 * sizeof(unsigned long) * LONGS_PER_CHAR(c1->width, c1->height)))) return 1;
	return 0;
}

__COLD_ATTR__ static void UNREF_CHARSET(struct console_charset *c)
{
	if (c == &default_charset) return;
	if (__unlikely(!c->refcount)) KERNEL$SUICIDE("UNREF_CHARSET: UNREFERENCING CHARSET WITH NO REFERENCES");
	if (!--c->refcount) free(c);
}

static const HANDLE_OPERATIONS console_operations = {
	SPL_X(SPL_TTY),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,	/* clone -- default */
	console_lookup,	/* lookup -- none */
	NULL,	/* create -- none */
	NULL,	/* delete -- none */
	NULL,	/* rename -- none */
	NULL,	/* lookup io -- none */
	NULL,	/* instantiate */
	NULL,	/* leave */
	NULL,	/* detach */
	NULL,	/* close */
	KERNEL$NO_OPERATION,
	console_write,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	console_ioctl,

	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
};

__COLD_ATTR__ static void console_init_root(HANDLE *h, void *dummy)
{
	h->op = &console_operations;
	h->flags = 0;
	bell_timer.fn = bell_timer_fn;
	SET_TIMER_NEVER(&bell_timer);
}

__COLD_ATTR__ static void cons_done(void)
{
	if (del_upd) KERNEL$DEL_TIMER(&delayed_update), del_upd = 0;
	cons_update();
	update_cursor();
	KERNEL$DEL_TIMER(&bell_timer);
	WQ_WAKE_ALL(&vc_switch_wait);
}
