#include <SPAD/LIBC.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <FCNTL.H>
#include <UNISTD.H>
#include <STDLIB.H>
#include <SPAD/IOCTL.H>
#include <SPAD/DL.H>
#include <SPAD/CHARSET.H>
#include <SPAD/TTY.H>

#include "VT100GR.I"
#include "FONTFMT.H"
#include "../CONSCTRL.H"

#define FONT_PREFIX		"ETC.:/FONT/"
#define FONT_SUFFIX		".FNT"
#define FONT_DEFAULT_VGA	"9X16"
#define FONT_DEFAULT_EGA	"8X16"

static const char *charset_name = NULL;
static const char *fontfile = NULL;
static int verbose = 1, dflt = 0;

#define N_FRAME_TYPES	4
#define N_FRAME_CHARS	11

/* !!! TODO:
	allow mapping only letters in specific language
*/

static const unsigned short frames[N_FRAME_TYPES][N_FRAME_CHARS] = {
	{ 0x250c, 0x252c, 0x2510, 0x251c, 0x253c, 0x2524, 0x2514, 0x2534, 0x2518, 0x2500, 0x2502 },
	{ 0x2554, 0x2566, 0x2557, 0x2560, 0x256c, 0x2563, 0x255a, 0x2569, 0x255d, 0x2550, 0x2551 },
	{ 0x2552, 0x2564, 0x2555, 0x255e, 0x256a, 0x2561, 0x2558, 0x2567, 0x255b, 0x2550, 0x2502 },
	{ 0x2553, 0x2565, 0x2556, 0x255f, 0x256b, 0x2562, 0x2559, 0x2568, 0x255c, 0x2500, 0x2551 },
};

static const char * const frame_desc[N_FRAME_TYPES] = {
	"SINGLE LINE",
	"DOUBLE LINE",
	"SINGLE-VERTICAL DOUBLE-HORIZONTAL",
	"DOUBLE-VERTICAL SINGLE-HORIZONTAL",
};

static int h;
static CHARSET_8B *charset, *cp437;
static FALLBACK *fallback;
static struct font_header fh;
static int fontmode;
static char vga_9bit;
static unsigned bmp_slots;

struct char_in_font {
	LIST_ENTRY list;
	__s32 uni;
	off_t off;
	struct char_in_font *same;
	unsigned long data[1];
};

#define UNI_HASH_SIZE		4096
#define UNI_HASH(u)		((u) & (UNI_HASH_SIZE - 1))

static XLIST_HEAD uni_hash[UNI_HASH_SIZE];

static struct char_in_font *find_char(__s32 uni)
{
	struct char_in_font *ch;
	XLIST_FOR_EACH(ch, &uni_hash[UNI_HASH(uni)], struct char_in_font, list) if (__likely(ch->uni == uni)) return ch;
	return NULL;
}

static const unsigned short vga_classes_limit[3] = { 256, 224, 32 };
static unsigned short vga_classes[3];
static struct char_in_font *class_chars[3][256];
static unsigned short vga_classes_backup[3];
static struct char_in_font *class_chars_backup[3][256];

#define err_not_found_in_font		0
#define err_invalid_vga_shape		1
#define err_vga_noframe_class_overflow	2
#define err_vga_frame_class_overflow	3
#define err_vga_charset_overflow	4
#define err_n				5

static const char * const err_desc[err_n][2] = {
	"IT WAS NOT FOUND IN FONT", "THEY WERE NOT FOUND IN FONT",
	"IT HAS INVALID SHAPE FOR VGA 9-PIXEL FONT GENERATOR", "THEY HAVE INVALID SHAPE FOR VGA 9-PIXEL FONT GENERATOR",
	"OF VGA NOFRAME CLASS OVERFLOW", "OF VGA NOFRAME CLASS OVERFLOW",
	"OF VGA FRAME CLASS OVERFLOW", "OF VGA FRAME CLASS OVERFLOW",
	"OF VGA CHARSET OVERFLOW", "OF VGA CHARSET OVERFLOW",
};

static unsigned short err[err_n];

static void map_character_uni(__s32 uni);

static void map_character(CHARSET_8B *charset, __u8 chr)
{
	__s32 uni = CHARSET$8B2UNI(charset, chr);
	map_character_uni(uni);
}

static void map_character_uni(__s32 uni)
{
	struct char_in_font *ch;
	unsigned char vga_class;
	unsigned i;
	KERNEL$THREAD_MAY_BLOCK();
	if (__unlikely(uni == -1)) return;
	ch = find_char(uni);
	if (__unlikely(!ch)) {
		err[err_not_found_in_font]++;
		return;
	}
	vga_class = 0;
	if (vga_9bit) {
		unsigned y;
		unsigned char class1 = 0, class2 = 0;
		for (y = 0; y < fh.height; y++) {
			char last = __BT(ch->data, y * fh.width + (fh.width - 1));
			char prelast = __BT(ch->data, y * fh.width + (fh.width - 2));
			if (__likely(!last)) {
				if (__unlikely(prelast)) class1 = 1;
			} else {
				if (__likely(prelast)) {
					class2 = 1;
				} else {
					invl_shape:
					err[err_invalid_vga_shape]++;
					return;
				}
			}
		}
		if (__unlikely((vga_class = class1 + class2 * 2) == 3)) goto invl_shape;
	}
	for (i = 0; i < vga_classes[vga_class]; i++) if (__unlikely(class_chars[vga_class][i] == ch)) return;
	for (i = 0; i < vga_classes[vga_class]; i++) if (__unlikely(!memcmp(class_chars[vga_class][i]->data, ch->data, bmp_slots * sizeof(unsigned long)))) {
		ch->same = class_chars[vga_class][i];
		return;
	}
	if (__unlikely(vga_classes[0] + vga_classes[1] + vga_classes[2] >= 256)) goto charset_over;
	if (__unlikely(vga_classes[vga_class] >= vga_classes_limit[vga_class])) {
		if (vga_class == 1) err[err_vga_noframe_class_overflow]++;
		if (vga_class == 2) err[err_vga_frame_class_overflow]++;
		if (vga_class == 0) charset_over: err[err_vga_charset_overflow]++;
		return;
	}
	class_chars[vga_class][vga_classes[vga_class]++] = ch;
}

static void save_state(void)
{
	memcpy(vga_classes_backup, vga_classes, sizeof vga_classes);
	memcpy(class_chars_backup, class_chars, sizeof class_chars);
}

static void restore_state(void)
{
	memcpy(vga_classes, vga_classes_backup, sizeof vga_classes);
	memcpy(class_chars, class_chars_backup, sizeof class_chars);
}

static int is_frame_uni_char(__s32 uni)
{
	unsigned k, l;
	for (k = 0; k < N_FRAME_TYPES; k++) for (l = 0; l < N_FRAME_CHARS; l++) if (__unlikely(frames[k][l] == uni)) return 1;
	return 0;
}

static int is_vt100_char(__s32 uni)
{
	unsigned i;
	if (__unlikely(uni < 0)) return 0;
	for (i = 0; i < 256; i++) if (__unlikely(vt100gr[i] == uni)) return 1;
	return 0;
}

static struct char_in_font *result[256];
static int empty_char;

static int try_place(struct char_in_font *ch, int cls, int pos)
{
	if (__unlikely((unsigned)pos >= 0x100)) return 0;
	if (__unlikely(result[pos] != NULL)) return 0;
	if (cls && (cls == 2) ^ (pos >= 192 && pos < 224)) return 0;
	result[pos] = ch;
	if (empty_char < 0) {
		unsigned i;
		for (i = 0; i < bmp_slots; i++) if (__likely(ch->data[i])) goto noempty;
		empty_char = pos;
		noempty:;
	}
	return 1;
}

static void place_char(struct char_in_font *ch, int cls)
{
	__u8 c;
	int pref_pos;
	pref_pos = CHARSET$UNI28B(cp437, ch->uni);
	if (__likely(try_place(ch, cls, pref_pos))) return;
	pref_pos = CHARSET$UNI28B(charset, ch->uni);
	if (try_place(ch, cls, pref_pos)) return;
	c = 0x80;
	do {
		KERNEL$THREAD_MAY_BLOCK();
		if (try_place(ch, cls, c)) return;
	} while (++c != 0x80);
	KERNEL$SUICIDE("SETFONT: CAN'T PLACE CHARACTER %08X IN CLASS %d", ch->uni, cls);
}

#define place_passes	4

static __finline__ unsigned place_pass(__s32 uni)
{
	if (uni >= 0x20 && uni < 0x7f) return 0;
	if (is_frame_uni_char(uni)) return 1;
	if (uni >= 0xa0) return 2;
	return 3;
}

static void place_class(int cls)
{
	unsigned i;
	unsigned idx;
	for (i = 0; i < place_passes; i++) for (idx = 0; idx < vga_classes[cls]; idx++) {
		struct char_in_font *ch = class_chars[cls][idx];
		if (place_pass(ch->uni) == i) place_char(ch, cls);
	}
}

static __u8 xlate[4][256];

static __u8 find_xlate(__s32 uni)
{
	struct char_in_font *ch;
	unsigned order;
	unsigned i;
	__s32 *fb;
	KERNEL$THREAD_MAY_BLOCK();
	if (__unlikely(uni < 0)) return empty_char;
	for (i = 0; i < 0x100; i++) if (result[i] && result[i]->uni == uni) return i;
	ch = find_char(uni);
	if (ch->same) for (i = 0; i < 0x100; i++) if (result[i] && result[i]->uni == ch->same->uni) return i;
	for (order = 0; (fb = CHARSET$GET_FALLBACK(fallback, uni, order)); order++) if (fb[0] != -1 && fb[1] == -1) {
		for (i = 0; i < 0x100; i++) if (result[i] && result[i]->uni == fb[0]) return i;
	}
	return empty_char;
}

static int SET_BLOB(int h, FBLOB *blob)
{
	/* copied to SETKBD.C */
	int r;
	char *str, *d, *a;
	if (__unlikely(h < 0)) return -EBADF;
	str = KERNEL$HANDLE_PATH(h);
	if (__unlikely(!str)) return -errno;
	d = strchr(str, ':');
	if (__unlikely(!d)) KERNEL$SUICIDE("SET_BLOB: INVALID HANDLE %d PATH: %s", h, str);
	a = alloca(d - str + 1);
	*(char *)mempcpy(a, str, d - str) = 0;
	r = KERNEL$SET_LOGICAL(a, 0, 0, NULL, NULL, blob);
	write(h, "", 0);
	return r;
}

static void RESET_KBD(void)
{
	char kbd_name[32];
	IOCTLRQ io;
	io.h = KERNEL$STDIN();
	if (__likely(io.h >= 0)) {
		io.ioctl = IOCTL_TTY_GET_KEYBOARD_NAME;
		io.param = 0;
		io.v.ptr = (unsigned long)kbd_name;
		io.v.len = sizeof(kbd_name);
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (__likely(io.status >= 0) && *kbd_name) {
			DLIMGRQ img;
			const char *a[3];
			a[0] = img.filename;
			a[1] = kbd_name;
			a[2] = NULL;
			img.argv = a;
			img.cwd = KERNEL$CWD();
			img.std_in = KERNEL$STDIN();
			img.std_out = KERNEL$STDOUT();
			img.std_err = KERNEL$STDERR();
			img.device = 0;
			strcpy(img.filename, "SETKBD.EXE");
			SYNC_IO_CANCELABLE(&img, KERNEL$DL_LOAD_IMAGE);
		}
	}
}

static const struct __param_table params[7] = {
	"", __PARAM_STRING, 1, __MAX_STR_LEN,
	"", __PARAM_STRING, 1, __MAX_STR_LEN,
	"QUIET", __PARAM_BOOL, ~0, 0,
	"VERBOSE", __PARAM_BOOL, ~0, 2,
	"SYSDEFAULT", __PARAM_BOOL, ~0, 1,
	"DEFAULT", __PARAM_BOOL, ~0, 2,
	NULL, 0, 0, 0,
};

static void *const vars[7] = {
	&charset_name,
	&fontfile,
	&verbose,
	&verbose,
	&dflt,
	&dflt,
	NULL,
};

int main(int argc, const char * const argv[])
{
	int i, f;
	int r, r1, r2;
	int state = 0;
	const char * const *arg = argv;
	struct letter_header *lh;
	IOCTLRQ io;
	struct font_blob *blob;
	char *cn;
	const char *blobfile;
	for (i = 0; i < UNI_HASH_SIZE; i++) INIT_XLIST(&uni_hash[i]);
	memset(vga_classes, 0, sizeof vga_classes);
	if (__unlikely(__parse_params(&arg, &state, params, vars, NULL, NULL, NULL))) {
		badsyn:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}
	if (__unlikely(dflt)) {
		if (__unlikely(charset_name != NULL) || __unlikely(fontfile != NULL)) goto badsyn;
		blob = __sync_calloc(1, sizeof(FBLOB));
		if (__unlikely(!blob)) {
			r = -errno;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T ALLOC MEMORY: %s", strerror(-r));
			goto ret0;
		}
		blob->size = dflt == 1 ? 0 : sizeof(FBLOB);
		blob->type = BLOB_TTY_FONT;
		blob->l_refcount = 1;
		SET_BLOB(KERNEL$STDIN(), (FBLOB *)blob);
		SET_BLOB(KERNEL$STDOUT(), (FBLOB *)blob);
		SET_BLOB(KERNEL$STDERR(), (FBLOB *)blob);
		KERNEL$UNREF_BLOB((FBLOB *)blob);
		RESET_KBD();
		r = 0;
		goto ret0;
	}
	if (__unlikely(!charset_name)) goto badsyn;

	io.h = KERNEL$STDOUT();
	io.ioctl = IOCTL_TTY_GET_FONT_MODE;
	try_stderr:
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	if (__likely(io.h >= 0)) SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);
	else io.status = -EBADF;
	if (__unlikely(io.status < 0)) {
		r = io.status;
		if (r != -EINTR) {
			if (io.h != KERNEL$STDERR()) {
				io.h = KERNEL$STDERR();
				goto try_stderr;
			}
			if (r == -EBADF) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: NO CONTROL TERMINAL"), r = -EINVAL;
			else _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: STDOUT AND STDERR ARE NOT TERMINALS: %s", strerror(-r));
		}
		goto ret0;
	}
	if (__unlikely(io.status == RESULT_GET_FONT_MODE_NONE)) {
		r = -EOPNOTSUPP;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: TERMINAL DOESN'T SUPPORT FONT LOAD");
		goto ret0;
	}
	if (io.status & RESULT_GET_FONT_MODE_ANY) fontmode = RESULT_GET_FONT_MODE_ANY;
	else if (io.status & RESULT_GET_FONT_MODE_VGA) fontmode = RESULT_GET_FONT_MODE_VGA;
	else if (io.status & RESULT_GET_FONT_MODE_EGA) fontmode = RESULT_GET_FONT_MODE_EGA;
	else {
		r = -EINVAL;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: TERMINAL REPORTS UNKNOWN FONT LOAD METHOD %lX", io.status);
		goto ret0;
	}

	blobfile = "";
	if (!fontfile) {
		static char fntfile[__MAX_STR_LEN];
		io.ioctl = IOCTL_TTY_GET_FONT_FILE;
		io.param = 0;
		io.v.ptr = (unsigned long)&fntfile;
		io.v.len = sizeof fntfile;
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);
		if (!io.status) {
			fontfile = fntfile;
			if (fontfile[0] && !strpbrk(fontfile, ":/.")) goto file_ok;
		}
		if (fontmode == RESULT_GET_FONT_MODE_EGA) fontfile = FONT_PREFIX FONT_DEFAULT_EGA FONT_SUFFIX;
		else fontfile = FONT_PREFIX FONT_DEFAULT_VGA FONT_SUFFIX;
		blobfile = fontfile;
	} else if (!strpbrk(fontfile, ":/.")) {
		char *fnt;
		file_ok:
		blobfile = fontfile;
		fnt = alloca(strlen(fontfile) + strlen(FONT_PREFIX) + strlen(FONT_SUFFIX) + 1);
		stpcpy(stpcpy(stpcpy(fnt, FONT_PREFIX), fontfile), FONT_SUFFIX);
		fontfile = fnt;
	}

	h = open(fontfile, O_RDONLY);
	if (__unlikely(h == -1)) {
		r = -errno;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T OPEN FONT FILE %s: %s", fontfile, strerror(-r));
		goto ret0;
	}
	errno = EFTYPE;
	if (__unlikely(pread(h, &fh, sizeof fh, 0) != sizeof fh)) {
		r = -errno;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T READ FONT FILE HEADER: %s", strerror(-r));
		goto ret1;
	}
	if (__unlikely(memcmp(fh.signature, FONT_SIGNATURE, sizeof fh.signature))) {
		r = -EFTYPE;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: NOT A FONT FILE");
		goto ret1;
	}
	if (__unlikely(!fh.width) || __unlikely(!fh.height) || __unlikely(fh.letter_size < (sizeof(struct letter_header) + fh.width + fh.height + 7) / 8)) {
		r = -EFTYPE;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: INVALID FONT HEADER");
		goto ret1;
	}
	bmp_slots = LONGS_PER_CHAR(fh.width, fh.height);
	lh = alloca(fh.letter_size);
	for (i = 0; i < __32LE2CPU(fh.n_entries_le); i++) {
		unsigned o, px;
		struct char_in_font *ch;
		off_t off = sizeof(struct font_header) + fh.reserved_bytes + (off_t)i * fh.letter_size;
		errno = EFTYPE;
		if (__unlikely(pread(h, lh, fh.letter_size, off) != fh.letter_size)) {
			r = -errno;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T READ CHARACTER %d: %s", i, strerror(-r));
			goto ret1;
		}
		ch = find_char(__32LE2CPU(lh->uni_le));
		if (__unlikely(ch != NULL)) {
			r = -EFTYPE;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: DUPLICATE CHARACTER FOR UNICODE %X", (unsigned)__32LE2CPU(lh->uni_le));
			goto ret1;
		}
		ch = __sync_calloc(1, sizeof(struct char_in_font) + (bmp_slots - 1) * sizeof(unsigned long));
		if (__unlikely(!ch)) {
			r = -errno;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T ALLOC MEMORY: %s", strerror(-r));
			goto ret1;
		}
		ch->uni = __32LE2CPU(lh->uni_le);
		ch->off = off;
		ADD_TO_XLIST(&uni_hash[UNI_HASH(ch->uni)], &ch->list);
		o = 0;
		px = fh.height * fh.width;
		for (o = 0; o < px; o++)
			if (((__u8 *)(lh + 1))[o / 8] & (1 << (~o & 7))) __BS(ch->data, o);
	}
	cn = CHARSET$RESOLVE_ALIAS(charset_name);
	if (__unlikely(!cn)) {
		r = -errno;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T RESOLVE ALIAS %s: %s", charset_name, strerror(-r));
		goto ret1;
	}
	if (__unlikely(!(charset = CHARSET$OPEN_8B(cn)))) {
		r = -errno;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T OPEN CHARSET %s: %s", charset_name, strerror(-r));
		goto ret15;
	}
	if (__unlikely(!(cp437 = CHARSET$OPEN_8B("CP437")))) {
		r = -errno;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T OPEN CHARSET CP437: %s", strerror(-r));
		goto ret2;
	}
	fallback = CHARSET$OPEN_FALLBACK(NULL);

	vga_9bit = 0;
	if (__likely(fontmode == RESULT_GET_FONT_MODE_VGA)) {
		if (__unlikely(fh.width == 8)) vga_9bit = 0;
		else if (__likely(fh.width == 9)) vga_9bit = 1;
		else {
			r = -EINVAL;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: VGA CAN ONLY USE 8 OR 9 PIXELS WIDE FONTS");
			goto ret3;
		}
	} else if (__unlikely(fontmode == RESULT_GET_FONT_MODE_EGA)) {
		if (__unlikely(fh.width != 8)) {
			r = -EINVAL;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: EGA CAN ONLY USE 8 PIXELS WIDE FONTS");
			goto ret3;
		}
	}
	memset(err, 0, sizeof err);
	for (i = 0x20; i < 0x100; i++) map_character(charset, i);
	for (i = 0; i < 0x20; i++) if (__likely(!(CTRL_ALWAYS & (1 << i)))) map_character(charset, i);
	for (i = 0; i < err_n; i++) if (__unlikely(err[i])) {
		if (verbose >= 1) _eprintf("%d CHARACTER%s NOT MAPPED BECAUSE %s\n", err[i], err[i] == 1 ? " WAS" : "S WERE", err_desc[i][err[i] != 1]);
	}
	for (f = 0; f < N_FRAME_TYPES; f++) {
		memset(err, 0, sizeof err);
		save_state();
		for (i = 0; i < N_FRAME_CHARS; i++) map_character_uni(frames[f][i]);
		for (i = 0; i < err_n; i++) if (__unlikely(err[i])) {
			if (verbose >= 2) _eprintf("UNABLE TO MAP %s FRAMES BECAUSE %s\n", frame_desc[f], err_desc[i][1]);
			restore_state();
			break;
		}
		if (__unlikely(!f)) {
			memset(err, 0, sizeof err);
			for (i = 0; i < 256; i++) if (__unlikely(vt100gr[i] >= 0)) {
				if (is_frame_uni_char(vt100gr[i])) continue;
				map_character_uni(vt100gr[i]);
			}
			for (i = 0; i < err_n; i++) if (__unlikely(err[i])) {
				if (verbose >= 2) _eprintf("%d VT100 GRAPHICS CHARACTER%s NOT MAPPED BECAUSE %s\n", err[i], err[i] == 1 ? " WAS" : "S WERE", err_desc[i][err[i] != 1]);
			}
		}
	}
	memset(err, 0, sizeof err);
	for (i = 0; i < 0x100; i++) if (__likely(i >= 0x20) || __likely(!(CTRL_ALWAYS & (1 << i)))) {
		__s32 uni = CHARSET$8B2UNI(cp437, i);
		if (__unlikely(uni < 0)) continue;
		if (__unlikely(is_frame_uni_char(uni)) || __unlikely(is_vt100_char(uni))) continue;
		map_character_uni(uni);
	}
	for (i = 0; i < err_n; i++) if (__unlikely(err[i])) {
		if (verbose >= 2) _eprintf("%d CP437 CHARACTER%s NOT MAPPED BECAUSE %s\n", err[i], err[i] == 1 ? " WAS" : "S WERE", err_desc[i][err[i] != 1]);
	}
	/*_eprintf("%d %d %d\n", vga_classes[0], vga_classes[1], vga_classes[2]);*/

	empty_char = -1;
	memset(result, 0, sizeof result);
	place_class(1);
	place_class(2);
	place_class(0);
	if (__unlikely(empty_char < 0)) {
		if (verbose >= 1) _eprintf("NO SPACE CHARACTER MAPPED");
		for (i = 0; i < 0x100; i++) if (result[i] && result[i]->uni == 0x20) {
			empty_char = i;
			goto have_empty;
		}
		for (i = 0; i < 0x100; i++) if (result[i] && result[i]->uni == 0xa0) {
			empty_char = i;
			goto have_empty;
		}
		empty_char = 0x20;
		have_empty:;
	}
	for (i = 0; i < 0x100; i++) {
		__s32 uni = CHARSET$8B2UNI(charset, i);
		xlate[0][i] = find_xlate(uni);
		if (__unlikely(vt100gr[i] >= 0)) xlate[1][i] = find_xlate(vt100gr[i]);
		else xlate[1][i] = xlate[0][i];
		uni = CHARSET$8B2UNI(cp437, i);
		xlate[2][i] = find_xlate(uni);

		if (result[i]) {
			int val;
			uni = result[i]->uni;
			val = CHARSET$UNI28B_FALLBACK(charset, fallback, uni);
			if (val == -1) val = ' ';
			xlate[3][i] = val;
		} else {
			xlate[3][i] = ' ';
		}
	}
	
	blob = __sync_calloc(1, sizeof(struct font_blob) + (256 * bmp_slots - 1) * sizeof(unsigned long));
	if (__unlikely(!blob)) {
		r = -errno;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: CAN'T ALLOC MEMORY: %s", strerror(-r));
		goto ret3;
	}
	blob->size = sizeof(struct font_blob) + (256 * bmp_slots - 1) * sizeof(unsigned long);
	blob->type = BLOB_TTY_FONT;
	blob->l_refcount = 1;
	{
		char *cn_end;
		char *cn_base = strrchr(cn, '/');
		if (!cn_base) cn_base = strrchr(cn, ':');
		else cn_base++;
		if (!cn_base) cn_base = cn;
		else cn_base++;
		cn_end = strrchr(cn_base, '.');
		if (!cn_end) cn_end = cn_base + strlen(cn_base);
		if (__likely(cn_end - cn_base < sizeof(blob->charset))) {
			*(char *)mempcpy(blob->charset, cn_base, cn_end - cn_base) = 0;
			__upcase(blob->charset);
		}
	}
	if (__likely(strlen(blobfile) < sizeof(blob->fontfile)))
		strcpy(blob->fontfile, blobfile);
	blob->w = fh.width;
	blob->h = fh.height;
	memcpy(blob->xlate, xlate, sizeof blob->xlate);
	for (i = 0; i < 0x100; i++) {
		if (__likely(result[i] != NULL)) memcpy(blob->data + i * bmp_slots, result[i]->data, bmp_slots * sizeof(unsigned long));
	}
	r1 = SET_BLOB(KERNEL$STDOUT(), (FBLOB *)blob);
	if (__unlikely(r1 == -EINTR)) {
		r = -EINTR;
		goto ret4;
	}
	r2 = SET_BLOB(KERNEL$STDERR(), (FBLOB *)blob);
	if (r1 && r2 && r2 != -EINTR) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETFONT: COULD NOT SET BLOB FOR NEITHER STDOUT(%s) NOR STDERR(%s)", strerror(-r1), strerror(-r2));
		r = r1;
		goto ret4;
	}
	SET_BLOB(KERNEL$STDIN(), (FBLOB *)blob);
	/*__debug_printf("%d %d %d\n", vga_classes[0], vga_classes[1], vga_classes[2]);*/
	RESET_KBD();

	r = 0;
	ret4:
	KERNEL$UNREF_BLOB((FBLOB *)blob);
	ret3:
	CHARSET$CLOSE_FALLBACK(fallback, 0);
	CHARSET$CLOSE_8B(cp437, 0);
	ret2:
	CHARSET$CLOSE_8B(charset, 0);
	ret15:
	free(cn);
	ret1:
	close(h);
	ret0:
	for (i = 0; i < UNI_HASH_SIZE; i++) while (!XLIST_EMPTY(&uni_hash[i])) {
		struct char_in_font *ch = LIST_STRUCT(uni_hash[i].next, struct char_in_font, list);
		DEL_FROM_LIST(&ch->list);
		free(ch);
	}
	return r;
}
