#include <ARCH/SETUP.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV.H>
#include <SPAD/TIMER.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/VM.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/IO.H>
#include <ARCH/MOV.H>
#include <SPAD/CMOS.H>
#include <SPAD/CONSOLE.H>
#include <SPAD/TTY.H>

#include "VESA.H"
#include "CONS.H"
#include "VGAREG.H"

#define DELAYED_UPDATE_FREQ	25

#define CARD_MDA	0
#define CARD_CGA	1
#define CARD_EGA	2
#define CARD_VGA	3

c_char *base;
int cols;
int rows;
char console_gfx = 0;
char on_kernel_cons = 0;
static char card_type;
int can_do_color;
static int can_do_vesa;
static int snow_hack = 0;
static unsigned long port;

static int ring_size = 512;
int vcs;

static c_char apply_softcursor(c_char b);
static void cons_update(void);
static int cons_getattr(int color, int flags);
static void cons_write(int x, int y, int chr);
static void cons_setcursor(int x, int y, int type);
static void cons_clear_cached(void);

struct console_charset {
/* xlate: 0: normal, 1: VT100 graphics, 2: ESC[11m (cp437), 3: reverse normal, for mouse copying */
	unsigned char xlate[4][256];
	unsigned refcount;
	__u8 width;
	__u8 height;
	unsigned long data[1];
};

static struct console_charset *console_charsets[MAX_VCS];

static const struct console_charset default_charset = {
#include "PCC_MAP.I"
	, 0, 0, 0, { 0 }
};

int vga_retrace_errors = 0;

int cons_wait_for_v_retrace(void)
{
	u_jiffies_lo_t j, jj;
	/* the problem here is that we don't know if we missed the retrace
	   because of many interrupts or the controller doesn't support retrace.
	   Rising SPL high while waiting is not acceptable. We tolerate few
	   timeouts and then disable wait retrace for 1000 frames and try to
	   wait one frame again */
	if (__unlikely(vga_retrace_errors > RETRACE_ERROR_TOLERANCE)) {
		vga_retrace_errors--;
		return -EIO;
	}
	j = jj = KERNEL$GET_JIFFIES_LO();
	while (__unlikely(io_inb(port - VGA_CRTC_IDX + VGA_INPUT_REG) & VGA_INPUT_REG_VRETRACE)) {
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(RETRACE_TIMEOUT))) {
			retrace_error:
			vga_retrace_errors++;
			if (vga_retrace_errors == RETRACE_ERROR_TOLERANCE + 1)
				vga_retrace_errors = RETRACE_ERROR_TOLERANCE + RETRACE_TRY_AGAIN;
			return -EIO;
		}
		jj = KERNEL$GET_JIFFIES_LO();
	}
	while (__likely(!(io_inb(port - VGA_CRTC_IDX + VGA_INPUT_REG) & VGA_INPUT_REG_VRETRACE))) {
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(RETRACE_TIMEOUT))) goto retrace_error;
		jj = KERNEL$GET_JIFFIES_LO();
	}
	vga_retrace_errors = 0;
	return 0;
}

#define cons_is_hv_retrace() (io_inb(port - VGA_CRTC_IDX + VGA_INPUT_REG) & (VGA_INPUT_REG_HRETRACE | VGA_INPUT_REG_VRETRACE))

static void cons_wait_for_hv_retrace(void)
{
	u_jiffies_lo_t j, jj;
	if (__likely(cons_is_hv_retrace()))
		goto ret_err_0;
	if (__unlikely(vga_retrace_errors > RETRACE_ERROR_TOLERANCE)) {
		vga_retrace_errors--;
		return;
	}
	j = jj = KERNEL$GET_JIFFIES_LO();
	while (!cons_is_hv_retrace()) {
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(RETRACE_TIMEOUT))) {
			vga_retrace_errors++;
			if (vga_retrace_errors == RETRACE_ERROR_TOLERANCE + 1)
				vga_retrace_errors = RETRACE_ERROR_TOLERANCE + RETRACE_TRY_AGAIN;
			return;
		}
		jj = KERNEL$GET_JIFFIES_LO();
	}
	ret_err_0:
	vga_retrace_errors = 0;
}

static __finline__ void snow_wait_for_v_retrace(void)
{
	if (__unlikely(snow_hack)) {
		cons_wait_for_v_retrace();
	}
}

static __finline__ void snow_check_for_hv_retrace(void)
{
	if (__unlikely(snow_hack) && __unlikely(!cons_is_hv_retrace())) {
		cons_wait_for_v_retrace();
	}
}

static void cons_hw_set_font(const struct console_charset *c);

static __finline__ void cons_write(int x, int y, int chr)
{
	if (__unlikely(console_gfx)) return;
#if __DEBUG >= 2
	if (__unlikely((unsigned)x >= cols) || __unlikely((unsigned)y >= rows)) KERNEL$SUICIDE("cons_write: OUT OF RANGE: x == %d, y == %d", x, y);
#endif
	if (__unlikely(snow_hack)) {
		cons_wait_for_hv_retrace();
	}
	base[x + y * cols] = __16CPU2LE(chr);
}

static c_char invert_char(c_char b)
{
	if (__unlikely(!can_do_color)) return b ^ 0x0800;
	else return ((b) & 0x88ff) | (((b) & 0x7000) >> 4) | (((b) & 0x0700) << 4);
}

static c_char mouse_invert_char(c_char b)
{
	if (__likely(can_do_color) && __unlikely(!(((b >> 8) ^ (b >> 12)) & 7))) {
		return b ^ 0x7700;
	}
	return invert_char(b);
}

static __finline__ c_char drag_invert_char(c_char b)
{
	return mouse_invert_char(b);
}

static __finline__ int cons_validate_font(struct font_blob *font)
{
	if (__unlikely(card_type != CARD_EGA) && __unlikely(card_type != CARD_VGA)) return 0;
	return (font->w == 8 || (__likely(font->w == 9) && __likely(card_type == CARD_VGA))) && __likely(font->h > 0) && __likely(font->h <= 32);
}

#include "CONS.I"

static c_char apply_softcursor(c_char b)
{
	int type = tt[avc].cursor_type;
	/* it is already assumed that type & CURSOR_SOFT */
	c_char orig = b;
	b |= (type >> 8) & 0xff00;
	b ^= type & 0xff00;
	if (type & 0x20 && (orig & 0x7000) == (b & 0x7000)) b ^= 0x7000;
	if (type & 0x40 && (b & 0x700) == ((b & 0x7000) >> 4)) b ^= 0x700;
	if (type & 0x80) b = invert_char(b);
	return b;
}

static void cons_update(void)
{
	int i;
	c_char *a;
	if (__unlikely(console_gfx)) return;
	snow_wait_for_v_retrace();
	for (i = 0, a = base; i < rows; i++, a += cols) {
		if (__unlikely(snow_hack) && __unlikely(!(i & 0x3))) {
			snow_check_for_hv_retrace();
		}
#ifdef __ENDIAN_LITTLE
		if ((__unlikely(mouse_visible) && __unlikely(i == mouse_y))
		 || (__unlikely(drag_visible) && __unlikely(i >= drag_start_y) && __unlikely(i <= drag_end_y))
		 || (__unlikely(tt[avc].cursor_type & CURSOR_SOFT) && __unlikely(i == tt[avc].y + scrollback[avc]))) {
#endif
			update_line_slow(i);
#ifdef __ENDIAN_LITTLE
		} else {
			c_char *line = lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + i)];
			memcpy(a, line, cols * sizeof(c_char));
		}
#endif
	}
}


/*
 * color: low 4 bits - fg, high 4 bits - bg
 * attr: attributes: TTYF_xxx
 */
static int cons_getattr(int color, int flags)
{
	int attr;
	if (!can_do_color) {
		if (!(flags & TTYF_REVERSE)) {
			if (!(flags & TTYF_BRIGHT)) {
				if (!(flags & TTYF_UNDERLINE))
					return 0x07;
				else
					return 0x01;
			} else {
				if (!(flags & TTYF_UNDERLINE))
					return 0x0F;
				else
					return 0x09;
			}
		} else {
			if (!(flags & TTYF_BRIGHT))
				return 0x70;
			else
				return 0x7F;
		}
	}
	attr = color;
	if (flags & TTYF_UNDERLINE) attr = (attr & 0xf0) | t->ul_color;
	else if (flags & TTYF_DIM) attr = (attr & 0xf0) | t->half_color;
	if (flags & TTYF_REVERSE) attr = ((attr >> 4) & 0x0f) | ((attr << 4) & 0xf0);
	if (flags & TTYF_BLINK) attr |= 0x80;
	if (flags & TTYF_BRIGHT) attr |= 0x08;
	return attr;
}

static int lv = -1;
static int lt = -1;

/* x can be == cols */
static void cons_setcursor(int x, int y, int type)
{
	int val;
	if (__unlikely(console_gfx)) return;
	if (on_kernel_cons) {
		KERNEL$CONSOLE_SET_CURSOR(x, y);
	}
	if (__unlikely(x == cols)) x--;
	if (__unlikely((unsigned)y >= (unsigned)rows)) y = rows + 1;
	val = x + y * cols;
	if (__unlikely(type != lt)) {
		int height;
		int from, to;
		KERNEL$CONSOLE_LOCK();
		io_outb_p(port, VGA_CRTC_MAXIMUM_SCAN_LINE);
		height = (io_inb(port + 1) & VGA_CRTC_MAXIMUM_SCAN_LINE_LINE) + 1;
		KERNEL$CONSOLE_UNLOCK();
		lt = type;
		to = height - (height < 10 ? 1 : 2);
		if (type == CURSOR_NONE) {
			from = 31;
			to = 30;
		} else if (type == CURSOR_UNDERLINE) {
			from = height - (height < 10 ? 2 : 3);
		} else if (type == CURSOR_LOWER_THIRD) {
			from = height * 2 / 3;
		} else if (type == CURSOR_LOWER_HALF) {
			from = height / 2;
		} else if (type == CURSOR_TWO_THIRDS) {
			from = height / 3;
		} else {
			from = 1;
			to = height;
		}
		KERNEL$CONSOLE_LOCK();
		io_outb_p(port, VGA_CRTC_CURSOR_START);
		io_outb_p(port + 1, (io_inb_p(port + 1) & 0xc0) | from);
		io_outb_p(port, VGA_CRTC_CURSOR_END);
		io_outb_p(port + 1, (io_inb_p(port + 1) & 0xe0) | to);
		KERNEL$CONSOLE_UNLOCK();
	}
	if (val != lv) {
		lv = val;
		KERNEL$CONSOLE_LOCK();
		io_outb_p(port, VGA_CRTC_CURSOR_HI);
		io_outb_p(port + 1, val >> 8);
		io_outb_p(port, VGA_CRTC_CURSOR_LO);
		io_outb_p(port + 1, val);
		KERNEL$CONSOLE_UNLOCK();
	}
}

static void cons_clear_cached(void)
{
	lv = -1;
	lt = -1;
}

/* Updating videomemory for idiots.
   I really have a S3 AGP card where this is needed. */
#define WRITE_REPEATS	10

static void write_fontmem(volatile __u32 *addr, __u32 value)
{
	int repeat = 0;
	write_again:
	*addr = value;
	if (__unlikely(snow_hack) && (__unlikely(*addr != value) || __unlikely(*addr != value)) && ++repeat < WRITE_REPEATS)
		goto write_again;
}

/* Dumb ATI ES1000 card supports only zero bank */
#define FONT_BANK	0

static __u32 default_font[8192 / 4];

static void save_restore_default_font(int restore)
{
	snow_wait_for_v_retrace();
	if (!restore)
		memcpy(default_font, KERNEL$ZERO_BANK + 0xa0000, sizeof default_font);
	else if (!snow_hack)
		memcpy(KERNEL$ZERO_BANK + 0xa0000, default_font, sizeof default_font);
	else {
		int i;
		for (i = 0; i < sizeof default_font / 4; i++) {
			snow_check_for_hv_retrace();
			write_fontmem((__u32 *)(KERNEL$ZERO_BANK + 0xa0000) + i, default_font[i]);
		}
	}
}

static __u8 misc, current_pixels, wanted_pixels, changing_width, set_default;

static void cons_hw_set_font_restore(void)
{
	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, __unlikely((wanted_pixels ^ current_pixels) & VGA_TS_CLOCKING_MODE_98DM) ? 0 : VGA_TS_RESET_AR);
	io_outb_p(VGA_TS_IDX, VGA_TS_PLANE_WRITE_ENABLE);
	io_outb_p(VGA_TS_REG, 0x03);
	io_outb_p(VGA_TS_IDX, VGA_TS_MEMORY_MODE);
	io_outb_p(VGA_TS_REG, 1 | VGA_TS_MEMORY_MODE_EXT_MEM);
	io_outb_p(VGA_TS_IDX, VGA_TS_CHARACTER_MAP);
	io_outb_p(VGA_TS_REG, set_default ? 0x00 : FONT_BANK + (FONT_BANK * 4));
	if (__unlikely(changing_width)) {
		io_outb_p(VGA_TS_IDX, VGA_TS_CLOCKING_MODE);
		io_outb_p(VGA_TS_REG, (current_pixels & ~VGA_TS_CLOCKING_MODE_98DM) | wanted_pixels);
	}
	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, VGA_TS_RESET_AR | VGA_TS_RESET_SR);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_READ_MAP);
	io_outb_p(VGA_GDC_REG, 0x00);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_GRAPHICS_MODE);
	io_outb_p(VGA_GDC_REG, VGA_GDC_GRAPHICS_MODE_HOST_OE);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_MISC);
	io_outb_p(VGA_GDC_REG, misc);
	if (__unlikely(changing_width)) {
		__u8 atc_ctrl, output;
		output = io_inb_p(VGA_OUTPUT_REG);
		if (__likely(!(output & VGA_OUTPUT_REG_CLOCK_2))) {
			output = (output | VGA_OUTPUT_REG_CLOCK_1) - (wanted_pixels * VGA_OUTPUT_REG_CLOCK_1);
			io_outb_p(VGA_OUTPUT_REG_WRITE, output);
		}
		io_inb_p(port - VGA_CRTC_IDX + VGA_INPUT_REG);
		io_outb_p(VGA_ATC_IDX_REG, VGA_ATC_CONTROL | VGA_ATC_IDX_REG_PAS);
		atc_ctrl = io_inb_p(VGA_ATC_REG_READ);
		atc_ctrl = (atc_ctrl | VGA_ATC_CONTROL_LGE) ^ (wanted_pixels * VGA_ATC_CONTROL_LGE);
		io_outb_p(VGA_ATC_IDX_REG, atc_ctrl);
		io_outb_p(VGA_ATC_IDX_REG, VGA_ATC_H_PIXEL_PANNING | VGA_ATC_IDX_REG_PAS);
		io_outb_p(VGA_ATC_IDX_REG, 8 ^ (wanted_pixels * 8));
	}
}

static void cons_hw_set_font(const struct console_charset *c)
{
#define USER_FONT_ADDR	(0xa0000 + FONT_BANK * 0x4000)
	static char default_saved = 0;

	const unsigned long *fontdata;
	unsigned fontslot, px, skipmask, i;
	__u32 *addr;
	if (__unlikely(console_gfx)) return;
	if (__unlikely(card_type != CARD_EGA) && __unlikely(card_type != CARD_VGA)) return;
	set_default = c == &default_charset;

	KERNEL$CONSOLE_LOCK();
	io_outb_p(VGA_TS_IDX, VGA_TS_CLOCKING_MODE);
	current_pixels = io_inb_p(VGA_TS_REG);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_MISC);
	misc = io_inb_p(VGA_GDC_REG);
	KERNEL$CONSOLE_UNLOCK();

	wanted_pixels = c->width == 8;
	changing_width = (wanted_pixels ^ current_pixels) & VGA_TS_CLOCKING_MODE_98DM;
	if (__unlikely(card_type != CARD_VGA)) changing_width = 0;

	KERNEL$CONSOLE_SET_GRAPHICS(cons_hw_set_font_restore);

	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, VGA_TS_RESET_AR);
	io_outb_p(VGA_TS_IDX, VGA_TS_PLANE_WRITE_ENABLE);
	io_outb_p(VGA_TS_REG, 0x04);
	io_outb_p(VGA_TS_IDX, VGA_TS_MEMORY_MODE);
	io_outb_p(VGA_TS_REG, 1 | VGA_TS_MEMORY_MODE_EXT_MEM | VGA_TS_MEMORY_MODE_OE_DIS);
	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, VGA_TS_RESET_AR | VGA_TS_RESET_SR);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_READ_MAP);
	io_outb_p(VGA_GDC_REG, 0x02);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_GRAPHICS_MODE);
	io_outb_p(VGA_GDC_REG, 0x00);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_MISC);
	io_outb_p(VGA_GDC_REG, VGA_GDC_MISC_MEMORY_MAP_A000_BFFF);

	if (!default_saved) {
		save_restore_default_font(0);
		default_saved = 1;
	}

	if (c != &default_charset) {
		snow_wait_for_v_retrace();
		for (i = 0,
		     fontdata = c->data,
		     px = c->width * c->height,
		     fontslot = LONGS_PER_CHAR(c->width, c->height),
		     addr = (__u32 *)(KERNEL$ZERO_BANK + USER_FONT_ADDR),
		     skipmask = c->width == 9 ? 7 : -1
		     ; i < 256; i++, fontdata += fontslot) {
			unsigned bitpos = 0;
			__u32 value = 0;
			unsigned x;
			for (x = 0; x < px; x++) {
				if (__unlikely(bitpos == 32)) {
					write_fontmem(addr, __32CPU2LE(value));
					addr++;
					value = 0, bitpos = 0;
				}
				value |= __BT(fontdata, x) << (bitpos ^ 7);
				bitpos++;
				x += __unlikely(!(bitpos & skipmask));
			}
			snow_check_for_hv_retrace();
			write_fontmem(addr, __32CPU2LE(value));
			addr = (__u32 *)(((unsigned long)addr + 32) & ~31UL);
		}
	} else {
		save_restore_default_font(1);
	}
	__write_barrier2();
	cons_hw_set_font_restore();
	KERNEL$CONSOLE_SET_GRAPHICS(NULL);
#undef USER_FONT_ADDR
}

__COLD_ATTR__ static void probe_hw_console(void)
{
	int equip;
	can_do_vesa = 0;
	can_do_color = *(__u8 *)(KERNEL$ZERO_BANK + 0x449) != 7;
	base = (void *)(*(__u8 *)(KERNEL$ZERO_BANK + 0x449) == 7 ? 0xb0000 : 0xb8000);
	port = (*(__u8 *)(KERNEL$ZERO_BANK + 0x449) == 7 ? VGA_BASE_MONO + VGA_CRTC_IDX : VGA_BASE_COLOR + VGA_CRTC_IDX);
	cols = *(__u16 *)(KERNEL$ZERO_BANK + 0x44a);
	rows = *(__u8 *)(KERNEL$ZERO_BANK + 0x484) + 1;
	if (rows <= 1 || rows >= 101) rows = 25;
	if (cols <= 1 || cols >= 301) cols = 80;
		/* test for MDA/CGA/EGA/VGA, method from FreeBSD>=4 */
	equip = CMOS$READ(0x14);
	if (equip >= 0 && (equip >> 4) & 3) {
		if (!can_do_color) card_type = CARD_MDA;
		else card_type = CARD_CGA;
	} else {
		__u8 cursor;
		card_type = CARD_EGA;
		io_outb_p(port, VGA_CRTC_CURSOR_START);
		cursor = io_inb_p(port + 1);
		io_outb_p(port + 1, cursor | 0x20);
		if (io_inb_p(port + 1) != (cursor | 0x20)) goto ega;
		io_outb_p(port + 1, cursor & ~0x20);
		if (io_inb_p(port + 1) != (cursor & ~0x20)) goto ega;
		card_type = CARD_VGA;
		can_do_vesa = 1;
		ega:
		io_outb_p(port + 1, cursor);
	}
}

/*
ARGS:	[BASE=#B8000]
	[PORT=#3D4]
	[COLS=80]
	[ROWS=25]
	[COLOR]
	[MONO]
*/

char dev_name[__MAX_STR_LEN];

static char port_locked = 0;
static char vga_locked = 0;
static IO_RANGE port_range;
static IO_RANGE vga_range;

static void *lnte, *dlrq;

static const struct __param_table console_params[13] = {
	"BASE", __PARAM_PTR, 0xa0000, 0x100000,
	"PORT", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - 1,
	"COLS", __PARAM_INT, 2, 301,
	"ROWS", __PARAM_INT, 2, 101,
	"COLOR", __PARAM_BOOL, ~0, 1,
	"MONO", __PARAM_BOOL, ~0, 0,
	"NOVESA", __PARAM_BOOL, ~0, 0,
	"SNOW", __PARAM_BOOL, ~0, 1,
	"SCROLLBACK", __PARAM_INT, 1, MAX_RING_SIZE + 1,
	"VC", __PARAM_INT, 1, MAX_VCS + 1,
	"MAX_V_REFRESH", __PARAM_UNSIGNED_INT, 50, 10000,
	"MAX_H_REFRESH", __PARAM_UNSIGNED_INT, 30, 10000,
	NULL, 0, 0, 0,
};

static void *const vars[13] = {
	&base,
	&port,
	&cols,
	&rows,
	&can_do_color,
	&can_do_color,
	&can_do_vesa,
	&snow_hack,
	&ring_size,
	&vcs,
	&max_v_refresh,
	&max_h_refresh,
	NULL,
};

static int unload(void *p, void **release, const char * const argv[]);

__COLD_ATTR__ int main(int argc, const char * const argv[])
{
	int r;
	int i, j;
	unsigned x, y;
	MALLOC_REQUEST mrq;
	const char * const *arg = argv;
	int state = 0;
	u_jiffies_lo_t j1, j2;
	int jc;

	vcs = MAX_VCS;
	probe_hw_console();
	if (__parse_params(&arg, &state, console_params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: SYNTAX ERROR");
		return -EBADSYN;
	}
	_snprintf(dev_name, __MAX_STR_LEN, "PCCONS@" MEM_ID, (unsigned long)base);

	if ((unsigned long)base == KERNEL$CONSOLE_BASE_PHYS_ADDR())
		on_kernel_cons = 1;

	port_range.start = port - 4;
	port_range.len = 8;
	port_range.name = dev_name;
/* port may be already locked by the kernel --- so allow failure */
	if (!KERNEL$REGISTER_IO_RANGE(&port_range)) port_locked = 1;

	if (can_do_vesa) {
		vga_range.start = VGA_BASE;
		vga_range.len = 0x10;
		vga_range.name = dev_name;
		if (KERNEL$REGISTER_IO_RANGE(&vga_range)) goto skip_vesa;
		vga_locked = 1;
		if (PROBE_VESA()) {
			can_do_vesa = 0;
			KERNEL$UNREGISTER_IO_RANGE(&vga_range);
			vga_locked = 0;
		}
	}
	skip_vesa:
	if (ring_size < rows || (ring_size & (ring_size - 1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: BAD SCROLLBACK (MUST BE POWER OF 2 AND GREATER THAN NUMBER OF ROWS)");
		r = -EBADSYN;
		goto ret0;
	}

	memset(lines, 0, sizeof(lines));
	for (i = 0; i < vcs; i++) for (j = 0; j < ring_size; j++) {
		mrq.size = sizeof(c_char) * cols;
		SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(mrq.status < 0)) {
			if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: COULD NOT ALLOCATE LINES BUFFER");
			r = mrq.status;
			goto ret1;
		}
		memset(mrq.ptr, 0, mrq.size);
		lines[i][j] = mrq.ptr;
	}

	avc = 0;
	for (i = 0; i < MAX_VCS; i++) console_charsets[i] = (struct console_charset *)&default_charset;
	for (i = 0; i < vcs; i++) {
		KERNEL$THREAD_MAY_BLOCK();
		set_write_vc(i), tty_init();
	}
	set_write_vc(0);
	base = (void *)((char *)base + (unsigned long)KERNEL$ZERO_BANK);
	for (y = 0; y < rows; y++)
		for (x = 0; x < cols; x++)
			lines[0][RING_MASK(scrpos[0] + y)][x] = __16LE2CPU(base[x + y * cols]);

	io_outb_p(port, VGA_CRTC_CURSOR_HI);
	x = io_inb_p(port + 1);
	io_outb_p(port, VGA_CRTC_CURSOR_LO);
	x = (x << 8) + io_inb_p(port + 1);
	t->x = x % cols;
	if ((t->y = x / cols) >= rows) t->y = 0;
	validate_tty_state();

	j1 = KERNEL$GET_JIFFIES_LO();
	jc = 0;
	while (KERNEL$CONSOLE_READ((char *)&KERNEL$LIST_END, sizeof KERNEL$LIST_END, &last_cons_update)) {
		j2 = KERNEL$GET_JIFFIES_LO();
		if (j2 != j1) if (++jc == 2) break;
		j1 = j2;
	}

	r = KERNEL$REGISTER_DEVICE(dev_name, "PCCONS.SYS", 0, NULL, console_init_root, VESA_VIDEOMODE_DCALL, "CONSOLE", NULL, unload, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret1;
	}
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;
	ret1:
	for (i = 0; i < vcs; i++) for (j = 0; j < ring_size; j++) {
		free(lines[i][j]);
	}
	ret0:
	if (port_locked) KERNEL$UNREGISTER_IO_RANGE(&port_range);
	if (vga_locked) KERNEL$UNREGISTER_IO_RANGE(&vga_range);
	return r;
}

__COLD_ATTR__ static int unload(void *p, void **release, const char * const argv[])
{
	int r;
	int i, j;
	RAISE_SPL(SPL_DEV);
	if ((r = VESA_TEST_UNLOAD())) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS DEVICE", dev_name);
		return r;
	}
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(lnte, argv))) {
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	RAISE_SPL(SPL_TTY);
	VESA_DONE();
	cons_done();
	LOWER_SPL(SPL_ZERO);
	for (i = 0; i < vcs; i++) for (j = 0; j < ring_size; j++) {
		free(lines[i][j]);
	}
	if (port_locked) KERNEL$UNREGISTER_IO_RANGE(&port_range);
	if (vga_locked) KERNEL$UNREGISTER_IO_RANGE(&vga_range);
	*release = dlrq;
	return 0;
}

__COLD_ATTR__ long PCCONS_VIDEOMODE(int cmd, int vc, va_list args)
{
	switch (cmd) {
		case VIDEOMODE_GET_FONT_MODE: {
			if (card_type == CARD_EGA) return RESULT_GET_FONT_MODE_EGA;
			if (card_type == CARD_VGA) return RESULT_GET_FONT_MODE_VGA;
			return RESULT_GET_FONT_MODE_NONE;
		}
		case VIDEOMODE_SET_FONT: {
			PROC *proc = va_arg(args, PROC *);
			FBLOB *blob_addr = va_arg(args, FBLOB *);
			size_t blob_size = va_arg(args, size_t);
			IORQ *rq = va_arg(args, IORQ *);
			return CONS_SET_FONT(vc, proc, blob_addr, blob_size, rq);
		}
		case VIDEOMODE_GET_DEFAULT_CHARSET: {
			return (long)"ISO-8859-1";
		}
		default: {
			return -ENXIO;
		}
	}
}

