#include <SPAD/SYNC.H>
#include <SPAD/LIBC.H>
#include <SYS/TYPES.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TTY.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/TIMER.H>
#include <SPAD/DL.H>
#include <SPAD/PCI.H>
#include <VALUES.H>
#include <ASM/IO.H>
#include <ARCH/MOV.H>

#include <VGA.H>
#include "VGAREG.H"

#define S3_TRIO_FIFO_TIMEOUT	(JIFFIES_PER_SECOND / 2)
#define S3_VIRGE_FIFO_TIMEOUT	(JIFFIES_PER_SECOND / 5)
#define RETRACE_TIMEOUT		(JIFFIES_PER_SECOND / 20)

static char *cons = "PCCONS@MEM_B8000:";
static const char dev_name[] = "VIDEO$S3";

static io_t io_base;

#define S3_911		0
#define S3_924		1
#define S3_801		2
#define S3_805		3
#define S3_928		4
#define S3_864		5
#define S3_964		6
#define S3_TRIO32	7
#define S3_TRIO64	8
#define S3_866		9
#define S3_868		10
#define S3_968		11
#define S3_765		12

static const char * const s3_chipname[] = {
"911", "924", "801", "805", "928",
"864", "964", "Trio32", "Trio64", "866", "868", "968", "Trio64V+"};

static __s8 s3_chiptype;

/*#define S3_OLD_STEPPING		0x01
static char s3_flags;*/

static const struct dac *dac = NULL;
static int accel = 1;
static pci_id_t pci_id;
static char pci_valid = 0;
static unsigned long pci_chiptype = 0;
static __u8 *pci_regs = NULL;

static const struct __param_table params[3] = {
	"CONSOLE", __PARAM_STRING, 1, __MAX_STR_LEN,
	"NOACCEL", __PARAM_BOOL, ~0, 0,
	NULL, 0, 0, 0,
};

static void *const vars[3] = {
	&cons,
	&accel,
	NULL,
};

#define T_964_0			0x01
#define T_964_1			0x02
#define T_TRIO			0x03
#define T_AURORA64VP		0x04

#define T_968			0x05
#define T_TRIO64UVP		0x06
#define T_TRIO64V2_DXGX		0x07

#define VIRGE			0x11
#define VIRGE_VX		0x12
#define VIRGE_DX_GX		0x13
#define VIRGE_GX_2		0x14
#define VIRGE_MX		0x15
#define VIRGE_MXP		0x16
#define VIRGE_TRIO_3D		0x17
#define VIRGE_TRIO_3D_2X	0x18

#define SAVAGE_3D		0x21
#define SAVAGE_4		0x22
#define SAVAGE_2000		0x23
#define SAVAGE_MX		0x24
#define SAVAGE_PRO		0x25
#define SAVAGE_SUPER		0x26

static const struct pci_id_s s3_pci_cards[] = {

	{ 0x5333, 0x88D0, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 964", T_964_0 },
	{ 0x5333, 0x88D1, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 964", T_964_1 },
	{ 0x5333, 0x8811, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio", T_TRIO },
	{ 0x5333, 0x8812, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Aurora64 VP", T_AURORA64VP },
	{ 0x5333, 0x88f0, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 968", T_968 },
	{ 0x5333, 0x8814, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio64 UVP", T_TRIO64UVP },
	{ 0x5333, 0x8901, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio64 DX/GX", T_TRIO64V2_DXGX },

	{ 0x5333, 0x5631, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE", VIRGE },
	{ 0x5333, 0x883D, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE VX", VIRGE_VX },
	{ 0x5333, 0x8A01, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE DX/GX", VIRGE_DX_GX },
	{ 0x5333, 0x8A10, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE GX 2", VIRGE_GX_2 },
	{ 0x5333, 0x8C01, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE MX", VIRGE_MX },
	{ 0x5333, 0x8C03, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE MX+", VIRGE_MXP },
	{ 0x5333, 0x8904, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio3D", VIRGE_TRIO_3D },
	{ 0x5333, 0x8A13, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio3D/2X", VIRGE_TRIO_3D_2X },

	{ 0x5333, 0x8A20, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage 3D", SAVAGE_3D },
	{ 0x5333, 0x8A21, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage 3D-MV", SAVAGE_3D },
	{ 0x5333, 0x8A22, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage 4", SAVAGE_4 },
	{ 0x5333, 0x9102, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage 2000", SAVAGE_2000 },
	{ 0x5333, 0x9102, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage 2000", SAVAGE_2000 },
	{ 0x5333, 0x8C10, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage MX/MV", SAVAGE_MX },
	{ 0x5333, 0x8C11, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage MX", SAVAGE_MX },
	{ 0x5333, 0x8C12, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage IX/MV", SAVAGE_MX },
	{ 0x5333, 0x8C13, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Savage IX", SAVAGE_MX },
	{ 0x5333, 0x8A25, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ProSavage PM133", SAVAGE_PRO },
	{ 0x5333, 0x8A26, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ProSavage KM133", SAVAGE_PRO },
	{ 0x5333, 0x8D01, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ProSavage PN133", SAVAGE_PRO },
	{ 0x5333, 0x8D02, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ProSavage KN133", SAVAGE_PRO },
	{ 0x5333, 0x8D03, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ProSavage DDR", SAVAGE_PRO },
	{ 0x5333, 0x8D04, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ProSavage DDR-K", SAVAGE_PRO },
	{ 0x5333, 0x8C22, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage MX 128", SAVAGE_SUPER },
	{ 0x5333, 0x8C24, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage MX 64", SAVAGE_SUPER },
	{ 0x5333, 0x8C26, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage MX 64C", SAVAGE_SUPER },
	{ 0x5333, 0x8C2A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage IX 128 SDR", SAVAGE_SUPER },
	{ 0x5333, 0x8C2B, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage IX 128 DDR", SAVAGE_SUPER },
	{ 0x5333, 0x8C2C, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage IX 64 SDR", SAVAGE_SUPER },
	{ 0x5333, 0x8C2D, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage IX 64 DDR", SAVAGE_SUPER },
	{ 0x5333, 0x8C2E, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage IXC 64 SDR", SAVAGE_SUPER },
	{ 0x5333, 0x8C2F, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 SuperSavage IXC 64 DDR", SAVAGE_SUPER },

	{ 0 },
};

#define VGA_DWORD_IO

#include "VGA.I"
#include "DAC.I"

static void s3_set_timing(struct video_mode *v, struct tty_videomode_params *p);
static long s3_get_pixel_clock(struct video_mode *v, unsigned px);
static int s3_set_display_start(struct video_mode *v, unsigned addr, int waitretrace);

static long s3t_accel_avail(struct video_mode *v, int ac);
static void s3t_accel_init(struct video_mode *v);
static void *s3t_accel_do(struct video_mode *v, unsigned op, union accel_param *ap);

static long s3v_accel_avail(struct video_mode *v, int ac);
static void s3v_accel_init(struct video_mode *v);
static void *s3v_accel_do(struct video_mode *v, unsigned op, union accel_param *ap);

static struct driver_function s3_functions[] = {
	"ACCEL_AVAIL", NULL,
	"ACCEL_INIT", NULL,
	"ACCEL_EXIT", NULL,
	"ACCEL_DO", NULL,
	"SET_VIDEO_TIMING", NULL,
	"GET_PIXEL_CLOCK", NULL,
	"SET_DISPLAY_START", s3_set_display_start,
	NULL, NULL,
};

#define fn_accel_avail		s3_functions[0].ptr
#define fn_accel_init		s3_functions[1].ptr
#define fn_accel_exit		s3_functions[2].ptr
#define fn_accel_do		s3_functions[3].ptr
#define fn_set_timing		s3_functions[4].ptr
#define fn_get_pixel_clock	s3_functions[5].ptr
#define fn_set_display_start	s3_functions[6].ptr

static int test_wr(unsigned reg, __u8 idx, __u8 mask)
{
	__u8 old, new1, new2;
	io_outb(io_base + reg, idx);
	old = io_inb(io_base + reg + 1);
	io_outb(io_base + reg + 1, old & ~mask);
	new1 = io_inb(io_base + reg + 1) & mask;
	io_outb(io_base + reg + 1, old | mask);
	new2 = io_inb(io_base + reg + 1) & mask;
	io_outb(io_base + reg + 1, old);
	return !new1 && new2 == mask;
}

static int s3_probe(void)
{
	int r;
	__u8 old_11, old_38, config;
	unsigned id, rev;
	RAISE_SPL(SPL_TTY);
	io_base = io_inb(VGA_OUTPUT_REG) & 1 ? VGA_BASE_COLOR : VGA_BASE_MONO;
	old_11 = in_cr(0x11);
	out_cr(0x11, 0x00);
	old_38 = in_cr(0x38);
	out_cr(0x38, 0x00);
	if (test_wr(VGA_CRTC_IDX, 0x35, 0x0f)) {
		not_found:
		r = -ENODEV;
		ret:
		out_cr(0x11, old_11);
		out_cr(0x38, old_38);
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	out_cr(0x38, 0x48);
	if (!test_wr(VGA_CRTC_IDX, 0x35, 0x0f)) goto not_found;
	out_cr(0x39, 0xa5);
	id = in_cr(0x30);
	rev = id & 0x0f;
	if (id >= 0xe0) {
		id |= in_cr(0x2e) << 8;
		rev |= in_cr(0x2f) << 4;
	}
	config = in_cr(0x36);
	s3_chiptype = -1;
	/*s3_flags = 0;*/
	switch (id & 0xfff0) {
		case 0x0080:
			if (rev == 1) s3_chiptype = S3_911;
			if (rev == 2) s3_chiptype = S3_924;
			break;
		case 0x00a0:
			switch (config & 0x03) {
				case 0x00:
				case 0x01:
					s3_chiptype = S3_805;
					/*if ((rev & 0x0f) < 2) s3_flags |= S3_OLD_STEPPING;*/
					break;
				case 0x03:
					s3_chiptype = S3_801;
					/*if ((rev & 0x0f) < 2) s3_flags |= S3_OLD_STEPPING;*/
					break;
			}
			break;
		case 0x0090:
			s3_chiptype = S3_928;
			/*if ((rev & 0x0f) < 4) s3_flags |= S3_OLD_STEPPING;*/
			break;
		case 0x00b0:
			s3_chiptype = S3_928;
			break;
		case 0x00c0:
			s3_chiptype = S3_864;
			break;
		case 0x00d0:
			s3_chiptype = S3_964;
			break;
		case 0x10e0:
			s3_chiptype = S3_TRIO32;
			break;
		case 0x3de0:
		case 0x31e0:
		case 0x01e0:
		case 0x04e0:
		case 0x11e0:
		case 0x13e0:
			if (rev & 0x0400) s3_chiptype = S3_765;
			else s3_chiptype = S3_TRIO64;
			break;
		case 0x80e0:
			s3_chiptype = S3_866;
			break;
		case 0x90e0:
			s3_chiptype = S3_868;
			break;
		case 0xf0e0:
		case 0xb0e0:
			s3_chiptype = S3_968;
			break;
		default:
			if (pci_valid)
				s3_chiptype = S3_TRIO64;
			break;
	}
	if (s3_chiptype == -1) {
		r = -ENXIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: UNKNOWN CHIP TYPE. ID %04X, REV %03X", id, rev);
		goto ret;
	}
	if (pci_valid && pci_chiptype >= T_AURORA64VP && s3_chiptype < S3_TRIO64)
		s3_chiptype = S3_TRIO64;	/* needed for ViRGE GX/2 which identifies itself as Trio32 */
	if (s3_chiptype == S3_TRIO64 || s3_chiptype == S3_765) {
		if ((r = s3_trio64_dac.probe(s3_chiptype, pci_chiptype))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: UNABLE TO INITIALIZE DAC. CHIP %s", s3_chipname[s3_chiptype]);
			goto ret;
		}
		dac = &s3_trio64_dac;
	} else {
		if (!s3_sdac.probe(s3_chiptype, pci_chiptype)) dac = &s3_sdac;
		else if (!s3_gendac.probe(s3_chiptype, pci_chiptype)) dac = &s3_gendac;
		/*else {
			r = -ENXIO;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: UNKNOWN DAC. CHIP %s", s3_chipname[s3_chiptype]);
			goto ret;
		}*/
	}

	r = 0;
	goto ret;
}

static void s3_set_timing(struct video_mode *v, struct tty_videomode_params *p)
{
	struct tty_videomode_params pp = *p;
	if (pci_chiptype == SAVAGE_MX || pci_chiptype == SAVAGE_SUPER) {
		/* I don't have any sample of these, can't test it */
		return;
	}
	if (pci_chiptype == VIRGE_MX || pci_chiptype == VIRGE_MXP) {
		/* If LCD is ON, don't mess with it. BIOS set it up fine */
		if (__likely(in_sr(0x31) & 0x10)) return;

		/* some anomaly regarding double scanned modes.
		   Maybe it is needed on other chipsets too.
		   Virge DX/GX definitely doesn't want it. */
		if (pp.v >> !!(pp.flags & VIDEO_PARAMS_DOUBLE_SCAN) <= 384)
			pp.pixel_clock *= 2;
	}
	out_cr(0x38, 0x48);
	out_cr(0x39, 0xa5);

	if (__unlikely(!(pp.pixel_clock = dac->set_pixel_clock(v->bpp, v->x, pp.pixel_clock)))) return;

	/* For Trio3D and 24-bpp, the values in registers are completely fscked
	   up. I have no clue what to set there. Horizontal timing seem to be
	   multiplied by 3/2, but simple multiplication won't produce the
	   picture, there are some magic additions needed as well.
	   
	   Ignore it, and we get at least higher refresh rate, without
	   centering the image. */
	if (pci_chiptype == VIRGE_TRIO_3D && v->bpp == 24) return;

	if (s3_chiptype >= S3_911) out_cr(0x40, in_cr(0x40) | 0x01);
	pp.h = dac->map_horizontal_crtc(v->bpp, v->x, pp.pixel_clock, pp.h);
	pp.h_sync_start = dac->map_horizontal_crtc(v->bpp, v->x, pp.pixel_clock, pp.h_sync_start);
	pp.h_sync_end = dac->map_horizontal_crtc(v->bpp, v->x, pp.pixel_clock, pp.h_sync_end);
	pp.h_total = dac->map_horizontal_crtc(v->bpp, v->x, pp.pixel_clock, pp.h_total);
	/*
	__debug_printf("h: %d, h_sync_start: %d, h_sync_end: %d, h_total: %d\n", pp.h, pp.h_sync_start, pp.h_sync_end, pp.h_total);
	__debug_printf("v: %d, v_sync_start: %d, v_sync_end: %d, v_total: %d\n", pp.v, pp.v_sync_start, pp.v_sync_end, pp.v_total);
	__debug_printf("px: %d\n", pp.pixel_clock);
	__debug_printf("was h total: %d\n", in_cr(0));
	__debug_printf("was h: %d\n", in_cr(1));
	__debug_printf("was h blank start: %d\n", in_cr(2));
	__debug_printf("was h blank end: %d\n", in_cr(3));
	__debug_printf("was h sync start: %d\n", in_cr(4));
	__debug_printf("was h blank/sync end: 0x%x\n", in_cr(5));
	__debug_printf("was v total: %d\n", in_cr(6));
	__debug_printf("was ovf: 0x%x\n", in_cr(7));*/
	if (pp.h_total > 4096) pp.h_total = 4096;
	setup_vga_timing(v, &pp);
	/*
	__debug_printf("is h total: %d\n", in_cr(0));
	__debug_printf("is h: %d\n", in_cr(1));
	__debug_printf("is h blank start: %d\n", in_cr(2));
	__debug_printf("is h blank end: %d\n", in_cr(3));
	__debug_printf("is h sync start: %d\n", in_cr(4));
	__debug_printf("is h blank/sync end: 0x%x\n", in_cr(5));
	__debug_printf("is v total: %d\n", in_cr(6));
	__debug_printf("is ovf: 0x%x\n", in_cr(7));*/
	if (s3_chiptype >= S3_801) {
		int m, n;
		int i, j;
		if (pci_chiptype >= VIRGE) goto skip_mclk;
		n = 0xff;
		if (s3_chiptype >= S3_864 || s3_chiptype == S3_801 || s3_chiptype == S3_805) {
	/* How to calculate these values correctly? I don't know ...
	   ... in 864 they have these timings wrong in their own vesa bios! */
			int clock = (pp.pixel_clock * ((v->bpp + 7) >> 3)) / 1000;
			int mc;
			if (v->total_mem < 2048 * 1024 || s3_chiptype == S3_TRIO32) clock *= 2;
			if (__likely(mclk)) mc = mclk;
			else if (s3_chiptype == S3_864) mc = 40000;
			else if (s3_chiptype == S3_801 || s3_chiptype == S3_805) mc = 50000;
			else mc = 60000;
			m = (((__s64)mc * 720 / 1000 + 16867) * 89736 / (clock + 39000) - 21154) / 1000;
			if (v->total_mem < 2048 * 1024 || s3_chiptype == S3_TRIO32) m /= 2;
			/*__debug_printf("%d, %d %d: %d %d\n", v->total_mem, clock, s3_chiptype, m, n);*/
			if (m > 31) m = 31;
			if (m < 0) {
				m = 0;
				if (s3_chiptype != S3_864) {
/* 864 will trash screen on 800x600x16m32@75 with this enabled.
   Maybe more cards will need to be excluded here */
					n = 16;
				}
			}
		} else if (v->total_mem == 512 * 1024 || v->x > 1200) {
			m = 0;
		} else if (v->total_mem == 1024 * 1024) {
			m = 2;
		} else {
			m = 20;
		}
		out_cr(0x54, m << 3);
		out_cr(0x60, n);
		out_cr(0x55, (in_cr(0x55) & 0x08) | 0x40);
		skip_mclk:

		out_cr(0x5e, (((pp.v_total - 2) & 0x400) >> 10) |
			     (((pp.v - 1) & 0x400) >> 9) |
			     (((pp.v_sync_start) & 0x400) >> 8) |
			     (((pp.v_sync_start) & 0x400) >> 6) | 0x40);
		i = ((((pp.h_total >> 3) - 5) & 0x100) >> 8) |
		    ((((pp.h >> 3) - 1) & 0x100) >> 7) |
		    ((((pp.h_sync_start >> 3) - 1) & 0x100) >> 6) |
		    ((pp.h_sync_start & 0x800) >> 7);
		if ((pp.h_sync_end >> 3) - (pp.h_sync_start >> 3) > 64) i |= 0x08;
		if ((pp.h_sync_end >> 3) - (pp.h_sync_start >> 3) > 32) i |= 0x20;
		j = (in_cr(0x00) + ((i & 0x01) << 8) + in_cr(0x04) + ((i & 0x10) << 4) + 1) / 2;
		if (j - (in_cr(0x04) + ((i & 0x10) << 4)) < 4) {
			if (in_cr(0x04) + ((i & 0x10) << 4) + 4 <= in_cr(0x00) + ((i & 0x01) << 8)) j = in_cr(0x04) + ((i & 0x10) << 4) + 4;
			else j = in_cr(0x00) + ((i & 0x01) << 8) + 1;
		}
		out_cr(0x3b, j);
		i |= (j & 0x100) >> 2;
		out_cr(0x3c, (in_cr(0x00) + ((i & 0x01) << 8)) / 2);
		out_cr(0x5d, (in_cr(0x5d) & 0x80) | i);
	}
	if (pp.flags & VIDEO_PARAMS_INTERLACED) out_cr(0x42, in_cr(0x42) | 0x20);
	else out_cr(0x42, in_cr(0x42) & ~0x20);
	/*{
		out_cr(0x38, 0x48);
		out_cr(0x39, 0xa5);
		out_cr(0x58, in_cr(0x58) & ~0x14);
		out_cr(0x40, 0x01);
		out_cr(0x53, in_cr(0x53) & ~0x0f);
		out_cr(0x54, 0xa0);
		io_outw(S3A_MULTIFUNC_CNTL, 0xe000);
		__debug_printf("%04x\n", io_inw(S3A_GP_STAT));
	}*/
}

static long s3_get_pixel_clock(struct video_mode *v, unsigned px)
{
	unsigned clk;
	clk = dac->get_pixel_clock(v->bpp, v->x, px);
	if (__unlikely(!clk) || __unlikely(clk > MAXINT)) return -EINVAL;
	return clk;
}

/*
   Some S3 Vesa bioses have this function seriously broken --- waitretrace flips
   the image immediatelly AFTER the retrace, so the old image is displayed for
   one more frame. If the CPU is fast enough to write to the old page, screen
   flicker happens in mplayer. --- I reimplemented it correctly.
*/

static int s3_set_display_start(struct video_mode *v, unsigned addr, int waitretrace)
{
#if 0
	if (__unlikely(pci_chiptype == VIRGE_TRIO_3D)) addr &= ~1;
#endif
	KERNEL$DI();	/* Minimize the probability that incorrect values are
			sampled by the card */
	out_cr(0x38, 0x48);
	out_cr(0x39, 0xa5);
	out_cr(0x0d, addr >> 2);
	out_cr(0x0c, addr >> 10);
#if 0		/* that trick with ATC doesn't seem to work */
	io_inb(VGA_BASE_COLOR + VGA_INPUT_REG);
	io_outb(VGA_ATC_IDX_REG, VGA_ATC_H_PIXEL_PANNING | VGA_ATC_IDX_REG_PAS);
	io_outb(VGA_ATC_IDX_REG, (io_inb(VGA_ATC_REG_READ) & 0xf0) | ((addr & 3) << 1));
#endif
	if (__likely(pci_chiptype >= VIRGE)) {
		out_cr(0x69, addr >> 18);
	} else {
		io_outb(io_base + VGA_CRTC_REG, (in_cr(0x31) & 0xcf) | (addr & 0xc0000) >> 14);
		if (__likely(s3_chiptype >= S3_801)) {
			io_outb(io_base + VGA_CRTC_REG, (in_cr(0x51) & 0xfc) | (addr & 0x300000) >> 20);
		}
	}
	KERNEL$EI();
	return waitretrace;
}

/*
 * S3 Trio accelerator
 */

#define S3A_SUBSYS_STAT		0x42e8
#define S3A_SUBSYS_CNTL		0x42e8
#define S3A_ADVFUNC_CNTL	0x4ae8
#define S3A_CUR_Y		0x82e8
#define S3A_CUR_X		0x86e8
#define S3A_DESTY_AXSTP		0x8ae8
#define S3A_DESTX_DIASTP	0x8ee8
#define S3A_MAJ_AXIS_PCNT	0x96e8
#define S3A_GP_STAT		0x9ae8
#define S3A_CMD			0x9ae8
#define S3A_FRGD_COLOR		0xa6e8
#define S3A_WRT_MASK		0xaae8
#define S3A_FRGD_MIX		0xbae8
#define S3A_MULTIFUNC_CNTL	0xbee8

static IO_RANGE s3a_ports[] = {
	{ { NULL, NULL }, S3A_SUBSYS_STAT,	2, dev_name },
	{ { NULL, NULL }, S3A_ADVFUNC_CNTL,	2, dev_name },
	{ { NULL, NULL }, S3A_CUR_Y,		2, dev_name },
	{ { NULL, NULL }, S3A_CUR_X,		2, dev_name },
	{ { NULL, NULL }, S3A_DESTY_AXSTP,	2, dev_name },
	{ { NULL, NULL }, S3A_DESTX_DIASTP,	2, dev_name },
	{ { NULL, NULL }, S3A_MAJ_AXIS_PCNT,	2, dev_name },
	{ { NULL, NULL }, S3A_GP_STAT,		2, dev_name },
	{ { NULL, NULL }, S3A_FRGD_COLOR,	2, dev_name },
	{ { NULL, NULL }, S3A_WRT_MASK,		2, dev_name },
	{ { NULL, NULL }, S3A_FRGD_MIX,		2, dev_name },
	{ { NULL, NULL }, S3A_MULTIFUNC_CNTL,	2, dev_name },
	{ { NULL, NULL }, 0,			0, NULL },
};

static char s3a_ports_locked = 0;
static char s3a_should_disable = 0;
static char s3t_accel_available;

static void s3t_enable_access(void)
{
	__u8 cr40;
	out_cr(0x38, 0x48);
	out_cr(0x39, 0xa5);
	cr40 = in_cr(0x40);
	if (!(cr40 & 1)) {
		io_outb(io_base + VGA_CRTC_REG, cr40 | 1);
		s3a_should_disable = 1;
	}
}

static void s3t_disable_access(void)
{
	if (s3a_should_disable) {
		out_cr(0x38, 0x48);
		out_cr(0x39, 0xa5);
		io_outb(io_base + VGA_CRTC_REG, in_cr(0x40) & ~1);
		s3a_should_disable = 0;
	}
}

static void s3t_unlock(void)
{
	int i;
	s3t_disable_access();
	if (s3a_ports_locked) {
		for (i = 0; s3a_ports[i].len; i++)
			KERNEL$UNREGISTER_IO_RANGE(&s3a_ports[i]);
		s3a_ports_locked = 0;
	}
}

static void s3t_error(char *str, unsigned bit)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "%s, BIT %04X, STATUS %04X", str, bit, io_inw(S3A_GP_STAT));
	s3t_accel_available = 0;
}

#define S3T_WAITQUEUE(n, err)						\
do {									\
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;			\
	unsigned bit = 1 << (8 - (n));					\
	while (jj = KERNEL$GET_JIFFIES_LO(), inw(S3A_GP_STAT) & bit) {	\
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_TRIO_FIFO_TIMEOUT))) {									\
			s3t_error("TIMEOUT WAITING FOR QUEUE", bit);	\
			err;						\
			break;						\
		}							\
	}								\
} while (0)

#define S3T_WAITQUEUE_16_32(v, n16, n32, err)	S3T_WAITQUEUE((v)->bpp <= 16 ? (n16) : (n32), err)

#define S3T_WAITIDLE(err)	S3T_WAITQUEUE(-1, err)

static long s3t_accel_avail(struct video_mode *v, int ac)
{
	unsigned bpp, vx;
	bpp = (v->bpp + 7) >> 3;
	vx = v->scanline / bpp;
	if (__unlikely(v->scanline % bpp)) return 0;
	if (vx != 640 &&
	    vx != 800 &&
	    __unlikely(vx != 1024) &&
	    __unlikely(vx != 1152) &&
	    __unlikely(vx != 1280) &&
	    __unlikely(vx != 1600) &&
	    __unlikely(vx != 2048)) return 0;
	if (v->bpp != 8 &&
	    v->bpp != 15 &&
	    v->bpp != 16 &&
	    v->bpp != 24 &&
	    __unlikely(v->bpp != 32)) return 0;
	switch (ac) {
		case VGA_AVAIL_ACCEL:
			return ACCELFLAG_FILLBOX | ACCELFLAG_SETFGCOLOR | ACCELFLAG_SCREENCOPY | ACCELFLAG_SETMODE | ACCELFLAG_SYNC;
		default:
			return 0;
	}
}

static void s3t_accel_init(struct video_mode *v)
{
	__u8 cr50;
	unsigned bpp, vx;
	s3t_accel_available = 0;
	if (__unlikely(!s3t_accel_avail(v, VGA_AVAIL_ACCEL))) return;
	bpp = (v->bpp + 7) >> 3;
	vx = v->scanline / bpp;
	s3t_enable_access();
	cr50 = in_cr(0x50);
	cr50 &= 0x0e;
	switch (vx) {
		case 640: cr50 |= 0x40; break;
		case 800: cr50 |= 0x80; break;
		case 1024:
			if (__unlikely(in_cr(0x31) & 2)) return;
			cr50 |= 0x00; break;
		case 1152: cr50 |= 0x01; break;
		case 1280: cr50 |= 0xc0; break;
		case 1600: cr50 |= 0x81; break;
		case 2048:
			if (__unlikely(!(in_cr(0x31) & 2))) return;
			cr50 |= 0x00; break;
		default: return;
	}
	switch (v->bpp) {
		case 8: cr50 |= 0x00; break;
		case 15:
		case 16: cr50 |= 0x10; break;
		case 24: cr50 |= 0x20; break;
		case 32: cr50 |= 0x30; break;
		default: return;
	}
	io_outb(io_base + VGA_CRTC_REG, cr50);
	io_outw(S3A_ADVFUNC_CNTL, v->x == 1024 ? 0x7 : 0x3);
	io_outw(S3A_SUBSYS_CNTL, 0x9000);	/* reset it */
	io_outw(S3A_SUBSYS_CNTL, 0x5000);
	io_inw(S3A_SUBSYS_STAT);
	io_outw(S3A_MULTIFUNC_CNTL, 0x500c);/* copied from X, I have no clue what does it do */
	S3T_WAITQUEUE(5, return);
	io_outw(S3A_MULTIFUNC_CNTL, 0x1000);		/* scissors, top */
	io_outw(S3A_MULTIFUNC_CNTL, 0x2000);		/* scissors, left */
	io_outw(S3A_MULTIFUNC_CNTL, 0x3000 | (v->y * v->images - 1));/* scissors, bottom */
	io_outw(S3A_MULTIFUNC_CNTL, 0x4000 | (v->x - 1));/* scissors, right */
	s3t_accel_available = 1;
}

static void s3t_accel_exit(struct video_mode *v)
{
	if (s3t_accel_available) io_outw(S3A_ADVFUNC_CNTL, 0);
	s3t_disable_access();
}

static void *s3t_accel_do(struct video_mode *v, unsigned op, union accel_param *ap)
{
	if (__unlikely(!s3t_accel_available)) return __ERR_PTR(-EOPNOTSUPP);
	__write_barrier2();
	if (__likely((op & _ACCEL_OP) == ACCEL_FILLBOX)) {
		if (__unlikely(__GET_FIELD(op, _ACCEL_ROP) != ROP_COPY))
			return __ERR_PTR(-EINVAL);
		S3T_WAITQUEUE_16_32(v, 4, 6, return __ERR_PTR(-EIO));
		io_outw(S3A_MULTIFUNC_CNTL, 0xa000);
		io_outw(S3A_FRGD_COLOR, ap->fillbox.c);
		if (v->bpp > 16) io_outw(S3A_FRGD_COLOR, ap->fillbox.c >> 16);
		io_outw(S3A_FRGD_MIX, 0x0027);
		io_outw(S3A_WRT_MASK, 0xffff);
		if (v->bpp > 16) io_outw(S3A_WRT_MASK, 0xffff);
		S3T_WAITQUEUE(5, return __ERR_PTR(-EIO));
		io_outw(S3A_CUR_X, ap->fillbox.x);
		io_outw(S3A_CUR_Y, ap->fillbox.y);
		io_outw(S3A_MAJ_AXIS_PCNT, ap->fillbox.w - 1);
		io_outw(S3A_MULTIFUNC_CNTL, ap->fillbox.h - 1);
		io_outw(S3A_CMD, 0x40b1);
	} else if (__likely((op & _ACCEL_OP) == ACCEL_SCREENCOPY)) {
		unsigned cmd;
		if (__unlikely(__GET_FIELD(op, _ACCEL_ROP) != ROP_COPY))
			return __ERR_PTR(-EINVAL);
		ap->screencopy.w--;
		ap->screencopy.h--;
		S3T_WAITQUEUE_16_32(v, 3, 4, return __ERR_PTR(-EIO));
		io_outw(S3A_MULTIFUNC_CNTL, 0xa000);
		io_outw(S3A_FRGD_MIX, 0x0067);
		io_outw(S3A_WRT_MASK, 0xffff);
		if (v->bpp > 16) io_outw(S3A_WRT_MASK, 0xffff);
		cmd = 0xc0b1;
		if (ap->screencopy.x1 < ap->screencopy.x2) {
			ap->screencopy.x1 += ap->screencopy.w;
			ap->screencopy.x2 += ap->screencopy.w;
			cmd &= ~0x0020;
		}
		if (ap->screencopy.y1 < ap->screencopy.y2) {
			ap->screencopy.y1 += ap->screencopy.h;
			ap->screencopy.y2 += ap->screencopy.h;
			cmd &= ~0x0080;
		}
		S3T_WAITQUEUE(7, return __ERR_PTR(-EIO));
		io_outw(S3A_CUR_X, ap->screencopy.x1);
		io_outw(S3A_CUR_Y, ap->screencopy.y1);
		io_outw(S3A_DESTX_DIASTP, ap->screencopy.x2);
		io_outw(S3A_DESTY_AXSTP, ap->screencopy.y2);
		io_outw(S3A_MAJ_AXIS_PCNT, ap->screencopy.w);
		io_outw(S3A_MULTIFUNC_CNTL, ap->screencopy.h);
		io_outw(S3A_CMD, cmd);
	} else if (__likely((op & _ACCEL_OP) == ACCEL_SYNC)) {
		S3T_WAITIDLE(return __ERR_PTR(-EIO));
		return NULL;
	} else {
		return __ERR_PTR(-EOPNOTSUPP);
	}
	if (!(op & _ACCEL_ASYNC)) {
		S3T_WAITIDLE(return NULL);
	}
	return NULL;
}

/*
 * S3 ViRGE accelerator
 */

#define S3V_REGOFFS	0x01000000
#define S3V_REGSIZE	0x00010000

#define S3V_INREG(r)		mmio_inl(pci_regs + (r))
#define S3V_OUTREG(r, v)	mmio_outl(pci_regs + (r), v)

static int s3v_cmd_bpp;
static char s3v_in_progress;
static char s3v_patch_32;

static void s3v_error(char *str)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "%s, STATUS %08X, ADV CTRL %08X", str, (unsigned)S3V_INREG(0x8504), (unsigned)S3V_INREG(0x850c));
	s3v_cmd_bpp = -1;	/* disable the accelerator */
}

#define S3V_WAITFIFO(n, err)						\
do {									\
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;			\
	__u32 sub_stat;							\
	while (jj = KERNEL$GET_JIFFIES_LO(),				\
		sub_stat = S3V_INREG(0x8504),				\
		(__likely(pci_chiptype != VIRGE_GX_2) &&		\
		__likely(pci_chiptype != VIRGE_TRIO_3D_2X) ?		\
			__unlikely(((sub_stat >> 8) & 0x1f) < (n)) :	\
			__unlikely(!((sub_stat >> 9) & 0x60))		\
		)							\
	) {								\
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {									\
			s3v_error("TIMEOUT WAITING FOR FIFO");		\
			err;						\
			break;						\
		}							\
	}								\
} while (0)

#define S3V_WAITCMD(err)						\
do {									\
	if (__unlikely(pci_chiptype == VIRGE_GX_2) ||			\
	    __unlikely(pci_chiptype == VIRGE_TRIO_3D_2X)) {		\
		u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;		\
		while (jj = KERNEL$GET_JIFFIES_LO(),			\
			__unlikely((S3V_INREG(0x850c) & 0x07c0) != 0x0400)) {\
			if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {								\
				s3v_error("TIMEOUT WAITING FOR COMMAND");\
				err;					\
				break;					\
			}						\
		}							\
	}								\
} while (0)

#define S3V_WAITIDLE(err)						\
do {									\
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;			\
	__u32 sub_stat;							\
	while (jj = KERNEL$GET_JIFFIES_LO(),				\
		sub_stat = S3V_INREG(0x8504),				\
		(__likely(pci_chiptype != VIRGE_TRIO_3D) ?		\
			__unlikely((sub_stat & 0x3f00) < 0x3000) :	\
			__unlikely((sub_stat & 0x20002000) != 0x20002000)\
		)							\
	) {								\
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {									\
			s3v_error("TIMEOUT WAITING FOR IDLE");		\
			err;						\
			break;						\
		}							\
	}								\
} while (0)

#define S3V_WAITCMDFIFO(err)						\
do {									\
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;			\
	while (jj = KERNEL$GET_JIFFIES_LO(),				\
		__unlikely(pci_chiptype == VIRGE_TRIO_3D) ?		\
			__unlikely((S3V_INREG(0x8504) & 0x5f00) != 0x5f00) :\
		__unlikely(pci_chiptype == VIRGE_MX) ||			\
		__unlikely(pci_chiptype == VIRGE_MXP) ||		\
		__unlikely(pci_chiptype == VIRGE_GX_2) ||		\
		__unlikely(pci_chiptype == VIRGE_TRIO_3D_2X) ?		\
			__unlikely(!(S3V_INREG(0x850c) & 0x400)) :	\
			__unlikely(!(S3V_INREG(0x850c) & 0x200))	\
									\
	) {								\
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {									\
			s3v_error("TIMEOUT WAITING FOR COMMAND FIFO");	\
			err;						\
			break;						\
		}							\
	}								\
} while (0)

/* All types of cards generate miniature pixel corruption when FIFO is used
   --- probably the FIFO writes the register before previous command reads it
   --- so we always wait until the engine is idle */

#undef S3V_WAITFIFO
#define S3V_WAITFIFO(n, err)		S3V_WAITIDLE(err)

#define S3V_NO_FILLBOX	(__unlikely(pci_chiptype == VIRGE_TRIO_3D))

static long s3v_accel_avail(struct video_mode *v, int ac)
{
	if (v->bpp != 8 &&
	    v->bpp != 15 &&
	    v->bpp != 16 &&
	    v->bpp != 24 &&
	    __unlikely(v->bpp != 32)) return 0;
	if (__unlikely((v->x << (v->bpp == 32)) >= 2048) ||
	    __unlikely(v->y >= 2048)) return 0;
/* Trio3D/2X can't do this trick with 16-bit accelerator and 32-bit display.
   The 32-bit display puts so much load on the memory that the accelerator
   produces temporary underruns.
   Luckily, Trio3D/2X has 24-bit modes in its Vesa BIOS
   TODO: test if this is required on on GX 2 */
	if ((__unlikely(pci_chiptype == VIRGE_GX_2) ||
	     __unlikely(pci_chiptype == VIRGE_TRIO_3D_2X)) &&
	   __unlikely(v->bpp == 32)) return 0;

	switch (ac) {
		case VGA_AVAIL_ACCEL:
			return (S3V_NO_FILLBOX ? 0 : ACCELFLAG_FILLBOX | ACCELFLAG_SETFGCOLOR) | ACCELFLAG_SCREENCOPY | ACCELFLAG_SETMODE | ACCELFLAG_SYNC;
		default:
			return 0;
	}
}

static void s3v_accel_init(struct video_mode *v)
{
	unsigned stride, scis;
	int cb;
	s3v_cmd_bpp = -1;
	s3v_in_progress = 0;
	s3v_patch_32 = 0;
	if (__unlikely(!s3v_accel_avail(v, VGA_AVAIL_ACCEL))) return;
	if (v->bpp == 8) cb = 0;
	else if (v->bpp == 15 || v->bpp == 16) cb = 4;
	else if (v->bpp == 24) cb = 8;
	else if (__likely(v->bpp == 32)) cb = 4, s3v_patch_32 = 1;
	else return;
	out_cr(0x53, 0x08);
	S3V_WAITFIFO(6, return);
	S3V_OUTREG(0xa4d4, 0);
	S3V_OUTREG(0xa4d8, 0);
	stride = v->scanline;
	S3V_OUTREG(0xa4e4, stride | (stride << 16));
	S3V_OUTREG(0xa4dc, v->x << s3v_patch_32);
	scis = v->y * v->images;
	if (scis > 2047) scis = 2047;
	S3V_OUTREG(0xa4e0, scis);
	S3V_WAITCMD(return);
	S3V_OUTREG(0xa500, 0x78000000);
	s3v_cmd_bpp = cb;
}

static void *s3v_accel_do(struct video_mode *v, unsigned op, union accel_param *ap)
{
	int command = s3v_cmd_bpp;
	if (__unlikely(command == -1)) return __ERR_PTR(-EOPNOTSUPP);
	if (__unlikely(!s3v_in_progress)) {
		/* Flush framebuffer writes */
		__write_barrier2();
		S3V_WAITCMDFIFO(return __ERR_PTR(-EIO));
	}
	if (__likely((op & _ACCEL_OP) == ACCEL_FILLBOX)) {
		if (__unlikely(__GET_FIELD(op, _ACCEL_ROP) != ROP_COPY))
			return __ERR_PTR(-EINVAL);
		if (S3V_NO_FILLBOX)
			return __ERR_PTR(-EOPNOTSUPP);
		command |= 0x17e00120;
		if (__unlikely(s3v_patch_32)) {
			ap->fillbox.w <<= 1;
			ap->fillbox.x <<= 1;
			if (__unlikely(((ap->fillbox.c >> 16) & 0xff) != (ap->fillbox.c & 0xff))) {
				return __ERR_PTR(-EINVAL);
			}
		}
		S3V_WAITFIFO(4, return __ERR_PTR(-EIO));
		S3V_OUTREG(0xa4f4, ap->fillbox.c);
		S3V_OUTREG(0xa504, ((ap->fillbox.w - 1) << 16) | ap->fillbox.h);
		S3V_OUTREG(0xa50c, (ap->fillbox.x << 16) | ap->fillbox.y);
		S3V_WAITCMD(return __ERR_PTR(-EIO));
		S3V_OUTREG(0xa500, command);
		s3v_in_progress = 1;
	} else if (__likely((op & _ACCEL_OP) == ACCEL_SCREENCOPY)) {
		if (__unlikely(__GET_FIELD(op, _ACCEL_ROP) != ROP_COPY))
			return __ERR_PTR(-EINVAL);
		command |= 0x07980020;
		if (__unlikely(s3v_patch_32)) {
			ap->screencopy.w <<= 1;
			ap->screencopy.x1 <<= 1;
			ap->screencopy.x2 <<= 1;
		}
		if (ap->screencopy.x1 < ap->screencopy.x2) {
			ap->screencopy.x1 += ap->screencopy.w - 1;
			ap->screencopy.x2 += ap->screencopy.w - 1;
			command &= ~0x02000000;
		}
		if (ap->screencopy.y1 < ap->screencopy.y2) {
			ap->screencopy.y1 += ap->screencopy.h - 1;
			ap->screencopy.y2 += ap->screencopy.h - 1;
			command &= ~0x04000000;
		}
		S3V_WAITFIFO(4, return __ERR_PTR(-EIO));
		S3V_OUTREG(0xa504, ((ap->screencopy.w - 1) << 16) | ap->screencopy.h);
		S3V_OUTREG(0xa508, (ap->screencopy.x1 << 16) | ap->screencopy.y1);
		S3V_OUTREG(0xa50c, (ap->screencopy.x2 << 16) | ap->screencopy.y2);
		S3V_WAITCMD(return __ERR_PTR(-EIO));
		S3V_OUTREG(0xa500, command);
		s3v_in_progress = 1;
	} else if (__likely((op & _ACCEL_OP) == ACCEL_SYNC)) {
		S3V_WAITIDLE(return __ERR_PTR(-EIO));
		return NULL;
	} else {
		return __ERR_PTR(-EOPNOTSUPP);
	}
	if (!(op & _ACCEL_ASYNC)) {
		S3V_WAITIDLE(return NULL);
	}
	return NULL;
}


#if 0
static long s3s_accel_avail(struct video_mode *v, int ac)
{
	if (v->bpp != 8 &&
	    v->bpp != 15 &&
	    v->bpp != 16 &&
	    v->bpp != 24 &&
	    __unlikely(v->bpp != 32)) return 0;
	return ACCELFLAG_FILLBOX | ACCELFLAG_SETFGCOLOR | ACCELFLAG_SCREENCOPY | ACCELFLAG_SETMODE | ACCELFLAG_SYNC;
}

static void s3s_accel_init(struct video_mode *v)
{
}

static void *s3s_accel_do(struct video_mode *v, unsigned op, union accel_param *ap)
{
	return __ERR_PTR(-EOPNOTSUPP);
}
#endif

static int s3_unload(void *p, void **release, const char * const argv[]);

static void *dlrq, *lnte;

int main(int argc, const char * const argv[])
{
	int r;
	int i;
	const char * const *arg = argv;
	int state = 0;
	DLLZRQ lz;
	char *chip_name;
	char gstr[__MAX_STR_LEN];
	__u64 mem_res;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}

	lz.handle = KERNEL$DLRQ();
	lz.caller = NULL;
	strcpy(lz.modname, "PCI");
	SYNC_IO_CANCELABLE(&lz, KERNEL$DL_LOAD_LAZY);
	if (lz.status < 0) {
		goto no_pci;
	}
	if (PCI$FIND_DEVICE(s3_pci_cards, 0, 0, 0, PCI$TEST_LIST, &pci_id, &chip_name, &pci_chiptype, 0)) goto no_pci;
	PCI$ENABLE_DEVICE(pci_id, PCI_COMMAND_IO | PCI_COMMAND_MEMORY);
	pci_valid = 1;
	no_pci:

	*KERNEL$ERROR_MSG() = 0;
	if (!(r = s3_probe())) {
		if (dac) {
			fn_set_timing = s3_set_timing;
			fn_get_pixel_clock = s3_get_pixel_clock;
		}
	} else {
		if (!*KERNEL$ERROR_MSG()) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: S3 CARD NOT FOUND");
		goto ret0;
	}

	if (pci_valid) {
		_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, pci_id));
	} else {
		_printf("%s: %s\n", dev_name, s3_chipname[s3_chiptype]);
	}

	if (pci_chiptype >= VIRGE && pci_chiptype < SAVAGE_3D && accel) {
		mem_res = PCI$READ_MEM_RESOURCE(pci_id, 0, NULL);
		if (!mem_res) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
			goto accel_done;
		}
		pci_regs = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(mem_res + S3V_REGOFFS, S3V_REGSIZE, PAT_UC);
		if (__IS_ERR(pci_regs)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNABLE TO MAP MEM RESOURCE");
			goto accel_done;
		}
		_printf("%s: FRAMEBUFFER %"__64_format"X, ViRGE ACCELERATOR %"__64_format"X\n", dev_name, mem_res, mem_res + S3V_REGOFFS);
		fn_accel_avail = s3v_accel_avail;
		fn_accel_init = s3v_accel_init;
		fn_accel_do = s3v_accel_do;
		goto accel_done;
	}

	if (pci_chiptype >= SAVAGE_3D && accel) {
		/* not supported now */
		goto accel_done;
	}

	if (accel) {
		for (i = 0; s3a_ports[i].len; i++) {
			r = KERNEL$REGISTER_IO_RANGE(&s3a_ports[i]);
			if (r) {
				KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR S3 ACCELERATOR: %s", s3a_ports[i].start, s3a_ports[i].start + s3a_ports[i].len - 1, strerror(-r));
				for (i--; i >= 0; i--) KERNEL$UNREGISTER_IO_RANGE(&s3a_ports[i]);
				goto accel_done;
			}
		}
		s3a_ports_locked = 1;
		RAISE_SPL(SPL_TTY);
		s3t_enable_access();
		if (io_inw(S3A_GP_STAT) == 0xffff) {
			s3t_unlock();
			LOWER_SPL(SPL_ZERO);
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "S3 ACCELERATOR NOT PRESENT ON IO PORT "IO_FORMAT, (io_t)S3A_GP_STAT);
			goto accel_done;
		}
		S3T_WAITIDLE(s3t_unlock(); goto accel_done);
		s3t_disable_access();
		LOWER_SPL(SPL_ZERO);
		fn_accel_avail = s3t_accel_avail;
		fn_accel_init = s3t_accel_init;
		fn_accel_exit = s3t_accel_exit;
		fn_accel_do = s3t_accel_do;
		_printf("%s: S3 PRE-ViRGE ACCELERATOR AT xxE8\n", dev_name);
		goto accel_done;
	}

	accel_done:

	r = KERNEL$DCALL(cons, "CONSOLE", CONSOLE_CMD_INSTALL_DRIVER, (void *)s3_functions);
	if (r < 0) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: CAN'T ATTACH TO CONSOLE DEVICE %s", cons);
		goto ret0;
	}

	r = KERNEL$REGISTER_DEVICE(dev_name, "S3.SYS", 0, NULL, NULL, NULL, NULL, NULL, s3_unload, &lnte, cons, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", dev_name, strerror(-r));
		goto ret1;
	}

	dlrq = KERNEL$TSR_IMAGE();
	strlcpy(KERNEL$ERROR_MSG(), dev_name, __MAX_STR_LEN);
	return 0;

	ret1:
	KERNEL$DCALL(cons, "CONSOLE", CONSOLE_CMD_RELEASE_DRIVER);
	ret0:
	RAISE_SPL(SPL_TTY);
	s3t_unlock();
	LOWER_SPL(SPL_ZERO);
	if (pci_regs) KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(pci_regs, S3V_REGSIZE);
	if (pci_valid) PCI$FREE_DEVICE(pci_id);
	return r;
}

static int s3_unload(void *p, void **release, const char * const argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	KERNEL$DCALL(cons, "CONSOLE", CONSOLE_CMD_RELEASE_DRIVER);
	RAISE_SPL(SPL_TTY);
	s3t_unlock();
	LOWER_SPL(SPL_ZERO);
	if (pci_regs) KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(pci_regs, S3V_REGSIZE);
	if (pci_valid) PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
